
#include <ctype.h>
#include <stdlib.h>
#define _XTAL_FREQ 48000000
#include <xc.h>

#include "app_device_cdc_basic.h"
#include "app_led_usb_status.h"

#include "usb.h"
#include "usb_device.h"
#include "usb_device_cdc.h"

#include "config.h"
#include "flash.h"
#include "system.h"
#include "strings.h"

#if defined(_PIC16F1459_H_)
#define TEST_BOARD
#endif

config_t active_config, saved_config;
status_t status;
extern unsigned short ms_timer;

#ifdef DEBUG_OSC
const unsigned long default_config[12] @ 0x1FC0 = { 0, 50, 500, 450, 0xFFFFFFFF, 0, 1000, 60000, 0, 0, 60000, 1 };
#else
const unsigned long default_config[12] @ 0x1FC0 = { 0, 50, 1000, 1000, 0xFFFFFFFF, 1, 1000, 60000, 3, 0, 60000, 1 };
#endif

static uint8_t readBuffer[CDC_DATA_OUT_EP_SIZE];
static char cmdBuffer[255];
unsigned char cmdBufferPos;

/* commands:

   help
   show [active|saved] config
   show status
   set trigger on (high|low|rise|fall|change)
   set trigger delay [min|max|fixed] (<time>|VR[12] <time> to <time>)
   set on time [min|max] [resettable] (<time>|VR[12] <time> to <time>)
   set off time (<delay>|VR[12] <delay> to <delay>))
   set led timer (infinite|<time>|VR[12] <time> to <time>)
   pull-up (on|off)
   save
   revert

   pin configurations for test board:

   RC2 = trigger input, not inverted, external pull-up resistor
   RB7 = trigger input bias pin (high-impedance or pull high)
   RA5 = relay driver output, active-low
   RB4/AN10 = trimpot VR1
   RB5/AN11 = trimpot VR2
   RC1 = on LED, active-high
   RB6 = off LED, active-high

   pin configurations for final version:

   RC2 = trigger input, inverted, external pull-up resistor
   RC4 = trigger input bias pin (high-impedance or pull high)
   RC0 = relay driver output, active-high
   RC3/AN7 = trimpot VR1
   RA4/AN3 = trimpot VR2
   RC5 = on LED, active-high
   RA5 = off LED, active-high
 */

static unsigned short ReadADC(unsigned char channel) {
    ADCON0 = (channel << 2) | 3;
    __delay_us(200);
    ADCON0bits.GO = 1;
    while( ADCON0bits.GO )
        ;
    return (((unsigned short)ADRESH)<<8) | ADRESL;
}

static void copy_config_from_flash() {
    HEFLASH_readBlock((char*)&saved_config.padding[0], 0x1FC0, 32);
    HEFLASH_readBlock((char*)&saved_config.padding[8], 0x1FE0, 32);
}

static void copy_config_to_flash() {
    HEFLASH_writeBlock(0x1FC0, (char*)&saved_config.padding[0], 32);
    HEFLASH_writeBlock(0x1FE0, (char*)&saved_config.padding[8], 32);
}

#if 0
static void default_config(config_t* cfg) {
  memset(cfg, 0, sizeof(*cfg));
  cfg->input_delay_ms = 50;
  cfg->trimpots[0].controls = controls_on_time;
  cfg->trimpots[0].min = 1000;
  cfg->trimpots[0].max = 60000;
  cfg->trimpots[1].controls = controls_off_time;
  cfg->trimpots[1].max = 60000;
  cfg->led_on_time_ms = 0xFFFFFFFFUL;
  cfg->pullup_resistor_enabled = 1;
}
#endif

static void print_num(char** dest, unsigned long val, unsigned char min_div) {
    unsigned long dig_val = min_div;
    while( dig_val < val ) {
        unsigned long temp = dig_val * 10;
        if( temp > dig_val && temp <= val )
            dig_val = temp;
        else
            break;
    }
    do {
        unsigned char dig = val / dig_val;
        *(*dest)++ = '0' + (char)dig;
        val -= dig * dig_val;
        dig_val /= 10;
    } while( dig_val );
}

// maximum length of result: 11 chars
static void print_ms(char** dest, unsigned long ms) {
    unsigned long s, m, h;
    unsigned char min, zero;

    s = ms / 10;
    s /= 10; // this takes up less flash space than dividing by 1000
    s /= 10; // because we're already dividing an unsigned long by 10 elsewhere
    m = s / 60;
    h = m / 60;
    min = 1;
    zero = (ms == 0);

    ms -= s * 1000;
    s -= m * 60;
    m -= h * 60;
    if( h ) {
        print_num(dest, h, 1);
        *(*dest)++ = 'h';
        min = 10;
    }
    if( m ) {
        print_num(dest, m, min);
        *(*dest)++ = 'm';
        min = 10;
    }
    if( s ) {
        print_num(dest, s, min);
        *(*dest)++ = (ms && !h) ? '.' : 's';
        min = 10;
    }
    if( zero || (ms && !h) ) {
        print_num(dest, ms, min > 1 ? 100 : 1);
        if( !s )
            *(*dest)++ = 'm';
        *(*dest)++ = 's';
    }
}

static unsigned char try_show_trimpot_config(char** ptr, config_t* cfg, trimpot_controls_t which) {
    unsigned char num;
    if( cfg->trimpots[0].controls == which ) {
       num = 0;
    } else if( cfg->trimpots[1].controls == which ) {
       num = 1;
    } else {
       return 0;
    }
    print_ms(ptr, cfg->trimpots[num].min);
    FLASH_read_string(ptr, STR__to_);
    print_ms(ptr, cfg->trimpots[num].max);
    FLASH_read_string(ptr, STR___VR);
    *(*ptr)++ = '1' + num;
    *(*ptr)++ = ')';
    return 1;
}

// maximum length of result: 93+93+47+48 - 21x2 = 239 chars
static void show_config(char** ptr, config_t* cfg) {
  // maximum string length for this section: 16+9+14+32+2+15+5 = 93 chars
  FLASH_read_string(ptr, STR__r_nOn_when_input_);//16 chars

  switch(cfg->input_polarity) {
  case trigger_on_high:
    FLASH_read_string(ptr, STR_is_high);
    break;
  case trigger_on_low:
    FLASH_read_string(ptr, STR_is_low);
    break;
  case trigger_on_rising_edge:
    FLASH_read_string(ptr, STR_goes_high);//9 chars
    break;
  case trigger_on_falling_edge:
    FLASH_read_string(ptr, STR_goes_low);
    break;
//case trigger_on_change:
  default:
    FLASH_read_string(ptr, STR_changes);
    break;
  }
  if( cfg->input_delay_ms ) {
    *(*ptr)++ = ' ';
    if( cfg->delay_type == delay_is_minimum )
        FLASH_read_string(ptr, STR_for_at_least_);
    else if( cfg->delay_type == delay_is_maximum )
        FLASH_read_string(ptr, STR_for_under_);
    else
        FLASH_read_string(ptr, STR_after_);
    if( !try_show_trimpot_config(ptr, cfg, controls_delay_time) ) {//maximum 32 chars
      print_ms(ptr, cfg->input_delay_ms);
    }
  } else {
    FLASH_read_string(ptr, STR___immediately);
  }
  *(*ptr)++ = ' ';//2 chars
  *(*ptr)++ = '(';
  if( cfg->pullup_resistor_enabled ) {
      FLASH_read_string(ptr, STR_pull_up_on);
  } else {
      FLASH_read_string(ptr, STR_pull_up_off);//15 chars
  }
  FLASH_read_string(ptr, STR____r_nStays_on_);//5+9 chars

  // maximum string length for this section: 9+4+9+32+13+23+3 = 93 chars
  if( cfg->relay_on_time_ms == 0xFFFFFFFF ) {
    if( cfg->on_time_type == on_time_is_maximum ) {
      FLASH_read_string(ptr, STR_until_input_changes__r_n);
    } else {
      FLASH_read_string(ptr, STR_until_power_cycles__r_n);
    }
  } else {
    FLASH_read_string(ptr, STR_for_);//4 chars
    if( cfg->on_time_type == on_time_is_minimum ) {
      FLASH_read_string(ptr, STR_at_least_);//9 chars
    }
    if( !try_show_trimpot_config(ptr, cfg, controls_on_time) ) {//maximum 32 chars
      print_ms(ptr, cfg->relay_on_time_ms);
    }
    if( cfg->on_time_resettable ) {
      FLASH_read_string(ptr, STR___resettable_);//13 chars
    }
    if( cfg->on_time_type == on_time_is_maximum ) {
      FLASH_read_string(ptr, STR__or_until_input_changes);//23 chars
    }
  }
  FLASH_read_string(ptr, STR___r_nOff_for__);//3+12 chars

  // maximum string length for this section: 12+32+3 = 47 chars
  if( !try_show_trimpot_config(ptr, cfg, controls_off_time) ) {//maximum 32 chars
    print_ms(ptr, cfg->relay_off_time_ms);
  }

  // maximum string length for this section: 11+32+5 = 48 chars
  FLASH_read_string(ptr, STR___r_nLED_timer__);//3+11 chars
  if( !try_show_trimpot_config(ptr, cfg, controls_led_on_time) ) {//maximum 32 chars
    if( cfg->led_on_time_ms == 0xFFFFFFFFUL ) {
      FLASH_read_string(ptr, STR_infinite);
    } else {
      print_ms(ptr, cfg->led_on_time_ms);
    }
  }
  FLASH_read_string(ptr, STR___r_n);//3 chars
  FLASH_read_string(ptr, STR__r_n);//2 chars
}

// maximum length of result: 16+15+51+115 = 197 chars
static void show_status(char** ptr) {//, status_t* stat) {
  unsigned char i;

  // maximum string length for this section: 9+4+3 = 16 chars
  FLASH_read_string(ptr, STR__r_nInput__);//9 chars
  if( status.input_state ) {
    FLASH_read_string(ptr, STR_high);//4 chars
  } else {
    FLASH_read_string(ptr, STR_low);
  }
  FLASH_read_string(ptr, STR___r_nRelay__);//3+7 chars

  // maximum string length for this section: 3+12 = 15 chars
  if( status.state != triggered && status.state != triggered_timeout ) {
    FLASH_read_string(ptr, STR_de_);//3 chars
  }
  FLASH_read_string(ptr, STR_energised__r_nState__);//12 + 7 chars

  // maximum string length for this section: 7+18+11+1+11+3 = 51 chars
  if( status.state == waiting ) {
    FLASH_read_string(ptr, STR_waiting_for_trigger);
  } else if( status.state == pre_trigger ) {
    FLASH_read_string(ptr, STR_trigger_delay_for_);//18 chars
    print_ms(ptr, status.timer);//maximum 11 chars
    *(*ptr)++ = '/';
    print_ms(ptr, status.delay_time);//maximum 11 chars
  } else if( status.state == post_trigger ) {
    FLASH_read_string(ptr, STR_relay_off_for_);
    print_ms(ptr, status.timer);
    *(*ptr)++ = '/';//1 char
    print_ms(ptr, status.off_time);
  } else {
    FLASH_read_string(ptr, STR_relay_on_for_);
    print_ms(ptr, status.timer);
    *(*ptr)++ = '/';
    print_ms(ptr, status.on_time);
  }
  FLASH_read_string(ptr, STR___r_n);// 3 chars

  // maximum string length for this section: 2x(3+10+13+6+3+4+11+1+3)+2 = 111 chars
  for( i = 0; i < 2; ++i ) {
      *(*ptr)++ = 'V'; // 3 chars
      *(*ptr)++ = 'R';
      *(*ptr)++ = '1'+i;
      FLASH_read_string(ptr, STR__controls_);//10 chars
      if( active_config.trimpots[i].controls == controls_nothing ) {
          FLASH_read_string(ptr, STR_nothing);
      } else {
          unsigned long val;
          switch( active_config.trimpots[i].controls ) {
              case controls_on_time:
                  FLASH_read_string(ptr, STR_on_time);
                  val = status.on_time;
                  break;
              case controls_delay_time:
                  FLASH_read_string(ptr, STR_trigger_delay);//13 chars
                  val = status.delay_time;
                  break;
              case controls_off_time:
                  FLASH_read_string(ptr, STR_off_time);
                  val = status.off_time;
                  break;
              default:
                  FLASH_read_string(ptr, STR_LED_on_time);
                  val = status.led_time;
                  break;
          }
          FLASH_read_string(ptr, STR___now_);//6 chars
          print_num(ptr, (status.trimpots[i] + status.trimpots[i] / 512) * 50 / 512, 1);//maximum 3 chars
          FLASH_read_string(ptr, STR_pct_equ_);//4 chars
          print_ms(ptr, val);//maximum 11 chars
          *(*ptr)++ = ')';//1 char
      }
      FLASH_read_string(ptr, STR___r_n);// 3 chars
  }
  FLASH_read_string(ptr, STR__r_n);//2 chars
}

static unsigned char get_ms(char** ptr, unsigned long* to_here, char space) {
    unsigned long ret = 0, val, pointval;
    unsigned long mult;
    char* endptr = 0;
    unsigned char point = 0;

    while( 1 ) {
        endptr = *ptr;
        val = strtoul(endptr, &endptr, 10);
        if( endptr != *ptr ) {
            switch( *endptr ) {
                case 's':
                    mult = 1000UL;
                    break;
                case 'm':
                    if( endptr[1] == 's' ) {
                        mult = 1UL;
                        ++endptr;
                    } else {
                        mult = 60000UL;
                    }
                    break;
                case 'h':
                    mult = 3600000UL;
                    break;
                case '.':
                    pointval = val;
                    point = 1;
                    mult = 0;
                    *ptr = endptr+1;
                    continue;
                default:
                    return 0;
            }
            if( point ) {
                char* temp = *ptr;
                ret += pointval * mult;
                while( temp != endptr ) {
                    mult /= 10;
                    ++temp;
                }
                if( !mult )
                    return 0;
            }
            ret += val * mult;
            *ptr = endptr+1;
            if( **ptr == space ) {
                *to_here = ret;
                return 1;
            }
        } else {
            break;
        }
    }
    return 0;
}

static unsigned char get_trimpot_config(char** ptr, trimpot_controls_t controls) {
    unsigned char vrnum = (*ptr)[2] - '1';
    unsigned long min, max;
    *ptr += 4;
    if( get_ms(ptr, &min, ' ') && (*ptr)[0] == ' ' && (*ptr)[1] == 't' && (*ptr)[2] == 'o' && (*ptr)[3] == ' ' ) {
        *ptr += 4;
        if( get_ms(ptr, &max, '\0') ) {
            active_config.trimpots[vrnum].min = min;
            active_config.trimpots[vrnum].max = max;
            active_config.trimpots[vrnum].controls = controls;
            return 1;
        }
    }
    return 0;
}

static unsigned char get_time_or_trimpot(char** ptr, unsigned long* ms_to_here, trimpot_controls_t controls) {
    if( (*ptr)[0] == 'v' && (*ptr)[1] == 'r' && ((*ptr)[2] == '1' || (*ptr)[2] == '2') && (*ptr)[3] == ' ' ) {
        return get_trimpot_config(ptr, controls);
    } else {
        unsigned char ret = get_ms(ptr, ms_to_here, '\0');
        if( ret ) {
            if( active_config.trimpots[0].controls == controls )
                active_config.trimpots[0].controls = controls_nothing;
            if( active_config.trimpots[1].controls == controls )
                active_config.trimpots[1].controls = controls_nothing;
        }
        return ret;
    }
}

static void ParseCommand() {
    char* ptr = cmdBuffer;
    unsigned char i;

    if( !*ptr ) {
        FLASH_read_string(&ptr, STR__r_n);
    } else if( !FLASH_cmp_string(&ptr, STR_help) ) {
        FLASH_read_string(&ptr, STR_1);
    } else if( !FLASH_cmp_string(&ptr, STR_show_config) || !FLASH_cmp_string(&ptr, STR_show_active_config) ) {
        show_config(&ptr, &active_config);
        if( memcmp(&active_config, &saved_config, sizeof(active_config)) && (ptr-cmdBuffer) <= 255-20 ) {
            ptr -= 2;
            FLASH_read_string(&ptr, STR_Changes_not_saved__r_n);//20 chars
            FLASH_read_string(&ptr, STR__r_n);//2 chars
        }
    } else if( !FLASH_cmp_string(&ptr, STR_show_saved_config) ) {
        show_config(&ptr, &saved_config);
    } else if( !FLASH_cmp_string(&ptr, STR_show_status) ) {
        show_status(&ptr);
    } else if( !FLASH_cmp_string(&ptr, STR_save) ) {
        memcpy(&saved_config, &active_config, sizeof(saved_config));
        copy_config_to_flash();
        FLASH_read_string(&ptr, STR__r_nSaved__r_n_r_n);
    } else if( !FLASH_cmp_string(&ptr, STR_reset) ) {
        RESET();
    } else if( !FLASH_cmp_string(&ptr, STR_revert) ) {
        copy_config_from_flash();
        memcpy(&active_config, &saved_config, sizeof(active_config));
        FLASH_read_string(&ptr, STR__r_nReverted__r_n_r_n);
    } else if( !FLASH_cmp_string(&ptr, STR_pull_up_on) ) {
        active_config.pullup_resistor_enabled = 1;
        goto done;
    } else if( !FLASH_cmp_string(&ptr, STR_pull_up_off) ) {
        active_config.pullup_resistor_enabled = 0;
        goto done;
    } else if( !FLASH_cmp_string(&ptr, STR_set_) ) {
        if( !FLASH_cmp_string(&ptr, STR_trigger_) ) {
            if( !FLASH_cmp_string(&ptr, STR_on_) ) {
                if( !FLASH_cmp_string(&ptr, STR_high) )
                    active_config.input_polarity = trigger_on_high;
                else if( !FLASH_cmp_string(&ptr, STR_low) )
                    active_config.input_polarity = trigger_on_low;
                else if( !FLASH_cmp_string(&ptr, STR_rise) )
                    active_config.input_polarity = trigger_on_rising_edge;
                else if( !FLASH_cmp_string(&ptr, STR_fall) )
                    active_config.input_polarity = trigger_on_falling_edge;
                else if( !FLASH_cmp_string(&ptr, STR_change) )
                    active_config.input_polarity = trigger_on_change;
                else
                    goto UnknownCommand;
            done:
                ptr = cmdBuffer;
                FLASH_read_string(&ptr, STR__r_nDone__r_n_r_n);
            } else if( !FLASH_cmp_string(&ptr, STR_delay_) ) {
                delay_type_t delay_type;
                if( !FLASH_cmp_string(&ptr, STR_min_) ) {
                    delay_type = delay_is_minimum;
                } else if( !FLASH_cmp_string(&ptr, STR_max_) ) {
                    delay_type = delay_is_maximum;
                } else if( !FLASH_cmp_string(&ptr, STR_fixed_) ) {
                    delay_type = delay_is_fixed;
                } else {
                    delay_type = delay_is_minimum; // default
                }
                if( get_time_or_trimpot(&ptr, &active_config.input_delay_ms, controls_delay_time) ) {
                    active_config.delay_type = delay_type;
                    goto done;
                } else {
                    goto UnknownCommand;
                }
            } else {
                goto UnknownCommand;
            }
        } else if( !FLASH_cmp_string(&ptr, STR_on_time_) ) {
            on_time_type_t on_time_type = on_time_is_exact;
            unsigned char on_time_resettable;
            if( !FLASH_cmp_string(&ptr, STR_min_) ) {
                on_time_type = on_time_is_minimum;
            } else if( !FLASH_cmp_string(&ptr, STR_max_) ) {
                on_time_type = on_time_is_maximum;
            }
            on_time_resettable = !FLASH_cmp_string(&ptr, STR_resettable_);
            if( get_time_or_trimpot(&ptr, &active_config.relay_on_time_ms, controls_on_time) ) {
                active_config.on_time_type = on_time_type;
                active_config.on_time_resettable = on_time_resettable;
                goto done;
            } else {
                goto UnknownCommand;
            }
        } else if( !FLASH_cmp_string(&ptr, STR_off_time_) ) {
            if( !get_time_or_trimpot(&ptr, &active_config.relay_off_time_ms, controls_off_time) )
                goto UnknownCommand;
            goto done;
        } else if( !FLASH_cmp_string(&ptr, STR_led_timer_) ) {
            if( !FLASH_cmp_string(&ptr, STR_infinite) ) {
                active_config.led_on_time_ms = 0xFFFFFFFFUL;
            } else if( !get_time_or_trimpot(&ptr, &active_config.led_on_time_ms, controls_led_on_time) ) {
                goto UnknownCommand;
            }
            goto done;
        } else {
            goto UnknownCommand;
        }
    } else {
    UnknownCommand:
        ptr = cmdBuffer;
        FLASH_read_string(&ptr, STR__r_nUnknown_command__r_n);
    }
    putUSBUSART(cmdBuffer, ptr-cmdBuffer);
}

static void setup_pins() {
#ifdef TEST_BOARD
//  RC2 = trigger input, inverted, external pull-up resistor
    ANSELCbits.ANSC2 = 0;
//  RB7 = trigger input bias pin (high-impedance or pull high)
    LATBbits.LATB7 = 1;
//  RA5 = relay driver output, active-low
    LATAbits.LATA5 = 1;
    TRISAbits.TRISA5 = 0;
//  RB4/AN10 = trimpot VR1
//  RB5/AN11 = trimpot VR2
//  RC1 = on LED, active-high
    ANSELCbits.ANSC1 = 0;
    TRISCbits.TRISC1 = 0;
//  RC5 = off LED, active-high
    TRISCbits.TRISC5 = 0;
#else
//  RC2 = trigger input, inverted, external pull-up resistor
//  RC4 = trigger input bias pin (high-impedance or pull high)
    ANSELCbits.ANSC2 = 0;
    LATCbits.LATC4 = 1;
//  RC0 = relay driver output, active-high
    ANSELCbits.ANSC0 = 0;
    TRISCbits.TRISC0 = 0;
//  RC3/AN7 = trimpot VR1
//  RA4/AN3 = trimpot VR2
//  RC5 = on LED, active-high
    TRISCbits.TRISC5 = 0;
//  RA5 = off LED, active-high
    TRISAbits.TRISA5 = 0;
#endif
}


static void on_led_on() {
#ifdef TEST_BOARD
    LATBbits.LATB6 = 0;
    LATCbits.LATC1 = 1;
#else
    LATAbits.LATA5 = 0;
    LATCbits.LATC5 = 1;
#endif
}

static void off_led_on() {
#ifdef TEST_BOARD
    LATCbits.LATC1 = 0;
    LATBbits.LATB6 = 1;
#else
    LATCbits.LATC5 = 0;
    LATAbits.LATA5 = 1;
#endif
}

static void both_leds_off() {
#ifdef TEST_BOARD
    LATCbits.LATC1 = 0;
    LATBbits.LATB6 = 0;
#else
    LATAbits.LATA5 = 0;
    LATCbits.LATC5 = 0;
#endif
}

static void set_led_timer() {
    status.led_timer = status.led_time+1;
}

static void relay_on() {
#ifdef TEST_BOARD
    LATAbits.LATA5 = 0;
#else
    LATCbits.LATC0 = 1;
#endif
    if( active_config.on_time_type == on_time_is_exact )
        status.state = triggered_timeout;
    else
        status.state = triggered;
    status.timer = 0;
    if( status.led_time ) {
        set_led_timer();
        on_led_on();
    }
    status.retrigger = 0;
}

static void relay_off() {
#ifdef TEST_BOARD
    LATAbits.LATA5 = 1;
#else
    LATCbits.LATC0 = 0;
#endif
    if( status.off_time ) {
        status.state = post_trigger;
        status.timer = 0;
    } else {
        status.state = waiting;
    }
    if( status.led_time ) {
        set_led_timer();
        off_led_on();
    }
}

static void update_trimpot_vals() {
    unsigned char i;
    unsigned long diff;

    for( i = 0; i < 2; ++i ) {
        if( active_config.trimpots[i].controls != controls_nothing ) {
            diff = active_config.trimpots[i].max - active_config.trimpots[i].min;
            diff = (unsigned long long)diff * status.trimpots[i] / 1023;
            diff += active_config.trimpots[i].min;
            switch( active_config.trimpots[i].controls ) {
                case controls_on_time:
                    status.on_time = diff;
                    break;
                case controls_delay_time:
                    status.delay_time = diff;
                    break;
                case controls_off_time:
                    status.off_time = diff;
                    break;
                default:
                    status.led_time = diff;
                    break;
            }
        }
    }
}

MAIN_RETURN main(void) {
#if defined(USE_INTERNAL_OSC)
    //Make sure to turn on active clock tuning for USB full speed
    //operation from the INTOSC
    OSCCON = 0xFC;  //HFINTOSC @ 16MHz, 3X PLL, PLL enabled
    ACTCON = 0x90;  //Active clock tuning enabled for USB
#endif

    setup_pins();

    copy_config_from_flash();
//    if( ((unsigned char*)&saved_config)[0] == 0xFF )
//        default_config(&saved_config);
    memcpy(&active_config, &saved_config, sizeof(active_config));

    ADCON1 = 32|64|128;
    USBDeviceInit();
    OPTION_REG = 0xD4;
    INTCONbits.PEIE = 1;
    INTCONbits.TMR0IE = 1;

    while(1) {
        unsigned short local_ms_timer;

        INTCONbits.GIE = 0;
        local_ms_timer = ms_timer;
        ms_timer = 0;
        INTCONbits.GIE = 1;
        if( local_ms_timer > 0 ) {
            unsigned char last_input_state = status.input_state;
            unsigned char trig, untrig;

#ifdef TEST_BOARD
            status.input_state = PORTCbits.RC2;
#else
            status.input_state = !PORTCbits.RC2;
#endif
#ifdef TEST_BOARD
            if( active_config.pullup_resistor_enabled ) {
                TRISBbits.TRISB7 = 0;
            } else {
                TRISBbits.TRISB7 = 1;
            }
#else
            if( active_config.pullup_resistor_enabled ) {
                TRISCbits.TRISC4 = 0;
                LATCbits.LATC4 = 1;
            } else {
                TRISCbits.TRISC4 = 1;
            }
#endif

            status.on_time = active_config.relay_on_time_ms;
            status.delay_time = active_config.input_delay_ms;
            status.off_time = active_config.relay_off_time_ms;
            status.led_time = active_config.led_on_time_ms;
            update_trimpot_vals();

            if( status.led_timer ) {
                if( status.led_timer <= local_ms_timer ) {
                    both_leds_off();
                    status.led_timer = 0;
                } else {
                    status.led_timer -= local_ms_timer;
                }
            }
            switch( active_config.input_polarity ) {
                case trigger_on_high:
                    trig = (status.input_state != 0);
                    untrig = (status.input_state == 0);
                    break;
                case trigger_on_low:
                    trig = (status.input_state == 0);
                    untrig = (status.input_state != 0);
                    break;
                case trigger_on_rising_edge:
                    trig = (status.input_state != 0 && last_input_state == 0);
                    untrig = (status.input_state == 0);
                    break;
                case trigger_on_falling_edge:
                    trig = (status.input_state == 0 && last_input_state != 0);
                    untrig = (status.input_state != 0);
                    break;
//              case trigger_on_change:
                default:
                    trig = (status.input_state != last_input_state);
                    untrig = trig;
                    break;
            }
//            trig = (active_config.input_polarity == trigger_on_high         && status.input_state != 0) ||
//                   (active_config.input_polarity == trigger_on_low          && status.input_state == 0) ||
//                   (active_config.input_polarity == trigger_on_rising_edge  && status.input_state != 0 && last_input_state == 0) ||
//                   (active_config.input_polarity == trigger_on_falling_edge && status.input_state == 0 && last_input_state != 0) ||
//                   (active_config.input_polarity == trigger_on_change       && status.input_state != last_input_state);
//            untrig = ((active_config.input_polarity == trigger_on_high || active_config.input_polarity == trigger_on_rising_edge)  && status.input_state == 0) ||
//                     ((active_config.input_polarity == trigger_on_low  || active_config.input_polarity == trigger_on_falling_edge) && status.input_state != 0) ||
//                      (active_config.input_polarity == trigger_on_change                                                           && status.input_state != last_input_state);
            switch( status.state ) {
                case waiting:
                    if( trig ) {
                        if( status.delay_time ) {
                            status.state = pre_trigger;
                            status.timer = 0;
                        } else {
                            relay_on();
                        }
                        break;
                    }
                    break;
                case pre_trigger:
                    if( active_config.delay_type == delay_is_fixed ) {
                        status.timer += local_ms_timer;
                        if( status.timer >= status.delay_time ) {
                            relay_on();
                        }
                    } else if( untrig ) {
                        if( active_config.delay_type == delay_is_maximum ) {
                            if( status.timer <= status.delay_time ) {
                                relay_on();
                            } else {
                                status.state = waiting;
                            }
                        }
                    } else {
                        if( status.input_state != last_input_state )
                            status.timer = 0;
                        else
                            status.timer += local_ms_timer;

                        if( active_config.delay_type != delay_is_maximum && status.timer >= status.delay_time ) {
                            relay_on();
                        }
                    }
                    break;
                case triggered:
                    status.timer += local_ms_timer;
                    if( untrig ) {
                        if( active_config.on_time_type == on_time_is_maximum ) {
                            relay_off();
                        } else {
                            status.state = triggered_timeout;
                        }
                    }
                    break;
                case triggered_timeout:
                    status.timer += local_ms_timer;
                    if( status.timer >= status.on_time )
                        relay_off();
                    break;
                case post_trigger:
                    status.timer += local_ms_timer;
                    if( status.timer >= status.off_time )
                        status.state = waiting;
                    break;
            }

            if( active_config.on_time_resettable && (status.state == triggered || status.state == triggered_timeout) ) {
                if( trig && !status.retrigger ) {
                    if( status.delay_time ) {
                        status.retrigger = 1;
                        status.retrigger_timer = 0;
                    } else {
                        relay_on();
                    }
                } else if( status.retrigger ) {
                    if( untrig ) {
                        if( active_config.delay_type == delay_is_maximum && status.retrigger_timer <= status.delay_time )
                            relay_on();
                        else
                            status.retrigger = 0;
                    } else {
                        status.retrigger_timer += local_ms_timer;
                        if( active_config.delay_type == delay_is_minimum && status.retrigger_timer >= status.delay_time ) {
                            relay_on();
                        }
                    }
                }
            }

#ifdef TEST_BOARD
            status.trimpots[0] = ReadADC(10);
            status.trimpots[1] = ReadADC(11);
#else
            status.trimpots[0] = ReadADC(7);
            status.trimpots[1] = ReadADC(3);
#endif
        }

        if( USBGetDeviceState() >= CONFIGURED_STATE ) {
            if( !USBIsDeviceSuspended() ) {
                if( USBUSARTIsTxTrfReady() == true ) {
                    uint8_t i, parsed;
                    uint8_t numBytesRead;

                    numBytesRead = getsUSBUSART(readBuffer, sizeof(readBuffer));
                    parsed = 0;
                    for( i = 0; i < numBytesRead; ++i ) {
                        if( cmdBufferPos == sizeof(cmdBuffer) ) {
                            // this should really be memmove but since dst<src, memcpy should work and it makes the program more compact
                            memcpy(cmdBuffer, cmdBuffer+1, sizeof(cmdBuffer)-1);
                            --cmdBufferPos;
                        }
                        if( readBuffer[i] == '\b' ) {
                            if( cmdBufferPos )
                                --cmdBufferPos;
                            if( numBytesRead == 1 ) {
                                readBuffer[1] = ' ';
                                readBuffer[2] = '\b';
                                numBytesRead = 3;
                            }
                        } else if( readBuffer[i] != '\n' ) {
                            cmdBuffer[cmdBufferPos] = tolower((char)readBuffer[i]);
                            if( cmdBuffer[cmdBufferPos] == '\r' ) {
                                cmdBuffer[cmdBufferPos] = '\0';
                                ParseCommand();
                                ++parsed;
                                cmdBufferPos = 0;
                            } else {
                                ++cmdBufferPos;
                            }
                        }
                    }
                    if( !parsed ) {
                        putUSBUSART(readBuffer, numBytesRead);
                    }
                }
                CDCTxService();
            }
        } else {
            USBTryDeviceAttach();
        }
    }
}
