/*
code to retrieve temperature data from :-
- rempote DS18B20 temperature sensors via Xbee and a picaxe 20X2 serial to i2c bridge
- local i2c mcp9808 temperature sensor
*/

#include <Wire.h>

//i2c device addresses
#define mcp9808_addr 0x18     // MCP9808 temperature sensor (0x18 - 7bit, 0x30 - 8bit)
#define picaxe_addr 0x54      // picaxe 20x2 slave address  (0x54 - 7bit, 0xA8 - 8bit) 

boolean picaxe_20x2 = true;   //remote 20X2/DS18B20 remote temperature sensors PRESENT
boolean mcp9808 = true;       //local mcp9808 i2c temperature sensor PRESENT

word mcpdata;                 //general purpose word for data read from mcp9808, picaxe20X2

//temperature related items
float temperatureA;           // the actual calculated temperature in degC from the remote DS18B20 sensor 1
float temperatureB;           // the actual calculated temperature in degC from the remote DS18B20 sensor 2
float temperatureC;           // the actual calculated temperature in degC from the local mcp9808 chip


void setup()
  {
  
    Serial.begin(9600);
    
    Wire.setClock(400000);    // set i2c frequency to 400kHz
    Wire.begin();

   
// Initialise the MCP9808 i2c temperature sensor

    Wire.beginTransmission(mcp9808_addr);
    Wire.write(0x01);             // select config register
    Wire.write(0x00);             // Continuous conversion mode, Power-up default  
    Wire.write(0x00); 
    Wire.endTransmission();

    Wire.beginTransmission(mcp9808_addr);
    Wire.write(0x08);             // select resolution register
    Wire.write(0x03);             // 0.0625 degC  
    Wire.endTransmission();
}


void loop()
{

if(picaxe_20x2 == true)
{
    picaxe_read(0);
    if (mcpdata == 0x5251)                    //valid data prefixed with the letters RQ (R=52, Q=51) as the first 2 bytes in 20X2 scratchpad
                                              //as sent by remote picaxe14M2 (see 14M2 code)
     {    
      picaxe_read(4);                         //picaxe pointer 4 (temperatureA)
      temperatureA = Tconvert(0x0800);        //pass the sign bit for DS18B20, mcpdata returned
      Serial.print("DS18B20-1 ");
      if(temperatureA >= 0) {Serial.print("+");}
      Serial.print(temperatureA);
      Serial.println("degC");
        
      picaxe_read(2);                         //picaxe pointer 2 (temperatureB)
      temperatureB = Tconvert(0x0800);        //pass the sign bit for DS18B20, mcpdata returned
      Serial.print("DS18B20-2 ");
      if(temperatureB >= 0) {Serial.print("+");}
      Serial.print(temperatureB);
      Serial.println("degC");
     }
}
     
  if (mcp9808 == true)
  {    
    mcp9808_read();
    temperatureC = Tconvert(0x1000);          //pass the mcp9808 sign bit 12, mcpdata returned
    Serial.print("MCP9808-1 ");
    if(temperatureC >= 0) {Serial.print("+");}
    Serial.print(temperatureC);
    Serial.println("degC");
  }

  Serial.println();
  
  delay(5000);
}


//FUNCTIONS

// ################
// ### Tconvert ###
// ################
//converts 12 bit temperature data to 16 bit temperature data to enable easy 2's complement calculations
//note...this overwrites mcp9808 data bit11 (2^7 = 128degC)
//the sign bit is passed to this function (0x0800 for DS18B20 or 0x1000 for mcp9808)
//the function returns a signed floating temperature number (TdegC)

float Tconvert(word sign_bit)      
{ 
    float TdegC;
    
    if((mcpdata & sign_bit) == sign_bit)    //test the sign bit to see if temperature is -ve
      {
        mcpdata = (mcpdata | 0xf800);       //if -ve insert 1's in bits 15,14,13,12,11
        mcpdata =  ~(mcpdata) +1;           //2's complement conversion
        TdegC = ((mcpdata * 0.0625) * -1);  //convert to signed floating number
      } 
      else 
      {
        mcpdata = (mcpdata & 0x07ff);       //otherwise if +ve insert 0's
        TdegC = ((mcpdata * 0.0625));       //convert to signed floating number
      }

return TdegC;
}      

// ########################
// ### picaxe 20X2 read ###
// ########################

void picaxe_read(byte pointer)
{
    //read
    Wire.beginTransmission(picaxe_addr);  
    Wire.write(pointer); // Select scratchpad pointer 
    Wire.endTransmission();
    
    Wire.requestFrom(picaxe_addr,2);   // get 2 bytes from the 20x2 slave scratchpad memory (temperatureA & temperatureB)
    byte sp_0 = Wire.read();  //MSB temperatureA
    byte sp_1 = Wire.read();  //LSB
    
    mcpdata = (sp_0 * 256) + sp_1;    // combine into a word
}

// ####################
// ### mcp9808_read ###
// ####################

void mcp9808_read()
{
    //read
    Wire.beginTransmission(mcp9808_addr);  
    Wire.write(0x05); // Select data register 
    Wire.endTransmission();
    
    Wire.requestFrom(mcp9808_addr,2);   // get 4 bytes from the 20x2 slave scratchpad memory
    byte sp_0 = Wire.read();  //MSB
    byte sp_1 = Wire.read();  //LSB
    
    mcpdata = (sp_0 * 256) + sp_1;    // combine into a word
}
