// Connects to WiFi, requests NTP data and sends out as valid GPS sentences (RMC, GSA and GGA), with offset added to millis()
// Also shows Wifi status via custom ESP82 sentence
// Implements 64bit millis() for long term installation
// Settings menu with save to EEPROM with defaults (baudrate, WIFI SSID, WIFI PASS, NTP Server, Dummy COORDS for TZ detecting clock)
// uses IP API to get approx latitude/longitude
// LED status- on solid while connecting to Wi-Fi, flickers during NMEA data transmit
// working with analog clock and HiVis Clock
// V9 Add 1PPS feature
// V10 change srollover to 64 bit to prevent glitching
// V11 general tidyup (variable location/scope)

#include <EEPROM.h>
#define BAUDRATE_DEFAULT 9600
#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>
#include <WiFiUdp.h>
HTTPClient http;
WiFiUDP udp;
IPAddress hostIP;
//GPIO4 is D2 on D1 Mini
#define PPSPIN 4

//default settings
char ssid[80] = "SSID";                   //needs to be changed in settings
char pass[80] = "PASSWORD";               //needs to be changed in settings
char host[80] = "pool.ntp.org";           //should work OK
char dummy[80]= "3351.000,S,15112.000,E"; //somewhere in Sydney
char ipapi[]="http://ip-api.com/line?fields=lat,lon";
char hex[]="0123456789ABCDEF";            //for writing out checksum
unsigned long baudrate=9600;

unsigned long long boott=(43099ULL*86400ULL)<<32;    //1/1/2018
unsigned long ts=0;
unsigned long dsn;            //day serial number
int date,mon,yr;              //calculated from daysn
bool ntplock=false;           //set after we get a valid NTP return
bool ntphit=false;            //an attempt was made to get NTP
unsigned long long srollover;      //for detecting when a new second rolls over
long lastadjust=0;            //track last adjustment, signed to see if +/-
unsigned long lastNTP;        //last time NTP was attempted
unsigned long long tnow;      //64bit time to work with
unsigned long tshort;         //32 bit time in seconds

void setup() {
  pinMode(LED_BUILTIN,OUTPUT);//active low on some devices
  digitalWrite(LED_BUILTIN,LOW);
  pinMode(PPSPIN,OUTPUT);
  pinMode(PPSPIN,LOW);
  int httpCode;               //for HTTP call
  int httptries=10;           //retries for IP API
  unsigned long long ttemp;        //for storing millis() while working
  delay(1000);
  loadconfig();               //load from EEPROM, including baudrate
  Serial.begin(baudrate);
  WiFi.mode(WIFI_STA);
  WiFi.begin(ssid, pass);
  while(WiFi.status()!=WL_CONNECTED){
    ttemp=millis64();
    if((ttemp%1000)<srollover){       //do this each second to let connected device know we're awake
      togglePPS();
      doupdate();
      doRMC();
      doGGA();
      doGSA();
      doESPstat();
    }
    checkinput();
    srollover=ttemp%1000;
    yield();                          //to stop watchdog resets
  }
  digitalWrite(LED_BUILTIN,HIGH);     //LED off now Wi-Fi connected
  while(httptries>0){                 //use IP API to find lat/lon
    ttemp=millis64();
    http.begin(ipapi);                //URL
    httpCode=http.GET();              //fetch
    if(httpCode==200){                //valid data returned?
      setlatlon(http.getString());    //set lat/lon from data
      httptries=0;                    //got our data, so continue
    }
    http.end();
    httptries--;
    if((ttemp%1000)<srollover){       //do this each second to let connected device know we're awake
      togglePPS();
      doupdate();
      doRMC();
      doGGA();
      doGSA();
      doESPstat();
    }
    srollover=ttemp%1000;
    yield();                          //to stop watchdog resets
  }

  udp.begin(2390);
  getNTP();                           //try to get time
  lastNTP=millis();
}

void loop() {
  unsigned long long ttemp;
  if((millis()-lastNTP>3600000UL)||(!ntplock&&(millis()-lastNTP>60000UL))){   //by default, try every hour, or every minute if not locked on
    getNTP();
    lastNTP=millis();
  }
  ttemp=millis64();
  if((ttemp%1000)<srollover){           //output sentences each second, will glitch to (1)296ms every 71 days
    digitalWrite(LED_BUILTIN,LOW);      //LED on
    togglePPS();
    doupdate();
    doRMC();
    doGGA();
    doGSA();
    doESPstat();
    digitalWrite(LED_BUILTIN,HIGH);     //LED off
  }
  checkinput();
  srollover=ttemp%1000;
}

void getNTP(){
  byte NTPpacket[48]="";
  unsigned long long ts3,ts4;
  unsigned long long newboott,ts1,ts2;           //to measure adjustment, outgoing/return timestamps
  int i;
  ntphit=true;            //flag that we're trying to do an update
  for(i=0;i<48;i++){      //clear packet contents 
    NTPpacket[i]=0;
  }
  NTPpacket[0] = 0xE3;    //set default values for outgoing packet to server
  NTPpacket[2] = 0x06;
  NTPpacket[3] = 0xEC;
  NTPpacket[12]  = 0x31;
  NTPpacket[13]  = 0x4E;
  NTPpacket[14]  = 0x31;
  NTPpacket[15]  = 0x34;
  udp.beginPacket(host,123);    //NTP requests are to port 123
  udp.write(NTPpacket,48);      //buffer 48 bytes
  udp.endPacket();              //send 48 bytes
  ts1=millis64();               //outgoing timestamp
  while((millis64()-ts1<500UL)&(!udp.parsePacket())){}      //wait up to a 0.5s for packet to return
  ts2=millis64();               //return timestamp
  udp.read(NTPpacket,48);       //save the returned packet
  ts3=(((unsigned long long)NTPpacket[32])<<56)|(((unsigned long long)NTPpacket[33])<<48)|(((unsigned long long)NTPpacket[34])<<40)|(((unsigned long long)NTPpacket[35])<<32)|(((unsigned long long)NTPpacket[36])<<24)|(((unsigned long long)NTPpacket[37])<<16)|(((unsigned long long)NTPpacket[38])<<8)|(((unsigned long long)NTPpacket[39]));
  ts4=(((unsigned long long)NTPpacket[40])<<56)|(((unsigned long long)NTPpacket[41])<<48)|(((unsigned long long)NTPpacket[42])<<40)|(((unsigned long long)NTPpacket[43])<<32)|(((unsigned long long)NTPpacket[44])<<24)|(((unsigned long long)NTPpacket[45])<<16)|(((unsigned long long)NTPpacket[46])<<8)|(((unsigned long long)NTPpacket[47]));
  if((ts3!=0)&&(ts4!=0)){                           //if valid packet returned
    newboott=ts3-NTPfromms(ts1+(ts2-ts1)/2);        //calculate time offset
    lastadjust=msfromNTP(((long long)newboott-(long long)boott));    //cast to signed so we can check for +/- adjustment
    boott=newboott;                                 //update time offset
    ntplock=true;                                   //fix is good
  }
}

void dodsn(unsigned long dsn){
  byte dim[]={0,31,28,31,30,31,30,31,31,30,31,30,31};   //days in month
  yr=1900;
  mon=1;
  date=dsn+1;                       //1/1/1900 has timestamp 0, ie date of 0 is 1
  while(date>365){                  //count years             
    if(((yr%4)==0)&&(yr!=1900)){
      date--;
    }
    yr++;
    date=date-365;
  }
  if(((yr%4)==0)&&(yr!=1900)){dim[2]++;}  //if a leap year, Feb has 29
  while(date>dim[mon]){                   //count months
    date=date-dim[mon];
    mon++;
  }
  if((date==0)&&(mon==1)){                //glitch on 31/12 of leap years
    yr--;
    date=31;
    mon=12;
  }
}

unsigned long long NTPfromms(unsigned long long t){      //converts a time in ms to time in 64 bit NTP, max precision, will rollover after 194 years
  return ((t<<22)/1000)<<10;
}

unsigned long msfromNTP(unsigned long long t){      //converts a 64 bit NTP time into ms
  return ((t*1000)>>32);                            //resolution will be lost however we do this
}

int seconds(unsigned long t){
  return t%60;
}

int minutes(unsigned long t){
  return (t/60)%60;
}

int hours(unsigned long t){
  return (t/3600)%24;
}

void doESPstat(){                       //prints a sentence that looks like GPS data, but gives us WIFI info
  char sentence[82]="";                 //WiFi.status(),WiFi.localIP(),ntp data received, have we just tried to hit NTP?, last adjustment value in ms
  byte checksum=0x00;
  sprintf(sentence,"$ESP82,%02d,%03d.%03d.%03d.%03d,%01d,%01d,%01d*",WiFi.status(),WiFi.localIP()[0],WiFi.localIP()[1],WiFi.localIP()[2],WiFi.localIP()[3],ntplock,ntphit,lastadjust);
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);
  ntphit=false;                         //clear this flag
}

void doupdate(){                          //update sentence values
  tnow=boott+NTPfromms(millis64());       //get current time
  tshort=(tnow)>>32;                      //get integer seconds part
  dodsn(tshort/86400);                    //calculate date
}

void doRMC(){                             //output an RMC sentence
  char sentence[82]="";                   //standard RMC: time, fix, coordinates, date, checksum
  byte checksum=0x00;
  sprintf(sentence,"$GPRMC,%02d%02d%02d.%03d,%c,%s,0.00,000.00,%02d%02d%02d,,,*" ,hours(tshort),minutes(tshort),seconds(tshort),((unsigned long)(tnow))/4294968UL,ntplock?'A':'V',dummy,date,mon,yr%100);  
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);      
}

void doGSA(){                           //ouput a GSA sentence
  char sentence[82]="";
  byte checksum=0x00;
  sprintf(sentence,"$GPGSA,A,%c,,,,,,,,,,,,,1.00,1.00,1.00,*",ntplock?'3':'1');
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);        
}

void doGGA(){                           //output a GGA sentence
  char sentence[82]="";
  byte checksum=0x00;
  sprintf(sentence,"$GPGGA,%02d%02d%02d.%03d,%s,%c,04,1.0,0.0,M,0.0,M,,*" ,hours(tshort),minutes(tshort),seconds(tshort),((unsigned long)(tnow))/4294968UL,dummy,ntplock?'8':'0');
  for(int i=1;i<strlen(sentence)-1;i++){
    checksum=checksum^sentence[i];
  }
  sentence[strlen(sentence)]=hex[(checksum>>4)&0xF];
  sentence[strlen(sentence)]=hex[checksum&0xF];
  Serial.println(sentence);          
}

unsigned long long millis64(){
  static unsigned long lo,hi;   //hi and lo 32 bits
  unsigned long newlo=millis(); //check millis()
  if(newlo<lo){hi++;}           //rollover has happened
  lo=newlo;                     //save for next check
  return ((unsigned long long)(hi) << 32) | ((unsigned long long)(lo));  //return 64 bit result
}

void checkinput(){              //see if user wants to change settings
  if(Serial.available()){
    if(Serial.read()=='~'){
      delay(100);
      while(Serial.available()){Serial.read();}   //purge stream before input
      doconfig();
    }
  }
}

void doconfig(){  
  bool done=false;
  int d,i;
  EEPROM.begin(1024);
  showmenu();
  while(!done){
    if(Serial.available()){
      d=Serial.read();
      if(d>31){Serial.println((char)d);}  
      switch(d){
        case '1':
          baudrate=4800;
          EEPROM.put(0,baudrate);
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          showmenu();
          break;
        case '2':
          baudrate=9600;
          EEPROM.put(0,baudrate);
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          showmenu();
          break;
        case '3':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter SSID:"));
          getstr(ssid);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(4+i,ssid[i]);                //save to EEPROM
          }
          showmenu();
          break;
        case '4':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter password:"));
          getstr(pass);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(84+i,pass[i]);               //save to EEPROM
          }
          showmenu();
          break;
        case '5':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter NTP Server:"));
          getstr(host);                               //get input
          for(i=0;i<80;i++){
            EEPROM.write(164+i,host[i]);              //save to EEPROM
          }
          showmenu();
          break;
        case '6':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Enter Dummy Coords (DDMM.SS,N/S,DDDMMSS,E/W):"));
          getstr(dummy);                              //get input
          for(i=0;i<80;i++){
            EEPROM.write(244+i,dummy[i]);             //save to EEPROM
          }
          showmenu();
          break;
        case '9':
          delay(100);
          while(Serial.available()){Serial.read();}   //purge stream before input
          Serial.println(F("Exit config, saved"));
          EEPROM.end();
          done=true;
          break;
      }
    }
  }  
}

void loadconfig(){
  int i;
  Serial.begin(9600);
  EEPROM.begin(1024);
  EEPROM.get(0,baudrate);
  Serial.println(baudrate);
  if((baudrate!=4800)&&(baudrate!=9600)){     //sanity check for corrupt EEPROM
    baudrate=BAUDRATE_DEFAULT;                //load sane defaults
    EEPROM.put(0,baudrate);                   //save safe defaults      
    for(i=0;i<80;i++){
      EEPROM.write(4+i,ssid[i]);
      EEPROM.write(84+i,pass[i]);
      EEPROM.write(164+i,host[i]);
      EEPROM.write(244+i,dummy[i]);
    }
    Serial.println(F("Defaults written to EEPROM"));        
  }else{
    for(i=0;i<80;i++){                        //load from eeprom
      ssid[i]=EEPROM.read(4+i);
      pass[i]=EEPROM.read(84+i);
      host[i]=EEPROM.read(164+i);
      dummy[i]=EEPROM.read(244+i);
    }
  ssid[79]=0;           //null terminate in case we have garbage
  pass[79]=0;
  host[79]=0;
  dummy[79]=0;
  }
  EEPROM.end();
  delay(100);
  Serial.end();
}

void showmenu(){
  Serial.println();
  Serial.println(F("NTP GPS Source Setup:"));
  Serial.print(F("Current Baudrate:"));
  Serial.println(baudrate);
  Serial.println(F("1.Set 4800 Baudrate"));
  Serial.println(F("2.Set 9600 Baudrate"));
  Serial.print(F("3.Set SSID. Current:"));
  Serial.println(ssid);
  Serial.print(F("4.Set Password. Current:"));
  Serial.println(pass);
  Serial.print(F("5.Set NTP Server. Current:"));
  Serial.println(host);
  Serial.print(F("6.Set Dummy Coords. Current:"));
  Serial.println(dummy);
  Serial.println(F("9.Exit and save"));
  Serial.println(F("Enter a number:"));
}

void getstr(char *s){
  char buf[100]="";
  bool done=false;
  int i,d;
  while(!done){
    if(Serial.available()){
      d=Serial.read();
      if(d>31){
        Serial.write(d);
        buf[strlen(buf)]=d;   
      }      
      if((d==13)||(strlen(buf)>78)){                //jump out
        done=true;
        Serial.println();
        delay(100);
        while(Serial.available()){Serial.read();}   //purge stream before input
      }
    }
  }
  for(i=0;i<80;i++){
    s[i]=buf[i];
  }
  s[79]=0;        //null terminate
}

void setlatlon(String a){               //extract latitude/longitude in RMC format from string in lat(CR)lon format. Works to nearest minute as this is <2km
  int lonindex;
  float lat,lon; 
  int lati,latf,loni,lonf;              //latitude/longitude integer/fractional parts
  char ns='N';
  char ew='E';                          //for positive values
  lat=a.toFloat();                      //latitude is first line
  lonindex=a.indexOf('\n')+1;           //look for /n
  lon=a.substring(lonindex).toFloat();  //longitude is on second line
  if((lat==0)||(lon==0)){return;}       //if either are zero, data is probably wrong
  if(lat<0){                            //check if negative and change compass direction
    ns='S';
    lat=-lat;
  }
  if(lon<0){                            //check if negative and change compass direction
    ew='W';
    lon=-lon;
  }
  lati=lat;
  latf=(lat-lati)*60;                   //counts up to 59 minutes
  loni=lon;
  lonf=(lon-loni)*60;                   //counts up to 59 minutes  
  sprintf(dummy,"%02d%02d.000,%c,%03d%02d.000,%c",lati,latf,ns,loni,lonf,ew);
}

void togglePPS(){
  pinMode(PPSPIN,HIGH);
  delay(10);
  pinMode(PPSPIN,LOW);  
}

