/* 
 File:   main.c        * Author: Peter
 Created on 29 June 2015, 8:30 PM
 Fridge door open alarm

 The input to the unit is closed when the door is open: this initiates the 
 timing process.
 
 An extend pushbutton allows for extending the door open time to
 facilitate stocking of the fridge or cleaning etc.

 The battery voltage is checked on each door open and compared to a reference 
 voltage derived from a silicon diode. If the battery voltage drops below a
 specified voltage (e.g. 2 V) then  upon opening the unit beeps 3 times.
 A reference voltage is gained by forward biasing a diode using an output turned
 on and using an analogue input to monitor the voltage across the diode.
 The user can vary the threshold voltage in the program to account for varying
 diode characteristics. 

 When the door is closed the alarm is in sleep mode and current consumption was
 measured at less than 1uA. Timing of the door open state is effected by
 charging a capacitor and, putting the unit to sleep and then waking with
 an intterupt triggered by the voltage across the capacitor falling to the input
 off threshold as it slowly discharges. This triggers the sounder to operate.
 If the extend button is pressed then this cycle occurs a number of times
 before the alarm is triggered.
 */

#include <stdio.h>
#include <stdlib.h>
#include <xc.h>
#include "user.h"   

#pragma config WDTE = OFF       // Watchdog Timer Enable bit (WDT disabled)
#pragma config PWRTE = OFF      // Power-up Timer Enable bit (PWRT disabled)
#pragma config MCLRE = OFF      // MCLR Pin Func Sel pin func is digi in
#pragma config CP = OFF         // Code Protection bit 
#pragma config CPD = OFF        // Data Code Protection bit 
#pragma config BOREN = OFF      // Brown Out Detect (BOR disabled)
#pragma config IESO = OFF       // Internal External Switchover bit 
#pragma config FCMEN = OFF      // Fail-Safe Clock Monitor Enabled bit 
#pragma config FOSC = 100;    //INTOSCIO Oscillator Selection bits (INTOSC osc)

#pragma OPTION_REG =0B00000000;
#pragma INTCON =0B00000000;
#define _XTAL_FREQ 8000000
#pragma PCON=0b00000001;

//function prototypes
void alarm(int, int,int);
void doorOpenSleep();

#define ON 1
#define OFF 0
#define vRef 0.67           //reference voltage for low battery alarm
#define tripLowVolts 2.0    //when battery to this level, alarms
#define sounderVol 122
#define threeBeeps 3
#define twoBeeps 2
#define sounderFreq1 250
#define sounderFreq2 254

#define doorClosed 0        //when door closed switch circuit is open
#define doorOpen 1
#define wakeUpPin GPIObits.GP0          //Capacitor charge/discharge
#define pwrVoltRef GPIObits.GP1         //supply to reference diode for ADC
#define sounder GPIObits.GP2            //To gate of Piezo sounder mosfet
#define extendPushButton GPIObits.GP3   //to enable longer open time
#define voltRefIn AN3                   //Monitor ref diode for ADC
#define reedSwitch GPIObits.GP5         //Reed switch input

volatile int sleepNumber = 4; //initialise to number of time periods
//possible when extend P/B pressed
int lowVoltTripCount = (((vRef/tripLowVolts)*1023)/4)-2; //multiply by 1023(10
                        //bit analog), result is shifted left 2 places (divide
                        //by 4) to use only 8 most significant bits
int volumeCalc;
int trial;
volatile int wokenFromDoorOpenSleep=0;
volatile int doorOpenSleepMode=0;
volatile int doorClosedSleep=0;

void main(void)
{
    __delay_ms(200);            //time to settle when first powered up 
    TRISIO = 0b00111000;        // 
    ANSEL = 0B01111000;         //<3:0>- AN3 only analog input <6:4>x11=int osc
    OSCCONbits.IRCF = 0b111;    //int osc 8MHz
    WPU = 0B00000000;           //no weak pull ups
    OPTION_REG = 0B00000000;
    CMCON0 = 7;                 //0111 means there is no comparator setup
    sounder = 1;
    pwrVoltRef = 0;             //pwr to ref diode off
    while (1)
    {
        while ((reedSwitch == doorClosed)) //when door is closed, contact opens,
        {
            IOC = 0B00100000;   //setup for reed (GPIO5) to wake from sleep
            TRISIO = 0b00111001; //set GP0 as input, use  for door open sleep
                                 //this is the ultra low power wake up pin
            GPIE = 1;           //enable GPIO change interrupt
            SLEEP();            //sleep & wait for reed to close (door open)
            GPIE = 0;           //wake up routine
            trial = GPIO;       //must read GPIO on wake to clear wake up
            GPIF = 0;
        }

/*door opens, reed closes, wake and do battery voltage test
wake up when door opens (contact closes)first do battery voltage test,
alarm if low, then perform as usual. The measurement of the battery voltage is
performed opposite to what ADC would normally do. Volts in on pin 3, AN3/GP4,
is derived from the voltage across a reference diode. The battery voltage on 
the chip's supply pins is used as the analog ref.
As the battery voltage falls the analog reference falls with it and as a
consequence the voltage measured across the diode appears to rise
--in reality the voltage does not change but the count increases as the ref
falls with the falling battery voltage. Three beeps on opening indicates
 low battery
*/
        pwrVoltRef = 1;     //turn on supply for battery test ref diode
        __delay_ms(10);     //time to settle
        ADCON0 = 0b00001111; //turn on ADC converter, do analog conversion
        __delay_ms(10);     //time for conversion
        pwrVoltRef = 0;     //turn off supply for battery test reference diode
        if (ADRESH >= lowVoltTripCount) //only need to look at most significant
                            //8 bits of ADC result
        { // if above value then do low battery beep
            alarm(sounderVol, threeBeeps, sounderFreq2); //vol(CCPR1L),number
                                                         //of pulses,freq(PR2)
        }
        //*************door open sleep************************************
        doorClosedSleep == 0;
        sleepNumber = 4; //initialise sleepNumber before door opens
        // If extend button pressed then counts up from 0
        while ((reedSwitch == doorOpen)) //do short sleep while door open
        {
            doorOpenSleep(); //sleep & wait for reed to open or time out
            asm("nop");
            if ((reedSwitch == doorOpen) && (sleepNumber >= 4))
            {
                alarm(sounderVol, twoBeeps, sounderFreq1); //vol(CCPR1L),
                                                    //number of pulses,freq(PR2)
            }
            sleepNumber = sleepNumber + 1; //increment sleep number
            if (sleepNumber > 7)
            {
                sleepNumber = 7;
            }// back to sleep
        }
    }
}

void doorOpenSleep()
{
    //using the ultra low power wake up function, using change on input
    //config for wake up delay: charge a capacitor, goto sleep and wake when
    //it's voltage drops to the point where input changes state
    TRISIO = 0b00111000;    //set GP0 (pin7)  as an output
    wakeUpPin = 1;          //turn on GP0 to charge capacitor
    __delay_ms(240);         //allow time to charge, increase for v large caps
    IOC = 0B00101001;       //look for change on inputs to wake up
    TRISIO = 0b00111001;    //set GP0 as an input
    trial = GPIO;           //after wake up must read GPIO
    GPIF = 0;
    GPIE = 1;               //peripheral inpterupt enable
    SLEEP();                //go to sleep and wait till capacitor discharges to
                            //wake up or door closes or extend button pressed
    trial = GPIO;           //after wake up must read GPIO
    GPIE = 0;
    trial = GPIO;
    GPIF = 0;
    __delay_ms(20);
    if (extendPushButton == 0) //if button pressed extend time by 4 cycles
        //before sounder goes off
    {
        sleepNumber = 0;    //set number of cycles to 0
    }

    if (reedSwitch == doorClosed)
    {
        doorClosedSleep = 1;
        sleepNumber = 4;
        TRISIO = 0b00111000; //set GP0 (pin7)  as an output
        wakeUpPin = 0;      //low to let capacitor drain
        __delay_ms(60);      //allow time to discharge capacitor
        TRISIO = 0b00111001; //set GP0 as an input to minimise load
    }
    return;
}