
/*
 * GP0/AN0 = output (high=enable dividers)
 * GP1/AN1 = analog input, battery voltage (threshold = 4V)
 * GP2/AN2 = analog input, thermistor voltage (threshold = 1V)
 * GP4/AN3 = analog input, upper threshold voltage set
 * GP5 = output, Mosfet gate drive
 */

#include <xc.h>

#pragma config FOSC=INTRCIO,WDTE=ON,MCLRE=ON,BOREN=OFF
#define _XTAL_FREQ 4000000

void delay_ms(unsigned char num_ms) {
    unsigned short i = num_ms * 4;
    while( i-- )
        __delay_us(250);
}

unsigned short read_ADC(unsigned char input) {
    ADCON0 = 0x81 | (input << 2);
    delay_ms(1);
    ADCON0 = 0x83 | (input << 2);
    while( ADCON0 & 2 )
        ;

    return ADRESL | (((unsigned short)ADRESH) << 8);
}

unsigned short time_on, time_off;

void interrupt t1int() {
    T1CON = 0;
    if( GP5 ) {
        GP5 = 0;
        TMR1H = time_off>>8;
        TMR1L = time_off;
    } else {
        GP5 = 1;
        TMR1H = time_on>>8;
        TMR1L = time_on;
    }
    T1CON = 1;
    // clear timer1 interrupt flag
    PIR1 = 0;
}

void main(void) {
    unsigned char cur_duty = 0, target_duty = 0, hysteresis = 0;
    unsigned short pwm_set_voltage;
    unsigned char pwm_delay_val = 10;

    GPIO = 0;
    CMCON = 7; // disable comparators
    TRISIO = 0b11011110; // GP0 and GP5 are outputs
    ANSEL = 0x1E;
    // enable timer1 overflow interrupt
    PIE1 = 1;
    PIR1 = 0;
    INTCON = 0xC0;

    pwm_set_voltage = read_ADC(2);
    if( pwm_set_voltage < 200 ) {
        unsigned char pwm_delay;
        if( pwm_set_voltage > 10 )
            pwm_delay = 2000 / pwm_set_voltage;
        else
            pwm_delay = 200;
        eeprom_write(0, pwm_delay);
        pwm_delay_val = pwm_delay;
    } else {
        pwm_delay_val = eeprom_read(0);
        if( pwm_delay_val > 200 )
            pwm_delay_val = 10;
    }

    while(1) {
        unsigned short batt = 0;

        asm("CLRWDT");
        GP0 = 1;
        batt = read_ADC(1);
        if( batt < 819 - hysteresis ) {
            hysteresis = 0;
            ADCON0 = 0;
            T1CON = 0;
            GP0 = 0;
            GP5 = 0;
            asm("SLEEP");
        } else {
            unsigned short thermistor, max_speed;

            hysteresis = 40; // 0.2V

            thermistor = read_ADC(2);
            max_speed = read_ADC(3);
            ADCON0 = 0;

            if( thermistor < 205 ) {
                target_duty = 0;
            } else if( max_speed <= 205 ) {
                target_duty = 100;
            } else {
                unsigned long duty = 10 + (thermistor - 205) * 73710UL / ((max_speed - 205) * (unsigned long)batt);
                if( duty > 100 )
                    target_duty = 100;
                else
                    target_duty = duty;
            }

            if( target_duty != cur_duty ) {
                if( target_duty > cur_duty ) {
                    if( cur_duty == 0 )
                        cur_duty = 10;
                    else if( cur_duty < 100 )
                        ++cur_duty;
                } else {
                    if( cur_duty == 10 )
                        cur_duty = 0;
                    else
                        --cur_duty;
                }
                if( cur_duty > 0 && cur_duty < 100 ) {
                    PIE1 = 0;
                    time_on = 0xFFFF - cur_duty * pwm_delay_val;
                    time_off = 0xFFFF - (100 - cur_duty) * pwm_delay_val;
                    PIE1 = 1;
                    T1CON = 1;
                } else {
                    T1CON = 0;
                    GP5 = (cur_duty ? 1 : 0);
                }
            }
            delay_ms(46);
        }
    }
}