  ' -----------------------------------------------------------
  ' Altimeter.bas
  ' A program for Micromite to use the Elecrow GY-68 Barometer
  ' module (based on the Bosch BMP180 chip) and the DHT22/AM2302
  ' Temperature & Humidity Sensor, as an Altimer for ultra-light
  ' aircraft and hang gliders & also as a mini weather station.
  '
  ' Written by Jim Rowe for Silicon Chip.
  ' Last revision 27/11/2017 at 1:45 pm
  '
  ' Notes:
  ' 1. Communication with the BMP180 chip is via the Micromite's
  ' I2C port, which uses pin 17 for SCL and pin 18 for SDA.
  ' 2. Although the chip's I2C address is specified as EFh for
  ' reading and EEh for writing, MMBasic's I2C function expects a
  ' 7-bit I2C address since it adds the lsb read/write bit automatically.
  ' So in this program we use 77h as the chip's I2C address.
  ' 3. The data output of the DHT22 temp/humidity chip is connected to
  ' pin 21 of the Micromite, which is also fitted with a 4.7k pullup
  ' resistor.
  ' 4. To work out the altitude above MSL (mean sea level), we need
  ' to use the International Barometric Formula, which assumes that the
  ' air pressure at MSL (PaMSL) is 1013.25hPa. However when the actual
  ' barometric pressure is higher than 1013.25hPa, this can result in
  ' negative altitude readings when you are at very low altitudes
  ' (i.e., near ground level). To avoid this problem, the program allows
  ' the user to change the reference air pressure for the altitude
  ' calculation from PaMSL to the current barometric pressure 'on the
  ' ground' in the area concerned (found by making a QNH request). This
  ' QNH pressure can then be keyed in to become the alternative altitude
  ' reference.
  ' 5. The current altitude reference (MSL or QNH) is indicated on the
  ' display in both modes (Weather Station or Altimeter).
  ' 6. The program also allows the user to change the altitude units
  ' from metres to feet, and vice-versa. The units in current use are
  ' also indicated on the display in both modes.
  ' 7. Whenever the display mode, altitude units or altitude reference
  ' is changed, the current values are saved in non-volatile memory.
  ' ----------------------------------------------------------
  
  OPTION AUTORUN ON
  OPTION EXPLICIT
  
  DIM AS INTEGER AC1 = 0    ' BMP-180 cal parameter AC1 (16 bits)
  DIM AS INTEGER AC2 = 0    ' BMP-180 cal parameter AC2 (16 bits)
  DIM AS INTEGER AC3 = 0    ' BMP-180 cal parameter AC3 (16 bits)
  DIM AS INTEGER AC4 = 0    ' BMP-180 cal parameter AC4 (16 bits)
  DIM AS INTEGER AC5 = 0    ' BMP-180 cal parameter AC5 (16 bits)
  DIM AS INTEGER AC6 = 0    ' BMP-180 cal parameter AC6 (16 bits)
  DIM AS INTEGER B1 = 0     ' BMP-180 cal parameter B1 (16 bits)
  DIM AS INTEGER B2 = 0     ' BMP-180 cal parameter B2 (16 bits)
  DIM AS INTEGER MB = 0     ' BMP-180 cal parameter MB (16 bits)
  DIM AS INTEGER MC = 0     ' BMP-180 cal parameter MC (16 bits)
  DIM AS INTEGER MD = 0     ' BMP-180 cal parameter MD (16 bits)
  DIM AS INTEGER UT = 0     ' uncompensated temperature reading
  DIM AS INTEGER UP = 0     ' uncompensated pressure reading
  DIM AS INTEGER DScrn = 0  ' Default display screen index (0=WS, 1=Altim)
  DIM AS INTEGEr DAunit = 0 ' Default altitude units index (0=m, 1=ft)
  DIM AS INTEGER DAref = 0  ' Default altitude ref index (0=MSL, 1=arbGL)
  DIM AS INTEGER IntFlg = 0 ' flag to show if screen has been touched
  
  DIM AS FLOAT Temp         ' average temperature for display
  DIM AS FLOAT Temp1        ' compensated temperature reading (BMP180)
  DIM AS FLOAT Temp2        ' temperature reading from DHT22
  DIM AS FLOAT RelH         ' relative humidity reading from DHT22
  DIM AS FLOAT Pres         ' compensated pressure reading
  DIM AS FLOAT ArefPres = 1013.25 ' ref pressure for calculating altitude
  DIM AS FLOAT Alt          ' altitude above MSL (or QNH)
  DIM AS FLOAT X1, X2, X3, B3, B4, B5, B6, B7   ' working variables
 
  DIM AS STRING AltUnit$ = "metres above " ' string for saving altim units
  DIM AS STRING AltRef$ = "MSL"       ' string for saving altim reference
  DIM AS STRING AltString$            ' composite string used for display
  DIM AS STRING KPStr$                ' char returned by touchscreen keypad
  DIM AS STRING KPInStr$              ' string returned from keypad
  DIM AS string QNHString$            ' string used to input QNH pressure
   
  Const DBlue = RGB(0,0,128)
  CONST Bone = RGB(255,255,192)
  CONST White = RGB(WHITE)
  CONST Black = RGB(BLACK)
  CONST Red = RGB(RED)
  CONST Green = RGB(GREEN)
  CONST Pink = RGB(255,210,210)
  
  PIN(17) = 0             ' set pin 17 to low and
  SETPIN 17, DOUT         ' declare it a digital output (SCL)
  SETPIN 15, INTL, TchInt ' call TchInt to set flag when screen touched
  
  DIM INTEGER CData(21)   ' one-dim array to save cal data bytes from EEPROM
  DIM INTEGER TByte(1)    ' one dim array to save raw UT data bytes
  DIM INTEGER PByte(2)    ' one-dim array to save raw UP data bytes
  
  I2C OPEN 100, 200       ' enable the Micromite I2C module in master mode
  VAR RESTORE             ' retrieve default values from non-volatile memory
  ' ----------------------------------------------------------------------
  ' first go fetch the BMP180 chip's calibration data
    I2C WRITE &H77,0,1,&HAA     ' send first EEPROM reg address to be read
    I2C READ &H77,0,22,CData()  ' then read 22 bytes & save in CData() array

  ' The cal parameter bytes should now have been retrieved from the EEPROM
  ' registers of the BMP180, but we now need to assemble each pair and then
  ' mask them back to 16 bits. For all except AC4, AC5 and AC6 we also need
  ' to convert them into signed integers if they are >= 32768
  AC1 = ((CData(0) << 8) OR CData(1)) AND &H000000000000FFFF
  IF AC1 >= 32768 THEN AC1 = (AC1 - 65535)
  AC2 = ((CData(2) << 8) OR CData(3)) AND &H000000000000FFFF
  IF AC2 >= 32768 THEN AC2 = (AC2 - 65535)
  AC3 = ((CData(4) << 8) OR CData(5)) AND &H000000000000FFFF
  IF AC3 >= 32768 THEN AC3 = (AC3 - 65535)
  AC4 = ((CData(6) << 8) OR CData(7)) AND &H000000000000FFFF
  AC5 = ((CData(8) << 8) OR CData(9)) AND &H000000000000FFFF
  AC6 = ((CData(10) << 8) OR CData(11)) AND &H000000000000FFFF
  B1 = ((CData(12) << 8) OR CData(13)) AND &H000000000000FFFF
  IF B1 >= 32768 THEN B1 = (B1 - 65535)
  B2 = ((CData(14) << 8) OR CData(15)) AND &H000000000000FFFF
  IF B2 >= 32768 THEN B2 = (B2 - 65535)
  MB = ((CData(16) << 8) OR CData(17)) AND &H000000000000FFFF
  IF MB >= 32768 THEN MB = (MB - 65535)
  MC = ((CData(18) << 8) OR CData(19)) AND &H000000000000FFFF
  IF MC >= 32768 THEN MC = (MC - 65535)
  MD = ((CData(20) << 8) OR CData(21)) AND &H000000000000FFFF
  IF MD >= 32768 THEN MD = (MD - 65535)
  ' ----------------------------------------------------------------
  ' start of main part of program
  ' first show the default screen from params saved in NV memory
  ' (DScrn,DAunit,DAref,ArefPres,AltUnit$,AltRef$)
  AltString$ = AltUnit$ + AltRef$   ' restore AltString$
  IF DScrn = 0 THEN
    ShowWStnScrn    ' weather station display
  ELSE
    ShowAltmScrn    ' altimeter display
  END IF
  ' main program loop starts here
DO
  GoReadTemp          ' first get an uncomp temperature reading 
  GoReadPressure      ' also the uncompensated pressure (BMP-180)
  DoTheMaths          ' then do the maths to get Temp1 and Pres
  GoReadTandRH        ' then get Temp2 and RelH from DHT22
  GetAvTemp           ' work out average Temp from Temp1 and Temp2
  DispRes             ' and display the results on screen
  IF (IntFlg = 1 AND TOUCH(Y)>=MM.VRes*5/8) THEN MakeChanges  
  PAUSE 1000          ' set to loop about once every second
LOOP
END                   ' end of main program loop, subroutines follow
  
  ' *****************************************************************
  ' subroutine to set IntFlag when screen is touched
SUB TchInt
  IntFlg = 1
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display the weather station screen
SUB ShowWStnScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, "Air Temp: ", LM, 1,2, Green, Black
  TEXT MM.HRes/2, MM.VRes/8, "00.0degC", LM, 1,2, White, Black
  TEXT 10, MM.VRes/4, "RelHumid: ", LM, 1,2, Green, Black
  TEXT MM.HRes/2, MM.VRes/4, "00%", LM, 1,2, White, Black  
  TEXT 10, MM.VRes*3/8, "Air Pres:" , LM, 1,2, Green, Black
  TEXT MM.HRes/2, MM.VRes*3/8, "0000.0hPa", LM, 1,2, White, Black
  TEXT 10, MM.VRes/2, "Altitude: ", LM, 1,2, Green, Black
  TEXT MM.HRes/2, MM.VRes/2, "0000", LM, 1,2, White, Black
  TEXT MM.HRes/2, MM.VRes*5/8, AltString$, CM, 1,2, Green, Black
  RBOX 20, MM.VRes*11/16, MM.HRes*7/8, MM.VRes/4, 10, Red, Pink
  TEXT MM.HRes/2, MM.VRes*3/4, "TOUCH TO CHANGE", CM, 1,2, Black, Pink
  TEXT MM.HRes/2, MM.VRes*7/8, "MODE OR UNITS", CM, 1,2, Black, Pink
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to display the altimeter screen
SUB ShowAltmScrn
  CLS Black
  RBOX 0,0, MM.HRes-2, MM.VRes-2, 5, RGB(Cyan), Black
  TEXT 10, MM.VRes/8, "ALTITUDE:", LM, 1,2, Green, Black
  TEXT MM.HRes/2, MM.VRes*3/8, "0000", CM, 1,6, White,Black
  TEXT MM.HRes/2, MM.VRes*5/8, AltString$, CM, 1,2, Green,Black 
  RBOX 20, MM.VRes*11/16, MM.HRes*7/8, MM.VRes/4, 10, Red, White
  TEXT MM.HRes/2, MM.VRes*3/4, "TOUCH TO CHANGE", CM, 1,2, Black, White
  TEXT MM.HRes/2, MM.VRes*7/8, "MODE OR UNITS", CM, 1,2, Black, White  
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to show change screen & respond by making the changes
SUB MakeChanges
  IntFlg = 0      ' first clear the int flag ready for a fresh touch
  ' then show the 'make changes' screen
  CLS Black
  ' show top pair of buttons
  RBOX 5, 0, MM.HRes/2 -10, MM.VRes/4, 10, Red, Pink
  RBOX MM.HRes/2, 0, MM.HRes/2 -5, MM.VRes/4, 10, Red, Bone
  TEXT MM.HRes/4, MM.VRes/16, "WEATHER", CM, 1,2, Black, Pink
  TEXT MM.HRes/4, MM.VRes*3/16, "STN MODE", CM, 1,2, Black, Pink
  TEXT MM.HRes*3/4, MM.VRes/16, "ALTIMETER", CM, 1,2, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*3/16, "MODE", CM, 1,2, Black, Bone
  ' show second pair of buttons
  RBOX 5, MM.VRes/4, MM.HRes/2 -10, MM.VRes/4, 10, Red, Green
  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/2 -5, MM.VRes/4, 10, Red, Pink
  TEXT MM.HRes/4, MM.VRes*3/8, "METRES", CM, 1,2, Black, Green
  TEXT MM.HRes*3/4, MM.VRes*3/8, "FEET", CM, 1,2, Black, Pink
  ' show third pair of buttons
  RBOX 5, MM.VRes/2, MM.HRes/2 -10, MM.VRes/4, 10, Red, Bone
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/2 -5, MM.VRes/4, 10, Red, Bone
  TEXT MM.HRes/4, MM.VRes*9/16, "MSL", CM, 1,2, Black, Bone
  TEXT MM.HRes/4, MM.VRes*11/16, "REFERENCE", CM, 1,2, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*9/16, "INPUT QNH", CM, 1,2, Black, Bone
  TEXT MM.HRes*3/4, MM.VRes*11/16, "REFERENCE", CM,1,2, Black, Bone
  ' show bottom "Exit without any changes" button
  RBOX 5, MM.VRes*3/4, MM.HRes*31/32, MM.VRes/4, 10, Red, White
  TEXT MM.HRes/2, MM.VRes*13/16, "EXIT WITHOUT", CM, 1,2, Red, White
  TEXT MM.HRes/2, MM.VRes*15/16, "ANY CHANGES", CM, 1,2, Red, White  
    
FM2: IF TOUCH(X) = -1 THEN GOTO FM2   ' keep looping until user touches screen
  IF TOUCH(Y)<MM.VRes/4 THEN
    IF TOUCH(X)<MM.HRes/2 THEN
        DScrn = 0 
    ELSE
        DScrn = 1
    END IF
  END IF
  IF (TOUCH(Y)>MM.VRes/4) AND (TOUCH(Y)<MM.VRes/2) THEN
      IF TOUCH(X)<MM.HRes/2 THEN
        DAunit = 0                  ' set DAunit to zero and
        AltUnit$ = "metres above "  ' change unit string to metres
      ELSE
        DAunit = 1                  ' change DAunit to 1 and
        AltUnit$ = "feet above "    ' change unit string to feet
      END IF
  END IF
  IF (TOUCH(Y)>MM.VRes/2) AND (TOUCH(Y)<MM.VRes*3/4) THEN
      IF TOUCH(X)<MM.HRes/2) THEN
        DAref = 0              ' change altim reference to MSL
        AltRef$ = "MSL"
        ArefPres = 1013.25     ' change ArefPres to MSL value
      ELSE
        DAref = 1              ' change altim ref to QNH
        AltRef$ = "QNH"
        FeedinQNH              ' and go get QNH pressure
      END IF
  END IF
  IF TOUCH(Y)>MM.VRes*3/4 THEN GOTO Skip ' touched exit button, so do so
  VAR SAVE DScrn,DAunit,DAref,ArefPres,AltUnit$,AltRef$  ' save params in NV memory
  AltString$ = AltUnit$ + AltRef$ 
Skip: IF DScrn = 0 THEN
        ShowWStnScrn
      ELSE
        ShowAltmScrn
      END IF
  IntFlg = 0    ' clear int flag before leaving
END SUB  
  ' -----------------------------------------------------------------
  ' subroutine to read uncompensated temperature (UT) from BMP180
SUB GoReadTemp
  I2C WRITE &H77,0,2,&HF4,&H2E  ' send 2Eh to control register F4
  PAUSE 5                       ' then wait 5ms for measurement
  I2C WRITE &H77,0,1,&HF6       ' send first address to read (F6h)
  I2C READ &H77,0,2, TByte()    ' then read the two UT bytes
  ' then glue them together to get the UT reading
  UT = ((TByte(0) << 8) OR TByte(1)) AND &H000000000000FFFF
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to read uncompensated pressure (UP) from BMP180
SUB GoReadPressure
  I2C WRITE &H77,0,2,&HF4,&H34  ' send 34h to control register F4
  PAUSE 10                      ' then wait 10ms for measurement
  I2C WRITE &H77,0,1, &HF6      ' send first address to read (F6h)
  I2C READ &H77,0,3, PByte()    ' then read the three UP bytes
  ' then glue them together to get the UP reading (oss = 00)
  UP = (((PByte(0) << 16) OR (PByte(1) << 8) OR PByte(2)) >> 8) AND &H0000000000FFFFFF
END SUB
  ' -----------------------------------------------------------------
  ' subroutine to 'do the maths' for calculating compensated Temp1 and Pres
  ' based on expressions given in Bosch's BMP180 datasheet on page 15.
SUB DoTheMaths
  X1 = ((UT - AC6) * AC5)/32768
  IF X1 + MD <> 0 THEN X2 = (MC * 2048)/(X1 + MD) ELSE X2 = 0
  B5 = X1 + X2
  Temp1 = (B5 + 8)/16        ' this should be the comp temp in 0.1 deg C
  Temp1 = (Temp1/10)          ' so now it should be in deg C.
  '
  ' now to crunch the data to get the pressure (Pres)
  B6 = B5 - 4000
  X1 = (B2 * (B6 * B6/4096))/2048
  X2 = (AC2 * B6)/2048
  X3 = X1 + X2
  B3 = (((AC1 * 4) + X3) + 2)/4
  X1 = (AC3*B6)/8192
  X2 = (B1 * (B6 * B6/4096))/65536
  X3 = ((X1 + X2) + 2)/4
  B4 = AC4 * (X3 + 32768)/32768
  B7 = (UP - B3) * 50000
  IF B4 = 0 THEN
    Pres = 0
  ELSEIF B7 < &H80000000 THEN
    Pres = (B7 * 2)/B4
  ELSE
    Pres = (B7/B4) * 2
  END IF
  X1 = (Pres/256) * (Pres/256)
  X1 = (X1 * 3038)/65536
  X2 = (-7357*Pres)/65536
  Pres = Pres + (X1 + X2 + 3791)/16     ' this gives pressure in Pascals,
  Pres = (Pres * 0.01)                  ' so now convert to mBar (= hPa)
  Alt = 44330*(1 - (Pres/ArefPres)^0.190295)  ' then work out altitude in metres
  IF DAunit = 1 THEN Alt = Alt * 3.28   ' convert to feet if DAunit=1 (units = ft)
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to display measurements on the selected screen
SUB DispRes
  AltString$ = AltUnit$ + AltRef$   ' make up AltString$
  IF DScrn = 0 THEn   ' if the WSScrn is in use, use it
    TEXT MM.HRes/2, MM.VRes/8,(STR$(Temp,2,1) + "degC"), LM, 1,2, White, Black
    TEXT MM.HRes/2, MM.VRes/4,(STR$(RelH,2,0)+ "%"), LM, 1,2, White, Black
    TEXT MM.HRes/2, MM.VRes*3/8,(STR$(Pres,4,1) + "hPa"), LM, 1,2, White, Black    
    TEXT MM.HRes/2, MM.VRes/2,STR$(Alt,5,0), LM, 1,2, White, Black
    TEXT MM.HRes/2, MM.VRes*5/8, AltString$, CM, 1,2, Green, Black    
  ELSE      ' otherwise (DScrn = 1) use the Altimeter screen
    TEXT MM.HRes/2, MM.VRes*3/8,STR$(Alt,5,0), CM, 1,6, White,Black
    TEXT MM.HRes/2, MM.VRes*5/8, AltString$, CM, 1,2, Green, Black  
  END IF
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to find the temperature Temp2 (degrees C) and relative
  ' humidity (%) from the DHT22/AM2302 module.
SUB GoReadTandRH
  HUMID 21, Temp2, RelH
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to find the average of Temp1 and Temp2
SUB GetAvTemp
  Temp = (Temp1 + Temp2)/2.00
END SUB
  ' -------------------------------------------------------------------
  ' subroutine to allow user to key/touch in current QNH pressure as 
  ' ground reference pressure for altitude calaculation
SUB FeedinQNH
  KPadDraw    ' first draw keypad
  KPInStr$ = SPACE$(6)   ' now clear the input string
  DO
    InCharFrmKP   ' then go get a character/string from the keypad (> KPStr$)
    SELECT CASE KPStr$    ' and analyse it
      CASE "OK"    ' OK touched
        IF KPInStr$ = SPACE$(6) THEN EXIT DO  ' if KPInStr$ null string, exit now
        QNHString$ = LEFT$(KPInStr$, 6)  ' otherwise make QNHString$ = KPInStr$
        TEXT MM.HRes/16, MM.VRes*7/16, QNHString$, LM, 1,2, Red, White  ' show it
        ArefPres = VAL(QNHString$)   ' then update ArefPres
        VAR SAVE ArefPres         ' save it in flash RAM for non-volatility
        EXIT DO  ' then leave
      CASE "0" TO "9", "-", "."
        KPInStr$ = RIGHT$(KPInStr$, 5) ' a numeral, so shorten KPInStr$
        KPInStr$ = KPInStr$ + KPStr$  ' then add the new digit to it
        TEXT MM.HRes/16, MM.VRes*7/16, KPInStr$, LM, 1,2, Red, White
      CASE CHR$(08)   ' must be a backspace
        IF LEN(KPInStr$) <> 0 THEN  ' if LEn(KPInStr$) not 0, then shorten by 1
        KPInStr$ = Left$(KPInStr$, LEN(KPInStr$) -1)
        TEXT MM.HRes/16, MM.VRes*7/16, KPInStr$ + " ", LM, 1,2, Red, White '& re-show
        ELSE    ' if LEN(KPInStr$ = 0) then just restore to 6 spaces
        KPInstr$ = SPACE$(6)
        END IF
    END SELECT
    PAUSE 250
  LOOP    ' keep looping until OK button is pressed
  PAUSE 250 

END SUB
  ' -------------------------------------------------------------------
  ' subroutine to draw keypad on right, OK & backspace buttons lower L
SUB KPadDraw
  CLS Black
  ' keypad boxes
  RBOX MM.HRes/2, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, 0, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes/2, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  
  RBOX MM.HRes/2, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*4/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  RBOX MM.HRes*5/6, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4-4, 10, Black, Bone
  ' boxes at lower left for backspace and OK buttons, plus legends
  RBOX 0, MM.VRes*3/4, MM.HRes/6-4, MM.VRes/4, 10, Black, Bone
  RBOX MM.HRes/6, MM.VRes*3/4, MM.HRes/3-4, MM.VRes/4, 10, Black, Bone
  TEXT MM.HRes/12, MM.VRes*7/8, "<-", CM, 1, 2, Black, Bone
  TEXT MM.HRes/3, MM.VRes*7/8, "OK", CM, 1, 3, Red, Bone
  ' legends for the keypad buttons
  TEXT MM.HRes*7/12, MM.VRes/8, "7", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes/8, "8", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes/8, "9", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*3/8, "4", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*3/8, "5", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*3/8, "6", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*5/8, "1", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*5/8, "2", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*5/8, "3", CM, 1, 3, Black, Bone
  
  TEXT MM.HRes*7/12, MM.VRes*7/8, "-", CM, 1, 3, Black, Bone
  TEXT MM.HRes*9/12, MM.VRes*7/8, "0", CM, 1, 3, Black, Bone
  TEXT MM.HRes*11/12, MM.VRes*7/8, ".", CM, 1, 3, Black, Bone
END SUB
' -----------------------------------------------------------------------  
SUB InCharFrmKP   ' sub to get a char from the touch keypad
ICFK2: IF TOUCH(X) = -1 THEN GOTO ICFK2 'wait until keypad touched
  ' also loop back if no valid button touched
  IF TOUCH(X) < MM.HRes/2 AND TOUCH(Y) < MM.VRes*3/4 THEN GOTO ICFK2
  SELECT CASE TOUCH(X)  ' now decide which button was touched
    CASE 0 TO MM.HRes/6-1
      KPStr$ = CHR$(08)  ' must have been backspace button
    CASE MM.HRes/6 TO MM.HRes/2-1
      KPStr$ = "OK"      ' must have been OK button
    CASE MM.HRes/2 TO MM.HRes*2/3-1  'first column of keypad itself
      SELECT CASE TOUCH(Y)  ' must be first column
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "7"   ' must be the 7 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "4"   ' must be the 4 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "1"   ' must be the 1 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "-"   ' must be the hyphen button
      END SELECT
    CASE MM.HRes*2/3 TO MM.HRes*5/6-1 'centre column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "8"   ' must be the 8 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "5"   ' must be the 5 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "2"   ' must be the 2 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "0"   ' must be the 0 button
      END SELECT
    CASE MM.HRes*5/6 TO MM.HRes   ' last column of keypad
      SELECT CASE TOUCH(Y)
        CASE 0 TO MM.VRes/4 -1  ' if it's in the first row
          KPStr$ = "9"   ' must be the 9 button
        CASE MM.VRes/4 TO MM.VRes/2 -1  'try second row
          KPStr$ = "6"   ' must be the 6 button
        CASE MM.VRes/2 TO MM.Vres*3/4 -1  'try third row
          KPStr$ = "3"   ' must be the 3 button
        CASE MM.VRes*3/4 TO MM.VRes   ' try fourth row
          KPStr$ = "."   ' must be the DP button
      END SELECT
  END SELECT
END SUB
  ' -------------------------------------------------------------------
CSub HUMID integer, float, float
0000002B 27BDFFF8 AFBF0004 00852023 03E42021 ACC40000 8FBF0004 03E00008
27BD0008 27BDFFE0 AFBF001C 00002021 3C059D00 24A50040 27A60010 0411FFF1
00000000 8FA30010 3C029D00 8C4200BC 3C049D00 24840414 0040F809 00832021
8FBF001C 03E00008 27BD0020 00041080 000419C0 00621823 00642021 000418C0
3C029D00 8C420000 3C047735 34849400 8C420000 0082001B 004001F4 00002012
0064001B 008001F4 03E00008 00001012
27BDFFB8 AFBF0044 AFBE0040 AFB7003C AFB60038 AFB50034 AFB40030 AFB3002C
AFB20028 AFB10024 AFB00020 00808821 00A0A821 3C029D00 8C420000 8C430000
3C020098 34429680 0062102B 10400003 00C0B021 0411FFC7 00000000 3C109D00
8E02001C 8E240000 0040F809 24050006 8E020010 8E240000 24050009 0040F809
00003021 8E02001C 8E240000 0040F809 2405000E 8E02001C 8E240000 0040F809
24050005 8E020004 0040F809 240403E8 8E02001C 8E240000 0040F809 24050006
8E020080 240403E8 0040F809 00002821 AEC20000 AEA20000 0411FFB8 24040190
00409821 0000A021 40944800 00008021 10000005 3C129D00 40104800 0270102B
1440008E 8FBF0044 8E420020 0040F809 8E240000 1440FFF8 00000000 10000007
3C129D00 40104800 0270102B 50400004 8E420020 10000081 8FBF0044 8E420020
0040F809 8E240000 1040FFF6 00000000 10000007 3C129D00 40104800 0270102B
50400004 8E420020 10000074 8FBF0044 8E420020 0040F809 8E240000 1440FFF6
00001021 00001821 AFA20010 AFA30014 0000B821 10000005 3C129D00 40104800
0270102B 14400065 8FBF0044 8E420020 0040F809 8E240000 1040FFF8 00000000
40944800 10000006 8E420020 40104800 0270102B 14400059 8FBF0044 8E420020
0040F809 8E240000 1440FFF8 8FA30010 000317C2 8FA50014 0005F040 005EF025
00031040 AFA20018 0411FF6C 24040032 0050102B 8FA30018 00431025 AFA20010
26F70001 24050028 16E5FFE0 AFBE0014 001E1600 8FA50010 00052202 00442025
001E1200 00058E02 00518825 7CA33C00 308200FF 00621821 322200FF 00621821
93A50014 30A200FF 00621021 8FA30010 00431026 304200FF 1440002F 001EA603
3C109D00 8E130064 8E020080 30847FFF 0040F809 00002821 00409021 8E020080
2404000A 0040F809 00002821 02402021 0260F809 00402821 00408021 AEA20000
3C020080 8FA50010 00451024 5040000C 3C109D00 3C029D00 8C520058 8C420080
2404FFFF 0040F809 2405FFFF 02002021 0240F809 00402821 AEA20000 3C109D00
8E120064 8E020080 02202021 0040F809 02802821 00408821 8E020080 2404000A
0040F809 00002821 02202021 0240F809 00402821 AEC20000 8FBF0044 8FBE0040
8FB7003C 8FB60038 8FB50034 8FB40030 8FB3002C 8FB20028 8FB10024 8FB00020
03E00008 27BD0048 20555043 65657073 6F742064 6F6C206F 000A0D77
End CSub
' ---------------------------------------------------------------------
