#include <LiquidCrystal.h>
LiquidCrystal lcd(8,9,4,5,6,7);

#include <TimerOne.h>
#define STROBE_OUT_PIN 12
#define LOW_PIN 11
#define INV_DUTY 10

#define RPM_LOWER 60
#define RPM_UPPER 9000

#define DISP_UPDATE 200
#define SAMP_UPDATE 10
#define SAMP_THRESHOLD 3
#define SMOOTHING 2

enum lcdKeys_t {
  LCD_NO_KEY,
  LCD_SELECT,
  LCD_LEFT,
  LCD_DOWN,
  LCD_UP,
  LCD_RIGHT  
};

bool useRPM=true;
bool strobeOn=false;
unsigned int rpmOut=1000;
unsigned int rpmStep=10;
unsigned long lastDisp=0;
unsigned long lastSamp=0;
unsigned int rpmIn=0;
int rpmStepIndex=0;
#define RPM_STEP_COUNT 6
unsigned int rpmSteps[RPM_STEP_COUNT]={10,20,50,100,200,500};

void setup() {
  lcd.begin(16,2);
  Serial.begin(115200);
  pinMode(STROBE_OUT_PIN,OUTPUT);
  pinMode(LOW_PIN,OUTPUT);
  digitalWrite(LOW_PIN,LOW);  //fake ground
  digitalWrite(STROBE_OUT_PIN,LOW);
  Timer1.attachInterrupt(strobe);
  setRPM(0);
}

void loop() {
  if((millis()-lastDisp)>DISP_UPDATE){
    lastDisp=lastDisp+DISP_UPDATE;
    doDisp();   //handle key capture and display update
  }
  if((millis()-lastSamp)>SAMP_UPDATE){
    lastSamp=lastSamp+SAMP_UPDATE;
    doSamp();   //handle key capture and display update
  }
}

void doSamp(){
  static bool state=false;    //for edge detection
  static unsigned int dcLevel=0;       //smoothed samples for baseline
  static unsigned long zeroEdge=0;    //for multiple edge counts (higher rates)
  static unsigned long thisEdge=0;    
  unsigned long temp;
  static int edgeCount=0;   
  unsigned int sample;
  sample=analogRead(A1);
  if((state==false)&&(sample>(dcLevel+SAMP_THRESHOLD))){  //rising edge
    thisEdge=millis();    
    state=true;
    edgeCount=edgeCount+1;
    if((thisEdge-zeroEdge)>300){   //count multiple edges if possible
      temp=(60000UL*edgeCount)/(thisEdge-zeroEdge);
      rpmIn=temp;
      //Serial.println(edgeCount);
      edgeCount=0;        //reset count
      zeroEdge=thisEdge; //and reset time
    }
  }
  if((state==true)&&(sample<(dcLevel-SAMP_THRESHOLD))){  //falling edge, to reset 
    state=false;
  }
  dcLevel=(dcLevel*(SMOOTHING-1)+sample)/SMOOTHING;
}

void doDisp(void){
  static lcdKeys_t lastk=LCD_NO_KEY;
  lcdKeys_t k=LCD_NO_KEY;
  k=getKey(analogRead(0));
  analogRead(A1); //avoid noise on photodiode reads
  if(lastk==LCD_NO_KEY){  //possible keydown
    switch(k){
      case LCD_SELECT:  //toggle RPM/Hz
        if(useRPM){
          useRPM=false;
        }else{
          useRPM=true;
        }
        break;
      case LCD_LEFT:
        if(strobeOn){
          strobeOn=false;
          setRPM(0);
        }else{
          strobeOn=true;
          setRPM(rpmOut);
        }
        break;
      case LCD_RIGHT:
        rpmStepIndex=rpmStepIndex+1;
        if(rpmStepIndex>=RPM_STEP_COUNT){rpmStepIndex=0;}
        rpmStep=rpmSteps[rpmStepIndex];
        break;
      default: break;  //ignore others
    }
  }
  lastk=k;
  if(k==LCD_UP){
    rpmOut=rpmOut+rpmStep;
    if(rpmOut>RPM_UPPER){
      rpmOut=RPM_UPPER;
    }
    if(strobeOn){setRPM(rpmOut);}
  }
  if(k==LCD_DOWN){
    if(rpmOut>=(RPM_LOWER+rpmStep)){ //avoid wrapping on unsigned
      rpmOut=rpmOut-rpmStep;
      if(strobeOn){setRPM(rpmOut);}
    }
  }
  if(k==LCD_SELECT){    //help page
    lcd.setCursor(0, 0);
    lcd.print("S=Units U/D=Rate");
    lcd.setCursor(0, 1);
    lcd.print("L=Strobe R=Steps");
  }else{                  //normal display
    lcd.setCursor(0, 0);
    showSpeed(&lcd,rpmOut);
    if(strobeOn){
      lcd.print(" ON  ");
    }else{
      lcd.print(" OFF ");
    }
    if(rpmStep<10){lcd.write(' ');}
    if(rpmStep<100){lcd.write(' ');}
    lcd.write('x');
    lcd.print(rpmStep);
    lcd.setCursor(0, 1);
    lcd.print("^OUT  IN>");
    showSpeed(&lcd,rpmIn);
  }
}


lcdKeys_t getKey(int a){ //convert ADC value to key
  if(a>831){return LCD_NO_KEY;}
  if(a>523){return LCD_SELECT;}
  if(a>331){return LCD_LEFT;}
  if(a>177){return LCD_DOWN;}
  if(a>50){return LCD_UP;}
  return LCD_RIGHT;
}

void showSpeed(Print* p, unsigned int r){ //takes value in RPM and converts depending on useRPM
  bool lzb=true;
  char d;
  if(useRPM){   //xxxxRPM
    if(r>9999){
      p->print("-HI-");
    }else{
      d=r/1000;
      r=r%1000;
      if(d>0){lzb=false;}
      if(lzb){p->write(' ');}else{p->write(d+'0');}
      d=r/100;
      r=r%100;
      if(d>0){lzb=false;}
      if(lzb){p->write(' ');}else{p->write(d+'0');}
      d=r/10;
      r=r%10;
      if(d>0){lzb=false;}
      if(lzb){p->write(' ');}else{p->write(d+'0');}
      p->write(r+'0');
    }
    p->print("RPM");
  }else{        //xxx.xHz
    r=r/6;      // ratio is 60, but one decimal place
    if(r>9999){
      p->print(" -HI-");
    }else{
      d=r/1000;
      r=r%1000;
      if(d>0){lzb=false;}
      if(lzb){p->write(' ');}else{p->write(d+'0');}
      d=r/100;
      r=r%100;
      if(d>0){lzb=false;}
      if(lzb){p->write(' ');}else{p->write(d+'0');}
      d=r/10;
      r=r%10;
      p->write(d+'0');
      p->write('.');
      p->write(r+'0');
    }
    p->print("Hz");

  }

}

void strobe(void){
  static char c=0;
  c++;
  if(c>(INV_DUTY-1)){
    c=0;
    digitalWrite(STROBE_OUT_PIN,HIGH);
  }else{
    digitalWrite(STROBE_OUT_PIN,LOW);
  }
}

void setRPM(unsigned int r){
  long p;
  if(r==0){
    Timer1.stop();
    digitalWrite(STROBE_OUT_PIN,LOW); //force off
  }else{
    p=(60000000UL/INV_DUTY)/r; //60s x 1000000us/s / 10 interrupts
    Timer1.initialize(p);
  }
}
