/*  
 * USB CDC-ACM Demo
 *
 * This file may be used by anyone for any purpose and may be used as a
 * starting point making your own application using M-Stack.
 *
 * It is worth noting that M-Stack itself is not under the same license as
 * this file.
 *
 * M-Stack is distributed in the hope that it will be useful, but WITHOUT
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or
 * FITNESS FOR A PARTICULAR PURPOSE.  For details, see sections 7, 8, and 9
 * of the Apache License, version 2.0 which apply to this file.  If you have
 * purchased a commercial license for this software from Signal 11 Software,
 * your commerical license superceeds the information in this header.
 *
 * Alan Ott
 * Signal 11 Software
 * 2014-05-12
 
 Modified Tim Blythman 2024 to behave as USB-serial bridge with DTR
 //DTR was RC3 in revA PCB
 RX     RC5
 TX     RC4
 DTR    RA5
 RX_LED RC0
 TX_LED RC1
 
 //using MPLAB X 6.15, XC8 1.45 and DFP 1.1.53   
 */

#define _XTAL_FREQ 16000000
#include "usb.h"
#include <xc.h>
#include <string.h>
#include "usb_config.h"
#include "usb_ch9.h"
#include "usb_cdc.h"
#include "hardware.h"
#include "io.h"

#ifdef MULTI_CLASS_DEVICE
static uint8_t cdc_interfaces[] = { 0 };
#endif

#define IN_EP_LIMIT 32

bool dtrState;
char lcChange=0;
char brkReq=0;      //a break has been requested by the host

static struct cdc_line_coding line_coding =
{
	9600,
	CDC_CHAR_FORMAT_1_STOP_BIT,
	CDC_PARITY_NONE,
	8,
};

char hex[]="0123456789ABCDEF";

int main(void){
    hardware_init();
    ioInit();
    uartInit();
    uartSetBR(line_coding.dwDTERate);
    tmr2Init();    
    GIE=1;          //interrupts on

#ifdef MULTI_CLASS_DEVICE
	cdc_set_interface_list(cdc_interfaces, sizeof(cdc_interfaces));
#endif
	usb_init();
    while (1) {
        if(lcChange){           //line coding change from host
            lcChange=0;
            uartSetBR(line_coding.dwDTERate);   //16F145x only supports 8N1 in hardware
        }
        if(brkReq){             //break request from host
            while(PIE1bits.TXIE){}  //wait till buffer empty to maintain data order
            while(PIR1bits.TXIF==0){}   //wait til TX done
            TXSTAbits.SENDB=1;
            TXREG=0;
            txFlicker=LED_FLICKER_TIME*10;
            brkReq=0;
        }
        if (!usb_is_configured()){          //default mode if not enumerated            
            TRISCbits.TRISC4=1; //TX floating //disable output
            DTR_PIN=1;          //data terminal is not ready!
            uart_rx_buffer_tail=uart_rx_buffer_head;    //mark as empty
        }else{
            TRISCbits.TRISC4=0; //TX output
        }
		// Send data to the PC 
        if (usb_is_configured() &&
		    !usb_in_endpoint_halted(2) &&
		    !usb_in_endpoint_busy(2)) {
			int i,k;
			unsigned char *buf = usb_get_in_buffer(2);
            k=uartAvailable();
            if(k){
                if(k>IN_EP_LIMIT){k=IN_EP_LIMIT;}     //limit data sent            
                uartReceiveN(k,buf);
                usb_send_in_buffer(2, k);
            }
		}
		// Handle data received from the host 
		if (usb_is_configured() &&
		    !usb_out_endpoint_halted(2) &&
		    usb_out_endpoint_has_data(2) &&
            (brkReq==0)                   ){     //stall if break pending
			const unsigned char *out_buf;
			size_t out_buf_len;			
			out_buf_len = usb_get_out_buffer(2, &out_buf);
            if(out_buf_len){
                txFlicker=LED_FLICKER_TIME;
                if(uartTXBufferSpace()>=out_buf_len){   //keep EP waiting if no room in ring buffer
                    uartSendN(out_buf_len,out_buf);
                    usb_arm_out_endpoint(2);
                }
            }            
		}
    }
}

/* Callbacks. These function names are set in usb_config.h. */
void app_set_configuration_callback(uint8_t configuration)
{

}

uint16_t app_get_device_status_callback()
{
	return 0x0000;
}

void app_endpoint_halt_callback(uint8_t endpoint, bool halted)
{

}

int8_t app_set_interface_callback(uint8_t interface, uint8_t alt_setting)
{
	return 0;
}

int8_t app_get_interface_callback(uint8_t interface)
{
	return 0;
}

void app_out_transaction_callback(uint8_t endpoint)
{

}

void app_in_transaction_complete_callback(uint8_t endpoint)
{

}

int8_t app_unknown_setup_request_callback(const struct setup_packet *setup)
{
	/* To use the CDC device class, have a handler for unknown setup
	 * requests and call process_cdc_setup_request() (as shown here),
	 * which will check if the setup request is CDC-related, and will
	 * call the CDC application callbacks defined in usb_cdc.h. For
	 * composite devices containing other device classes, make sure
	 * MULTI_CLASS_DEVICE is defined in usb_config.h and call all
	 * appropriate device class setup request functions here.
	 */
	return process_cdc_setup_request(setup);
}

int16_t app_unknown_get_descriptor_callback(const struct setup_packet *pkt, const void **descriptor)
{
	return -1;
}

void app_start_of_frame_callback(void)
{

}

void app_usb_reset_callback(void)
{

}

/* CDC Callbacks. See usb_cdc.h for documentation. */

int8_t app_send_encapsulated_command(uint8_t interface, uint16_t length)
{
	return -1;
}

int16_t app_get_encapsulated_response(uint8_t interface,
                                      uint16_t length, const void **report,
                                      usb_ep0_data_stage_callback *callback,
                                      void **context)
{
	return -1;
}

int8_t app_set_comm_feature_callback(uint8_t interface,
                                     bool idle_setting,
                                     bool data_multiplexed_state)
{
	return -1;
}

int8_t app_clear_comm_feature_callback(uint8_t interface,
                                       bool idle_setting,
                                       bool data_multiplexed_state)
{
	return -1;
}

int8_t app_get_comm_feature_callback(uint8_t interface,
                                     bool *idle_setting,
                                     bool *data_multiplexed_state)
{
	return -1;
}

int8_t app_set_line_coding_callback(uint8_t interface,
                                    const struct cdc_line_coding *coding)
{
	line_coding = *coding;
    lcChange=1;
	return 0;
}

int8_t app_get_line_coding_callback(uint8_t interface,
                                    struct cdc_line_coding *coding)
{
	/* This is where baud rate, data, stop, and parity bits are set. */
	*coding = line_coding;
	return 0;
}

int8_t app_set_control_line_state_callback(uint8_t interface,
                                           bool dtr, bool dts)
{
	dtrState=dtr;
    if(dtrState){           //DTR change from host
        DTR_PIN=0;
    }else{
        DTR_PIN=1;
    }    
    return 0;
}

int8_t app_send_break_callback(uint8_t interface, uint16_t duration)
{
	brkReq=1;
    return 0;
}


#ifdef _PIC14E
void interrupt isr()
{
	usb_service();
    uartISR();
    tmr2ISR();
}
#elif _PIC18

#ifdef __XC8
void interrupt high_priority isr()
{
	usb_service();
}
#elif _PICC18
#error need to make ISR
#endif

#endif
