/* sketch for testing MAX7219 serial LED display driver
   with an Arduino Uno or compatible
   Version 2 - gives an expanding star on the 8x8 LED array
   written by Jim Rowe (Silicon Chip)
   Last updated 1/12/2016 at 3:35 pm

   Note: the following interface connections
   are assumed by the Arduino SPI library:
   SLAVE                      ARDUINO (MASTER)
   SS/CS-bar/LOAD  <-  from   DigIO pin 10
   DataOut/MOSI    <-  from   DigIO pin 11 (MOSI) or ICSP-4
   DataIn/MISO     ->  to     DigIO pin 12 (MISO) or ICSP-1
   SCK             <-  from   DigIO pin 13 (SCK) or ICSP-3
*/
#include <SPI.h>

// global variables
unsigned long outData;   // used for the outgoing 16-bit data to the MAX7219

const byte SSPin = 10;    // declares IO pin 10 as SSPin
const byte MOSIPin = 11;  // and IO pin 11 as MOSIPin
const byte MISOPin = 12;  // and IO pin 12 as MISOPin
const byte SCKPin = 13;   // and finally IO pin 13 as SCKPin

void setup()
{
  pinMode(SSPin, OUTPUT);         // make D10 an output for SS/Load
  digitalWrite(SSPin, LOW);       // and initialise it to LOW
  pinMode(MOSIPin, OUTPUT);       // make D11 an output for MOSI data
  digitalWrite(MOSIPin, LOW);     // and initialise to LOW
  pinMode(MISOPin, INPUT);        // make D12 an input for MISO data
                                  // (not used in this sketch)
  pinMode(SCKPin, OUTPUT);        // make D13 an output for SCK
  digitalWrite(SCKPin,LOW);       // and initialise to LOW
  
  SPI.begin();             // start up SPI interface
    
  // now we set the initial values for the MAX7219 mode registers
  outData = 0x0C01;   // sets the SHUTDOWN reg for normal operation
  SpiWrite();
  outData = 0x0900;   // sets the DECODE MODE reg for no decoding
  SpiWrite();
  outData = 0x0A07;   // sets the INTENSITY reg for 8/16 intensity
  SpiWrite();
  outData = 0x0B07;   // sets the SCAN LIMIT reg for all 8 digits
  SpiWrite();
  outData = 0x0F00;   // sets the DISPLAY-TEST reg for normal operation
  SpiWrite();
}  // end of setup function loop

void loop()
{
  // first blank the whole array
  outData = 0x0100;    // send a column of 0s to digit 0
  SpiWrite();
  outData = 0x0200;    // and digit 1
  SpiWrite();
  outData = 0x0300;    // and digit 2
  SpiWrite();
  outData = 0x0400;    // and digit 3
  SpiWrite();
  outData = 0x0500;    // and digit 4
  SpiWrite();
  outData = 0x0600;    // and digit 5
  SpiWrite();
  outData = 0x0700;    // and digit 6
  SpiWrite(); 
  outData = 0x0800;    // and digit 7
  SpiWrite();
  delay(1000);          // then pause briefly
  
  outData = 0x0418;    // now send first 4 dots of star
  SpiWrite();
  outData = 0x0518;
  SpiWrite();
  delay(250);           // pause very briefly

  outData = 0x0324;    // before sending next 4 dots
  SpiWrite();
  outData = 0x0624;
  SpiWrite();
  delay(250);           // very brief pause again
  
  outData = 0x0242;    // now send next 4 dots
  SpiWrite();
  outData = 0x0742;
  SpiWrite();
  delay(250);           // very brief pause again
   
  outData = 0x033C;    // then add 4 extra dots
  SpiWrite();
  outData = 0x043E;
  SpiWrite();
  outData = 0x057C;
  SpiWrite();
  outData = 0x063C;
  SpiWrite();
  delay(250);         // very brief pause again

  outData = 0x0181;    // now add the final outer 8 dots
  SpiWrite();
  outData = 0x0252;
  SpiWrite();
  outData = 0x074A;
  SpiWrite();
  outData = 0x0881;
  SpiWrite();
  delay(3000);         // now pause for visible impact
  
}   // end of main loop

// =====================================================================
// SpiWrite() function: to send 16 bits of data to the MAX7219 via SPI
// =====================================================================

void SpiWrite()
{
  digitalWrite(SCKPin,LOW);      // make sure SCK pin is low
  digitalWrite(MOSIPin,LOW);     // also the MOSI pin
  digitalWrite(SSPin,LOW);       // then take the SS pin low as well
  
  // first set the SPI port for 5MHz clock, data sent MSB first,
  // data mode 0 -- with SCK idle low (CPOL = 0) and
  //                MOSI data read on rising edge (CPHA = 0)
  // then begin the SPI transaction
  SPI.beginTransaction(SPISettings(5000000, MSBFIRST, SPI_MODE0));
  long recVal16;          // local variable for SPI received data (not used)
  
  recVal16 = SPI.transfer16(outData);  // send 16 bits of data out
  
  digitalWrite(SSPin, HIGH); // pull SS pin high to load into MAX7219
  delayMicroseconds(2);      // pause for 2us to make sure it's loaded
  digitalWrite(SSPin, LOW);   // then bring SS pin low again

  SPI.endTransaction();
  digitalWrite(MOSIPin,LOW);     // drop the MOSI pin
}  // end of SpiWrite() function

// ======================================================================




