#include <Wire.h>
#include "XC3715.h"
#include "HMC5883L_Simple.h"
#include <EEPROM.h>

HMC5883L_Simple Compass;
#define ROTATION_OFFSET 180

//buffer for display
unsigned int display[4]={0,0,0,0};

//buttons
#define BUTTON_1 A0
#define BUTTON_2 A1
#define BUTTON_3 A2

unsigned long buttonTime=0;
unsigned long dispTime=0;
#define BUTTON_DELAY 100
#define DISP_DELAY 400

//EEPROM
#include <EEPROM.h>
#define EEPROM_SIG (0x1234)
#define EEPROM_LOCATION (1)
#define SAVE_DELAY (10000)
bool saveNeeded=false;
unsigned long saveTime=0;

//display modes and parameters
struct saveData{
  int okSig;
  int dispBright;
  int declination;
  bool showDegrees;
};

saveData cur;
saveData initial={
  .okSig=EEPROM_SIG,
  .dispBright=15,
  .declination=0,
  .showDegrees=true
};

void setup() {
  Serial.begin(115200);
  delay(2000)  ;
  EEPROM.get(EEPROM_LOCATION,cur);
  dumpSettings();
  if(cur.okSig!=EEPROM_SIG){
    cur=initial;
    Serial.println("Defaults loaded");
    dumpSettings();
  }
  Wire.begin();
  XC3715_init();
  XC3715_brightness(cur.dispBright);
  Compass.SetDeclination(cur.declination, 0, 'E');  //just an int so we can pass -ve numbers
  Compass.SetSamplingMode(COMPASS_SINGLE);
  Compass.SetScale(COMPASS_SCALE_130);
  Compass.SetOrientation(COMPASS_HORIZONTAL_X_NORTH);
  pinMode(BUTTON_1,INPUT_PULLUP);
  pinMode(BUTTON_2,INPUT_PULLUP);
  pinMode(BUTTON_3,INPUT_PULLUP);
}

void loop() {
  if((millis()-buttonTime)>BUTTON_DELAY){buttonTime=buttonTime+BUTTON_DELAY;doButtons();}
  if((millis()-dispTime)>DISP_DELAY){dispTime=dispTime+DISP_DELAY;doDisplay();}
  if((saveNeeded)&&((millis()-saveTime)>SAVE_DELAY)){saveEEPROM();}
}

void doDisplay(){
  int h;
  float heading;
  if(digitalRead(BUTTON_1)==false){    //editing declination
    h=abs(cur.declination);
    if(h<10){
      display[0]=0; //blank
    }else{
      display[0]=XC3715_c[(h/10)%10];  //tens
    }
    display[1]=XC3715_c[(h/1)%10];  //units
    display[2]=XC3715_DEGREES;
    if(cur.declination<0){    //west
      display[3]=XC3715_c['W'-XC3715_LETTER_OFFSET];
    }else{      //east
      display[3]=XC3715_c['E'-XC3715_LETTER_OFFSET];
    }
    XC3715_data(display);
  }else{                  //normal display, depends on showDegrees
    heading = Compass.GetHeadingDegrees();
    h=heading;
    h=(h+360+ROTATION_OFFSET)%360;
    Serial.print("Heading: \t");
    Serial.println(h);   
    if(cur.showDegrees){
      if(h<100){
        display[0]=0; //blank
        if(h<10){
          display[1]=0; //blank
        }else{
          display[1]=XC3715_c[(h/10)%10];  //tens
        }
      }else{
        display[0]=XC3715_c[(h/100)%10];  //hundreds
        display[1]=XC3715_c[(h/10)%10];  //tens
      }
      display[2]=XC3715_c[(h/1)%10];  //units
      display[3]=XC3715_DEGREES;
      XC3715_data(display);
    }else{      
      switch(((h+22)/45)%8){
        case 0: //north
          display[0]=XC3715_c['N'-XC3715_LETTER_OFFSET];
          display[1]=0;
          display[2]=0;
          display[3]=XC3715_arrow[0];
          break;
        case 1: //north-east
          display[0]=XC3715_c['N'-XC3715_LETTER_OFFSET];
          display[1]=XC3715_c['E'-XC3715_LETTER_OFFSET];
          display[2]=0;
          display[3]=XC3715_arrow[7];   //to face north
          break;
        case 2: //east
          display[0]=XC3715_c['E'-XC3715_LETTER_OFFSET];
          display[1]=0;
          display[2]=0;
          display[3]=XC3715_arrow[6];
          break;
        case 3: //south-east
          display[0]=XC3715_c['S'-XC3715_LETTER_OFFSET];
          display[1]=XC3715_c['E'-XC3715_LETTER_OFFSET];
          display[2]=0;
          display[3]=XC3715_arrow[5];   //to face north
          break;
        case 4: //south
          display[0]=XC3715_c['S'-XC3715_LETTER_OFFSET];
          display[1]=0;
          display[2]=0;
          display[3]=XC3715_arrow[4];
          break;
        case 5: //south-west
          display[0]=XC3715_c['S'-XC3715_LETTER_OFFSET];
          display[1]=XC3715_c['W'-XC3715_LETTER_OFFSET];
          display[2]=0;
          display[3]=XC3715_arrow[3];   //to face north
          break;
        case 6: //west
          display[0]=XC3715_c['W'-XC3715_LETTER_OFFSET];
          display[1]=0;
          display[2]=0;
          display[3]=XC3715_arrow[2];
          break;
        case 7: //north-west
          display[0]=XC3715_c['N'-XC3715_LETTER_OFFSET];
          display[1]=XC3715_c['W'-XC3715_LETTER_OFFSET];
          display[2]=0;
          display[3]=XC3715_arrow[1];   //to face north
          break;
        default:
          display[0]=192;   //dashes; error
          display[1]=192;
          display[2]=192;
          display[3]=192;
          break;
      }
      XC3715_data(display);
    }
  }
}

void doButtons(){
  static bool old1state=true; //released
  bool new1state=digitalRead(BUTTON_1);
  if(new1state==false){   //button 1 pressed, set declination
    if(digitalRead(BUTTON_2)==0){
      cur.declination=cur.declination-1;
      if(cur.declination<-99){cur.declination=-99;}
      Compass.SetDeclination(cur.declination, 0, 'E');  //just an int so we can pass -ve numbers
      doDisplay();    //update
      flagSave();     //save is needed
    }
    if(digitalRead(BUTTON_3)==0){
      cur.declination=cur.declination+1;
      if(cur.declination>99){cur.declination=99;}
      Compass.SetDeclination(cur.declination, 0, 'E');  //just an int so we can pass -ve numbers
      doDisplay();    //update
      flagSave();     //save is needed
    }
    if(old1state==true){    //edge, button pressed
      if(cur.showDegrees){
        cur.showDegrees=false;
      }else{
        cur.showDegrees=true;
      }
      doDisplay();    //update
      flagSave();     //save is needed
    }
  }else{              //button 1 released, set brightness
    if(digitalRead(BUTTON_2)==0){
      cur.dispBright=cur.dispBright-1;
      if(cur.dispBright<0){cur.dispBright=0;}
      XC3715_brightness(cur.dispBright);
      doDisplay();    //update
      flagSave();     //save is needed
    }
    if(digitalRead(BUTTON_3)==0){
      cur.dispBright=cur.dispBright+1;
      if(cur.dispBright>15){cur.dispBright=15;}
      XC3715_brightness(cur.dispBright);
      doDisplay();    //update
      flagSave();     //save is needed
    }
  }
  old1state=new1state;
}

void flagSave(void){
  saveNeeded=true;
  saveTime=millis();
}

void saveEEPROM(void){
  unsigned int saveText[4]={XC3715_c['S'-XC3715_LETTER_OFFSET],XC3715_c['A'-XC3715_LETTER_OFFSET],XC3715_c['V'-XC3715_LETTER_OFFSET],XC3715_c['E'-XC3715_LETTER_OFFSET]};
  XC3715_data(saveText);
  EEPROM.put(EEPROM_LOCATION,cur);
  Serial.println("Saved to EEPROM");
  delay(500);
  saveNeeded=false;
  //display will update itself soon enough  
}

void dumpSettings(void){
  Serial.print("SIG:");
  Serial.println(cur.okSig);
  Serial.print("BRG:");
  Serial.println(cur.dispBright);
  Serial.print("DEC:");
  Serial.println(cur.declination);
  Serial.print("DEG:");
  Serial.println(cur.showDegrees);
}