#include <ESP8266WiFi.h>
#include <ESP8266HTTPClient.h>

#define LED_PIN D4

//TCP client for mail server
WiFiClient client;
#define STASSID "xxxxxxxx"
#define STAPSK "xxxxxxxx"
#define CONNECT_TIMEOUT 30000
#define SMTPUSER "xxxxxxxx"
#define SMTPPASS "xxxxxxxx"
#define SMTPHOST "mail.smtp2go.com"
#define SMTPPORT 2525
#define SMTPFROM "xxxxxxxx"
#define SMTPTO "xxxxxxxx"
#define SMTPSUBJECT "Ultrasonic Email Alerter"
#define SMTPBODY "This is a test."
#define BASE64_BUFFER 512
#define SMTP_BUFFER 1024
char smtpSubject[SMTP_BUFFER];
char smtpBody[SMTP_BUFFER];

//NTP uses code from NTPClient.ino example
//NTP, uses UDP
#include <WiFiUdp.h>
unsigned int localPort = 2390;  // local port to listen for UDP packets
const int NTP_PACKET_SIZE = 48;  // NTP time stamp is in the first 48 bytes of the message
byte packetBuffer[NTP_PACKET_SIZE];  // buffer to hold incoming and outgoing packets
WiFiUDP udp;
#define NTP_SERVER "pool.ntp.org"
unsigned long ntpOffset=0;

//date/time handling
int date,mon,yr;              //calculated by daysn()
int h,m,s;                    //hour, minute, second
int lastHour,lastDate;        //to check for rollover
bool flag=false;              //does NTP neet update?
char timeBuffer[128];
unsigned long oldMillis=0;    //to help detect rollover

//ultrasonic
#define ECHO D1
#define TRIG D2
//58 gives result in cm, max sets timeout
#define U_DIVIDER 58
#define U_MAX 200
int dist=0;     //global to access where needed

const char* getStatus(int d){   //use this to convert distance into status
  if(d==0){
    return "Sensor error.";
  }else if(d<50){
    return "Garage door open.";
  }else if(d<100){
    return "Car present, door closed.";
  }else{
    return "Car not present, door closed.";
  }
}

void setup() {
  Serial.begin(115200);
  usonicsetup();
  Serial.println();
  WiFi.begin(STASSID, STAPSK);
  Serial.printf("Connecting to %s.\r\n", WiFi.SSID().c_str());
  while (WiFi.status() != WL_CONNECTED) {
    delay(500);
    Serial.print(".");
    if(millis()>CONNECT_TIMEOUT){
      Serial.println("\r\nCould not connect, rebooting");
      delay(100);
      ESP.restart();
    }
  }
  Serial.println("");
  Serial.print("Connected! IP address: ");
  Serial.println(WiFi.localIP());
  udp.begin(localPort);
  Serial.printf("UDP started on port %d.\r\n",udp.localPort());
  pinMode(LED_PIN,OUTPUT);
  digitalWrite(LED_PIN,HIGH); //active LOW
  doTime();     //try once at boot
  updateTime();
  lastHour=h;
  lastDate=date;
}

void loop() {
  int d;  //serial data
  updateTime();
  if(Serial.available()){
    d=Serial.read();
    if(d=='t'){doDummy();}
    if(d=='~'){doReport();}
  }
  if((WiFi.status() == WL_CONNECTED)&&(ntpOffset>0)){
    digitalWrite(LED_PIN,LOW); //active LOW
  }else{
    digitalWrite(LED_PIN,HIGH); //active LOW
  }
  if(h!=lastHour){  //hour has rolled over
    lastHour=h;
    Serial.println("Hour rollover");
    Serial.print(timeBuffer);    
    //doDummy();    //for testing
    doReport();   //for going live
  }
  if(date!=lastDate){
    lastDate=date;
    Serial.println("Day rollover");
    Serial.print(timeBuffer);    
    flag=true;
  }
  if((flag==true)&&(m>30)){   //update NTP at 30 minutes after midnight to avoid glitches    
    flag=false;
    doTime();
    updateTime();
  }
}

void doReport(void){
  Serial.println("\r\nReport triggered");
  dist=usonic(U_MAX*U_DIVIDER)/U_DIVIDER;      
  Serial.printf("Measured %d cm.\r\n",dist);
  Serial.println(getStatus(dist));
  Serial.print(timeBuffer);    
  sprintf(smtpSubject,"Ultrasonic Email Alerter: %s",getStatus(dist));
  sprintf(smtpBody,"%sMeasured %d cm.\r\n%s",timeBuffer,dist,getStatus(dist));
  Serial.println("---EMAIL CONTENT---");
  Serial.println(smtpSubject);
  Serial.println(smtpBody);
  Serial.println("-------------------");
  doEmail();
}

void doDummy(void){   //note that this is much the same as the above, but without doEmail()
  Serial.println("\r\nDummy report triggered");
  dist=usonic(U_MAX*U_DIVIDER)/U_DIVIDER;      
  Serial.printf("Measured %d cm.\r\n",dist);
  Serial.println(getStatus(dist));
  Serial.print(timeBuffer);    
  sprintf(smtpSubject,"Ultrasonic Email Alerter: %s",getStatus(dist));
  sprintf(smtpBody,"%sMeasured %d cm.\r\n%s",timeBuffer,dist,getStatus(dist));
  Serial.println("---EMAIL CONTENT---");
  Serial.println(smtpSubject);
  Serial.println(smtpBody);
  Serial.println("-------------------");
}

void updateTime(void){
  unsigned long epoch;
  unsigned long mTemp;
  mTemp=millis();
  if(mTemp<oldMillis){    //rollover has happened
    ntpOffset=ntpOffset+4294967UL   ;//adjust
    Serial.print("millis() has rolled over, offset adjusted.");
    //ESP.restart();    //might be an option to properly reset the time
    delay(100);
  }
  oldMillis=mTemp;
  epoch=ntpOffset+mTemp/1000;
  dodsn(epoch/86400UL);   
  h=(epoch/3600)%24;
  m=(epoch/60)%60;
  s=epoch%60;
  sprintf(timeBuffer,"Time is %02d:%02d:%02d on %02d/%02d/%04d (UTC)\r\n",h,m,s,date,mon,yr); //hh:mm:ss, dd/mm/yyyy
}

void doTime(void){
  int cb;
  unsigned long t=0;
  Serial.println("Getting time from NTP");
  sendNTPpacket();
  delay(1000);
  cb=udp.parsePacket();
  if(cb){
    Serial.printf("%d bytes received.\r\n",cb);
    udp.read(packetBuffer, NTP_PACKET_SIZE);  // read the packet into the buffer
    t=packetBuffer[40];
    t=(t<<8)|packetBuffer[41];
    t=(t<<8)|packetBuffer[42];
    t=(t<<8)|packetBuffer[43];
    Serial.printf("NTP is 0x%x\r\n",t);
    oldMillis=millis();
    ntpOffset=t-(oldMillis/1000);
    updateTime();
    Serial.print(timeBuffer);
  }else{
    Serial.println("No response.");
  }
}

void doEmail(void){
  int r=0;  
  Serial.println("Sending email.");
  digitalWrite(LED_PIN,HIGH); //active LOW, flicker LED to show activity
  r=client.connect(SMTPHOST,SMTPPORT);
  if(r==0){
    Serial.printf("Could not connect to %s\r\n",SMTPHOST);
    return;
  }
  client.print("EHLO\r\nAUTH LOGIN\r\n");
  client.print(to64(SMTPUSER));
  client.print("\r\n");
  client.print(to64(SMTPPASS));
  client.print("\r\nMAIL FROM: ");
  client.print(SMTPFROM);
  client.print("\r\nRCPT TO: ");
  client.print(SMTPTO);
  client.print("\r\nDATA\r\n"); //end of header
  client.print("From: ");       //repeated in body
  client.print(SMTPFROM);
  client.print("\r\nTo: ");
  client.print(SMTPTO);
  client.print("\r\nSubject: ");
  client.print(smtpSubject);
  client.print("\r\n\r\n");
  client.print(smtpBody);
  client.print("\r\n.\r\n");  //this ends the message
  delay(1000);
  Serial.printf("%d bytes received.\r\n--------------\r\n",client.available());
  while(client.available()){
    Serial.write(client.read());
  }
  Serial.println("--------------");
}

// send an NTP request to the time server at the given address
// modified to use address
void sendNTPpacket(void) {
  Serial.println("sending NTP packet...");
  // set all bytes in the buffer to 0
  memset(packetBuffer, 0, NTP_PACKET_SIZE);
  // Initialize values needed to form NTP request
  // (see URL above for details on the packets)
  packetBuffer[0] = 0b11100011;  // LI, Version, Mode
  packetBuffer[1] = 0;           // Stratum, or type of clock
  packetBuffer[2] = 6;           // Polling Interval
  packetBuffer[3] = 0xEC;        // Peer Clock Precision
  // 8 bytes of zero for Root Delay & Root Dispersion
  packetBuffer[12] = 49;
  packetBuffer[13] = 0x4E;
  packetBuffer[14] = 49;
  packetBuffer[15] = 52;
  // all NTP fields have been given values, now
  // you can send a packet requesting a timestamp:
  udp.beginPacket(NTP_SERVER, 123);  // NTP requests are to port 123
  udp.write(packetBuffer, NTP_PACKET_SIZE);
  udp.endPacket();
}


char* to64(const char* s){  //convert to base 64
  static char buf[BASE64_BUFFER];
  static const char b[]="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
  unsigned i,n,k,p,bPtr;
  for(i=0;i<BASE64_BUFFER;i++){buf[i]=0;} //clear
  n=strlen(s);
  if(n>(BASE64_BUFFER/2)){return buf;}  //too long, return empty
  k=0;
  p=0;
  bPtr=0;
  for(i=0;i<n;i=i+3){
    k=s[i];
    k=k<<8;
    if((i+1)<n){
      k=k|s[i+1];
    }else{
      p=p+1;
    }
    k=k<<8;
    if((i+2)<n){
      k=k|s[i+2];
    }else{
      p=p+1;
    }
    buf[bPtr+3]=b[k&0x3F];
    k=k>>6;
    buf[bPtr+2]=b[k&0x3F];
    k=k>>6;
    buf[bPtr+1]=b[k&0x3F];
    k=k>>6;
    buf[bPtr+0]=b[k&0x3F];
    bPtr=bPtr+4;    
  }
  bPtr=bPtr-p;
  while(p){
    buf[bPtr]='=';
    bPtr++;
    p--;
  }
  return buf;
}

void dodsn(unsigned long dsn){
  byte dim[]={0,31,28,31,30,31,30,31,31,30,31,30,31};   //days in month
  yr=1900;
  mon=1;
  date=dsn+1;                       //1/1/1900 has timestamp 0, ie date of 0 is 1
  while(date>365){                  //count years             
    if(((yr%4)==0)&&(yr!=1900)){
      date--;
    }
    yr++;
    date=date-365;
  }
  if(((yr%4)==0)&&(yr!=1900)){dim[2]++;}  //if a leap year, Feb has 29
  while(date>dim[mon]){                   //count months
    date=date-dim[mon];
    mon++;
  }
  if((date==0)&&(mon==1)){                //glitch on 31/12 of leap years
    yr--;
    date=31;
    mon=12;
  }
}

void usonicsetup(void){
  pinMode(ECHO, INPUT);
  pinMode(TRIG, OUTPUT);
  digitalWrite(TRIG, LOW);
}

long usonic(long utimeout){ //utimeout is maximum time to wait for return in us
  long b;
  if(digitalRead(ECHO)==HIGH){return 0;} //if echo line is still low from last result,return 0;
  digitalWrite(TRIG, HIGH); //send trigger pulse
  delay(1);
  digitalWrite(TRIG, LOW);
  long utimer=micros();
  while((digitalRead(ECHO)==LOW)&&((micros()-utimer)<1000)){} //wait for pin state to change-return starts after 460us typically or timeout (eg if not connected)
  utimer=micros();
  while((digitalRead(ECHO)==HIGH)&&((micros()-utimer)<utimeout)){} //wait for pin state to change
  b=micros()-utimer;
  if(b==0){b=utimeout;}
  return b;
}