' -----------------------------------------------------------
' Hello World program for Micromite to drive serial I2C LCD
' Adapted from a Maximite I2CLCD program by John Gerrard
' Changed to suit Micromite & Chinese I2C LCD module
' (different connections from PCF8574 to LCD)
' by Jim Rowe. Last revision 15/11/2016 at 3:45 pm

' LCD functions of PCF8571 output bits P0 - P7:
' Name      8574 bit  LCD function
' DB4       = P4      Data Line 4 (pin 11)
' DB5       = P5      Data Line 5 (pin 12)
' DB6       = P6      Data Line 6 (pin 13)
' DB7       = P7      Data Line 7 (pin 14)
' RS        = P0      Reg select (pin 4) 0 = Cmd 1 = Data
' EN        = P2      EN (pin 6) 0 = Idle, 1 = Active
' R/Wbar    = P1      R/W-bar (pin 5, should be 0 for writing)
' BL        = P3      Backlight control (Q1 & LCD pin 16)
' ----------------------------------------------------------

OPTION AUTORUN ON
OPTION EXPLICIT

' declare the PCF8574 I2C address (all links out)
DIM AS INTEGER I2CAddr  = &H27  ' (A2=A1=A0=1)

' masks for preparing I2C bytes for writing to 8574 & LCD
DIM AS INTEGER RSCmask = &B00000000 ' select cmd register (RS = 0)
DIM AS INTEGER RSDmask = &B00000001 ' or data register (RS = 1)
DIM AS INTEGER ENmask  = &B00000100 ' Enable (active = P2 high)
DIM AS INTEGER BLmask  = &B00001000 ' Turn on backlighting (P3 = 1)
DIM AS INTEGER i, index               ' counters
DIM AS INTEGER aByte, RSbit, temp     ' variables

Dim AS INTEGER CNT(7) ' array to store LCD config command bytes
CNT(0) = &B00110011   ' &H33 --> 8-bit / 8-bit mode init
CNT(1) = &B00110010   ' &H32 --> 8-bit / 4-bit mode
CNT(2) = &B00101000   ' &H28 --> 4-bit mode, 2 lines, 5x7 pixels
CNT(3) = &B00001000   ' &H08 --> disp off, cursor off, no blink
CNT(4) = &B00000001   ' &H01 --> clear screen, return home cmd
CNT(5) = &B00000110   ' &H06 --> increment cursor, cursor move
CNT(6) = &B00001100   ' &H0C --> disp on, cursor off, no blink

DIM AS STRING Line1$ = "Hello, world!"   ' string for line 1
DIM AS STRING Line2$ = "Silicon Chip"	  ' string for line 2

PAUSE 200        ' wait 200ms for LCD to stabilise after powerup
I2C OPEN 100,100 ' enable I2C (bus speed 100kHz, 100ms timeout)
InitLCD          ' then go initialise the LCD + controller

' main program loop starts here
DO
  aByte = CNT(4)         ' clear LCD screen, return home
  SendCmdByte
  PAUSE 400              ' pause for 400ms to give it time...
  ' now send first line of text to the LCD via the PCF8574
  For i = 1 To Len(Line1$)
    aByte = ASC(Mid$(Line1$, i, 1))
    SendDataByte
  Next i
  PAUSE 2000             ' pause 2s before doing second line
  aByte = &B11000000     ' now put cursor at start of line 2
  SendCmdByte
  'and send the second line of text
  For i = 1 To Len(Line2$)
    aByte = Asc(Mid$(Line2$, i, 1))
    SendDataByte
  Next i
  PAUSE 3000             ' pause 3 sec before looping back 
LOOP

END ' end of main part of program, subs follow 
' -----------------------------------------------------------------
'  subroutine to initialise the LCD
SUB InitLCD
    For index = 0 To 6
      aByte = CNT(index)
      SendCmdByte
      IF index = 4 THEN
        PAUSE 20  ' pause for 20ms to allow full clear
      END IF
      PAUSE 50    ' pause to let LCD controller settle down
    Next
END SUB
' -----------------------------------------------------------------
' subroutine to send a command byte (aByte) to the LCD controller
SUB SendCmdByte
    RSbit = RSCmask  ' make sure we're sending to cmd reg (RS=0)
    ' then prepare the more significant nibble
    temp = (aByte AND &HF0) Or RSbit OR BLmask
    DirectSend             ' and send it
    ' then prepare the less significant nibble
    temp = ((aByte AND &H0F) << 4) Or RSbit OR BLmask
    DirectSend             ' and send it also   
END SUB
' -----------------------------------------------------------------
' subroutine to send a data byte (aByte) to the LCD controller
SUB SendDataByte
    RSbit = RSDmask  ' make sure we're sending to data reg (RS=1)
    ' then prepare the more significant nibble (from aByte)
    temp = (aByte AND &B01110000) Or RSbit OR BLmask
    DirectSend              ' and send it
    ' then prepare the less significant nibble (from aByte)
    temp = ((aByte AND &H0F) << 4) OR RSbit OR BLmask
    DirectSend              ' and send it also
END SUB
' -----------------------------------------------------------------
' subroutine to send a nibble (in temp) to the LCD
SUB DirectSend
    temp = temp Xor ENmask            ' make EN=1
    I2C WRITE I2CAddr, 0, 1, temp     ' send it with EN=1
    Pause 10                          ' pause for 10ms
    temp = temp Xor ENmask            ' now make EN=0 again
    I2C WRITE I2CAddr, 0, 1, temp     ' before sending again
END SUB
' ----------------------------------------------------------------
