// Ultrasonic Theremin using PWM to play samples
// Samples played at rate dependent on distance read by ultrasonic sensor
// Giving theremin-like effect

// Sample timer interrupt runs 16bit Timer 1 at FREQ Hz (try 22050)
// PWM timer runs at 62500Hz on 8bit Timer 2 (max possible with 16MHz processor)
// Timer 0 is left unaltered, and is needed for delay/millis in ultrasonic functions
// sample is held in the sample array (sine, piano and theremin samples).

#include "sample.c"

#define VOL_SENSOR 1     // Set to 1 if you have a second ultrasonic sensor for controlling volume.
#define MAX_DIST   5800  // Max detection distance in microseconds.
#define FREQ       22050 // Playback frequency.
#define AMP_AUDIN  3     // This can be changed to other PWM-enabled pins if you
                         // know to change the corresponding registry access on the chip.

// Ultrasonic HC-SR04 unit interface
// define pins here
// if using fixed power pins, set to a negative number, and they will be ignored
#define UVCC  8
#define UTRIG 9
#define UECHO 10
#define UGND  11

// Ultrasonic pins for second sensor used for volume control.
// We use the analog pins as digital pins on the Uno board for the volume-control sensor.
#if VOL_SENSOR == 1
  #define V_UVCC  A2
  #define V_UTRIG A3
  #define V_UECHO A4
  #define V_UGND  A5
#endif

volatile unsigned long i = 0;
volatile unsigned long u = 0, u_target = 0, volume;

void setup()
{
  usonicsetup(UVCC, UTRIG, UECHO, UGND);
  #if VOL_SENSOR == 1
    usonicsetup(V_UVCC, V_UTRIG, V_UECHO, V_UGND);
  #endif
  TMRsetup();
}

unsigned long last = 0;

void loop()
{
  unsigned long timeout = 0;
  long t1, t2; // Return time for sensor 1 and 2.

  // up to 1m
  usonic(MAX_DIST, &t1, &t2);
  // invert so closer to sensor is higher
  t1 = MAX_DIST - t1;
  t2 = MAX_DIST - t2;

  // Timeout for pitch-controlling sensor.
  if (t1 > MAX_DIST - 100)
  {
    timeout++;
    t1 = last;
  }
  else //handle edge cases
  {
    timeout = 0;
  }

  last = timeout > 2 ? 0 : t1;
  
  volume = t2 ? t2 * 256 / MAX_DIST : 256;

  u_target = t1 * 200L; // 50 gives closer to true sample rate, but doesn't sound as good
}

// gets triggered FREQ times/second
ISR(TIMER1_COMPA_vect)
{
  unsigned long val;

  //val = sine256[(i >> 18) % 256]; // If you want to use sine256 uncomment this, and comment the line below.
  val = pgm_read_byte(&theremin[i >> 18]);
  val -= 128;
  val = val * volume / 256;
  val += 128;
  OCR2B = val; // load sample, change name to change sample.
  i += u;      // step forward to next sample

  // Makes the step size more uniform.
  if (u_target > u)      {u += 40L;}
  else if (u_target < u) {u -= 40L;}
}

void TMRsetup()
{
  // Timer 1 set up as a FREQ Hz sample interrupt, only common thing this affects is servo library
  TCCR1A = 0;
  TCCR1B = _BV(WGM12) | _BV(CS10);
  TCNT1  = 0;
  OCR1A  = F_CPU / FREQ;
  TIMSK1 = _BV(OCIE1A);

  // PWM output fixed to pin 3 on TMR2 at 62.5kHz, PWM duty cycle is set by OCR2B
  pinMode(AMP_AUDIN, OUTPUT);
  TCCR2A = _BV(COM2B1) | _BV(WGM21) | _BV(WGM20);
  TCCR2B = _BV(CS20);
  TIMSK2 = 0;
}

// Sets the four pin locations (in order) on the ultrasonic sensor.
void usonicsetup(int vcc, int trig, int echo, int gnd)
{
  if (gnd > -1) // set up ground pin if defined
  {
    pinMode(gnd, OUTPUT);
    digitalWrite(gnd, LOW);
  }
  if (vcc > -1) // set up VCC pin if defined
  {
    pinMode(vcc, OUTPUT);
    digitalWrite(vcc, HIGH);
  }

  pinMode(echo, INPUT);    // ECHO pin is input
  pinMode(trig, OUTPUT);   // TRIG pin is output
  digitalWrite(trig, LOW);
}

// utimeout is maximum time to wait for return in us
void usonic(long utimeout, long *ret1, long *ret2)
{
  long utimer[4] = {-1, -1, -1, -1}, utimer_start;
  *ret1 = *ret2 = 0;

  // if UECHO line is still low from last result, return 0
  if (digitalRead(UECHO) == HIGH) {*ret1 = 0;}
  else                            {*ret1 = -1;}

  #if VOL_SENSOR == 1
    if (digitalRead(V_UECHO) == HIGH) {*ret2 = 0;}
    else                              {*ret2 = -1;}
  #endif
    
  if (*ret1 == 0 && *ret2 == 0) {return;}

  /*
  if (*ret1 == -1) {digitalWrite(UTRIG, HIGH);}
  if (*ret2 == -1 && VOL_SENSOR) {digitalWrite(V_UTRIG, HIGH);}
  delay(1);
  if (*ret1 == -1) {digitalWrite(UTRIG, LOW);}
  if (*ret2 == -1 && VOL_SENSOR) {digitalWrite(V_UTRIG, LOW);}
  */

  // send trigger pulses
  if (*ret1 == -1) {digitalWrite(UTRIG, HIGH); delay(1); digitalWrite(UTRIG, LOW);}

  #if VOL_SENSOR == 1
    if (*ret2 == -1) {digitalWrite(V_UTRIG, HIGH); delay(1); digitalWrite(V_UTRIG, LOW);}
  #endif

  #ifdef V_UECHO
    #define V_ECHO_LOW digitalRead(V_UECHO) == LOW
  #else
    #define V_ECHO_LOW 1
  #endif

  // wait for pin state to change - return starts after 460us typically
  utimer_start = micros();
  while ( (*ret1 == 0 || digitalRead(UECHO) == LOW) 
          && (*ret2 == 0 || V_ECHO_LOW) 
          && micros() - utimer_start < 1000 ) {
    ;
  }
    
  if (*ret1 == 0) {utimer[0] = utimer[2] = 0;}
  if (*ret2 == 0) {utimer[1] = utimer[3] = 0;}

  utimer_start = micros();
  while ( (utimer[2] == -1 || utimer[3] == -1) && micros() - utimer_start < utimeout )
  {
    if (digitalRead(UECHO) == HIGH && utimer[0] == -1)
    {
      utimer[0] = micros();
    }
    if (utimer[0] != -1 && digitalRead(UECHO) == LOW && utimer[2] == -1)
    {
      utimer[2] = micros();
    }

    #if VOL_SENSOR == 1
      if (digitalRead(V_UECHO) == HIGH && utimer[1] == -1)
      {
        utimer[1] = micros();
      }
      if (utimer[1] != -1 && digitalRead(V_UECHO) == LOW && utimer[3] == -1)
      {
        utimer[3] = micros();
      }
    #endif
  }

  if (utimer[2] == -1) {*ret1 = utimeout;}
  else                 {*ret1 = utimer[2] - utimer[0];}

  if (utimer[3] == -1) {*ret2 = utimeout;}
  else                 {*ret2 = utimer[3] - utimer[1];}
}

