// Ultrasonic Theremin using PWM to play samples.
// Samples played at rate dependent on distance read by ultrasonic sensor
// giving theremin-like effect.

// Sample timer interrupt runs 16bit Timer 1 at FREQ Hz (try 22050)
// PWM timer runs at 62500Hz on 8bit Timer 2 (max possible with 16MHz processor)
// Timer 0 is left unaltered, and is needed for delay/millis in ultrasonic functions
// sample is held in the sample array (sine, piano and theremin samples).

#include "sample.c"

#define MAX_DIST   5800  // Max detection distance in microseconds.
#define FREQ       22050 // Playback frequency.
#define AMP_AUDIN  3     // This can be changed to other PWM-enabled pins if you
                         // know to change the corresponding registry access on the chip.
                         
// Ultrasonic HC-SR04 unit interface
// define pins here
// if using fixed power pins, set to a negative number, and they will be ignored.
#define UVCC  8
#define UTRIG 9
#define UECHO 10
#define UGND  11

volatile unsigned long i = 0;
volatile unsigned long u = 0, u_target = 0;

void setup()
{
  usonicsetup(UVCC, UTRIG, UECHO, UGND);
  TMRsetup();
}

unsigned long last = 0;

void loop()
{
  unsigned long timeout = 0, t; // Return time for sensor.

  t = usonic(MAX_DIST);
  // Invert so closer to sensor is higher.
  t = MAX_DIST - t;

  // Timeout for sensor.
  if (t > MAX_DIST - 100)
  {
    timeout++;
    t = last;
  }
  else
  {
    // Handle edge cases.
    timeout = 0;
  }

  last = timeout > 2 ? 0 : t;

  // 50 gives closer to true sample rate, but doesn't sound as good.
  u_target = t * 200L;
}

// gets triggered FREQ times/second
ISR(TIMER1_COMPA_vect)
{
  //OCR2B = sine256[(i >> 18) % 256]; // If you want to use sine256 uncomment this, and comment the line below.
  OCR2B = pgm_read_byte(&theremin[i >> 18]); // Load sample, change name to change sample.
  
  i += u; // Step forward to next sample.

  // Makes the step size more uniform.
  if (u_target > u)      {u += 40L;}
  else if (u_target < u) {u -= 40L;}
}

void TMRsetup()
{
  // Timer 1 set up as a FREQ Hz sample interrupt, only common thing this affects is servo library.
  TCCR1A = 0;
  TCCR1B = _BV(WGM12) | _BV(CS10);
  TCNT1  = 0;
  OCR1A  = F_CPU / FREQ;
  TIMSK1 = _BV(OCIE1A);

  // PWM output fixed to pin 3 on TMR2 at 62.5kHz, PWM duty cycle is set by OCR2B.
  pinMode(AMP_AUDIN, OUTPUT);
  TCCR2A = _BV(COM2B1) | _BV(WGM21) | _BV(WGM20);
  TCCR2B = _BV(CS20);
  TIMSK2 = 0;
}

// Sets the four pin locations (in order) on the ultrasonic sensor.
void usonicsetup(int vcc, int trig, int echo, int gnd)
{
  if (gnd > -1) // set up ground pin if defined
  {
    pinMode(gnd, OUTPUT);
    digitalWrite(gnd, LOW);
  }
  if (vcc > -1) // set up vcc pin if defined
  {
    pinMode(vcc, OUTPUT);
    digitalWrite(vcc, HIGH);
  }

  pinMode(echo, INPUT);    // echo pin is input
  pinMode(trig, OUTPUT);   // trig pin is output
  digitalWrite(trig, LOW);
}

// utimeout is maximum time to wait for return in us.
long usonic(long utimeout)
{
  long b, utimer;

  // if UECHO line is still low from last result, return 0
  if (digitalRead(UECHO) == HIGH) {return 0;}

  // send trigger pulse
  digitalWrite(UTRIG, HIGH);
  delay(1);
  digitalWrite(UTRIG, LOW);

  utimer = micros();
  // Wait for pin state to change - return starts after 460us typically.
  while ((digitalRead(UECHO) == LOW) && ((micros() - utimer) < 1000)) {
    ;
  }
  
  utimer = micros();
  // Wait for pin state to change.
  while ((digitalRead(UECHO) == HIGH) && ((micros() - utimer) < utimeout)) {
    ;
  }
  
  b = micros() - utimer;

  return b;
}
