#include "LCD.h"
#include "Arial_round_16x24.c"
#include "esp_camera.h"
#include <WiFi.h>

//camera includes and variables
#define CAMERA_MODEL_AI_THINKER // selected in camera_pins.h
//includes some helper functions camInit,showCamID, BMPHeader_t struct
//also getRow and getPixel for reading bitmaps
#include "camera_pins.h"
int e;      //error status
sensor_t * s;
uint16_t camID; //see camera_pid_t enum
#define OV2640_MODE_SVGA 1
#define OV2640_MODE_UXGA 0

//filesystem on Flash
#include "FS.h"
#include <LittleFS.h>
File root, currentFile;
char fileName[]="/00000.BMP";
unsigned int fileNumber=0;
unsigned int dispNumber=0;
float zoomLevel=1;

//this will hold a camera_fb_t followed by the buf pixel data, nb buf will need to be set
#define RAM_BUF_SIZE 32768
uint8_t ramFB[RAM_BUF_SIZE];
camera_fb_t* readFB=(camera_fb_t*)ramFB;
int n=0;

//buttons
button b0=  {.x=320,.y= 18,.w=140,.h=35,.text="CAPTURE",.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange
button b1=  {.x=320,.y= 68,.w=140,.h=35,.text="ZOOM+"  ,.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange
button b2=  {.x=320,.y=118,.w=140,.h=35,.text="ZOOM-"  ,.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange
button b3=  {.x=320,.y=168,.w=140,.h=35,.text="NEXT"   ,.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange
button b4=  {.x=320,.y=218,.w=140,.h=35,.text="PREV"   ,.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange
button b5=  {.x=320,.y=268,.w=140,.h=35,.text="FORMAT" ,.pressed=0,.visible=1,.lastchange=0};//x,y,w,h,text,pressed,visible,lastchange

void setup() {
  Serial.begin(115200);
  displaySetup(); //this inits LCD controller, touch and backlight PWM
  setrotation(1);
  clear(DARK_GREY);
  drawbutton(&b0);
  drawbutton(&b1);
  drawbutton(&b2);
  drawbutton(&b3);
  drawbutton(&b4);
  drawbutton(&b5);
  showZoom(zoomLevel);
  e=camInit();
  if(e==ESP_OK){
    Serial.println("Camera initialised");
  }else{
    Serial.printf("Camera init error %d.\r\n",e);
  }
  s = esp_camera_sensor_get();
  camID = s->id.PID;
  showCamID(camID);
  setZoom(zoomLevel);
  e=s->set_quality(s,10);
  if(e==ESP_OK){
    Serial.println("OK");
  }else{
    Serial.printf("Error %d.\r\n",e);
  }
  if(LittleFS.begin()){
    Serial.println("LittleFS mounted OK");
    root=LittleFS.open("/");
  }else{
    Serial.println("LittleFS not mounted, attempting format");
    if(LittleFS.format()){
      if(LittleFS.begin()){
        Serial.println("LittleFS formatted and mounted OK");
        root=LittleFS.open("/");
      }else{
        Serial.println("LittleFS not mounted, not available");      
      }
    }
  }
  reportFSstatus();
  fileNumber=0;
  setFN(fileNumber);
  while(LittleFS.exists(fileName)){
    fileNumber=fileNumber+1;
    setFN(fileNumber);
  }
  Serial.printf("%s is available.\r\n",fileName);
  if(showFile(dispNumber)==0){
    box(32,96,287,287,BLACK);
    showarray(80,292,"NO IMAGES ",Arial_round_16x24,YELLOW,BLACK);
  }
}


void loop() {       //this displays at actual size
  camera_fb_t *fb = NULL;
  uint8_t* out;     //for bitmap data
  size_t out_len;   //length of bitmap data
  e=ESP_OK;
  touchCoords t;    //for reading touch
  static bool takeImg=0;  //flag to capture image
  fb = esp_camera_fb_get();
  if(fb){
    if(frame2bmp(fb,&out,&out_len)){    //convert from jpg to bmp
      drawThumb(out,118,5,85,64,3);     //scaled down by 3
      if(takeImg){
        drawBitMap(out,32,96,256,192);      //this should centre the image
        showarray(80,292,"CAPTURED  ",Arial_round_16x24,YELLOW,BLACK);
        takeImg=0;
        if(getFreeFile()){
          Serial.printf("Saving to %s\r\n",fileName);          
          showarray(80,292,"SAVING    ",Arial_round_16x24,YELLOW,BLACK);
          currentFile=LittleFS.open(fileName,"w");
          if(currentFile){
            int bCount=currentFile.write((uint8_t*)fb,sizeof(*fb));
            bCount=bCount+currentFile.write(fb->buf,fb->len);
            if(bCount==(sizeof(*fb)+fb->len)){
              Serial.printf("%d bytes written OK.\r\n",bCount);
              showarray(80,292,"SAVED     ",Arial_round_16x24,YELLOW,BLACK);
            }else{
              Serial.printf("Write did not complete %d/%d bytes written.\r\n",bCount,(sizeof(*fb)+fb->len));
            }
            currentFile.close();
            reportFSstatus();
          }else{
            Serial.println("Could not open file for saving");
          }
        }else{
          Serial.println("No space to save");
        }
      }
    }
    free(out);                //release bitmap buffer
  }
  esp_camera_fb_return(fb);   //release frame buffer
  //check touch
  t=getTouchCoords();
  if(handleButton(&b0,t)==BUTTON_DOWN){ //capture
    takeImg=1;
  }
  if(handleButton(&b1,t)==BUTTON_DOWN){ //zoom+
    zoomLevel=zoomLevel+0.5;
    if(zoomLevel>4){zoomLevel=4;}
    setZoom(zoomLevel);
    showZoom(zoomLevel);
  }
  if(handleButton(&b2,t)==BUTTON_DOWN){ //zoom-
    zoomLevel=zoomLevel-0.5;
    if(zoomLevel<1){zoomLevel=1;}
    setZoom(zoomLevel);
    showZoom(zoomLevel);
  }
  if(handleButton(&b3,t)==BUTTON_DOWN){ //next
    dispNumber=dispNumber+1;
    if(dispNumber>fileNumber){dispNumber=fileNumber;}
    if(showFile(dispNumber)==0){
      box(32,96,287,287,BLACK);
      showarray(80,292,"NOT FOUND ",Arial_round_16x24,YELLOW,BLACK);
    }
  }
  if(handleButton(&b4,t)==BUTTON_DOWN){ //prev
    if(dispNumber>0){dispNumber=dispNumber-1;}
    if(showFile(dispNumber)==0){
      box(32,96,287,287,BLACK);
      showarray(80,292,"NOT FOUND ",Arial_round_16x24,YELLOW,BLACK);
    }
  }
  if(handleButton(&b5,t)==BUTTON_DOWN){ //format
    dispNumber=0;
    box(32,96,287,287,BLACK);
    showarray(80,292,"FORMATTED ",Arial_round_16x24,YELLOW,BLACK);
    root.close();
    if(LittleFS.format()){
      if(LittleFS.begin()){
        Serial.println("LittleFS formatted and mounted OK");
        root=LittleFS.open("/");
      }else{
        Serial.println("LittleFS not mounted, not available");      
      }
    }
    reportFSstatus();
  }
}

void drawThumb(uint8_t * bm,unsigned int x,unsigned int y,unsigned int w,unsigned int h, unsigned int s){ //scaled down /s simple subsmapling
  unsigned int bmw, bmh;
  unsigned int u,v;               //counters  
  BMPHeader_t* header;  
  uint8_t* rowData;
  header=(BMPHeader_t*)bm;  //header is at start of bitmap data
  bmw=abs(header->width);
  bmh=abs(header->height);
  setarea(x,y,x+w-1,y+h-1);
  digitalWrite(LCDCS,LOW);
  for(v=0;v<h*s;v=v+s){
    rowData=getRow(header,v);
    if(rowData){
      for(u=0;u<w;u=u+1){   //RGB triplets
        data8(rowData[u*s*3+2]);  //RGB<>BGR order
        data8(rowData[u*s*3+1]);
        data8(rowData[u*s*3+0]);
      }
    }
  }
  digitalWrite(LCDCS,HIGH);
}

void drawBitMap( uint8_t * bm,unsigned int x,unsigned int y,unsigned int w,unsigned int h){  //draw bm to x,y, cropping (centred) to w,h if needed
  unsigned int bmw, bmh,cvw,cvh;  //bitmap, canvas= actual area drawn
  unsigned int bmx=0,bmy=0;       //top left corner of drawn part of bitmap
  unsigned int u,v;               //counters
  BMPHeader_t* header;  
  uint8_t* rowData;
  header=(BMPHeader_t*)bm;  //header is at start of bitmap data
  bmw=abs(header->width);
  bmh=abs(header->height);
  if(bmw>w){cvw=w;bmx=(bmw-cvw)/2;}else{cvw=bmw;x=x+(w-bmw)/2;}
  if(bmh>h){cvh=h;bmy=(bmh-cvh)/2;}else{cvh=bmh;y=y+(h-bmh)/2;}
  setarea(x,y,x+cvw-1,y+cvh-1);
  digitalWrite(LCDCS,LOW);
  for(v=0;v<cvh;v++){
    rowData=getRow(header,v+bmy);
    if(rowData){
      for(u=0;u<cvw*3;u=u+3){   //RGB triplets
        data8(rowData[u+bmx*3+2]);  //RGB<>BGR order
        data8(rowData[u+bmx*3+1]);
        data8(rowData[u+bmx*3+0]);
      }
    }else{
      for(u=0;u<cvw*3;u++){
        data8(128);   //grey
      }
    }
  }
  digitalWrite(LCDCS,HIGH);
  Serial.printf("%d x %d bitmap cropped to %d x %d pixels for display.\r\n",bmw,bmh,cvw,cvh);
}

void setZoom(float z){
  unsigned int x;
  if(z<1){z=1;}
  if(z>4){z=4;}
  x=1024/z;
  setScale(x);
}

void setScale(unsigned int x){    //x-dimension of scanned area
  int unused   = 0;
  int cam_offset_x=512;
  int cam_offset_y=512;
  int cam_total_x=384;
  int cam_total_y=288;
  int cam_width  =256;
  int cam_height =192;
  cam_total_x=x;
  cam_total_y=(x*3)/4;
  cam_offset_x=((1600-cam_total_x)/2)&0xFFF8;   //multiple of 8
  cam_offset_y=((1200-cam_total_y)/2)&0xFFF8;   //multiple of 8
  e=s->set_res_raw(s, OV2640_MODE_UXGA, unused, unused, unused, cam_offset_x, cam_offset_y, cam_total_x, cam_total_y, cam_width, cam_height, unused, unused);      
}

void setFN(unsigned int n){  //array of format "/00000.BMP" in fileName
  unsigned int r=100000;
  char p=1;
  while(p<6){
    n=n%r;
    r=r/10;
    fileName[p]=n/r+'0';  
    p=p+1;
  }
}

bool getFreeFile(void){
  fileNumber=0;
  setFN(fileNumber);
  while(LittleFS.exists(fileName)&&(fileNumber<100000)){
    fileNumber=fileNumber+1;
    setFN(fileNumber);
  }
  if(LittleFS.exists(fileName)){
    return false;
  }else{
    return true;
  }
}

void reportFSstatus(void){
  char p[]="  0%";
  int n;
  if(root){
    Serial.println("File system OK");
    Serial.printf("%d bytes used out of %d total bytes.\r\n",LittleFS.usedBytes(),LittleFS.totalBytes());
    if(LittleFS.totalBytes()>100){
      n=(LittleFS.usedBytes()*100)/LittleFS.totalBytes();
      if(n>100){n=100;}
      if(n<0){n=0;}
    }else{
      n=100;
    }
    if(n>9){
      p[1]=((n/10)%10)+'0';
      if(n>99){
        p[0]=((n/100)%10)+'0';
      }
    }    
    p[2]=((n/1)%10)+'0';
    showarray(220,20,p,Arial_round_16x24,YELLOW,BLACK);            
  }else{
    Serial.println("File system not available");
    showarray(160,20,"----",Arial_round_16x24,YELLOW,BLACK);            
  }
}

int showFile(unsigned int n){
  uint8_t* out;     //for bitmap data
  size_t out_len;   //length of bitmap data
  int retval=0;
  setFN(n);
  currentFile=LittleFS.open(fileName,"r");
  if(currentFile){
    Serial.printf("File is %d bytes.\r\n",currentFile.size());
    if(currentFile.size()<=RAM_BUF_SIZE){
      int bCount=0;
      bCount=currentFile.read(ramFB,currentFile.size());
      if(currentFile.size()==bCount){
        Serial.println("File read OK");
        readFB->buf=&ramFB[sizeof(camera_fb_t)];  //data is stored just past camera_fb_t
        Serial.printf("Framebuffer is %d bytes, %d x %d pixels.\r\n",readFB->len, readFB->width, readFB->height);          
        if(frame2bmp(readFB,&out,&out_len)){    //convert from jpg to bmp
          drawBitMap(out,32,96,256,192);
          Serial.println("Image loaded OK.");
          showarray(80,292,fileName,Arial_round_16x24,YELLOW,BLACK);          
          retval=1;
        }else{
          Serial.println("Could not decode image.");
        }        
        free(out);                //release bitmap buffer  
      }else{
        Serial.println("Could not read file");
      }
    }else{
      Serial.printf("File too big for buffer %d/%d bytes.\r\n",currentFile.size(),RAM_BUF_SIZE);
    }
  }else{
    Serial.println("Could not open file.");
  }
  currentFile.close();      
  return retval;
}

void showZoom(float z){
  int n=z*10;
  char s[]="x1.0";
  s[1]=((n/10)%10)+'0';
  s[3]=((n/1)%10)+'0';
  showarray(32,20,s,Arial_round_16x24,YELLOW,BLACK);          
}