#include <avr/wdt.h>
#include "i2cmaster.h"
//#include "nunchuck.h"
#include "ArduinoWirelessNunchuk.h"
#include "TimerOne.h"
#include "CapacitiveSensor.h"
#include "XBee.h"

#define FALSE 0
#define TRUE 1

#define CHANNEL1ON  23
#define CHANNEL2ON  22
#define CHANNEL3ON  8
#define CHANNEL4ON  7

#define BLUE1       13
#define GREEN1      11
#define RED1        6

#define BLUE2       9
#define GREEN2      5
#define RED2        10

#define STROBE      LED_BUILTIN_RX

#define LED1_CHARLIE  A0
#define LED2_CHARLIE  A1
#define LED3_CHARLIE  A2

#define TOUCH_SEND      12
#define TOUCH_RECEIVE	4
#define TOUCH_TIMEOUT	50		//milliseconds
#define TOUCH_SAMPLES	2		//Samples to average
#define TOUCH_INTERVAL	100		//Milliseconds between readings. Does this need to go to zero?
#define TOUCH_THRESHOLD	10		

#define X_THRESHOLD       20      //Trigger the channel sweep
#define CHANNEL_SWEEP_PERIOD  100   //ms
#define MAG_THRESHOLD     5       //Joystick deadzone
#define STROBE_DURATION   100     // in millis  -- 100, 200, 1000, 2000, 4000 == 10Hz, 5Hz, 1Hz, .5Hz, and .125Hz
#define STROBE_DUTY       10      // on time, in millis
#define ACC_FACTOR        0.4     //Ratio of joystick magnitude to acceleration magnitude (higher is more accel)
#define GRAVITY           720
#define ACC_GAIN          2.0
#define Y_THRESHOLD       100

typedef enum {
  LOCAL_MANUAL,
  LOCAL_AUTO,  
  NUM_MODES
} MODES;

ArduinoWirelessNunchuk nunchuck = ArduinoWirelessNunchuk();

CapacitiveSensor touch_button = CapacitiveSensor(TOUCH_SEND, TOUCH_RECEIVE);

XBee xbee = XBee();
XBeeResponse response = XBeeResponse();
// create reusable response objects for responses we expect to handle 
ZBRxResponse rx = ZBRxResponse();
ModemStatusResponse msr = ModemStatusResponse();

int rgb_colors[3], old_c_button = LOW;  
uint16_t hue, auto_hue = 100, new_auto_hue = 0;
uint8_t saturation, brightness, auto_brightness = 100, new_auto_brightness = 0, old_mode = 0, channel_sweep = 0, colour_sweep_brightness = 0, colour_sweep_retrigger = 0;
volatile uint8_t strobe_flag = 0, strobe_duty = 0, mode = 0;
volatile uint16_t strobe_timer = 0;
unsigned long strobe_on = 0, strobe_off = 0, touch_sensor = 0, channel_sweep_timer = 0;

const byte dim_curve[] = {
    0,   1,   1,   2,   2,   2,   2,   2,   2,   3,   3,   3,   3,   3,   3,   3,
    3,   3,   3,   3,   3,   3,   3,   4,   4,   4,   4,   4,   4,   4,   4,   4,
    4,   4,   4,   5,   5,   5,   5,   5,   5,   5,   5,   5,   5,   6,   6,   6,
    6,   6,   6,   6,   6,   7,   7,   7,   7,   7,   7,   7,   8,   8,   8,   8,
    8,   8,   9,   9,   9,   9,   9,   9,   10,  10,  10,  10,  10,  11,  11,  11,
    11,  11,  12,  12,  12,  12,  12,  13,  13,  13,  13,  14,  14,  14,  14,  15,
    15,  15,  16,  16,  16,  16,  17,  17,  17,  18,  18,  18,  19,  19,  19,  20,
    20,  20,  21,  21,  22,  22,  22,  23,  23,  24,  24,  25,  25,  25,  26,  26,
    27,  27,  28,  28,  29,  29,  30,  30,  31,  32,  32,  33,  33,  34,  35,  35,
    36,  36,  37,  38,  38,  39,  40,  40,  41,  42,  43,  43,  44,  45,  46,  47,
    48,  48,  49,  50,  51,  52,  53,  54,  55,  56,  57,  58,  59,  60,  61,  62,
    63,  64,  65,  66,  68,  69,  70,  71,  73,  74,  75,  76,  78,  79,  81,  82,
    83,  85,  86,  88,  90,  91,  93,  94,  96,  98,  99,  101, 103, 105, 107, 109,
    110, 112, 114, 116, 118, 121, 123, 125, 127, 129, 132, 134, 136, 139, 141, 144,
    146, 149, 151, 154, 157, 159, 162, 165, 168, 171, 174, 177, 180, 183, 186, 190,
    193, 196, 200, 203, 207, 211, 214, 218, 222, 226, 230, 234, 238, 242, 248, 255,
};

void setup() {
  //wdt_enable(WDTO_250MS);

  pinMode(STROBE, OUTPUT); 
  digitalWrite(STROBE, LOW); 

  nunchuck.init();
  
  Serial.begin(57600);
  delay(100);
  //while(!Serial);
  Serial.println("B Speedie Enterprises: Nunchuck v2");

  //Serial1.begin(57600);
  //xbee.begin(Serial1);
  
  pinMode(CHANNEL1ON, OUTPUT);
  pinMode(CHANNEL2ON, OUTPUT);
  pinMode(CHANNEL3ON, OUTPUT);
  pinMode(CHANNEL4ON, OUTPUT);

  digitalWrite(CHANNEL1ON, HIGH);
  digitalWrite(CHANNEL2ON, HIGH);
  digitalWrite(CHANNEL3ON, HIGH);
  digitalWrite(CHANNEL4ON, HIGH);    
  
  //pinMode(BUTTON, INPUT_PULLUP);
  
  touch_button.set_CS_Timeout_Millis(TOUCH_TIMEOUT);

  //Timer1.initialize(100000);   //100ms period
  //Timer1.attachInterrupt(strobe_ISR); 

}

void loop() {   
  wdt_reset();

  bool button_pressed = FALSE;  
  touch_sensor = touch_button.capacitiveSensor(TOUCH_SAMPLES);    

	if (touch_sensor >= TOUCH_THRESHOLD) 
	{
		button_pressed = TRUE;
    mode++;   
    if (mode >= NUM_MODES) mode = 0; 
    delay(500); //debounce		
	}
  
  switch(mode){
    case LOCAL_MANUAL:   
      pinMode(LED1_CHARLIE, OUTPUT);
      pinMode(LED2_CHARLIE, INPUT);
      pinMode(LED3_CHARLIE, OUTPUT);

      digitalWrite(LED1_CHARLIE, HIGH);      
      digitalWrite(LED3_CHARLIE, LOW);

      nunchuck.update();      
    break;

    case LOCAL_AUTO:
      pinMode(LED1_CHARLIE, OUTPUT);
      pinMode(LED2_CHARLIE, INPUT);
      pinMode(LED3_CHARLIE, OUTPUT);

      digitalWrite(LED1_CHARLIE, LOW);      
      digitalWrite(LED3_CHARLIE, HIGH);
    break;
/*
    case 2:
      pinMode(LED1_CHARLIE, INPUT);
      pinMode(LED2_CHARLIE, OUTPUT);
      pinMode(LED3_CHARLIE, OUTPUT);

      digitalWrite(LED1_CHARLIE, LOW);      
      digitalWrite(LED3_CHARLIE, HIGH);     
    break;

    case 3:
      pinMode(LED1_CHARLIE, INPUT);
      pinMode(LED2_CHARLIE, OUTPUT);
      pinMode(LED3_CHARLIE, OUTPUT);
     
      digitalWrite(LED2_CHARLIE, HIGH);
      digitalWrite(LED3_CHARLIE, LOW);      
    break;

    case 4:
      pinMode(LED1_CHARLIE, OUTPUT);
      pinMode(LED2_CHARLIE, OUTPUT);
      pinMode(LED3_CHARLIE, INPUT);

      digitalWrite(LED1_CHARLIE, LOW);
      digitalWrite(LED2_CHARLIE, HIGH);     
    break;

    case 5:
      pinMode(LED1_CHARLIE, OUTPUT);
      pinMode(LED2_CHARLIE, OUTPUT);
      pinMode(LED3_CHARLIE, INPUT);

      digitalWrite(LED1_CHARLIE, HIGH);
      digitalWrite(LED2_CHARLIE, LOW);      
    break;
    */

    default:
      mode = 0;
  }

  //Treat joystick as cartesian plane where centre is (0,0)
  int32_t x_normalised = nunchuck.analogX - 0x7f;
  int32_t y_normalised = nunchuck.analogY - 0x7f;

  //Convert to polar
  uint8_t magnitude = sqrt(sq(x_normalised) + sq(y_normalised));
  int16_t angle = round(atan2(y_normalised, x_normalised) * 180/3.14159265); // radians to degrees and rounding
  //We want something in the range [0, 360]
  if (angle < 0) {
    angle += 360;
  }     
     
  // set HSB values 
  float joystick_component = map(constrain(magnitude, 0, 90), 0, 90, 0, 255) * (1-ACC_FACTOR);     //0 to 255 at 0% accel factor
  float acceleration_component = (nunchuck.accelZ - GRAVITY) * ACC_FACTOR;                                   //0 to 255 at 100% accel factor. Gravity included  
  
  //Gain up the acceleration to give good response from zero and also to stick to the rail a little bit
  acceleration_component *= ACC_GAIN;

  float brightness_f = joystick_component + acceleration_component;  
  brightness_f = constrain(brightness_f, 0, 255);
  brightness = (uint8_t) brightness_f;  
 
  //If holding down Z button, ignore gravity
  if (nunchuck.zButton) {
    brightness = map(constrain(magnitude, 0, 90), 0, 90, 0, 255);
  } 
    
  //Add a dead zone for the joystick so we don't get jitter
  if (magnitude < MAG_THRESHOLD) {
    brightness = 0;
    colour_sweep_retrigger = 0; //colour sweep cancels once the joystick goes to zero
  }
  
  hue        = angle;                              // hue is a number between 0 and 360
  saturation = 0xff;                               // saturation is a number between 0 - 255 

  //Push forward and back for colour sweep
  if (nunchuck.accelY < Y_THRESHOLD) {    
    colour_sweep_brightness = 0xff;   
    colour_sweep_retrigger = 1;
  }

  if (colour_sweep_brightness > 0) {
    brightness = colour_sweep_brightness;
    colour_sweep_brightness -= 5;
  }
  else {
    if (colour_sweep_retrigger) brightness = 0;
  }

  //AUTO MODE
  if (mode == LOCAL_AUTO) {
    hue = auto_hue;
    brightness = auto_brightness;
    delay(25);
  }  

  if (new_auto_hue == auto_hue) {
    new_auto_hue = random(0, 360);
  }

  if (new_auto_brightness == auto_brightness) {
    new_auto_brightness = random(100, 255);
  }
  
  if (new_auto_hue > auto_hue) {
    auto_hue++;
  }
  else {
    auto_hue--;
  }

  if (new_auto_brightness > auto_brightness) {
    auto_brightness++;
  }
  else {
    auto_brightness--;
  }  
  
  getRGB(hue,saturation,brightness,rgb_colors);   // converts HSB to RGB 
 /*
  analogWrite(RED1, rgb_colors[0]);            // red value in index 0 of rgb_colors array
  analogWrite(GREEN1, rgb_colors[1]);            // green value in index 1 of rgb_colors array
  analogWrite(BLUE1, rgb_colors[2]);            // blue value in index 2 of rgb_colors array  

  analogWrite(RED2, rgb_colors[0]);            // red value in index 0 of rgb_colors array
  analogWrite(GREEN2, rgb_colors[1]);            // green value in index 1 of rgb_colors array
  analogWrite(BLUE2, rgb_colors[2]);            // blue value in index 2 of rgb_colors array  
  */

  //Shake left and right for channel sweep
  if (nunchuck.accelX < X_THRESHOLD) {
    //prevent re-triggering
    if (channel_sweep == 0) {
      channel_sweep = 4;   
      channel_sweep_timer = millis();   
    }
  }
  switch(channel_sweep) {
    case 1:
      digitalWrite(CHANNEL1ON, HIGH); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 2:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, HIGH); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 3:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, HIGH); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 4:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, HIGH); 
    break;
    default:
      digitalWrite(CHANNEL1ON, HIGH); 
      digitalWrite(CHANNEL2ON, HIGH); 
      digitalWrite(CHANNEL3ON, HIGH); 
      digitalWrite(CHANNEL4ON, HIGH); 
    break;      
  }

  if (channel_sweep > 0) {
    if (millis() > (channel_sweep_timer + CHANNEL_SWEEP_PERIOD)) {  
    channel_sweep--;  
    channel_sweep_timer = millis();  
    }
  } 

  //Handle Strobe
  unsigned long now = millis();  
  if (nunchuck.cButton) {    
      if ((old_c_button == LOW)||(now > strobe_off)) {
        //c just pressed, or next cycle elapsed, set timers
        strobe_on = now + STROBE_DUTY;
        strobe_off = now + STROBE_DURATION;
      }
      if (now < strobe_on) {             
        digitalWrite(STROBE, HIGH);

        analogWrite(RED1, 0xff);            // red value in index 0 of rgb_colors array
        analogWrite(GREEN1, 0xff);            // green value in index 1 of rgb_colors array
        analogWrite(BLUE1, 0xff);            // blue value in index 2 of rgb_colors array  

        analogWrite(RED2, 0xff);            // red value in index 0 of rgb_colors array
        analogWrite(GREEN2, 0xff);            // green value in index 1 of rgb_colors array
        analogWrite(BLUE2, 0xff);            // blue value in index 2 of rgb_colors array  
      } 
      else {
        digitalWrite(STROBE, LOW);

        analogWrite(RED1, 0);            // red value in index 0 of rgb_colors array
        analogWrite(GREEN1, 0);            // green value in index 1 of rgb_colors array
        analogWrite(BLUE1, 0);            // blue value in index 2 of rgb_colors array  

        analogWrite(RED2, 0);            // red value in index 0 of rgb_colors array
        analogWrite(GREEN2, 0);            // green value in index 1 of rgb_colors array
        analogWrite(BLUE2, 0);            // blue value in index 2 of rgb_colors array  
      }
      old_c_button = HIGH;
  }
  else {
      digitalWrite(STROBE, LOW);
      old_c_button = LOW; 

      analogWrite(RED1, rgb_colors[0]);            // red value in index 0 of rgb_colors array
      analogWrite(GREEN1, rgb_colors[1]);            // green value in index 1 of rgb_colors array
      analogWrite(BLUE1, rgb_colors[2]);            // blue value in index 2 of rgb_colors array  

      analogWrite(RED2, rgb_colors[0]);            // red value in index 0 of rgb_colors array
      analogWrite(GREEN2, rgb_colors[1]);            // green value in index 1 of rgb_colors array
      analogWrite(BLUE2, rgb_colors[2]);            // blue value in index 2 of rgb_colors array  
  }   
/*
  digitalWrite(RED1, HIGH);            // red value in index 0 of rgb_colors array
  digitalWrite(GREEN1, HIGH);            // green value in index 1 of rgb_colors array
  digitalWrite(BLUE1, HIGH);            // blue value in index 2 of rgb_colors array  

  digitalWrite(RED2, HIGH);            // red value in index 0 of rgb_colors array
  digitalWrite(GREEN2, HIGH);            // green value in index 1 of rgb_colors array
  digitalWrite(BLUE2, HIGH);            // blue value in index 2 of rgb_colors array  
  */
}

void getRGB(int hue, int sat, int val, int colors[3]) { 
  /* from: https://www.kasperkamperman.com/blog/arduino/arduino-programming-hsb-to-rgb/
    convert hue, saturation and brightness ( HSB/HSV ) to RGB
     The dim_curve is used only on brightness/value and on saturation (inverted).
     This looks the most natural.      
  */
 
  val = dim_curve[val];
  sat = 255-dim_curve[255-sat];
 
  int r;
  int g;
  int b;
  int base;
 
  if (sat == 0) { // Acromatic color (gray). Hue doesn't mind.
    colors[0]=val;
    colors[1]=val;
    colors[2]=val;  
  } else  { 
 
    base = ((255 - sat) * val)>>8;
 
    switch(hue/60) {
  case 0:
    r = val;
    g = (((val-base)*hue)/60)+base;
    b = base;
  break;
 
  case 1:
    r = (((val-base)*(60-(hue%60)))/60)+base;
    g = val;
    b = base;
  break;
 
  case 2:
    r = base;
    g = val;
    b = (((val-base)*(hue%60))/60)+base;
  break;
 
  case 3:
    r = base;
    g = (((val-base)*(60-(hue%60)))/60)+base;
    b = val;
  break;
 
  case 4:
    r = (((val-base)*(hue%60))/60)+base;
    g = base;
    b = val;
  break;
 
  case 5:
    r = val;
    g = base;
    b = (((val-base)*(60-(hue%60)))/60)+base;
  break;
    }
 
    colors[0]=r;
    colors[1]=g;
    colors[2]=b; 
  }   
}
/*
void strobe_ISR() { 
  switch(channel_sweep) {
    case 1:
      digitalWrite(CHANNEL1ON, HIGH); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 2:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, HIGH); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 3:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, HIGH); 
      digitalWrite(CHANNEL4ON, LOW); 
    break;
    case 4:
      digitalWrite(CHANNEL1ON, LOW); 
      digitalWrite(CHANNEL2ON, LOW); 
      digitalWrite(CHANNEL3ON, LOW); 
      digitalWrite(CHANNEL4ON, HIGH); 
    break;
    default:
      digitalWrite(CHANNEL1ON, HIGH); 
      digitalWrite(CHANNEL2ON, HIGH); 
      digitalWrite(CHANNEL3ON, HIGH); 
      digitalWrite(CHANNEL4ON, HIGH); 
    break;      
  }
  if (channel_sweep > 0) {
    channel_sweep--;    
  }

  if (mode == 1) {
    //AUTO MODE
    //If old and new numbers are the same, generate new targets to fade to
    if (new_auto_hue == auto_hue) {
      new_auto_hue = random(0, 360);
    }
    if (new_auto_brightness == auto_brightness) {
      new_auto_brightness = random(100, 255);
    }
  
    if (new_auto_hue > auto_hue) {
      auto_hue++;
    }
    else {
      auto_hue--;
    }
    if (new_auto_brightness > auto_brightness) {
      auto_brightness++;
    }
    else {
      auto_brightness--;
    }  
  }
}*/
