/*
 * File:   MAX112XX.c
 * Author: ppros
 *
 * Created on 5 February 2022, 1:49 PM
 */

//##############################################################################
//    filename:        MAX112XX.c
//##############################################################################
//    ADC library for MAX 11207 and compatible ADCs  ...
//##############################################################################
//      Author:            	P.Prosser
//      Company:        	TGM
//      Adapted:            XX
//      Revision:        	0.1 (for XC16)
//      Date:         		Feb 2022
//     	Compiler            XC16 v1.7
//      MPLAB X             v6.00
//      MCC                 v3
//##############################################################################

#define FOSC    8000000UL
#define FCY     (FOSC/2)
#include "mcc_generated_files/system.h"    //ASO
#include <xc.h>
#include <libpic30.h> /* required for delays */
#include "Low_Ohms.h"
#include "MAX112XX.h"
#include <stdio.h> /* include I/O facilities */
#include "mcc_generated_files/pin_manager.h"
#include "mcc_generated_files/spi1_driver.h"
#include "LCD_lib_busy.h"
#include "mcc_generated_files/spi1_types.h"



//##############################################################################
//
// Byte Write to the ADC requires:
// 1. Assert CS Low
// 2. Write SPI in normal fashion
//        SDO Data out   Tds prior to SCLK
//        SCLK rising edge clocks data into ADC 
//        Hold SDO Tdh after clk
// This is a BYTE WRITE ONLY function
// CS is de-asserted after CMD plus BYTE, use READ to get data back out!
//##############################################################################
unsigned char MAX112XX_Register_Write_Byte(uint8_t Register_To_Write, uint8_t Value_To_Write)
{
    uint8_t b;
    // initialise the ADC
    
    
    MAX112XX_CS_Assert;
    __delay_us(MAX112XX_Tiny_Delay_us);
    b = spi1_exchangeByte(Register_To_Write); /* dont care about b */
    __delay_us(MAX112XX_Tiny_Delay_us);
    /* now write to SPI the data for register */
    b = spi1_exchangeByte(Value_To_Write); /* dont care about b */
    __delay_us(MAX112XX_Tiny_Delay_us);
    MAX112XX_CS_DeAssert;
    return(1);
}

//##############################################################################
//
// Write byte to the ADC requires:
// 1. Assert CS Low
// 2. Write SPI in normal fashion
//        SDO Data out   Tds prior to SCLK
//        SCLK rising edge clocks data into ADC 
//        Hold SDO Tdh after clk
// This is a BYTE WRITE ONLY function
// CS is de-asserted after CMD plus BYTE, use READ to get data back out!
//##############################################################################
unsigned char MAX112XX_Write_Byte(uint8_t Value_To_Write)
{
    uint8_t b;
    
    MAX112XX_CS_Assert;
    __delay_us(MAX112XX_Tiny_Delay_us);
    b = spi1_exchangeByte(Value_To_Write); /* dont care about b */
    __delay_us(MAX112XX_Tiny_Delay_us);
    MAX112XX_CS_DeAssert;
    return(1);
}


//##############################################################################
//
// Word Write to the ADC requires:
// 1. Assert CS Low
// 2. Write SPI in normal fashion
//        SDO Data out   Tds prior to SCLK
//        SCLK rising edge clocks data into ADC 
//        Hold SDO Tdh after clk
// This is a WRITE ONLY function
// CS is deasserted after CMD plus 24 bit word, use READ to get data back out!
//##############################################################################
unsigned char MAX112XX_Register_Write_Word(uint8_t Register_To_Write, long int Value_To_Write)
{
    long int TMP;
    uint8_t TMP_Val;
    
    // initialise the ADC
    MAX112XX_CS_Assert;
    __delay_us(MAX112XX_Tiny_Delay_us);
    spi1_writeByte(Register_To_Write);
    spi1_readByte(); /* clear read char from buffer */
    __delay_us(MAX112XX_Tiny_Delay_us);
    /* first byte */
    TMP = Value_To_Write & 0x00ff0000;
    TMP = TMP >> 16;
    TMP_Val = (uint8_t) TMP;
    spi1_writeByte(TMP_Val);
    spi1_readByte(); /* clear read char from buffer */
/* second byte */
    TMP = Value_To_Write & 0x0000ff00;
    TMP = TMP >> 8;
    TMP_Val = (uint8_t) TMP;
    spi1_writeByte(TMP_Val);
    spi1_readByte(); /* clear read char from buffer */
/* final byte */
    TMP = Value_To_Write & 0x000000ff;
    TMP_Val = (uint8_t) TMP;
    spi1_writeByte(TMP_Val);
    spi1_readByte(); /* clear read char from buffer */
__delay_us(MAX112XX_Tiny_Delay_us);
    MAX112XX_CS_DeAssert;
    return(1);
}


//##############################################################################
//
// Byte READ from the ADC requires:
// 1. Assert CS Low
// 2. Write SPI in normal fashion with correct command asking to read something
//        SDO Data out   Tds prior to SCLK
//        SCLK rising edge clocks data into ADC 
//        Hold SDO Tdh after clk
// This is a BYTE READ ONLY function
// CS is de-asserted after CMD plus BYTE READ in
//##############################################################################
unsigned char MAX112XX_Register_Read_Byte(uint8_t Register_To_Read, unsigned char *Value_Read)
{ uint8_t b;
    // initialise the ADC
    MAX112XX_CS_Assert;
    __delay_us(MAX112XX_Tiny_Delay_us);
    b = spi1_exchangeByte(Register_To_Read);   /* dont care about b */
    __delay_us(MAX112XX_Tiny_Delay_us);
    *Value_Read = spi1_exchangeByte(b);   /* dont care about b */
    __delay_us(MAX112XX_Tiny_Delay_us);
    MAX112XX_CS_DeAssert;
    return(1);
}


//##############################################################################
//
// ADC READ 5SPS:
// Read ADC
// - Mode 0
// - 5SPS rate
// - Return data
//##############################################################################
unsigned char MAX112XX_Read_ADC_5SPS(long int *ADC_Value_Read)
{ 
    long int r, cnt = 0;
    char Read_Val;

    MAX112XX_Write_Byte(MAX112XX_Convert_5SPS);
    // wait before checking if ADC done - minimise bus noise
    __delay_ms(MAX112XX_200ms_Read_Delay_ms);
        do{
    MAX112XX_Register_Read_Byte(MAX112XX_Stat1_RD_CMD, &Read_Val);
    if (cnt > MAX112XX_Read_Loop_Max)
        LCD_TextOut(0,0,(unsigned char *)" ADC Read Error ");   // could be optimized    
    cnt++;
    } while(Read_Val & MAX112XX_Stat_Bit);
    MAX112XX_Register_Read_Word(MAX112XX_Data_Read_Cmd , (unsigned long int *) &r);
    *ADC_Value_Read = r;
    return(1);
}

//##############################################################################
//
// ADC READ 10SPS:
// Read ADC
// - Mode 0
// - 10SPS rate
// - Return data
//##############################################################################
unsigned char MAX112XX_Read_ADC_10SPS(long int *ADC_Value_Read)
{ 
    long int r;
    char Read_Val;

    MAX112XX_Write_Byte(MAX112XX_Convert_10SPS);
    // wait before checking if ADC done - minimise bus noise
    __delay_ms(MAX112XX_100ms_Read_Delay_ms);
        do{
    MAX112XX_Register_Read_Byte(MAX112XX_Stat1_RD_CMD, (unsigned char *) &Read_Val);
    LCD_TextOut(0,0,(unsigned char *)"ERR - Reading   ");   // could be optimized    
    } while(Read_Val & MAX112XX_Stat_Bit);
    MAX112XX_Register_Read_Word(MAX112XX_Data_Read_Cmd , (unsigned long int *)&r);
    *ADC_Value_Read = r;
    return(1);
}



//##############################################################################
//
// Word READ from the ADC requires:
// 1. Assert CS Low
// 2. Write SPI in normal fashion with correct command asking to read something
//        SDO Data out   Tds prior to SCLK
//        SCLK rising edge clocks data into ADC 
//        Hold SDO Tdh after clk
// This is a WORD READ ONLY function
// CS is de-asserted after CMD plus three BYTE READS 
//##############################################################################
unsigned char MAX112XX_Register_Read_Word(uint8_t Register_To_Read, unsigned long int *Value_Read)
{
    uint8_t b;
    
    MAX112XX_CS_Assert;
    __delay_us(MAX112XX_Tiny_Delay_us);
    b = spi1_exchangeByte(Register_To_Read);
    __delay_us(MAX112XX_Tiny_Delay_us);
    /* first byte */
    b = spi1_exchangeByte(0);
    *Value_Read = b;
    *Value_Read = *Value_Read <<8;
    __delay_us(MAX112XX_Tiny_Delay_us);
    /* second byte */
    b = spi1_exchangeByte(0);
    *Value_Read += b;
    *Value_Read = *Value_Read <<8;
    __delay_us(MAX112XX_Tiny_Delay_us);
    /* third byte */
    b = spi1_exchangeByte(0);
    *Value_Read += b;

    MAX112XX_CS_DeAssert;

    return(1);
}




//##############################################################################
//
// Initialise the ADC requires:
// 1. Power up Delay by ADC_Power_Up_Delay_ms
// 2.
//
//##############################################################################
unsigned char MAX112XX_Init(void)
{
//    char Temp_String_1[20];
    
    __delay_ms(MAX112XX_Power_Up_Delay_ms);

    // initialise the ADC Interface
    MAX112XX_CS_DeAssert;   /* start with CS deasserted */     
    // Set MAX112XX CTRL1 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl1_WR_CMD, MAX112XX_Ctrl1_Unipolar_Default);
    // Set MAX112XX CTRL3 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Write MAX112XX_Convert_5SPS to device sets 5SPS
    MAX112XX_Write_Byte(MAX112XX_Self_Cal_5SPS);
    //wait while calibrates
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    // Set MAX112XX CTRL2 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl2_WR_CMD, MAX112XX_Ctrl2_Default);
    // Note MAX112XX  System Offset Calibration is NOT USED
    // Note MAX112XX   System Gain Calibration is NOT USED
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Note MAX112XX  Self-calibration Offset is of interest only
    // Note MAX112XX  Self-calibration Gain register is of interest only
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    MAX112XX_Write_Byte(MAX112XX_Self_Cal_5SPS);
    //wait while calibrates
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    return(1);
}


//##############################################################################
//
// Initialise the ADC in Bipolar mode requires:
// 1. Power up Delay by ADC_Power_Up_Delay_ms
// 2.
//
//##############################################################################
unsigned char MAX112XX_Bipolar_Set(void)
{
//    char Temp_String_1[20];
    
    // initialise the ADC Interface
    MAX112XX_CS_DeAssert;   /* start with CS deasserted */     
    // Set MAX112XX CTRL1 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl1_WR_CMD, MAX112XX_Ctrl1_Bipolar_Default);
    // Set MAX112XX CTRL3 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Write MAX112XX_Convert_5SPS to device sets 5SPS
    MAX112XX_Write_Byte(MAX112XX_Self_Cal_5SPS);
    //wait while calibrates
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    // Set MAX112XX CTRL2 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl2_WR_CMD, MAX112XX_Ctrl2_Default);
    // Note MAX112XX  System Offset Calibration is NOT USED
    // Note MAX112XX   System Gain Calibration is NOT USED
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Note MAX112XX  Self-calibration Offset is of interest only
    // Note MAX112XX  Self-calibration Gain register is of interest only
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    return(1);
}


//##############################################################################
//
// Initialise the ADC in Unipolar Mode requires:
// 1. Power up Delay by ADC_Power_Up_Delay_ms
// 2.
//
//##############################################################################
unsigned char MAX112XX_Unipolar_Set(void)
{
//    char Temp_String_1[20];
    
    // initialise the ADC Interface
    MAX112XX_CS_DeAssert;   /* start with CS deasserted */     
    // Set MAX112XX CTRL1 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl1_WR_CMD, MAX112XX_Ctrl1_Unipolar_Default);
    // Set MAX112XX CTRL3 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Write MAX112XX_Convert_5SPS to device sets 5SPS
    MAX112XX_Write_Byte(MAX112XX_Self_Cal_5SPS);
    //wait while calibrates
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    // Set MAX112XX CTRL2 Register - we need to set this
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl2_WR_CMD, MAX112XX_Ctrl2_Default);
    // Note MAX112XX  System Offset Calibration is NOT USED
    // Note MAX112XX   System Gain Calibration is NOT USED
    MAX112XX_Register_Write_Byte(MAX112XX_Ctrl3_WR_CMD, MAX112XX_Ctrl3_Default);
    // Note MAX112XX  Self-calibration Offset is of interest only
    // Note MAX112XX  Self-calibration Gain register is of interest only
    __delay_ms(MAX112XX_Self_Cal_Time_ms);
    return(1);
}



