#ifndef DCC_H
#define	DCC_H

//also includes SLIP processor

#include <xc.h> 
#include "portbits.h"
#include "globalPins.h"
#include "uart.h"

#define PACKET_MAX_BYTES 8
#define DCC_FORWARD 1
#define DCC_REVERSE 0
#define DCC_PREAMBLE_BITS 11
#define DCC_PACKET_TIMEOUT_COUNT (1363)

//SLIP
#define SLIP_END (192)
#define SLIP_ESC (219)
#define SLIP_ESC_END (220)
#define SLIP_ESC_ESC (221)
#define SLIP_PACKET_SIZE 32
#define CHAINID_INVALID (0xFF)
#define SLIP_DCC_IMMEDIATE ('B')
#define SLIP_HOST_CHECK ('C')
#define SLIP_HOST_RESPONSE ('D')
#define SLIP_SYSTEM_COMMAND ('I')
#define SLIP_SYSTEM_COMMAND_OFF (0)
#define SLIP_SYSTEM_COMMAND_ON  (1)

//Host check parameters
#define COUNTER_INTERVAL 125
#define TRIP_TIME 900
#define DCC_PACKET_INTERVAL 250
#define HOST_CHECK_INTERVAL 5000

typedef struct{
  unsigned int address; //XC8 can't do 14-bit bitfield, so use unsigned int
  unsigned int useLongAddress:1;
  unsigned int speed:7; 
  unsigned int dir:1;   //1=forward
  unsigned int F0:1;
  unsigned int F1:1;
  unsigned int F2:1;
  unsigned int F3:1;
  unsigned int F4:1;
} loco_t;

//packet contents
typedef struct {
  uint8_t byteCount;
  uint8_t data[PACKET_MAX_BYTES];
} dccPacket_t;

//SLIP
typedef struct  {  
  unsigned char byteCount;
  char data[SLIP_PACKET_SIZE];
} slipPacket_t;

typedef struct {
  unsigned char byte1;
  unsigned char byte2;
} slip_bytes_t;

extern slipPacket_t packetsToHost;
extern slipPacket_t packetsFromHost;

//remotes
#define REMOTE_MAX 5
#define REMOTE_TTL_REFRESH 2

typedef struct  {  
  char index;
  char id;
  char ttl;
} remote_t;

extern remote_t remotes[REMOTE_MAX];
extern char remoteCount;
extern char packetRX;
extern char dccSwitch;    //user control
extern dccPacket_t dccRx;

void idlePacket(dccPacket_t* p);  //load idle packet into p
void resetPacket(dccPacket_t* p); //load reset packet into p
void addChecksum(dccPacket_t* p); //add checksum byte
void addSpeed128(dccPacket_t* p, loco_t loco);
void addF04(dccPacket_t* p, loco_t loco);
void setAddress(dccPacket_t* p, loco_t loco);
void speedPacket128(dccPacket_t* p, loco_t loco);
void F04Packet(dccPacket_t* p, loco_t loco);
void DCCopsLongWrite(dccPacket_t* p,loco_t loco,int cv, unsigned char data);  //NB: address (from loco) needed s-9.2.1_dcc_extended_packet_formats: s2.3.7.3 Configuration Variable Access Instruction - Long Form
char checkPacketMatch(dccPacket_t* p1, dccPacket_t* p2);    //simple equality over valid bytes (ie up to len)
//SLIP
char processFeed(slipPacket_t* p, char c); //feed c into p, return true completion
slip_bytes_t slipEncode(unsigned char b); //SLIP encoding
void sendSlipByte( void(*f)(const char),slip_bytes_t s);
void sendSLIPpacket(void(*f)(const char),slipPacket_t* p);
unsigned char slipChecksum(slipPacket_t* p);
void addSlipChecksum(slipPacket_t* p);
void dccInit(void); //set up for programmed DCC output
void dccTXISR(void);  //call at 58us intervals
void sendHostCheck(void);
void handleSLIP(void);
void dccRXISR(void);    //DCC receive ISR
void t4init(void);      //22us DCC packet receive
void cleanPacket(dccPacket_t* p);    //check checksum, void if invalid

#endif	/* DCC_H */

