#include "dcc.h"
char dccSwitch=0;    //user control

slip_bytes_t slipEnd={SLIP_END,0};  //need to use this so it doesn't get encoded
slipPacket_t packetsToHost={0,""};
slipPacket_t packetsFromHost={0,""};

remote_t remotes[REMOTE_MAX];
char remoteCount=0;
dccPacket_t thisPacket,nextPacket,pktIdle,dccIn,dccRx;
const uint8_t bitmasks[]={0,128,64,32,16,8,4,2,1}; //leading 0 supplies byte start, trailing 1 comes from next preamble
char packetRX=0;
unsigned int packetTimeout=0;

void resetPacket(dccPacket_t* p){
  p->data[0]=0;
  p->data[1]=0;
  p->byteCount=2;
  addChecksum(p);
}

void idlePacket(dccPacket_t* p){
  p->data[0]=255;
  p->data[1]=0;
  p->byteCount=2;
  addChecksum(p);
}

void addChecksum(dccPacket_t* p){
  uint8_t cx=0;
  uint8_t i;
  for(i=0;i<p->byteCount;i++){cx=cx^p->data[i];}
  p->data[p->byteCount]=cx;
  p->byteCount++;
}

void addSpeed128(dccPacket_t* p, loco_t loco){
  p->data[p->byteCount]=0x3F;   //128 step
  p->byteCount++;
  if(loco.dir){
    p->data[p->byteCount]=(loco.speed&0x7F)|128;
  }else{
    p->data[p->byteCount]=(loco.speed&0x7F);
  }
  p->byteCount++;
  addChecksum(p);
}

void addF04(dccPacket_t* p, loco_t loco){
  p->data[p->byteCount]=0x80; //F0-F0 packet
  if(loco.F0){p->data[p->byteCount]=p->data[p->byteCount]|0x10;}
  if(loco.F1){p->data[p->byteCount]=p->data[p->byteCount]|0x01;}
  if(loco.F2){p->data[p->byteCount]=p->data[p->byteCount]|0x02;}
  if(loco.F3){p->data[p->byteCount]=p->data[p->byteCount]|0x04;}
  if(loco.F4){p->data[p->byteCount]=p->data[p->byteCount]|0x08;}
  p->byteCount++;
  addChecksum(p);
}

void setAddress(dccPacket_t* p, loco_t loco){
  if(loco.useLongAddress){
    p->data[0]=0xC0|((loco.address>>8)&0x3F);
    p->data[1]=loco.address&0xFF;    
    p->byteCount=2;
  }else{
    p->data[0]=loco.address&0x7F;
    p->byteCount=1;
  }
}

void speedPacket128(dccPacket_t* p, loco_t loco){
  setAddress(p,loco);
  addSpeed128(p,loco);
}

void F04Packet(dccPacket_t* p, loco_t loco){
  setAddress(p,loco);
  addF04(p,loco);
}

//operations mode programming
void DCCopsLongWrite(dccPacket_t* p,loco_t loco,int cv, unsigned char data){  //s-9.2.1_dcc_extended_packet_formats: s2.3.7.3 Configuration Variable Access Instruction - Long Form
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  setAddress(p,loco);  
  p->data[p->byteCount+0]=0xEC|((cv>>8)&0x3); //top two bits of cv
  p->data[p->byteCount+1]=cv&0xFF; //lower 8 bits of cv
  p->data[p->byteCount+2]=data; //value
  p->byteCount=p->byteCount+3;
  addChecksum(p);  
}

char checkPacketMatch(dccPacket_t* p1, dccPacket_t* p2){    //simple equality over valid bytes (up to len)
    unsigned char n=p1->byteCount;
    unsigned char i;
    if(n!=p2->byteCount){return 0;}
    for(i=0;i<n;i++){if(p1->data[0]!=p2->data[0]){return 0;}}
    return 1;
}

//SLIP
char processFeed(slipPacket_t* p, char c){ //feed c into p, return true on completion
  //p is working
  static char last=0;
  if(c==SLIP_END){
    last=0; //reset
    if(slipChecksum(p)==0){
        if(p->byteCount==0){
            return 0;   //empty packet
        }else{
            return 1; //p is complete                    
        }
    }else{
        p->byteCount=0;
        return 0;   //cx error, blank
    }
  }else if(c==SLIP_ESC){
    last=SLIP_ESC;        //nothing added
  }else{
    if(last==SLIP_ESC){
      if(c==SLIP_ESC_END){
        p->data[p->byteCount]=SLIP_END;
        p->byteCount++;
      }else if(c==SLIP_ESC_ESC){
        p->data[p->byteCount]=SLIP_ESC;
        p->byteCount++;        
      } //otherwise ignore, unknown state
    }else{
      p->data[p->byteCount]=c;
      p->byteCount++;
    }
    last=0;
    if(p->byteCount>(SLIP_PACKET_SIZE-2)){p->byteCount=SLIP_PACKET_SIZE-2;} //avoid overflow, drop extra    
  }
  return 0;
}

slip_bytes_t slipEncode(unsigned char b){ //SLIP encoding
    slip_bytes_t r;
    r.byte1=b;  //default
    r.byte2=0;
    if(b==SLIP_END){
        r.byte1=SLIP_ESC;
        r.byte2=SLIP_ESC_END;
    }else if(b==SLIP_ESC){
        r.byte1=SLIP_ESC;
        r.byte2=SLIP_ESC_ESC;        
    }
    return r;
}

void sendSlipByte(void(*f)(const char),slip_bytes_t s){
    (*f)(s.byte1);
    if(s.byte2){
        (*f)(s.byte2);
    }
}

void sendSLIPpacket(void(*f)(const char),slipPacket_t* p){  //assume has cx
    char i;
    for(i=0;i<p->byteCount;i++){
        sendSlipByte(f,slipEncode(p->data[i]));
    }    
    sendSlipByte(f,slipEnd);
}

unsigned char slipChecksum(slipPacket_t* p){
  int i;
  unsigned char cx=0;
  for(i=0;i<p->byteCount;i++){
    cx=cx^p->data[i];
  }
  return cx;
}

void addSlipChecksum(slipPacket_t* p){
  unsigned char cx;
  cx=slipChecksum(p);
  p->data[p->byteCount]=cx;
  p->byteCount++;
}

void dccInit(void){
    idlePacket(&pktIdle);
    //all outputs off
    anselbits.DCCOUT_EN=0;
    trisbits.DCCOUT_EN=0;
    latbits.DCCOUT_EN=0;
    anselbits.DCCOUT_A=0;
    trisbits.DCCOUT_A=0;
    latbits.DCCOUT_A=0;
    anselbits.DCCOUT_B=0;
    trisbits.DCCOUT_B=0;
    latbits.DCCOUT_B=0;
    unsigned char intcon=INTCON;
    GIE=0;
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 0; //clear PPSLOCKED bit
    pps.DCCOUT_A=PPSO_LAT;
    pps.DCCOUT_B=PPSO_LAT;
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 1; //set PPSLOCKED bit           
    INTCON=intcon;   
    //T2 provides 58us interrupt
    T2CON=0;                //1:1 PS and OUTPS
    T2HLT=0;                //free running, not gated
    T2CLKCON=1;             //FOSC/4 = 8MHz
    T2TMR=0;
    T2CONbits.CKPS=3;       //=> 1MHz
    T2PR=57;                //58us
    T2CONbits.ON=1;         //run
    TMR2IF=0;               //clear
    TMR2IE=1;               //enable
    PEIE=1;
}

void dccTXISR(void){  //call at 58us intervals
    static char phase=0;
    static char bitptr=0;
    static signed char byteptr=-2; //preamble
    static char bit=1;
    if(phase==2){
        latbits.DCCOUT_A=0;
        latbits.DCCOUT_B=1;
    }else if(phase>3){
        latbits.DCCOUT_B=0;
        latbits.DCCOUT_A=1;
        phase=0;
        bitptr++;      //next bit
        if(bitptr>8){  //next byte
            bitptr=0;
            byteptr++;
            if(byteptr>=thisPacket.byteCount){ //next packet
                byteptr=-2;
                //get next packet
                if(nextPacket.byteCount){
                    thisPacket=nextPacket;
                    nextPacket.byteCount=0; //flag that it has been consumed
                }else{
                    thisPacket=pktIdle;
                }
            }
        }
    }
    if(phase==0){                             //only do calcs at start of bit
        if(byteptr<0){
            bit=1;                          //preamble bits are always one
        }else{
            bit=(thisPacket.data[byteptr]&bitmasks[bitptr])?1:0;
        }
    }
    phase++;
    phase=phase+bit; //double skip for 1's to give shorter bit time
}

void sendHostCheck(void){
    char i;
    packetsFromHost.data[0]=SLIP_HOST_CHECK;
    packetsFromHost.data[1]=1;
    packetsFromHost.byteCount=2;
    addSlipChecksum(&packetsFromHost);   //rejig cx                   
    sendSLIPpacket(&uart1_send,&packetsFromHost);
    remoteCount=0;
    for(i=0;i<REMOTE_MAX;i++){
        if(remotes[i].ttl){
            remotes[i].ttl--;
            remoteCount++;  //count active
        }else{
            remotes[i].index=0;
            remotes[i].id=0;
        }
    }
}

void handleSLIP(void){
    char i;
    while(uart1_available()){            
        if(processFeed(&packetsToHost,uart1_receive())){
            packetRX=1;
            if(packetsToHost.data[0]==SLIP_DCC_IMMEDIATE){ //DCC packet
                if(nextPacket.byteCount==0){    //space available
                    if(packetsToHost.byteCount<(PACKET_MAX_BYTES+2)){
                        for(i=0;i<packetsToHost.byteCount-2;i++){
                            nextPacket.data[i]=packetsToHost.data[i+1];
                            nextPacket.byteCount=packetsToHost.byteCount-2;
                        }
                    }                    
                }
            }
            if(packetsToHost.data[0]==SLIP_HOST_RESPONSE){ //host response
                if(packetsToHost.data[1]<REMOTE_MAX){
                    remotes[packetsToHost.data[1]].index=packetsToHost.data[1];
                    remotes[packetsToHost.data[1]].id=packetsToHost.data[2];
                    remotes[packetsToHost.data[1]].ttl=REMOTE_TTL_REFRESH;
                }
            }
            if(packetsToHost.data[0]==SLIP_SYSTEM_COMMAND){ //sys command
                if(packetsToHost.data[1]==SLIP_SYSTEM_COMMAND_ON){dccSwitch=1;}
                if(packetsToHost.data[1]==SLIP_SYSTEM_COMMAND_OFF){dccSwitch=0;}
            }
            packetsToHost.byteCount=0;  //reset
        }
    }    
}

void dccRXISR(void){
    //bit level logic
    static unsigned char now=0;
    static unsigned char last=0;
    static unsigned char count=0;
    static unsigned char thisBit=0;
    //for preamble/stop/start detection
    static signed char bitcount=-DCC_PREAMBLE_BITS; //<0 means preamble
    static unsigned char bitmask=128;
    static unsigned char thisByte=0;
    static unsigned char usePlusInput=0;
    if(usePlusInput){
        if(portbits.DCCIN_A){now=1;}else{now=0;}
    }else{
        if(portbits.DCCIN_B){now=0;}else{now=1;}
    }    
    if(now){
        count++;
    }else{
        if(last){ //falling edge, act 
            if(count<4){        //1 bit
                thisBit=1;
            }else{  //0 bit
                thisBit=0;
            }
            if(bitcount<0){     //still counting preamble bits                    
                if(thisBit){
                    bitcount++;     //advance count
                }else{
                    bitcount=-DCC_PREAMBLE_BITS;    //reset preamble count
                }
            }else if(bitcount==0){  //ready for start 0
                if(thisBit==0){     //start
                    bitcount=1;
                    bitmask=128;
                    thisByte=0;
                }           //otherwise keep count at 0
            }else{          //handle data bits and subsequent start bits
                if(thisBit){
                    thisByte=thisByte|bitmask;
                }
                bitmask=bitmask>>1;                    
                bitcount++;
                if(bitcount>9){ //end of byte
                    dccIn.data[dccIn.byteCount]=thisByte; //add to outgoing
                    dccIn.byteCount++;
                    bitcount=bitcount-9;    //move back (should be to 1)
                    bitmask=128;
                    thisByte=0;
                    if(dccIn.byteCount>=PACKET_MAX_BYTES){dccIn.byteCount=PACKET_MAX_BYTES-1;}  //clip
                    if(thisBit){    //end of packet
                        if(dccRx.byteCount==0){  //if buffer full, ignore
                            dccRx=dccIn;
                        }
                        dccIn.byteCount=0;
                        bitcount=-DCC_PREAMBLE_BITS;    //reset preamble count
                    }                        
                }
            }
        }
        count=0;
    }
    last=now;
    packetTimeout++;
    if(packetTimeout>DCC_PACKET_TIMEOUT_COUNT){
        packetTimeout=0;
        if(usePlusInput){   //toggle rail if we don't see anything on this rail
            usePlusInput=0;
        }else{
            usePlusInput=1;                
        }
    }
}

void t4init(void){
    //T4 provides 22us interrupt
    T4CON=0;                //1:1 PS and OUTPS
    T4HLT=0;                //free running, not gated
    T4CLKCON=1;             //FOSC/4 = 8MHz
    T4TMR=0;
    T4CONbits.CKPS=3;       //=> 1MHz
    T4PR=21;                //22us
    T4CONbits.ON=1;         //run
    TMR4IF=0;               //clear
    TMR4IE=1;               //enable
    PEIE=1;    
}

void cleanPacket(dccPacket_t* p){    //check checksum, void if invalid
    unsigned char c=0;
    unsigned char i;
    unsigned char len=p->byteCount;    
    for(i=0;i<len;i++){c=c^p->data[i];}
    if(c){
        p->byteCount=0;
    }else{
        packetTimeout=0;
    }
}
