#include "util.h"

slip_bytes_t slipEnd={SLIP_END,0};  //need to use this so it doesn't get encoded
volatile unsigned char minTimers[LOCO_COUNT];
volatile unsigned char maxTimers[LOCO_COUNT];
volatile unsigned char screenTick=0;
volatile unsigned char tickDone=0;
volatile unsigned char heartbeatTimer=HEARTBEAT_INTERVAL/TICK_INTERVAL;
unsigned char chainID=CHAINID_INVALID;

slipPacket_t packetsToHost={0,""};
slipPacket_t packetsFromHost={0,""};

loco_t locos[LOCO_COUNT];
loco_t curLoco;
char updateFlags[LOCO_COUNT];
dccPacket_t speedPacket;
dccPacket_t functionPacket;
char potActive[LOCO_COUNT];   //stopped or using pot
char potMatched=0;  //is pot ready on this screen?
const char* sysMsg="";
__eeprom unsigned char f1Mode=FUNCTION_MOM;
__eeprom unsigned char f2Mode=FUNCTION_MOM;
__eeprom loco_t savedLocos[LOCO_COUNT]; //zeros are fine

unsigned char getSpeed(void){
    static unsigned int last=0;
    unsigned int cur=getADC(VPOT);
    if(labs(((long)last)-((long)cur))>512){
        last=cur;
    }
    return (unsigned char)(last/512);
}

unsigned char getTwoDigits(void){
    static unsigned int last=0;
    unsigned char ret=0;
    unsigned int cur=getADC(VPOT);
    if(labs(((long)last)-((long)cur))>650){ //hysteresis
        last=cur;
    }
    ret=(unsigned char)(last/656);
    if(ret==1){ret=0;}  //avoid sending speed step 1
    return ret;
}

void sendDCCPacket(dccPacket_t* p){    //SLIP
    char i;
    sendSlipByte(&uart1_send,slipEncode('B'));
    for(i=0;i<p->byteCount;i++){
        sendSlipByte(&uart1_send,slipEncode(p->data[i]));
    }
    sendSlipByte(&uart1_send,slipEncode('B'));  //checksum of DCC packet should be zero, so this works
    sendSlipByte(&uart1_send,slipEnd);
}

slip_bytes_t slipEncode(unsigned char b){ //SLIP encoding
    slip_bytes_t r;
    r.byte1=b;  //default
    r.byte2=0;
    if(b==SLIP_END){
        r.byte1=SLIP_ESC;
        r.byte2=SLIP_ESC_END;
    }else if(b==SLIP_ESC){
        r.byte1=SLIP_ESC;
        r.byte2=SLIP_ESC_ESC;        
    }
    return r;
}

void sendSlipByte(void(*f)(const char),slip_bytes_t s){
    (*f)(s.byte1);
    if(s.byte2){
        (*f)(s.byte2);
    }
}

char processFeed(slipPacket_t* p, char c){ //feed c into p, return true on completion
  //p is working
  static char last=0;
  if(c==SLIP_END){
    last=0; //reset
    if(slipChecksum(p)==0){
        return 1; //p is complete        
    }else{
        p->byteCount=0;
        return 0;   //cx error, blank
    }
  }else if(c==SLIP_ESC){
    last=SLIP_ESC;        //nothing added
  }else{
    if(last==SLIP_ESC){
      if(c==SLIP_ESC_END){
        p->data[p->byteCount]=SLIP_END;
        p->byteCount++;
      }else if(c==SLIP_ESC_ESC){
        p->data[p->byteCount]=SLIP_ESC;
        p->byteCount++;        
      } //otherwise ignore, unknown state
    }else{
      p->data[p->byteCount]=c;
      p->byteCount++;
    }
    last=0;
    if(p->byteCount>(SLIP_PACKET_SIZE-2)){p->byteCount=SLIP_PACKET_SIZE-2;} //avoid overflow, drop extra    
  }
  return 0;
}

char getSLIPlen(slipPacket_t* p){   //how many bytes needed on wire
    char n=0;
    char i;
    for(i=0;i<p->byteCount;i++){
        if(slipEncode(p->data[i]).byte2){n++;}
        n++;
    }
    return n+1;    //+1 for trailing END
}

void sendSLIPpacket(void(*f)(const char),slipPacket_t* p){  //assume has cx
    char i;
    for(i=0;i<p->byteCount;i++){
        sendSlipByte(f,slipEncode(p->data[i]));
    }    
    sendSlipByte(f,slipEnd);
}

void handleThruTraffic(void){
    char f=0;
    slipPacket_t hostResponse;
    while(uart1_available()){
        f=processFeed(&packetsFromHost,uart1_receive());
        if(f){
            if(packetsFromHost.byteCount>1){
                if(packetsFromHost.data[0]=='C'){
                   chainID=packetsFromHost.data[1];
                   packetsFromHost.data[1]=chainID+1; //increment host addressing
                   packetsFromHost.byteCount=2;
                   addSlipChecksum(&packetsFromHost);   //rejig cx                   
                   heartbeatTimer=HEARTBEAT_INTERVAL/TICK_INTERVAL;
                }
            }
            sendSLIPpacket(&uart2_send,&packetsFromHost);
            packetsFromHost.byteCount=0;   //empty
            //send response back to host
            hostResponse.data[0]='D';
            hostResponse.data[1]=chainID;
            hostResponse.data[2]=(unsigned char)MUIcx;
            hostResponse.byteCount=3;
            addSlipChecksum(&hostResponse);   //add cx                   
            sendSLIPpacket(&uart1_send,&hostResponse);  
        }
    }
    while(uart2_available()){
        f=processFeed(&packetsToHost,uart2_receive());
        if(f){
            sendSLIPpacket(&uart1_send,&packetsToHost);
            packetsToHost.byteCount=0;   //empty
        }
    }
}

unsigned char slipChecksum(slipPacket_t* p){
  int i;
  unsigned char cx=0;
  for(i=0;i<p->byteCount;i++){
    cx=cx^p->data[i];
  }
  return cx;
}

void addSlipChecksum(slipPacket_t* p){
  unsigned char cx;
  cx=slipChecksum(p);
  p->data[p->byteCount]=cx;
  p->byteCount++;
}

void drawMain(char n){  //draw main page for slot n
    OLEDcol=2;
    OLEDpage=0;
    OLEDchar(n+'1',arial);          //col 1
    OLEDchar(':',arial);            //col 2
    if(locos[n].address){
        textScanShort(locos[n].address);
        if(locos[n].useLongAddress==0){
            textLZB(8);
            OLEDchararray(textScanBuf+7,arial);    //col 3-7    
            OLEDchararray("  ",arial);
        }else{
            OLEDchararray(textScanBuf+5,arial);    //col 3-7                        
        }        
    }else{
        OLEDchararray("  ---",arial);    //col 3-7            
    }
    //messages go here    
    if(potMatched){
        if(locos[n].address==0){
            OLEDchararray(" HOLD >",arial);
        }else{
            OLEDchararray(" READY ",arial);
        }        
    }else if(locos[n].speed<getSpeed()){
        OLEDchararray(" <SP   ",arial);        
    }else{
        OLEDchararray("  SP>  ",arial);                
    }
    //2nd line
    OLEDcol=2;
    OLEDpage=4;
    if(locos[n].dir==DCC_FORWARD){
        OLEDchar('F',arial);        //col 1
    }else{
        OLEDchar('R',arial);                        
    }
    OLEDchar(':',arial);                  //col 2      
    if(potActive[n]){
        textScanShort(locos[n].speed);
        textLZB(8);
        OLEDchararray(textScanBuf+7,arial);   //col 3-5             
    }else{
        OLEDchararray("---",arial);   //col 3-5
    }
    OLEDchar(' ',arial);                        //col 6
    if(locos[n].F0){OLEDchar('L',arial);}else{OLEDchar('.',arial);} //col 7
    if(locos[n].F1){OLEDchar('1',arial);}else{OLEDchar('.',arial);}
    if(locos[n].F2){OLEDchar('2',arial);}else{OLEDchar('.',arial);} //col 9    
    OLEDchar(' ',arial);                        //col 10
    if(chainID==CHAINID_INVALID){
        OLEDchararray("---",arial);   //col 11-14
    }else{
        textScanShort(chainID);
        textLZB(8);
        OLEDchararray(textScanBuf+7,arial);   //col 3-5             
    }    
}

void getAddress(char n){
    char done=0;
    unsigned char a[3]={0,0,0};  //short,high digits of long,low digits of long
    unsigned int aa=0;
    unsigned char m=0;  //mode=short address
    locos[n].speed=0;   //make sure it's stopped
    if(locos[n].address){   //and send out on the wire
        speedPacket128(&speedPacket,locos[n]);
        F04Packet(&functionPacket,locos[n]);
        sendDCCPacket(&speedPacket);
        sendDCCPacket(&functionPacket);
        sendDCCPacket(&speedPacket);
        sendDCCPacket(&functionPacket);
    }
    setLED(LED_SET_OFF);                    
    while(done==0){
        OLEDcol=2;
        OLEDpage=0;
        OLEDchararray("ADDRESS: ",arial);
        a[m]=getTwoDigits();
        if(a[m]>99){a[m]=99;}        
        if(m==0){
            textScanShort(a[0]);
            textLZB(8);
            OLEDchararray(textScanBuf+7,arial);            
            OLEDchararray("  ",arial);
        }else{
            aa=a[1];
            aa=aa*100+a[2];
            textScanShort(aa);
            OLEDchararray(textScanBuf+5,arial);            
        }
        OLEDcol=2;
        OLEDpage=4;
        OLEDchararray("v EXIT  > DONE ",smallFont);
        switch(m){
            case 0:  OLEDchararray("^^^   ",smallFont);break;
            case 1:  OLEDchararray("^^^---",smallFont);break;
            default: OLEDchararray("---^^^",smallFont);break;
        }
        OLEDcol=2;
        OLEDpage=6;
        OLEDchararray("F0 SHORT. F1/F2 LONG.",smallFont);
        if(checkButton(BUTTON_DOWN)){done=1;}   //cancel
        if(checkButton(BUTTON_UP)||checkButton(BUTTON_SEL)){//OK
            if(m==0){
                locos[n].address=a[0];
                locos[n].useLongAddress=0;
            }else{
                locos[n].address=aa;
                locos[n].useLongAddress=1;
            }
            done=1;
        }
        if(checkButton(BUTTON_F0)){m=0;}
        if(checkButton(BUTTON_F1)){m=1;}
        if(checkButton(BUTTON_F2)){m=2;}
        handleThruTraffic();
        sendTraffic();
    }
}

void drawSys(void){
    OLEDcol=2;
    OLEDpage=0;
    OLEDchararray("SYS:",arial);
    if(sysMsg[0]==0){
        OLEDcol=40;
        OLEDpage=0;
        OLEDchararray(ICON_DOWN_ARRAY ICON_UP_ARRAY,iconSmall);
        OLEDchararray("DCC OFF/ON",smallFont);
        OLEDcol=40;
        OLEDpage=2;
        OLEDchararray("F0 SAVE LOCOS",smallFont);
    }else{
        OLEDchararray(sysMsg,arial);
    }    
    OLEDcol=2;
    OLEDpage=4;
    OLEDchararray("F1:",arial);
    if(f1Mode==FUNCTION_MOM){
        OLEDchararray("MOM ",arial);
    }else{
        OLEDchararray("TOG ",arial);
    }
    OLEDchararray("F2:",arial);
    if(f2Mode==FUNCTION_MOM){
        OLEDchararray("MOM ",arial);
    }else{
        OLEDchararray("TOG ",arial);
    }
    
}

void sendTraffic(void){ //any pending data, only one per tick
    static char i=0; 
    char j;
    if(tickDone){
        tickDone=0;
        for(j=0;j<LOCO_COUNT;j++){  //only try for LOCO_COUNT times
            i=i+1;
            if(i>=LOCO_COUNT){i=0;} //but work in round robin fashion
            tickDone=0;
            if((updateFlags[i] && (minTimers[i]==0)) || (maxTimers[i]==0)){
                minTimers[i]=MIN_INTERVAL/TICK_INTERVAL;
                maxTimers[i]=MAX_INTERVAL/TICK_INTERVAL;
                if(updateFlags[i]){updateFlags[i]--;}
                if(locos[i].address){
                    speedPacket128(&speedPacket,locos[i]);
                    F04Packet(&functionPacket,locos[i]);
                    sendDCCPacket(&speedPacket);
                    sendDCCPacket(&functionPacket);
                    return; //stop after sending 1
                }
            }                    
        }
    }
}

void idleFunction(void){
    handleThruTraffic();
    sendTraffic();    
}