#include "io.h"

unsigned int adcref1,dacref1;
unsigned int adcref2,dacref2;
unsigned int adcref4,dacref4;
unsigned char MUI[9];
unsigned int MUIcx;
void(*idleCallback)(void)=0;

void setIdleCallback(void(*f)(void)){
    idleCallback=f;
}

void callCallback(void){
    if(idleCallback){
        idleCallback();
    }
}

void ioInit(void){
    unsigned int i;
    unsigned char d;
    //these are in DIA, see pic16f1xxxx.h
    adcref1=readNVM(DIA_FVRA1X);
    dacref1=readNVM(DIA_FVRC1X);
    adcref2=readNVM(DIA_FVRA2X);
    dacref2=readNVM(DIA_FVRC2X);
    adcref4=readNVM(DIA_FVRA4X);
    dacref4=readNVM(DIA_FVRC4X);
    for(i=0;i<9;i++){
        d=(unsigned char)readNVM(DIA_MUI+i);
        MUI[i]=d;
        MUIcx=MUIcx+d;
    }
    //set everything as analog inputs with pullup to avoid leakage
    ANSELA=0xFF;
    ANSELB=0xFF;
    ANSELC=0xFF;
    //switches
    anselbits.S1=0; //digital inputs with pullups
    anselbits.S2=0;
    anselbits.S3=0;
    anselbits.S4=0;
    anselbits.S5=0;
    anselbits.S6=0;
    trisbits.S1=1;
    trisbits.S2=1;
    trisbits.S3=1;
    trisbits.S4=1;
    trisbits.S5=1;
    trisbits.S6=1;
    wpubits.S1=1;
    wpubits.S2=1;
    wpubits.S3=1;
    wpubits.S4=1;
    wpubits.S5=1;
    wpubits.S6=1;
    //analogs
    anselbits.VPOT=1;
    wpubits.VPOT=0;
    //LEDs output off
    anselbits.LED_GREEN=0;
    trisbits.LED_GREEN=0;
    latbits.LED_GREEN=0;
    anselbits.LED_RED=0;
    trisbits.LED_RED=0;
    latbits.LED_RED=0;
}

void t0init(void){      //count at 244Hz (ie no interrupts, use raw 16-bit counter value)
    //16bit
/*
    T0CON0=0;               //reset, 8-bit, 1:1 postscale
    T0CON1=0;               //T0CKI, ASYNC off, PS=1:1
    //TMR0L=0;
    //TMR0H=0;                //reset
    T0CON0bits.MD16=1;      //16-bit, period is 65536
    T0CON1bits.CS=5;        //MFINTOSC 500kHz
    //T0CON1bits.CS=4;        //LFINTOSC 31kHz
    //T0CON1bits.CS=2;        //FOSC/4    
    T0CON1bits.ASYNC=1;     //ASYNC
    T0CON1bits.T0CKPS=11;    //1:2^n
    //T0CON0bits.OUTPS=14;    //1:15   1:15*65536 => 4Hz
    T0CON0bits.EN=1;        //on
    //PIR0bits.TMR0IF=0;      //clear
    //PIE0bits.TMR0IE=1;      //enable
*/
    //8bit
    T0CON0=0;               //reset, 8-bit, 1:1    
    T0CON1bits.CS=2;        //FOSC/4=8MHz
    T0CON1bits.ASYNC=1;     //ASYNC
    T0CON1bits.T0CKPS=7;    //1:128
    T0CON0bits.OUTPS=TICK_INTERVAL/2-1;
    TMR0H=125;              //~2ms x (OUTPS+1)
    T0CON0bits.EN=1;        //on
    PIR0bits.TMR0IF=0;      //clear
    PIE0bits.TMR0IE=1;      //enable
}

void initADC(void){
    ADCON0=0;           //reset ADC, single ended, use FOSC
    ADCON1=0;
    ADCON2=0;           //legacy mode
    ADCON3=0;
    ADPRE=0;
    ADACQ=0;
    ADCAP=0;            //extra capacitance
    //ADCLK=63;
    ADCON0bits.CS=1;    //ADCRC    
    ADCON0bits.ADFM0=1; //right justify, two's complement
    ADREF=0;            //VDD    
    //ADREF=3;            //FVR
    ADCON0bits.ADON=1;  //turn on, ready
}

void deInitADC(void){
    ADCON0=0;
}

unsigned int getADC(char c){
    char i;
    unsigned int r=0;
    ADCON0bits.IC=0;    //switch back to single ended
    ADPCH=ADC_VSS;
    ADCON0bits.GO=1;    //dummy VCC
    while(ADCON0bits.GO){}            
    ADPCH=c;
    ADCON0bits.GO=1;    //dummy
    while(ADCON0bits.GO){}            
    for(i=0;i<ADC_OS;i++){
        ADCON0bits.GO=1;
        while(ADCON0bits.GO){}            
        r=r+ADRES;
    }
    return r;
}

unsigned int getVCC(void){  //in mV
    unsigned long r;
    unsigned char adref=ADREF;
    ADREF=0;            //VDD    
    r=getADC(FVRBUF1);
    ADREF=adref;
    if(r<(adcref4/2)){return 8192;}    //possible overflow/error
    return (unsigned int)(((unsigned long)adcref4)*(4096UL)/r);
}

unsigned int readNVM(unsigned int add){ //maps top bit to NVMREGS
    NVMREGS=0;
    if(add&0x8000){NVMREGS=1;}
    NVMADR=add;
    NVMCON1bits.RD=1;   //start read
    return NVMDAT;    
}

void initFVR_DAC(char n){
    FVRCON=(FVRCON&0xF3)|0x80|((n<<2)&0xC);    
}

void initFVR_ADC(char n){
    FVRCON=(FVRCON&0xFC)|0x80|(n&3);
}

void deInitFVR(void){
    FVRCON=0;
}

char checkButton(char n){
    if(buttonState(n)){
        while(buttonState(n)){
            callCallback();
        }
        delay(BUTTON_DELAY); //debouncing
        return SHORT_PRESS;
    }
    return 0;    
}

char checkButtonLongShort(char n){  //return long or short
    char t=0;
    char r=SHORT_PRESS;
    if(buttonState(n)){
        while(buttonState(n)){
            callCallback();
            t++;
            delay(BUTTON_DELAY);
            if(t>(BUTTON_LONG/BUTTON_DELAY)){r=LONG_PRESS;}
        }
        delay(10); //debouncing
        return r;
    }
    return 0;        
}

char buttonState(char n){
    switch(n){
#ifdef S1
        case 1: return PRESSED(S1); break;
#endif
#ifdef S2
        case 2: return PRESSED(S2); break;
#endif
#ifdef S3
        case 3: return PRESSED(S3); break;
#endif
#ifdef S4
        case 4: return PRESSED(S4); break;
#endif
#ifdef S5
        case 5: return PRESSED(S5); break;
#endif
#ifdef S6
        case 6: return PRESSED(S6); break;
#endif
#ifdef S7
        case 7: return PRESSED(S7); break;
#endif
#ifdef S8
        case 8: return PRESSED(S8); break;
#endif
    }
    return 0;
}

char anyButton(void){
#ifdef S1
    if(PRESSED(S1)){return 1;}
#endif
#ifdef S2
    if(PRESSED(S2)){return 2;}
#endif
#ifdef S3
    if(PRESSED(S3)){return 3;}
#endif
#ifdef S4
    if(PRESSED(S4)){return 4;}
#endif
#ifdef S5
    if(PRESSED(S5)){return 5;}
#endif
#ifdef S6
    if(PRESSED(S6)){return 6;}
#endif
#ifdef S7
    if(PRESSED(S7)){return 7;}
#endif
#ifdef S8
    if(PRESSED(S8)){return 8;}
#endif
    return 0;    
}

char buttonPressAndHold(void){  //return repeats when held
    char t;
    static char last=0;
    char now=anyButton();   //what is pressed?
    if(now==0){  //return immediately unless button pressed
        last=0;
        return 0;
    }else{
        if(last){
            t=HOLD_DELAY/DELAY_INCREMENT;
            while((t--)&&anyButton()){
                delay(DELAY_INCREMENT);
                callCallback();
            }
            last=anyButton();
            if(last){
                return now;
            }else{
                return 0;   //stop double-skip on release after hold
            }            
        }else{
            t=PRESS_DELAY/DELAY_INCREMENT;
            while((t--)&&anyButton()){
                delay(DELAY_INCREMENT);
                callCallback();
            }
            last=anyButton();
            return now;
        }        
    }    
}

void setLED(char n){
    switch(n){
        case LED_SET_OFF: latbits.LED_GREEN=0;latbits.LED_RED=0;break;
        case LED_SET_RED: latbits.LED_GREEN=0;latbits.LED_RED=1;break;
        case LED_SET_GREEN: latbits.LED_GREEN=1;latbits.LED_RED=0;break;
    }
}