#include "uart.h"

#ifdef UART_TX1_PIN
volatile unsigned char uart1_txhead,uart1_txtail;
unsigned char uart1_txbuf[UART1_TXBUFSIZE];
#define UART_TX1_RESET uart1_txhead=0;uart1_txtail=0;anselbits.UART_TX1_PIN=0;trisbits.UART_TX1_PIN=0;
#else
#define UART_TX1_RESET ;
#endif

#ifdef UART_RX1_PIN
volatile unsigned char uart1_rxhead,uart1_rxtail;
unsigned char uart1_rxbuf[UART1_RXBUFSIZE];
#define UART_RX1_RESET uart1_rxhead=0;uart1_rxtail=0;anselbits.UART_RX1_PIN=0;trisbits.UART_RX1_PIN=1;wpubits.UART_RX1_PIN=1;
#else
#define UART_RX1_RESET ;
#endif

#ifdef UART_TX2_PIN
volatile unsigned char uart2_txhead,uart2_txtail;
unsigned char uart2_txbuf[UART2_TXBUFSIZE];
#define UART_TX2_RESET uart2_txhead=0;uart2_txtail=0;anselbits.UART_TX2_PIN=0;trisbits.UART_TX2_PIN=0;
#else
#define UART_TX2_RESET ;
#endif

#ifdef UART_RX2_PIN
volatile unsigned char uart2_rxhead,uart2_rxtail;
unsigned char uart2_rxbuf[UART2_RXBUFSIZE];
#define UART_RX2_RESET uart2_rxhead=0;uart2_rxtail=0;anselbits.UART_RX2_PIN=0;trisbits.UART_RX2_PIN=1;wpubits.UART_RX2_PIN=1;
#else
#define UART_RX2_RESET ;
#endif

void uart1_deInit(void){
    TX1STA=0;           //reset    
    RC1STA=0;           //reset    
}

void uart1_init(unsigned int b){    //b is constant for baudcon register
    unsigned char iflag;
    iflag=INTCON;               //save interrupt flags
    INTCONbits.GIE=0;           //disable interrupts globally
    PPSLOCK=0x55;
    PPSLOCK=0xAA;
    PPSLOCKED=0;
#ifdef UART_TX1_PIN
    pps.UART_TX1_PIN=PPSO_TX1_OUT;
#endif
#ifdef UART_RX1_PIN
    RX1PPS=UART_RX1_PIN;
#endif
    PPSLOCK = 0x55;
    PPSLOCK = 0xAA;
    PPSLOCKED=1;
    INTCON=iflag;       //restore interrupts    
    UART_TX1_RESET;     //pins and buffers
    UART_RX1_RESET;     //pins and buffers
    //BAUD/COMMON
    TX1STA=0;           //reset, async mode
    BAUD1CON=8;         //16bit BRG, low speed
    TX1STAbits.BRGH=1;  //high speed
    SP1BRG=b;           //could be derived from _XTAL_FREQ
#ifdef UART_TX1_PIN
    TX1STAbits.TXEN=1;  //TX on
    //TXIE is set when there is data in buffer
#endif
#ifdef UART_RX1_PIN
    RC1STA=0;           //reset
    RC1STAbits.CREN=1;  //enable receive
    RC1IE=1;
    PEIE=1;         //allow interrupts, GIE set in main
#endif
    RC1STAbits.SPEN=1;  //enable serial port
}

void uart2_deInit(void){
    TX2STA=0;           //reset    
    RC2STA=0;           //reset    
}

void uart2_init(unsigned int b){    //b is constant for baudcon register
    unsigned char iflag;
    iflag=INTCON;               //save interrupt flags
    INTCONbits.GIE=0;           //disable interrupts globally
    PPSLOCK=0x55;
    PPSLOCK=0xAA;
    PPSLOCKED=0;
#ifdef UART_TX2_PIN
    pps.UART_TX2_PIN=PPSO_TX2_OUT;
#endif
#ifdef UART_RX2_PIN
    RX2PPS=UART_RX2_PIN;
#endif
    PPSLOCK = 0x55;
    PPSLOCK = 0xAA;
    PPSLOCKED=1;
    INTCON=iflag;       //restore interrupts    
    UART_TX2_RESET;     //pins and buffers
    UART_RX2_RESET;     //pins and buffers
    //BAUD/COMMON
    TX2STA=0;           //reset, async mode
    BAUD2CON=8;         //16bit BRG, low speed
    TX2STAbits.BRGH=1;  //high speed
    SP2BRG=b;           //could be derived from _XTAL_FREQ
#ifdef UART_TX2_PIN
    TX2STAbits.TXEN=1;  //TX on
    //TXIE is set when there is data in buffer
#endif
#ifdef UART_RX2_PIN
    RC2STA=0;           //reset
    RC2STAbits.CREN=1;  //enable receive
    RC2IE=1;
    PEIE=1;         //allow interrupts, GIE set in main
#endif
    RC2STAbits.SPEN=1;  //enable serial port
}

#ifdef UART_TX1_PIN
void uart1_send(const char c){
    while(((uart1_txhead+1)%UART1_TXBUFSIZE)==uart1_txtail){}         //blocking
    TX1IE=0;                   //disable while we add to avoid buffer corruption
    uart1_txbuf[uart1_txhead]=c;                                  //add to queue
    uart1_txhead=(uart1_txhead+1)%UART1_TXBUFSIZE;           //increment pointer
    TX1IE=1;               //enable interrupts again (or any time there is data)    
}

void uart1_sends(const char *c){
    while(*c){
        uart1_send(*c);
        c++;
    }    
}

unsigned char uart1_sendFree(void){   //bytes free for sending in uart1_txbuf
    return (unsigned char)((uart1_txtail+(UART1_TXBUFSIZE-1)-uart1_txhead)%UART1_TXBUFSIZE);    
}
#endif

#ifdef UART_RX1_PIN
char uart1_receive(void){
  char c;
  if(uart1_rxhead==uart1_rxtail){return 0xFF;}    //empty
  c=uart1_rxbuf[uart1_rxtail];
  uart1_rxtail=(uart1_rxtail+1)%UART1_RXBUFSIZE;
  return c;        
}

unsigned char uart1_available(void){
    return (unsigned char)((uart1_rxhead+UART1_RXBUFSIZE-uart1_rxtail)%UART1_RXBUFSIZE);
}
#endif

#ifdef UART_TX2_PIN
void uart2_send(const char c){
    while(((uart2_txhead+1)%UART2_TXBUFSIZE)==uart2_txtail){}         //blocking
    TX2IE=0;                   //disable while we add to avoid buffer corruption
    uart2_txbuf[uart2_txhead]=c;                                  //add to queue
    uart2_txhead=(uart2_txhead+1)%UART2_TXBUFSIZE;           //increment pointer
    TX2IE=1;               //enable interrupts again (or any time there is data)    
}

void uart2_sends(const char *c){
    while(*c){
        uart2_send(*c);
        c++;
    }    
}

unsigned char uart2_sendFree(void){   //bytes free for sending in uart2_txbuf
    return (unsigned char)((uart2_txtail+(UART2_TXBUFSIZE-1)-uart2_txhead)%UART2_TXBUFSIZE);    
}
#endif

#ifdef UART_RX2_PIN
char uart2_receive(void){
  char c;
  if(uart2_rxhead==uart2_rxtail){return 0xFF;}    //empty
  c=uart2_rxbuf[uart2_rxtail];
  uart2_rxtail=(uart2_rxtail+1)%UART2_RXBUFSIZE;
  return c;        
}

unsigned char uart2_available(void){
    return (unsigned char)((uart2_rxhead+UART2_RXBUFSIZE-uart2_rxtail)%UART2_RXBUFSIZE);
}
#endif

void uartISR(void){
    char d;
#ifdef UART_TX1_PIN
    if(TX1IF && TX1IE){
       if(uart1_txhead!=uart1_txtail){    //data in buffer
           TX1REG=uart1_txbuf[uart1_txtail]; //write clears flag
           uart1_txtail=(uart1_txtail+1)%UART1_TXBUFSIZE;
       }else{
           TX1IE=0;              //no more data, shut down
       }
    }
#endif
#ifdef UART_RX1_PIN
    if(RC1IF && RC1IE){
        if(RC1STAbits.OERR){RC1STAbits.CREN=0;RC1STAbits.CREN=1;}   //reset overflow error (serious, but better than locking up)
        RC1IF=0;           //clear flag
        d=RC1REG;        //do read
        if(((uart1_rxhead+1)%UART1_RXBUFSIZE)!=uart1_rxtail){ //avoid overflow by dropping
            uart1_rxbuf[uart1_rxhead]=d;
            uart1_rxhead=(uart1_rxhead+1)%UART1_RXBUFSIZE;
        }
    }
#endif
#ifdef UART_TX2_PIN
    if(TX2IF && TX2IE){
       if(uart2_txhead!=uart2_txtail){    //data in buffer
           TX2REG=uart2_txbuf[uart2_txtail]; //write clears flag
           uart2_txtail=(uart2_txtail+1)%UART2_TXBUFSIZE;
       }else{
           TX2IE=0;              //no more data, shut down
       }
    }
#endif
#ifdef UART_RX2_PIN
    if(RC2IF && RC2IE){
        if(RC2STAbits.OERR){RC2STAbits.CREN=0;RC2STAbits.CREN=1;}   //reset overflow error (serious, but better than locking up)
        RC2IF=0;           //clear flag
        d=RC2REG;        //do read
        if(((uart2_rxhead+1)%UART2_RXBUFSIZE)!=uart2_rxtail){ //avoid overflow by dropping
            uart2_rxbuf[uart2_rxhead]=d;
            uart2_rxhead=(uart2_rxhead+1)%UART2_RXBUFSIZE;
        }
    }
#endif
}
