#include "dcc.h"

Ticker DCC_tick;
char dcc1on,dcc2on;
dccPacket_t pkt1, pkt2, pktIdle, nextPkt2, pktReset;   //current packets
dccPacket_t nextPkt1[DCC_PACKET_QUEUE_SIZE];

char packetQueueSpace(void){
  if(nextPkt1[DCC_PACKET_QUEUE_SIZE-1].byteCount==0){
    return 1;
  }else{
    return 0;
  }
}

char queuePacket(dccPacket_t p){
  if(packetQueueSpace()){
    noInterrupts();
    nextPkt1[DCC_PACKET_QUEUE_SIZE-1]=p;
    packQueue();
    interrupts();
    return 1; //OK
  }else{
    return 0; //not done
  }
}

void packQueue(void){ //bump packets to top of queue, towards nextPkt1[0]
  int i;
  for(i=0;i<(DCC_PACKET_QUEUE_SIZE-1);i++){
    if(nextPkt1[i].byteCount==0){  //room to shuffle up
      nextPkt1[i]=nextPkt1[i+1];
      nextPkt1[i+1].byteCount=0;
    }
  }
}

void dumpPacket(dccPacket_t* p){
  int i;
  Serial.printf("DCC %d: ",p->byteCount);
  for(i=0;i<p->byteCount;i++){
    Serial.printf("%02x ",p->data[i]);
  }
  Serial.println();
}

void DCCinit(void){
  pinMode(DCC1A,OUTPUT);  //idle state
  pinMode(DCC1B,OUTPUT);
  pinMode(DCC1EN,OUTPUT);
  pinMode(DCC2A,OUTPUT);
  pinMode(DCC2B,OUTPUT);
  digitalWrite(DCC1A,LOW);
  digitalWrite(DCC1B,LOW);
  digitalWrite(DCC1EN,LOW);
  digitalWrite(DCC2A,LOW);
  digitalWrite(DCC2B,LOW);
  idlePacket(&pktIdle);
  resetPacket(&pktReset);
  dcc1on=0;
  dcc2on=0;
  DCC_tick.attach(DCC_PERIOD,DCCcallback);
}

void DCCcallback(void){
  static const uint8_t bitmasks[]={0,128,64,32,16,8,4,2,1}; //leading 0 supplies byte start, trailing 1 comes from next preamble
  static uint8_t phase1=0;
  static uint8_t phase2=0;
  static uint8_t bitptr1=0;
  static uint8_t bitptr2=0;
  static int8_t byteptr1=-2; //<0 => preamble
  static int8_t byteptr2=-2; //<0 => preamble
  static uint8_t bit1=1;
  static uint8_t bit2=1;
  //DCC1
  if(dcc1on){
    if(phase1==2){
      digitalWrite(DCC1EN,HIGH);
      digitalWrite(DCC1A,LOW);
      digitalWrite(DCC1B,HIGH);
    }else if(phase1>3){
      digitalWrite(DCC1EN,HIGH);
      digitalWrite(DCC1A,HIGH);
      digitalWrite(DCC1B,LOW);
      phase1=0;
      bitptr1++;      //next bit
      if(bitptr1>8){  //next byte
        bitptr1=0;
        byteptr1++;
        if(byteptr1>=pkt1.byteCount){ //next packet
          byteptr1=-2;
          //get next packet
          if(nextPkt1[0].byteCount){
            pkt1=nextPkt1[0];
            nextPkt1[0].byteCount=0; //flag that it has been consumed
          }else{
            pkt1=pktIdle;
          }
          packQueue();  //do this when possible
        }
      }
    }
    if(phase1==0){                             //only do calcs at start of bit
      if(byteptr1<0){
        bit1=1;                          //preamble bits are always one
      }else{
        bit1=(pkt1.data[byteptr1]&bitmasks[bitptr1])?1:0;
      }
    }
    phase1++;
    phase1=phase1+bit1; //double skip for 1's to give shorter bit time
  }else{
    digitalWrite(DCC1A,LOW);
    digitalWrite(DCC1B,LOW);
    digitalWrite(DCC1EN,LOW);
    bitptr1=0;
    byteptr1=-2;  //reset, will restart current packet
  }
  //output for DCC2
  if(dcc2on){
    if(phase2==2){
      digitalWrite(DCC2A,LOW);
      digitalWrite(DCC2B,HIGH);
    }else if(phase2>3){
      digitalWrite(DCC2A,HIGH);
      digitalWrite(DCC2B,LOW);
      phase2=0;
      bitptr2++;      //next bit
      if(bitptr2>8){  //next byte
        bitptr2=0;
        byteptr2++;
        if(byteptr2>=pkt2.byteCount){ //next packet
          byteptr2=-2;
          //get next packet
          if(nextPkt2.byteCount){ //load next, but don't default to idle
            pkt2=nextPkt2;
            nextPkt2.byteCount=0; //flag that it has been consumed
          }          
        }
      }
    }
    if(phase2==0){                             //only do calcs at start of bit
      if(byteptr2<0){
        bit2=1;                          //preamble bits are always one
      }else{
        bit2=(pkt2.data[byteptr2]&bitmasks[bitptr2])?1:0;
      }
    }
    phase2++;
    phase2=phase2+bit2; //double skip for 1's to give shorter bit time
  }else{
    digitalWrite(DCC2A,LOW);
    digitalWrite(DCC2B,LOW);
    bitptr2=0;
    byteptr2=-2;  //reset, this should restart current packet
  }  
}

void resetPacket(dccPacket_t* p){
  p->data[0]=0;
  p->data[1]=0;
  p->byteCount=2;
  addChecksum(p);
}

void idlePacket(dccPacket_t* p){
  p->data[0]=255;
  p->data[1]=0;
  p->byteCount=2;
  addChecksum(p);
}

void addChecksum(dccPacket_t* p){
  uint8_t cx=0;
  uint8_t i;
  for(i=0;i<p->byteCount;i++){cx=cx^p->data[i];}
  p->data[p->byteCount]=cx;
  p->byteCount++;
}

void addSpeed128(dccPacket_t* p, loco_t loco){
  p->data[p->byteCount]=0x3F;   //128 step
  p->byteCount++;
  if(loco.dir){
    p->data[p->byteCount]=(loco.speed&0x7F)|128;
  }else{
    p->data[p->byteCount]=(loco.speed&0x7F);
  }
  p->byteCount++;
  addChecksum(p);
}

void addF04(dccPacket_t* p, loco_t loco){
  p->data[p->byteCount]=0x80; //F0-F0 packet
  if(loco.F0){p->data[p->byteCount]=p->data[p->byteCount]|0x10;}
  if(loco.F1){p->data[p->byteCount]=p->data[p->byteCount]|0x01;}
  if(loco.F2){p->data[p->byteCount]=p->data[p->byteCount]|0x02;}
  if(loco.F3){p->data[p->byteCount]=p->data[p->byteCount]|0x04;}
  if(loco.F4){p->data[p->byteCount]=p->data[p->byteCount]|0x08;}
  p->byteCount++;
  addChecksum(p);
}

void addAddress(dccPacket_t* p, loco_t loco){
  if(loco.useLongAddress){
    p->data[0]=0xC0|((loco.address>>8)&0x3F);
    p->data[1]=loco.address&0xFF;    
    p->byteCount=2;
  }else{
    p->data[0]=loco.address&0x7F;
    p->byteCount=1;
  }
}

void speedPacket128(dccPacket_t* p, loco_t loco){
  addAddress(p,loco);
  addSpeed128(p,loco);
}

void F04Packet(dccPacket_t* p, loco_t loco){
  addAddress(p,loco);
  addF04(p,loco);
}

//service mode/programming packets
//direct mode
void DCCserviceDirectVerifyByte(dccPacket_t* p, int cv, byte data){        //verify cv has data  
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  p->data[0]=0x74|(cv>>8);   //2 msb of address
  p->data[1]=cv&0xFF;        //8 lsb of address
  p->data[2]=data;
  p->byteCount=3;
  addChecksum(p);  
}

void DCCserviceDirectVerifyBit(dccPacket_t* p, int cv, byte bt, byte bv){  //verify that cv has bit bt equal to bv
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  p->data[0]=0x78|(cv>>8);   //2 msb of address
  p->data[1]=cv&0xFF;        //8 lsb of address
  p->data[2]=0xE0|((bv&1)<<3)|(bt&7);
  p->byteCount=3;
  addChecksum(p);  
}

void DCCserviceDirectWriteByte(dccPacket_t* p, int cv, byte data){        //write cv with data  
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  p->data[0]=0x7C|(cv>>8);   //2 msb of address
  p->data[1]=cv&0xFF;        //8 lsb of address
  p->data[2]=data;
  p->byteCount=3;
  addChecksum(p);  
}

void DCCserviceDirectWriteBit(dccPacket_t* p, int cv, byte bt, byte bv){  //write cv bit bt with bv
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  p->data[0]=0x78|(cv>>8);   //2 msb of address
  p->data[1]=cv&0xFF;        //8 lsb of address
  p->data[2]=0xF0|((bv&1)<<3)|(bt&7);
  p->byteCount=3;
  addChecksum(p);  
}

//Address only modes, effectively a subset of Physical modes with only CV 1 supported
void DCCserviceAddressOnlyVerify(dccPacket_t* p, byte data){        //verify address (7bits only) 
  p->data[0]=0x70;             //verify
  p->data[1]=data&0x7F;        //7 lsb of address
  p->byteCount=2;
  addChecksum(p);  
}

void DCCserviceAddressOnlyWrite(dccPacket_t* p, byte data){        //write address (7bits only) 
  p->data[0]=0x74;             //write
  p->data[1]=data&0x7F;        //7 lsb of address
  p->byteCount=2;
  addChecksum(p);  
}

//physical modes
void DCCservicePhysicalVerify(dccPacket_t* p, int cv, byte data){        //verify mapped cv with data  
  switch(cv){
    case 1 : p->data[0]=0x70;break;  // verify Register 1 RRR=000
    case 2 : p->data[0]=0x71;break;  // verify Register 2 RRR=011
    case 3 : p->data[0]=0x72;break;  // verify Register 3 RRR=000
    case 4 : p->data[0]=0x73;break;  // verify Register 4 RRR=011
    case 29: p->data[0]=0x74;break;  // verify Register 5 RRR=100    
    //Register 6 (RRR=101) is page register
    case 7 : p->data[0]=0x76;break;  // verify Register 7 RRR=110
    case 8 : p->data[0]=0x77;break;  // verify Register 8 RRR=111
    default: p->byteCount=0; return;break;    //not supported by this mode
  }
  p->data[1]=data;
  p->byteCount=2;
  addChecksum(p);  
}

void DCCservicePhysicalWrite(dccPacket_t* p, int cv, byte data){        //write mapped cv with data  
  switch(cv){
    case 1 : p->data[0]=0x78;break;  // verify Register 1 RRR=000
    case 2 : p->data[0]=0x79;break;  // verify Register 2 RRR=011
    case 3 : p->data[0]=0x7A;break;  // verify Register 3 RRR=000
    case 4 : p->data[0]=0x7B;break;  // verify Register 4 RRR=011
    case 29: p->data[0]=0x7C;break;  // verify Register 5 RRR=100    
    //Register 6 (RRR=101) is page register
    case 7 : p->data[0]=0x7E;break;  // verify Register 7 RRR=110
    case 8 : p->data[0]=0x7F;break;  // verify Register 8 RRR=111
    default: p->byteCount=0; return;break;    //not supported by this mode
  }
  p->data[1]=data;
  p->byteCount=2;
  addChecksum(p);  
}

//page register setting
void DCCservicePageNumberVerify(dccPacket_t* p, int cv){
  int page=(cv+3)/4;
  p->data[0]=0x75;  //page register verify
  p->data[1]=page&0xFF;
  p->byteCount=2;
  addChecksum(p);  
}

void DCCservicePageNumberWrite(dccPacket_t* p, int cv){
  int page=(cv+3)/4;
  p->data[0]=0x7D;  //page register write
  p->data[1]=page&0xFF;
  p->byteCount=2;
  addChecksum(p);  
}

//paged mode, note that these are the same as physical when page=1
void DCCservicePagedVerify(dccPacket_t* p, int cv, byte data){  //check that PageNumber is set prior
  int reg=(cv+3)%4;
  p->data[0]=0x70|reg;  //page register write
  p->data[1]=data;
  p->byteCount=2;
  addChecksum(p);  
}

void DCCservicePagedWrite(dccPacket_t* p, int cv, byte data){  //check that PageNumber is set prior
  int reg=(cv+3)%4;
  p->data[0]=0x78|reg;  //page register write
  p->data[1]=data;
  p->byteCount=2;
  addChecksum(p);  
}

//operations mode programming
void DCCopsLongWrite(dccPacket_t* p,loco_t loco,int cv, byte data){  //s-9.2.1_dcc_extended_packet_formats: s2.3.7.3 Configuration Variable Access Instruction - Long Form
  cv=(cv-1)&0x3FF;            //CV1= HW address 0, 10 bits only
  addAddress(p,loco);  
  p->data[p->byteCount+0]=0xEC|((cv>>8)&0x3); //top two bits of cv
  p->data[p->byteCount+1]=cv&0xFF; //lower 8 bits of cv
  p->data[p->byteCount+2]=data; //value
  p->byteCount=p->byteCount+3;
  addChecksum(p);  
}

char checkPacketMatch(dccPacket_t* p1, dccPacket_t* p2){    //simple equality over valid bytes (up to len)
    unsigned char n=p1->byteCount;
    unsigned char i;
    if(n!=p2->byteCount){return 0;}
    for(i=0;i<n;i++){if(p1->data[0]!=p2->data[0]){return 0;}}
    return 1;
}
