'=========================================================================
' Tiny85 Graphical Thermocouple Temperature Meter
'
' Author: Andrew Woodfield
' Date: May 2023
' File Name: TCmeter_JLX_t85_Vxx.bas     (where xx is version number)
' ----------------------------------------------------------------------
' Program Description:
'
' A Tiny85 is used to measure temperatures from 0C to (in theory) 1023C
' using a K-type thermocouple and displays the results using a JLX12864
' 128X64 mono LCD. The Tiny85 and LCD operate from a 3V3 supply. (The LCD
' is a 3V3 type!!!)
'
' The LCD connects to the Tiny85 via a modified 4-wire SPI interface
' i.e. Sda, Sclk, A0 and Cs. Reset is wih LOW RC pulse at power-on.
'  SDA= PB.1 , SCLK = PB.0 , A0 = PB.2 , RST = RC , CS = PB.5
' Note: CS shared PB.1 with user pushbuttons!
'
' Three pushbuttons are connected to ADC0 to control Up and Down selection
' of the max temperature of the plotted graph, the temperature sample rate,
' and Start/Stop meter operation. Max temperatures can range from 50 - 600C
' in 50C steps with a default of 300C. Sample times are 1, 2, 3, 4, 5, 10
' and 30 seconds, and 1, 2, 5, 10 and 15 minutes. The default is 3 seconds.
'
' The processor waits for 900mS after each 'per second' sample is measured
' and displayed and the STOP user key has been checked. During this time,
' the processor is put to sleep using the Powerdown sleep mode.
'
' The UC1701 display routines are adapted from Kim Inkeun HL3LZV 2014.05.24
' via the Bascom Forum. Other display routines have been added for this
' application.
'
'=========================================================================
' Fuse settings:
'
' LOCK Bit Byte : 0ffh (No locks)
'
' Fuse EXTd Byte: 0ffh (No self-programming enabled
'
' Fuse HIGH Byte: 05fh
'   RSTDISBL 0 Reset DISABLED !!!
'       DWEN 1 DW not enabled
'      SPIEN 0 SPI programming enabled
'      WDTON 1 Watchdog timer off
'     EESAVE 1 EEPROM *NOT* preserved in erase
'  BODLEVEL2 1 BOD disabled
'  BODLEVEL1 1
'  BODLEVEL0 1
'
' Fuse LOW Byte:  0e2h
'     CKDIV8 1 NOT divided by 8
'      CKOUT 1 Not enabled
'       SUT1 1 Slow rising power
'       SUT0 0
'     CKSEL3 0 8 MHz internal RC clock
'     CKSEL2 0
'     CKSEL1 1
'     CKSEL0 0
'
'-------------------------------------------------------------------------

$regfile = "attiny85.dat"
$crystal = 4000000                                          '8MHz RC clock is divided by 2 in code below
'
Config portb.0 = Output                                     'Serial Clock                                   '
Config Portb.1 = Output                                     'Serial Data
Config Portb.2 = Output                                     'RS (A0) ( Low = command / High = Data )

Config Pinb.3 = Input                                       'Thermocouple - and balanced ADC input
Config Pinb.4 = Input                                       'Thermocouple + and balanced ADC input
Config portb.5 = output                                     'CS ( Active Low )   *******Also used as input later*******
'
A0 Alias Portb.2                                            'Command or Data
Rs Alias Portb.7                                            'Reset LCD        ***NOT a legitimate port here - Used to fool SPI driver
Cs Alias Portb.5                                            'Chip enable
'
'------------------------------------------------------------------------
Declare Sub Setlcdline(byval Yy As Byte)                    'Select the LCD line (0 ~ 63)
Declare Sub Setlcdpage(byval Page As Byte)                  'Select the LCD page (0 ~ 7)
Declare Sub Setlcdcol(byval Xx As Byte )                    'Select LCD column (0 ~ 127)
Declare Sub Setlcdswreset
Declare Sub Setlcdvolt(byval V5 As Byte )                   'V_LCD Contrast (0~7)
Declare Sub Setlcdvolm(byval Vol As Byte )                  'Contrast (0 ~ 63)
Declare Sub Setlcdinit                                      'Initialise LCD
Declare Sub Setlcdclear(byval Zz As Byte)                   'Clear LCD and can set on or off (0~255 where 0=> clear, 255=> fill all dots)
Declare Sub Lcdouttext                                      'Display TEXT on GLCD
Declare Sub Printfont                                       '8 x 6 Font GLCD Out Ssbroutine
declare Sub Setlcdpix                                       'Graphics page splash routine
declare Sub writelcdbyte                                    'write a byte to the LCD at current location
'-------------------------------
Dim Iii As Byte
Dim Xxx As Byte
Dim Jjj As Byte
Dim Kkk As Byte
Dim Vvv As Byte
Dim Vvv2 As Byte
Dim Length As Byte
Dim Loops As Byte
Dim Sign As String * 1
Dim Vvv1 As Byte
Dim Xx1 As Byte
Dim Text As String * 21

dim Tadc as word                                            'thermocouple voltage from adc
dim Vkey as word                                            'pushbutton adc input voltage

dim startkeyflag as bit                                     'flag for start pushbutton status (Open=0, Closed=1)
dim changeflag as bit                                       'flags change of up or down key
dim stopkeyexit as bit                                      'flags stop key press during measurement (0=No keypress, 1=stop pressed)

dim sampletime as word                                      'sample time in seconds
dim timetext as string * 3                                  'converts sampletime to text for lcd
dim timepointer as byte                                     'pointer to sample rate table
dim waitandsee as word                                      'counter for waitandsee sample time delay loop

dim avgcount as Byte                                        'counter for averaging loop
dim avgtemp as word                                         'averaged adc measurements

dim xaxis as byte                                           'LCD column pointer
dim lcdy as byte                                            'LCD row calculation
dim lcdrow as byte                                          'LCD row pointer
dim value as byte                                           'LCD row pixel value
dim temp as word                                            'temporary variable

dim maxheat as word                                         'max temperature to be reported on graph
dim mt2 as Word                                             'other y-axis marker values
dim mt1 as word
dim denom as word                                           'division denominator for y value pixel plot

dim offset as byte                                          'signed thermocouple temperature offset (Range: 0 to 40 equiv to -19 to +20 where offset of 20=0C)
dim delta as byte                                           'temporary calculation byte for offset
dim exitflag as bit                                         'flag to detect offset setup exit
dim dummy1 as eram word                                     'reserve first two bytes in EEPROM (Not used)
dim eoffset as eram byte                                    'reserve EEPROM space for user set thermocouple offset

'-------------------------------------------------------------------------
'

config clockdiv = 2                                         'divide RC clock by 2 to give 4MHz clock

'initialise variables

startkeyflag = 0                                            'start measurement flag
stopkeyexit = 0                                             'flags stop button event
timepointer = 3                                             'point to the third value in the sample time lookup table (3 sec)
maxheat = 300                                               'FSD y axis
offset = eoffset                                            'read stored offset value

Config Spi = Soft , Dout = Pinb.1 , Ss = None , Clock = Pinb.0
Spiinit                                                     'uses SPI driver

Waitms 300                                                  'Wait for supply voltage to stabilise

Setlcdinit                                                  'initialise the LCD
Setlcdclear(0)                                              'and then clear the LCD

Setlcdpix : wait 2                                          'write splash screen
Setlcdclear(0)

'check if next pressed on power-up or during splash write/wait
'yes - cal routine
'no - carry on

Config pinb.5 = input                                       'change to an input for the adc measurement
config adc = single , prescaler = auto , reference = AVCC   'starts automatically

For avgcount = 1 to 8                                       'sample eight times but keep only the last
   Vkey = getadc(0)
next avgcount

if Vkey <= 400 then                                         'has NEXT been pressed?

     'cal routine
     'screen is cleared and routine waits for NEXT key to be released before
     'measuring and reporting temperature with the current stored user offset.
     'if user alters the displayed value with UP/DOWN to correct the offset
     'then the new offset value is saved. Cal routine ends when NEXT key is pressed

      Setlcdclear(0)                                        'clear screen

     'wait for any key pressed to be released

      while Vkey <= 800
         waitms 100                                         'key release/debounce delay
         Vkey = getadc(0)
      wend

      exitflag = 0                                          'initialise flags
      changeflag = 0

      if offset > 40 then                                   'if first time then use zero-equiv value
         offset = 20
         set changeflag
      end if

      while exitflag = 0

            'offset is a signed value (range: 0 to 40) equiv to -19 to +20 where offset of 20=0C

            if offset < 20 then                             'convert offset to actual value and sign and convert measured temperature
               delta = 20 - offset
'                  if Tadc <= delta then Tadc = 0 else Tadc = Tadc - delta       'limit at zero
             else
               delta = offset - 20
'                  Tadc = Tadc + delta
            end if

            timetext = str(delta)                           'convert number to text for lcd
            if delta < 10 then insertchar timetext , 1 , " "       ' add a space if temperature is only one digit

            Config portb.5 = output                         'change PB5 config again for LCD write

            Setlcdpage 1 : Setlcdcol 0                      'show value on line 2 of LCD
             Text = "TEMP OFFSET: " : Lcdouttext
             if offset < 20 then text = "-" else text = " " : lcdouttext       'show sign if required
             text = timetext : Lcdouttext
             text = "C  " : Lcdouttext                      'units

            Config pinb.5 = input                           'change to an input for the adc measurement

'               config adc = single , prescaler = auto , reference = AVCC       'starts automatically

            For avgcount = 1 to 8                           'sample eight times but keep only the last
               Vkey = getadc(0)
            next avgcount

            select case Vkey                                'has a pushbutton been pressed?
               case is <= 400 :                             'NEXT key?
                  exitflag = 1                              'time to go...
               case is <= 600:                              'DECR key?
                  decr offset
                  if offset > 40 then offset = 0            'lower range limit
                  set changeflag
               case is <= 800 :                             'INCR key pressed?
                  incr offset
                  if offset > 40 then offset = 40           'upper range limit
                  set changeflag
            end select

            waitms 200                                      'key debounce release delay

     wend

     if changeflag = 1 then eoffset = offset                'update EEPROM if offset value changed
     waitms 50                                              'allow time for the write
     offset = eoffset                                       'update program value too

end if                                                      'NEXT not pressed so fall through to begin measurement


Do

   Setlcdclear(0)                                           'clear screen

   'let user select y-axis max temperature value

   changeflag = 1                                           'flag for first time through so current sample rate is reported

   while startkeyflag <> 1                                  'repeat until the Start button is pressed.....

      Config pinb.5 = input                                 'change to an input for the adc measurement
      config adc = single , prescaler = auto , reference = AVCC       'starts automatically

      'read the user pushbuttons

      For avgcount = 1 to 8                                 'sample eight times but keep only the last
         Vkey = getadc(0)
      next avgcount

      select case Vkey                                      'has a pushbutton been pressed?
         case is <= 400 :                                   'NEXT key?
            startkeyflag = 1                                'flag the sample time setting
            changeflag = 1
         case is <= 600:                                    'DECR key?
            maxheat = maxheat - 50
            if maxheat <= 50 then maxheat = 50              'limit minimum to 50C
            changeflag = 1
         case is <= 800 :                                   'INCR key pressed?
            maxheat = maxheat + 50                          'increase the max temperature
            if maxheat >= 600 then maxheat = 600            'limit max to 600C
            changeflag = 1
      end select

      if changeflag = 1 then                                'was a key pressed?

         while Vkey < 850
            Vkey = getadc(0)                                'yes, so wait for key to be released
         wend

         Config portb.5 = output                            'change PB5 config again for LCD write

         changeflag = 0                                     'reset the flag for next time

         timetext = str(maxheat)                            'convert it to text for lcd

         Setlcdpage 3 : Setlcdcol 0                         'show new sample rate on line 5 of LCD
          Text = "MAX TEMP AXIS: " : Lcdouttext
          text = timetext : Lcdouttext
          text = "C  " : Lcdouttext

         waitms 200                                         'wait for end of LCD write

      end if

   wend
                                                            'calculate other y-axis values
   mt2 = maxheat : shift mt2 , left , 1 : mt2 = mt2 \ 3     'mt2=2/3 maxtemp
   mt1 = maxheat \ 3                                        'mt1=1/3 maxtemp
   denom = maxheat \ 56 : incr denom                        'denominator for y value pixel plot on graph


   'now let user select sample time

   changeflag = 1                                           'flag for first time through so current sample rate is reported
   startkeyflag = 0                                         'reset for this routine

   while startkeyflag <> 1                                  'repeat until the Start button is pressed.....

      Config pinb.5 = input                                 'change to an input for the adc measurement
      config adc = single , prescaler = auto , reference = AVCC       'starts automatically

      'read the user pushbuttons

      For avgcount = 1 to 8                                 'sample eight times but keep only the last
         Vkey = getadc(0)
      next avgcount

      select case Vkey                                      'has a pushbutton been pressed?
         case is <= 400 :                                   'START key?
            startkeyflag = 1                                'flag the measurement start
            changeflag = 1
         case is <= 600:                                    'DECR key?
            decr timepointer
            if timepointer = 0 then timepointer = 12
            changeflag = 1
         case is <= 800 :                                   'INCR key pressed?
            incr timepointer                                'update the sample time selection
            if timepointer >= 13 then timepointer = 1       'desired timepointer range is 1-12
            changeflag = 1
      end select

      if changeflag = 1 then                                'was a key pressed?

         while Vkey < 850
            Vkey = getadc(0)                                'yes, so wait for key to be released
         wend

         Config portb.5 = output                            'change PB5 config again for LCD write

         changeflag = 0                                     'reset the flag for next time

         sampletime = lookup(timepointer , sampletable)     'read new sampletime from table
         timetext = str(sampletime)                         'convert it to text for lcd

         Setlcdpage 5 : Setlcdcol 0                         'show new sample rate on line 5 of LCD
          Text = "SAMPLE RATE: " : Lcdouttext
          text = timetext : Lcdouttext
          text = " SEC   " : Lcdouttext

         waitms 200                                         'wait for end of LCD write

      end if

   wend

      'start key was pressed so sample the temperature and plot result!

   startkeyflag = 0                                         'clear the flag for next time

   Setlcdclear(0) : Waitms 200                              'clear LCD screen


' read TC and write value on graph every sample period


' adc channel (7) in the ATtiny85 selects a differential adc with +ve on PB4 (pin 3)
' and -ve of PB3 (pin 2) with adc gain of x20. PB3/Pin 2 is tied to ground for the
' thermocouple. Software configures the adc is with a 1.1V reference so that the
' adc reads (1.1V/1023)/20 = 53.8uV/bit and FSD (1023 bits)=55mV
' The K-type thermocouple generates 41uV/K giving a change of (53.8/41) = 1.3C
' per bit. Therefore O degrees C=0V and adc(7)=0 and (say) 300C=12.3mV where adc(7)=229
' The adc value should be multiplied by 1.3 to convert it to the temperature in C.
' An approx integer method is to multiply the adc value by 64 and divide by 49 (1.306)

   config adc = single , prescaler = auto , reference = INTERNAL_1.1       'starts automatically

   do

      'write y-axis and y-axis lines and axis values and value markers

      Setlcdpage 0 : Setlcdcol 0 : Text = str(maxheat) : Lcdouttext       'write y-axis value marker then y-axis line character
      Setlcdpage 0 : Setlcdcol 17 : lcdy = 128 : writelcdbyte : lcdy = 255 : writelcdbyte
      Setlcdpage 1 : Setlcdcol 18 : writelcdbyte
      Setlcdpage 2 : Setlcdcol 0 : Text = str(mt2) : Lcdouttext
      Setlcdpage 2 : Setlcdcol 17 : lcdy = 128 : writelcdbyte : lcdy = 255 : writelcdbyte
      Setlcdpage 3 : Setlcdcol 18 : writelcdbyte
      Setlcdpage 4 : Setlcdcol 0 : Text = str(mt1) : Lcdouttext
      Setlcdpage 4 : Setlcdcol 17 : lcdy = 128 : writelcdbyte : lcdy = 255 : writelcdbyte
      Setlcdpage 5 : Setlcdcol 18 : writelcdbyte
      Setlcdpage 6 : Setlcdcol 0 : Text = "  0" : Lcdouttext
      Setlcdpage 6 : Setlcdcol 17 : lcdy = 128 : writelcdbyte : lcdy = 255 : writelcdbyte
      Setlcdpage 7 : Setlcdcol 32 : Text = "RATE: " : Lcdouttext
      Text = str(sampletime) : lcdouttext                   'convert it to text for lcd
      Text = " SEC   " : lcdouttext                         'trailing blanks resolve number length differences

      'write x-axis

      Setlcdpage 6 : Setlcdcol 19
      for xaxis = 19 to 127 :
         lcdy = 128 : writeLCDbyte
      next xaxis

      'measure and plot the data

      for xaxis = 19 to 127                                 'column pointer on LCD

         avgtemp = 0                                        'initialise

         For avgcount = 1 to 8                              'average the measurements due to noise
            Tadc = getadc(7)                                'get current thermocouple value
            avgtemp = avgtemp + Tadc                        'sum them all
         next avgcount

         shift avgtemp , right , 3                          'div by 8 to get average
         Tadc = avgtemp                                     'and save result

         'now convert to temperature in C

         SHIFT Tadc , LEFT , 6                              'multiply 10-bit adc value x 64 - Shift by 6 just fits in 16-bit word!
         Tadc = Tadc \ 49                                   'value now in degrees C

         if offset < 20 then                                'convert offset to actual value and sign and convert measured temperature
            delta = 20 - offset
            if Tadc <= delta then Tadc = 0 else Tadc = Tadc - delta       'limit at zero
          else
            delta = offset - 20
            Tadc = Tadc + delta
         end if

         'write value in lower left corner

         timetext = str(Tadc)                               'convert number to text for lcd
         if Tadc < 100 then insertchar timetext , 1 , " "   ' add a space if temperature is only two digits

         Setlcdpage 7 : Setlcdcol 0                         'shift pointer
          text = timetext : Lcdouttext                      'write temperature

         'convert to column value
         ' 64 LCD column pixels => lowest row (8 pixels) reserved for axis line and vertical marks
         ' leaving 56 pixels for data plot. If graph y-axis goes from 0C to 300C then each pixel
         ' is, say, (300/56 = 5.4) 6 degrees C. If maxtemp is, say, 500C, then each pixel is
         ' (500/56=8.9) 9 degrees C. And so forth...


         temp = Tadc \ denom                                '6 oC / pixel when maxtemp=300C
         lcdrow = temp \ 8                                  '8 vertical pixels per byte
         value = 6                                          'use value as a temporary register
         value = 6 - lcdrow                                 'because LCD row 0 is at the top
         lcdrow = value                                     'first row reserved for axis
         value = temp mod 8                                 'remainder shows pixel to write
         lcdy = 128                                         'bit7=1
         shift lcdy , right , value                         'converts value to pixel (bit7 is lowest, bit 0 is highest)

         Setlcdpage lcdrow : Setlcdcol xaxis                'point to the correct location on the LCD
         if lcdrow = 6 then lcdy = lcdy + 128               'rewrite x-axis if data is in lowest line
         writelcdbyte                                       'write the pixel

'         wait sampletime                                    'This is the main timing for samples and is adequately accurate (for now)

            'check for a STOP key press and flag it while waiting for the sampletime delay ...

         for waitandsee = 1 to sampletime                   'test each second for STOP key press while waiting

            Config pinb.5 = input                           'change to an input for the adc measurement
            config adc = single , prescaler = auto , reference = AVCC       'starts automatically

            'read the user pushbuttons

            Vkey = getadc(0)
            Vkey = getadc(0)
            Vkey = getadc(0)

            if Vkey <= 400 then                             'was the STOP key pressed?

               stopkeyexit = 1                              'flag event if STOP key pressed
               waitandsee = sampletime                      'and force/clean the waitandsee for-next sampletime delay and graph display loop for the exit
               xaxis = 127

               while Vkey < 850
                  Vkey = getadc(0)                          'wait for STOP key to be released
               wend

'               waitms 200                                   'plus a little time for the user to really leave the START/STOP key alone

            end if

            Config portb.5 = output                         'no valid key press so clean up by changing PB5 config again for LCD write
            config adc = single , prescaler = auto , reference = INTERNAL_1.1       'and reconfigure the ADC

            if stopkeyexit = 0 then waitms 900              'trim as required to give approx one second delay

         next waitandsee

      next xaxis                                            'point to next column

   'repeat the graph write unless STOP is pressed

      Setlcdclear(0) : Waitms 200                           'clear LCD screen for next screen plot

      if stopkeyexit = 1 then exit do                       'but exit the measurement if STOP was pressed

   loop

   'program can only arrive here if STOP key was pressed so clean up

   stopkeyexit = 0

   'and loop back to allow adjustment of max temperature and sample rate if required and to start another measurement

Loop


End



'*********************************************************************************************************
' LCD subroutines
'**********************************************************************************************************
'
'---------------------------------------------------------------------------------------
Sub Setlcdline(byval Yy As Byte)
      Vvv = &B01000000 + Yy                                 'set start line => ( 0 ~ 63 )
      A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
End Sub Setlcdline(byval Yy As Byte)
'---------------------------------------------------------------------------------------
Sub Setlcdpage(byval Page As Byte)
     Vvv = &B10110000 + Page                                ' Page =>(&B1011CCCC, 0000=>page1 ~0111=>page8 )
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
End Sub Setlcdpage(byval Page As Byte)
'---------------------------------------------------------------------------------------
Sub Setlcdcol(byval Xx As Byte)
        Vvv = Xx : Vvv1 = Xx
        Shift Vvv , Right , 4 : Vvv = Vvv
                Vvv = &B00010000 + Vvv
        Shift Vvv1 , Left , 4 : Shift Vvv1 , Right , 4
'             Vvv = Vvv1 + 16
' Setting Column address upper bit (&B0001CCCC, &H8:0000 )
       A0 = 0 : Cs = 0 : Spiout Vvv1 , 1 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

' Setting Column address lower bit (&B0000CCCC,&H3 : 0000 )
'       A0 = 0 : Cs = 0 : Spiout Vvv1 , 1 : Cs = 1 : A0 = 1
End Sub Setlcdcol(byval Xx As Byte)
'-----------------------------------------------------------------------------------------
Sub Setlcdswreset                                           ' LCD Software Reset => &B11100010
       Vvv = &B11100010
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Waitms 500 : Cs = 1 : A0 = 1
End Sub Setlcdswreset
'-----------------------------------------------------------------------------------------
Sub Setlcdvolt(byval V5 As Byte )                           ' &B00100ccc ( 0 ~ 7 )
      Vvv = &B00100000 + V5
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
End Sub Setlcdvolt(byval V5 As Byte )
'-----------------------------------------------------------------------------------------
Sub Setlcdvolm(byval Vol As Byte )
    Vvv = Vol : Vvv1 = &B10000001                           ' The Electronic Volume => 2byte
       A0 = 0 : Cs = 0 : Spiout Vvv1 , 1 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
End Sub Setlcdvolm(byval Vol As Byte )
'-----------------------------------------------------------------------------------------
Sub Setlcdinit
    Call Setlcdswreset

      ' LCD Display start line (&B01CCCCCC, 0 ~ 63 )
           Vvv = &H40
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

      ' LCD ADC Select => Normal=&B10100000, Reverse=&B10100001 )
           Vvv = &HA0
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

      ' Commom output mode => &B11000000=> Normal,&B11001000=>:reverse )
           Vvv = &HC8
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

      ' Set Inverse Display (&B1010011C, &HA6= Not Mirror )
           Vvv = &B10100110
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

      ' LCD Bias Setting 1/9 bias (&B1010001C, 0=> 1/9 , 1=> 1/7 )
           Vvv = &B10100010
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

      ' Power Control Setting => (&00101CCC, 111=>Booster,Vreg,V/F all ON )
           Vvv = &B00101111
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
      ' Set Booster Ratio  => (2 Byte => &hF8, &H00 x4 )
           Vvv = &HF8 : Vvv1 = &H00
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Spiout Vvv1 , 1 : Cs = 1 : A0 = 1
     Call Setlcdvolt(4)
     Call Setlcdvolm(36)
      ' Display On => 'On => &B10101111, Off=> &B10101110
           Vvv = &B10101111
       A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1

 ' No indicator
           Vvv = &HAC : Vvv1 = &H00 : Vvv2 = &HAF
    A0 = 0 : Cs = 0 : Spiout Vvv , 1 : Spiout Vvv1 , 1 : Spiout Vvv2 , 1 : Cs = 1 : A0 = 1

    Call Setlcdclear(0)
    Call Setlcdcol(0)                                       '0 ~ 127
    Call Setlcdpage(0)                                      '0 ~ 7
End Sub Setlcdinit
'------------------------------------------------------------------------------------------------------
Sub Setlcdclear(byval Zz As Byte)
    For Jjj = 0 To 7
            Setlcdpage Jjj
        For Iii = 0 To 127
               Setlcdpage Jjj : Setlcdcol Iii
            Vvv = Zz
            A0 = 1 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
        Next Iii
    Next Jjj

End Sub Setlcdclear(byval Zz As Byte)
'-------------------------------------------------------------------------------------------------------
Sub Printfont
    For Kkk = 0 To 5
        Read Xxx
             A0 = 1 : Cs = 0 : Spiout Xxx , 1 : Cs = 1 : A0 = 1
    Next Kkk
End Sub Printfont
'-------------------------------------------------------------------------------------------------------
Sub Lcdouttext
  Length = Len(text)
      For Loops = 1 To Length
          Sign = Mid(text , Loops , 1)
              If Sign = " " Then Restore 10:
              If Sign = "0" Then Restore 0:
              If Sign = "1" Then Restore 1:
              If Sign = "2" Then Restore 2:
              If Sign = "3" Then Restore 3:
              If Sign = "4" Then Restore 4:
              If Sign = "5" Then Restore 5:
              If Sign = "6" Then Restore 6:
              If Sign = "7" Then Restore 7:
              If Sign = "8" Then Restore 8:
              If Sign = "9" Then Restore 9:
              If Sign = "A" Or Sign = "a" Then Restore A:
              If Sign = "B" Or Sign = "b" Then Restore B:
              If Sign = "C" Or Sign = "c" Then Restore C:
              If Sign = "D" Or Sign = "d" Then Restore D:
              If Sign = "E" Or Sign = "e" Then Restore E:
              If Sign = "F" Or Sign = "f" Then Restore F:
              If Sign = "G" Or Sign = "g" Then Restore G:
              If Sign = "H" Or Sign = "h" Then Restore H:
              If Sign = "I" Or Sign = "i" Then Restore I:
              If Sign = "J" Or Sign = "j" Then Restore J:
              If Sign = "K" Or Sign = "k" Then Restore K:
              If Sign = "L" Or Sign = "l" Then Restore L:
              If Sign = "M" Or Sign = "m" Then Restore M:
              If Sign = "N" Or Sign = "n" Then Restore N:
              If Sign = "O" Or Sign = "o" Then Restore O:
              If Sign = "P" Or Sign = "p" Then Restore P:
              If Sign = "Q" Or Sign = "q" Then Restore Q:
              If Sign = "R" Or Sign = "r" Then Restore R:
              If Sign = "S" Or Sign = "s" Then Restore S:
              If Sign = "T" Or Sign = "t" Then Restore T:
              If Sign = "U" Or Sign = "u" Then Restore U:
              If Sign = "V" Or Sign = "v" Then Restore V:
              If Sign = "W" Or Sign = "w" Then Restore W:
              If Sign = "X" Or Sign = "x" Then Restore X:
              If Sign = "Y" Or Sign = "y" Then Restore Y:
              If Sign = "Z" Or Sign = "z" Then Restore Z:
              If Sign = "-" then restore {045}:
              If Sign = "=" then restore {061}:
              If Sign = ":" then restore {058}:
                 Printfont
      Next Loops
End Sub Lcdouttext
'------------------------------------------------------------------------------------------------------
Sub Setlcdpix
    Restore splashscreen
    For Jjj = 2 to 5
    Iii = 0
    Setlcdpage Jjj : Setlcdcol Iii
       For Iii = 0 To 127
           read Vvv
           A0 = 1 : Cs = 0 : Spiout Vvv , 1 : Cs = 1 : A0 = 1
       Next Iii
    Next Jjj
End Sub
'------------------------------------------------------------------------------------------------------
Sub writelcdbyte
    A0 = 1 : Cs = 0 : Spiout lcdy , 1 : Cs = 1 : A0 = 1     'write one byte
End sub
'------------------------------------------------------------------------------------------------------



'==================================================================================================================
' LCD Text Tables
'==================================================================================================================
'
10:                                                         'blank
Data &H00 , &H00 , &H00 , &H00 , &H00 , &H00
0:                                                          '0-9
Data &H00 , &H3E , &H41 , &H41 , &H3E , &h00
1:
Data &H00 , &H00 , &H42 , &H7F , &H40 , &H00
2:
Data &H00 , &H72 , &H49 , &H49 , &H46 , &H00
3:
Data &H00 , &H22 , &H49 , &H49 , &H36 , &H00
4:
Data &h18 , &h14 , &h12 , &h7F , &h10 , &h00
5:
Data &h00 , &h4F , &h49 , &h49 , &h31 , &h00
6:
Data &h00 , &h3E , &h49 , &h49 , &h32 , &h00
7:
Data &h00 , &h03 , &h61 , &h19 , &h07 , &h00
8:
Data &h00 , &h36 , &h49 , &h49 , &h36 , &h00
9:
Data &h00 , &h26 , &h49 , &h49 , &h3E , &h00

A:                                                          'A-Z
Data &H00 , &H78 , &H16 , &H11 , &H16 , &H78
B:
Data &H00 , &H7F , &H49 , &H49 , &H49 , &H36
C:
Data &H00 , &H3E , &H41 , &H41 , &H41 , &H22
D:
Data &H00 , &H7F , &H41 , &H41 , &H41 , &H3E
E:
Data &H00 , &H7F , &H49 , &H49 , &H49 , &H41
F:
Data &H00 , &H7F , &H09 , &H09 , &H09 , &H01
G:
Data &H00 , &H3E , &H41 , &H41 , &H49 , &H3A
H:
Data &H00 , &H7F , &H08 , &H08 , &H08 , &H7F
I:
Data &H00 , &H00 , &H41 , &H7F , &H41 , &H00
J:
Data &H00 , &H30 , &H40 , &H40 , &H40 , &H3F
K:
Data &H00 , &H7F , &H08 , &H14 , &H22 , &H41
L:
Data &H00 , &H7F , &H40 , &H40 , &H40 , &H40
M:
Data &H00 , &H7F , &H02 , &H0C , &H02 , &H7F
N:
Data &H00 , &H7F , &H06 , &H08 , &H30 , &H7F
O:
Data &H00 , &H3E , &H41 , &H41 , &H41 , &H3E
P:
Data &H00 , &H7F , &H09 , &H09 , &H09 , &H06
Q:
Data &H00 , &H3E , &H41 , &H51 , &H21 , &H5E
R:
Data &H00 , &H7F , &H09 , &H19 , &H29 , &H46
S:
Data &H00 , &H26 , &H49 , &H49 , &H49 , &H32
T:
Data &H00 , &H01 , &H01 , &H7F , &H01 , &H01
U:
Data &H00 , &H3F , &H40 , &H40 , &H40 , &H3F
V:
Data &H00 , &H0F , &H30 , &H40 , &H30 , &H0F
W:
Data &H00 , &H3F , &H40 , &H38 , &H40 , &H3F
X:
Data &H00 , &H63 , &H14 , &H08 , &H14 , &H63
Y:
Data &H00 , &H07 , &H08 , &H70 , &H08 , &H07
Z:
Data &H00 , &H61 , &H51 , &H49 , &H45 , &H43
{045}:                                                      '-
Data &H18 , &H18 , &H18 , &H18 , &H18 , &H18
{061}:                                                      '=
Data &H0A , &H0A , &H0A , &H0A , &H0A , &H0A
{058}:                                                      ':
Data &H00 , &H00 , &H14 , &H00 , &H00 , &H00


'--------------------------------------------------------
sampletable:                                                'sample times are word values so they all end with %
data 0% , 1% , 2% , 3% , 4% , 5% , 10% , 30% , 60% , 120% , 300% , 600% , 900%       'seconds
                                              ' 1     2       5     10     15 mins

'--------------------------------------------------------

Splashscreen:
Data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFE , &hFF , &h01 , &h11 , &hF7 , &hF6 , &h00 , &hFF , &hFF , &h11 , &h31 , &hEF , &hCE , &h00 , &hFE , &hFF , &h11 , &h11 , &hFF , &hFE , &h00 , &hFF , &hFF , &h11 , &h11 , &h1F , &h0E , &h00 , &hFF , &hFF , &h10 , &h10 , &hFF , &hFF , &h00 , &h01 , &h01 , &hFF , &hFF , &h01 , &h01 , &h00 , &hFF , &hFF , &h18 , &h30 , &hFF , &hFF , &h00 , &hFE , &hFF , &h01 , &h11,
Data &hF7 , &hF6 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &hF0 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h40 , &hA0 , &h40 , &h00 , &hC0 , &hE0 , &h20 , &hE0 , &hC0 , &h00 , &h00 , &h00 , &h00,
Data &h00 , &h08 , &h08 , &hF8 , &hF8 , &h08 , &h08 , &h00 , &hF8 , &hF8 , &h80 , &h80 , &hF9 , &hF9 , &h01 , &hF9 , &hF8 , &h88 , &h89 , &h89 , &h00 , &hF8 , &hF9 , &h89 , &h88 , &h79 , &h71 , &h00 , &hF8 , &hF9 , &h61 , &hC0 , &h61 , &hF9 , &hF8 , &h00 , &hF0 , &hF8 , &h08 , &h09 , &hF9 , &hF0 , &h00 , &hF1 , &hF9 , &h08 , &h09 , &h39 , &h31 , &h01 , &hF1 , &hF9 , &h08 , &h09 , &hF9 , &hF0 , &h00 , &hF9 , &hF9 , &h00 , &h00 , &hF9 , &hF9 , &h01,
Data &hF9 , &hF8 , &h08 , &h08 , &hF8 , &hF0 , &h00 , &hF8 , &hF8 , &h00 , &h00 , &h00 , &h00 , &hF8 , &hF8 , &h88 , &h88 , &h88 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h80 , &h40 , &h30 , &h08 , &h04 , &h02 , &h02 , &h04 , &h08 , &h70 , &h80 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h0F , &h1F , &h10 , &h1C , &h0C , &h00 , &h00 , &h00 , &h00,
Data &h00 , &h00 , &h00 , &h0F , &h4F , &h40 , &hC0 , &hC0 , &h4F , &h4F , &h00 , &hC0 , &hCF , &h0F , &h00 , &hCF , &hCF , &h08 , &hC8 , &hC8 , &h40 , &h4F , &h4F , &h00 , &hC1 , &hCF , &h4E , &h40 , &hCF , &h8F , &h00 , &hC0 , &hC0 , &h0F , &h0F , &h00 , &hC7 , &hCF , &h08 , &h88 , &hCF , &h47 , &h40 , &hC7 , &h8F , &h08 , &hC8 , &hCE , &h06 , &h00 , &h07 , &hCF , &hC8 , &h08 , &hCF , &hC7 , &h40 , &h47 , &h4F , &h08 , &h48 , &h4F , &hC7 , &hC0,
Data &h4F , &h4F , &h41 , &h01 , &hC1 , &hC0 , &h40 , &h4F , &h4F , &h08 , &hC8 , &hC8 , &h40 , &h4F , &hCF , &h88 , &h08 , &h08 , &h00 , &h00 , &h00 , &h00 , &h00 , &hFF , &h00 , &h00 , &h00 , &h00 , &hC0 , &h20 , &h1C , &h03 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h03 , &h04 , &h18 , &h20 , &h40 , &h40 , &h30 , &h08 , &h04 , &h08 , &h10 , &h60 , &h80 , &h00 , &h00 , &h80 , &h40 , &h20 , &h00 , &h00 , &h00,
Data &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h7F , &h7F , &h00 , &h00 , &h00 , &h7F , &h7F , &h04 , &h04 , &h7F , &h7F , &h00 , &h7F , &h7F , &h44 , &h44 , &h44 , &h00 , &h7F , &h7F , &h04 , &h0C , &h7F , &h77 , &h00 , &h7F , &h7F , &h03 , &h06 , &h03 , &h7F , &h7F , &h00 , &h3F , &h7F , &h40 , &h40 , &h7F , &h3F , &h00 , &h7F , &h7F , &h03 , &h06 , &h03 , &h7F , &h7F , &h00 , &h7F , &h7F , &h44 , &h44 , &h44 , &h00 , &h00 , &h00 , &h7F , &h7F,
Data &h00 , &h00 , &h00 , &h00 , &h7F , &h7F , &h44 , &h44 , &h44 , &h00 , &h7F , &h7F , &h04 , &h0C , &h7F , &h73 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h00 , &h1F , &h10 , &h14 , &h14 , &h12 , &h11 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h10 , &h11 , &h11 , &h10 , &h10 , &h10 , &h10 , &h10 , &h00,
