#include "dcc.h"

loco_t loco;
dcc_packet_t dccIn,dccRx,lastPacket;
unsigned int packetTimeout=0;
unsigned char locoTimeout=0;
unsigned char ackTimer=0;
unsigned char ackPol=0;
unsigned int accelTimer=0;
unsigned int accelCount=0;
unsigned char usePlusInput=0;
const dcc_config_t configDefault=CONFIG_DEFAULT;    //backup
__eeprom dcc_config_t nvConfig=CONFIG_DEFAULT;      //in EEPROM
unsigned char wireState[DCC_FUNCTION_WIRE_COUNT]; //white, yellow, green, brown
unsigned char speedCurve[DCC_SPEED_TABLE_SIZE];
//CV29 =[0]
//CV1 =[1]
//CV19=[2]
//CV17/CV18:16 bits=[4][3] (little endian)
//CV2;      //[5]
//CV3;      //[6]
//CV4;      //[7]
//CV5;      //[8]
//CV6;      //[9]
//CV33;    //F0F white=1  [10]
//CV34;    //F0R yellow=2 [11]
//CV35;    //F1 green=4  [12]
//CV36;    //F2 brown=8  [13]
//CV37;    //F3 none=0   [14]
//CV49;     //white      [15]
//CV50;     //yellow     [16]
//CV51;     //green      [17]
//CV52;     //brown      [18]
//CV11;     //timeout    [19]
//CV47;     //custom features    [20]
const unsigned char CVindex[DCC_CVS_IN_USE]={
    0xFF,   1,   5,   6,   7,   8,   9,0xFF,0xFF,0xFF,0xFF,  19,0xFF,0xFF,0xFF,0xFF,
    0xFF,   4,   3,   2,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,   0,0xFF,0xFF,
    0xFF,  10,  11,  12,  13,  14,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,0xFF,  20,
    0xFF,  15,  16,  17,  18
};
const unsigned char CVORmask[CONFIG_SIZE]={0,0,0,0,192,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0,0};
const unsigned char CVANDmask[CONFIG_SIZE]={35,127,255,255,255,255,255,255,255,255,15,15,15,15,15,255,255,255,255,255,31};
const unsigned char bitMasks[8]={1,2,4,8,16,32,64,128};

void dccInit(void){ //io, timers, structs
    //inputs/no pullup
    trisbits.SENSEMINUS=1;
    wpubits.SENSEMINUS=0;
    trisbits.SENSEPLUS=1;
    wpubits.SENSEPLUS=0;
    //t2    //22us ISR
    T2CON=0;                //1:1 PS and OUTPS
    T2HLT=0;                //free running, not gated
    T2CLKCON=1;             //FOSC/4 = 8MHz
    T2TMR=0;
    T2PR=176;               //22us
    T2CONbits.ON=1;         //run
    TMR2IF=0;               //clear
    TMR2IE=1;               //enable
    PEIE=1;
    //t0    //4Hz Function ISR
    T0CON0=0;               //reset, 8-bit, 1:1 postscale
    T0CON1=0;               //T0CKI, ASYNC off, PS=1:1
    T0CON0bits.MD16=1;      //65536 count
    T0CON1bits.CS=2;        //FOSC/4=8MHz
    T0CON1bits.ASYNC=1;     //ASYNC
    T0CON1bits.T0CKPS=1;    //1:2
    T0CON0bits.OUTPS=14;    //1:15 => 4Hz
    TMR0L=0;                //reset
    TMR0H=0;               
    T0CON0bits.EN=1;        //on
    TMR0IF=0;               //clear
    TMR0IE=1;               //enable        
}

void cleanPacket(dcc_packet_t* p){    //check checksum, void if invalid
    unsigned char c=0;
    unsigned char i;
    unsigned char len=p->len;    
    for(i=0;i<len;i++){c=c^p->data[i];}
    if(c){
        p->len=0;
    }else{
        packetTimeout=0;
    }
}

void dccISR(void){
    //bit level logic
    static unsigned char now=0;
    static unsigned char last=0;
    static unsigned char count=0;
    static unsigned char thisBit=0;
    //for preamble/stop/start detection
    static signed char bitcount=-DCC_PREAMBLE_BITS; //<0 means preamble
    static unsigned char bitmask=128;
    static unsigned char thisByte=0;
    static unsigned char functionBit=0;
    if(TMR0IF && TMR0IE){TMR0IF=0;
        functionBit=functionBit>>1;
        if(functionBit==0){functionBit=128;}
        if(loco.config.CV49&functionBit){trisbits.F0FCON=0;}else{trisbits.F0FCON=1;}
        if(loco.config.CV50&functionBit){trisbits.F0RCON=0;}else{trisbits.F0RCON=1;}
        if(loco.config.CV51&functionBit){trisbits.F1CON=0; }else{trisbits.F1CON=1; }
        if(loco.config.CV52&functionBit){trisbits.F2CON=0; }else{trisbits.F2CON=1; }
        locoTimeout++;  //4Hz update is convenient
    }
    if(TMR2IF && TMR2IE){TMR2IF=0;
        if(usePlusInput){
            if(portbits.SENSEPLUS){now=1;}else{now=0;}
        }else{
            if(portbits.SENSEMINUS){now=0;}else{now=1;}
        }    
        if(now){
            count++;
        }else{
            if(last){ //falling edge, act 
                if(count<4){        //1 bit
                    thisBit=1;
                }else{  //0 bit
                    thisBit=0;
                }
                if(bitcount<0){     //still counting preamble bits                    
                    if(thisBit){
                        bitcount++;     //advance count
                    }else{
                        bitcount=-DCC_PREAMBLE_BITS;    //reset preamble count
                    }
                }else if(bitcount==0){  //ready for start 0
                    if(thisBit==0){     //start
                        bitcount=1;
                        bitmask=128;
                        thisByte=0;
                    }                   //otherwise keep count at 0
                }else{      //handle data bits and subsequent start bits
                    if(thisBit){
                        thisByte=thisByte|bitmask;
                    }
                    bitmask=bitmask>>1;                    
                    bitcount++;
                    if(bitcount>9){ //end of byte
                        dccIn.data[dccIn.len]=thisByte; //add to outgoing
                        dccIn.len++;
                        bitcount=bitcount-9;    //move back (should be to 1)
                        bitmask=128;
                        thisByte=0;
                        if(dccIn.len>=DCC_DATA_BYTES){dccIn.len=DCC_DATA_BYTES-1;}  //clip
                        if(thisBit){    //end of packet
                            if(dccRx.len==0){  //if buffer full, ignore
                                dccRx=dccIn;
                            }
                            dccIn.len=0;
                            bitcount=-DCC_PREAMBLE_BITS;    //reset preamble count
                        }                        
                    }
                }
            }
            count=0;
        }
        last=now;
        packetTimeout++;
        if(packetTimeout>DCC_PACKET_TIMEOUT_COUNT){
            packetTimeout=0;
            if(usePlusInput){   //toggle rail if we don't see anything on this rail
                usePlusInput=0;
            }else{
                usePlusInput=1;                
            }
        }
        accelTimer++;
        if(ackTimer){ackTimer--;}
    }    
}

dcc_address_t getAddress(dcc_packet_t* p){
    dcc_address_t r={.address=0,.type=ADD_INVALID};
    if(p->len==0){return r;}
    unsigned int b0=p->data[0];
    unsigned int b1=p->data[1];
    if(b0==0){
        if(b1==0){
            r.type=ADD_RESET;
        }else{
            r.type=ADD_BROADCAST;
        }
    }else if(b0==255){   //idle packet
        r.type=ADD_IDLE;
    }else if(b0<112){   //short address
        r.address=b0;
        r.type=ADD_SHORT;
    }else if(b0<128){   //service mode
        r.type=ADD_SERVICE;        
    }else if((b0>191)&&(b0<232)){   //long address
        r.address=b0<<8;
        r.address=(r.address|b1)&0x3FFF;
        r.type=ADD_LONG;
    }
    return r;
}

void doAck(void){
    if(ackTimer==0){
        ackTimer=DCC_ACK_TIMER_COUNT;
        if(ackPol){ackPol=0;}else{ackPol=1;}    //toggle direction
    }
}

void parseDCCops(dcc_packet_t* p,loco_t* t){  //act on packet with respect to t
    dcc_address_t add; 
    add=getAddress(p);
    unsigned char rev=t->config.reverse;
    unsigned char* d=0;
    unsigned char s=0;   
    if(t->config.consistAddress){
        if(add.type!=ADD_SHORT){return;}
        if(add.address!=t->config.consistAddress){return;}
    }else{
        if(t->config.useLongAddress){
            if(add.type!=ADD_LONG){return;}
            if(add.address!=((t->config.longAddress)&0x3FFF)){return;}
        }else{
            if(add.type!=ADD_SHORT){return;}
            if(add.address!=t->config.shortAddress){return;}
        }   
    }
    locoTimeout=0;  //valid packet received to this address 
    //set pointer to data
    if(t->config.consistAddress){    //use consist
        if(add.type==ADD_SHORT){
            d=&p->data[1];
            if(t->config.consistReverse){rev=rev^1;}
        }        
    }else{
        if(t->config.useLongAddress){   //use long
            if(add.type==ADD_LONG){d=&p->data[2];}
        }else{  //use short
            if(add.type==ADD_SHORT){d=&p->data[1];}
        }                
    }    
    if(d){
        if((d[0]&DCC_BASELINE_MASK)==DCC_BASELINE){
            if(d[0]&DCC_BASELINE_DIR_BIT){rev=rev^1;}
            t->live.targetDir=rev;
            s=d[0]&DCC_BASELINE_SPEED_BITS;
            if(t->config.use28step){
                t->live.targetSpeed=s*8;
                if(d[0]&DCC_BASELINE_EXTRA_SPEED_BIT){t->live.targetSpeed=t->live.targetSpeed+4;}                
            }else{
                t->live.targetSpeed=s*8;
                if(d[0]&DCC_BASELINE_F0_BIT){
                    t->live.F0=1;
                }else{
                    t->live.F0=0;                    
                }
            }
            if(s==1){   //emergency stop
                t->live.targetSpeed=0;
                t->live.speed=0;
            }
        }else if((d[0]&DCC_EXT_128_STEP_MASK)==DCC_EXT_128_STEP){
            if(d[1]&DCC_EXT_DIR_BIT){rev=rev^1;}
            t->live.targetDir=rev;            
            s=d[1]&DCC_EXT_SPEED_BITS;
            t->live.targetSpeed=s;
            if(s==1){   //emergency stop
                t->live.targetSpeed=0;
                t->live.speed=0;
            }
        }else if((d[0]&DCC_EXT_FUNCTION_1_MASK)==DCC_EXT_FUNCTION_1){
            if(t->config.use28step){if(d[0]&DCC_EXT_F0_MASK){t->live.F0=1;}else{t->live.F0=0;}}
            if(d[0]&DCC_EXT_F1_MASK){t->live.F1=1;}else{t->live.F1=0;}
            if(d[0]&DCC_EXT_F2_MASK){t->live.F2=1;}else{t->live.F2=0;}
            if(d[0]&DCC_EXT_F3_MASK){t->live.F3=1;}else{t->live.F3=0;}
            if(d[0]&DCC_EXT_F4_MASK){t->live.F4=1;}else{t->live.F4=0;}
        }                    
    }  
}

void parseDCCopsOther(dcc_packet_t* p,loco_t* t){  //act on packet with respect to t
    //check non-consist address for ops mode programming packets, long form: s-9.2.1: 2.3.7.3, write only
    dcc_address_t add; 
    add=getAddress(p);
    unsigned char* d=0;
    unsigned int cv=0;      //can be up to 10 bits
    unsigned char cvVal=0;
    //ignore consist address
    if(t->config.useLongAddress){
        if(add.type!=ADD_LONG){return;}
        if(add.address!=((t->config.longAddress)&0x3FFF)){return;}
        d=&p->data[2];
    }else{
        if(add.type!=ADD_SHORT){return;}
        if(add.address!=t->config.shortAddress){return;}
        d=&p->data[1];
    }   
    locoTimeout=0;  //valid packet received to this address 
    if(checkPacketMatch(&lastPacket,p)){        //two consecutive matching packets for action
        if((d[0]&DCC_EXT_LONG_OPS_WRITE_MASK)==DCC_EXT_LONG_OPS_WRITE){
            cv=d[0];
            cv=cv<<8;
            cv=((cv|d[1])&0x3FF)+1;        
            cvVal=d[2];
            if((cv!=1)&&(cv!=17)&&(cv!=18)&&(cv!=29)){    //don't modify these in ops mode
                writeCV(cv,cvVal);
            }
        }
    }
}

void parseDCCservice(dcc_packet_t* p,loco_t* t){  //act on packet with respect to t
    dcc_address_t add; 
    add=getAddress(p);
    unsigned int cv=0;      //can be up to 10 bits
    unsigned char cvVal=0;
    unsigned char cvBit=0;    
    unsigned char cvMode=0; //depends on command
    if((p->data[0]&DCC_SERVICE_PACKET_MASK)==DCC_SERVICE_PACKET){    //service
        if(checkPacketMatch(&lastPacket,p)){        //two consecutive matching packets for action
            if(p->len==3){  //paged/physical
                cvMode=p->data[0]&7;
                cvVal=p->data[1];
                switch(cvMode){
                    case 0: cv=t->live.pageReg*4-3;break;
                    case 1: cv=t->live.pageReg*4-2;break;
                    case 2: cv=t->live.pageReg*4-1;break;
                    case 3: cv=t->live.pageReg*4;break;
                    case 4: cv=29;break;    //physical
                    case 5: loco.live.pageReg=cvVal;doAck();break; //page set, ack this
                    case 6: cv=DCC_MODEL_ID_CV;break;    //physical
                    case 7: cv=DCC_MANUF_ID_CV;break;    //physical                            
                }
                if(cvMode!=5){  //not page reg
                    if((p->data[0])&0x08){ //write
                        //cv=8 & cvVal=8 => decoder reset
                        if((cv==DCC_MANUF_ID_CV)&&(cvVal==8)){
                            //do reset
                            nvConfig=configDefault;
                            t->config=configDefault;  
                            doAck();
                        }else{
                            if(writeCV(cv,cvVal)){doAck();}
                        }
                    }else{  //verify
                        if(readCV(cv)==cvVal){doAck();}
                    }
                }
            }else if(p->len==4){    //direct
                cvMode=p->data[0]&0x0C;
                cv=p->data[0];
                cv=cv<<8;
                cv=((cv|p->data[1])&0x3FF)+1;   //+1 offset is prescribed in the standards
                cvVal=p->data[2];
                switch(cvMode){
                    case 0x04: //verify byte
                        if(readCV(cv)==cvVal){doAck();}
                        break;
                    case 0x0C: //write byte
                        if((cv==DCC_MANUF_ID_CV)&&(cvVal==8)){
                            //do reset
                            nvConfig=configDefault;
                            t->config=configDefault;  
                            doAck();
                        }else{
                            if(writeCV(cv,cvVal)){doAck();}
                        }
                        break;
                    case 0x08:  //bit manipulation
                        cvBit=cvVal;
                        if((cvBit&0xE0)==0xE0){ 
                            if(cvBit&0x10){ //write bit
                                if(readCV(cv)>=0){
                                    cvVal=(unsigned char)readCV(cv);
                                    if(cvBit&0x08){ //set
                                        cvVal=cvVal|bitMasks[cvBit&7];
                                    }else{  //clear
                                        cvVal=cvVal&(bitMasks[cvBit&7]^0xFF);                                            
                                    }
                                    if(writeCV(cv,cvVal)){doAck();}
                                }//else fail                                    
                            }else{      //verify bit
                                if(readCV(cv)>=0){
                                    cvVal=(unsigned char)readCV(cv);
                                    if(cvBit&0x08){ //check set
                                        if(cvVal&bitMasks[cvBit&7]){doAck();}
                                    }else{  //check clear
                                        if((cvVal&bitMasks[cvBit&7])==0){doAck();}                                            
                                    }                                        
                                }//else fail                                    
                            }
                        }
                        break;
                    default:break;      //do nothing                            
                }
            }
        }                    
    }    
}

void parseDCCbroadcast(dcc_packet_t* p,loco_t* t){    //handle broadcast here, handle stop only
    dcc_address_t add; 
    add=getAddress(p);
    unsigned char* d=0;
    unsigned char s=0;   
    d=&p->data[1];
    if((d[0]&DCC_BASELINE_MASK)==DCC_BASELINE){
        s=d[0]&DCC_BASELINE_SPEED_BITS;
        if(s==1){   //emergency stop
            t->live.targetSpeed=0;
            t->live.speed=0;
        }else if(s==0){   //normal stop
            t->live.targetSpeed=0;
        }
    }else if((d[0]&DCC_EXT_128_STEP_MASK)==DCC_EXT_128_STEP){
        s=d[1]&DCC_EXT_SPEED_BITS;
        if(s==1){   //emergency stop
            t->live.targetSpeed=0;
            t->live.speed=0;
        }else if(s==0){   //normal stop
            t->live.targetSpeed=0;
        }
    }    
}

void parseDCC(dcc_packet_t* p,loco_t* t){  //act on packet with respect to t
    dcc_address_t add; 
    add=getAddress(p);
    //mode transitions; check if service/reset etc
    switch(t->live.serviceMode){
        case DCC_MODE_NONE:
            if(add.type==ADD_RESET){
                t->live.serviceMode=DCC_MODE_RESET;
                resetLoco(t);
            }else if(
                    (add.type==ADD_SHORT)||
                    (add.type==ADD_LONG)||
                    (add.type==ADD_IDLE)||
                    (add.type==ADD_CONSIST)||
                    (add.type==ADD_BROADCAST)){                
                t->live.serviceMode=DCC_MODE_OPS;
            }
            break;
        case DCC_MODE_RESET:    //satisfy sequencing to enter service mode
            if(add.type==ADD_SERVICE){t->live.serviceMode=DCC_MODE_SERVICE;}
            else if(add.type!=ADD_RESET){t->live.serviceMode=DCC_MODE_NONE;}
            break;
        case DCC_MODE_SERVICE:
            if((add.type!=ADD_RESET)&&(add.type!=ADD_SERVICE)){
                t->live.serviceMode=DCC_MODE_NONE;
            }
            break;
        case DCC_MODE_OPS:
            if(add.type==ADD_RESET){
                t->live.serviceMode=DCC_MODE_RESET;
                resetLoco(t);
            }
            break;
    }
    //mode actions        
    if(t->live.serviceMode==DCC_MODE_OPS){
        parseDCCops(&dccRx,&loco);           //check for normal/consist addressed packet in ops mode
        parseDCCopsOther(&dccRx,&loco);      //check for normal addressed packet in ops mode (ie ops mode programming)    
    }else if(t->live.serviceMode==DCC_MODE_SERVICE){ //note that service mode does not require addressing        
        parseDCCservice(&dccRx,&loco);
    }//otherwise, nothing to do in DCC_MODE_RESET/DCC_MODE_NONE modes
    if(add.type==ADD_BROADCAST){
        parseDCCbroadcast(&dccRx,&loco);
    }
}

void resetLoco(loco_t* t){  //timeout/service mode etc
    t->live.targetSpeed=0;
    t->live.speed=0;
    t->live.F0=0;
    t->live.F1=0;
    t->live.F2=0;
    t->live.F3=0;
    t->live.F4=0;  
    //note that pageReg is not affected by resets
}

void mapSpeed(unsigned char start, unsigned char mid, unsigned char hi){    //note start, mid, hi are 8-bit values, but map is 7-bit
    unsigned char i,j;
    unsigned int n,n1;
    if(hi<2){hi=255;}
    if(mid<2){
        n=start;
        n=n+hi;
        n=n/2;
        mid=(unsigned char)n;
    }
    j=(DCC_SPEED_TABLE_SIZE/2)-3;
    for(i=1;i<(DCC_SPEED_TABLE_SIZE/2-2);i++){    //1st half 3..63
        n=start;
        n=n*j;
        n1=mid;
        n1=n1*i;
        n=n+n1;
        n=n/(DCC_SPEED_TABLE_SIZE/2-2);
        speedCurve[i+2]=(unsigned char)n;
        j--;
    }
    j=(DCC_SPEED_TABLE_SIZE/2)-2;
    for(i=1;i<(DCC_SPEED_TABLE_SIZE/2-1);i++){    //2nd half 65..126
        n=mid;
        n=n*j;
        n1=hi;
        n1=n1*i;
        n=n+n1;
        n=n/(DCC_SPEED_TABLE_SIZE/2-1);
        speedCurve[i+(DCC_SPEED_TABLE_SIZE/2)]=(unsigned char)n;
        j--;
    }
    //fix these
    speedCurve[0]=0;
    speedCurve[1]=0;
    speedCurve[2]=start;
    speedCurve[DCC_SPEED_TABLE_SIZE/2]=mid;
    speedCurve[DCC_SPEED_TABLE_SIZE-1]=hi;
}

int readCV(unsigned int cv){
    unsigned char n;
    int r=-1;
    if(cv<DCC_CVS_IN_USE){
        n=CVindex[cv];
        if(n<CONFIG_SIZE){        //else unimplemented
            r=loco.config.a[n];
        }        
    }    
    if(cv==DCC_MANUF_ID_CV){r=DCC_MANUF_ID;}
    if(cv==DCC_MODEL_ID_CV){r=DCC_MODEL_ID;}    
    return r;
}

unsigned char writeCV(unsigned int cv, unsigned char d){    //0 on fail
    unsigned char n;
    if(cv>=DCC_CVS_IN_USE){
        return 0;
    }else{
        n=CVindex[cv];
        if(n>=CONFIG_SIZE){
            return 0;
        }else{
            d=d | CVORmask[n]; //masking unused bits
            d=d & CVANDmask[n]; //masking unused bits
            //implementation:
            loco.config.a[n]=d;
            if(nvConfig.a[n]!=d){nvConfig.a[n]=d;}  //save to nv if not the same
            //followups, recalculate speed table etc
            if((cv==2)||(cv==5)||(cv==6)){
                mapSpeed(loco.config.CV2,loco.config.CV6,loco.config.CV5);    //current values
            }
        }        
    }
    return 1;
}

char checkPacketMatch(dcc_packet_t* p1, dcc_packet_t* p2){    //simple equality over valid bytes (up to len)
    unsigned char n=p1->len;
    unsigned char i;
    if(n!=p2->len){return 0;}
    for(i=0;i<n;i++){if(p1->data[0]!=p2->data[0]){return 0;}}
    return 1;
}

unsigned char getPWM(loco_t* t, unsigned char v){    //check all necessary parameters and supply a value
    unsigned int n=speedCurve[t->live.speed];
    if(v<DCC_NORMALISED_VOLTAGE_MIN){v=DCC_NORMALISED_VOLTAGE_MIN;} //avoid /0
    if(t->config.voltageControl){
        n=(n*DCC_NORMALISED_VOLTAGE)/v;
        if(n>255){n=255;}
        return (unsigned char)n;
    }else{
        return (unsigned char)n;
    }    
}

unsigned char getFuncPWM(char s, unsigned char v){  //find PWM setting to match s with supply=v, v is in 0.1V
    unsigned int n=s;
    if(v<DCC_NORMALISED_VOLTAGE_MIN){v=DCC_NORMALISED_VOLTAGE_MIN;} //avoid /0
    n=(n*2560)/v;
    if(n>255){n=255;}
    return (unsigned char)n;
}

