//DCC (digital command control)
//structs, defs etc
#ifndef DCC_H
#define	DCC_H

#include "globalPins.h"
#include "portbits.h"
#include "io.h"

#define DCC_SPEEDCURVE_DEFAULT .CV2=0,.CV5=0,.CV6=0   //start/hi/mid
#define DCC_FX_DEFAULT .CV49=255,.CV50=255,.CV51=255,.CV52=255      //all solid
#define DCC_WIREMAPPING_DEFAULT .CV33=1,.CV34=2,.CV35=4,.CV36=8,.CV37=0     //default mapping
#define DCC_CUSTOM_DEFAULT .voltageControl=0,.functionVoltage=0

#define CONFIG_DEFAULT {.use28step=1,.shortAddress=3,.longAddress=0|0xC000,DCC_WIREMAPPING_DEFAULT,DCC_FX_DEFAULT,DCC_SPEEDCURVE_DEFAULT,.CV11=0,DCC_CUSTOM_DEFAULT}
//#define CONFIG_DEFAULT {.use28step=1,.useLongAddress=1,.shortAddress=3,.longAddress=400|0xC000,DCC_WIREMAPPING_DEFAULT,DCC_FX_DEFAULT,DCC_SPEEDCURVE_DEFAULT,0}
//#define CONFIG_DEFAULT {.use28step=1,.shortAddress=3,.consistAddress=9,.consistReverse=1,DCC_WIREMAPPING_DEFAULT,DCC_FX_DEFAULT,DCC_SPEEDCURVE_DEFAULT,0}

#define DCC_MANUF_ID (13)
#define DCC_MODEL_ID (0x5C)
#define DCC_MANUF_ID_CV (8)
#define DCC_MODEL_ID_CV (7)
#define DCC_SPEED_TABLE_SIZE 128
#define DCC_PREAMBLE_BITS 11
#define DCC_DATA_BYTES 7
#define DCC_ACCEL_PERIOD_COUNT (318)
//packet filtering masks/values
#define DCC_BASELINE (0x40)
#define DCC_BASELINE_MASK (0xC0)
#define DCC_EXT_128_STEP (0x3F)
#define DCC_EXT_128_STEP_MASK (0xFF)
#define DCC_EXT_FUNCTION_1 (0x80)
#define DCC_EXT_FUNCTION_1_MASK (0xE0)
#define DCC_EXT_FUNCTION_2 (0xA0)
#define DCC_EXT_FUNCTION_2_MASK (0xE0)
#define DCC_EXT_CONSIST (0x12)
#define DCC_EXT_CONSIST_MASK (0xFE)
#define DCC_EXT_LONG_OPS_WRITE (0xEC)
#define DCC_EXT_LONG_OPS_WRITE_MASK (0xFC)
#define DCC_BASELINE_DIR_BIT (0x20)
#define DCC_BASELINE_F0_BIT (0x10)
#define DCC_BASELINE_SPEED_BITS (0x0F)
#define DCC_BASELINE_EXTRA_SPEED_BIT (0x10)
#define DCC_EXT_DIR_BIT (0x80)
#define DCC_EXT_SPEED_BITS (0x7F)
#define DCC_EXT_F0_MASK (0x10)
#define DCC_EXT_F1_MASK (0x01)
#define DCC_EXT_F2_MASK (0x02)
#define DCC_EXT_F3_MASK (0x04)
#define DCC_EXT_F4_MASK (0x08)
#define DCC_SERVICE_PACKET_MASK (0xF0)
#define DCC_SERVICE_PACKET (0x70)
#define DCC_FUNCTION_WIRE_COUNT (4)
#define DCC_CVS_IN_USE (53)
#define DCC_ACK_TIMER_COUNT (255)   //5.6ms
//#define DCC_ACK_TIMER_COUNT (227)   //5.0ms
#define DCC_PACKET_TIMEOUT_COUNT (1363)
#define DCC_NORMALISED_VOLTAGE (160)    //in steps of 0.1V, ie 160=16V
#define DCC_NORMALISED_VOLTAGE_MIN (60)  //in steps of 0.1V, ie 60=6V

//packet decode
typedef enum{
    ADD_INVALID=0,
    ADD_SHORT=1,
    ADD_LONG,
    ADD_BROADCAST,
    ADD_IDLE,   
    ADD_RESET,
    ADD_SERVICE,
    ADD_CONSIST            
} dcc_address_type_t;

typedef struct{
    unsigned int address;
    dcc_address_type_t type;
} dcc_address_t;

typedef enum{
    DCC_MODE_NONE=0,
    DCC_MODE_RESET,
    DCC_MODE_SERVICE,
    DCC_MODE_OPS         
} dcc_loco_mode_t;

//based roughly on TCS decoders
#define CONFIG_SIZE 21
typedef union{
struct {    //these are save to non-volatile
        unsigned int reverse:1;
        unsigned int use28step:1;
        unsigned int enableDC:1;
        unsigned int biDiComms:1;        
        unsigned int speedTable:1;
        unsigned int useLongAddress:1; //CV29 =[0]
        unsigned int :2;
        unsigned int shortAddress:7; //CV1 =[1]
        unsigned int :1;
        unsigned int consistAddress:7; //CV19=[2]
        unsigned int consistReverse:1;
        unsigned int longAddress;    //CV17/CV18:16 bits=[4][3] (little endian)
        unsigned char CV2;      //[5]
        unsigned char CV3;      //[6]
        unsigned char CV4;      //[7]
        unsigned char CV5;      //[8]
        unsigned char CV6;      //[9]
        unsigned char CV33;    //F0F white=1  [10]
        unsigned char CV34;    //F0R yellow=2 [11]
        unsigned char CV35;    //F1 green=4  [12]
        unsigned char CV36;    //F2 brown=8  [13]
        unsigned char CV37;    //F3 none=0   [14]
        unsigned char CV49;     //white      [15]
        unsigned char CV50;     //yellow     [16]
        unsigned char CV51;     //green      [17]
        unsigned char CV52;     //brown      [18]
        unsigned char CV11;   //loco timeout [19]
        unsigned int functionVoltage:4;  //CV47, custom features [20]
        unsigned int voltageControl:1;
        unsigned int :3;
};
    unsigned char a[CONFIG_SIZE];   //map to bytes to allow programming byte access
}dcc_config_t;

//live loco attributes
typedef struct{
    unsigned char speed:7;        //subject to accel/decel, but not speed mapping
    unsigned int :1;
    unsigned char targetSpeed:7;  //per instructions on wire
    unsigned int :1;
    unsigned int dir:1;         //forward=1 as per baseline packet
    unsigned int targetDir:1;
    unsigned int F0:1;
    unsigned int F1:1;
    unsigned int F2:1;
    unsigned int F3:1;
    unsigned int F4:1;      //not used, but present in packets that are decoded
    dcc_loco_mode_t serviceMode;
    unsigned char pageReg;  //for service mode page mode addressing
} dcc_live_loco_t;

//loco
typedef struct{
    dcc_live_loco_t live;
    dcc_config_t config;
} loco_t;

//packet contents
typedef struct {
    unsigned char data[DCC_DATA_BYTES];  //max for non-extended packets
    unsigned char checksum;     //not used for reception
    unsigned char len;      //data bytes (not inc checksum)
} dcc_packet_t;

extern loco_t loco;
extern dcc_packet_t dccIn,dccRx,lastPacket;
extern const dcc_config_t configDefault;
extern __eeprom dcc_config_t nvConfig;
extern unsigned char wireState[DCC_FUNCTION_WIRE_COUNT]; //white, yellow, green, brown
extern unsigned char speedCurve[DCC_SPEED_TABLE_SIZE];
extern unsigned char ackTimer;
extern unsigned char ackPol;
extern unsigned int accelTimer;
extern unsigned int accelCount;
extern unsigned char locoTimeout;
void dccInit(void); //io, timers, structs
void cleanPacket(dcc_packet_t* p);    //check checksum, void if invalid
void dccISR(void);
dcc_address_t getAddress(dcc_packet_t* p);
void parseDCC(dcc_packet_t* p, loco_t* t);  //act on packet with respect to loco
void resetLoco(loco_t* t);  //timeout/service mode etc
void mapSpeed(unsigned char start, unsigned char mid, unsigned char hi);    //note these are 8-bit values, but map is 7-bit
int readCV(unsigned int cv);    //-1 on fail
unsigned char writeCV(unsigned int cv, unsigned char d);    //0 on fail
char checkPacketMatch(dcc_packet_t* p1, dcc_packet_t* p2);    //simple equality over valid bytes (up to len)
void parseDCCops(dcc_packet_t* p,loco_t* t);  //act on packet with respect to t
void parseDCCopsOther(dcc_packet_t* p,loco_t* t);  //act on packet with respect to t
void parseDCCservice(dcc_packet_t* p,loco_t* t);  //act on packet with respect to t
void parseDCCbroadcast(dcc_packet_t* p,loco_t* t);    //handle broadcast here, handle stop only
unsigned char getPWM(loco_t* t, unsigned char v);    //check all necessary parameters and supply a value, v is in 0.1V
unsigned char getFuncPWM(char s, unsigned char v);  //find PWM setting to match s with supply=v, v is in 0.1V

#endif	/* DCC_H */

