#include "esp_camera.h"
#include <WiFi.h>

#define CAMERA_MODEL_AI_THINKER // Has PSRAM
#include "camera_pins.h"

//See the following for useful functions/structs/enums:
//sensor.h
//esp_camera.h
//img_converters.h

int e;      //error status
sensor_t * s;

typedef struct __attribute__ ((packed)) {        
  uint16_t  magic;            // Magic identifier: BM
  uint32_t  size;             // File size in bytes
  uint16_t  p1;
  uint16_t  p2;
  uint32_t  offset;           // Offset to image data
  uint32_t  dib_header_size;  // DIB Header size in bytes
  int32_t   width;            // Width of the image
  int32_t   height;           // Height of image
  uint16_t  planes;           // Number of colour planes
  uint16_t  bpp;              // Bits per pixel
  uint32_t  comp;             // Compression type
  uint32_t  image_bytes;      // Image size in bytes
  int32_t   x_res;            // Pixels per meter
  int32_t   y_res;            // Pixels per meter
  uint32_t  num_colors;       // Number of colours  
  uint32_t  important_colors; // Important colours 
} BMPHeader_t;
#define BM_MAGIC (0x4D42)

BMPHeader_t* bmHeader;

char asciiArt[]=" .:-=+*#%@";
#define ASCIIART 10
#define ASCIIW 50
#define ASCIIH 20

#include "FS.h"
#include "SD.h"
#include "SPI.h"
#define VSPI_MISO   2
#define VSPI_MOSI   15
#define VSPI_SCLK   14
#define VSPI_SS     13
SPIClass * vspi = NULL;
char fName[16]="0000000000";
uint64_t cardSize=0;

void setup() {
  uint16_t camID; //see camera_pid_t enum
  delay(1000);
  Serial.begin(115200);
  Serial.setDebugOutput(true);
  Serial.println();
  //SD card
  vspi = new SPIClass(VSPI);
  vspi->begin(VSPI_SCLK, VSPI_MISO, VSPI_MOSI, VSPI_SS); //SCLK, MISO, MOSI, SS
  if(SD.begin(13,*vspi)){
    Serial.println("SD card initialised.");
    cardSize = SD.cardSize();
    Serial.printf("SD Card Size: %llu MB, %llu MB free.\r\n", cardSize>>20,(cardSize-SD.usedBytes())>>20);
  }else{
    Serial.println("SD card init failed.");
  }
  //camera
  e=camInit();
  if(e==ESP_OK){
    Serial.println("Camera initialised");
  }else{
    Serial.printf("Camera init error %d.\r\n",e);
  }
  s = esp_camera_sensor_get();
  camID = s->id.PID;
  showCamID(camID);
  showHelp();
}

void loop() {
  int d,i,x,y,w,h;
  int t;
  unsigned long tm=0;
  camera_fb_t *fb = NULL;
  uint8_t* out;     //for bitmap data
  size_t out_len;   //length of bitmap data
  if(Serial.available()){
    d=Serial.read();
    if(d=='s'){ //save to SD
      tm=millis();
      e=ESP_OK;
      fb = esp_camera_fb_get();
      if(fb){
        Serial.println("Framebuffer captured");
        Serial.printf("%d bytes, %d x %d pixels.\r\n",fb->len,fb->width,fb->height);
        showPixFormat(fb->format);
        setFileName(millis());
        fName[1]='/';
        //Serial.println(fName+1);    //restrict to last 8 chars for compatibility
        if(!SD.exists(&fName[1])){
          File file = SD.open(&fName[1],FILE_WRITE,true);
          if(file){
            i=file.write(fb->buf,fb->len);
            Serial.printf("%d bytes written to %s.\r\n",i,&fName[1]);
            Serial.printf("SD Card: %llu MB free.\r\n", (cardSize-SD.usedBytes())>>20);
          }else{
            Serial.printf("Could not create file %s\r\n",&fName[1]);
          }
          file.close();
        }else{
          Serial.printf("File with same name already exists.");
        }
      }else{
        Serial.println("Camera capture failed");
      }
      esp_camera_fb_return(fb);   //need to release this when done
      Serial.printf("%lu ms taken.\r\n\r\n",millis()-tm);
    }    
    if(d=='a'){
      tm=millis();
      camera_fb_t *fb = NULL;
      e=ESP_OK;
      fb = esp_camera_fb_get();
      if(fb){
        if(frame2bmp(fb,&out,&out_len)){    //convert from jpg to bmp
          bmHeader=(BMPHeader_t*)out;
          w=abs(bmHeader->width);
          h=abs(bmHeader->height);
          if((w>50)&&(h>50)){
            for(y=0;y<ASCIIH;y++){
              for(x=0;x<ASCIIW;x++){
                Serial.write(asciiArt[greyScale(getPixel(bmHeader,x*w/ASCIIW,y*h/ASCIIH))*ASCIIART/256]);
              }
              Serial.println();
            }
          }else{
            Serial.println("BMP corrupted.");    
          }
        }else{
          Serial.println("Convert to BMP capture failed.");  
        }
      }else{
        Serial.println("Camera capture failed.");
      }
      free(out);                  //release bitmap buffer
      esp_camera_fb_return(fb);   //need to release this when done
      Serial.printf("%lu ms taken.\r\n\r\n",millis()-tm);
    }
    if(d=='p'){
      tm=millis();
      e=ESP_OK;
      fb = esp_camera_fb_get();
      if(fb){
        Serial.println("Framebuffer captured");
        Serial.printf("%d bytes, %d x %d pixels.\r\n",fb->len,fb->width,fb->height);
        showPixFormat(fb->format);
      }else{
        Serial.println("Camera capture failed");
      }
      if(frame2bmp(fb,&out,&out_len)){    //convert from jpg to bmp
        Serial.println("Convert to bitmap OK");
        Serial.printf("Bitmap is %d bytes.\r\n",out_len);
        hexDump(out,64);      //show BMP headers
        bmHeader=(BMPHeader_t*)out;
        Serial.printf("Bitmap is %d x %d pixels.\r\n",abs(bmHeader->width),abs(bmHeader->height));
        if(bmHeader->magic==BM_MAGIC){
          Serial.println("BM header found");
          Serial.printf("Size: %d bytes\r\n",bmHeader->size);
          Serial.printf("Data offset: 0x%x\r\n",bmHeader->offset);
        }else{
          Serial.println("BM header not found");
        }
      }else{
        Serial.println("Convert to bitmap fail");
      }
      free(out);                  //release bitmap buffer
      esp_camera_fb_return(fb);   //need to release this when done
      Serial.printf("%lu ms taken.\r\n\r\n",millis()-tm);
    }
    if((d>='0')&&(d<='9')){
      e=s->set_framesize(s, (framesize_t)(d-'0'));
      if(e==ESP_OK){
        Serial.println("Set OK");
      }else{
        Serial.printf("Error %d.\r\n",e);
      }
      d='i';  //show info results too
    }
    if(d=='v'){
      t=s->status.vflip;
      if(t){
        s->set_vflip(s,0);
      }else{
        s->set_vflip(s,1);
      }
      d='i';
    }
    if(d=='h'){
      t=s->status.hmirror;
      if(t){
        s->set_hmirror(s,0);
      }else{
        s->set_hmirror(s,1);
      }
      d='i';
    }
    if(d=='c'){
      t=s->status.contrast;
      t=t-1;
      if(t<-2){t=-2;}
      s->set_contrast(s,t);
      d='i';
    }
    if(d=='C'){
      t=s->status.contrast;
      t=t+1;
      if(t>2){t=2;}
      s->set_contrast(s,t);
      d='i';
    }
    if(d=='b'){
      t=s->status.brightness;
      t=t-1;
      if(t<-2){t=-2;}
      s->set_brightness(s,t);
      d='i';
    }
    if(d=='B'){
      t=s->status.brightness;
      t=t+1;
      if(t>2){t=2;}
      s->set_brightness(s,t);
      d='i';
    }
    if(d=='q'){
      t=s->status.quality;
      t=t-1;
      if(t<0){t=0;}
      s->set_quality(s,t);
      d='i';
    }
    if(d=='Q'){
      t=s->status.quality;
      t=t+1;
      if(t>63){t=63;}
      s->set_quality(s,t);
      d='i';
    }
    if(d=='i'){
      Serial.println("Sensor information and status");
      showFrameSize(s->status.framesize);
      showPixFormat(s->pixformat);
      Serial.printf("Brightness: %d\r\n",s->status.brightness);
      Serial.printf("Contrast: %d\r\n",s->status.contrast);
      Serial.printf("H-Flip: %d\r\n",s->status.hmirror);
      Serial.printf("V-Flip: %d\r\n",s->status.vflip);
      Serial.printf("Quality: %d\r\n",s->status.quality);
      Serial.println("");
    }
    if(d=='?'){showHelp();}
  }
}

void showHelp(){
  Serial.println("Press ? for this help.");
  Serial.println("Press c/C to decrease/increase contrast.");
  Serial.println("Press b/B to decrease/increase brightness.");
  Serial.println("Press q/Q to decrease/increase (lower number = higher quality).");
  Serial.println("Press 0-9 to set frame size.");
  Serial.println("Press p to take photo.");
  Serial.println("Press a to take photo and display as ASCII art.");
  Serial.println("Press i for status.");
  Serial.println("Press h to toggle horizontal flip.");
  Serial.println("Press v to toggle vertical flip.");
  Serial.println("Press s save JPG image to SD card.");
  Serial.println();
}

void showFrameSize(int n){
  switch(n){
    case FRAMESIZE_96X96: Serial.println("96X96");break;
    case FRAMESIZE_QQVGA: Serial.println("QQVGA 160x120");break;
    case FRAMESIZE_QCIF: Serial.println("QCIF 176x144");break;
    case FRAMESIZE_HQVGA: Serial.println("HQVGA 240x176");break;
    case FRAMESIZE_240X240: Serial.println("240X240");break;
    case FRAMESIZE_QVGA: Serial.println("QVGA 320x240");break;
    case FRAMESIZE_CIF: Serial.println("CIF 400x296");break;
    case FRAMESIZE_HVGA: Serial.println("HVGA 480x320");break;
    case FRAMESIZE_VGA: Serial.println("VGA 640x480");break;
    case FRAMESIZE_SVGA: Serial.println("SVGA 800x600");break;
    case FRAMESIZE_XGA: Serial.println("XGA 1024x768");break;
    case FRAMESIZE_HD: Serial.println("HD 1280x720");break;
    case FRAMESIZE_SXGA: Serial.println("SXGA 1280x1024");break;
    case FRAMESIZE_UXGA: Serial.println("UXGA 1600x1200");break;
    default: Serial.println("Frame size unknown.");break;
  }
}

void showCamID(int n){
  switch(n){
    case OV9650_PID: Serial.println("Camera is OV9650.");break;
    case OV7725_PID: Serial.println("Camera is OV7725.");break;
    case OV2640_PID: Serial.println("Camera is OV2640.");break;
    case OV3660_PID: Serial.println("Camera is OV3660.");break;
    case OV5640_PID: Serial.println("Camera is OV5640.");break;
    case OV7670_PID: Serial.println("Camera is OV7670.");break;
    case NT99141_PID: Serial.println("Camera is NT99141.");break;
    case GC2145_PID: Serial.println("Camera is GC2145.");break;
    case GC032A_PID: Serial.println("Camera is GC032A.");break;
    case GC0308_PID: Serial.println("Camera is GC0308.");break;
    case BF3005_PID: Serial.println("Camera is BF3005.");break;
    case BF20A6_PID: Serial.println("Camera is BF20A6.");break;
    case SC101IOT_PID: Serial.println("Camera is SC101IOT.");break;
    case SC030IOT_PID: Serial.println("Camera is SC030IOT.");break;
    case SC031GS_PID: Serial.println("Camera is SC031GS.");break;
    default: Serial.println("Camera ID unknown.");break;
  }
}

void showPixFormat(int n){
  switch(n){
    case PIXFORMAT_RGB565: Serial.println("PIXFORMAT_RGB565");break;
    case PIXFORMAT_YUV422: Serial.println("PIXFORMAT_YUV422");break;
    case PIXFORMAT_YUV420: Serial.println("PIXFORMAT_YUV420");break;
    case PIXFORMAT_GRAYSCALE: Serial.println("PIXFORMAT_GRAYSCALE");break;
    case PIXFORMAT_JPEG: Serial.println("PIXFORMAT_JPEG");break;
    case PIXFORMAT_RGB888: Serial.println("PIXFORMAT_RGB888");break;
    case PIXFORMAT_RAW: Serial.println("PIXFORMAT_RAW");break;
    case PIXFORMAT_RGB444: Serial.println("PIXFORMAT_RGB444");break;
    case PIXFORMAT_RGB555: Serial.println("PIXFORMAT_RGB555");break;
    default: Serial.println("Format unknown.");break;
  }
}

int camInit(){
  camera_config_t config;
  config.ledc_channel = LEDC_CHANNEL_0;
  config.ledc_timer = LEDC_TIMER_0;
  config.pin_d0 = Y2_GPIO_NUM;
  config.pin_d1 = Y3_GPIO_NUM;
  config.pin_d2 = Y4_GPIO_NUM;
  config.pin_d3 = Y5_GPIO_NUM;
  config.pin_d4 = Y6_GPIO_NUM;
  config.pin_d5 = Y7_GPIO_NUM;
  config.pin_d6 = Y8_GPIO_NUM;
  config.pin_d7 = Y9_GPIO_NUM;
  config.pin_xclk = XCLK_GPIO_NUM;
  config.pin_pclk = PCLK_GPIO_NUM;
  config.pin_vsync = VSYNC_GPIO_NUM;
  config.pin_href = HREF_GPIO_NUM;
  config.pin_sccb_sda = SIOD_GPIO_NUM;
  config.pin_sccb_scl = SIOC_GPIO_NUM;
  config.pin_pwdn = PWDN_GPIO_NUM;
  config.pin_reset = RESET_GPIO_NUM;
  config.xclk_freq_hz = 20000000;
  config.frame_size = FRAMESIZE_UXGA;
  config.pixel_format = PIXFORMAT_JPEG; // for streaming
  //config.pixel_format = PIXFORMAT_RGB565; // for face detection/recognition
  config.grab_mode = CAMERA_GRAB_WHEN_EMPTY;
  config.fb_location = CAMERA_FB_IN_PSRAM;
  config.jpeg_quality = 12;
  config.fb_count = 1;
  
  // if PSRAM IC present, init with UXGA resolution and higher JPEG quality
  //                      for larger pre-allocated frame buffer.
  if(config.pixel_format == PIXFORMAT_JPEG){
    if(psramFound()){
      config.jpeg_quality = 10;
      config.fb_count = 2;
      config.grab_mode = CAMERA_GRAB_LATEST;
    } else {
      // Limit the frame size when PSRAM is not available
      config.frame_size = FRAMESIZE_SVGA;
      config.fb_location = CAMERA_FB_IN_DRAM;
    }
  } else {
    // Best option for face detection/recognition
    config.frame_size = FRAMESIZE_240X240;
  }

  // camera init
  esp_err_t err = esp_camera_init(&config);
  if (err != ESP_OK) {
    return err;
  }

  sensor_t * s = esp_camera_sensor_get();
  // initial sensors are flipped vertically and colors are a bit saturated
  if (s->id.PID == OV3660_PID) {
    s->set_vflip(s, 1); // flip it back
    s->set_brightness(s, 1); // up the brightness just a bit
    s->set_saturation(s, -2); // lower the saturation
  }
  // drop down frame size for higher initial frame rate
  if(config.pixel_format == PIXFORMAT_JPEG){
    s->set_framesize(s, FRAMESIZE_QVGA);
  }
  return ESP_OK;
}

#define LINESIZE 16

void hexDump(uint8_t* a,int n){
  int i,j;
  Serial.println("                       0  1  2  3  4  5  6  7  8  9  A  B  C  D  E  F");
  for(j=0;j<n;j=j+LINESIZE){  //16 per line
    if(j<16){Serial.write('0');}
    if(j<256){Serial.write('0');}
    if(j<4096){Serial.write('0');}
    Serial.print(j,HEX);
    Serial.write(' ');
    for(i=0;i<LINESIZE;i++){
      if((a[i+j]>31)&&(a[i+j]<127)){  //ascii
        Serial.write(a[i+j]);
      }else{
        Serial.write('.');
      }
    }
    for(i=0;i<LINESIZE;i++){
      Serial.write(' ');
      if (a[i+j]<0x10){Serial.write('0');}          
      Serial.print(a[i+j], HEX);
    }
    Serial.println();
  }
}

uint32_t getPixel(BMPHeader_t* b, int x, int y){   //returns as 0x00RRGGBB or 0xFFFFFFFF on fail
  uint32_t r=0;
  uint8_t* d;
  uint8_t* p;
  int lineLen;
  if(x<0){return 0xFFFFFFFF;}
  if(y<0){return 0xFFFFFFFF;}
  if(x>=abs(b->width)){return 0xFFFFFFFF;}
  if(y>=abs(b->height)){return 0xFFFFFFFF;}
  d=((uint8_t*)b)+b->offset;      //start of Data
  lineLen=(abs(b->width)*3+3)&0xFFFFFFFC;   //32bit padded
  if(b->height<0){  //negative height => scan from top to bottom
    p=d+x*3+y*lineLen;
    r=(p[0]<<16)|(p[1]<<8)|(p[2]);
    return r;    
  }else{  //scan from bottom to top
    p=d+x*3+(abs(b->height)-1-y)*lineLen;
    r=(p[0]<<16)|(p[1]<<8)|(p[2]);
    return r;    
  }
  return 0xFFFFFFFF;
}

uint8_t greyScale(uint32_t c){
  int n;
  n=(c>>16)&0xFF;
  n=n+((c>>8)&0xFF);
  n=n+((c)&0xFF);
  n=n/3;
  return n;
}

void setFileName(unsigned long n){
  int i;
  unsigned long p=1000000000UL;
  unsigned long q;
  for(i=0;i<10;i++){
    q=n/p;
    fName[i]=q+'0';
    n=n-p*q;
    p=p/10;
  }
  fName[10]='.';
  fName[11]='J';
  fName[12]='P';
  fName[13]='G';
  fName[14]=0;
}