#include "MCUFRIEND_kbv.h"
#if defined(USE_SERIAL)
 //#include <SPI.h>
#include "mcufriend_serial.h"
 //uint8_t running;
#elif defined(USE_KEIL)
#include "pin_freedom_8.h"
#define CTL_INIT()   { RD_OUTPUT; WR_OUTPUT; CD_OUTPUT; CS_OUTPUT; RESET_OUTPUT; }
#define WriteCmd(x)  { CD_COMMAND; write16(x); }
#define WriteData(x) { CD_DATA; write16(x); }
#else
#include "mcufriend_shield.h"
#endif

#define wait_ms(ms)  delay(ms)
#define AUTO_READINC    (1<<0)
#define TWO_PARM_WINDOW (1<<1)
#define MV_AXIS         (1<<2)
#define REV_SCREEN      (1<<3)
#define READ_BGR        (1<<4)
#define XSA_XEA_16BIT   (1<<5)

#if defined USE_GFX_KBV
MCUFRIEND_kbv::MCUFRIEND_kbv():Adafruit_GFX(240, 320)
{
    // we can not access GPIO pins until AHB has been enabled.
}
#else
MCUFRIEND_kbv::MCUFRIEND_kbv(int CS, int RS, int WR, int RD, int RST):Adafruit_GFX(240, 320)
{
    // we can not access GPIO pins until AHB has been enabled.
}
#endif

static uint8_t done_reset;

void MCUFRIEND_kbv::reset(void)
{
    done_reset = 1;
    setWriteDir();
    CTL_INIT();
    CS_IDLE;
    RD_IDLE;
    WR_IDLE;
    RESET_IDLE;
    wait_ms(50);
    RESET_ACTIVE;
    wait_ms(100);
    RESET_IDLE;
    wait_ms(100);
}

void MCUFRIEND_kbv::WriteCmdData(uint16_t cmd, uint16_t dat)
{
    CS_ACTIVE;
    WriteCmd(cmd);
    WriteData(dat);
    CS_IDLE;
}

static void WriteCmdParamN(uint16_t cmd, int8_t N, uint8_t * block)
 //void MCUFRIEND_kbv::WriteCmdParamN(uint16_t cmd, int8_t N, uint8_t * block)
{
    CS_ACTIVE;
    WriteCmd(cmd);
    CD_DATA;
    while (N-- > 0) {
        uint8_t u8 = *block++;
        write8(u8);
    }
    CS_IDLE;
}

static inline void WriteCmdParam4(uint8_t cmd, uint8_t d1, uint8_t d2, uint8_t d3, uint8_t d4)
{
#if 1
    //    uint8_t d[4] = {d1, d2, d3, d4};
    uint8_t d[4];
    d[0] = d1, d[1] = d2, d[2] = d3, d[3] = d4;
    WriteCmdParamN(cmd, 4, d);
#else
    CS_ACTIVE;
    WriteCmd(cmd);
    CD_DATA;
    write8(d1);
    write8(d2);
    write8(d3);
    write8(d4);
    CS_IDLE;
#endif
}

uint16_t MCUFRIEND_kbv::readReg(uint16_t reg)
{
    uint16_t ret;
    if (!done_reset)
        reset();
    CS_ACTIVE;
    WriteCmd(reg);
    setReadDir();
    CD_DATA;
#if defined(USE_SERIAL)
    //    READ_8(ret);    //dummy
#endif
    READ_16(ret);
    RD_IDLE;
    CS_IDLE;
    setWriteDir();
    return ret;
}

uint32_t MCUFRIEND_kbv::readReg32(uint16_t reg)
{
    uint16_t h, l;
    CS_ACTIVE;
    WriteCmd(reg);
    setReadDir();
    CD_DATA;
    READ_16(h);
    READ_16(l);
    RD_IDLE;
    CS_IDLE;
    setWriteDir();
    return ((uint32_t) h << 16) | (l);
}

uint16_t MCUFRIEND_kbv::readID(void)
{
    uint16_t ret, ret2;
    uint8_t msb;
#if 0
    ret = readReg(0);
    msb = ret >> 8;
    if ((ret & 0xFF) == 0x89)   // it might be safer to test 0x8989
        return 0x1289;
    // a write-only 16-bit bus will return 0x0012 when reading reg(0x12)
    // a write-only register will return 0x1212 when reading reg(0x12) - I think
    if (ret == 0 || msb == (ret & 0xFF)) {
        ret = readReg32(0x04);  //0x8552 for ST7789V
        if (ret == 0x8552)
            return 0x7789;
        ret = readReg32(0xD3);  //0x9341,0x9488 (0x2200 ? ILI9481)
        if (ret == 0x2200) {
            if ((readReg32(0xBF) & 0xFFFF) == 0x0494)
                return 0x9481;  //0x0494 for ILI9481
        }
        msb = ret >> 8;
        if (ret == 0 || msb == (ret & 0xFF)) {
            if ((readReg32(0xEF) & 0xFFFF) == 0x0493)
                return 0x9327;  //0x0493 for ILI9327
        }
    }
    return ret;
#else
    ret = readReg(0);           //forces a reset() if called before begin()
	if (ret == 0x0089 || ret == 0x8989)
	    return 0x1289;
    ret = readReg32(0xBF);      //for ILI9481
    if (ret == 0x0494)
        return 0x9481;
    ret = readReg32(0xEF);      //for ILI9327
    if (ret == 0x0493)
        return 0x9327;
    ret = readReg32(0x04);      //0x8552 for ST7789V
    if (ret == 0x8552)
        return 0x7789;
    ret = readReg32(0xD3);      //for ILI9488, 9486, 9340, 9341
    msb = ret >> 8;
    if (msb == 0x93 || msb == 0x94)
        return ret;             //0x9488, 9486, 9340, 9341
    if (ret == 0x00D3)
        return ret;             //16-bit write-only bus
    return readReg(0);          //0154, 7783, 9320, 9325, 9335, B509
#endif
}

 // independent cursor and window registers.   S6D0154, ST7781 increments.  ILI92320/5 do not.  
int16_t MCUFRIEND_kbv::readGRAM(int16_t x, int16_t y, uint16_t * block, int16_t w, int16_t h)
{
    uint16_t ret, dummy, _MR = _MW;
    int16_t n = w * h, row = 0, col = 0;
    if (_lcd_capable & TWO_PARM_WINDOW)
        _MR = 0x2E;
    setAddrWindow(x, y, x + w - 1, y + h - 1);
    while (n > 0) {
        if (!(_lcd_capable & TWO_PARM_WINDOW)) {
            WriteCmdData(_MC, x + col);
            WriteCmdData(_MP, y + row);
        }
        CS_ACTIVE;
        WriteCmd(_MR);
        setReadDir();
        CD_DATA;
        switch (_lcd_ID) {
        case 0x0139:
        case 0x0154:
        case 0x7783:
            READ_16(dummy);
            goto loop;
        case 0x1289:
        case 0x9327:
        case 0x9481:
        case 0x9486:
            READ_8(dummy);
        case 0x3082:
          loop:
            while (n-- > 0) {
                READ_16(ret);   //already in 565 format
                if (_lcd_capable & READ_BGR)
                    ret = (ret & 0x07E0) | (ret >> 11) | (ret << 11);
                *block++ = ret;
            }
            break;
        case 0x7789:
        case 0x9341:
        case 0x9488:
            uint8_t r, g, b, tmp;
            READ_8(r);
            while (n-- > 0) {
                READ_8(r);
                READ_8(g);
                READ_8(b);
                if (_lcd_capable & READ_BGR) {
                    tmp = r;
                    r = b;
                    b = tmp;
                }
                *block++ = color565(r, g, b);
            }
            break;
        default:
            // 0x6809, 0x9320, 0x9325, 0x9335
            READ_16(dummy);
            READ_16(ret);
            if (_lcd_capable & READ_BGR)
                ret = (ret & 0x07E0) | (ret >> 11) | (ret << 11);
            *block++ = ret;
            if (++col >= w) {
                col = 0;
                if (++row >= h)
                    row = 0;
            }
            n--;
            break;
        }
        RD_IDLE;
        CS_IDLE;
        setWriteDir();
    }
    if (!(_lcd_capable & TWO_PARM_WINDOW))
        setAddrWindow(0, 0, width() - 1, height() - 1);
    return 0;
}

void MCUFRIEND_kbv::setRotation(uint8_t r)
{
    uint16_t GS, SS, ORG, REV = _lcd_rev;
    uint8_t val, d[3];
    //    Adafruit_GFX::setRotation(r & 3);   // needs to know the name of the base class
    rotation = r & 3;           // just perform the operation ourselves on the protected variables
    _width = (rotation & 1) ? HEIGHT : WIDTH;
    _height = (rotation & 1) ? WIDTH : HEIGHT;
    switch (rotation) {
    case 0:                    //PORTRAIT:
        val = 0x40;             //MY=0, MX=1, MV=0, ML=0
        break;
    case 1:                    //LANDSCAPE:
        val = 0xF0;             //MY=1, MX=1, MV=1, ML=1
        break;
    case 2:                    //PORTRAIT_REV:
        val = 0x90;             //MY=1, MX=0, MV=0, ML=1
        break;
    case 3:                    //LANDSCAPE_REV:
        val = 0x20;             //MY=0, MX=0, MV=1, ML=0
        break;
    }
    switch (_lcd_ID) {
    case 0x6809:
    case 0x9325:
    case 0x4535:
    case 0x7783:
        val ^= 0x80;            //.kbv ILI9320 has NOT inverted logic on GS (MY)
    case 0x9320:
    case 0x9335:
        _MC = 0x20, _MP = 0x21, _MW = 0x22, _SC = 0x50, _EC = 0x51, _SP = 0x52, _EP = 0x53;
        GS = (val & 0x80) ? (1 << 15) : 0;
        SS = (val & 0x40) ? (1 << 8) : 0;
        ORG = (val & 0x20) ? (1 << 3) : 0;
        WriteCmdData(0x01, SS); // set Driver Output Control
        WriteCmdData(0x60, GS | 0x2700);        // Gate Scan Line (0xA700)
        WriteCmdData(0x03, ORG | 0x1030);       // set GRAM write direction and BGR=1.
        break;
    case 0x1289:
        _MC = 0x4E, _MP = 0x4F, _MW = 0x22, _SC = 0x44, _EC = 0x44, _SP = 0x45, _EP = 0x46;
        if (rotation & 1)
            val ^= 0xD0;        // exchange Landscape modes
        GS = (val & 0x80) ? (1 << 14) | (1 << 12) : 0;  //called TB (top-bottom)
        SS = (val & 0x40) ? (1 << 9) : 0;       //called RL (right-left)
        ORG = (val & 0x20) ? (1 << 3) : 0;      //called AM
        _lcd_drivOut = GS | SS | (REV << 13) | 0x093F;  //REV=0, BGR=1, MUX=319
        WriteCmdData(0x01, _lcd_drivOut);       // set Driver Output Control
        WriteCmdData(0x11, ORG | 0x6070);       // set GRAM write direction and BGR=1.
        break;
    case 0x0139:
        _MC = 0x20, _MP = 0x21, _MW = 0x22, _SC = 0x46, _EC = 0x46, _SP = 0x48, _EP = 0x47;
        goto S6Dxxxx;
    case 0x0154:
        _MC = 0x20, _MP = 0x21, _MW = 0x22, _SC = 0x37, _EC = 0x36, _SP = 0x39, _EP = 0x38;
      S6Dxxxx:
        GS = (val & 0x80) ? (1 << 9) : 0;
        SS = (val & 0x40) ? (1 << 8) : 0;
        ORG = (val & 0x20) ? (1 << 3) : 0;
        WriteCmdData(0x01, GS | SS | 0x0028);   // set Driver Output Control
        WriteCmdData(0x03, ORG | 0x1030);       // set GRAM write direction and BGR=1.
        break;
    case 0xB509:
        _MC = 0x200, _MP = 0x201, _MW = 0x202, _SC = 0x210, _EC = 0x211, _SP = 0x212, _EP = 0x213;
        GS = (val & 0x80) ? (1 << 15) : 0;
        SS = (val & 0x40) ? (1 << 8) : 0;
        ORG = (val & 0x20) ? (1 << 3) : 0;
        WriteCmdData(0x01, SS); // set Driver Output Control
        WriteCmdData(0x400, GS | 0x6200);
        WriteCmdData(0x03, ORG | 0x1030);       // set GRAM write direction and BGR=1.
        break;
    case 0x9486:
        _MC = 0x2A, _MP = 0x2B, _MW = 0x2C, _SC = 0x2A, _EC = 0x2A, _SP = 0x2B, _EP = 0x2B;
        GS = (val & 0x80) ? (1 << 6) : 0;       //MY
        SS = (val & 0x40) ? (1 << 5) : 0;       //MX
        val = (val & 0x20) | 0x08;      //MV, BGR
        d[0] = 0;
        d[1] = GS | SS | 0x02;  //MY, MX
        d[2] = 0x3B;
        WriteCmdParamN(0x36, 1, &val);
        WriteCmdParamN(0xB6, 3, d);
        break;
        //    case 0x9327:
    case 0x9481:
        if (val & 0x80)
            val |= 0x01;        //GS
        if ((val & 0x40))
            val |= 0x02;        //SS
        val &= ~0x10;           //ML=0           
    case 0x7789:
    case 0x9327:
    case 0x9341:
    case 0x9488:
        //        if (_lcd_ID == 0x9327) val &= ~0x10;   //ML = 0
        _MC = 0x2A, _MP = 0x2B, _MW = 0x2C, _SC = 0x2A, _EC = 0x2A, _SP = 0x2B, _EP = 0x2B;
        val |= 0x08;            //BGR
        WriteCmdParamN(0x36, 1, &val);
        break;
    case 0x3082:
        _MC = 0x02, _MP = 0x03, _MW = 0x0F, _SC = 0x02, _EC = 0x06, _SP = 0x03, _EP = 0x07;     // (portrait)
        if (val & 0x80)
            val |= 0x02;        //flip vert
        if (!(val & 0x40))
            val |= 0x01;        //flip horiz
        WriteCmdData(0x000D, val & 0x20 ? 0 : 1);       //MV
        val &= ~0x30;           //MV=0, ML=0           
        WriteCmdData(0x000C, val);
        break;
    }
    if ((rotation & 1) && ((_lcd_capable & MV_AXIS) == 0)) {
        uint16_t x;
        x = _MC, _MC = _MP, _MP = x;
        x = _SC, _SC = _SP, _SP = x;    //.kbv check 0139
        x = _EC, _EC = _EP, _EP = x;    //.kbv check 0139
    }
    setAddrWindow(0, 0, width() - 1, height() - 1);
    vertScroll(0, HEIGHT, 0);   //reset scrolling after a rotation
}

void MCUFRIEND_kbv::drawPixel(int16_t x, int16_t y, uint16_t color)
{
    // MCUFRIEND just plots at edge if you try to write outside of the box:
    if (x < 0 || y < 0 || x >= width() || y >= height())
        return;
    if (_lcd_capable & TWO_PARM_WINDOW) {
        WriteCmdParam4(_MC, x >> 8, x, x >> 8, x);
        WriteCmdParam4(_MP, y >> 8, y, y >> 8, y);
    } else {
        WriteCmdData(_MC, x);
        WriteCmdData(_MP, y);
    }
    WriteCmdData(_MW, color);
}

void MCUFRIEND_kbv::setAddrWindow(int16_t x, int16_t y, int16_t x1, int16_t y1)
{
    if (_lcd_capable & TWO_PARM_WINDOW) {
        WriteCmdParam4(_MC, x >> 8, x, x1 >> 8, x1);
        WriteCmdParam4(_MP, y >> 8, y, y1 >> 8, y1);
    } else {
        WriteCmdData(_MC, x);
        WriteCmdData(_MP, y);
        if (_lcd_capable & XSA_XEA_16BIT) {
            if (rotation & 1)
                y1 = y = (y1 << 8) | y;
            else
                x1 = x = (x1 << 8) | x;
        }
        WriteCmdData(_SC, x);
        WriteCmdData(_SP, y);
        WriteCmdData(_EC, x1);
        WriteCmdData(_EP, y1);
    }
}

void MCUFRIEND_kbv::fillRect(int16_t x, int16_t y, int16_t w, int16_t h, uint16_t color)
{
    int16_t end;
    if (w < 0) {
        w = -w;
        x -= w;
    }                           //+ve w
    end = x + w;
    if (x < 0)
        x = 0;
    if (end > width())
        end = width();
    w = end - x;
    if (h < 0) {
        h = -h;
        y -= h;
    }                           //+ve h
    end = y + h;
    if (y < 0)
        y = 0;
    if (end > height())
        end = height();
    h = end - y;

    setAddrWindow(x, y, x + w - 1, y + h - 1);
    CS_ACTIVE;
    WriteCmd(_MW);
    if (h > w) {
        end = h;
        h = w;
        w = end;
    }
    uint8_t hi = color >> 8, lo = color & 0xFF;
    CD_DATA;
    while (h-- > 0) {
        end = w;
#if (defined(__AVR_ATmega2560__) && defined(USE_MEGA_16BIT_SHIELD)) || (defined(__SAM3X8E__) && defined(USE_DUE_16BIT_SHIELD))
#if defined(__SAM3X8E__)
#define DUE_STROBE {WR_ACTIVE;WR_ACTIVE;WR_ACTIVE;WR_ACTIVE;WR_ACTIVE;  WR_IDLE;WR_IDLE;WR_IDLE;WR_IDLE;WR_IDLE;}
#else
#define DUE_STROBE {WR_ACTIVE; WR_IDLE;}
#endif
        write_16(color);        //we could just do the strobe
        lo = end & 7;
        hi = end >> 3;
        if (hi)
            do {
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
                DUE_STROBE;
            } while (--hi > 0);
        while (lo-- > 0) {
            DUE_STROBE;
        }
#else
        do {
            write8(hi);
            write8(lo);
        } while (--end != 0);
#endif
    }
    CS_IDLE;
    if (!(_lcd_capable & TWO_PARM_WINDOW))
        setAddrWindow(0, 0, width() - 1, height() - 1);
}

void MCUFRIEND_kbv::pushColors(uint16_t * block, int16_t n, bool first)
{
    uint16_t color;
    CS_ACTIVE;
    if (first) {
        WriteCmd(_MW);
    }
    CD_DATA;
    while (n-- > 0) {
        color = *block++;
        write16(color);
    }
    CS_IDLE;
}

void MCUFRIEND_kbv::pushColors(const uint8_t * block, int16_t n, bool first)
{
    uint16_t color;
    uint8_t h, l;
    CS_ACTIVE;
    if (first) {
        WriteCmd(_MW);
    }
    CD_DATA;
    while (n-- > 0) {
        l = pgm_read_byte(block++);
        h = pgm_read_byte(block++);
        color = h << 8 | l;
        write16(color);
    }
    CS_IDLE;
}

void MCUFRIEND_kbv::vertScroll(int16_t top, int16_t scrollines, int16_t offset)
{
    int16_t vsp = top + offset; // vertical start position
    int16_t sea = top + scrollines - 1;
    if (vsp < 0)
        vsp += HEIGHT;          //keep in unsigned range
    if (sea >= HEIGHT)
        vsp -= HEIGHT;          //keep in unsigned range
    switch (_lcd_ID) {
    case 0x7783:
        WriteCmdData(0x61, _lcd_rev);   //!NDL, !VLE, REV
        WriteCmdData(0x6A, vsp);        //VL#
        break;
    case 0x0139:
        WriteCmdData(0x41, sea);        //SEA
        WriteCmdData(0x42, top);        //SSA
        WriteCmdData(0x43, vsp - top);  //SST
        break;
    case 0x0154:
        WriteCmdData(0x31, sea);        //SEA
        WriteCmdData(0x32, top);        //SSA
        WriteCmdData(0x33, vsp - top);  //SST
        break;
    case 0x1289:
        WriteCmdData(0x41, vsp);        //VL#
        break;
    case 0xB509:
        WriteCmdData(0x401, (1 << 1) | _lcd_rev);       //VLE, REV 
        WriteCmdData(0x404, vsp);       //VL# 
        break;
    case 0x7789:
    case 0x9327:
    case 0x9341:
    case 0x9481:
    case 0x9486:
    case 0x9488:
        uint8_t d[6];           // for multi-byte parameters
        int16_t bfa;            // bottom fixed area
        bfa = HEIGHT - top - scrollines;
        if (_lcd_ID == 0x9327) {        //panel is wired for 240x432 
            if (rotation == 1 || rotation == 2) {
                if (scrollines == HEIGHT) {
                    scrollines = 432;   // we get a glitch but hey-ho
                    vsp -= 432 - HEIGHT;
                }
                if (vsp < 0)
                    vsp += 432;
            }
            bfa = 432 - top - scrollines;
        }
        d[0] = top >> 8;        //TFA
        d[1] = top;
        d[2] = scrollines >> 8; //VSA
        d[3] = scrollines;
        d[4] = bfa >> 8;        //BFA
        d[5] = bfa;
        WriteCmdParamN(0x33, 6, d);
        d[0] = vsp >> 8;        //VSP
        d[1] = vsp;
        WriteCmdParamN(0x37, 2, d);
        break;
    case 0x3082:
        // nuffink
        break;
    default:
        // 0x6809, 0x9320, 0x9325, 0x9335 can only scroll whole screen
        WriteCmdData(0x61, (1 << 1) | _lcd_rev);        //!NDL, VLE, REV
        WriteCmdData(0x6A, vsp);        //VL#
        break;
    }
}

void MCUFRIEND_kbv::invertDisplay(boolean i)
{
    _lcd_rev = ((_lcd_capable & REV_SCREEN) != 0) ^ i;
    uint16_t newval;
    switch (_lcd_ID) {
    case 0x7789:
    case 0x9327:
    case 0x9341:
    case 0x9481:
    case 0x9486:
    case 0x9488:
        CS_ACTIVE;
        if (_lcd_rev) {
            WriteCmd(0x21);
        } else {
            WriteCmd(0x20);
        }
        CS_IDLE;
        break;
    case 0x6809:
    case 0x9320:
    case 0x9325:
    case 0x9335:
    case 0x4535:
    case 0x7783:
        WriteCmdData(0x61, _lcd_rev);
        break;
    case 0x0139:
    case 0x0154:
        WriteCmdData(0x07, 0x13 | (_lcd_rev << 2));     //.kbv kludge
        break;
    case 0x1289:
        if (_lcd_rev)
            _lcd_drivOut |= (1 << 13);
        else
            _lcd_drivOut &= ~(1 << 13);
        WriteCmdData(0x01, _lcd_drivOut);
        break;
    case 0xB509:
        WriteCmdData(0x401, (1 << 1) | _lcd_rev);       //.kbv kludge VLE 
        break;
    case 0x3082:
        // nuffink
        break;
    default:
        break;
    }
}

#define TFTLCD_DELAY 0xFFFF
#define TFTLCD_DELAY8 0xFF
static void init_table(const void *table, int16_t size)
{
    uint8_t *p = (uint8_t *) table;
    while (size > 0) {
        uint8_t cmd = pgm_read_byte(p++);
        uint8_t len = pgm_read_byte(p++);
        if (cmd == TFTLCD_DELAY8) {
            delay(len);
            len = 0;
        } else {
            CS_ACTIVE;
            WriteCmd(cmd);
            CD_DATA;
            for (uint8_t d = 0; d < len; d++) {
                uint8_t x = pgm_read_byte(p++);
                write8(x);
            }
            CS_IDLE;
        }
        size -= len + 2;
    }
}

#if 1
static void init_table16(const void *p, int16_t size)
{
    uint16_t cmd, d, *table = (uint16_t *) p;
    int i = 0;
    size /= sizeof(uint16_t);
    while (i < size) {
        cmd = pgm_read_word(&table[i++]);
        d = pgm_read_word(&table[i++]);
        if (cmd == TFTLCD_DELAY)
            delay(d);
        else {
            CS_ACTIVE;
            WriteCmd(cmd);
            WriteData(d);
            CS_IDLE;
        }
    }
}
#else
static void init_table16(const void *table, int16_t size)
{
    uint16_t *p = (uint16_t *) table;
    while (size > 0) {
        uint16_t cmd = pgm_read_word(p++);
        uint16_t d = pgm_read_word(p++);
        if (cmd == TFTLCD_DELAY)
            delay(d);
        else
            WriteCmdData(cmd, d);
        size -= 2 * sizeof(int16_t);
    }
}
#endif

void MCUFRIEND_kbv::begin(uint16_t ID)
{
    int16_t *p16;               //so we can "write" to a const protected variable.
    reset();
    if (ID == 0)
        ID = 0x9341;
    switch (_lcd_ID = ID) {
/*
	static const uint16_t _regValues[] PROGMEM = {
    0x0000, 0x0001, // start oscillation
    0x0007, 0x0000, //  source output control 0 D0 
    0x0013, 0x0000, // power control 3 off
    0x0011, 0x2604, //    
    0x0014, 0x0015, //   
    0x0010, 0x3C00, //  
 //    0x0013, 0x0040, // 
 //    0x0013, 0x0060, //     
 //    0x0013, 0x0070, // 
    0x0013, 0x0070, // power control 3 PON PON1 AON
       
    0x0001, 0x0127, //      driver output control
 //    0x0002, 0x0700, //  field 0 b/c waveform xor waveform
    0x0003, 0x1030, //    
    0x0007, 0x0000, //    
    0x0008, 0x0404, //    
    0x000B, 0x0200, // 
    0x000C, 0x0000, //   
    0x00015,0x0000, //     
       
    //gamma setting    
    0x0030, 0x0000,      
    0x0031, 0x0606,    
    0x0032, 0x0006,    
    0x0033, 0x0403,  
    0x0034, 0x0107,  
    0x0035, 0x0101, 
    0x0036, 0x0707,   
    0x0037, 0x0304,   
    0x0038, 0x0A00,     
    0x0039, 0x0706,     
       
    0x0040, 0x0000,     
    0x0041, 0x0000,      
    0x0042, 0x013F,    
    0x0043, 0x0000,   
    0x0044, 0x0000,     
    0x0045, 0x0000,     
    0x0046, 0xEF00,    
    0x0047, 0x013F,     
    0x0048, 0x0000,     
    0x0007, 0x0011,  
    0x0007, 0x0017,     
};
*/
    case 0x0139:
        _lcd_capable = AUTO_READINC | REV_SCREEN | XSA_XEA_16BIT;
        static const uint16_t S6D0139_regValues[] PROGMEM = {
            0x0000, 0x0001,     //Start oscillator
            0x0011, 0x1a00,     //Power Control 2
            0x0014, 0x2020,     //Power Control 4
            0x0010, 0x0900,     //Power Control 1
            0x0013, 0x0040,     //Power Control 3
            0x0013, 0x0060,     //Power Control 3
            0x0013, 0x0070,     //Power Control 3
            0x0011, 0x1a04,     //Power Control 2
            0x0010, 0x2f00,     //Power Control 1
            0x0001, 0x0127,     //Driver Control: SM=0, GS=0, SS=1, 240x320
            0x0002, 0x0100,     //LCD Control:  (.kbv was 0700) FLD=0, BC= 0, EOR=1
            0x0003, 0x1030,     //Entry Mode:    TR1=0, DFM=0, BGR=1, I_D=3   
            0x0007, 0x0000,     //Display Control: everything off
            0x0008, 0x0808,     //Blank Period:  FP=98, BP=8
            0x0009, 0x0000,     //f.k.
            0x000b, 0x0000,     //Frame Control:
            0x000c, 0x0000,     //Interface Control: system i/f
            0x0040, 0x0000,     //Scan Line
            0x0041, 0x0000,     //Vertical Scroll Control
            0x0042, 0x013f,     //Screen 1 End
            0x0043, 0x0000,     //Screen 1 start
            0x0044, 0x00ef,     //Screen 2 end
            0x0045, 0x0000,     //Screen 2 start
            0x0046, 0xef00,     //Horiz address H=end, L=start
            0x0047, 0x013f,     //Vert end
            0x0048, 0x0000,     //Vert start
            0x0007, 0x0014,     //Display Control: SPT=1, REV=1
            0x0007, 0x0016,     //Display Control: SPT=1, REV=1, display on
            0x0007, 0x0017,     //Display Control: SPT=1, REV=1, display on, GON
        };
        init_table16(S6D0139_regValues, sizeof(S6D0139_regValues));
        break;

    case 0x0154:
        _lcd_capable = AUTO_READINC | REV_SCREEN;
        static const uint16_t S6D0154_regValues[] PROGMEM = {
            0x11, 0x001A,
            0x12, 0x3121,
            0x13, 0x006C,
            0x14, 0x4249,

            0x10, 0x0800,
            TFTLCD_DELAY, 10,
            0x11, 0x011A,
            TFTLCD_DELAY, 10,
            0x11, 0x031A,
            TFTLCD_DELAY, 10,
            0x11, 0x071A,
            TFTLCD_DELAY, 10,
            0x11, 0x0F1A,
            TFTLCD_DELAY, 10,
            0x11, 0x0F3A,
            TFTLCD_DELAY, 30,

            0x01, 0x0128,
            0x02, 0x0100,
            0x03, 0x1030,
            0x07, 0x1012,
            0x08, 0x0303,
            0x0B, 0x1100,
            0x0C, 0x0000,
            0x0F, 0x1801,
            0x15, 0x0020,
            /*
               0x50,0x0101,
               0x51,0x0603,
               0x52,0x0408,
               0x53,0x0000,
               0x54,0x0605,
               0x55,0x0406,
               0x56,0x0303,
               0x57,0x0303,
               0x58,0x0010,
               0x59,0x1000,
             */
            0x07, 0x0012,
            TFTLCD_DELAY, 40,
            0x07, 0x0013,       /* GRAM Address Set */
            0x07, 0x0017,       /* Display Control DISPLAY ON */

            0x36, 0x00EF,
            0x37, 0x0000,
            0x38, 0x013F,
            0x39, 0x0000,
        };
        init_table16(S6D0154_regValues, sizeof(S6D0154_regValues));

        break;

    case 0x1289:
        _lcd_capable = 0 | XSA_XEA_16BIT | REV_SCREEN;
        // came from MikroElektronika library http://www.hmsprojects.com/tft_lcd.html
        static const uint16_t SSD1289_regValues[] PROGMEM = {
            0x0000, 0x0001,
            0x0003, 0xA8A4,
            0x000C, 0x0000,
            0x000D, 0x080C,     // was 0x800C
            0x000E, 0x2B00,
            0x001E, 0x00B7,
            0x0001, 0x2B3F,     // was 0x2B3F,
            0x0002, 0x0400,     // was 0x0600
            0x0010, 0x0000,
            0x0011, 0x6070,     // was 0x6070
            0x0005, 0x0000,
            0x0006, 0x0000,
            0x0016, 0xEF1C,
            0x0017, 0x0003,
            0x0007, 0x0233,
            0x000B, 0x0000,
            0x000F, 0x0000,
            0x0041, 0x0000,
            0x0042, 0x0000,
            0x0048, 0x0000,
            0x0049, 0x013F,
            0x004A, 0x0000,
            0x004B, 0x0000,
            0x0044, 0xEF95,
            0x0045, 0x0000,
            0x0046, 0x013F,
            0x0030, 0x0707,
            0x0031, 0x0204,
            0x0032, 0x0204,
            0x0033, 0x0502,
            0x0034, 0x0507,
            0x0035, 0x0204,
            0x0036, 0x0204,
            0x0037, 0x0502,
            0x003A, 0x0302,
            0x003B, 0x0302,
            0x0023, 0x0000,
            0x0024, 0x0000,
            0x0025, 0x8000,
            0x004f, 0x0000,
            0x004e, 0x0000,
        };
        init_table16(SSD1289_regValues, sizeof(SSD1289_regValues));
        break;

    case 0x3082:
        _lcd_capable = 0 | AUTO_READINC;
        // came from ITEAD Studio PDF
        static const uint16_t IM13082_regValues[] PROGMEM = {
            0x0001, 0x0008,     // Backlight
            0x0002, 0x0000,     // XSA address (portrait)
            0x0003, 0x0000,     // YSA address (portrait)
            0x0004, 0x0000,     // Page display
            0x0005, 0x0000,     // Page r/w
            0x0006, 0x013F,     // XEA (portrait)
            0x0007, 0x031F,     // YEA (portrait)
            0x000B, 0x0000,     // Sleep
            0x000C, 0x0000,     // MADCTL
            0x000D, 0x0000,     // increment / Decrement
            0x000F, 0x0000,     // Data Channel
        };
        init_table16(IM13082_regValues, sizeof(IM13082_regValues));
        p16 = (int16_t *) & HEIGHT;
        *p16 = 800;             // (portrait)
        p16 = (int16_t *) & WIDTH;
        *p16 = 480;             // (portrait)
        break;

    case 0x4535:
        _lcd_capable = 0 | REV_SCREEN;
        static const uint16_t LGDP4535_regValues[] PROGMEM = {
            0x15, 0x0030,       // Set the internal vcore voltage                                               
            0x9A, 0x0010,       // Start internal OSC 
            0x11, 0x0020,       // set SS and SM bit 
            0x10, 0x3428,       // set 1 line inversion 
            0x12, 0x0002,       // set GRAM write direction and BGR=1  
            0x13, 0x1038,       // Resize register 
            TFTLCD_DELAY, 40,
            0x12, 0x0012,       // set the back porch and front porch 
            TFTLCD_DELAY, 40,
            0x10, 0x3420,       // set non-display area refresh cycle ISC[3:0] 
            0x13, 0x3045,       // FMARK function 
            TFTLCD_DELAY, 70,
            0x30, 0x0000,       // RGB interface setting 
            0x31, 0x0402,       // Frame marker Position 
            0x32, 0x0307,       // RGB interface polarity 
            0x33, 0x0304,       // SAP, BT[3:0], AP, DSTB, SLP, STB 
            0x34, 0x0004,       // DC1[2:0], DC0[2:0], VC[2:0] 
            0x35, 0x0401,       // VREG1OUT voltage 
            0x36, 0x0707,       // VDV[4:0] for VCOM amplitude 
            0x37, 0x0305,       // SAP, BT[3:0], AP, DSTB, SLP, STB 
            0x38, 0x0610,       // DC1[2:0], DC0[2:0], VC[2:0] 
            0x39, 0x0610,       // VREG1OUT voltage 
            0x01, 0x0100,       // VDV[4:0] for VCOM amplitude 
            0x02, 0x0300,       // VCM[4:0] for VCOMH 
            0x03, 0x1030,       // GRAM horizontal Address 
            0x08, 0x0808,       // GRAM Vertical Address 
            0x0A, 0x0008,
            0x60, 0x2700,       // Gate Scan Line 
            0x61, 0x0001,       // NDL,VLE, REV 
            0x90, 0x013E,
            0x92, 0x0100,
            0x93, 0x0100,
            0xA0, 0x3000,
            0xA3, 0x0010,
            0x07, 0x0001,
            0x07, 0x0021,
            0x07, 0x0023,
            0x07, 0x0033,
            0x07, 0x0133,
        };
        init_table16(LGDP4535_regValues, sizeof(LGDP4535_regValues));
        break;

    case 0x7783:
        _lcd_capable = AUTO_READINC | REV_SCREEN;
        static const uint16_t ST7781_regValues[] PROGMEM = {
            0x00FF, 0x0001,     //can we do 0xFF
            0x00F3, 0x0008,
            //  LCD_Write_COM(0x00F3,

            0x00, 0x0001,
            0x0001, 0x0100,     // Driver Output Control Register (R01h)
            0x0002, 0x0700,     // LCD Driving Waveform Control (R02h)
            0x0003, 0x1030,     // Entry Mode (R03h)
            0x0008, 0x0302,
            0x0009, 0x0000,
            0x0010, 0x0000,     // Power Control 1 (R10h)
            0x0011, 0x0007,     // Power Control 2 (R11h)
            0x0012, 0x0000,     // Power Control 3 (R12h)
            0x0013, 0x0000,     // Power Control 4 (R13h)
            TFTLCD_DELAY, 50,
            0x0010, 0x14B0,     // Power Control 1 (R10h)
            TFTLCD_DELAY, 10,
            0x0011, 0x0007,     // Power Control 2 (R11h)
            TFTLCD_DELAY, 10,
            0x0012, 0x008E,     // Power Control 3 (R12h)
            0x0013, 0x0C00,     // Power Control 4 (R13h)
            0x0029, 0x0015,     // NVM read data 2 (R29h)
            TFTLCD_DELAY, 10,
            0x0030, 0x0000,     // Gamma Control 1
            0x0031, 0x0107,     // Gamma Control 2
            0x0032, 0x0000,     // Gamma Control 3
            0x0035, 0x0203,     // Gamma Control 6
            0x0036, 0x0402,     // Gamma Control 7
            0x0037, 0x0000,     // Gamma Control 8
            0x0038, 0x0207,     // Gamma Control 9
            0x0039, 0x0000,     // Gamma Control 10
            0x003C, 0x0203,     // Gamma Control 13
            0x003D, 0x0403,     // Gamma Control 14
            0x0050, 0x0000,     // Window Horizontal RAM Address Start (R50h)
            0x0051, 240 - 1,    // Window Horizontal RAM Address End (R51h)
            0x0052, 0X0000,     // Window Vertical RAM Address Start (R52h)
            0x0053, 320 - 1,    // Window Vertical RAM Address End (R53h)
            0x0060, 0xA700,     // Driver Output Control (R60h) .kbv was 0xa700
            0x0061, 0x0001,     // Driver Output Control (R61h)
            0x0090, 0X0029,     // Panel Interface Control 1 (R90h)

            // Display On
            0x0007, 0x0133,     // Display Control (R07h)
            TFTLCD_DELAY, 50,
        };
        init_table16(ST7781_regValues, sizeof(ST7781_regValues));
        break;
    case 0x7789:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS;
        static const uint8_t ST7789_regValues[] PROGMEM = {
            0x01, 0,            //Soft Reset
            (0x11), 0,          //exit SLEEP mode
            TFTLCD_DELAY8, 100, // if from Sleep
            (0x36), 1, 0x80,    //MADCTL: memory data access control
            //        (0x3A), 1, 0x66,      //COLMOD: Interface Pixel format *** I use 262K-colors in 18bit/pixel format when using 8-bit interface to allow 3-bytes per pixel
            (0x3A), 1, 0x55,    //COLMOD: Interface Pixel format  *** I use 65K-colors in 16bit/pixel (5-6-5) format when using 16-bit interface to allow 1-byte per pixel
            (0xB2), 5, 0x0C, 0x0C, 0x00, 0x33, 0x33,    //PORCTRK: Porch setting [08 08 00 22 22] PSEN=0 anyway
            (0xB7), 1, 0x35,    //GCTRL: Gate Control [35]
            (0xBB), 1, 0x2B,    //VCOMS: VCOM setting VCOM=1.175 [20] VCOM=0.9
            (0xC0), 1, 0x04,    //LCMCTRL: LCM Control [2C]
            (0xC2), 2, 0x01, 0xFF,      //VDVVRHEN: VDV and VRH Command Enable [01 FF]
            (0xC3), 1, 0x11,    //VRHS: VRH Set VAP=4.4, VAN=-4.4 [0B]
            (0xC4), 1, 0x20,    //VDVS: VDV Set [20]
            (0xC6), 1, 0x0F,    //FRCTRL2: Frame Rate control in normal mode [0F]
            (0xD0), 2, 0xA4, 0xA1,      //PWCTRL1: Power Control 1 [A4 A1]
            (0xE0), 14, 0xD0, 0x00, 0x05, 0x0E, 0x15, 0x0D, 0x37, 0x43, 0x47, 0x09, 0x15, 0x12, 0x16, 0x19,     //PVGAMCTRL: Positive Voltage Gamma control        
            (0xE1), 14, 0xD0, 0x00, 0x05, 0x0D, 0x0C, 0x06, 0x2D, 0x44, 0x40, 0x0E, 0x1C, 0x18, 0x16, 0x19,     //NVGAMCTRL: Negative Voltage Gamma control
            (0x2A), 4, 0x00, 0x00, 0x00, 0xEF,  //X address set
            (0x2B), 4, 0x00, 0x00, 0x01, 0x3F,  //Y address set
            0x29, 0,            //Display On
            TFTLCD_DELAY8, 10,
        };
        static const uint8_t ST7789_regValues_arcain6[] PROGMEM = {
            0x01, 0,            //Soft Reset
            (0x11), 0,          //exit SLEEP mode
            TFTLCD_DELAY8, 100, // if from Sleep
            (0x36), 1, 0x00,    //MADCTL: memory data access control
            (0x3A), 1, 0x05,    //COLMOD: Interface Pixel format  *** I use 65K-colors in 16bit/pixel (5-6-5) format when using 16-bit interface to allow 1-byte per pixel
            (0xB2), 5, 0x0C, 0x0C, 0x00, 0x33, 0x33,    //PORCTRK: Porch setting [08 08 00 22 22] PSEN=0 anyway
            (0xB7), 1, 0x35,    //GCTRL: Gate Control [35]
            (0xBB), 1, 0x35,    //VCOMS: VCOM setting VCOM=??? [20] VCOM=0.9
            (0xC0), 1, 0x2C,    //LCMCTRL: LCM Control [2C]
            (0xC2), 2, 0x01, 0xFF,      //VDVVRHEN: VDV and VRH Command Enable [01 FF]
            (0xC3), 1, 0x13,    //VRHS: VRH Set VAP=???, VAN=-??? [0B]
            (0xC4), 1, 0x20,    //VDVS: VDV Set [20]
            (0xC6), 1, 0x0F,    //FRCTRL2: Frame Rate control in normal mode [0F]
            (0xCA), 1, 0x0F,    //REGSEL2 [0F]
            (0xC8), 1, 0x08,    //REGSEL1 [08]
            (0x55), 1, 0x90,    //WRCACE  [00]
            (0xD0), 2, 0xA4, 0xA1,      //PWCTRL1: Power Control 1 [A4 A1]
            (0xE0), 14, 0xD0, 0x00, 0x06, 0x09, 0x0B, 0x2A, 0x3C, 0x55, 0x4B, 0x08, 0x16, 0x14, 0x19, 0x20,     //PVGAMCTRL: Positive Voltage Gamma control        
            (0xE1), 14, 0xD0, 0x00, 0x06, 0x09, 0x0B, 0x29, 0x36, 0x54, 0x4B, 0x0D, 0x16, 0x14, 0x21, 0x20,     //NVGAMCTRL: Negative Voltage Gamma control
            //        (0x2A), 4, 0x00, 0x00, 0x00, 0xEF,      //X address set
            //        (0x2B), 4, 0x00, 0x00, 0x01, 0x3F,      //Y address set
            0x29, 0,            //Display On
            TFTLCD_DELAY8, 10,
        };
        init_table(ST7789_regValues, sizeof(ST7789_regValues)); //
        break;

    case 0x9320:
        _lcd_capable = 0 | REV_SCREEN | READ_BGR;
        static const uint16_t ILI9320_regValues[] PROGMEM = {
            0xe5, 0x8000,
            0x00, 0x0001,
            0x01, 0x100,
            0x02, 0x0700,
            0x03, 0x1030,
            0x04, 0x0000,
            0x08, 0x0202,
            0x09, 0x0000,
            0x0A, 0x0000,
            0x0C, 0x0000,
            0x0D, 0x0000,
            0x0F, 0x0000,
            //-----Power On sequence-----------------------
            0x10, 0x0000,
            0x11, 0x0007,
            0x12, 0x0000,
            0x13, 0x0000,
            TFTLCD_DELAY, 50,
            0x10, 0x17B0,
            0x11, 0x0007,
            TFTLCD_DELAY, 10,
            0x12, 0x013A,
            TFTLCD_DELAY, 10,
            0x13, 0x1A00,
            0x29, 0x000c,
            TFTLCD_DELAY, 10,
            //-----Gamma control-----------------------
            0x30, 0x0000,
            0x31, 0x0505,
            0x32, 0x0004,
            0x35, 0x0006,
            0x36, 0x0707,
            0x37, 0x0105,
            0x38, 0x0002,
            0x39, 0x0707,
            0x3C, 0x0704,
            0x3D, 0x0807,
            //-----Set RAM area-----------------------
            0x50, 0x0000,
            0x51, 0x00EF,
            0x52, 0x0000,
            0x53, 0x013F,
            0x60, 0x2700,
            0x61, 0x0001,
            0x6A, 0x0000,
            0x21, 0x0000,
            0x20, 0x0000,
            //-----Partial Display Control------------
            0x80, 0x0000,
            0x81, 0x0000,
            0x82, 0x0000,
            0x83, 0x0000,
            0x84, 0x0000,
            0x85, 0x0000,
            //-----Panel Control----------------------
            0x90, 0x0010,
            0x92, 0x0000,
            0x93, 0x0003,
            0x95, 0x0110,
            0x97, 0x0000,
            0x98, 0x0000,
            //-----Display on-----------------------
            0x07, 0x0173,
            TFTLCD_DELAY, 50,
        };
        init_table16(ILI9320_regValues, sizeof(ILI9320_regValues));
        break;
    case 0x6809:
    case 0x9325:
    case 0x9335:
        _lcd_capable = 0 | REV_SCREEN;
        static const uint16_t ILI9325_regValues[] PROGMEM = {
            0xE5, 0x78F0,       // set SRAM internal timing
            0x01, 0x0100,       // set Driver Output Control
            0x02, 0x0200,       // set 1 line inversion
            0x03, 0x1030,       // set GRAM write direction and BGR=1.
            0x04, 0x0000,       // Resize register
            0x05, 0x0000,       // .kbv 16bits Data Format Selection
            0x08, 0x0207,       // set the back porch and front porch
            0x09, 0x0000,       // set non-display area refresh cycle ISC[3:0]
            0x0A, 0x0000,       // FMARK function
            0x0C, 0x0000,       // RGB interface setting
            0x0D, 0x0000,       // Frame marker Position
            0x0F, 0x0000,       // RGB interface polarity
            // ----------- Power On sequence ----------- //
            0x10, 0x0000,       // SAP, BT[3:0], AP, DSTB, SLP, STB
            0x11, 0x0007,       // DC1[2:0], DC0[2:0], VC[2:0]
            0x12, 0x0000,       // VREG1OUT voltage
            0x13, 0x0000,       // VDV[4:0] for VCOM amplitude
            0x07, 0x0001,
            TFTLCD_DELAY, 200,  // Dis-charge capacitor power voltage
            0x10, 0x1690,       // SAP, BT[3:0], AP, DSTB, SLP, STB
            0x11, 0x0227,       // Set DC1[2:0], DC0[2:0], VC[2:0]
            TFTLCD_DELAY, 50,   // wait_ms 50ms
            0x12, 0x000D,       // 0012
            TFTLCD_DELAY, 50,   // wait_ms 50ms
            0x13, 0x1200,       // VDV[4:0] for VCOM amplitude
            0x29, 0x000A,       // 04  VCM[5:0] for VCOMH
            0x2B, 0x000D,       // Set Frame Rate
            TFTLCD_DELAY, 50,   // wait_ms 50ms
            0x20, 0x0000,       // GRAM horizontal Address
            0x21, 0x0000,       // GRAM Vertical Address
            // ----------- Adjust the Gamma Curve ----------//

            0x30, 0x0000,
            0x31, 0x0404,
            0x32, 0x0003,
            0x35, 0x0405,
            0x36, 0x0808,
            0x37, 0x0407,
            0x38, 0x0303,
            0x39, 0x0707,
            0x3C, 0x0504,
            0x3D, 0x0808,

            //------------------ Set GRAM area ---------------//
            0x50, 0x0000,       // Horizontal GRAM Start Address
            0x51, 0x00EF,       // Horizontal GRAM End Address
            0x52, 0x0000,       // Vertical GRAM Start Address
            0x53, 0x013F,       // Vertical GRAM Start Address
            0x60, 0xA700,       // Gate Scan Line (0xA700)
            0x61, 0x0001,       // NDL,VLE, REV .kbv
            0x6A, 0x0000,       // set scrolling line
            //-------------- Partial Display Control ---------//
            0x80, 0x0000,
            0x81, 0x0000,
            0x82, 0x0000,
            0x83, 0x0000,
            0x84, 0x0000,
            0x85, 0x0000,
            //-------------- Panel Control -------------------//
            0x90, 0x0010,
            0x92, 0x0000,
            0x07, 0x0133,       // 262K color and display ON
        };
        init_table16(ILI9325_regValues, sizeof(ILI9325_regValues));
        break;
    case 0x9327:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS;
        static const uint8_t ILI9327_regValues[] PROGMEM = {
            0x01, 0,            //Soft Reset
            0x28, 0,            //Display Off
            //            0xE0, 1, 0x20,      //NV Memory Write [00]
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 100,
            0xB0, 1, 0x00,      //Disable Protect for cmds B1-DF, E0-EF, F0-FF 
            //            0xD1, 3, 0x00, 0x71, 0x19,  //VCOM control [00 40 0F]
            //            0xD0, 3, 0x07, 0x01, 0x08,  //Power Setting [07 04 8C]
            0xC1, 4, 0x10, 0x10, 0x02, 0x02,    //Display Timing [10 10 02 02]
            0xC0, 6, 0x00, 0x35, 0x00, 0x00, 0x01, 0x02,        //Panel Drive [00 35 00 00 01 02 REV=0,GS=0,SS=0
            0xC5, 1, 0x04,      //Frame Rate [04]
            0xD2, 2, 0x01, 0x04,        //Power Setting [01 44]
            //            0xC8, 15, 0x04, 0x67, 0x35, 0x04, 0x08, 0x06, 0x24, 0x01, 0x37, 0x40, 0x03, 0x10, 0x08, 0x80, 0x00,
            //            0xC8, 15, 0x00, 0x77, 0x77, 0x04, 0x04, 0x00, 0x00, 0x00, 0x77, 0x00, 0x00, 0x08, 0x00, 0x00, 0x00,
            0xCA, 1, 0x00,      //DGC LUT ???
            0xEA, 1, 0x80,      //3-Gamma Function Enable
            //                     0xB0, 1, 0x03,      //Enable Protect
            0x36, 1, 0x48,      // Memory Access
            0x3A, 1, 0x55,      //Pixel read=565, write=565
            0x2A, 4, 0x00, 0x00, 0x00, 0xEF,    // wid: 0, 239
            0x2B, 4, 0x00, 0x00, 0x01, 0x8F,    // ht: 0, 399
            0x30, 4, 0x00, 0x00, 0x01, 0x8F,    // Partial Area: 0, 399
            0x29, 0,            //Display On
        };
        init_table(ILI9327_regValues, sizeof(ILI9327_regValues));
        p16 = (int16_t *) & HEIGHT;
        *p16 = 400;
        p16 = (int16_t *) & WIDTH;
        *p16 = 240;
        break;
    case 0x9341:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS;
        static const uint8_t ILI9341_regValues_2_4[] PROGMEM = {        // BOE 2.4"
            0x01, 0,            // software reset
            TFTLCD_DELAY8, 50,  // .kbv
            0xCF, 3, 0x00, 0x81, 0x30,  //Power Control B [00 81 30]
            0xED, 4, 0x64, 0x03, 0x12, 0x81,    //Power On Seq [55 01 23 01]
            0xE8, 3, 0x85, 0x10, 0x78,  //Driver Timing A [04 11 7A]
            0xCB, 5, 0x39, 0x2C, 0x00, 0x34, 0x02,      //Power Control A [39 2C 00 34 02]
            0xF7, 1, 0x20,      //Pump Ratio [10]
            0xEA, 2, 0x00, 0x00,        //Driver Timing B [66 00]
            0xB1, 2, 0x00, 0x1B,        //Frame Control [00 1B]
            //            0xB6, 2, 0x0A, 0xA2, 0x27, //Display Function [0A 82 27 XX]    .kbv SS=1  
            0xB4, 1, 0x00,      //Inversion Control [02] .kbv NLA=1, NLB=1, NLC=1
            0xC0, 1, 0x21,      //Power Control 1 [26]
            0xC1, 1, 0x11,      //Power Control 2 [00]
            0xC5, 2, 0x3F, 0x3C,        //VCOM 1 [31 3C]
            0xC7, 1, 0xB5,      //VCOM 2 [C0]
            0x36, 1, 0x48,      //Memory Access [00]
            0xF2, 1, 0x00,      //Enable 3G [02]
            0x26, 1, 0x01,      //Gamma Set [01]
            0xE0, 15, 0x0f, 0x26, 0x24, 0x0b, 0x0e, 0x09, 0x54, 0xa8, 0x46, 0x0c, 0x17, 0x09, 0x0f, 0x07, 0x00,
            0xE1, 15, 0x00, 0x19, 0x1b, 0x04, 0x10, 0x07, 0x2a, 0x47, 0x39, 0x03, 0x06, 0x06, 0x30, 0x38, 0x0f,
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 150,
            0x29, 0,            //Display On
            0x3A, 1, 0x55,      //Pixel Format [66]
        };
        static const uint8_t ILI9341_regValues_ada[] PROGMEM = {        // Adafruit_TFTLCD only works with EXTC=0
            0x01, 0,            // software reset
            TFTLCD_DELAY8, 50,
            0x28, 0,            //Display Off
            //                     0xF6, 3, 0x00, 0x01, 0x00,  //Interface Control needs EXTC=1 TM=0, RIM=0
            0xF6, 3, 0x01, 0x01, 0x03,  //Interface Control needs EXTC=1 RM=1, RIM=1
            0xB0, 1, 0x40,      //RGB Signal [40] RCM=2
            0xB4, 1, 0x00,      //Inversion Control [02] .kbv NLA=1, NLB=1, NLC=1
            0xC0, 1, 0x23,      //Power Control 1 [26]
            0xC1, 1, 0x10,      //Power Control 2 [00]
            0xC5, 2, 0x2B, 0x2B,        //VCOM 1 [31 3C]
            0xC7, 1, 0xC0,      //VCOM 2 [C0]
            0x36, 1, 0x88,      //Memory Access [00]
            0xB1, 2, 0x00, 0x1B,        //Frame Control [00 1B]
            0xB7, 1, 0x07,      //Entry Mode [00]
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 150,
            0x29, 0,            //Display On
            0x3A, 1, 0x55,      //Pixel Format [66]
        };
#if !defined(USE_SERIAL)
        if (readReg32(0xD3) != 0x9341) {
            init_table(ILI9341_regValues_ada, sizeof(ILI9341_regValues_ada));
            _lcd_capable |= REV_SCREEN | READ_BGR;
            invertDisplay(false);
        } else
#endif
            init_table(ILI9341_regValues_2_4, sizeof(ILI9341_regValues_2_4));   //
        break;
    case 0x9481:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS | READ_BGR;
        static const uint8_t ILI9481_regValues[] PROGMEM = {    // Atmel MaxTouch
            0x01, 0,            //Soft Reset
            TFTLCD_DELAY8, 50,
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 20,
            0xD0, 3, 0x07, 0x42, 0x18,  //Set Power
            0xD1, 3, 0x00, 0x07, 0x10,  // Set VCOM
            0xD2, 2, 0x01, 0x02,        // Set Power for Normal Mode
            0xC0, 5, 0x10, 0x3B, 0x00, 0x02, 0x11,      //Set Panel Driving
            0xC5, 1, 0x08,      //Frame Rate
            0xC8, 12, 0x00, 0x32, 0x36, 0x45, 0x06, 0x16, 0x37, 0x75, 0x77, 0x54, 0x0C, 0x00,
            0x36, 1, 0x0A,      //Memory Access [00]
            0x3A, 1, 0x55,      //Interlace Pixel Format [XX]

            0x2A, 4, 0x00, 0x00, 0x01, 0x3F,
            0x2B, 4, 0x00, 0x00, 0x01, 0xE0,
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 120,
            0x29, 0,            //Display On
            TFTLCD_DELAY8, 25,
        };
        init_table(ILI9481_regValues, sizeof(ILI9481_regValues));
        p16 = (int16_t *) & HEIGHT;
        *p16 = 480;
        p16 = (int16_t *) & WIDTH;
        *p16 = 320;
        break;
    case 0x9486:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS | REV_SCREEN;
        static const uint8_t ILI9486_regValues[] PROGMEM = {
            0x01, 0,            //Soft Reset
            TFTLCD_DELAY8, 50,
            0x28, 0,            //Display Off
            0xC0, 2, 0x0d, 0x0d,        //Power Control 1 [0x0E0E]
            0xC1, 2, 0x43, 0x00,        //Power Control 2 [0x4300]
            0xC2, 1, 0x00,      //Power Control 3
            0xC5, 4, 0x00, 0x48, 0x00, 0x48,    //VCOM  Control 1 [0x00400040]
            0xB4, 1, 0x00,      //Inversion Control
            0xB6, 3, 0x00, 0x22, 0x3B,  // Display Function Control  .kbv SS=1
            0xE0, 15, 0x0f, 0x31, 0x2b, 0x0c, 0x0e, 0x08, 0x4e, 0xf1, 0x37, 0x07, 0x10, 0x03, 0x0e, 0x09, 0x00,
            0xE1, 15, 0x00, 0x0e, 0x14, 0x03, 0x11, 0x07, 0x31, 0xC1, 0x48, 0x08, 0x0f, 0x0c, 0x31, 0x36, 0x0f,
            0x20, 0,            // Display Inversion OFF
            0x36, 1, 0x0A,      //Memory Access
            0x3A, 1, 0x55,      //Interlace Pixel
            //            0x21, 0,            //Invert display !!!
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 50,
            0x29, 0,            //Display On
        };
        init_table(ILI9486_regValues, sizeof(ILI9486_regValues));
#if (defined(__AVR_ATmega2560__) && defined(USE_MEGA_16BIT_SHIELD))
        _lcd_capable &= ~(REV_SCREEN);  //the 16-bit shield is different
#endif
        p16 = (int16_t *) & HEIGHT;
        *p16 = 480;
        p16 = (int16_t *) & WIDTH;
        *p16 = 320;
        break;
    case 0x9488:
        _lcd_capable = AUTO_READINC | TWO_PARM_WINDOW | MV_AXIS;
        static const uint8_t ILI9488_regValues_max[] PROGMEM = {        // Atmel MaxTouch
            0x01, 0,            //Soft Reset
            TFTLCD_DELAY8, 50,
            0x28, 0,            //Display Off
            0xC0, 2, 0x10, 0x10,        //Power Control 1 [0E 0E]
            0xC1, 1, 0x41,      //Power Control 2 [43]
            0xC5, 4, 0x00, 0x22, 0x80, 0x40,    //VCOM  Control 1 [00 40 00 40]
            0x36, 1, 0x68,      //Memory Access [00]
            0xB0, 1, 0x00,      //Interface     [00]
            0xB1, 2, 0xB0, 0x11,        //Frame Rate Control [B0 11]
            0xB4, 1, 0x02,      //Inversion Control [02]
            0xB6, 3, 0x02, 0x02, 0x3B,  // Display Function Control [02 02 3B] .kbv NL=480
            0xB7, 1, 0xC6,      //Entry Mode      [06]
            0x3A, 1, 0x55,      //Interlace Pixel Format [XX]
            0xF7, 4, 0xA9, 0x51, 0x2C, 0x82,    //Adjustment Control 3 [A9 51 2C 82]
            0x11, 0,            //Sleep Out
            TFTLCD_DELAY8, 150,
            0x29, 0,            //Display On
        };
        init_table(ILI9488_regValues_max, sizeof(ILI9488_regValues_max));
        p16 = (int16_t *) & HEIGHT;
        *p16 = 480;
        p16 = (int16_t *) & WIDTH;
        *p16 = 320;
        break;
    case 0xB509:
        _lcd_capable = REV_SCREEN;
        static const uint16_t R61509V_regValues[] PROGMEM = {
            0x0000, 0x0000,
            0x0000, 0x0000,
            0x0000, 0x0000,
            0x0000, 0x0000,
            TFTLCD_DELAY, 15,
            0x0400, 0x6200,     //NL=0x31 (49) i.e. 400 rows
            0x0008, 0x0808,
            //gamma
            0x0300, 0x0C00,
            0x0301, 0x5A0B,
            0x0302, 0x0906,
            0x0303, 0x1017,
            0x0304, 0x2300,
            0x0305, 0x1700,
            0x0306, 0x6309,
            0x0307, 0x0C09,
            0x0308, 0x100C,
            0x0309, 0x2232,

            0x0010, 0x0016,     //69.5Hz         0016
            0x0011, 0x0101,
            0x0012, 0x0000,
            0x0013, 0x0001,

            0x0100, 0x0330,     //BT,AP
            0x0101, 0x0237,     //DC0,DC1,VC
            0x0103, 0x0D00,     //VDV
            0x0280, 0x6100,     //VCM
            0x0102, 0xC1B0,     //VRH,VCMR,PSON,PON
            TFTLCD_DELAY, 50,

            0x0001, 0x0100,
            0x0002, 0x0100,
            0x0003, 0x1030,     //1030
            0x0009, 0x0001,
            0x000C, 0x0000,
            0x0090, 0x8000,
            0x000F, 0x0000,

            0x0210, 0x0000,
            0x0211, 0x00EF,
            0x0212, 0x0000,
            0x0213, 0x018F,     //432=01AF,400=018F
            0x0500, 0x0000,
            0x0501, 0x0000,
            0x0502, 0x005F,     //???
            0x0401, 0x0001,     //REV=1
            0x0404, 0x0000,
            TFTLCD_DELAY, 50,

            0x0007, 0x0100,     //BASEE
            TFTLCD_DELAY, 50,

            0x0200, 0x0000,
            0x0201, 0x0000,
        };
        init_table16(R61509V_regValues, sizeof(R61509V_regValues));
        p16 = (int16_t *) & HEIGHT;
        *p16 = 400;
        break;
    }
    _lcd_rev = ((_lcd_capable & REV_SCREEN) != 0);
    setRotation(0);             //PORTRAIT
    invertDisplay(false);
}
