
#define _SUPPRESS_PLIB_WARNING
#define _DISABLE_OPENADC10_CONFIGPORT_WARNING
#include <plib.h>
#include <xc.h>
#include <math.h>
#include <string.h>
#include "dee_emulation_pic32.h"

/*
 *  PIC32MX170F256D-I/P pin mapping:
 *
 *  1 RB9        : LED Matrix K8
 *  2 RC6        : LED Matrix A7
 *  3 RC7        : LED Matrix A8
 *  4 RC8        : LED Matrix A9
 *  5 RC9        : LED Matrix A10
 *  6 VSS
 *  7 VCAP
 *  8 RB10/PGED2 : PGED2 / S1
 *  9 RB11/PGEC2 : PGEC2 / S2
 * 10 RB12/AN12  : LED82 cathode
 * 11 RB13/AN11  : Brightness trimpot
 * 12 RA10/PGED  : LED83 cathode
 * 13  RA7/PGEC  : LED84 cathode
 * 14 RB14/AN10  : Right input x100
 * 15 RB15/AN9   : Left input x100
 * 16 AVSS
 * 17 AVDD
 * 18 MCLR
 * 19 RA0/AN0    : Left input x1
 * 20 RA1/AN1    : Right input x1
 * 21 RB0/AN2    : Left input x10
 * 22 RB1/AN3    : Right input x10
 * 23 RB2/AN4    : LED Matrix K1
 * 24 RB3/AN5    : LED Matrix K2
 * 25 RC0/AN6    : LED Matrix A1
 * 26 RC1/AN7    : LED Matrix A2
 * 27 RC2/AN8    : LED Matrix A3
 * 28 VDD
 * 29 VSS
 * 30 RA2/CLKI   : LED87 cathode
 * 31 RA3/CLKO   : LED86 cathode
 * 32 RA8        : LED85 cathode
 * 33 RB4        : LED Matrix K3
 * 34 RA4        : LED88 cathode
 * 35 RA9        : LED81 cathode
 * 36 RC3        : LED Matrix A4
 * 37 RC4        : LED Matrix A5
 * 38 RC5        : LED Matrix A6
 * 39 VSS
 * 40 VDD
 * 41 RB5/PGED3  : LED Matrix K4
 * 42 RB6/PGEC3  : LED Matrix K5
 * 43 RB7        : LED Matrix K6
 * 44 RB8        : LED Matrix K7
 */

// DEVCFG3
#pragma config PMDL1WAY = OFF           // Peripheral Module Disable Configuration (Allow only one reconfiguration)
#pragma config IOL1WAY = ON             // Peripheral Pin Select Configuration (Allow only one reconfiguration)

// DEVCFG2
#pragma config FPLLIDIV = DIV_2
#pragma config FPLLMUL = MUL_20
#pragma config FPLLODIV = DIV_2

// DEVCFG1
#pragma config FNOSC = FRCPLL           // Oscillator Selection Bits (Fast RC Osc with PLL)
#pragma config FSOSCEN = OFF            // Secondary Oscillator Enable (Disabled)
#pragma config IESO = OFF               // Internal/External Switch Over (Diasbled)
#pragma config POSCMOD = OFF            // Primary Oscillator Configuration (External clock/Primary osc disabled)
#pragma config OSCIOFNC = OFF           // CLKO Output Signal Active on the OSCO Pin (Disabled)
#pragma config FPBDIV = DIV_2           // Peripheral Clock Divisor (Pb_Clk is Sys_Clk)
#pragma config FCKSM = CSECMD           // Clock Switching and Monitor Selection (Clock Switch Enabled, FSCM Disabled)
#pragma config WDTPS = PS1024           // Watchdog Timer Postscaler (1:1024, ~1s)
#pragma config WINDIS = OFF             // Watchdog Timer Window Enable (Watchdog Timer is in Non-Window Mode)
#pragma config FWDTEN = OFF             // Watchdog Timer Enable (WDT Disabled (SWDTEN Bit Controls))
//#pragma config FWDTWINSZ = WINSZ_25     // Watchdog Timer Window Size (Window Size is 25%)

// DEVCFG0
#pragma config DEBUG = OFF              // Background Debugger Enable (Debugger is Enabled)
#pragma config JTAGEN = OFF             // JTAG Enable (JTAG Port Disabled)
#pragma config ICESEL = ICS_PGx2        // ICE/ICD Comm Channel Select (Communicate on PGEC2/PGED2)
#pragma config PWP = OFF                // Program Flash Write Protect (Disable)
#pragma config BWP = OFF                // Boot Flash Write Protect bit (Protection Disabled)
#pragma config CP = OFF                 // Code Protect (Protection Disabled)

#define SYS_FREQ (40000000L)
#define NUM_SAMPLES 1024
#define TIMER_PERIOD 32768
#define MAX_NUM_AVGS 40
#define BTN_TMR_DIV 256
#define DEBOUNCE_MS 2

volatile unsigned char button_1_pressed, button_2_pressed, both_buttons_pressed, button_1_longpress, button_2_longpress;
unsigned short boot_PORTB_state;
void __ISR(_TIMER_4_VECTOR,IPL2SOFT) __T4Interrupt(void) {
    ++button_1_longpress;
    T4CONbits.ON = 0;
    T5CONbits.ON = 0;
    IFS0bits.T4IF = 0;
}
void __ISR(_TIMER_5_VECTOR,IPL2SOFT) __T5Interrupt(void) {
    ++button_2_longpress;
    T4CONbits.ON = 0;
    T5CONbits.ON = 0;
    IFS0bits.T5IF = 0;
}
static void S1_pressed() {
    if( !T4CONbits.ON ) {
        TMR4 = 0;
        T4CONbits.TCKPS = 7; // divide by 256
        IFS0bits.T4IF = 0;
        IPC4bits.T4IP = 2;
        IEC0bits.T4IE = 1;
        T4CONbits.ON = 1;
    }
}
static void S2_pressed() {
    if( !T5CONbits.ON ) {
        TMR5 = 0;
        T5CONbits.TCKPS = 7; // divide by 256
        IFS0bits.T5IF = 0;
        IPC5bits.T5IP = 2;
        IEC0bits.T5IE = 1;
        T5CONbits.ON = 1;
    }
}
void __ISR(_CHANGE_NOTICE_VECTOR,IPL3SOFT) __CNInterrupt(void) {
    if( CNSTATBbits.CNSTATB10 ) {
        if( !PORTBbits.RB10 ) {
            S1_pressed();
        } else {
            boot_PORTB_state |= 1<<10;
            if( T4CONbits.ON ) {
                if( TMR4 >= SYS_FREQ / BTN_TMR_DIV * DEBOUNCE_MS / 1000 ) {
                    if( T5CONbits.ON && TMR5 >= SYS_FREQ / BTN_TMR_DIV * DEBOUNCE_MS / 1000 ) {
                        ++both_buttons_pressed;
                        TMR5 = 0;
                        T5CONbits.ON = 0;
                    } else {
                        ++button_1_pressed;
                    }
                }
                T4CONbits.ON = 0;
            }
        }
    }
    if( CNSTATBbits.CNSTATB11 ) {
        if( !PORTBbits.RB11 ) {
            S2_pressed();
        } else {
            boot_PORTB_state |= 1<<11;
            if( T5CONbits.ON ) {
                if( TMR5 >= SYS_FREQ / BTN_TMR_DIV * DEBOUNCE_MS / 1000 ) {
                    if( T4CONbits.ON && TMR4 >= SYS_FREQ / BTN_TMR_DIV * DEBOUNCE_MS / 1000 ) {
                        ++both_buttons_pressed;
                        TMR4 = 0;
                        T4CONbits.ON = 0;
                    } else {
                        ++button_2_pressed;
                    }
                }
                T5CONbits.ON = 0;
            }
        }
    }
    IFS1bits.CNBIF = 0;
}

unsigned char LED_test_mode, LED_test_mode_S2_presses;
static void setup_IOs() {
  LATA = 0;
  LATB = 0;
  LATC = 0;

  TRISB &= ~(0xFF<<2);  // LED Matrix cathodes
  ANSELB &= ~(0xFF<<2); // LED Matrix cathodes
  TRISC &= ~0x3FF;      // LED Matrix anodes
  ANSELC &= ~0x3FF;     // LED Matrix anodes

  LATAbits.LATA8 = 1;    // LED1 cathode
  TRISAbits.TRISA8 = 0;  // LED1 cathode
  LATAbits.LATA3 = 1;    // LED2 cathode
  TRISAbits.TRISA3 = 0;  // LED2 cathode
  LATAbits.LATA2 = 1;    // LED3 cathode
  TRISAbits.TRISA2 = 0;  // LED3 cathode
  LATAbits.LATA4 = 1;    // LED4 cathode
  TRISAbits.TRISA4 = 0;  // LED4 cathode
  LATAbits.LATA9 = 1;    // LED5 cathode
  TRISAbits.TRISA9 = 0;  // LED5 cathode
  LATBbits.LATB12 = 1;   // LED6 cathode
  TRISBbits.TRISB12 = 0; // LED6 cathode
  ANSELBbits.ANSB12 = 0; // LED6 cathode
  LATAbits.LATA10 = 1;   // LED7 cathode
  TRISAbits.TRISA10 = 0; // LED7 cathode
  LATAbits.LATA7 = 1;    // LED8 cathode
  TRISAbits.TRISA7 = 0;  // LED8 cathode

  CNPDBbits.CNPDB13 = 1; // brightness trimpot

  CNPUBbits.CNPUB10 = 1;
  CNPUBbits.CNPUB11 = 1;
  CNENBbits.CNIEB10 = 1;
  CNENBbits.CNIEB11 = 1;
  CNCONBbits.ON = 1;
  IPC8bits.CNIP = 3;
  IFS1bits.CNBIF = 0;
#ifndef __DEBUG
  IEC1bits.CNBIE = 1;
  boot_PORTB_state = PORTB;
  if( !(boot_PORTB_state&(1<<10)) )
      LED_test_mode = 1;
  if( !(boot_PORTB_state&(1<<11)) )
      S2_pressed();
#endif
}

int left, right, leftavg = 32767, rightavg = 32767;
int BrightnessTrimpotValues[4];
volatile int BrightnessTrimpot, DisplayBrightness, samples_since_low_amp_reading;
int samples[2][2][NUM_SAMPLES];
int samplepos, subsample;
volatile unsigned int samplebuf, ADC_blanking, flash_timer, last_pot_read;
static void ProcessSamples(void) {
    int a, b, c, d, e, f;
    if( AD1CON2bits.BUFS ) {
        a = ADC1BUF0;
        b = ADC1BUF1;
        c = ADC1BUF2;
        d = ADC1BUF3;
        e = ADC1BUF4;
        f = ADC1BUF5;
    } else {
        a = ADC1BUF8;
        b = ADC1BUF9;
        c = ADC1BUFA;
        d = ADC1BUFB;
        e = ADC1BUFC;
        f = ADC1BUFD;
    }
    if( e >= 510 || e <= -511 ) {
        if( c >= 510 || c <= -511 ) {
            left = a * 23 * 23;
        } else {
            left = c * 23;
        }
        if( samples_since_low_amp_reading < 64 )
            ++samples_since_low_amp_reading;
    } else {
        left = e;
        if( samples_since_low_amp_reading > 4 )
            samples_since_low_amp_reading -= 4;
        else
            samples_since_low_amp_reading = 0;
    }
    if( f >= 510 || f <= -511 ) {
        if( d >= 510 || d <= -511 ) {
            right = b * 23 * 23;
        } else {
            right = d * 23;
        }
        if( samples_since_low_amp_reading < 64 )
            ++samples_since_low_amp_reading;
    } else {
        right = f;
        if( samples_since_low_amp_reading > 4 )
            samples_since_low_amp_reading -= 4;
        else
            samples_since_low_amp_reading = 0;
    }

    if( ADC_blanking ) {
        if( ADC_blanking == 7 ) {
            AD1CSSLbits.CSSL10 = 0;
            AD1CSSLbits.CSSL11 = 1;
        } else if( ADC_blanking >= 1 && ADC_blanking <= 4 && AD1CSSLbits.CSSL11 ) {
            BrightnessTrimpotValues[4-ADC_blanking] = f;
            if( ADC_blanking == 1 ) {
                int val = (/*BrightnessTrimpotValues[0] + BrightnessTrimpotValues[1] + */BrightnessTrimpotValues[2] + BrightnessTrimpotValues[3] + 512*2) / 2;
                if( val < 64 )
                    val = 64;
                BrightnessTrimpot = val;
                last_pot_read = flash_timer;
                AD1CSSLbits.CSSL10 = 1;
                AD1CSSLbits.CSSL11 = 0;
            }
        }
        --ADC_blanking;
        subsample = 0;
    } else {
        if( subsample == 0 ) {
            samples[samplebuf][0][samplepos] = left;
            samples[samplebuf][1][samplepos] = right;
            subsample = 1;
        } else {
            samples[samplebuf][0][samplepos] = (samples[samplebuf][0][samplepos] + left) / 2;
            samples[samplebuf][1][samplepos] = (samples[samplebuf][1][samplepos] + right) / 2;

            if( ++samplepos == NUM_SAMPLES ) {
                samplepos = 0;
                samplebuf ^= 1;
            }
            subsample = 0;
        }
    }
    IFS0bits.AD1IF = 0;
}
void __ISR(_ADC_VECTOR,IPL1SOFT) ADCinterrupt(void) {
    ProcessSamples();
}

static void setup_ADC() {
    ANSELAbits.ANSA0 = 1;
    ANSELAbits.ANSA1 = 1;
    ANSELBbits.ANSB0 = 1;
    ANSELBbits.ANSB1 = 1;
    ANSELBbits.ANSB13 = 1;
    ANSELBbits.ANSB14 = 1;
    ANSELBbits.ANSB15 = 1;
    AD1CON1bits.ASAM = 1;
    AD1CON1bits.SSRC = 7;
    AD1CON1bits.FORM = 5;
    AD1CON2bits.BUFM = 1;
    AD1CON2bits.SMPI = 6-1;
    AD1CON2bits.CSCNA = 1;
    AD1CON3bits.ADCS = 0;
    AD1CON3bits.SAMC = 8;
    AD1CSSLbits.CSSL0 = 1;
    AD1CSSLbits.CSSL1 = 1;
    AD1CSSLbits.CSSL2 = 1;
    AD1CSSLbits.CSSL3 = 1;
    AD1CSSLbits.CSSL9 = 1;
    AD1CSSLbits.CSSL10 = 1;
    IFS0bits.AD1IF = 0;
    IPC5bits.AD1IP = 1;
    IEC0bits.AD1IE = 1;
    AD1CON1bits.ON = 1;
}

volatile unsigned int bank_states[8], bank_dim_masks[8][3], bank_dim_vals[8][4];
unsigned char which_LED_cathodes = 0, which_dim_cathode, which_dim_bank;
void __ISR(_TIMER_1_VECTOR,IPL4SOFT) __T1Interrupt(void) {
    if( TMR2 > TIMER_PERIOD/2 ) {
        unsigned int LEDs_to_turn_on;

        TMR3 = 0;
        PR3 = TIMER_PERIOD * 2;
        LEDs_to_turn_on = bank_states[which_LED_cathodes];
        ++flash_timer;

        if( LEDs_to_turn_on || (LATC&0x3FF) ) {
            LATCCLR = 0x3FF;
            if( LEDs_to_turn_on ) {
                LATBCLR = 0xFF<<2;
                LATBSET = 4 << which_LED_cathodes;
                LATCSET = bank_states[which_LED_cathodes];
                which_dim_cathode = which_LED_cathodes;
                if( LEDs_to_turn_on & bank_dim_masks[which_LED_cathodes][0] ) {
                    which_dim_bank = 0;
                    PR3 = DisplayBrightness * (128 - bank_dim_vals[which_LED_cathodes][which_dim_bank]) / 4 * TIMER_PERIOD / 1024 / 32;
                } else if( LEDs_to_turn_on & bank_dim_masks[which_LED_cathodes][1] ) {
                    which_dim_bank = 1;
                    PR3 = DisplayBrightness * (128 - bank_dim_vals[which_LED_cathodes][which_dim_bank]) / 4 * TIMER_PERIOD / 1024 / 32;
                } else if( LEDs_to_turn_on & bank_dim_masks[which_LED_cathodes][2] ) {
                    which_dim_bank = 2;
                    PR3 = DisplayBrightness * (128 - bank_dim_vals[which_LED_cathodes][which_dim_bank]) / 4 * TIMER_PERIOD / 1024 / 32;
                } else if( LEDs_to_turn_on && (DisplayBrightness < 1024 || bank_dim_vals[which_LED_cathodes][3]) ) {
                    which_dim_bank = 3;
                    PR3 = DisplayBrightness * (128 - bank_dim_vals[which_LED_cathodes][which_dim_bank]) / 4 * TIMER_PERIOD / 1024 / 32;
                }
            }
            if( samples_since_low_amp_reading < 32 || (flash_timer-last_pot_read) > 48 )
                ADC_blanking = 7;
        }
        which_LED_cathodes = (which_LED_cathodes + 1) & 7;
        TMR2 = 0;
    }
    IFS0bits.T1IF = 0;
}

void __ISR(_TIMER_3_VECTOR,IPL4SOFT) __T3Interrupt(void) {
    if( !ADC_blanking && (samples_since_low_amp_reading < 32 || (flash_timer-last_pot_read) > 48) )
        ADC_blanking = 4;
    PR3 = TIMER_PERIOD * 2;
    if( which_dim_bank == 3 ) {
        LATCCLR = 0x3FF;
    } else {
        LATCCLR = bank_dim_masks[which_dim_cathode][which_dim_bank];
        while( ++which_dim_bank < 3 ) {
            if( (LATC&0x3FF) & bank_dim_masks[which_dim_cathode][which_dim_bank] ) {
                PR3 = DisplayBrightness * bank_dim_vals[which_LED_cathodes][which_dim_bank] / 4 * TIMER_PERIOD / 1024 / 32;
                break;
            }
        }
        if( which_dim_bank == 3 && (LATC&0x3FF) && (DisplayBrightness < 1024 || bank_dim_vals[which_LED_cathodes][which_dim_bank]) )
            PR3 = DisplayBrightness * (128 - bank_dim_vals[which_LED_cathodes][which_dim_bank]) / 4 * TIMER_PERIOD / 1024 / 32;
    }
    IFS0bits.T3IF = 0;
}

static void setup_TMR1() {
  PR1 = TIMER_PERIOD;
  IFS0bits.T1IF = 0;
  IPC1bits.T1IP = 4;
  IEC0bits.T1IE = 1;
  T1CONbits.TON = 1;

  T2CONbits.TON = 1;

  PR3 = TIMER_PERIOD * 2;
  IFS0bits.T3IF = 0;
  IPC3bits.T3IP = 4;
  IEC0bits.T3IE = 1;
  T3CONbits.TON = 1;
}

unsigned char Sensitivity, save_reference_level, adjust_reference_level, MaxLevel = 1;
int MeterMode, ShowMeterMode, ShowMeterModeTimer, ShowNoiseNull;
float ReferenceLevels_dBV[4][2] = { { -10.0f, -10.0f }, { 0.0f, 0.0f }, { 1.79f, 1.79f }, { 7.0f, 7.0f } };
union {
    struct {
        float left_rms, right_rms, left_peak, right_peak;
    } _float;
    struct {
        unsigned int left_rms, right_rms, left_peak, right_peak;
    } _int;
} noise_null;
unsigned int avg_history_pos, peak_history_pos, num_avgs, num_peak;

static void SaveSettings() {
    DataEEWrite( MaxLevel|(Sensitivity<<2)|(MeterMode<<4), 1 );
}

static void SaveNoiseNull() {
    DataEEWrite( noise_null._int.left_rms, 2 );
    DataEEWrite( noise_null._int.right_rms, 3 );
    DataEEWrite( noise_null._int.left_peak, 4 );
    DataEEWrite( noise_null._int.right_peak, 5 );
}

static void LoadAvgPeakPeriod() {
    num_avgs = 24;
    num_peak = 24;
    DataEERead(&num_avgs, 6);
    if( num_avgs < 1 || num_avgs > MAX_NUM_AVGS )
        num_avgs = 24;
    DataEERead(&num_peak, 7);
    if( num_peak < 1 || num_peak > MAX_NUM_AVGS )
        num_peak = 24;
}

static void SaveAvgPeakPeriod() {
    DataEEWrite( num_avgs, 6 );
    DataEEWrite( num_peak, 7 );
}

static void LoadReferenceLevels() {
    DataEERead((unsigned int*)&ReferenceLevels_dBV[0][0], 8);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[0][1], 9);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[1][0], 10);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[1][1], 11);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[2][0], 12);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[2][1], 13);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[3][0], 14);
    DataEERead((unsigned int*)&ReferenceLevels_dBV[3][1], 15);
}

static void SaveReferenceLevel(unsigned int which) {
    DataEEWrite(*((unsigned int*)&ReferenceLevels_dBV[which][0]), 8 + which * 2);
    DataEEWrite(*((unsigned int*)&ReferenceLevels_dBV[which][1]), 9 + which * 2);
}

signed char LED_Relative_Brightness[80];
static void LoadLEDBrightnessCalibration() {
    int i;
    for( i = 0; i < 80/4; ++i )
        DataEERead( &((unsigned int*)LED_Relative_Brightness)[i], 16 + i );
}

static void SaveLEDBrightnessCalibration() {
    int i;
    for( i = 0; i < 80/4; ++i )
        DataEEWrite( ((unsigned int*)LED_Relative_Brightness)[i], 16 + i );
}

static void LoadSettings() {
    unsigned int data = 0;
    if( DataEERead(&data, 1) == 0 ) {
        MaxLevel = data&3;
        Sensitivity = (data>>2)&3;
        MeterMode = (data>>4)&7;
        if( MeterMode > 4 )
            MeterMode = 0;
    }
    DataEERead(&noise_null._int.left_rms, 2);
    DataEERead(&noise_null._int.right_rms, 3);
    DataEERead(&noise_null._int.left_peak, 4);
    DataEERead(&noise_null._int.right_peak, 5);
    LoadAvgPeakPeriod();
    LoadReferenceLevels();
}

float left_rms, right_rms, left_peak, right_peak;
int last_samplebuf;
static inline float square(float val) {
    return val * val;
}
static int level_to_db(int min, int max, float offset, float level, float db_step) {
    int ret = 40 + (log10(level) + offset) * (20 / db_step);
    if( ret < min )
        return min;
    else if( ret > max )
        return max;
    else
        return ret;
}

static unsigned short calc_bar_dot(int bar, int dot, int min, int max) {
    unsigned short ret;
    if( bar > max )
        bar = max;
    if( bar >= min )
        ret = (1<<(bar-min))-1;
    else
        ret = 0;
    if( dot >= min && dot < max )
        ret |= (1<<(dot-min));
    return ret;
}
static void show_ref_level_bar(volatile unsigned int* here, float val) {
    unsigned int temp[4] = { 0, 0, 0, 0 };
    int a, b, i, j;
    a = (val + 20.0f) * 10.0f;
    b = a;
    a /= 10;
    b -= a * 10;
//    a += 19;
    if( a < 0 )
        a = 0;
    else if( a > 40 - 12 )
        a = 40 - 12;
    for( i = a; i < a + 12; ++i ) {
        if( i != a + 1 + b ) {
            j = i / 10;
            temp[j] |= 1<<(i - j * 10);
        }
    }
    here[0] = temp[0];
    here[1] = temp[1];
    here[2] = temp[2];
    here[3] = temp[3];
}
static void set_bar_dot(int left_bar, int left_dot, int right_bar, int right_dot) {
    if( adjust_reference_level == 2 ) {
        show_ref_level_bar(bank_states+0, ReferenceLevels_dBV[MaxLevel][1]);
    } else {
        bank_states[0] = calc_bar_dot( left_bar,  left_dot,  0, 10);
        bank_states[1] = calc_bar_dot( left_bar,  left_dot, 10, 20);
        bank_states[2] = calc_bar_dot( left_bar,  left_dot, 20, 30);
        bank_states[3] = calc_bar_dot( left_bar,  left_dot, 30, 40);
    }
    if( adjust_reference_level == 1 ) {
        show_ref_level_bar(bank_states+4, ReferenceLevels_dBV[MaxLevel][0]);
    } else {
        bank_states[4] = calc_bar_dot(right_bar, right_dot,  0, 10);
        bank_states[5] = calc_bar_dot(right_bar, right_dot, 10, 20);
        bank_states[6] = calc_bar_dot(right_bar, right_dot, 20, 30);
        bank_states[7] = calc_bar_dot(right_bar, right_dot, 30, 40);
    }
}

float left_rms_history[MAX_NUM_AVGS], left_peak_history[MAX_NUM_AVGS];
float right_rms_history[MAX_NUM_AVGS], right_peak_history[MAX_NUM_AVGS];
static void analyse_last_samplebuf() {
    int i, sens, left_bar, right_bar, left_dot, right_dot;
    float offsetL, offsetR;

    // compute average/min/max buffer sample values
    int left_avg = 0, left_max = 0, left_min = 1024 * 23 * 23;
    int right_avg = 0, right_max = 0, right_min = 1024 * 23 * 23;
    float left_avg_float, right_avg_float;
    for( i = 0; i < NUM_SAMPLES; ++i ) {
        left_avg  += samples[last_samplebuf][0][i];
        if( samples[last_samplebuf][0][i] < left_min )
            left_min = samples[last_samplebuf][0][i];
        else if( samples[last_samplebuf][0][i] > left_max )
            left_max = samples[last_samplebuf][0][i];
        right_avg += samples[last_samplebuf][1][i];
        if( samples[last_samplebuf][1][i] < right_min )
            right_min = samples[last_samplebuf][1][i];
        else if( samples[last_samplebuf][1][i] > right_max )
            right_max = samples[last_samplebuf][1][i];
    }
    left_avg_float = (float)left_avg / NUM_SAMPLES;
    right_avg_float = (float)right_avg / NUM_SAMPLES;

    // now compute RMS of buffer sample values with average subtracted
    left_rms = 0;
    right_rms = 0;
    for( i = 0; i < NUM_SAMPLES; ++i ) {
        left_rms  += square(samples[last_samplebuf][0][i] - left_avg_float);
        right_rms += square(samples[last_samplebuf][1][i] - right_avg_float);
    }
    left_rms /= NUM_SAMPLES;
    right_rms /= NUM_SAMPLES;
    // re-scale computed values from ADC units to voltage
    left_rms = sqrt(left_rms) * 3.3 * 2 / 1024 / 23 / 23;
    right_rms = sqrt(right_rms) * 3.3 * 2 / 1024 / 23 / 23;

    // convert min/max values to peak values (relative to average)
    left_peak = left_max - left_avg_float;
    if( left_avg_float - left_min > left_peak )
        left_peak = left_avg_float - left_min;
    right_peak = right_max - right_avg_float;
    if( right_avg_float - right_min > right_peak )
        right_peak = right_avg_float - right_min;
    // re-scale computed values from ADC units to voltage
    left_peak = left_peak * 3.3 / 1024 / 23 / 23;
    right_peak = right_peak * 3.3 / 1024 / 23 / 23;

    // average RMS values over history window
    left_rms_history[avg_history_pos] = left_rms;
    right_rms_history[avg_history_pos] = right_rms;
    if( MeterMode >= 3 ) {
        // VU meter
    } else {
        left_rms = 0;
        right_rms = 0;
        for( i = 0; i < num_avgs; ++i ) {
            left_rms += left_rms_history[i];
            right_rms += right_rms_history[i];
        }
        left_rms /= num_avgs;
        right_rms /= num_avgs;
    }

    // perform RMS noise subtration calculations
    left_rms = square(left_rms) - square(noise_null._float.left_rms);
    if( left_rms < 0 )
        left_rms = 0;
    else
        left_rms = sqrt(left_rms);
    right_rms = square(right_rms) - square(noise_null._float.right_rms);
    if( right_rms < 0 )
        right_rms = 0;
    else
        right_rms = sqrt(right_rms);

    if( MeterMode >= 3 ) {
        // VU meter
        static float VUpos[2] = { 0.0f, 0.0f }, VUvel[2] = { 0.0f, 0.0f };
        VUvel[0] += ( left_rms - VUpos[0]) / 13.0f;
        VUvel[1] += (right_rms - VUpos[1]) / 13.0f;
        VUpos[0] += VUvel[0];
        VUpos[1] += VUvel[1];
        if( VUpos[0] < 0 ) { // "needle" bounce off zero
            VUpos[0] = 0;
            VUvel[0] *= -0.125f;
        }
        if( VUpos[1] < 0 ) { // "needle" bounce off zero
            VUpos[1] = 0;
            VUvel[1] *= -0.125f;
        }
        VUvel[0] /= 1.68f;
        VUvel[1] /= 1.68f;
         left_rms = VUpos[0];
        right_rms = VUpos[0];
    }

    // compute peak values over history window
    left_peak_history[peak_history_pos] = left_peak;
    right_peak_history[peak_history_pos] = right_peak;
    left_peak = 0;
    right_peak = 0;
    for( i = 0; i < num_peak; ++i ) {
        if( left_peak_history[i] > left_peak )
            left_peak = left_peak_history[i];
        if( right_peak_history[i] > right_peak )
            right_peak = right_peak_history[i];
    }
    // subtract noise from peak values
    left_peak -= noise_null._float.left_peak;
    if( left_peak < 0 )
        left_peak = 0;
    right_peak -= noise_null._float.right_peak;
    if( right_peak < 0 )
        right_peak = 0;

    if( ++avg_history_pos == num_avgs )
        avg_history_pos = 0;
    if( ++peak_history_pos == num_peak )
        peak_history_pos = 0;

    if( save_reference_level && avg_history_pos == 0 ) {
        ReferenceLevels_dBV[MaxLevel][0] = log10(left_rms) * 20.0f;
        ReferenceLevels_dBV[MaxLevel][1] = log10(right_rms) * 20.0f;
        SaveReferenceLevel(MaxLevel);
        save_reference_level = 0;
    }

    // convert RMS & peak values to bar/dot positions based on current modes
    offsetL = ReferenceLevels_dBV[MaxLevel][0] / -20.0f;
    offsetR = ReferenceLevels_dBV[MaxLevel][1] / -20.0f;
    switch( Sensitivity ) {
        case 0:
            sens = 40;
            break;
        case 1:
            sens = 60;
            break;
        case 2:
            sens = 80;
            break;
        default:
            sens = 100;
            break;
    }

    switch( MeterMode ) {
        default: // Average + Peak
            left_bar = level_to_db(0, 40, offsetL, left_rms, sens / 40.0f);
            right_bar = level_to_db(0, 40, offsetR, right_rms, sens / 40.0f);
            left_dot = level_to_db(-1, 39, offsetL, left_peak, sens / 40.0f);
            right_dot = level_to_db(-1, 39, offsetR, right_peak, sens / 40.0f);
            break;
        case 1: // Average only
        case 4:
            left_bar = level_to_db(0, 40, offsetL, left_rms, sens / 40.0f);
            right_bar = level_to_db(0, 40, offsetR, right_rms, sens / 40.0f);
            left_dot = -1;
            right_dot = -1;
            break;
        case 2: // Peak only
        case 5:
            left_bar = level_to_db(0, 40, offsetL, left_peak, sens / 40.0f);
            right_bar = level_to_db(0, 40, offsetR, right_peak, sens / 40.0f);
            left_dot = -1;
            right_dot = -1;
            break;
    }
    set_bar_dot(left_bar, left_dot, right_bar, right_dot);
}

unsigned char LED_brightness_calibration, LED_brightness_calibration_pot_moved, set_avg_peak_period;
int LED_brightness_calibration_pot_pos;
static void UpdateLEDBrightnessCalculations() {
    int chn;
    for( chn = 0; chn < 2; ++chn ) {
        unsigned short brightness[40];
        int i, j;
        for( i = 0; i < 40; ++i )
            brightness[i] = 128 * 256;
        for( i = 0; i < 39; ++i ) {
            int val = LED_Relative_Brightness[i+(chn ? 40 : 0)];
            if( val > 0 ) {
                for( j = 0; j <= i; ++j )
                    brightness[j] = (unsigned int)brightness[j] * (unsigned int)(64 - val) / 64;
            } else if( val < 0 ) {
                for( j = i + 1; j < 40; ++j )
                    brightness[j] = (unsigned int)brightness[j] * (unsigned int)(64 + val) / 64;
            }
        }
        for( i = 0; i < 4; ++i ) {
            unsigned int maxbr = 0, k, restmask = 0x3FF, ii;
            for( j = i * 10; j < (i+1) * 10; ++j )
                if( brightness[j] > maxbr )
                    maxbr = brightness[j];
            if( maxbr < 16 * 256 )
                maxbr = 16 * 256;
            ii = i + (chn ? 4 : 0);
            bank_dim_vals[ii][3] = (128 * 256 - maxbr) / 256;
            for( j = 0; j < 3; ++j ) {
                int nextmaxbr = -1;
                for( k = i * 10; k < (i+1) * 10; ++k )
                    if( brightness[k] > nextmaxbr && brightness[k] < maxbr )
                        nextmaxbr = brightness[k];
                if( nextmaxbr >= 0 ) {
                    unsigned int mask = 0, temp;
                    for( k = i * 10; k < (i+1) * 10; ++k )
                        if( brightness[k] == nextmaxbr )
                            mask |= (1<<(k - i*10));
                    temp = nextmaxbr;
                    if( temp < 16 * 256 )
                        temp = 16 * 256;
                    bank_dim_vals[ii][j] = (128 * 256 - temp) / 256;
                    bank_dim_masks[ii][j] = (j == 2 || temp == 16 * 256 ? restmask : mask);
                    restmask &= ~mask;
                    maxbr = nextmaxbr;
                } else {
                    bank_dim_vals[ii][j] = 0;
                    bank_dim_masks[ii][j] = 0;
                }
            }
        }
    }
}
static void UpdateLEDStates() {
    if( ShowMeterMode && !LED_test_mode ) {
        LATAbits.LATA8  = !((flash_timer&128) && MeterMode < 3);
        LATAbits.LATA3  = !((flash_timer&128) && MeterMode >= 3);
        LATAbits.LATA2  = 1;
        LATAbits.LATA4  = 1;
        LATAbits.LATA9  = !((flash_timer&128) && (MeterMode == 0 || MeterMode == 3));
//      LATBbits.LATB12 = !((flash_timer&128) && (MeterMode == 1 || MeterMode == 4));
        if( (flash_timer&128) && (MeterMode == 1 || MeterMode == 4) )
            LATBCLR = 1<<12;
        else
            LATBSET = 1<<12;
        LATAbits.LATA10 = !((flash_timer&128) && (MeterMode == 2 || MeterMode == 5));
        LATAbits.LATA7  = 1;
        if( flash_timer-ShowMeterModeTimer >= 1024 )
            ShowMeterMode = 0;
    } else if( ShowNoiseNull && !LED_test_mode ) {
        LATAbits.LATA8  = 1;
        LATAbits.LATA3  = 1;
        LATAbits.LATA2  = 1;
        LATAbits.LATA4  = !((flash_timer&128) && (ShowNoiseNull == 2));
        LATAbits.LATA9  = 1;
//      LATBbits.LATB12 = !((flash_timer&128) && (MeterMode == 1 || MeterMode == 4));
        LATBSET = 1<<12;
        LATAbits.LATA10 = 1;
        LATAbits.LATA7  = !((flash_timer&128) && (ShowNoiseNull == 1));
        if( flash_timer-ShowMeterModeTimer >= 1024 )
            ShowNoiseNull = 0;
    } else {
        LATAbits.LATA8  = !(LED_test_mode ? (LED_test_mode_S2_presses != 1 || (flash_timer&128)) : (!LED_brightness_calibration && MaxLevel == 0));
        LATAbits.LATA3  = !(LED_test_mode ? (LED_test_mode_S2_presses != 2 || (flash_timer&128)) : (!LED_brightness_calibration && MaxLevel == 1));
        LATAbits.LATA2  = !(LED_test_mode ? (LED_test_mode_S2_presses != 3 || (flash_timer&128)) : (!LED_brightness_calibration && MaxLevel == 2));
        LATAbits.LATA4  = !(LED_test_mode || (!LED_brightness_calibration && MaxLevel == 3));
        LATAbits.LATA9  = !(LED_test_mode || (!LED_brightness_calibration && Sensitivity == 0));
//      LATBbits.LATB12 = !(Sensitivity == 1);
        if( LED_test_mode || (!LED_brightness_calibration && Sensitivity == 1) )
            LATBCLR = 1<<12;
        else
            LATBSET = 1<<12;
        LATAbits.LATA10 = !(LED_test_mode || (!LED_brightness_calibration && Sensitivity == 2));
        LATAbits.LATA7  = !(LED_test_mode || (!LED_brightness_calibration && Sensitivity == 3));
    }
    if( button_1_pressed ) {
        --button_1_pressed;
        if( LED_brightness_calibration ) {
            if( LED_brightness_calibration > 1 ) {
                if( LED_brightness_calibration_pot_moved )
                    LED_Relative_Brightness[LED_brightness_calibration-1] = (BrightnessTrimpot-512) / 8;
                --LED_brightness_calibration;
                if( LED_brightness_calibration == 40 )
                    UpdateLEDBrightnessCalculations();
                LED_brightness_calibration_pot_moved = 0;
                LED_brightness_calibration_pot_pos = BrightnessTrimpot;
            }
        } else if( set_avg_peak_period ) {
            if( set_avg_peak_period == 1 ) {
                if( num_avgs > 1 )
                    --num_avgs;
            } else {
                if( num_peak > 1 )
                    --num_peak;
            }
        } else if( adjust_reference_level ) {
            if( ReferenceLevels_dBV[MaxLevel][adjust_reference_level-1] > -20.0f )
                ReferenceLevels_dBV[MaxLevel][adjust_reference_level-1] -= 0.1f;
        } else {
            Sensitivity = (Sensitivity + 1) & 3;
            SaveSettings();
        }
    }
    if( button_2_pressed ) {
        --button_2_pressed;
        if( LED_test_mode ) {
            LED_test_mode_S2_presses = (LED_test_mode_S2_presses+1)&3;
        } else if( LED_brightness_calibration ) {
            if( LED_brightness_calibration < 80 ) {
                if( LED_brightness_calibration_pot_moved )
                    LED_Relative_Brightness[LED_brightness_calibration-1] = (BrightnessTrimpot-512) / 8;
                ++LED_brightness_calibration;
                if( LED_brightness_calibration == 40 || LED_brightness_calibration == 80 )
                    UpdateLEDBrightnessCalculations();
                LED_brightness_calibration_pot_moved = 0;
                LED_brightness_calibration_pot_pos = BrightnessTrimpot;
            }
        } else if( set_avg_peak_period ) {
            if( set_avg_peak_period == 1 ) {
                if( num_avgs < MAX_NUM_AVGS )
                    ++num_avgs;
            } else {
                if( num_peak < MAX_NUM_AVGS )
                    ++num_peak;
            }
        } else if( adjust_reference_level ) {
            if( ReferenceLevels_dBV[MaxLevel][adjust_reference_level-1] < 7.4f )
                ReferenceLevels_dBV[MaxLevel][adjust_reference_level-1] += 0.1f;
        } else {
            MaxLevel = (MaxLevel + 1) & 3;
            SaveSettings();
        }
    }
    if( both_buttons_pressed ) {
        --both_buttons_pressed;
        if( LED_brightness_calibration ) {
            LED_brightness_calibration = 0;
            SaveLEDBrightnessCalibration();
            UpdateLEDBrightnessCalculations();
        } else if( adjust_reference_level ) {
            ReferenceLevels_dBV[MaxLevel][adjust_reference_level-1] = ReferenceLevels_dBV[MaxLevel][2-adjust_reference_level];
        } else {
            if( MeterMode == 4 )
                MeterMode = 0;
            else
                ++MeterMode;
            ShowMeterMode = 1;
            ShowMeterModeTimer = flash_timer;
            SaveSettings();
        }
    }
    if( button_1_longpress ) {
        --button_1_longpress;
        if( LED_brightness_calibration ) {
            LED_Relative_Brightness[LED_brightness_calibration-1] = 0;
            LED_brightness_calibration_pot_moved = 0;
            LED_brightness_calibration_pot_pos = BrightnessTrimpot;
        } else if( set_avg_peak_period ) {
            set_avg_peak_period = 3 - set_avg_peak_period;
        } else if( adjust_reference_level ) {
            adjust_reference_level = 3-adjust_reference_level;
        } else {
            noise_null._float.left_rms = left_rms;
            noise_null._float.right_rms = left_rms;
            noise_null._float.left_peak = left_peak;
            noise_null._float.right_peak = left_peak;
            SaveNoiseNull();
            ShowNoiseNull = 1;
            ShowMeterModeTimer = flash_timer;
        }
    }
    if( button_2_longpress ) {
        --button_2_longpress;
        if( LED_brightness_calibration ) {
            memset(LED_Relative_Brightness, 0, sizeof(LED_Relative_Brightness));
            LED_brightness_calibration_pot_moved = 0;
            LED_brightness_calibration_pot_pos = BrightnessTrimpot;
        } else if( set_avg_peak_period ) {
            SaveAvgPeakPeriod();
            set_avg_peak_period = 0;
        } else if( adjust_reference_level ) {
            SaveReferenceLevel(MaxLevel);
            adjust_reference_level = 0;
        } else {
            if( !(boot_PORTB_state&(1<<11)) ) {
                LED_brightness_calibration = 1;
//                memset(LED_Relative_Brightness, 0, sizeof(LED_Relative_Brightness));
                LED_brightness_calibration_pot_moved = 0;
                LED_brightness_calibration_pot_pos = BrightnessTrimpot;
            } else {
                noise_null._float.left_rms = 0;
                noise_null._float.right_rms = 0;
                noise_null._float.left_peak = 0;
                noise_null._float.right_peak = 0;
                SaveNoiseNull();
                ShowNoiseNull = 2;
                ShowMeterModeTimer = flash_timer;
            }
        }
    }
}

static int realmain() {
    SYSTEMConfigPerformance(SYS_FREQ);
    INTEnableSystemMultiVectoredInt();
    INTEnableInterrupts();

    setup_IOs();
    setup_TMR1();
    setup_ADC();

    DataEEInit();
    LoadSettings();

#if 0
    LED_Relative_Brightness[29] = 96/2;
    LED_Relative_Brightness[33] = -36/2;
    LED_Relative_Brightness[37] = -24/2;
    LED_Relative_Brightness[29+40] = 96/2;
    LED_Relative_Brightness[33+40] = -36/2;
    LED_Relative_Brightness[37+40] = -36/2;
#endif
    LoadLEDBrightnessCalibration();
    UpdateLEDBrightnessCalculations();

    while(1) {
        if( LED_test_mode ) {
            bank_states[0] = 0x3FF;
            bank_states[1] = 0x3FF;
            bank_states[2] = 0x3FF;
            bank_states[3] = 0x3FF;
            bank_states[4] = 0x3FF;
            bank_states[5] = 0x3FF;
            bank_states[6] = 0x3FF;
            bank_states[7] = 0x3FF;
            if( BrightnessTrimpot > 63 )
                DisplayBrightness = BrightnessTrimpot+1;
            else
                DisplayBrightness = 64;

            if( (boot_PORTB_state&(1<<10)) ) {
                LED_test_mode = 0;
                if( LED_test_mode_S2_presses == 1 ) {
                    set_avg_peak_period = 1;
                } else if( LED_test_mode_S2_presses == 2 ) {
                    save_reference_level = 1;
                } else if( LED_test_mode_S2_presses == 3 ) {
                    if( ReferenceLevels_dBV[MaxLevel][0] < -20.0f )
                        ReferenceLevels_dBV[MaxLevel][0] = -20.0f;
                    if( ReferenceLevels_dBV[MaxLevel][1] < -20.0f )
                        ReferenceLevels_dBV[MaxLevel][1] = -20.0f;
                    adjust_reference_level = 1;
                }
            }
        } else if( set_avg_peak_period ) {
            set_bar_dot(0, ((flash_timer&256) || set_avg_peak_period != 1) ? num_avgs - 1 : -1, 0, ((flash_timer&256) || set_avg_peak_period != 2) ? num_peak - 1 : -1);
            if( BrightnessTrimpot > 63 )
                DisplayBrightness = BrightnessTrimpot+1;
            else
                DisplayBrightness = 64;
        } else if( LED_brightness_calibration ) {
            if( LED_brightness_calibration == 40 || LED_brightness_calibration == 80 ) {
                bank_states[0] = (LED_brightness_calibration == 40 ? 0x3FF : 0);
                bank_states[1] = (LED_brightness_calibration == 40 ? 0x3FF : 0);
                bank_states[2] = (LED_brightness_calibration == 40 ? 0x3FF : 0);
                bank_states[3] = (LED_brightness_calibration == 40 ? 0x3FF : 0);
                bank_states[4] = (LED_brightness_calibration == 80 ? 0x3FF : 0);
                bank_states[5] = (LED_brightness_calibration == 80 ? 0x3FF : 0);
                bank_states[6] = (LED_brightness_calibration == 80 ? 0x3FF : 0);
                bank_states[7] = (LED_brightness_calibration == 80 ? 0x3FF : 0);
                if( BrightnessTrimpot > 63 )
                    DisplayBrightness = BrightnessTrimpot+1;
                else
                    DisplayBrightness = 64;
//                UpdateLEDBrightnessCalculations();
            } else {
                int a, b, c, d, x, y;
                DisplayBrightness = 1024;

                a = LED_brightness_calibration-1;
                b = LED_brightness_calibration;
                c = 0;
                if( b >= 40 ) {
                    a -= 40;
                    b -= 40;
                    c = 4;
                }
                d = 4 - c;
                bank_states[c+0] = (a >= 0 && a < 10 ? (1<<a) : 0) | (b >= 0 && b < 10 ? (1<<b) : 0);
                a -= 10;
                b -= 10;
                bank_states[c+1] = (a >= 0 && a < 10 ? (1<<a) : 0) | (b >= 0 && b < 10 ? (1<<b) : 0);
                a -= 10;
                b -= 10;
                bank_states[c+2] = (a >= 0 && a < 10 ? (1<<a) : 0) | (b >= 0 && b < 10 ? (1<<b) : 0);
                a -= 10;
                b -= 10;
                bank_states[c+3] = (a >= 0 && a < 10 ? (1<<a) : 0) | (b >= 0 && b < 10 ? (1<<b) : 0);
                bank_states[d+0] = 0;
                bank_states[d+1] = 0;
                bank_states[d+2] = 0;
                bank_states[d+3] = 0;

                if( abs(LED_brightness_calibration_pot_pos - BrightnessTrimpot) > 4 )
                    LED_brightness_calibration_pot_moved = 1;

                x = y = 0;
                if( LED_brightness_calibration_pot_moved ) {
                    if( BrightnessTrimpot > 512 )
                        x = (BrightnessTrimpot-512)/4;
                    else if( BrightnessTrimpot < 512 )
                        y = (512-BrightnessTrimpot)/4;
                } else {
                    if( LED_Relative_Brightness[LED_brightness_calibration-1] >= 0 )
                        x = LED_Relative_Brightness[LED_brightness_calibration-1] * 2;
                    else
                        y = LED_Relative_Brightness[LED_brightness_calibration-1] * -2;
                }

                if( LED_brightness_calibration == 10 || LED_brightness_calibration == 20 || LED_brightness_calibration == 30 ||
                    LED_brightness_calibration == 50 || LED_brightness_calibration == 60 || LED_brightness_calibration == 70 ) {
                    bank_dim_vals[c+0][3] =                                                                            (LED_brightness_calibration == 10 || LED_brightness_calibration == 50 ? x : 0);
                    bank_dim_vals[c+1][3] = (LED_brightness_calibration == 10 || LED_brightness_calibration == 50 ? y : LED_brightness_calibration == 20 || LED_brightness_calibration == 60 ? x : 0);
                    bank_dim_vals[c+2][3] = (LED_brightness_calibration == 20 || LED_brightness_calibration == 60 ? y : LED_brightness_calibration == 30 || LED_brightness_calibration == 70 ? x : 0);
                    bank_dim_vals[c+3][3] = (LED_brightness_calibration == 30 || LED_brightness_calibration == 70 ? y :                                        0);
                    bank_dim_masks[c+0][0] = 0;
                    bank_dim_masks[c+0][1] = 0;
                    bank_dim_masks[c+0][2] = 0;
                    bank_dim_masks[c+1][0] = 0;
                    bank_dim_masks[c+1][1] = 0;
                    bank_dim_masks[c+1][2] = 0;
                    bank_dim_masks[c+2][0] = 0;
                    bank_dim_masks[c+2][1] = 0;
                    bank_dim_masks[c+2][2] = 0;
                    bank_dim_masks[c+3][0] = 0;
                    bank_dim_masks[c+3][1] = 0;
                    bank_dim_masks[c+3][2] = 0;
                } else {
                    bank_dim_vals[c+0][3] = 0;
                    bank_dim_vals[c+1][3] = 0;
                    bank_dim_vals[c+2][3] = 0;
                    bank_dim_vals[c+3][3] = 0;
//                    int bank = LED_brightness_calibration/10;
                    int n = (x == 0 ? b : a) + 30;
                    int v = (x == 0 ? y : x);
                    bank_dim_vals[c+0][0] = (n >= 0 && n < 10 ? v : 0);
                    bank_dim_masks[c+0][0] = (n >= 0 && n < 10 ? (1<<n) : 0);
                    bank_dim_masks[c+0][1] = 0;
                    bank_dim_masks[c+0][2] = 0;
                    n -= 10;
                    bank_dim_vals[c+1][0] = (n >= 0 && n < 10 ? v : 0);
                    bank_dim_masks[c+1][0] = (n >= 0 && n < 10 ? (1<<n) : 0);
                    bank_dim_masks[c+1][1] = 0;
                    bank_dim_masks[c+1][2] = 0;
                    n -= 10;
                    bank_dim_vals[c+2][0] = (n >= 0 && n < 10 ? v : 0);
                    bank_dim_masks[c+2][0] = (n >= 0 && n < 10 ? (1<<n) : 0);
                    bank_dim_masks[c+2][1] = 0;
                    bank_dim_masks[c+2][2] = 0;
                    n -= 10;
                    bank_dim_vals[c+3][0] = (n >= 0 && n < 10 ? v : 0);
                    bank_dim_masks[c+3][0] = (n >= 0 && n < 10 ? (1<<n) : 0);
                    bank_dim_masks[c+3][1] = 0;
                    bank_dim_masks[c+3][2] = 0;
                }
            }
        } else {
            int cur_samplebuf = samplebuf;
            if( last_samplebuf != cur_samplebuf ) {
                analyse_last_samplebuf();
                last_samplebuf = cur_samplebuf;
            }
            if( BrightnessTrimpot > 63 )
                DisplayBrightness = BrightnessTrimpot+1;
            else
                DisplayBrightness = 64;
        }
        UpdateLEDStates();
     }

  return (EXIT_SUCCESS);
}

int main() {
  if( (RCON & 0x18) == 0x18 ) {
    // The WDT caused a wake from sleep
    RCONCLR = 0x18;
    asm volatile ( "eret" ); // return from interrupt
  }
  if( (RCON & 0x14) == 0x14 ) {
    // The WDT caused a wake from idle
    RCONCLR = 0x14;
    asm volatile ( "eret" ); // return from interrupt
  }
  return realmain();
}

