/* sketch for 24bit LTC2400 based meter firmware
   written by Jim Rowe (Silicon Chip)
   (adapted from a sketch by J.Beale)
   Versoin 1.3, updated 26/07/2016 at 3:00 pm

   Note: the following interface connections
    are assumed by the SPI library:
   LTC2400  <---------> Arduino Uno or compatible
   pin 5 (CSbar) <-  from   DigIO pin 10 (SS-bar)
   pin 6 (SDO)   ->  to     DigIO pin 12 (MISO)
   pin 7 (SCK)   <-  from   DigIO pin 13 (SCK)
 */
#include <SPI.h>

const float Vref = 2.50000;     // Vref = the ADC reference voltage
const byte slaveSelectPin = 10; // digital IO pin 10 for CSbar/SS-bar
const byte MISOPin = 12;
const byte SCKPin = 13;
const int S1b1 = 8;      // give D8 the label S1b1
const int S1b2 = 7;      // give D7 the label S1b2
const int S1b3 = 6;      // give D6 the label S1b3
const int S1b4 = 5;      // give D5 the label S1b4
const int S1b5 = 4;      // give D4 the label S1b5
const int S1b6 = 3;      // give D3 the label S1b6

void setup() {
  Serial.begin(115200);                // start up serial comms to PC at 115200 baud, 8N1

  pinMode(slaveSelectPin, OUTPUT);     // make D10 an output for CS-bar
  digitalWrite(slaveSelectPin, HIGH);  // and initialise to HIGH
  pinMode(MISOPin, INPUT);             // make D12 an input for MISO signal
  pinMode(SCKPin, OUTPUT);             // make D13 an output for SCK
  digitalWrite(SCKPin,LOW);            // and initialise to low (for Ext SCK)
  
  pinMode(S1b1, INPUT_PULLUP);
  pinMode(S1b2, INPUT_PULLUP);
  pinMode(S1b3, INPUT_PULLUP);
  pinMode(S1b4, INPUT_PULLUP);
  pinMode(S1b5, INPUT_PULLUP);
  pinMode(S1b6, INPUT_PULLUP);

  // next initialise SPI port
  // speed max, data sent MSB first, data mode 0 -- with 
  // SCK idle low (CPOL = 0), MISO read on rising edge (CPHI = 0)
  SPI.beginTransaction(SPISettings(2000000, MSBFIRST, SPI_MODE0));
}

 // =====================================================================
 // main loop() begins here
 // =====================================================================

void loop() {
  static long last_sample, last_sample_flags;
  static bool last_sample_valid = false;
  static int last_sample_range_switch_pos;

  if( IsSampleReady() ) {
    last_sample_range_switch_pos = GetRangeSwitchPos();
    last_sample = SpiRead();
    last_sample_valid = (last_sample_range_switch_pos == GetRangeSwitchPos());
  }

  if( Serial.available() > 0 ) {
    char incomingByte = Serial.read();   // read in command from PC
     
    if(incomingByte == 'r' ) {
      char buf[2] = "0";
      buf[0] += GetRangeSwitchPos();
      Serial.println(buf);
    }       // end of 'read the position of S1' command reply section
    
    // this section is to get a sample from the ADC, convert to 24 bits,
    // then convert to volts and send back to PC
    else if( incomingByte == 't' || incomingByte == 'b' ) {
      float volts;
      while( !last_sample_valid || last_sample_range_switch_pos != GetRangeSwitchPos() ) {
        last_sample_range_switch_pos = GetRangeSwitchPos();
        last_sample = SpiRead();
        last_sample_valid = true;
      }
      last_sample_flags = (last_sample >> 24)&3;
      last_sample &= 0x0FFFFFFF;   // mask off four leading 'header' bits
      last_sample >>= 4;           // also shift out the four lowest bits
                                   // leaving just the 24 most significant bits
      switch (last_sample_flags) { // then act on the status bits
        case 1:                    // SIG bit = 0 & EXR bit = 1, so Vin < 0 (neg)
          last_sample -= 16777214; // so subtract FFFFFEh to swing negative
          break;
        case 2:                    // SIG bit = 1 & EXR bit = 0, so 0 < Vin < Vref
          break;                   // so there's nothing to be done to inSample
        case 3:                    // SIG bit & EXR bit both high, so Vin > Vref
          last_sample += 1;        // so simply add 1 to sample
          break;
       }
      // calculate the voltage using the relationship
      // volts = Vref * sample / 16777215 (FFFFFF = 16777215)
      volts = Vref * float(last_sample) / 16777215.0;
      String vString = String(volts, 7);
      if( incomingByte == 'b' ) {
        char buf[3] = "0 ";
        buf[0] += last_sample_range_switch_pos;
        Serial.print(buf);
      }
      Serial.println(vString);    //  send the result back to the PC
      last_sample_valid = false;
    }   // end of take a sample section
  }   // end of 'respond to a command from the PC' section
}      // end of main loop

// =====================================================================
// SpiRead() function: to fetch 4 bytes from the LTC2400 via SPI interface
// =====================================================================

long SpiRead() {
  unsigned long result = 0; // result declared a 32-bit long, init value 00000000h
  unsigned long b = 0;      // likewise for b

  digitalWrite(SCKPin,LOW);      // make sure LTC2400 is set for External SCK
  digitalWrite(slaveSelectPin,LOW); // then take the SS pin low to start txfr

  asm volatile("nop");
  asm volatile("nop");

  // wait for a sample to be available
  while( digitalRead(MISOPin) )
    ;

  b = SPI.transfer(0xff);  // get B3, most significant byte
  result = b<<8;           // shift left by 8 bits, transfer to result 
  b = SPI.transfer(0xff);  // get B2, next significant byte
  result |= b;             // bitwise OR with result
  result = result<<8;      // shift left by 8 bits again
  b = SPI.transfer(0xff);  // get B1, next significant byte
  result |= b;             // bitwise OR with result
  result = result<<8;      // shift left by 8 bits one more time
  b = SPI.transfer(0xff);  // get B0, least significant byte
  result |= b;             // bitwise OR with result
                           // which should give result = [B3,B2,B1,B0]
  digitalWrite(slaveSelectPin, HIGH); // pull SS pin high again to end txfr
  return(result);          // and return with 32-bit result
}  // end of SpiRead function

int GetRangeSwitchPos() {
  if( digitalRead(S1b1) == LOW )
    return 1;
  if( digitalRead(S1b2) == LOW )
    return 2;
  if( digitalRead(S1b3) == LOW )
    return 3;
  if( digitalRead(S1b4) == LOW )
    return 4;
  if( digitalRead(S1b5) == LOW )
    return 5;
  if( digitalRead(S1b6) == LOW )
    return 6;
  return 0;
}

bool IsSampleReady() {
  bool result;

  digitalWrite(13,LOW);      // make sure LTC2400 is set for External SCK
  digitalWrite(slaveSelectPin,LOW); // then take the SS pin low to start txfr
  asm volatile("nop");
  asm volatile("nop");
  result = !digitalRead(12);
  digitalWrite(slaveSelectPin, HIGH); // pull SS pin high again to end txfr
  return result;
}
  
//    end of code

