'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
'
' Boat Computer V6
' Geoff Graham, Nov 16
'
' Requires MMBasic 5.1 or later and an ILI9341 based LCD panel with touch
' Plus a GPS module
'
' V1: Original version
' V2: Modified to allow up to 57 POIs, suppressed heading when stationary
'     and improved rendering of the heading needle
' V3: Two fixes by Jim Hiley
'     Prevent a crash caused by POI's that are a long distance away.
'     Corrected erratic needle movement.  This bug was introduced in V2.
' V4: Fixed an issue that prevented selecting the SET button for some POIs
' V5: Changed the drawing of the blue border to accomodate a bug fix in V5.2
' V6: Changed the 'speed' variable to a float to improve Km/h accuracy
'
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

Option autorun on
Option explicit
Option default integer


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Don't like the colour scheme?
' You can change all the colours here!

' colours used in the main screen
Const cBGnd = RGB(96, 96, 150)      ' background colour
Const cBorder = RGB(blue)           ' border colour
Const cSpeed = RGB(white)           ' colour used for the speed
Const cDetail = RGB(yellow)         ' colour used for heading, poi, etc

' colours used in menus
Const cTitle = RGB(green)           ' title colour
Const cEntry = RGB(yellow)          ' colour used for the entry
Const cButton = RGB(cyan)           ' the key colour
Const cSave = RGB(white)            ' the save button

' other colours used in keypad entry screens
Const cDel = RGB(magenta)           ' the delete button
Const cSpecial = RGB(248, 184, 184) ' special buttons


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' the number of POIs allowed (the actual number is one more than this number)
Const nbrpoi = 56


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' global arrays
Dim string arg$(20)
Dim string baud(5) = ("", "4800", "9600", "14400", "19200", "38400")
Dim Integer key_coord(17, 5)
Dim String key_caption(17)



'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' global variables

' variables used to track the main display
Dim Integer keydwn, redraw = 1, DetailMode, LastDetails, SpeedMode, DemoMode

' variables involved with the time (these are saved in flash)
Dim Integer TimeZone = 36000, C24Hour, DaylightSaving

' details of the POIs (these are saved in flash)
Dim String POI.Title(nbrpoi) Length 10
Dim Integer POI.Lat(nbrpoi), POI.Lon(nbrpoi)

' data extracted from the GPS module
Dim integer year, month, day, hour, min, sec, lat, lon, heading, oldheading
Dim float speed

' misc global variables
Dim integer i, j, LastDeg, GotTouch = -1
Dim string str



'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' initialise the program and discover the GPS baud rate
CLS

' initialise the POI names
For i = 0 to nbrpoi : POI.Title(i) = "POI " + Str$(i + 1) : Next i

' demo mode is turned on by touching the center on power up
If Touch(X) > 80 And Touch(X) < 240 And Touch(Y) > 60 And Touch(Y) < 180 Then
  DemoMode = 1
  Text MM.HRes/2, MM.VRes/2 - 20, "Demo", CM, 2, 1
  Text MM.HRes/2, MM.VRes/2 + 20, "Mode", CM, 2, 1
  Do While Touch(X) <> -1 : Loop
  POI.Title(0) = "HARBOUR" : POI.Lat(0) = -1121262 : POI.Lon(0) = 4401721
  POI.Title(1) = "BOAT RAMP" : POI.Lat(1) = -1131062 : POI.Lon(1) = 4410721
Else
  VAR Restore
  Text MM.HRes/2, MM.VRes/2 - 20, "Waiting", CM, 2, 1
  Text MM.HRes/2, MM.VRes/2 + 20, "for GPS", CM, 2, 1
EndIf

' detect the GPS baud rate
' this is done by switching between baud rates until clear text is received
If Not DemoMode Then
  Do
    For i = 1 To 5
      WatchDog 4000
      Open "COM1:" + baud(i) As #1
      Pause 1000
      If Instr(Input$(255, #1), "$GP") > 0 Then Exit Do
      Close #1
      WatchDog 4000
      Open "COM1:" + baud(i) + ",INV" As #1
      Pause 1000
      If Instr(Input$(255, #1), "$GP") > 0 Then Exit Do
      Close #1
    Next i
    CLS
    Text MM.HRes/2, MM.VRes/2 - 20, "GPS Module", CM, 2, 1, RGB(red)
    Text MM.HRes/2, MM.VRes/2 + 20, "Not Found", CM, 2, 1, RGB(red)
  Loop
EndIf

' this interrupt is used to detect touch on the main screen
SetPin Peek(Byte Peek(Word &H9D000090) + 23), INTL, TouchDown


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this is the main program loop, it never exits
Do
  Do
    WatchDog 5000
    If DemoMode Then
      If Timer > 1000 Then
        arg$(0) = "GPRMC"
        Timer = 0
      EndIf
    Else
      Do While Input$(254, #1) <> "" : Loop  ' clean out anything in the buffer
      GetGPSRecord                           ' get a GPS record
    EndIf
    If GotTouch > 0 And GotTouch < MM.VRes/2 Then
      ' if the top half of the screen was touched switch between knots/kph/mph
      GotTouch = -1
      SpeedMode = (SpeedMode + 1) Mod 3
      If Not DemoMode Then VAR Save SpeedMode
      Pause 150
      redraw = 1
    ElseIf GotTouch > MM.VRes/2 Then
      ' if the bottom half was touched show the details menu
      SetPin Peek(Byte Peek(Word &H9D000090) + 23), OFF   ' no interrupts in the menus
      WatchDog 600000                        ' reset after 10 minutes
      SetDetailMode                          ' show the details menu
      SetPin Peek(Byte Peek(Word &H9D000090) + 23), INTL, TouchDown    ' reinstate the interrupt
      Pause 150
      redraw = 1
      GotTouch = -1
    EndIf
  Loop Until arg$(0) = "GPRMC"               ' we only want the RMC record
  If arg$(2) = "V" Then                      ' "V" means not locked on
    CLS
    Text MM.HRes/2, MM.VRes/2 - 20, "Searching For", CM, 2, 1
    Text MM.HRes/2, MM.VRes/2 + 20, "Satellites", CM, 2, 1
    redraw = 1
    Continue do                              ' go around again and keep looking
  EndIf

  ' the GPS has the correct data
  ' first extract the elements of the data that we want from the GPS record
  If DemoMode Then
    year = 16
    month = 10
    day = 17
    hour = 0
    min = 45
    sec = 23
    lat = -1131262
    lon = 4411721
    speed = 12
    heading = 352
  Else
   year = Val(Right$(arg$(9), 2))            ' extract the date
    month = Val(Mid$(arg$(9), 3, 2))
    day = Val(Left$(arg$(9), 2))
    hour = Val(Left$(arg$(1), 2))            ' extract the time
    min = Val(Mid$(arg$(1), 3, 2))
    sec = Val(Mid$(arg$(1), 5, 2))
    lat = Val(Mid$(arg$(3), 1, 2)) * 36000 + Val(Mid$(arg$(3), 3, 7)) * 600
    If arg$(4) = "S" Then lat = -lat
    lon = Val(Mid$(arg$(5), 1, 3)) * 36000 + Val(Mid$(arg$(5), 4, 7)) * 600
    If arg$(6) = "W" Then lon = -lon
    speed = Val(arg$(7))
    heading = Val(arg$(8))
    if speed < 2 then speed = 0 : heading = -1 ' stop any dithering when stopped
  EndIf

  arg$(0) = ""                               ' show that we have the data

  ' redraw the fixed details on the main screen
  If redraw = 1 Then
    If mm.ver < 5.02 Then
      Box 0, 0, MM.HRes-1, MM.VRes-1, 4, cBorder, cBGnd
    Else
      Box 0, 0, MM.HRes, MM.VRes, 4, cBorder, cBGnd
    EndIf
    If DetailMode <> 1 And DetailMode <> 2 Then DrawCompass
    If DetailMode >= 3 Then
      Select Case SpeedMode
        Case 0
          Text 10 + 32*4, 193, "nm", LB, 1, 2, cDetail, cBGnd
        Case 1
          Text 10 + 32*4, 193, "km", LB, 1, 2, cDetail, cBGnd
        Case 2
          Text 10 + 32*4, 193, "mi", LB, 1, 2, cDetail, cBGnd
      End Select
    End If
  EndIf

  redraw = 0

  ' update the speed
  Select Case SpeedMode
    Case 0
      Text 30, 25, Str$(Cint(speed), 2), , 3, 2, cSpeed, cBGnd
      Text 180, 30, "knots", , 2, 1, cSpeed, cBGnd
    Case 1
      Text 5, 25, Str$(Cint(speed * 1.852), 3), , 3, 2, cSpeed, cBGnd
      Text 205, 30, "km/h", , 2, 1, cSpeed, cBGnd
   Case 2
      Text 5, 25, Str$(Cint(speed * 1.15078), 3), , 3, 2, cSpeed, cBGnd
      Text 220, 30, "mph", , 2, 1, cSpeed, cBGnd
  End Select

  ' update the details panel (the lower half of the screen)
  Select Case DetailMode
    Case 0      ' display the heading
      If heading >= 0 Then
        Text 32, 165, Str$(heading, 3), ,3, 1, cDetail, cBGnd
      Else
        if heading <> oldheading Then Box 32, 165, 95, 50, 0, 0, cBGnd
        Line 38, 190, 58, 190, 4, cDetail
        Line 70, 190, 90, 190, 4, cDetail
        Line 102, 190, 122, 190, 4, cDetail
      Endif
      oldheading = heading
      Text 32*4, 155, "o", , 2, 1, cDetail, cBGnd
      DrawCompassNeedle heading
    Case 1      ' display the current coordinates
      CvtLat lat, str
      Text 315, 155, str, RT, 2, 1, cDetail, cBGnd
      CvtLon lon, str
      Text 315, 200, str, RT, 2, 1, cDetail, cBGnd
    Case 2      ' display the time
      i = hour *3600 + min *60 + sec + TimeZone
      If DaylightSaving Then i = i + 3600
      If i > 86400 Then i = i - 86400
      If i < 0 Then i = i + 86400
      If Not C24Hour Then
        j = i >= 12 * 3600
        If j Then i = i - 12 * 3600
        If i\3600 = 0 Then i = i + 12 * 3600
      EndIf
      str = Str$(i\3600, 2) + ":" + Str$((i Mod 3600) \ 60, 2, 0, "0") + ":" + Str$(i Mod 60, 2, 0, "0")
      Text 32, 145, str, , 3, 1, cDetail, cBGnd
      If Not C24Hour Then
        If j Then
          Text 160, 204, "PM", CT, 2, 1, cDetail, cBGnd
        Else
          Text 160, 204, "AM", CT, 2, 1, cDetail, cBGnd
        EndIf
      EndIf
    Case 3 to nbrpoi + 3     ' display POI
      Text 10, 235, POI.Title(DetailMode - 3), LB, 2, 1, cDetail, cBGnd
      DrawCompassNeedle POIHeading(POI.Lat(DetailMode - 3), POI.Lon(DetailMode - 3))
      PrintNum 10, 140, POIDistance(POI.Lat(DetailMode - 3), POI.Lon(DetailMode - 3))
  End Select

  ' continue looping forever
Loop




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' the following two subroutines are used to draw the compass or POI
' direction graphic
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' this draws the compass rose for the heading or direction rose for the POI
Sub DrawCompass
  Local integer x = 250, y = 160, r = 45

  Circle x, y, r, 0, 1, cDetail, cDetail

  Line x - r, y, x + r, y, 2, 0
  Line x, y - r, x, y + r, 2, 0
  Line x - (Cos(Rad(30)) * r), y + (Sin(Rad(30)) * r), x + (Cos(Rad(30)) * r), y - (Sin(Rad(30)) * r), 1, 0
  Line x - (Cos(Rad(60)) * r), y + (Sin(Rad(60)) * r), x + (Cos(Rad(60)) * r), y - (Sin(Rad(60)) * r), 1, 0
  Line x - (Cos(Rad(30)) * r), y - (Sin(Rad(30)) * r), x + (Cos(Rad(30)) * r), y + (Sin(Rad(30)) * r), 1, 0
  Line x - (Cos(Rad(60)) * r), y - (Sin(Rad(60)) * r), x + (Cos(Rad(60)) * r), y + (Sin(Rad(60)) * r), 1, 0

  Circle x, y, r - 10, 0, 1, cBGnd, cBGnd
  Circle x, y, 6, 0, 1, cDetail, cDetail

  If DetailMode = 0 Then
    Text x, y - r - 1, "N", CB, 1, 2, cDetail, cBGnd
    Text x + r + 2, y, "E", LM, 1, 2, cDetail, cBGnd
    Text x, y + r + 2, "S", CT, 1, 2, cDetail, cBGnd
    Text x - r - 2, y, "W", RM, 1, 2, cDetail, cBGnd
  Else
    Text x, y - r - 1, "^", CB, 1, 2, cDetail, cBGnd
    Text x + r + 2, y, "S", LM, 1, 2, cDetail, cBGnd
    Text x - r - 2, y, "P", RM, 1, 2, cDetail, cBGnd
  EndIf

End Sub


' this draws the compass needle for the heading or direction needle for the POI
Sub DrawCompassNeedle deg As integer
  Local integer x = 250, y = 160, r = 45
  Local Integer x1, y1, x2, y2, x3, y3
  Local float i
  Local integer ldeg

  If deg <> LastDeg Then
    Circle x, y, r - 10, 0, 1, cBGnd, cBGnd
    Circle x, y, 6, 0, 1, cDetail, cDetail
    LastDeg = deg
  EndIf

  If deg < 0 Then Exit Sub
  ldeg = deg - 90
  If ldeg < 0 Then ldeg = ldeg + 360
  x1 = GetX(ldeg - 90, 4)
  y1 = GetY(ldeg - 90, 4)
  x2 = GetX(ldeg + 90, 4)
  y2 = GetY(ldeg + 90, 4)
  x3 = GetX(ldeg, r - 10)
  y3 = GetY(ldeg, r - 10)
  DrawTriangles 1, x1, y1, x2, y2, x3, y3, cDetail

End Sub


' utility to get the x coord of a vector on the compass rose
Function GetX(deg As Float, r As Float) As Float
  Local Float t = deg
  If t > 359 Then t = t - 360
  If t < 0 Then t = t + 360
  GetX = 250 + Cos(Rad(t)) * r
End Function


' utility to get the y coord of a vector on the compass rose
Function GetY(deg As Float, r As Float) As Float
  Local Float t = deg
  If t > 359 Then t = t - 360
  If t < 0 Then t = t + 360
  GetY = 160 + Sin(Rad(t)) * r
End Function


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this is the touch interrupt used when the main screen is touched
' all it does is set a flag
Sub TouchDown
  GotTouch = Touch(y)
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' subroutine to get a GPS record into the array arg$()
Sub GetGPSRecord
  Local integer i
  Local string x$
  Do
    Do While Input$(1, #1) <> "$"
      If GotTouch <> -1 Then Exit Sub      ' exit if the screen has been touched
    Loop   ' wait for the start
    For i = 0 To 20
      arg$(i) = ""                         ' clear ready for data
      Do                                   ' loops until a specific exit
        If GotTouch <> -1 Then Exit Sub    ' exit if the screen has been touched
        x$ = Input$(1, #1)                 ' get the character
        If x$ = "," Then Exit Do           ' new data item, new field
        If x$ = "*" Then Exit Sub          ' end of record, so return with it
        arg$(i) = arg$(i) + x$             ' add to the data
      Loop                                 ' keep going
    Next i                                 ' increment the field
  Loop
End Sub




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' the following two sroutines are used to calculate the bearing and
' distance to a POI
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' get the heading to a POI
' this routine uses the global variables lat and lon which are set
' to the current latitude and longitude
Function POIHeading(POIlat As integer, POIlon As integer) As float
  Local float y, x, lat1, lon1, lat2, lon2
  If heading < 0 Then POIHeading = -1 : Exit Function
  lat1 = Rad(lat / 36000)
  lon1 = Rad(lon / 36000)
  lat2 = Rad(POIlat / 36000)
  lon2 = Rad(POIlon / 36000)

  y = Sin(lon2 - lon1) * Cos(lat2)
  x = Cos(lat1) * Sin(lat2) - Sin(lat1) * Cos(lat2) * Cos(lon2 - lon1)
  POIHeading = Deg(atan2(y, x)) - heading
  If POIHeading < 0 Then POIHeading = POIHeading + 360
  If POIHeading < 0 Then POIHeading = POIHeading + 360
End Function


' get the distance to a POI
' this routine uses the global variables lat and lon which are set
' to the current latitude and longitude
Function POIDistance(POIlat As integer, POIlon As integer) As string
  Local float y, x, lat1, lat2, n, m, a
  lat1 = Rad(lat / 36000)
  lat2 = Rad(POIlat / 36000)
  n = Rad((POIlat / 36000) - (lat / 36000))
  m = Rad((POIlon / 36000) - (lon / 36000))

  a = Sin(m/2) * Sin(m/2) + Cos(lat1) * Cos(lat2) * Sin(n/2) * Sin(n/2)
  if a > 1 then a = 1 ' sanity check
  n = (2 * atan2(Sqr(a), Sqr(1 - a)))

  ' convert the distance to the current units (nautical miles, kilometers, etc)
  Select Case SpeedMode
    Case 0
      n = 3440 * n
    Case 1
      n = 6371 * n
    Case 2
      n = 3958 * n
   End Select

   ' convert the distance to a four digit string
   Select Case n
     Case >= 1000
       POIDistance = Str$(n, 1, 0)
     Case >= 100
       POIDistance = Str$(n, 4, 0)
     Case >= 10
       POIDistance = Str$(n, 1, 1)
     Case Else
       POIDistance = Str$(n, 1, 2)
   End Select
End Function



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this is the main menu displayed when the lower half of the screen is touched
Sub SetDetailMode
  Local integer i, y, b, DetailPage
  Local Float f
  Local string s

  DetailPage = DetailMode\6

  Do
    CLS

    For y = 0 To 5
      ' draw the radio buttons
      If y + DetailPage * 6 = DetailMode Then
        Circle 10, y * MM.VRes/7 + MM.VRes/14, 10, 2, 1, cSave, cEntry
      Else
        Circle 10, y * MM.VRes/7 + MM.VRes/14, 10, 2, 1, cSave, 0
      EndIf
      ' get the entry name
      Select Case y + DetailPage * 6
        Case 0
          s = "Heading"
        Case 1
          s = "Latitude/Longitude"
        Case 2
          s = "Clock"
        Case Else
          s = POI.Title((y + DetailPage * 6) - 3)
      End Select
      ' draw the title of the entry
      Text 35, y * MM.VRes/7 + MM.VRes/14, s, LM, 1, 2, cEntry
      ' draw the SET button which allows an entry to be changed
      ' if they are not applicable draw them off the screen so that they cannot be used
      If DetailPage = 0 And y < 2 Then i = 999 Else i = MM.HRes - 100
'      If y + DetailPage * 6 > 1 Then DrawButton y, 0, i, y * MM.VRes/7, 64, 26 + 6, cButton, "SET"
      DrawButton y + 2, 0, i, y * MM.VRes/7, 64, 26 + 6, cButton, "SET"
    Next y

    ' draw the prev/next buttons
    ' if they are not applicable draw them off the screen so that they cannot be used
    If DetailPage = 0 Then i = 999 else i = 0
    DrawButton 0, 0, i, MM.Vres - MM.VRes/7 + 2, 120, 26 + 4, cSpecial, "< PREV"
    If DetailPage = (nbrpoi - 2) \ 6 Then i = 999 else i = MM.Hres - 120
    DrawButton 1, 0, i, MM.Vres - MM.VRes/7 + 2, 120, 26 + 4, cSpecial, "NEXT >"

    Do
      b = CheckButtonPress(0, 7)

      ' check if the PREV/NEXT buttons have been touched
      If b = 0 Or b = 1 Then
        pause 150
        CheckButtonRelease b
        DetailPage = DetailPage - 1 + b * 2
        Exit Do
      EndIf

      If DetailPage = 0 And b = 4 Then
        ' the SET button for the time has been touched
        pause 150
        CheckButtonRelease b
        GetTimeSettings
        If Not DemoMode Then VAR Save TimeZone, C24Hour, DaylightSaving
        Do While Touch(x) <> -1 : Loop
        Exit Do
      Else If b >= 2 Then
        ' the SET button for a POI has been touched
        pause 150
        CheckButtonRelease b
        b = b - 2
        GetPOI POI.Title((b + DetailPage * 6) - 3) , POI.Lat((b + DetailPage * 6) - 3), POI.Lon((b + DetailPage * 6) - 3)
        If Not DemoMode Then VAR Save POI.Title(), POI.Lat(), POI.Lon()
        Do While Touch(x) <> -1 : Loop
        Exit Do
      Else
        ' no SET buttons have been touched, now check the main entries
        ' if a main entry has been selected the display will revert to speed & detail
        y = Touch(y)
        If y > 0 and y < (MM.VRes/7)*6 And Touch(x) < MM.HRes - 120 Then
          For i = 0 To 5
            If y > (MM.VRes/7) * i and y < (MM.VRes/7) * (i + 1) Then
              DetailMode = i + DetailPage * 6
              Circle 10, i * MM.VRes/7 + MM.VRes/14, 10, 2, 1, cSave, cEntry
            Else
              Circle 10, i * MM.VRes/7 + MM.VRes/14, 10, 2, 1, cSave, 0
            Endif
          Next i
          pause 150
          Do While Touch(y) <> -1 : Loop
          If Not DemoMode Then VAR Save DetailMode
          Exit Sub
        EndIf
      EndIf
    Loop
  Loop
End Sub



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this displays a menu for a POI and allows the details to be changed
Sub GetPOI nam As string, LatL As integer, LonL As integer
  Local integer i, y
  Local string s, lats, lons
  Do
    CvtLat LatL, lats
    CvtLon LonL, lons
    CLS
    y = 3
    Text 0, 0, "POI Title", , 1, 2, cTitle
    DrawButton 0, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    y = y + 30
    Text 0, y, nam, , 2, 1, cEntry
    y = y + 35
    Text 0, y, "Latitude", , 1, 2, cTitle
    DrawButton 1, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    y = y + 30
    Text 0, y, lats, , 2, 1, cEntry
    y = y + 35
    Text 0, y, "Longitude", , 1, 2, cTitle)
    DrawButton 2, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    y = y + 30
    Text 0, y, lons, , 2, 1, cEntry
    DrawButton 3, 0, 0, MM.VRes - 37, 192, 36, cSpecial, "SET TO HERE"
    DrawButton 4, 0, MM.HRes - 113, MM.VRes - 37, 112, 36, cSave, "SAVE"

    Do
      ' keep looping waiting for a button to be touched
      Select Case CheckButtonPress(0, 4)
        Case 0
          CheckButtonRelease 0
          GetString nam           ' get the POI's name
          Exit Do
        Case 1
          CheckButtonRelease 1
          GetCoordinate 1, lats
          i = LatL : s = lats
          GetCoords s, i          ' get the POI's latitude
          If Abs(i) > 90 * 36000 Then
            MessageBox "Invalid", "Coordinate"
          Else
            LatL = i : lats = s
          EndIf
          Exit Do
        Case 2
          CheckButtonRelease 2
          GetCoordinate 0, lons
          i = LonL : s = lons
          GetCoords s, i          ' get the POI's longitude
          If Abs(i) > 180 * 36000 Then
            MessageBox "Invalid", "Coordinate"
          Else
            LonL = i : lons = s
          EndIf
          Exit Do
        Case 3
          CheckButtonRelease 3
          LatL = lat              ' set the POI to the current coordinates
          LonL = lon
          Exit Do
        Case 4
          CheckButtonRelease 4
          Exit Sub                ' exit to the main menu
      End Select
    Loop
  Loop
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' get a POI's name
Sub GetString str As String
  Local integer i, b, y, xt, yt, SIndex
  Local string strOriginal = str
  Local String SCap(47) = ("A","B","C","D","E","F","G","H","I","J","K","L","M","N","O","P","Q","R","S","T","U","V","W","X","Y","Z","1","2","3","4","5","6","7","8","9","0",",",".","?","-",":",";","(",")","@","#","$","&")
  Const bh = MM.VRes\5, bw = MM.HRes\6

  CLS
  SIndex = 0
  Text 0, 10, str, , 2, 1, cEntry, 0

  y = (MM.VRes / 5) * 4 + 5
  DrawButton 0, 0, 0, y, 64, 36, cDel, "DEL"
  DrawButton 1, 0, 80, y, 96, 36, cButton, "SPACE"
  DrawButton 2, 0, MM.HRes - 113, y, 112, 36, cSave, "SAVE"

  For i = 0 To 11
    DrawButton i + 3, 0, bw + bw * (i Mod 4) + 2, bh + bh * (i \ 4) + 2, bw - 4, bh - 4, cButton, SCap(i)
  Next i
  DrawRArrow bw, bh
  DrawLArrow bw, bh

  Do
    If Timer > 700 Then Text 0, 10, str + "_", , 2, 1, cEntry, 0 : Timer = 0
    If Timer > 500 Then Text 0, 10, str + " ", , 2, 1, cEntry, 0
    If Touch(x) <> -1 Then
      xt = Touch(x)
      yt = Touch(y)
      If xt > bw * 5 + 10 And yt > (bh/2) * 3 And yt < (bh/2) * 7 And SIndex < 32 Then
        DrawRArrowFilled bw, bh
        SIndex = SIndex + 12
        For i = 0 To 11
          DrawButton i + 3, 0, bw + bw * (i Mod 4) + 2, bh + bh * (i \ 4) + 2, bw - 4, bh - 4, cButton, SCap(i + SIndex)
        Next i
        Do While Touch(x) <> -1 : Loop
        DrawRArrow bw, bh
        Continue do
      EndIf
      If xt < bw And yt > (bh/2) * 3 And yt < (bh/2) * 7 And SIndex > 0 Then
        DrawLArrowFilled bw, bh
        SIndex = SIndex - 12
        For i = 0 To 11
          DrawButton i + 3, 0, bw + bw * (i Mod 4) + 2, bh + bh * (i \ 4) + 2, bw - 4, bh - 4, cButton, SCap(i + SIndex)
        Next i
        Do While Touch(x) <> -1 : Loop
        DrawLArrow bw, bh
        Continue do
      EndIf
      b = CheckButtonPress(0, 14)
      Select Case b
        Case 0
          If Len(str) > 0 Then
            str = Left$(str, Len(str) - 1)
            Text 0, 10, str + "  ", , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease 0
        Case 1
          If Len(str) < 9 Then
            str = str + " "
            Text 0, 10, str, , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease 1
        Case 2
          CheckButtonRelease 2
          If str = "" Then str = strOriginal
          Exit Do
        Case 3 To 14
          If Len(str) < 9 Then
            str = str + SCap(SIndex + b - 3)
            Text 0, 10, str, , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease b
      End Select
    EndIf
  Loop
End Sub

Sub DrawRArrow bw, bh
  Box bw * 5 + 10, (bh/2) * 3, bw, bh * 2, 1, 0, 0
  Line bw * 5 + 10, (bh/2) * 3, MM.HRes - 1, (bh/2) * 5, 1, cSpecial
  Line bw * 5 + 10, (bh/2) * 3, bw * 5 + 10, (bh/2) * 7, 1, cSpecial
  Line bw * 5 + 10, (bh/2) * 7, MM.HRes - 1, (bh/2) * 5, 1, cSpecial
End Sub

Sub DrawLArrow bw, bh
  Box 0, (bh/2) * 3, bw, bh * 2, 1, 0, 0
  Line bw - 10, (bh/2) * 3, 0, (bh/2) * 5, 1, cSpecial
  Line bw - 10, (bh/2) * 3, bw - 10, (bh/2) * 7, 1, cSpecial
  Line bw - 10, (bh/2) * 7, 0, (bh/2) * 5, 1, cSpecial
End Sub

Sub DrawRArrowFilled bw, bh
  Local integer x, y
  For x = bw * 5 + 10 To MM.HRes - 1
    y = bh * ((x - (bw * 5 + 10)) / ((MM.HRes - 1) - (bw * 5 + 10)))
    Line x, (bh/2) * 3 + y, x, (bh/2) * 7 - y, 1, cSpecial
  Next x
End Sub

Sub DrawLArrowFilled bw, bh
  Local integer x, y
  For x = 0 To bw - 10
    y = bh * (x / (bw - 10))
    Line x, bh/2 * 5 - y, x, bh/2 * 5 + y, 1, cSpecial
  Next x
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' get a POI's coordinate
' this can handle either latitude (lat = 1) or longitude (lat = 0)
Sub GetCoordinate lat As integer, str As string
  Local i, b
  Local String SCap(15) = ("7","8","9","E", "`","4","5","6","W","'","1","2","3",".",Chr$(34),"0")
  Local string strOriginal = str
  Const bh = MM.VRes\5, bw = MM.HRes\5

  Do While Left$(str, 1) = " " : str = Mid$(str, 2, 99) : Loop
  If lat Then SCap(3) = "N" : SCap(8) = "S"
  CLS
  Text 0, 10, str, , 2, 1, cEntry, 0

  For i = 0 To 14
    DrawButton i, 0, bw * (i Mod 5) + 2, bh + bh * (i \ 5) + 2, bw - 4, bh - 4, cButton, SCap(i)
  Next i

  DrawButton 15, 0, bw + 2, bh*4 + 2, bw - 4, bh - 4, cButton, "0"
  DrawButton 16, 0, bw*2 + 2, bh*4 + 2, bw - 4, bh - 4, cDel, "DEL"
  DrawButton 17, 0, bw*3 + 2, bh*4 + 2, bw * 2 - 4, bh - 4, cSave, "SAVE"

  Do
    If Timer > 700 Then Text 0, 10, str + "_", , 2, 1, cEntry, 0 : Timer = 0
    If Timer > 500 Then Text 0, 10, str + " ", , 2, 1, cEntry, 0
      b = CheckButtonPress(0, 17)
      Select Case b
        Case 0 To 15
          If Len(str) < 13 Then
            str = str + SCap(b)
            Text 0, 10, str, , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease b
        Case 16
          If Len(str) > 0 Then
            str = Left$(str, Len(str) - 1)
            Text 0, 10, str + "  ", , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease b
        Case 17
          CheckButtonRelease b
          If str = "" Then str = strOriginal
          Exit Do
      End Select
    EndIf
  Loop
End Sub




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' routines to convert a coordinate from a string to an integer and vice versa
' integers are the coordinate in tens of seconds
' strings can be ddd.dddd or ddd mm.mmmmm or ddd mm ss.s format followed by
' N, S, E or W.
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''


' convert an integer latitude to a string
Sub CvtLat lat As integer, lats As string
  Local Integer i
  i = Abs(lat)
  lats = Str$(i \ 36000, 2) + "`" + Str$((i Mod 36000) \ 600, 2, 0, "0") + "'" + Str$((i Mod 600) / 10, 2, 1, "0") + Chr$(34)
  If lat < 0 Then lats = lats + "S" Else lats = lats + "N"
End Sub


' convert an integer longitude to a string
Sub CvtLon lon As integer, lons As string
  Local Integer i
  i = Abs(lon)
  lons = Str$(i \ 36000, 2) + "`" + Str$((i Mod 36000) \ 600, 2, 0, "0") + "'" + Str$((i Mod 600) / 10, 2, 1, "0") + Chr$(34)
  If lon < 0 Then lons = lons + "W" Else lons = lons + "E"
End Sub


' convert a string coordinate to an integer
' this can handle either latitude or longitude
Sub GetCoords s As string, coord As integer
  Local float neg, deg, min, sec, degp, minp, secp
  Local integer i
  Local string t

  If UCase$(Left$(s, 1)) = "S" Then neg = 1 : s = Mid$(s, 2, 99)
  If UCase$(Right$(s, 1)) = "S" Then neg = 1 : s = Left$(s, Len(s) - 1)
  If UCase$(Left$(s, 1)) = "N" Then neg = 0 : s = Mid$(s, 2, 99)
  If UCase$(Right$(s, 1)) = "N" Then neg = 0 : s = Left$(s, Len(s) - 1)
  If UCase$(Left$(s, 1)) = "W" Then neg = 1 : s = Mid$(s, 2, 99)
  If UCase$(Right$(s, 1)) = "W" Then neg = 1 : s = Left$(s, Len(s) - 1)
  If UCase$(Left$(s, 1)) = "E" Then neg = 0 : s = Mid$(s, 2, 99)
  If UCase$(Right$(s, 1)) = "E" Then neg = 0 : s = Left$(s, Len(s) - 1)

  ' check the string for duplicate or invalid chars
  degp = Len(s) + 1 : minp = Len(s) + 1 : secp = Len(s) + 1 ' default str len
  For i = 1 To Len(s)
    Select Case Mid$(s, i, 1)
      Case "`"
        deg = deg + 1
        degp = i
      Case "'"
        min = min + 1
        minp = i
        If deg = 0 Then deg = 2
      Case Chr$(34)
        sec = sec + 1
        secp = i
        If deg = 0 Or min = 0 Then deg = 2
      Case ".", "0" To "9"
        ' do nothing
      Case Else
        deg = 2
    End Select
  Next i
  If s = "" Or deg > 1 Or min > 1 Or sec > 1 Then
    MessageBox "Invalid", "Coordinate"
    Exit Sub
  EndIf

  deg = Val(Mid$(s, 1, degp - 1))
  min = Val(Mid$(s, degp + 1, minp - 1))
  sec = Val(Mid$(s, minp + 1, secp - 1))
  coord = Cint(deg * 36000) + Cint(min * 600) + Cint(sec * 10)
  If neg Then coord = -coord
End Sub




''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this displays a menu for changing the time settings
Sub GetTimeSettings
  Local integer y
  Local string s
  Local float f
  Do
    CLS
    y = 3
    Text 0, 0, "Time Zone", , 1, 2, cTitle
    y = y + 30
    DrawButton 0, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    Text 0, y, Str$(TimeZone/3600, -1, 1), , 2, 1, cEntry
    y = y + 35
    Text 0, y, "Format", , 1, 2, cTitle
    y = y + 30
    DrawButton 1, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    If C24Hour Then s = "24 Hour" Else s = "12 Hour"
    Text 0, y, s, , 2, 1, cEntry
    y = y + 35
    Text 0, y, "Daylight Saving", , 1, 2, cTitle
    y = y + 30
    DrawButton 2, 0, MM.HRes - 116, y - 3, 116, 28, cButton, "CHANGE"
    If DaylightSaving Then s = "ON " Else s = "OFF"
    Text 0, y, s, , 2, 1, cEntry
    DrawButton 3, 0, MM.HRes - 113, MM.VRes - 37, 112, 36, cSave, "SAVE"

    Do
      Select Case CheckButtonPress(0, 3)
        Case 0
          CheckButtonRelease 0
          s = Str$(TimeZone/3600, -1, 1)
          GetTimeZone s
          f = Cint(Val(s) * 2) / 2
          If f > 12 Or f < -12 Then
            MessageBox "Invalid", "Time Zone"
          Else
            TimeZone = f * 3600
          EndIf
          Exit Do
        Case 1
          CheckButtonRelease 1
          C24Hour = Not C24Hour
          If C24Hour Then s = "24 Hour" Else s = "12 Hour"
          Text 0, 98, s, , 2, 1, cEntry
        Case 2
          CheckButtonRelease 2
          DaylightSaving = Not DaylightSaving
          If DaylightSaving Then s = "ON " Else s = "OFF"
          Text 0, 163, s, , 2, 1, cEntry
        Case 3
          CheckButtonRelease 3
          Exit Sub
      End Select
    Loop
  Loop
End Sub


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this displays a keypad for changing the time zone
Sub GetTimeZone str As string
  Local i, b
  Local String SCap(11) = ("7","8","9","-","4","5","6",".","1","2","3","0")
  Local string strOriginal = str
  Const bh = MM.VRes\5, bw = MM.HRes\4

  CLS
  For i = 0 To 11
    DrawButton i, 0, bw * (i Mod 4) + 2, bh + bh * (i \ 4) + 2, bw - 4, bh - 4, cButton, SCap(i)
  Next i

  DrawButton 12, 0, 0, bh*4 + 10, 16*4, bh - 12, cDel, "DEL"
  DrawButton 13, 0, MM.HRes - 113, bh*4 + 10, 112, bh - 12, cSave, "SAVE"
  If C24Hour Then DrawButton 12, 3      ' if 24 hour toggle button on

  Do
    If Timer > 700 Then Text bw, 10, str + "_", , 2, 1, cEntry, 0 : Timer = 0
    If Timer > 500 Then Text bw, 10, str + " ", , 2, 1, cEntry, 0
      b = CheckButtonPress(0, 13)
      Select Case b
        Case 0 To 11
          If Len(str) < 5 Then
            If b = 3 Then
              If str = "" Then str = "-"
            Else
              str = str + SCap(b)
            EndIf
            Text bw, 10, str, , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease b
        Case 12
          If Len(str) > 0 Then
            str = Left$(str, Len(str) - 1)
            Text bw, 10, str + "  ", , 2, 1, cEntry, 0
          EndIf
          CheckButtonRelease b
        Case 13
          CheckButtonRelease b
          If str = "" Then str = strOriginal
          Exit Do
      End Select
    EndIf
  Loop
End Sub





'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Draw buttons and get button presses
'
' The subrouting DrawButton will draw a button (normally used when drawing
' the screen for input).
'
' The function CheckButtonPress() will check if a button has been touched.
' If it has it will set it to selected (reverse video) and return with the
' button's number.
'
' The subroutine CheckButtonRelease will wait for the touch to be released
' and will then draw the button as normal.
'
' These routines use the global arrays key_coord() and key_caption() to
' track the coordinates and size of each button and save its caption.
'
' IMPORTANT: These routines set the watchdog to 10 minutes.  If a button
'            has not been pressed within this time the Micromite will
'            restart.
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' draw a button
Sub DrawButton n As Integer, mode As Integer, x As integer, y As integer, w As integer, h As integer, c As integer, s As string
  Local integer bc, fc

  If mode = 0 Then
    key_coord(n,0) = x : key_coord(n,1) = y : key_coord(n,2) = w : key_coord(n,3) = h
    key_coord(n,4) = c : key_caption(n) = s
  EndIf

  If mode > 1 Then
    bc = key_coord(n,4) : fc = 0    ' draw in reverse video if it is being touched
  Else
    bc = 0 : fc = key_coord(n,4)    ' a normal (untouched) button
  EndIf

  If key_coord(n,0) < MM.Hres Then
    RBox key_coord(n,0), key_coord(n,1), key_coord(n,2), key_coord(n,3), , key_coord(n,4), bc)
    If key_caption(n) = "`" Then
      Text key_coord(n,0) + key_coord(n,2)/2, key_coord(n,1) + key_coord(n,3)/4, "o", CM, 1, 1, fc, bc
    Else
      Text key_coord(n,0) + key_coord(n,2)/2, key_coord(n,1) + key_coord(n,3)/2, key_caption(n), CM, 1, 2, fc, bc
    EndIf
  EndIf
End Sub


' check if a button has been touch and animate the button's image
' returns the button's number
Function CheckButtonPress(startn As Integer, endn As Integer) As Integer
  Local Integer xt, yellowt, n

  CheckButtonPress = -1
  If Touch(x) <> -1 Then
    ' we have a touch
    WatchDog 600000                        ' reset after 10 minutes
    xt = Touch(x)
    yellowt = Touch(y)
    ' scan the array key_coord() to see if the touch was within the
    ' boundaries of a button
    For n = startn To endn
      If xt > key_coord(n,0) And xt < key_coord(n,0) + key_coord(n,2) And yellowt > key_coord(n,1) And yellowt < key_coord(n,1) + key_coord(n,3) Then
        ' we have a button press
        ' draw the button as pressed
        DrawButton n, 2
        CheckButtonPress = n
        Exit For
      EndIf
    Next n
  EndIf
End Function


' wait for the touch to be released and then draw the button as normal
Sub CheckButtonRelease n As integer
  ' if a button is currently down check if it has been released
  Do While Touch(x) <> -1 : Loop   ' wait for the button to be released
  DrawButton n, 1                  ' draw the button as normal (ie, not pressed)
End Sub



'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' this handy routine draws a message box with an OK button
' then waits for the button to be touched
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Sub MessageBox s1 As String, s2 As String
  Local integer w
  If Len(s1) > Len(s2) Then w = Len(s1) Else w = Len(s2)
  w = w * 8     ' get the width of the text (used for the box width)

  ' draw the box and the message in it
  RBox MM.HRes/2 - w - 20, 60, w * 2 + 40, 130, , RGB(yellow), 0
  Text MM.HRes/2, 70, s1, CT, 1, 2, RGB(white)
  Text MM.HRes/2, 100, s2, CT, 1, 2, RGB(white)

  ' draw the OK button
  RBox 110, 140, 100, 34, , cButton
  Text MM.HRes/2, 157, "OK", CM, 1, 2, cButton

  ' wait for the button to be touched
  Do While Not (Touch(x) > 110 And Touch(x) < 210 And Touch(y) > 140 And Touch(y) < 180) : Loop

  ' draw the OK button as depressed
  RBox 110, 140, 100, 34, , cButton, cButton
  Text MM.HRes/2, 157, "OK", CM, 1, 2, 0, cButton

  ' wait for the touch to be removed
  Do While Touch(x) <> -1 : Loop
End Sub


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' implementation of the atan2() function which is not natively in MMBasic
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Function atan2(y As float, x As float) As float
  If x > 0 Then atan2 = Atn(y / x)
  If x < 0 And y >= 0 Then atan2 = Atn(y  /x) + Pi
  If x = 0 And y > 0  Then atan2 = Pi / 2
  If x < 0 And y < 0  Then atan2 = Atn(y / x) - Pi
  If x = 0 And y < 0  Then atan2 = -Pi / 2
End Function


'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' used to print distance on the main screen
' the decimal point is drawn using a small circle
'''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
Sub PrintNum x As integer, y As integer, s As string
  Local i As integer
  i = Instr(s, ".")
  If i <> LastDetails Then redraw = 1     ' redraw if the nbr of digits changed
  LastDetails = i
  Text x, y, s, , 3, 1, cDetail, cBGnd
  If i > 0 Then
    x = x + (32 * i - 16)
    y = y + 45
    Circle x, y, 4, 0, 1, 0, cDetail
  EndIf
End Sub



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Routine to draw multiple triangles,
' written by Peter Mather (matherp on the Back Shed Forum)
' can be used to draw an existing triangle in the background colour and then re-write it somewhere else
'
' Parameters are:
'   number of triangles to draw
'   array x-coordinates of first points
'   array y-coordinates of first points
'   array x-coordinates of second points
'   array y-coordinates of second points
'   array x-coordinates of third points
'   array y-coordinates of third points
'   array colours of triangles
'
CSub DrawTriangles
     00000000
     27bdff90 afbf006c afbe0068 afb70064 afb60060 afb5005c afb40058 afb30054
     afb20050 afb1004c afb00048 afa40070 afa50074 afa60078 afa7007c 8c820004
     5c400007 afa00024 144000cc 8fbf006c 8c820000 104000ca 8fbe0068 afa00024
     afa0002c 3c159d00 8fa30074 8fa40024 00641021 8c510000 8fa50078 00a41021
     8c5e0000 8fa3007c 00641021 8c570000 8fa50080 00a41021 8c420000 afa20018
     8fa30084 00641021 8c420000 afa20028 8fa50088 00a41021 8c560000 8fa3008c
     00641021 8c540000 8fa40018 009e102a 10400008 8fa50018 03c01021 0080f021
     afa20018 02201021 02e08821 0040b821 8fa50018 02c5102a 10400007 8fa30018
     afb60018 00a0b021 02e01021 8fb70028 afa20028 8fa30018 007e102a 10400006
     03c01021 0060f021 afa20018 02201021 02e08821 0040b821 17d6001b 8fa50018
     02f1102a 14400006 02203821 0237102a 10400005 8fa40028 01000002 02e03821
     02e08821 8fa40028 0091102a 54400003 8fb10028 00e4102a 0082380b 8ea20048
     00fe3821 afb40010 8c420000 02202021 03c02821 02203021 0040f809 00f13823
     01000063 8fa4002c 00b61026 0002102b 00a21023 afa2001c 005e102a 1440002f
     03c08021 02f11023 afa20020 8fa30028 00711823 afa30030 00009021 00009821
     00be2023 afa40034 02def823 afbe0038 afb7003c afb60040 0060f021 0080b821
     03e0b021 0277001a 02e001f4 00002012 00912021 0256001a 02c001f4 00003012
     00d13021 00c4102a 50400005 8ea20048 00801021 00c02021 00403021 8ea20048
     afb40010 8c420000 02002821 0040f809 02003821 26100001 8fa20020 02629821
     8fa3001c 0070102a 1040ffe6 025e9021 8fbe0038 8fb7003c 8fb60040 02d0102a
     1440002a 8fa40028 00972023 afa4001c 8fa50018 02059023 72449002 8fa20028
     00511023 afa20020 021e9823 70539802 02c51023 afa20018 02def023 8fa30018
     0243001a 006001f4 00002012 00972021 027e001a 03c001f4 00003012 00d13021
     00c4102a 50400005 8ea20048 00801021 00c02021 00403021 8ea20048 afb40010
     8c420000 02002821 0040f809 02003821 26100001 8fa2001c 02429021 8fa30020
     02d0102a 1040ffe5 02639821 8fa4002c 24840001 afa4002c 8fa50024 24a50008
     afa50024 000417c3 8fa50070 8ca30004 0043182a 1460ff45 8fa30074 8ca30004
     14620006 8fbf006c 8ca20000 0082202b 1480ff3e 8fa30074 8fbf006c 8fbe0068
     8fb70064 8fb60060 8fb5005c 8fb40058 8fb30054 8fb20050 8fb1004c 8fb00048
     03e00008 27bd0070
End CSub
