
/*
 * PIC16LF1709 pin assignments:
 *
 *  1 VDD  +2.7V
 *  2 RA5  DISCHG3 (active high)
 *  3 RA4  DISCHG1 (active high)
 *  4 MCLR RESET
 *  5 RC5  DISCHG4 (active high)
 *  6 RC4  DISCHG0 (active high)
 *  7 RC3  DISCHG2 (active high)
 *  8 RC6  DISCHG5 (active high)
 *  9 RC7  DISCHG6 (active high)
 * 10 RB7  DISCHG7 (active high)
 * 11 RB6  Enable supply for IC8/charge pump
 * 12 AN11 Cell voltage / 2
 * 13 RB4  Cell select address 0
 * 14 RC2  Cell select address 1
 * 15 RC1  Cell select EN1
 * 16 RC0  Cell select EN0
 * 17 AN2  Balance enable (1V threshold)
 * 18 RA1  LED2 (active high) / PGC
 * 19 RA0  LED1 (active high) / PGD
 * 20 VSS  0V
 */

#include <xc.h> //Include XC8 compiler header file

// PIC16LF1709 Configuration Bit Settings

// CONFIG1
#pragma config FOSC = INTOSC    // Oscillator Selection Bits (INTOSC oscillator: I/O function on CLKIN pin)
#pragma config WDTE = SWDTEN    // Watchdog Timer Disable (software controlled)
#pragma config PWRTE = OFF      // Power-up Timer Enable (PWRT disabled)
#pragma config MCLRE = ON       // MCLR Pin Function Select (MCLR/VPP pin function is MCLR)
#pragma config CP = OFF         // Flash Program Memory Code Protection (Program memory code protection is disabled)
#pragma config BOREN = NSLEEP   // Brown-out Reset Enable (Brown-out Reset disabled in sleep mode, otherwise active)
#pragma config CLKOUTEN = OFF   // Clock Out Enable (CLKOUT function is disabled. I/O or oscillator function on the CLKOUT pin)
#pragma config IESO = OFF       // Internal/External Switchover Mode (Internal/External Switchover Mode is disabled)
#pragma config FCMEN = OFF      // Fail-Safe Clock Monitor Enable (Fail-Safe Clock Monitor is disabled)

// CONFIG2
#pragma config WRT = OFF        // Flash Memory Self-Write Protection (Write protection off)
#pragma config PPS1WAY = ON     // Peripheral Pin Select one-way control (The PPSLOCK bit cannot be cleared once it is set by software)
#pragma config ZCDDIS = ON      // Zero-cross detect disable (Zero-cross detect circuit is disabled at POR)
#pragma config PLLEN = OFF      // Phase Lock Loop disable
#pragma config STVREN = ON      // Stack Overflow/Underflow Reset Enable (Stack Overflow or Underflow will cause a Reset)
#pragma config BORV = LO        // Brown-out Reset Voltage Selection (Brown-out Reset Voltage (Vbor), low trip point selected.)
#pragma config LPBOR = ON       // Low-Power Brown Out Reset (Low-Power BOR is enabled)
#pragma config LVP = OFF        // Low-Voltage Programming Disable (High-voltage on MCLR/VPP must be used for programming)

#define _XTAL_FREQ 4000000
#define LED_RED    0
#define LED_GREEN  1

#pragma intrinsic(_delay)
extern __nonreentrant void _delay(unsigned long);

#define __delay_us(x) _delay((unsigned long)((x)*(_XTAL_FREQ/4000000.0)))
#define __delay_ms(x) _delay((unsigned long)((x)*(_XTAL_FREQ/4000.0)))

//END OF PIC SETTINGS

static void InitIOs() {
    TRISAbits.TRISA5 = 0;
    LATAbits.LATA5 = 0;

    TRISAbits.TRISA4 = 0;
    LATAbits.LATA4 = 0;
    ANSELAbits.ANSA4 = 0;

    TRISCbits.TRISC5 = 0;
    LATCbits.LATC5 = 0;

    TRISCbits.TRISC4 = 0;
    LATCbits.LATC4 = 0;

    TRISCbits.TRISC3 = 0;
    LATCbits.LATC3 = 0;
    ANSELCbits.ANSC3 = 0;

    TRISCbits.TRISC6 = 0;
    LATCbits.LATC6 = 0;
    ANSELCbits.ANSC6 = 0;

    TRISCbits.TRISC7 = 0;
    LATCbits.LATC7 = 0;
    ANSELCbits.ANSC7 = 0;

    TRISBbits.TRISB7 = 0;
    LATBbits.LATB7 = 0;

    TRISBbits.TRISB6 = 0;
    LATBbits.LATB6 = 0;

    TRISBbits.TRISB5 = 1;
    ANSELBbits.ANSB5 = 1;

    TRISBbits.TRISB4 = 0;
    LATBbits.LATB4 = 0;
    ANSELBbits.ANSB4 = 0;

    TRISCbits.TRISC2 = 0;
    LATCbits.LATC2 = 0;
    ANSELCbits.ANSC2 = 0;

    TRISCbits.TRISC1 = 0;
    LATCbits.LATC1 = 0;
    ANSELCbits.ANSC1 = 0;

    TRISCbits.TRISC0 = 0;
    LATCbits.LATC0 = 0;
    ANSELCbits.ANSC0 = 0;

    TRISAbits.TRISA2 = 1;
    ANSELAbits.ANSA2 = 1;

    TRISAbits.TRISA1 = 0;
    LATAbits.LATA1 = 0;
    ANSELAbits.ANSA1 = 0;

    TRISAbits.TRISA0 = 0;
    LATAbits.LATA0 = 0;
    ANSELAbits.ANSA0 = 0;
}

static void discharge_cell(signed char cell) {
    LATBbits.LATB7 = (cell == 0);
    LATCbits.LATC7 = (cell == 1);
    LATCbits.LATC6 = (cell == 2);
    LATCbits.LATC5 = (cell == 3);
    LATAbits.LATA5 = (cell == 4);
    LATCbits.LATC3 = (cell == 5);
    LATAbits.LATA4 = (cell == 6);
    LATCbits.LATC4 = (cell == 7);
}

static void select_cell_to_sample(signed char cell) {
    LATC &= ~3;
//    LATCbits.LATC0 = 0;
//    LATCbits.LATC1 = 0;
    LATBbits.LATB4 = !(cell&1);
    LATCbits.LATC2 = !(cell&2);
    if( cell >= 0 && cell < 8 )
        LATC |= 2-(cell>>2);
//    LATCbits.LATC0 = (cell >= 4 && cell < 8);
//    LATCbits.LATC1 = (cell >= 0 && cell < 4);
}

static void Sleep1s() {
    discharge_cell(-1);
    select_cell_to_sample(-1);
    asm("CLRWDT");
    WDTCONbits.WDTPS = 10; // 1 second interval
    WDTCONbits.SWDTEN = 1;
    SLEEP();
    WDTCONbits.SWDTEN = 0;
}

static void Sleep8s() {
    discharge_cell(-1);
    select_cell_to_sample(-1);
    asm("CLRWDT");
    WDTCONbits.WDTPS = 13; // 8 second interval
    WDTCONbits.SWDTEN = 1;
    SLEEP();
    WDTCONbits.SWDTEN = 0;
}

static void inamp_poweron(unsigned char on) {
    LATBbits.LATB6 = on;
}

#define PUMP_NUM_1 10
#define PUMP_NUM_2 2
static void pump_supply(int num) {
    while( num-- ) {
        inamp_poweron(0);
        __delay_us(75);
        inamp_poweron(1);
        if( num )
            __delay_us(100);
    }
}

static unsigned short read_adc(unsigned char channel) {
    unsigned short ret;
    ADCON0bits.CHS = channel;
    ADCON1bits.ADPREF = 0; // Vdd
    ADCON1bits.ADCS = 4;// Fosc/4      - 1; // Fosc/8
    ADCON1bits.ADFM = 1; // right justified
    ADCON0bits.ADON = 1;
    __delay_us(4);
    ADCON0bits.ADGO = 1;
    while( ADCON0bits.ADGO )
        ;
    ret = ADRES;
    ADCON0bits.ADON = 0;
    return ret;
}

#if 0
static unsigned char determine_num_cells() {
    unsigned char c;
    inamp_poweron(1);
    __delay_ms(25);
    for( c = 0; c < 8; ++c ) {
        pump_supply(PUMP_NUM_1);
        select_cell_to_sample(c);
        __delay_us(100);
        pump_supply(PUMP_NUM_2);
        if( read_adc(11) < 205 ) // 1024 = 5V, so 205 = ~1V
            break;
    }
    select_cell_to_sample(-1);
    inamp_poweron(0);
    return c;
}
#endif

static void led_on_off(unsigned char led, unsigned char on) {
    if( led )
        LATAbits.LATA1 = on;
    else
        LATAbits.LATA0 = on;
}

char num_cells;
unsigned short cell_voltages[8];
static void read_cell_voltages(unsigned char discharge_during_reading) {
    signed char c;
    inamp_poweron(1);
    __delay_ms(10);
    select_cell_to_sample(0);
    __delay_ms(10);
    pump_supply(20);
    select_cell_to_sample(num_cells-1);
    pump_supply(10);
    if( discharge_during_reading )
        discharge_cell(num_cells-1);
    __delay_us(500);
    for( c = num_cells-1; c >= 0; --c ) {
        cell_voltages[c] = read_adc(11);
        select_cell_to_sample(c-1);
        if( c != 0 ) {
            if( discharge_during_reading )
                discharge_cell(c-1);
            __delay_us(500);
        }
    }
    if( discharge_during_reading )
        discharge_cell(-1);
    inamp_poweron(0);
}

static signed char select_cell_to_discharge() {
    static signed char last_discharge = -1;
    unsigned short min = 1024, max = 0;
    unsigned char i, start;
    for( i = 0; i < num_cells; ++i ) {
        if( cell_voltages[i] < min )
            min = cell_voltages[i];
        if( cell_voltages[i] > max )
            max = cell_voltages[i];
    }
    if( max != min && max != min+1 ) {
        i = last_discharge + 1;
        while( i >= num_cells )
            i -= num_cells;
        while(1) {
            if( cell_voltages[i] == max ) {
                last_discharge = i;
                return i;
            }
            if( ++i == num_cells )
                i = 0;
        }
    }
    return -1;
}

static signed char determine_num_cells() {
    signed char top, cell;
    for( top = 7; top >= 0; --top ) {
        if( cell_voltages[top] >= 170 )
            break;
    }
    if( top < 1 )
        return -1;
    for( cell = 0; cell <= top; ++cell ) {
        if( cell_voltages[cell] < 170 || cell_voltages[cell] < cell_voltages[top] * 2 / 3 )
            return -1;
    }
    return top+1;
}

static unsigned char faulty_cell_present() {
    unsigned char cell;
    for( cell = 0; cell < num_cells; ++cell )
        if( cell_voltages[cell] < 170 )
            return 1;
    return 0;
}

static void error_flash() {
    unsigned char cell;
    for( cell = 0; cell < 10; ++cell ) {
        led_on_off(LED_RED, 1);
        __delay_ms(100);
        led_on_off(LED_RED, 0);
        __delay_ms(100);
    }
}

void main(void) {
    signed char cell = 0;//, counter = 0;
    unsigned char was_idle = 0, was_balancing = 0, did_anything = 0;
    InitIOs();

    OSCCON=0x6A; //Internal 4MHz oscillator, PLL disabled

    __delay_ms(500);

    while(1) {
        num_cells = 8;
        read_cell_voltages(0/*1*/);
        cell = determine_num_cells();
        if( cell == -1 ) {
            error_flash();
            __delay_ms(250);
        } else {
            num_cells = cell;
            for( cell = 0; cell < num_cells; ++cell ) {
                led_on_off(LED_GREEN, 1);
                __delay_ms(250);
                led_on_off(LED_GREEN, 0);
                __delay_ms(500);
            }
            __delay_ms(500);
            break;
        }
    }

    while(1) {
        unsigned short AN2_ADCval = read_adc(2);

        if( AN2_ADCval < 297 ) {
            if( was_balancing ) {
                led_on_off(LED_RED, 1);
                __delay_ms(1000);
                led_on_off(LED_RED, 0);

                was_balancing = 0;
            }
            Sleep8s();
            was_idle = 1;
        } else {
            if( was_idle ) {
                led_on_off(LED_RED, 1);
                led_on_off(LED_GREEN, 1);
                __delay_ms(250);
                led_on_off(LED_RED, 0);
                led_on_off(LED_GREEN, 0);
                __delay_ms(250);

                was_idle = 0;
            }

            discharge_cell(-1);
            read_cell_voltages(0);
            if( faulty_cell_present() ) {
                error_flash();
            } else {
                cell = select_cell_to_discharge();
                if( cell == -1 && did_anything ) {
                    __delay_ms(250); // let cell voltage rebound a bit and check again
                    cell = select_cell_to_discharge();
                }
                if( cell >= 0 ) {
                    unsigned char i;

                    discharge_cell(cell);

                    for( i = 0; i <= cell; ++i ) {
                        led_on_off(LED_GREEN, 1);
                        __delay_ms(100);
                        led_on_off(LED_GREEN, 0);
                        __delay_ms(100);
                    }
                    for( ; i < 10; ++i )
                        __delay_ms(200);
                    discharge_cell(-1);

                    did_anything = 1;
                    was_balancing = 1;
                } else {
                    led_on_off(LED_GREEN, 1);
                    if( did_anything )
                        __delay_ms(1000);
                    else
                        __delay_ms(5);
                    led_on_off(LED_GREEN, 0);
                    Sleep1s();

                    if( !did_anything )
                        Sleep1s();

                    was_balancing = 0;
                    did_anything = 0;
                }
            }
        }
    }
}
