
/*
  Pin assignments:
  
   1 (PD3/INT1)     - LED Cathode (Green)
   2 (PD4/XCK/T0)   - LED Cathode (Red)
   7 (PB6/XTAL1)    - PCM1972M CLOCK In
   8 (PB7/XTAL2)    - PCM1972M LOAD/SHIFT In
   9 (PD5/T1)       - LED Cathode (Blue)
  10 (PD6/AIN0)     - 
  11 (PD7/AIN1)     - 
  12 (PB0/ICP)      - PCM1972M DATA_IN
  13 (PB1/OC1A)     -
  14 (PB2/SS/OC1B)  - 
  15 (PB3/MOSI/OC2) - Program (MOSI)
  16 (PB4/MISO)     - Program (MISO)
  17 (PB5/SCK)      - Program (SCK)
  19 (ADC6)         -
  20 (AREF)         - 100nF Capacitor to AGND
  22 (ADC7)         - 
  23 (PC0/ADC0)     - Relay 1   Set coil (Active High)
  24 (PC1/ADC1)     - Relay 1 Reset coil (Active High)
  25 (PC2/ADC2)     - Relay 2   Set coil (Active High)
  26 (PC3/ADC3)     - Relay 2 Reset coil (Active High)
  27 (PC4/ADC4/SDA) - Relay 3   Set coil (Active High)
  28 (PC5/ADC5/SCL) - Relay 3 Reset coil (Active High)
  29 (PC6/RESET)    - Program (RESET)
  30 (PD0/RXD)      - 
  31 (PD1/TXD)      - 
  32 (PD2/INT0)     - IR in (inverted)

  Fuses: 
*/

#define F_CPU 1000000UL
#include <avr/io.h>
#include <util/delay.h>
#include <avr/interrupt.h>
#include <string.h>
#include <avr/eeprom.h>

static void delay_ms(unsigned short ms) {
  while(ms) {
    _delay_ms(1);
    --ms;
  }
}

#define IR_IN_VECT         INT0_vect
#define IR_READ()          (PIND&_BV(PD2))
#define IR_LED_ON()        PORTD &= ~_BV(PD3)
#define IR_LED_OFF()       PORTD |= _BV(PD5)

///////////////////////////////////////////////////////////////////////
// IR

volatile unsigned short rc5_code, rc5_final_code, rc5_timing;
volatile unsigned char rc5_bit, rc5_reset_timer;
static const unsigned char g_ir_leds = 1;
unsigned char led_save;

void cancel_rc5_reception() {
  EICRA = _BV(ISC01)|_BV(ISC00); // set INT0 to rising edge
  TIMSK1 &= ~_BV(TOIE1);
  rc5_bit = 0;
  if( g_ir_leds ) {
    IR_LED_OFF();
  }
}

union {
  unsigned long l;
  unsigned short s;
  unsigned char c;
} timer;

ISR(TIMER0_COMPA_vect) {
  ++timer.l;
}

ISR(TIMER1_OVF_vect) {
  if( rc5_bit == 1 || rc5_reset_timer ) {
    cancel_rc5_reception();
  } else {
    unsigned char level = IR_READ();
    rc5_code <<= 1;
    rc5_code |= !level;
    if( ++rc5_bit == 15 ) {
      rc5_final_code = (rc5_code&0x7F00)|((rc5_code>>1)&0x00FF)|0x8000;
      cancel_rc5_reception();
      if( g_ir_leds ) {
        IR_LED_OFF();
      }
    } else {
      TCNT1 = 0 - (rc5_timing>>1);
      rc5_reset_timer = 1;
    }
  }
}

#define RC5_PERIOD (2*0.00088889)
#define RC5_CYCLES (F_CPU*RC5_PERIOD)
#define RC5_MIN    (RC5_CYCLES / 1.2)
#define RC5_MAX    (RC5_CYCLES * 1.2)

ISR(IR_IN_VECT) {
  if( rc5_bit == 0 ) {
    // start reception
    TCNT1 = 0 - (unsigned short)RC5_MAX;
    TIFR1 |= _BV(TOV1);
    TIMSK1 |= _BV(TOIE1);
    rc5_reset_timer = 0;
    rc5_bit = 1;
    rc5_code = 0;
    if( g_ir_leds ) {
      IR_LED_ON();
    }
  } else if( rc5_bit == 1 ) {
    rc5_timing = TCNT1;
    TIMSK1 &= ~_BV(TOIE1);
    rc5_timing += (unsigned short)RC5_MAX;
    if( rc5_timing >= (unsigned short)RC5_MIN ) {
      TCNT1 = 0 - (rc5_timing>>1) - (rc5_timing>>2);
      TIMSK1 |= _BV(TOIE1);
      rc5_bit = 2;
      EICRA = _BV(ISC00); // set INT0 to rising or falling edge
    } else {
      cancel_rc5_reception();
    }
  } else if( rc5_reset_timer ) {
    TCNT1 = 0 - (rc5_timing>>1) - (rc5_timing>>2);
    rc5_reset_timer = 0;
  }
}

// codes: channel 1, channel 2, channel 3, channel 4, next channel, previous channel, volume up, volume down, mute, standby
#define NUM_IR_CODES 10
unsigned short g_codes[NUM_IR_CODES] = { 0x0001, 0x0002, 0x0003, 0x0004, 0x0020, 0x0021, 0x0010, 0x0011, 0x000d, 0x000c };
unsigned char g_codes_configured = 1;
#define IR_CHANNEL1     g_codes[0]
#define IR_CHANNEL2     g_codes[1]
#define IR_CHANNEL3     g_codes[2]
#define IR_CHANNEL4     g_codes[3]
#define IR_NEXT_CHANNEL g_codes[4]
#define IR_PREV_CHANNEL g_codes[5]
#define IR_VOLUME_UP    g_codes[6]
#define IR_VOLUME_DOWN  g_codes[7]
#define IR_MUTE         g_codes[8]
#define IR_STANDBY      g_codes[9]

void select_input(unsigned char which) {
  switch(which) {
  case 0:
    PORTC = _BV(PC3);
    _delay_ms(2.5);
    PORTC = _BV(PC4);
    _delay_ms(2.5);
    PORTC = 0;
    break;
  case 1:
    PORTC = _BV(PC2);
    _delay_ms(2.5);
    PORTC = _BV(PC4);
    _delay_ms(2.5);
    PORTC = 0;
    break;
  case 2:
    PORTC = _BV(PC0);
    _delay_ms(2.5);
    PORTC = _BV(PC5);
    _delay_ms(2.5);
    PORTC = 0;
    break;
  case 3:
    PORTC = _BV(PC1);
    _delay_ms(2.5);
    PORTC = _BV(PC5);
    _delay_ms(2.5);
    PORTC = 0;
    break;
  }
}

#define LOAD_SHIFT PB7
#define CLOCK      PB6
#define DATA       PB0

void send_volume_data(unsigned short data) {
  unsigned char i;

  if( data&32768 )
    PORTB |=  _BV(DATA);
  else
    PORTB &= ~_BV(DATA);
  PORTB &= ~_BV(LOAD_SHIFT);
  asm volatile("nop");
  for( i = 0; i < 16; ++i ) {
    asm volatile("nop");
    PORTB ^= _BV(CLOCK);
    asm volatile("nop");
    data <<= 1;
    PORTB ^= _BV(CLOCK);
    if( data&32768 )
      PORTB |=  _BV(DATA);
    else
      PORTB &= ~_BV(DATA);
  }
  asm volatile("nop");
  PORTB |= _BV(LOAD_SHIFT);
} 
void set_volume(unsigned char data) {
  // set left and right channels
  send_volume_data((unsigned short)data);
  send_volume_data((unsigned short)data|256);
}

int main(void) {
  unsigned char which_input, last_input = 4;
  unsigned char volume, last_volume = 255;
  unsigned short timer = 0;
  unsigned char saved_input, saved_volume;
  unsigned char muted = 0, standby = 0;
  unsigned short last_rc5_code = 0xFFFF;

  DDRB  = _BV(PB0)|_BV(PB6)|_BV(PB7);
  PORTB = _BV(PB7);

  DDRC  = _BV(PC0)|_BV(PC1)|_BV(PC2)|_BV(PC3)|_BV(PC4)|_BV(PC5);

  DDRD = _BV(PD3)|_BV(PD4)|_BV(PD5);
  PORTD = _BV(PD3)|_BV(PD4)|_BV(PD5);

  TCCR0A = _BV(WGM01);
  TCCR0B = _BV(CS01)|_BV(CS00);
  OCR0A = 156;
  TIMSK0 = _BV(OCIE0A);

  EICRA = _BV(ISC01)|_BV(ISC00); // set INT0 to rising edge
  EIMSK = _BV(INT0);
  TCCR1B = _BV(CS10); // enable timer
  sei();


  which_input = eeprom_read_byte((uint8_t*)0);
  if( which_input > 4 )
    which_input = 0;
  volume = eeprom_read_byte((uint8_t*)1);
  if( volume > 127 )
    volume = 0;
  saved_input = which_input;
  saved_volume = volume;

  delay_ms(250);
  while(1) {
    if( rc5_final_code ) {
      if( (rc5_final_code&0xFFF) == IR_CHANNEL1 ) {
        if( rc5_final_code != last_rc5_code ) {
          which_input = 0;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_CHANNEL2 ) {
        if( rc5_final_code != last_rc5_code ) {
          which_input = 1;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_CHANNEL3 ) {
        if( rc5_final_code != last_rc5_code ) {
          which_input = 2;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_CHANNEL4 ) {
        if( rc5_final_code != last_rc5_code ) {
          which_input = 3;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_NEXT_CHANNEL ) {
        if( rc5_final_code != last_rc5_code ) {
          if( ++which_input == 4 )
            which_input = 0;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_PREV_CHANNEL ) {
        if( rc5_final_code != last_rc5_code ) {
          if( --which_input == 0xFF )
            which_input = 3;
          muted = 0;
        }
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_VOLUME_UP ) {
        if( volume > 0 )
          --volume;
        timer = 1;
        muted = 0;
      } else if( (rc5_final_code&0xFFF) == IR_VOLUME_DOWN ) {
        if( volume < 127 )
          ++volume;
        timer = 1;
        muted = 0;
      } else if( (rc5_final_code&0xFFF) == IR_MUTE ) {
        if( rc5_final_code != last_rc5_code )
          muted = !muted;
        timer = 1;
      } else if( (rc5_final_code&0xFFF) == IR_STANDBY ) {
        if( rc5_final_code != last_rc5_code )
          standby = !standby;
        timer = 1;
      }
      last_rc5_code = rc5_final_code;
      rc5_final_code = 0;
    }
    if( which_input != last_input ) {
      select_input(which_input);
      last_input = which_input;
    }

    if( standby ) {
      PORTD |= _BV(PD4)|_BV(PD5);
    } else if( muted ) {
      PORTD &= ~(_BV(PD4)|_BV(PD5));
    } else {
      PORTD |= _BV(PD4);
      PORTD &= ~_BV(PD5);
    }

    if( (muted || standby ? 128 : volume) != last_volume ) {
      set_volume(muted || standby ? 128 : volume);
      last_volume = muted || standby ? 128 : volume;
    }
    if( timer ) {
      if( ++timer == 0 ) {
        last_rc5_code = 0xFFFF;
        if( saved_input != which_input ) {
          eeprom_write_byte((uint8_t*)0, which_input);
          saved_input = which_input;
        }
        if( saved_volume != volume ) {
          eeprom_write_byte((uint8_t*)1, volume);
          saved_volume = volume;
        }
      }
    }
  }
  return 0;
}
