/* Silicon_Chip_MicroMag3_control_sketch_V2.ino
 * A sketch to control the MicroMag3 magnetometer sensor
 * written by Jim Rowe for Silicon Chip, with help from
 * Nicholas Vinen and Tim Blythman.
 * The section to show the Heading has now been added.
 * 
 * Last updated 8/12/2023 at 11:05 am
 * Note that this sketch uses the Arduino SPI library to communicate
 * with the MicroMag3 3-axis magnetometer module.This library expects 
 * these pins on an Arduino Uno to be used for SPI communication:
 * MOSI: pin 11 (or ISCP4)
 * MISO: pin 12 (or ICSP1)
 * SCLK: pin 13 (or ICSP3)
 * In this sketch we are also using pin 7 to output the SSbar signal
 * used to open an SPI transaction, pin 5 to output the RESET signal 
 * pulse which announces the start of a transaction and pin 6 for 
 * receiving the DRDY signal from the MicroMag3 (which goes high
 * to show when measurement data is available). But when we tried
 * using the DRDY line to tell when the measurement data was ready,
 * we found that this caused the sketch to stop working. So we changed
 * the sketch to just pause for 40ms before getting the measurement
 * data. This is the same procedure used in other sketches used to
 * communicate with the MicroMag3.
 */
 
#include <SPI.h>
#include <Wire.h>

// define the pins used for SPI communication
#define MOSI 11
#define MISO 12
#define SCLK 13
#define SSbar 7
#define RESET 5
#define DRDY 6

byte CmdX = B01100001;  // command byte to request X axis reading
byte CmdY = B01100010;  // command byte to request Y axis reading
byte CmdZ = B01100011;  // command byte to request Z axis reading
byte Cmdbyte = 0;        // working byte for SPI commands
short RdgX = 0;   // to receive the X axis reading
short RdgY = 0;   // to receive the Y axis reading
short RdgZ = 0;   // to receive the  Z axis reading
short Reading = 0;    // working variable to receive readings
short DataIn = 0;
double Heading = 0;  // to carry the compass heading, in radians
short DegHdg = 0;  // to carry the heading in degrees
void setup() 
{
  Serial.begin(9600);
  pinMode(SSbar, OUTPUT);
  pinMode(RESET, OUTPUT);
  pinMode(MOSI, OUTPUT);
  pinMode(MISO, INPUT);
  pinMode(SCLK, OUTPUT);
  pinMode(DRDY, INPUT);
  Serial.println("A sketch to communicate with the MicroMag3");
  delay(2000);  // pause for 2 seconds before starting

  digitalWrite(SSbar, HIGH);  // deselect the MicroMag3 for a fresh start
  digitalWrite(RESET, LOW);   // and also set the RESET line low
  SPI.begin();    // then start the SPI library
}

void loop() 
{
Cmdbyte = CmdX;   // set Cmdbyte for getting X axis reading
GetaReading();    // then go get that reading
RdgX = Reading;   // save Reading as RdgX

Cmdbyte = CmdY;   // set Cmdbyte for getting Y axis reading
GetaReading();
RdgY = Reading;   // save Reading as RdgY

Cmdbyte = CmdZ;   // set Cmdbyte for getting Z axis reading
GetaReading();
RdgZ = Reading;   // and save it as RdgZ

Serial.print("X reading = ");
Serial.println(RdgX);
Serial.print("Y reading = ");
Serial.println(RdgY);
Serial.print("Z reading = ");
Serial.println(RdgZ);

FindHeading(); // now work out the heading
Serial.print("Heading = ");
Serial.print(Heading);
Serial.print(" radians");
DegHdg = Heading * 180/PI;
Serial.print(" or ");
Serial.print(DegHdg);
Serial.println(" in degrees");
Serial.println(" ");  // followed by a blank line for spacing
delay(5000); // pause for five seconds to allow reading
}

void GetaReading()
{
  digitalWrite(SSbar, LOW);  // drop the SSbar line to begin a transaction
  digitalWrite(RESET, HIGH);  // now pulse RESET line high for 2us
  delayMicroseconds(2);
  digitalWrite(RESET, LOW); // then send it low again
  SPI.beginTransaction(SPISettings(500000, MSBFIRST, SPI_MODE0));
  SPI.transfer(Cmdbyte);    // now send command byte to MicroMag3
  delay(40);    // then delay for 40ms to give it time to measure
  DataIn = SPI.transfer16(0x0000); // so we can get the 16-bit reading
  Reading = DataIn;     // and save it as Reading
  SPI.endTransaction(); // and end the transaction
  digitalWrite(SSbar, HIGH);  // also take SSbar high again
}

void FindHeading()
{
      Heading = PI/2 + (atan2(-RdgX,-RdgY));
}
