


; Speech Timer
; using IR remote from littlebird electronics

; Alarm tone set by the PWM frequency (fundamental tone) and bit for WARBLE, the modulation rate

; CPU configuration
	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc


;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB3  & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _HS_OSC

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF


; Define variables at memory locations

; EEPROM
EEPROM1		equ	H'00'	; preset time sets
EEPROM2		equ	H'01'	; preset main display dimming
EEPROM3		equ	H'02'	; preset lights display dimming
EEPROM4		equ	H'03'	; alarm volume

EEPROM5		equ	H'04'	; DISP1 store
EEPROM6		equ	H'05'	; DISP2 store
EEPROM7		equ	H'06'	; DISP3 store

EEPROM8		equ	H'07'	; green minutes warning (G_MIN)
EEPROM9		equ	H'08'	; green seconds warning (G_SEC)

EEPROMA		equ	H'09'	; amber minutes warning (A_MIN)
EEPROMB		equ	H'0A'	; amber seconds warning (A_SEC)

EEPROMC		equ	H'0B'	; red minutes warning (R_MIN)
EEPROMD		equ	H'0C'	; red seconds warning (R_SEC)

; RAM
; bank 0 

ACK			equ	H'20'	; acknowlwdge LED flag (clear is off, bit 0 set=on)
PRESETS		equ	H'21'	; preset time periods
COUNTER		equ	H'22'	; counter
REMOTE_A		equ	H'23'	; remote control Address byte
REM_A_BAR	equ	H'24'	; complementary value of address
REMOTE_C		equ	H'25'	; remote control Command byte
REM_C_BAR	equ	H'26'	; complementary value of command
BIT_COUNT		equ	H'27'	; 32 bit counter
STORE1			equ	H'28'	; delay store
STORE2			equ	H'29'	; delay store
STORE3			equ	H'2A'	; delay store	
TIME_MODE		equ	H'2B'	; timing mode (reset, run and stop)
COLON			equ	H'2C'	; seconds colon flash
CK_COUNT_LS	equ	H'2D'	; 50us counter for 500ms ls byte
CK_COUNT_MS	equ	H'2E'	; 50us counter for 500ms ms byte
msCOUNT_MS	equ H'2F'	; 110ms counter ms byte
msCOUNT_LS	equ H'30'	; 110ms counter ls bytes
SECONDS		equ	H'31'	; seconds counter
MINUTES		equ	H'32'	; minutes
PAUSE			equ	H'33'	; timer clock paused
RESET_TMR		equ	H'34'	; timer reset
BIN_0			equ	H'35'	; 8-bit binary value
BCD_0			equ	H'36'	; MS BCD value
BCD_1			equ	H'37'	; LS binary coded decimal value
TEMP			equ	H'38'	; data storage 
CNT_8			equ	H'39'	; counter in BCD routine
DISP1			equ	H'3A'	; display 1 (normally tens of minutes)	
DISP2			equ	H'3B'	; display 2 (normally minutes)	
DISP3			equ	H'3C'	; display 3 (normally tens of seconds)
PORTB_STO		equ	H'3D'	; store PORTB
PORTA_STO		equ	H'3E'	; store PORTA
MPX_CNT		equ	H'3F'	; multiplex rate counter 
MPX_CYCLE	equ	H'40'	; multiplex cycle counter
MPX_OFF		equ	H'41'	; multiplex off counter (switch off for dimming control)
OFF			equ	H'42'	; display off flag (for dimming)
DIM_MAIN		equ	H'43'	; dimming value for main display
DIM_LIGHTS		equ	H'44'	; dimming value for lights display
LOW_BAT.		equ	H'45'	; low battery flag
ALT_DISP		equ	H'46'	; alternative display 2-digit seconds or 3-digit minutes
ALL_DIG_MIN	equ	H'47'	; show 3-digits of minutes when >99minutes
PRES_FLG		equ	H'48'	; preset flag
G_MIN			equ	H'49'	; green preset minutes
G_SEC			equ	H'4A'	; green preset seconds
A_MIN			equ	H'4B'	; amber preset minutes
A_SEC			equ	H'4C'	; amber preset seconds
R_MIN			equ	H'4D'	; red preset minutes
R_SEC			equ	H'4E'	; red preset seconds
DIM_DIR			equ	H'4F'	; dimming direction main display
DIM_DIR1		equ	H'50'	; dimming direction warning LED display	
GAR_CNTR		equ	H'51'	; green, amber, red counter (0-3 for off, 1=green,2= amber and 3=red)	
RPT				equ	H'52'	; repeat counter
DIM_RATE		equ	H'53'	; dimming rate 2-speed rate
VOLUME		equ	H'54'	; alarm volume
VOL_TMR		equ	H'55'	; volume timer
VOL_CNT_MS	equ	H'56'	; volume counter ms byte
VOL_CNT_LS	equ	H'57'	; volume counter ls byte
VOL_UP_DN		equ	H'58'	; volume up or down direction
WARBLE		equ	H'59'	; warbling of PWM
BUZZ			equ	H'5A'	; buzzer sound flag
CHIRP			equ	H'5B'	; end of volume timer acknowledge chirp length
VOL_RATE		equ	H'5C'	; volume up or down rate
FASTR			equ	H'5D'	; timer up/down rate
; BCD to Binary
H_BYTE			equ	H'5E'	; high byte
L_BYTE			equ	H'5F'	; low byte
REG_0			equ	H'60'	; MSD of 5-digit BCD in right hand (low) nibble
REG_1			equ	H'61'	; BCD number (packed BCD)
REG_2			equ	H'62'	; BCD low order numbers
H_TEMP		equ	H'63'	; temporary register of high order binary result
L_TEMP			equ	H'64'	; temporary register of low order binary result	

; all banks ram
W_TMP			equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 
INT_FLG 		equ H'72'	; interrupt flag for IR decode
TEMP1			equ	H'73'	; temporary working register
ALM_SOUND	equ	H'74'	; flag for alarm
SND_PERD		equ	H'75'	; sound period
STORE_FLG1	equ	H'76'	; EEPROM storage flag (timer)
STORE_FLG2	equ	H'77'	; EEPROM storage flag (warning LEDs)		
RUN_FLG		equ	H'78'	; reset or run/paused indication flag

	ORG     2100		; preprogram EEPROM locations
	DE		D'00'	; no presets for timer
	DE		D'25'	; mid brightness main display 
	DE		D'25'	; mid  brightness lights display 
	DE		D'32'	; alarm volume (127 maximum)

	DE		D'00'	; DISP1
	DE		D'00'	; DISP2
	DE		D'00'	; DISP3

	DE		D'00'	; green minutes warning (G_MIN)
	DE		D'00'	; green seconds warning (G_SEC)

	DE		D'00'	; amber minutes warning (A_MIN)
	DE		D'00'	; amber seconds warning (A_SEC)

	DE		D'00'	; red minutes warning (R_MIN)
	DE		D'00'	; red seconds warning (R_SEC)
	
; define reset and interrupt vector start addresses

	org		0			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org    	 4			; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT

;_________________________________________

; lookup tables for 7-segment displays and LEDs. Uses PORTA and PORTB so separate lookups 
; for Dp on =RA2 low

; settings for display 1 and 2 for PORTA
; segment F=RA0, A=RA3, B=RA4
DISP1_PORTA
DISP2_PORTA
	addwf		PCL,f		; 0-9
;					  ba     f
	retlw		B'11100110'	; 0 abf
	retlw		B'11101111'	; 1 b
	retlw		B'11100111' 	; 2 ab
	retlw		B'11100111'	; 3 ab
	retlw		B'11101110'	; 4 bf
	retlw		B'11110110'	; 5 af
	retlw		B'11110110'	; 6 af
	retlw		B'11100111'	; 7 ab
	retlw		B'11100110'	; 8 abf
	retlw		B'11100110'	; 9 abf
	retlw		B'11111111'	; A for a dash 
	retlw		B'11100010'	; B for a '0 with colon'

; settings for display 1 and 2 for PORTB
DISP1_PORTB
DISP2_PORTB
	addwf		PCL,f		; 0-9
;				   g     e  dc 
	retlw		B'11101001'	; 0 
	retlw		B'11111101'	; 1 
	retlw		B'01101011' ;  2 
	retlw		B'01111001'	; 3 
	retlw		B'01111101'	; 4 
	retlw		B'01111001'	; 5 
	retlw		B'01101001'	; 6 
	retlw		B'11111101'	; 7
	retlw		B'01101001'	; 8 
	retlw		B'01111001'	; 9 
	retlw		B'01111111'	; A for a dash 
	retlw		B'11101001'	; B for 0 'with colon'

; settings for display 3 for PORTA  (Display 3 is upside down so Dp is upside and with different connections to segments compared to Disp1&2)
DISP3_PORTA
	addwf		PCL,f		; 0-5 normally used for timer clock but other digits can be used in settings
;					  gf     d
	retlw		B'11110110'	; 0 fd
	retlw		B'11110111'	; 1 f
	retlw		B'11101110' ;  2 dg
	retlw		B'11100110'	; 3 dfg
	retlw		B'11100111'	; 4 fg
	retlw		B'11100110'	; 5 dfg
	retlw		B'11100110'	; 6 dfg
	retlw		B'11110110' 	; 7 df
	retlw		B'11100110'	; 8 dfg
	retlw		B'11100110' 	; 9 dfg
;  symbols 
	retlw		B'11111111'	; 'A' seconds symbol (no segments required for PORTA)
	retlw		B'11110010'	; B for a 0 with colon
	retlw		B'11111111'	; C for minutes symbol (no segments required for PORTA)

; settings for display 3 for PORTB  (Display 3 is upside down and with different connections to segments compared to Disp1&2)
DISP3_PORTB
	addwf		PCL,f		; 0-5 normally used for timer clock
;				   c     b  ae 
	retlw		B'01101001'	; 0 cbae
	retlw		B'11111101'	; 1 e
	retlw		B'11101001' ;  2 eba
	retlw		B'11111001'	; 3 ae
	retlw		B'01111101'	; 4 ce
	retlw		B'01111011'	; 5 ca
	retlw		B'01101011'	; 6 abc
	retlw		B'11111101' 	; 7 e
	retlw		B'01101001'	; 8 abce
	retlw		B'01111001' 	; 9 ace
;  symbols
	retlw		B'01111101'	; A  ce for  " (seconds symbol)
	retlw		B'01101001'	; B for a 0 with colon
	retlw		B'01111111'	; C ce for  ' (seconds symbol)

; settings for green, amber and red LEDs and low battery/ ack. indicator
; using GAR_CNTR 0= none, 1= green, 2=amber, 3=red;
; Low Batt, acknowledge.= RA0 low
; Green=RA4, Amber = RA3, RED=RB1
DISP4_PORTA
	addwf		PCL,f		; 0-3 counter value
;					  GA		; green and amber
	retlw		B'11111111'	; 0	None
	retlw		B'11101111'	; 1	Green
	retlw		B'11110111' ;  2 	Amber
	retlw		B'11111111'	; 3	Red

DISP4_PORTB
	addwf		PCL,f		; 0-3 counter value
;					         R	; red
	retlw		B'11111111'	; 0	None
	retlw		B'11111111'	; 1	Green
	retlw		B'11111111' ;  2 	Amber
	retlw		B'11111101'	; 3	Red

; ************************************************************************************	
; start interrupt by saving w and status registers before altered by interrupt routine

INTERRUPT
	
	movwf	W_TMP			; w to w_tmp storage
	swapf	STATUS,w		; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0
	bcf		STATUS,RP1

; adjust interrupt rate with counter

	movlw	D'128'			; freq is 20MHz/4/2/255-128 = 50us 
	addwf	TMR0,f			; add to timer register and takes 2 cycles at 200ns each to start counting
	bcf		INTCON,TMR0IF	; clear TMR0 interrupt flag

	bsf		INT_FLG,0		; set flag to indicate an interrupt has occurred

; increase 110ms repeat counter
	incfsz	msCOUNT_LS,f	; increase 110ms counter ls byte
	goto	MPLX
	incf		msCOUNT_MS,f	; increase 110ms counter ms byte

; ---------------------------------------------------------------------------
MPLX; (on at MPX_CNT, off at MPX_OFF)
; Multiplex displays
; 50us x  counter value
	incf		MPX_CNT,f		; multiplex rate counter 
	
CF_RATE
; compare with count
	movf	MPX_CNT,w
	xorlw	D'50'			; 2.5ms per segment so with 4 displays refreshed each 10ms, ie 100Hz rate
	btfss	STATUS,Z
	goto	OFF_DISP		; not equal so not required ms period

; alarm sound
	btfss	BUZZ,0			; run when set
	goto	DECV
	incf		WARBLE,f
	btfsc	WARBLE,1		; bit sets repetition rate: bit 0: 2.5ms, bit 1: 5ms etc
	goto	WARB_ON
; start PWM for sound
	movf	VOLUME,w		; sets duty cycle
	movwf	CCPR1L		;  PWM
	goto	DECV
WARB_ON
	clrf		CCPR1L		; duty 0% 

DECV
; decrease volume counter
; check if zero
	movf	VOL_CNT_MS,w
	btfsc	STATUS,Z
	goto	ZERO_MPX		; stop reducing when VOL_CNT_MS is 0
; decrease
	decfsz	VOL_CNT_LS,f
	goto	ZERO_MPX
	decfsz	VOL_CNT_MS,f	; when ms byte goes to zero apply acknowledge chirp
; acknowledge chirp
	goto	ZERO_MPX
	movf	VOLUME,w
	movwf	CCPR1L		; chirp on
	movlw	D'15'			; set chirp length (25ms)
	movwf	CHIRP

ZERO_MPX
	movf	CHIRP,w
	btfsc	STATUS,Z
	goto	BY_CHIRP_CLEAR
	decfsz	CHIRP,f			; chirp length counter
	goto	BY_CHIRP_CLEAR
	clrf		CCPR1L		; end of chirp	

BY_CHIRP_CLEAR
; start at 0 again
	clrf		MPX_CNT	
	clrf		MPX_OFF	
	clrf		OFF			; reset display off flag		

	movf	MPX_CYCLE,w	; multiplex cycle counter
	btfsc	STATUS,Z
	goto	SHOW_DSP4
	movf	MPX_CYCLE,w	; multiplex cycle counter
	xorlw	D'1'
	btfsc	STATUS,Z
	goto	SHOW_DSP3
	movf	MPX_CYCLE,w	; multiplex cycle counter
	xorlw	D'2'
	btfsc	STATUS,Z
	goto	SHOW_DSP2
SHOW_DSP1
; set bits high
; clear display
	movlw	B'11111111'
	movwf	PORTA			; segments off
	movwf	PORTB
	movwf	PORTA_STO	
	movwf	PORTB_STO

; leading zero blanking
; if DISP1 is 0 then do not drive DISP1 segments
	movf	DISP1,w
	btfsc	STATUS,Z
	goto	SELECT1
NO_BLANK	
; load display	
	movf	DISP1,w
	call		DISP1_PORTA	; get values for driving 7-SEG via portA
	movwf	PORTA_STO
	movf	DISP1,w
	call		DISP1_PORTB	; get values for driving 7-SEG via portB
	movwf	PORTB_STO
SELECT1
	clrf		MPX_CYCLE	; counter starts again
; select digit1	
	bcf		PORTB_STO,5
	bcf		PORTB_STO,6
	movf	PORTB_STO,w
	movwf	PORTB
	movf	PORTA_STO,w
	movwf	PORTA
	goto	CLOCK	
SHOW_DSP2
; set bits high
; clear displays		
	movlw	B'11111111'
	movwf	PORTA			; segments off
	movwf	PORTB
	movwf	PORTA_STO
	movwf	PORTB_STO
; load display
	movf	DISP2,w
	call		DISP2_PORTA	; get values for driving 7-SEG via portA
	movwf	PORTA_STO	
	movf	DISP2,w
	call		DISP2_PORTB	; get values for driving 7-SEG via portB
	andwf	PORTB_STO,f
;  drive Dp
	btfsc	ALT_DISP,0		; alternative display flag, when set,  no colon 
	goto	DS_2
	btfss	COLON,0
	bcf		PORTA_STO,2		; Dp on
DS_2
	movf	PORTA_STO,w
	movwf	PORTA
; select digit2
	bsf		PORTB_STO,5
	incf		MPX_CYCLE,f
	bsf		PORTB_STO,6
	movf	PORTB_STO,w
	movwf	PORTB
	goto	CLOCK	
SHOW_DSP3
; set bits high
; clear display		
	movlw	B'11111111'
	movwf	PORTA			; segments off
	movwf	PORTB	
	movwf	PORTA_STO
	movwf	PORTB_STO
; load display	
	movf	DISP3,w
	call		DISP3_PORTA	; get values for driving 7-SEG via portA
	movwf	PORTA_STO	
	movf	DISP3,w
	call		DISP3_PORTB	; get values for driving 7-SEG via portB
	andwf	PORTB_STO,f
;  drive Dp
	btfsc	ALT_DISP,0		; alternative flag, when set,  no colon 
	goto	DS_3
	btfss	COLON,0
	bcf		PORTA_STO,2		; Dp on
DS_3
	incf		MPX_CYCLE,f
; select digit3
	bsf		PORTB_STO,5
	bcf		PORTB_STO,6
	movf	PORTB_STO,w
	movwf	PORTB
	movf	PORTA_STO,w
	movwf	PORTA
	goto	CLOCK	
SHOW_DSP4
; set bits high
; clear display		
	movlw	B'11111111'
	movwf	PORTA			; segments off
	movwf	PORTB
	movwf	PORTA_STO
	movwf	PORTB_STO
; load display
	movf	GAR_CNTR,w	; Green, Amber & Red counter
	call		DISP4_PORTA	; get values for driving LEDs via portA
	movwf	PORTA_STO
	movf	GAR_CNTR,w	; Green, Amber & Red counter
	call		DISP4_PORTB	; get values for driving LEDs via portB
	movwf	PORTB_STO
; check for acknowledge
	btfss	ACK,0			; if acknowledge LED required clear portA,0
	goto	CK_LOWBAT
	bcf		PORTA_STO,0
	clrf		LOW_BAT.
	goto	BY_LOW_BAT	; no check of low batt if acknowledge
CK_LOWBAT
; check for low batt.
	btfsc	LOW_BAT.,0		; low battery flag
	bcf		PORTA_STO,0	; if low battery show the LOW BATT. LED
BY_LOW_BAT
	incf		MPX_CYCLE,f
; select digit4	
	bcf		PORTB_STO,5
	bsf		PORTB_STO,6
	movf	PORTB_STO,w
	movwf	PORTB
	movf	PORTA_STO,w
	movwf	PORTA

; check if end of count down alarm
; stop PWM
	movf	SND_PERD,w	; sound period
	btfsc	STATUS,Z
	goto	CLOCK
	decfsz	SND_PERD,f	; reduce sound period
	goto	CLOCK
; buzzer off
	clrf		BUZZ			; BUZZer flag
	clrf		CCPR1L		; duty 0% 
	goto	CLOCK

;-------------------------------------------------------------------------------------------
OFF_DISP ; is display off?
	btfsc	OFF,0			; if set then display is already off (used for dimming switching off before next display)
	goto	CLOCK	
; increase counter for off 	
	incf		MPX_OFF,f		; multiplex rate off counter 

; compare with DIM_MAIN or DIM_LIGHTS
	movf	MPX_CYCLE,w
	xorlw	D'01'
	btfsc	STATUS,Z
	goto	TEST_LIGHTS		; when MPX_CYCLE is zero

	movf	MPX_OFF,w
	xorwf	DIM_MAIN,w
	btfss	STATUS,Z
	goto	CLOCK
	goto	CLR_DISP
TEST_LIGHTS
	movf	MPX_OFF,w
	xorwf	DIM_LIGHTS,w
	btfss	STATUS,Z
	goto	CLOCK
CLR_DISP
; start at 0 again
	clrf		MPX_OFF	
	bsf		OFF,0			; off flag
; clear displays		
	movlw	B'11111111'
	movwf	PORTA			; segments off
	movwf	PORTB

;-------------------------------------------------------------------------------------------------------------
CLOCK;  timer
; check if stopped
	btfss	PAUSE,0
	goto	A_D				; A_D   bypass clock changes if paused
; 10,000 (H2710) is 500ms
	incfsz	CK_COUNT_LS,f	; increase each 50us
	goto	CF_2710
	incf		CK_COUNT_MS,f

; compare with D10,000, H2710
CF_2710
	movf	CK_COUNT_LS,w
	xorlw	H'10'
	btfss	STATUS,Z
	goto	A_D				; not equal so not 500ms
	movf	CK_COUNT_MS,w
	xorlw	H'27'
	btfss	STATUS,Z
	goto	A_D				; not 500ms	
; start at 0 again
	clrf		CK_COUNT_LS
	clrf		CK_COUNT_MS		
	incf 	COLON,f		; toggle bit 0 for seconds flasher colon

	btfss	COLON,0	
	goto	A_D			; only update seconds each alternate 500ms period (ie 1s)

; increase or decrease (if preset=15 then decrease to 0:0)

	movf	PRESETS,w
	xorlw	D'15'
	btfss	STATUS,Z
	goto	CLOCK_UP
; down counter
	movf	SECONDS,w
	btfsc	STATUS,Z		; if zero reduce Minutes
	goto	DEC_MIN
	decf	SECONDS,f

; if both minutes and seconds are zero, sound alarm
	movf	MINUTES,w
	btfss	STATUS,Z
	goto	CF
	movf	SECONDS,w
	btfsc	STATUS,Z
; alarm
; **** remark out next line for no alarm at end of countdown timer
	bsf		ALM_SOUND,0	; flag for alarm
	goto	CF
	goto	CF
DEC_MIN
	movf	MINUTES,W
	btfsc	STATUS,Z
	goto	END_TMR
	movlw	D'59'
	movwf	SECONDS
	decf	MINUTES,f
	goto	CF

END_TMR
; alarm
; **** remark out next line for no alarm at end of countdown timer
	bsf		ALM_SOUND,0	; flag for alarm
	bcf		PAUSE,0		; pause when ended minutes
	bcf		COLON,0		; colon on
	goto	CF				; compare with minutes and seconds for warning LEDs

CLOCK_UP
; seconds
	incf		SECONDS,f		; seconds
; check for 60s 
	movf	SECONDS,w
	sublw	D'59'			; if seconds is 60s then result is negative

; NOTE (Testing using btfsc allows fast minutes count but does not check warning lights for when seconds are not zero ie for X:30s settings)
	btfss	STATUS,C 		; btfss required for normal running; btfsc testing
	clrf		SECONDS		; clear seconds when @ 60
	movf	SECONDS,w		; check if seconds are clear
	btfsc	STATUS,Z
	incf		MINUTES,f 		; next minutes when seconds clear

; check for 99 minutes
	movf	MINUTES,w
	sublw	D'98'
	btfsc	STATUS,C 

; allow next 4-lines for a 255 minute limit but after 99 has only 1m resolution	
;	bsf		ALL_DIG_MIN,0	; set when >99 minutes so 3-digits of minutes will be displayed
; check for 255 minutes
;	movf	MINUTES,w
;	xorlw	D'255' 
;	btfss	STATUS,Z 	 
	goto	CF				; compare with minutes and seconds for warning LEDs
	bcf		PAUSE,0		; pause when ended minutes
	bcf		COLON,0		; colon on
;_______________________________________________
CF; compare
; compare minutes and seconds with green, amber and red warning periods
; first test if minutes and seconds are zero then bypass testing
	movf	MINUTES,w
	btfss	STATUS,Z
	goto	CH_GRN	; minutes are not zero
	movf	SECONDS,w
	btfsc	STATUS,Z
	goto	DISP_CON	; bypass testing against warning periods
CH_GRN
	movf	G_MIN,w		; green minutes
	xorwf	MINUTES,w	; test minutes
	btfss	STATUS,Z
	goto	CH_AMB	; check amber
	movf	G_SEC,w	; green seconds
	xorwf	SECONDS,w; test seconds
	btfss	STATUS,Z
	goto	CH_AMB
	movlw	D'01'
	movwf	GAR_CNTR	; if equal to clock set to show green	

CH_AMB
	movf	A_MIN,w		; amber minutes
	xorwf	MINUTES,w	; test minutes
	btfss	STATUS,Z
	goto	CH_RED	; check red
	movf	A_SEC,w	; amber seconds
	xorwf	SECONDS,w; test seconds
	btfss	STATUS,Z
	goto	CH_RED
	movlw	D'02'
	movwf	GAR_CNTR	; if equal to clock set to show amber	

CH_RED
	movf	R_MIN,w		; red minutes
	xorwf	MINUTES,w	; test minutes
	btfss	STATUS,Z
	goto	DISP_CON
	movf	R_SEC,w	; red seconds
	xorwf	SECONDS,w; test seconds
	btfss	STATUS,Z
	goto	DISP_CON
	movlw	D'03'
	movwf	GAR_CNTR	; if equal to clock set to show red	

;_____________________________________________
; Display conversions converting to required BCD and display position
DISP_CON
	call		DISPLAY_CONV
	goto	A_D
;__________________________________________________________________
DISPLAY_CONV
	btfsc	ALL_DIG_MIN,0	; show 3-digits of minutes when >99minutes
	goto	MIN_DSP
	btfsc	ALT_DISP,0		; alternative  display
	goto	ALT_DISP_SHOW
; normal minutes/seconds display
; convert minutes to BCD for display
	movf	MINUTES,w
	movwf	BIN_0
	call		BCD			; convert to BCD
; BCD_0 and BCD_1 have unpacked decimal values
	movf	BCD_1,w		;  minutes
	andlw	H'0F'			; get ls 4-bits
	movwf	DISP2
	swapf	BCD_1,w		; 10's of minutes
	andlw	H'0F'			; get ls 4-bits
	movwf	DISP1

; converts seconds to decimal
	movf	SECONDS,w
	movwf	BIN_0
	call		BCD			; convert to BCD
; BCD_0 and BCD_1 have unpacked decimal values
	swapf	BCD_1,w		; 10's of seconds
	andlw	H'0F'	
	movwf	DISP3
	return					;goto	A_D

ALT_DISP_SHOW
; show 10's of seconds and seconds	; converts seconds to decimal
	movf	SECONDS,w
	movwf	BIN_0
	call		BCD			; convert to BCD
; BCD_0 and BCD_1 have unpacked decimal values
	swapf	BCD_1,w		; 10's of seconds
	andlw	H'0F'
	movwf	DISP1
	movf	BCD_1,w		; 1's of minutes
	andlw	H'0F'
	movwf	DISP2
; show seconds symbol "
	movlw	H'A'
	movwf	DISP3
	return					; goto	A_D

MIN_DSP
	bsf		ALT_DISP,0		; alternative  display flag set so no colon
; 3-digit minutes display
; convert minutes to BCD for display
	movf	MINUTES,w
	movwf	BIN_0
	call		BCD			; convert to BCD
; BCD_0 and BCD_1 have unpacked decimal values
	movf	BCD_0,w		; 100's of  minutes
	andlw	H'0F'			; get ls 4-bits
	movwf	DISP1
	swapf	BCD_1,w		; 10's of minutes
	andlw	H'0F'			; get ls 4-bits
	movwf	DISP2
	movf	BCD_1,w		; 1's of minutes
	andlw	H'0F'			; get ls 4-bits
	movwf	DISP3
	return
;______________________________________________________
A_D
; if LS byte is zero and ms byte rolls over to 0, then run A/D
	movf	msCOUNT_LS,w
	btfss	STATUS,Z
	goto	EOI_END
	incfsz	msCOUNT_MS,w; check if around 3.276s
	goto	EOI_END
	bsf		ADCON0,2		; GO/DONE bit start conversion
	movf	ADRESH,w
	sublw	D'112'
	btfss	STATUS,C

; if reading is above 112 then low battery (8.8V
	bsf		LOW_BAT.,0		; low battery flag

	movf	ADRESH,W
	sublw	D'105'
	btfsc	STATUS,C	
; if reading is below 105 then battery ok (120mV hysteresis)
	bcf		LOW_BAT.,0		; low battery flag

;________________________________________________________

; end of interrupt reclaim w and status 
EOI_END
	btfsc	ALM_SOUND,0	; flag for alarm sound at end of countdown
	goto 	SOUND_ALARM
	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS			; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   	W_TMP,w		; swap bits and into w register
	retfie					; return from interrupt

SOUND_ALARM
	clrf		GAR_CNTR		; warning LEDs off
	clrf		ALM_SOUND	; flag for alarm
	movlw	D'255'
	movwf	SND_PERD		; sound period
	bsf		BUZZ,0			; buzzer tone
	clrf		WARBLE
; start PWM for sound
	movf	VOLUME,w		; sets duty cycle
	movwf	CCPR1L		;  PWM
	goto	EOI_END

;********************************************************************************************** 
   	
; Set ports A & B

MAIN
	movlw	B'11111111'
	movwf	PORTA
	movwf	PORTB

	call		DELAY

; set inputs/outputs
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'		; comparators off
	movwf	CMCON
	movlw	B'00000001'		; Inputs/outputs
	movwf	TRISB			; port B data direction register
	movlw	B'10000000'		; 
	movwf	OPTION_REG	; TMR0 prescaler is 2, PORTB pullups off
	movlw   B'11100010'		; Inputs/ outputs 
	movwf   TRISA			; port A data direction register

; setup PWM
	movlw	H'FF'
	movwf	PR2				; PWM period register

; analog or IO
; set for AN1
	movlw	B'00000010'
	movwf	ANSEL			; all IO except AN1

; analog inputs, A/D
	movlw	B'00000000'	; left justified A/D result, Vdd to Vss A/D
	movwf	ADCON1
	bcf		STATUS,RP0	; select memory bank 0
	movlw	B'10001000'		; Fosc, channel 1 etc
	movwf	ADCON0
	bsf		ADCON0,ADON	; A/D on

; pwm set
	clrf		CCPR1L		; duty 0% 
	bcf		CCP1CON,4
	bcf		CCP1CON,5		; clear 10-bits
	bsf		T2CON,1		; if bit 0 is set then prescaler /4. If bit 1 set instead /16
	bsf		T2CON,2		; enable timer 2

	movlw	B'00001100'		; set PWM mode
	movwf	CCP1CON		; enable PWM operation
	
; initialise registers
	clrf		ACK			; acknowledge LED flag off
	clrf		TIME_MODE		; timing mode (reset, run and stop) set at reset
	clrf		COLON			; seconds colon flash
	clrf		CK_COUNT_LS	;  50us counter for 500ms ls byte
	clrf		CK_COUNT_MS	; 50us counter for 500ms ms byte
	clrf		msCOUNT_MS	; 110ms counter ms byte
	clrf		msCOUNT_LS	; 110ms counter ls bytes
	clrf		SECONDS		; seconds counter
	clrf		MINUTES		; minutes
	clrf		PAUSE			; timer clock paused
	clrf		RESET_TMR		; timer reset
	clrf		MPX_CNT		; multiplex rate counter 
	clrf		MPX_CYCLE	; multiplex cycle counter
	clrf		OFF			; display off flag
	clrf		MPX_OFF		; multiplex rate off counter
	clrf		LOW_BAT.		; low battery flag
	clrf		DISP1			; display 1 (tens of minutes)	
	clrf		DISP2			; display 2 (minutes)	
	clrf		DISP3			; display 3 (tens of seconds)
	clrf		GAR_CNTR		; green, amber, red counter (0-3 for off, 1=green,2= amber and 3=red)
	clrf		ALT_DISP		; flag for alternative display (2-digit seconds or 3-digit minutes) flag. So colon is off when flag is set
	clrf		ALL_DIG_MIN	; flag for showng 3-digits of minutes when >99minutes
	clrf		DIM_DIR			; dimming direction main display
	clrf		DIM_DIR1		; dimming direction for warning LED display
	clrf		VOL_CNT_MS	; volume counter ms byte
	clrf		VOL_CNT_LS	; volume counter ls byte
	clrf		VOL_UP_DN		; volume up or down flag
	clrf		BUZZ			; buzzer sound flag
	clrf		ALM_SOUND	; flag for alarm
	clrf		SND_PERD		; sound period
	clrf		STORE_FLG1	; EEPROM storage flag
	clrf		STORE_FLG2	; EEPROM storage flag
	clrf		RUN_FLG		; reset or run/paused indication flag

; get stored values
	movlw	EEPROM1		; stored preset (0=nil, 1=1-2m, 2=2-3m, 3=3-5m, 4=5-7m: 5-14 = SPEECH1-10 settings)
	call		EEREAD
	movwf	PRESETS		; preset time periods
	movlw	EEPROM2		; stored dimming for Main display
	call		EEREAD
	movwf	DIM_MAIN
	movlw	EEPROM3		; stored dimming for LIGHTS display
	call		EEREAD
	movwf	DIM_LIGHTS		; coloured lights brightness

	movlw	EEPROM4		; stored alarm volume
	call		EEREAD
	movwf	VOLUME		; volume

	call		SET_PRESETS
	goto	INTERRUPT_ENABLE

; ______________________________________________________________
SET_PRESETS; (subroutine)
	bsf		PRES_FLG,0	; set preset flag
; set green, amber and red LEDs off
	clrf		GAR_CNTR		; Green, Amber & Red counter

; load displays with presets (based on PRESETS value) or 00:0 

; PRESETS=1 (Table Topics 1-2m, green at 1m, amber at 1:30m/s and red at 2m) disqualified <1m>2:30m/s
; PRESETS=2 (Evaluation      2-3m, green at 2m, amber at 2:30m/s and red at 3m) disqualified <1:30m>3:30m/s
; PRESETS=3 (Tall Tales       3-5m, green at 3m, amber at 4m          and red at 5m) disqualified <2:30m/s>5:30m/s
; PRESETS=4 (Humorous      5-7m, green at 5m, amber at 6m          and red at 7m) disqualified <4:30m/s>7:30m/s
; Humorous has the same presets as International
; Note: Display 2 shows a dash by using H'A' for the lookup table.
;______________________________________________________
; PRESETS=5 SPEECH1 10m 7, 9 & 10m for G,A,R
; PRESETS=6 SPEECH2 15m 10, 13 & 15m for G,A,R
; PRESETS=7 SPEECH3 20m 15, 18 & 20m for G,A,R
; PRESETS=8 SPEECH4 30m 20, 25 & 30m for G,A,R
; PRESETS=9 SPEECH5 40m 30, 35 & 40m for G,A,R
; PRESETS=10 SPEECH6 50m 40, 45 & 50m for G,A,R
; PRESETS=11 SPEECH7 60m 50, 55 & 60m for G,A,R
; PRESETS=12 SPEECH8 70m 60, 65 & 70m for G,A,R
; PRESETS=13 SPEECH9 80m 70, 75 & 80m for G,A,R
; PRESETS=14 SPEECH10 90m 80, 85 & 90m for G,A,R

; Note: Display 3 shows a ' (minutes symbol) by loading a C for the lookup table

	movf	PRESETS,w
	btfsc	STATUS,Z
	goto	NIL_PRESET
	bsf		COLON,0	; blank colon
	movf	PRESETS,w
	xorlw	D'01'
	btfsc	STATUS,Z
	goto	TABLE_TOPICS 
	movf	PRESETS,w
	xorlw	D'02'
	btfsc	STATUS,Z
	goto	EVALUATION
	movf	PRESETS,w
	xorlw	D'03'
	btfsc	STATUS,Z
	goto	TALL_TALES
	movf	PRESETS,w
	xorlw	D'04'
	btfsc	STATUS,Z
	goto	HUMEROUS
; speech 1-10
; load common settings
; seconds are cleared
	clrf		G_SEC	; green seconds
	clrf		A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
	movlw	H'C'		; ' minutes symbol
	movwf	DISP3

	movf	PRESETS,w
	xorlw	D'05'
	btfsc	STATUS,Z
	goto	SPEECH1
	movf	PRESETS,w
	xorlw	D'06'
	btfsc	STATUS,Z
	goto	SPEECH2
	movf	PRESETS,w
	xorlw	D'07'
	btfsc	STATUS,Z
	goto	SPEECH3
	movf	PRESETS,w
	xorlw	D'08'
	btfsc	STATUS,Z
	goto	SPEECH4
	movf	PRESETS,w
	xorlw	D'09'
	btfsc	STATUS,Z
	goto	SPEECH5
	movf	PRESETS,w
	xorlw	D'10'
	btfsc	STATUS,Z
	goto	SPEECH6
	movf	PRESETS,w
	xorlw	D'11'
	btfsc	STATUS,Z
	goto	SPEECH7
	movf	PRESETS,w
	xorlw	D'12'
	btfsc	STATUS,Z
	goto	SPEECH8
	movf	PRESETS,w
	xorlw	D'13'
	btfsc	STATUS,Z
	goto	SPEECH9
	movf	PRESETS,w
	xorlw	D'14'
	btfsc	STATUS,Z
	goto	SPEECH10

	movf	PRESETS,w
	xorlw	D'15'
	btfsc	STATUS,Z
	goto	MANUAL

SPEECH1 
; PRESETS=5 SPEECH1 10m 7, 9 & 10m for G,A,R
; load green, amber and red presets
	movlw	D'7'
	movwf	G_MIN	; green minutes
	movlw	D'9'
	movwf	A_MIN	; amber minutes
	movlw	D'10'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'1'		; 10m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH2
; PRESETS=6 SPEECH2 15m 10, 13 & 15m for G,A,R
; load green, amber and red presets
	movlw	D'10'
	movwf	G_MIN	; green minutes
	movlw	D'13'
	movwf	A_MIN	; amber minutes
	movlw	D'15'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'1'		; 15m
	movwf	DISP1
	movlw	D'5'		; 
	movwf	DISP2
	return
SPEECH3
; PRESETS=7 SPEECH3 20m 15, 18 & 20m for G,A,R
; load green, amber and red presets
	movlw	D'15'
	movwf	G_MIN	; green minutes
	movlw	D'18'
	movwf	A_MIN	; amber minutes
	movlw	D'20'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'2'		; 20m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH4
; PRESETS=8 SPEECH4 30m 20, 25 & 30m for G,A,R
; load green, amber and red presets
	movlw	D'20'
	movwf	G_MIN	; green minutes
	movlw	D'25'
	movwf	A_MIN	; amber minutes
	movlw	D'30'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'3'		; 30m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH5
; PRESETS=9 SPEECH5 40m 30, 35 & 40m for G,A,R
; load green, amber and red presets
	movlw	D'30'
	movwf	G_MIN	; green minutes
	movlw	D'35'
	movwf	A_MIN	; amber minutes
	movlw	D'40'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'4'		; 40m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH6
; PRESETS=10 SPEECH6 50m 40, 45 & 50m for G,A,R
; load green, amber and red presets
	movlw	D'40'
	movwf	G_MIN	; green minutes
	movlw	D'45'
	movwf	A_MIN	; amber minutes
	movlw	D'50'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'5'		; 50m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH7
; PRESETS=11 SPEECH7 60m 50, 55 & 60m for G,A,R
; load green, amber and red presets
	movlw	D'50'
	movwf	G_MIN	; green minutes
	movlw	D'55'
	movwf	A_MIN	; amber minutes
	movlw	D'60'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'6'		; 60m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH8
; PRESETS=12 SPEECH8 70m 60, 65 & 70m for G,A,R
; load green, amber and red presets
	movlw	D'60'
	movwf	G_MIN	; green minutes
	movlw	D'65'
	movwf	A_MIN	; amber minutes
	movlw	D'70'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'7'		; 70m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH9
; PRESETS=13 SPEECH9 80m 70, 75 & 80m for G,A,R
; load green, amber and red presets
	movlw	D'70'
	movwf	G_MIN	; green minutes
	movlw	D'75'
	movwf	A_MIN	; amber minutes
	movlw	D'80'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'8'		; 80m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
SPEECH10
; PRESETS=14 SPEECH10 90m 80, 85 & 90m for G,A,R
; load green, amber and red presets
	movlw	D'80'
	movwf	G_MIN	; green minutes
	movlw	D'85'
	movwf	A_MIN	; amber minutes
	movlw	D'90'
	movwf	R_MIN	; red minutes
; load displays
	movlw	D'9'		; 90m
	movwf	DISP1
	movlw	D'0'		; 
	movwf	DISP2
	return
	
HUMEROUS ; & INTERNATIONAL ;  5-7m, green at 5m, amber at 6 and red at 7m disqualified <4m>7:30
; load green, amber and red presets
	movlw	D'5'
	movwf	G_MIN	; green minutes
	movlw	D'6'
	movwf	A_MIN	; amber minutes
	movlw	D'7'
	movwf	R_MIN	; red minutes
	clrf		G_SEC	; green seconds
	clrf		A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
; load displays
	movlw	D'5'		; 5m
	movwf	DISP1
	movlw	H'A'		; shows a dash
	movwf	DISP2
	movlw	D'7'		; 7m
	movwf	DISP3
	return

TALL_TALES ; 3-5m, green at 3m, amber at 4 and red at 5m disqualified <2m>5:30
; load green, amber and red presets
	movlw	D'3'
	movwf	G_MIN	; green minutes
	movlw	D'4'
	movwf	A_MIN	; amber minutes
	movlw	D'5'
	movwf	R_MIN	; red minutes
	clrf		G_SEC	; green seconds
	clrf		A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
; load displays
	movlw	D'3'		; 3m
	movwf	DISP1
	movlw	H'A'		; shows a dash
	movwf	DISP2
	movlw	D'5'		; 5m
	movwf	DISP3
	return

EVALUATION ; 2-3m, green at 2m, amber at 2:30 and red at 3m disqualified <1m30s>5:30
; load green, amber and red presets
	movlw	D'2'
	movwf	G_MIN	; green minutes
	movlw	D'2'
	movwf	A_MIN	; amber minutes
	movlw	D'3'
	movwf	R_MIN	; red minutes
	clrf		G_SEC	; green seconds
	movlw	D'30'
	movwf	A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
; load displays
	movlw	D'2'		; 2m
	movwf	DISP1
	movlw	H'A'		; shows a dash
	movwf	DISP2
	movlw	D'3'		; 3m
	movwf	DISP3
	bsf		COLON,0	; blank colon
	return

TABLE_TOPICS ; 1-2m, green at 1m, amber at 1:30 and red at 2m disqualified <1m>2:30
; load green, amber and red presets
	movlw	D'1'
	movwf	G_MIN	; green minutes
	movlw	D'1'
	movwf	A_MIN	; amber minutes
	movlw	D'2'
	movwf	R_MIN	; red minutes
	clrf		G_SEC	; green seconds
	movlw	D'30'
	movwf	A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
; load displays
	movlw	D'1'		; 1m
	movwf	DISP1
	movlw	H'A'		; shows a dash
	movwf	DISP2
	movlw	D'2'		; 2m
	movwf	DISP3
	return

NIL_PRESET
	bcf		COLON,0	; show colon
; clear green, amber and red presets
	movlw	D'0'
	movwf	G_MIN	; green minutes
	movwf	A_MIN	; amber minutes
	movwf	R_MIN	; red minutes
	clrf		G_SEC	; green seconds
	clrf		A_SEC	; amber seconds
	clrf		R_SEC	; red seconds
; load displays
	movlw	D'0'		; 
	movwf	DISP1
	movlw	H'B'		; 0 with colon
	movwf	DISP2
	movlw	H'B'		; 0 with colon
	movwf	DISP3
	return

MANUAL

; load displays
; store display values
	movlw	EEPROM5		; DISP1 store
	call		EEREAD		; 
	movwf	DISP1
	bcf		COLON,0		; show colon
	movlw	EEPROM6		; DISP2 store
	call		EEREAD		; 
	movwf	DISP2
	movlw	EEPROM7		; DISP3 store
	call		EEREAD		; 
	movwf	DISP3

; load EEPROM warning values for green, amber and red LEDs
	movlw	EEPROM8		; green minutes
	call		EEREAD
	movwf	G_MIN			; green minutes 
	movlw	EEPROM9		; green seconds
	call		EEREAD
	movwf	G_SEC			; green seconds

	movlw	EEPROMA		; amber minutes
	call		EEREAD
	movwf	A_MIN			; amber minutes 
	movlw	EEPROMB		; amber seconds
	call		EEREAD
	movwf	A_SEC			; amber seconds

	movlw	EEPROMC		; red minutes
	call		EEREAD
	movwf	R_MIN			; red minutes 
	movlw	EEPROMD		; red seconds
	call		EEREAD
	movwf	R_SEC			; red seconds

CONV
; convert to binary and load MINUTES and SECONDS
	swapf	DISP1,w
	andlw	B'11110000'		; separate out ms nibble
	movwf	REG_2			; ms nibble of ls byte (REG_0, REG_1 set at 0)
	movf	DISP2,w
	andlw	B'00001111'		; get ls nibble
	iorwf	REG_2,f			; combine 
	call		BCD_BIN		; binary result in H_BYTE, L_BYTE
	movf	L_BYTE,w
	movwf	MINUTES
	swapf	DISP3,w
	andlw	B'11110000'
	movwf	REG_2
	call		BCD_BIN		; binary result in H_BYTE, L_BYTE
	movf	L_BYTE,w
	movwf	SECONDS
	return

; storage subroutines
; store display values
STORE_DISP_VAL
	clrf		STORE_FLG1	; clear EEPROM storage flag for timer
	movlw	D'15'			; only if PRESETS=15
	xorwf	PRESETS,w
	btfss	STATUS,Z
	return
	movlw	EEPROM5		; DISP1 store
	call		EEREAD		; sets EEADR
	movf	DISP1,w
	call		EEWRITE
	movlw	EEPROM6		; DISP2 store
	call		EEREAD		; sets EEADR
	movf	DISP2,w
	call		EEWRITE
	movlw	EEPROM7		; DISP3 store
	call		EEREAD		; sets EEADR
	movf	DISP3,w
	call		EEWRITE
	return
STORE_RED_WARN
	bcf		STORE_FLG2,0	; clear red warning LED reminder flag to store in EEPROM 
	movlw	D'15'			; only if PRESETS=15
	xorwf	PRESETS,w
	btfss	STATUS,Z
	return
	movlw	EEPROMC
	call		EEREAD		; sets EEADR
	movf	MINUTES,w
	call		EEWRITE		; write value
	movlw	EEPROMD
	call		EEREAD		; sets EEADR
	movf	SECONDS,w
	call		EEWRITE		; write value
	return
STORE_AMB_WARN
	bcf		STORE_FLG2,1	; clear amber warning LED reminder flag to store in EEPROM 
	movlw	D'15'			; only if PRESETS=15
	xorwf	PRESETS,w
	btfss	STATUS,Z
	return
	movlw	EEPROMA
	call		EEREAD		; sets EEADR
	movf	MINUTES,w
	call		EEWRITE		; write value
	movlw	EEPROMB
	call		EEREAD		; sets EEADR
	movf	SECONDS,w
	call		EEWRITE		; write value
	return
STORE_GRN_WARN
	bcf		STORE_FLG2,2	; clear green warning LED reminder flag to store in EEPROM
	movlw	D'15'			; only if PRESETS=15
	xorwf	PRESETS,w
	btfss	STATUS,Z
	return
	movlw	EEPROM8
	call		EEREAD		; sets EEADR
	movf	MINUTES,w
	call		EEWRITE		; write value
	movlw	EEPROM9
	call		EEREAD		; sets EEADR
	movf	SECONDS,w
	call		EEWRITE		; write value
	return
;_________________________________________________________________________
; interrupt enable 
INTERRUPT_ENABLE
	bcf		INTCON,TMR0IF	; clear TMR0 interrupt flag
	bsf		INTCON,TMR0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE		; set global interrupt enable for above
;_________________________________________________________________________

; Monitor input for an IR reception

; monitor portb,0 for input
HIGH_WAIT
	bcf		ACK,0		; Ack LED off
; wait for PORTB,0 to go high
	btfss	PORTB,0
	goto	HIGH_WAIT

; Wait for a low at PORTB,0
LOW_WAIT

	btfss	PORTB,0
	goto	START_ROUTINE
	bcf		INT_FLG,0 	; clear flag that indicates an interrupt ocurred

; check storage flags
	btfsc	STORE_FLG1,0	;  EEPROM storage flag for timer
	call		STORE_DISP_VAL; store display values
	btfsc	STORE_FLG2,0	; red warning LED reminder flag to store in EEPROM 
	call		STORE_RED_WARN ; store warning 
	btfsc	STORE_FLG2,1	; amber warning LED reminder flag to store in EEPROM 
	call		STORE_AMB_WARN ; store warning
	btfsc	STORE_FLG2,2	; green warning LED reminder flag to store in EEPROM
	call		STORE_GRN_WARN	; store warning

	goto	LOW_WAIT	

START_ROUTINE
	call		START		; decode IR code

; on return w has 00, 99,  FF or A0
	xorlw	H'A0'		; if A0 (A0k) code is correct
	btfss	STATUS,Z	
	goto	HIGH_WAIT	; code not correct

; check command value
; if values are (incorectly) shifted left (ie ls bits sent left) codes are as follows. Seems to be what is described 
; with Arduino decoding (address H10)  	
; Operate = HD8 ; A = HF8 ; B = H78 ; C	= H58 ; < = H10 ; O = H20 ; >	= H80 ; ^	= HA0 ; V = 00

; If shifted right as it should be (ie ls bit first sent right)codes are (address H08)
; Operate = H1B
; A = H1F
; B = H1E
; C	= H1A
; < = H08
; O = H04
; >	= H01
; ^	= H05
; V = H00

;A: Alarm
	movlw	H'1F'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	A_OUT
;B: Brightness (of warning LEDs)
	movlw	H'1E'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	B_OUT
;C: Colour (cycle through warning LEDs colours)
	movlw	H'1A'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	C_OUT
;^: Up (presets up or show seconds)
	movlw	H'05'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	UP
;<: Reset
	movlw	H'08'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LEFT
;O: Pause/Stop
	movlw	H'04'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	ZERO
;>: Start
	movlw	H'01'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	RIGHT
;V: Down (presets down or show seconds)
	movlw	H'00'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	DOWN
; Operate: (Brightness of main 7-segment displays)
	movlw	H'1B'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	OPERATE
; no valid commands	
	call		REPEAT		; when repeat codes end, go to start of checking for codes
	goto	HIGH_WAIT

;____________________________________________________________

OPERATE ; main display brightness
; change dimming value (1-50)
	incf		DIM_DIR,f		; dimming direction change
	btfss	DIM_DIR,0
	goto	DIM_DN_MAIN
DIM_UP_MAIN	
; if DIM_MAIN > 25 set up by 3
	movf	DIM_MAIN,w
	sublw	D'25'
	movlw	D'3'				; by 3 if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	DIM_RATE
CONT_DIM	
	movf	DIM_MAIN,w
	movwf	TEMP1
	incf		TEMP1,f
	movf	TEMP1,w
	sublw	D'50'
	movlw	D'50'
	btfss	STATUS,C
	movwf	TEMP1
	movf 	TEMP1,w
	movwf	DIM_MAIN

	decfsz	DIM_RATE,f
	goto	CONT_DIM
	call		REPEAT		; check repeat codes
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	DIM_UP_MAIN	; continue dimming change
	goto	DIM_MAIN_END	

DIM_DN_MAIN
; if DIM_MAIN > 25 set down by 3
	movf	DIM_MAIN,w
	sublw	D'25'
	movlw	D'3'				; by 3  if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	DIM_RATE
CONT_DIM1
	movf	DIM_MAIN,w
	movwf	TEMP1
	decf	TEMP1,f
	movf	TEMP1,w
	sublw	D'6'				; mimimum diming value
	movlw	D'6'				; 
	btfsc	STATUS,C
	movwf	TEMP1
	movf	TEMP1,w
	movwf 	DIM_MAIN

	decfsz	DIM_RATE,f
	goto	CONT_DIM1
	call		REPEAT		; check repeat codes
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	DIM_DN_MAIN	; continue dimming change
;	goto		DIM_MAIN_END	

DIM_MAIN_END
; store in EEPROM	
	movlw	EEPROM2		; stored dimming for Main display
	call		EEREAD
	movf	DIM_MAIN,w
	call		EEWRITE
	goto	HIGH_WAIT
;________________________________________________________________

RIGHT ; start clock 
; if paused and PRESETS=15 convert DISP1,2,3 to Minutes, seconds (call CONV)
	movf	PAUSE,w
	btfss	STATUS,Z
	goto	BY_CONV1
	movf	PRESETS,w
	xorlw	D'15'
	btfss	STATUS,Z
	goto	BY_CONV1
	btfsc	RUN_FLG,0
	goto	BY_CONV1
	call		CONV			; display to Min/Sec
	bsf		RUN_FLG,0		; set to prevent CONV untill reset
BY_CONV1
; if PRESET flag set then call DISPLAY_CONV to show mins/sec
	btfsc	PRES_FLG,0
	call		DISPLAY_CONV	; show mins and seconds
	clrf		PRES_FLG		; preset flag cleared on clock start
	bsf		PAUSE,0		; start clock
REP1
	call		REPEAT		; check repeat codes
	goto	HIGH_WAIT
;__________________________________________________________
ZERO ; pause clock

; if paused and PRESETS=15 convert DISP1,2,3 to Minutes, seconds (call CONV)
	movf	PAUSE,w
	btfss	STATUS,Z
	goto	BY_CONV
	movf	PRESETS,w
	xorlw	D'15'
	btfss	STATUS,Z
	goto	BY_CONV
	btfsc	RUN_FLG,0
	goto	BY_CONV
	call		CONV			; display to Min/Sec
	bsf		RUN_FLG,0		; set to prevent CONV untill reset
BY_CONV
; if PRESET flag set then call DISPLAY_CONV to show mins/sec
	btfsc	PRES_FLG,0
	call		DISPLAY_CONV	; show mins and seconds
	clrf		PAUSE			; stop clock
	bcf		COLON,0		; show colon
	clrf		PRES_FLG		; presets flag clear so display is the minutes
	goto	REP1			; REP1 is  in the 'RIGHT'  action
;___________________________________________________________
LEFT ; reset
	clrf		PAUSE			; clock paused
	clrf		MINUTES
	clrf		SECONDS
	clrf		CK_COUNT_LS	; 500ms counter
	clrf		CK_COUNT_MS
	clrf		RUN_FLG
; when resetting clock
	clrf		ALL_DIG_MIN	; 3-digits of minutes when >99minutes
	clrf		ALT_DISP		; alternative display (2-digit seconds or 3-digit minutes) flag. So colon is off when flag is set
	call		SET_PRESETS	; set the initial preset values based on PRESETS value
	goto	REP1			; REP1 is found in the  'RIGHT' routine above
;___________________________________________________________

DOWN
; if presets (PRES_FLG,0 set) scroll down presets 15-0 
	btfss	PRES_FLG,0
	goto	ALT1

; if PRESETS=D15 and DISP1,2&3 are not 0 do not decrease PRESETS but DISP1,DISP2 and DISP3 instead down to 00:0
	movlw	D'15'	
	xorwf	PRESETS,w
	btfsc	STATUS,Z
	goto	DEC_DISP
RED_PRSTS
	clrf		MINUTES
	clrf		SECONDS
	movlw	D'1'	
	subwf	PRESETS,w
	btfsc	STATUS,C		; if negative stop
	decf	PRESETS,f

	call		SET_PRESETS	; set the initial preset values based on PRESETS value
; store in EEPROM	
	movlw	EEPROM1		; stored presets
	call		EEREAD
	movf	PRESETS,w
	call		EEWRITE
	goto	REP1
; if presets clear change ALT_DISP (inc or dec)
ALT1
	movlw	D'1'
	movwf 	ALT_DISP	; show alternative display
	call		DISPLAY_CONV	; display conversion
	call		REPEAT	; wait for repeat IR transmission
;  if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	ALT1		; continue alt change
	clrf		ALT_DISP	; return to normal display	
	call		DISPLAY_CONV	; display conversion
	goto	HIGH_WAIT

DEC_DISP
	movlw	D'3'
	movwf	FASTR		; timer up/down rate
DEC_DISP1
; check if zero
	movf	DISP1,w
	btfss	STATUS,Z
	goto	DEC_DISP_VAL	; not 0
	movf	DISP2,w
	btfss	STATUS,Z
	goto	DEC_DISP_VAL	; not 0
	movf	DISP3,w
	btfss	STATUS,Z
	goto	DEC_DISP_VAL
	clrf		SECONDS
	bsf		STORE_FLG1,0	; set EEPROM storage flag for timer as a reminder to store display values
	goto	RED_PRSTS	; All 0
DEC_DISP_VAL
; decrease DISP
	movf	DISP3,w
	btfsc	STATUS,Z
	goto	DEC2
	decf	DISP3,f 
	goto	REP_X
DEC2
	movlw	D'5'
	movwf	DISP3			; set at 5
	movf	DISP2,w
	btfsc	STATUS,Z
	goto	DEC1
	decf	DISP2,f
	goto	REP_X
DEC1
	movlw	D'9'
	movwf	DISP2
	movf	DISP1,w
	btfss	STATUS,Z
	decf	DISP1,f
REP_X
	clrf		SECONDS
	bsf		STORE_FLG1,0	; set EEPROM storage flag for timer as a reminder to store display values

; check if any warning LED is on
	movf	GAR_CNTR,w	; warning LED counter (0=off, 1=green, 2=amber, 3=red)  
	btfsc	STATUS,Z		; if clear,  no LEDs on
	goto	REP_X_GO
; get minutes and seconds value
	call		CONV			; convert from DISP1,2,3 to MIN/SEC
	movf	GAR_CNTR,w
	xorlw	D'01'			; green?
	btfsc	STATUS,Z
	goto	GRN_X
	movf	GAR_CNTR,w
	xorlw	D'02'			; amber?
	btfsc	STATUS,Z
	goto	AMB_X
	movf	GAR_CNTR,w
	xorlw	D'03'			; red?
	btfss	STATUS,Z
	goto	REP_X_GO
RED_X
	bsf		STORE_FLG2,0	; red warning LED reminder  flag to store in EEPROM
	goto	REP_X_GO
AMB_X
	bsf		STORE_FLG2,1	; amber warning LED reminder flag to store in EEPROM 
	goto	REP_X_GO
GRN_X
	bsf		STORE_FLG2,2	; green warning LED reminder flag to store in EEPROM
;	goto		REP_X_GO
REP_X_GO

; if DISP1-3 are zero, stop fast down
	movf	DISP1,w
	btfss	STATUS,Z	
	goto	DISP_NOT0
	movf	DISP2,w
	btfss	STATUS,Z	
	goto	DISP_NOT0
	movf	DISP3,w
	btfsc	STATUS,Z	
	goto	HIGH_WAIT
DISP_NOT0
	call		REPEAT		; check repeat codes
	xorlw	H'FF'			; if w=FF then repeat still in process
	btfss	STATUS,Z
	goto	HIGH_WAIT
	movf	FASTR,w		; timer up/down rate
	btfsc	STATUS,Z		; if zero faster rate
	goto	DEC_DISP1		; continue decreasing
	decf	FASTR,f
	goto	REP_X
;____________________________________________________________________________
UP
; if presets (PRES_FLG,0 set) scroll up presets 0-15 
 
	btfss	PRES_FLG,0
	goto	ALT1			; if presets clear (ALT1 is in down section above)

; if PRESETS=D15 increase DISP1,DISP2 and DISP3 
	movlw	D'15'	
	xorwf	PRESETS,w
	btfsc	STATUS,Z
	goto	INC_DISP
; increment presets
	movlw	D'1'	
	addwf	PRESETS,w
	sublw	D'15'
	btfss	STATUS,C		; if negative, stop
	goto	PRES15
	incf		PRESETS,f

	movlw	D'15'
	xorwf	PRESETS,w
	btfss	STATUS,Z
	goto	SET1
PRES15
	clrf		MINUTES
	clrf		SECONDS
	clrf		DISP1
	clrf		DISP2
	clrf		DISP3
SET1
	call		SET_PRESETS	; set the initial preset values based on PRESETS value
; store in EEPROM	
	movlw	EEPROM1		; stored presets
	call		EEREAD
	movf	PRESETS,w
	call		EEWRITE
	goto	REP1
INC_DISP
	movlw	D'3'
	movwf	FASTR		; timer up/down rate
INC_DISP1
; increase DISP
	movlw	D'1'
	addwf	DISP3,w
	sublw	D'5'
	btfss	STATUS,C
	goto	INC2
	incf		DISP3,f
	goto	REP_Y
INC2
	clrf		DISP3
	movlw	D'1'
	addwf	DISP2,w
	sublw	D'9'
	btfss	STATUS,C
	goto	INC1
	incf		DISP2,f
	goto	REP_Y
INC1
; if DISP1 is 9 then do not clear
	movf	DISP1,w
	xorlw	D'9'
	btfss	STATUS,Z
	clrf		DISP2
	movlw	D'1'
	addwf	DISP1,w
	sublw	D'9'
	btfsc	STATUS,C
	incf		DISP1,f

REP_Y
; store display values
	clrf		SECONDS
	bsf		STORE_FLG1,0	; set EEPROM storage flag for timer

; check if any warning LED is on
	movf	GAR_CNTR,w	; warning LED counter (0=off, 1=green, 2=amber, 3=red)  
	btfsc	STATUS,Z		; if clear,  no LEDs on
	goto	REP_Y_GO
; get minutes and seconds value
	call		CONV			; convert from DISP1,2,3 to MIN/SEC
	movf	GAR_CNTR,w
	xorlw	D'01'			; green?
	btfsc	STATUS,Z
	goto	GRN_Y
	movf	GAR_CNTR,w
	xorlw	D'02'			; amber?
	btfsc	STATUS,Z
	goto	AMB_Y
	movf	GAR_CNTR,w
	xorlw	D'03'			; red?
	btfss	STATUS,Z
	goto	REP_Y_GO
RED_Y
	bsf		STORE_FLG2,0	; red warning LED reminder to store in EEPROM flag
	goto	REP_Y_GO
AMB_Y
	bsf		STORE_FLG2,1	; amber warning LED reminder flagto store in EEPROM 
	goto	REP_Y_GO
GRN_Y
	bsf		STORE_FLG2,2	; green warning LED reminder flag to store in EEPROM
;	goto		REP_Y_GO	
REP_Y_GO
	call		REPEAT		; check repeat codes
	xorlw	H'FF'			; if w=FF then repeat still in process
	btfss	STATUS,Z
	goto	HIGH_WAIT
	movf	FASTR,w		; timer up/down rate
	btfsc	STATUS,Z		; if zero faster rate
	goto	INC_DISP1		; continue increasing
	decf	FASTR,f
	goto	REP_Y

;______________________________________________________________________:
A_OUT ; alarm
; set volume counter
	movlw	D'50'			; ~ 5.5s
	movwf	VOL_TMR		; volume timer starts changing volume after a set period of sound
;  volume counter, allows volume change immediately if counter is not 0	
	movf	VOL_CNT_MS,w
	btfsc	STATUS,Z
	goto	START_SOUND
	bsf		BUZZ,0			; allow buzzer sound 
	goto	REPV2			; bypass start sound and adjust volume with sound on

START_SOUND
	
	clrf		WARBLE
; start PWM for sound
	movf	VOLUME,w		; sets duty cycle
	movwf	CCPR1L		;  PWM
REP2
	movf	VOL_TMR,w
	btfsc	STATUS,Z
	goto	REPV
	decfsz	VOL_TMR,f		; adjust volume
	goto	SOUND

; adjust volume
REPV;
	clrf		BUZZ			; stop modulation
REPV2
	incf		VOL_UP_DN	,f	; volume up or down direction
REPV1
	btfss	BUZZ,0
	goto	CK_VOL
; decrease volume to Minimum volume, or increase to 127 (Maximum).
	btfss	VOL_UP_DN,0
	goto	VOL_UP
; volume down
; if VOLUME > 25 set down by 3
	movf	VOLUME,w
	sublw	D'25'
	movlw	D'6'				; by 6  if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	VOL_RATE
CONT_VOL1
	movf	VOLUME,w
	movwf	TEMP1
	decf	TEMP1,f
	movf	TEMP1,w
	sublw	D'3'				; mimimum value
	movlw	D'3'				; 
	btfsc	STATUS,C
	movwf	TEMP1
	movf	TEMP1,w
	movwf 	VOLUME

	decfsz	VOL_RATE,f
	goto	CONT_VOL1
	goto	CK_VOL

VOL_UP
; if VOLUME > 25 set up by 3
	movf	VOLUME,w
	sublw	D'25'
	movlw	D'6'				; by 6 if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	VOL_RATE
CONT_VOL	
	movf	VOLUME,w
	movwf	TEMP1
	incf		TEMP1,f
	movf	TEMP1,w
	sublw	D'127'			; maximum volume
	movlw	D'127'
	btfss	STATUS,C
	movwf	TEMP1
	movf 	TEMP1,w
	movwf	VOLUME

	decfsz	VOL_RATE,f
	goto	CONT_VOL

CK_VOL
	btfsc	BUZZ,0
	goto	SKIP_VOL
	movf	VOLUME,w
	movwf	CCPR1L		; adjust PWM
SKIP_VOL
	call		REPEAT 
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	REPV1

; stop PWM
	clrf		BUZZ			; stop warble sound
	clrf		CCPR1L		; duty 0% 
; store in EEPROM4 when complete	
	movlw	EEPROM4
	call		EEREAD		; sets EEADR
	movf	VOLUME,w
	call		EEWRITE
; set counter so that volume change can be immediate if counter is not zero (allows immediate change between volume up or down with a new A press)
	movlw	D'2'				; value x 640ms
	movwf	VOL_CNT_MS	; volume counter ms byte
	clrf		VOL_CNT_LS	; volume counter ls byte
	goto	HIGH_WAIT

SOUND; (ALARM sound)
	bsf		BUZZ,0			; set BUZZer flag
	call		REPEAT		; check repeat codes

; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfss	STATUS,Z
	goto	CEASE			; end of tone

	goto	REP2
CEASE
; stop PWM
	clrf		BUZZ			; BUZZer flag
	clrf		CCPR1L		; duty 0% 
	goto	HIGH_WAIT
;__________________________________________________________________________
B_OUT ; Lights brightness 
; change dimming value (1-50)
	incf		DIM_DIR1,f		; dimming direction change
	btfss	DIM_DIR1,0
	goto	DIM_DN_LIGHTS
DIM_UP_LIGHTS	
; if DIM_MAIN > 25 set up by 3
	movf	DIM_LIGHTS,w
	sublw	D'25'
	movlw	D'3'				; by 3 if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	DIM_RATE
CONT_DIM3	
	movf	DIM_LIGHTS,w
	movwf	TEMP1
	incf		TEMP1,f
	movf	TEMP1,w
	sublw	D'50'			; maximum brightness
	movlw	D'50'
	btfss	STATUS,C
	movwf	TEMP1
	movf 	TEMP1,w
	movwf	DIM_LIGHTS

	decfsz	DIM_RATE,f
	goto	CONT_DIM3
	call		REPEAT		; check repeat codes
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	DIM_UP_LIGHTS	; continue dimming change
	goto	DIM_LIGHTS_END	

DIM_DN_LIGHTS
; if DIM_LIGHTS > 25 set down by 3
	movf	DIM_LIGHTS,w
	sublw	D'25'
	movlw	D'3'				; by 3  if >25
	btfsc	STATUS,C
	movlw	D'1'				; or 1 if <
	movwf	DIM_RATE
CONT_DIM4
	movf	DIM_LIGHTS,w
	movwf	TEMP1
	decf	TEMP1,f
	movf	TEMP1,w
	sublw	D'6'				; mimimum diming value
	movlw	D'6'				; 
	btfsc	STATUS,C
	movwf	TEMP1
	movf	TEMP1,w
	movwf 	DIM_LIGHTS

	decfsz	DIM_RATE,f
	goto	CONT_DIM4
	call		REPEAT		; check repeat codes
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	DIM_DN_LIGHTS	; continue dimming change
;	goto		DIM_LIGHTS_END	

DIM_LIGHTS_END
; store in EEPROM	
	movlw	EEPROM3		; stored dimming for Lights display
	call		EEREAD
	movf	DIM_LIGHTS,w
	call		EEWRITE
	goto	HIGH_WAIT

;________________________________________________________________________________
C_OUT ; colour change/shift of green, amber and red lights
; increase GAR_CNTR 0-3
	incf		GAR_CNTR,w
	sublw	D'03'			; check against maximum of 3
	clrw
	btfsc	STATUS,C		; if >3 then clear
	incf		GAR_CNTR,w
	movwf	GAR_CNTR

; check if any warning LED is on
	movf	GAR_CNTR,w	; warning LED counter (0=off, 1=green, 2=amber, 3=red)  
	btfsc	STATUS,Z		; if clear,  no LEDs on
	goto	REP3
; get minutes and seconds value
	call		CONV			; convert from DISP1,2,3 to MIN/SEC
	movf	GAR_CNTR,w
	xorlw	D'01'			; green?
	btfsc	STATUS,Z
	goto	GRN_C
	movf	GAR_CNTR,w
	xorlw	D'02'			; amber?
	btfsc	STATUS,Z
	goto	AMB_C
	movf	GAR_CNTR,w
	xorlw	D'03'			; red?
	btfss	STATUS,Z
	goto	REP3
RED_C
	call		STORE_RED_WARN		; store red warning in EEPROM
	goto	REP3
AMB_C
	call		STORE_AMB_WARN		; store amber warning in EEPROM 
	goto	REP3
GRN_C
	call		STORE_GRN_WARN		; store green warning in EEPROM
;	goto		REP3
REP3
	call		REPEAT		; check repeat codes
; if w=FF repeat still running; if w=0 repeat ended
	xorlw	H'FF'
	btfsc	STATUS,Z
	goto	REP3
	goto	HIGH_WAIT

; ************************************************
; IR subroutines

; read IR signal that's based on pulse distance protocol
; start train is a 38kHz burst at 9ms and 4.5ms of no burst
; data sent as an 8-bit address (LSB first) then a complementary version of the address
; further data sent as an 8-bit command (LSB first) then a complementary version of the command
; logic 1 is 560us 38kHz burst and 1690us of no burst
; logic 0 is a 560us 38kHz burst and 560us of no burst
; At the end of the 32bit data is a tail pulse comprising 560us of a 38kHz burst
; data is sent once
; a repeat indication (transmit button held on) is 9000us 38kHz burst followed by 2250us no burst and a 560us 38kHz burst
; this is repeated each 110ms
; **********************************


START ; start of Infra Red reception
; start 110ms counter at beginning of code
	clrf		msCOUNT_MS	; 110ms counter
	clrf		msCOUNT_LS

; Check for start train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 4.5ms (no 38kHz signal)
; count 180 x 50us interrupts where PORTB,0 needs to stay low
; 
	movlw	D'160'	; allow an 8ms minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0
	bsf		ACK,0		; Ack LED on
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0
; PORTB,0 can now go high up to a period of 2ms
; wait for more interrupts
	movlw	D'40'	; 2ms
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_1
	btfsc	PORTB,0
	goto	NINE_ms		; PORTB,0 has gone high within the acceptable ~9ms low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_1
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_1
	retlw 	H'FF' 		; if PORTB,0 stays low  

NINE_ms	; PORTB,0 has stayed low and then gone high within the acceptable ~9ms low period
; Check for 4.5ms high
	bcf		ACK,0		; Ack LED off
	movlw	D'80'		; allow a 4ms minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_2
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_2 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_2

; PORTB,0 can now go low up to a period of 1ms
; wait for more interrupts
	movlw	D'20'		; 1ms
	movwf	COUNTER
; PORTB,0 can go low in this period
CK_B0_3
	btfss	PORTB,0
	goto	FOUR_POINT_FIVE_ms	; PORTB,0 has gone low within the acceptable ~4.5ms high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_3
	retlw	H'00'		; PORTB stays high

; end of start train check
FOUR_POINT_FIVE_ms
	
	bsf		ACK,0	; Ack LED on
	movlw	D'32'		; set up 32bit counter
	movwf	BIT_COUNT

START_DATA ; start of data 
; clear data
	clrf		REMOTE_A	; remote control Address byte
	clrf		REM_A_BAR	; complementary value of address
	clrf		REMOTE_C	; remote control Command byte
	clrf		REM_C_BAR	; complementary value of command

; start reading data
READ_DATA
; a 560us low
	movlw	D'8'		; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_4
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_4 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_4
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'		; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_5
	btfsc	PORTB,0
	goto	ONE_ZERO	; PORTB,0 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_5
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_5
	retlw 	H'FF' 		;  PORTB,0 stayss low  
; test for a zero (high for 560us) or a one (high for 1690us)
ONE_ZERO
	bcf		ACK,0		; Ack LED off
; Check for 560us high
	movlw	D'8'		; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_6
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_6 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_6

; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'		; 100us
	movwf	COUNTER

; PORTB,0 can go low in this period
CK_B0_7
	btfss	PORTB,0
	goto	LOGIC_0		; PORTB,0 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_7
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_7
	goto	CHECK_FOR_1

LOGIC_0	; logic zero detected
	bsf		ACK,0		; Ack LED on
	bcf		STATUS,C	; clear carry
; shift detected data into storage
SHIFT_DATA	
	rrf		REM_C_BAR,f	; complementary value of command
	rrf		REMOTE_C,f	; remote control Command byte
	rrf		REM_A_BAR,f	; complementary value of address
	rrf		REMOTE_A,f	; remote control Address byte
	

	decfsz	BIT_COUNT,f	; shift data for 32 bits
	goto	READ_DATA	; contine reading data
	goto	TAIL		; end of 32 bit data, check for the tail (560us of burst)
	
CHECK_FOR_1 ; check for a logic 1

; Check for 1690us high  (already 560us high) so check for 1130us
	movlw	D'15'	; allow a 1100s minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_8
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_8 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_8

; PORTB,0 can now go high up to a period of 50us extra
; wait for more interrupts
	movlw	D'10'	; 50us
	movwf	COUNTER

; PORTB,0 can go low in this period
CK_B0_9
	btfss	PORTB,0
	goto	LOGIC_1		; PORTB,0 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_9
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_9
	retlw	H'00'		; PORTB stays high	

LOGIC_1
	bsf		ACK,0		; Ack LED on
	bsf		STATUS,C	; data is a 1
	goto	SHIFT_DATA

TAIL ; end of data tail

; a 560us low
	movlw	D'8'	; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_10
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_10 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_10
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'	; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_11
	btfsc	PORTB,0
	goto	TAIL_DET	; PORTB,0 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_11
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_11
	retlw 	H'FF' 		; if PORTB,0 stays low  

TAIL_DET ; tail is detected
	bcf		ACK,0		; Ack LED off

; check if Address and Address bar are complementary

	comf	REM_A_BAR,w	; complementary value of address
	xorwf	REMOTE_A,w	; remote control Address byte
	btfss	STATUS,Z
	retlw	H'00'			; PORTB high

; ********************************************

; For the particular remote that uses 08 as address,
; check for a 08 to specify the IR transmitter used or change if different
; or remark out if for use with other IR remotes with different address
	movlw	H'08'
	xorwf	REMOTE_A,w	; value of address
	btfss	STATUS,Z
	retlw	H'99'			; indicate a goto	HIGH_WAIT
; *******************************************
; check if Command and Command bar are complementary

	comf	REM_C_BAR,w	; complementary value of command
	xorwf	REMOTE_C,w	; remote control Command byte
	btfss	STATUS,Z
	retlw	H'99'			; indicate a goto	HIGH_WAIT

	retlw	H'A0'			; if data is decoded correctly send an 'AO'k return message

;------------------------------------------------------------------------
REPEAT
	movlw	D'3'
REPEATx
	movwf	RPT
; wait for a low after 110ms (see msCOUNT_MS msCOUNT_LS	; 110ms counter)

; REPEAT is a 9ms burst followed by 2250us off and 560us burst repeated 110ms apart
; wait for mscount within a 110ms range
REPEAT1
KEEP_CHK_COUNT
	movf	msCOUNT_MS,w
	sublw	H'07'
	btfsc	STATUS,C
	goto	KEEP_CHK_COUNT

; wait for PORTB,0 to go low
KEEP_CHK_LOW
; check if ms count is less than H0A00
	movf	msCOUNT_MS,w
	sublw	H'0A'
	btfss	STATUS,C
	goto	REP_OFF	; repeat ended
	btfsc	PORTB,0	; when low check if within time frame allowed
	goto	KEEP_CHK_LOW

; start 110ms counter at beginning of repeat frame
	clrf		msCOUNT_MS	; 110ms counter
	clrf		msCOUNT_LS
; Check for repeat train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 2.25ms (no 38kHz signal) then a 560us burst
; count 180 x 50us interrupts where PORTB,0 needs to stay low
; 
	movlw	D'160'	; allow an 8ms minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_12
	bsf		ACK,0		; Ack LED on
	btfsc	PORTB,0
	goto	REP_OFF	; repeat ended if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_12 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_12
; PORTB,0 can now go high up to a period of 2ms
; wait for more interrupts
	movlw	D'40'	; 2ms
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_13
	btfsc	PORTB,0
	goto	NINE_ms_1	; PORTB,0 has gone high within the acceptable ~9ms low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_13
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_13
	goto	REP_OFF	; repeat ended

NINE_ms_1

	bcf		ACK,0		; Ack LED off
; Check for 2.25ms high
	movlw	D'40'	; allow a 2ms minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_14
	btfss	PORTB,0
	goto	REP_OFF	; repeat ended; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_14 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_14

; PORTB,0 can now go low up to a period of 500us
; wait for more interrupts
	movlw	D'10'	; 500us
	movwf	COUNTER
; PORTB,0 can go low in this period
CK_B0_15
	btfss	PORTB,0
	goto	TWOPOINTTWOFIVE_ms	; PORTB,0 has gone low within the acceptable ~2.25ms high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_15
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_15
	goto	REP_OFF		; repeat ended
TWOPOINTTWOFIVE_ms
; a 560us low
	bsf		ACK,0		; Ack LED on
	movlw	D'8'	; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_16
	btfsc	PORTB,0
	goto	REP_OFF	; repeat ended  if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_16 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_16
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'	; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_17
	btfsc	PORTB,0
	goto	REP_DET	; 
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_17
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_17
	goto	REP_OFF	; repeat ended  
REP_DET
	bcf		ACK,0
;	goto		REPEAT1
	retlw	H'FF'		 ; repeat still running ; PORTB,0 has gone high within the acceptable ~560us low period)
REP_OFF
	bcf		ACK,0
	decfsz	RPT,f
	goto	REPEAT1
	retlw	H'00'		; repeat off
		  	
; **********************************************************

; general subroutines
DELAY
	movlw	D'10'
	movwf	STORE3
DELAY1
	movlw	D'255'	;
	movwf	STORE1
LOOP2
	movlw	D'255'
	movwf	STORE2
LOOP1
	nop
	nop
	decfsz	STORE2,f
	goto	LOOP1
	decfsz	STORE1,f
	goto	LOOP2
	decfsz	STORE3,f
	goto	DELAY1
	return


; subroutine to read EEPROM memory 

EEREAD
	bcf		STATUS,RP0	; select bank 0
	bsf 		STATUS,RP1	; select memory bank 2
	movwf 	EEADR			; indirect special function register
	bsf 		STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD		; read EEPROM
	bcf 		STATUS,RP0	; select memory bank 2
	movf	EEDATA,w		; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM
EWRITE
EEWRITE	
	bcf 		STATUS,RP0	; bank 0
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf 		STATUS,RP0	; select memory bank 3
WR3	
	btfsc	EECON1,WR	; check if write complete 
	goto 	WR3			; not written yet
	bcf		INTCON,GIE		; disable interrupts
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'			; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'			; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
	bsf 		INTCON,GIE		; enable interrupts
WRITE
	btfsc	EECON1,WR	; skip if write complete 
	goto 	WRITE			; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP1	; 
	bcf 		STATUS,RP0	; select memory bank 0
	return					; value written 

; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN_0  
; Result in BCD is in BCD_0 & BCD_1.  
; BCD_0 is MSB, BCD_1 is LSB

BCD
	bcf		STATUS,C	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf		BCD_0
	clrf		BCD_1		; set BCD registers to 0 
LOOPBCD
	rlf		BIN_0,f		; shift left binary registers
	rlf		BCD_1,f		; MSB shift left
	rlf		BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ	; continue decimal adjust
	return				; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ
	movlw	BCD_1		; BCD LSB address
	movwf	FSR		; pointer for BCD1
	call		ADJBCD	; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call		ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD
ADJBCD
	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return				; end subroutine


BCD_BIN
; BCD to Binary; REG_0, REG_1, REG_2; BCD values (MS to LS bytes)
; H_BYTE, L_BYTE is binary value 

; clear initial registers as not using full capability of 5-digit BCD to 16-bit binary conversion

	clrf		REG_0		; high BCD number 
	clrf		REG_1		; mid BCD number
; start conversion

	clrf		H_BYTE		; high order binary result
	movf	REG_0,w
	andlw	H'0F'		; mask high nibble
	movwf	L_BYTE 
	call		TIMES_A
	swapf	REG_1,w
	call		TIMES_B
	movf	REG_1,w
	call		TIMES_B
	swapf	REG_2,w
	call		TIMES_B
	movf	REG_2,w
	andlw	H'0F'
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf		H_BYTE,f
	return		

TIMES_B
	andlw	H'0F'
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf		H_BYTE,f
TIMES_A
	bcf		STATUS,C
	rlf		L_BYTE,w	; times 2
	movwf	L_TEMP
	rlf		H_BYTE,w
	movwf	H_TEMP
; times 8
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
	bcf		STATUS,C
	rlf		L_BYTE,f		
	rlf		H_BYTE,f
; to mult by 10
	movf	L_TEMP,w
	addwf	L_BYTE,f
	btfsc	STATUS,C
	incf		H_BYTE,f
	movf	H_TEMP,w
	addwf	H_BYTE,f
	return	


	
	end


	
