; Processor pin allocations are as follows:
; Duty Cycle Meter
; shows % duty cycle on 2-digit display

; ports are allocated as follows 
 
; RA0 Output driving decimal point
; RA1 Output for relay
; RA2 Output disp 1
; RA3 Output disp 2
; RA4 Input from switches

; RB0 PWM input 
; RB1 e segment drive 
; RB2 d segment drive
; RB3 c segment drive
; RB4 b segment drive
; RB5 a segment drive
; RB6 f segment drive
; RB7 g segment drive

; CPU configuration
; 	
	list P=16F84A
	#include "p16f84A.inc"
	__config _XT_OSC & _WDT_OFF & _PWRTE_ON

; Define variables at memory locations

; EEPROM DATA


EEPROM1		equ	H'00'	; non-volatile storage for polarity edge 
EEPROM2		equ	H'01'	; non-volatile storage for detect threshold high or low
EEPROM3		equ	H'02'	; non-volatile storage for threshold level
EEPROM4		equ	H'03'	; non-volatile storage for hysteresis

; RAM

DISP1		equ	H'0C'	; working storage for Display1 value 
DISP2		equ	H'0D'	; working storage for Display2 value 
MODE		equ	H'0E'	; display mode
TEST_L		equ	H'0F'	; w store to test bit 0
STATUS_TMP 	equ 	H'10'	; temp storage for status during interrupt
W_TMP		equ	H'11'	; temporary storage for w during interrupt
VALUE_1		equ 	H'12'	; delay value storage			
VALUE_2		equ	H'13'	; delay value storage
FLAG_1		equ	H'14'	; 1, high pulse started; 2, low pulse started; 3, finished
FLAG_0		equ	H'15'	; bit 0 overrange flag
CNT_8		equ	H'16'	; BCD conversion counter 
TEMP		equ	H'17'	; temporary storage during BCD conversion
TEMP_1		equ	H'18'	; temporary working storage
TEMP_2		equ	H'19'	; temp storage

BCD_0		equ	H'1A'	; bcd MS value after conversion
BCD_1		equ	H'1B'	; bcd LS value
BIN_0		equ	H'1C'	; binary value for BCD conversion
PORT_A		equ	H'1D'   ; temp store of PORTA 
COUNT1		equ	H'1E'	; counter for high pulse ls byte
COUNT2		equ	H'1F'	; counter for high pulse ms byte
COUNT3		equ	H'20'	; counter for low pulse ls byte
COUNT4		equ	H'21'	; counter for low pulse ms byte
COUNT5		equ	H'22'	; latched counter for high pulse ls byte
COUNT6 		equ	H'23'	; latched counter for high pulse ms byte
COUNT7		equ	H'24'	; latched counter for low pulse ls byte
COUNT8		equ	H'25'	; latched counter for low pulse ms byte	
COUNT_5		equ	H'26'	; latched counter for high pulse ls byte store
COUNT_6 	equ	H'27'	; latched counter for high pulse ms byte store
COUNTO		equ	H'28'	; latched counter for timeout ls byte
COUNTT	 	equ	H'29'	; latched counter for Timeout ms byte 
DELCNT		equ	H'2A'	; delay counter
MPLX		equ	H'2B'	; multiplex counter	
TEMPW		equ	H'2C'	; store W when reading EEPROM

; math routines

TEMPB1          equ     0x31
TEMPB0          equ     0x32
TEMPD           equ     0x33    ; temporary storage
AARGB3		equ	0x34
AARGB2          equ     0x35
AARGB1          equ     0x36
AARGB0          equ     0x37
AARG            equ     0x37    ; most significant byte of argument A
BARGB1          equ     0x38
BARGB0          equ     0x39
BARG            equ     0x39    ; most significant byte of argument B
REMB1           equ     0x3A
REMB0           equ     0x3B    ; most significant byte of remainder
LOOPCOUNT       equ     0x3C    ; loop counter
STO1		equ	0x3D	; AARGB1 storage
STO2		equ	0x3E	; AARGB2 storage

; preprogram EEPROM DATA (00-3F from 2100-213F)
	
	ORG     2100		; start at 00
	
	DE	D'00', D'00', D'50', D'05'


; define reset and interrupt vector start addresses

	org	0	  	; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4		; interrupt vector 0004h, start interrupt routine here
	goto	INTRUPT		; go to start of interrupt routine, bypass subroutines

; subroutine to get seven segment display data.  
; RB7-RB1----gfabcde
 
SVNSEG	andlw	0x0F		; remove most significant bits if present prevents value >16h
	addwf	PCL,f		; add value of display to program counter
	retlw 	B'10000000'	; 7-segment code 0 
	retlw 	B'11100110'	; 7-segment code 1
	retlw 	B'01001000'	; 7-segment code 2
	retlw 	B'01000010'	; 7-segment code 3
	retlw 	B'00100110'	; 7-segment code 4
	retlw 	B'00010010'	; 7-segment code 5
	retlw 	B'00010000'	; 7-segment code 6
	retlw 	B'11000110'	; 7-segment code 7
	retlw 	B'00000000'	; 7-segment code 8
	retlw 	B'00000010'	; 7-segment code 9
	retlw	B'01111110'	; 7-segment code "-"

;******************************************************************************************* 

; INTERRUPT
; interrupt from counter used to multiplex display
; this sucessively switches on Disp1 & Disp2 sequence plus 
; blanking leading zeros on display for Disp2 
; uses internal timer to initiate display update
; also counts time between pulses
; start interrupt by saving w and status registers before altered by interrupt routine

INTRUPT	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf	STATUS,RP0	; select memory bank 0

	bcf	INTCON,T0IF	; clear interrupt flag
				; freq is approx 10MHz/4/256-210/(2) ~ 27,000Hz
	movf	MODE,w		; mode
	btfss	STATUS,z	; if zero then speed up interrupt
	goto	CLR_POR
	incf	MPLX,f		; increase multiplex rate counter
	btfss	MPLX,5		; when bit 5 high alter display
	goto	MOD_RT		; mode rate	

CLR_POR movlw	0xFF
	movwf	PORTB		; displays off for dead time between display multiplexing
	movf	PORTA,w		; check current display
	movwf	PORT_A		; store levels
	bsf	PORTA,3		; displays off at commons
	bsf	PORTA,2
	bsf	PORTA,0		; decimal point off

MOD_RT	movf	MODE,w		; mode
	btfss	STATUS,z	; if zero then speed up interrupt
	goto	DISP_M

	movlw	D'210'		; speed up timer interrupt rate 
	movwf	TMR0	

; measure pulse widths
; FLAG_1 bit 3 is set when positive edge detected so start counting
; bit1 set when started counting, bit 2 set when negative edge detected,
; bit 3 set when next positive edge detected. Transfer data to memory when finished

PWMCNT	btfsc	FLAG_1,1	; has counter sequence started
	goto	COUNT_P		; count positive edge start

	btfss	FLAG_1,3	; if flag set then set flag_1,1 when RB0 high
	goto	CK_LOW
	btfsc	PORTB,0		; is RB0 high
	bsf	FLAG_1,1
	goto	TIMEOUT

CK_LOW	btfss	PORTB,0		; is RB0 low
	bsf	FLAG_1,3	; set flag when RB0 is low
	goto	TIMEOUT
	
COUNT_P	incfsz	COUNT1,f	; if zero increment most significant byte
	goto	COUNT_T		; count test
	incfsz	COUNT2,f	; if zero flag overrange
	goto	COUNT_T
	bsf	FLAG_0,0
	
COUNT_T	btfsc 	FLAG_1,2	; has second counter started
	goto	COUNT_N
	btfsc	PORTB,0		; is level low
	goto	TIMEOUT
	bsf	FLAG_1,2
	goto	N_UP		; count when flag set
	
COUNT_N	btfsc	PORTB,0		; check if high again
	goto	END_CNT
N_UP	incfsz	COUNT3,f	; if zero increment most significant byte
	goto	TIMEOUT		; end
	incfsz	COUNT4,f	; if zero flag overrange
	goto	TIMEOUT
	bsf	FLAG_0,0	; overrange ie always high
	goto	TIMEOUT

; check if counters have reached enough counts otherwise count for more sets of pulses
	
END_CNT	movf	COUNT2,w
	sublw	B'00001000'	; compare with 4096 for ms counter
	btfss	STATUS,c	; if still less, not enough counts so continue
	goto	CLEAR_C		; clear counters

	bcf	FLAG_1,2	; clear negative edge flag to stop counting the pulse till edge
				; occurs again
	goto	TIMEOUT
			
CLEAR_C	clrf	FLAG_1		; clear flags
	clrf	FLAG_0
	movf	COUNT1,w
	movwf	COUNT5
	movf	COUNT2,w
	movwf	COUNT6
	movf	COUNT3,w
	movwf	COUNT7
	movf	COUNT4,w
	movwf	COUNT8
	clrf	COUNT1
	clrf	COUNT2
	clrf	COUNT3
	clrf	COUNT4
	clrf	COUNTT
	clrf	COUNTO

TIMEOUT	
	movlw	0x08
	addwf	COUNTO,f	; increase ls counter
	btfsc	STATUS,c	; if carry set then increase ms counter
	incfsz	COUNTT,f
	goto	DISPL	
	bsf	FLAG_0,1	; timeout ie frequency too low or always low

; drive displays

DISPL	btfss	MPLX,5		; when bit 5 high alter display
	goto	RECLAIM		; 
	clrf	MPLX		; start counter again
DISP_M	btfsc	PORT_A,2	; skip if display 1 lit
	goto	LT1		; display 1 lit
	movf	DISP2,w		; disp2 to w
	movwf	PORTB		; portB lights display
	bcf	PORTA,3		; disp2 powered
	btfsc	DISP2,0		; if clear bit 0 then no decimal point
	bcf	PORTA,0		; decimal point on
	goto	RECLAIM		; end of multiplex						
LT1	movf	DISP1,w
	movwf	PORTB
	bcf	PORTA,2		; disp1 powered
	btfsc	DISP1,0
	bcf	PORTA,0		; decimal point on
	

; end of interrupt reclaim w and status 

RECLAIM	swapf	STATUS_TMP,w	; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie			; return from interrupt

;********************************************************************************************** 
  	
; RESET		
; Set ports A & B

MAIN	clrf	FLAG_1		; clear flags
	clrf	FLAG_0		; overrange flag
	clrf	COUNT1
	clrf	COUNT2
	clrf	COUNT3
	clrf	COUNT4
	clrf	MPLX		; multiplex counter
	clrf	MODE		; display mode normal pulse width display
	movlw 	B'10000000'	; 7-segment code 0
	movwf	DISP2
	movwf	DISP1		; displays at 00 
	bsf	STATUS,RP0	; select memory bank 1
	movlw	B'00000001'	; w = 00000001 binary (RB0 input, RB1-RB7 output)
	movwf	TRISB		; port B data direction register
	movlw	B'11000000'	; w = 10000000 binary 
	movwf	OPTION_REG	; TMRO prescaler 2 division, PORTB pullups disabled
	movlw   B'00010000'	; w = 10000 binary (RA0-RA3 outputs, RA4 input)
	movwf   TRISA		; A port data direction register
	bcf	STATUS,RP0	; select memory bank 0
	movlw	B'11111110'	; w is all 1's for RB7-RB1, 0 for RB0
	movwf	PORTB		; portB outputs high
	movlw	B'00001101'
	movwf	PORTA		; portA RA0,RA2,RA3 outputs high, RA4 low 

; allow interrupts

	bsf	INTCON,T0IE	; set interrupt enable for timer
	bsf	INTCON,GIE	; set global interrupt enable 
	call	DELAY
	call	DELAY
	call	DELAY

; Check display mode
	
CK_MD	bsf	INTCON,GIE	; set global interrupt enable 
	movf	MODE,w
	btfsc	STATUS,z
	goto	MODE_Z		; Mode is zero so normal display
	movf	MODE,w
	xorlw	0x01		; polarity mode
	btfsc	STATUS,z	; 	
	goto	POLARTY		; polarity display
	movf	MODE,w
	xorlw	0x02		; Threshold Detect mode
	btfsc	STATUS,z	; 	
	goto	THRSHLD		; Threshold (high/low detect) display
	movf	MODE,w
	xorlw	0x03		; LEVEL mode
	btfsc	STATUS,z	; 	
	goto	LEVEL		; level display
	movf	MODE,w
	xorlw	0x04		; Hysteresis mode
	btfsc	STATUS,z	; 	
	goto	HYSTER		; Hysteresis value display
	clrf	MODE		; end of mode sequence
	goto	CK_MD
	
MODE_Z	bcf	INTCON,GIE	; clear global interrupt enable for above
	movf	PORTA,w
	bsf	PORTA,2		; DISP2 off
	bsf	PORTA,3		; DISP1 off
	nop
	btfsc	PORTA,4		; is switch closed
	goto	NORM		;
	bsf	INTCON,GIE	; set global interrupt enable for above 
	
	incf	MODE,f		; next mode
	call	POL_DSP		; polarity display
	call	DELAY
	goto	CK_MD

NORM	movwf	PORTA		; restore display drive
	bsf	INTCON,GIE	; set global interrupt enable for above 
	goto	CALC

POLARTY call	POL_DSP		; display polarity 
	call 	DELAY
	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bsf	PORTA,3
	nop
	btfss	PORTA,4		; is mode switch pressed
	goto	POL_MD
	bcf	PORTA,2		; DISP2 common 
	bcf	PORTA,3		; DISP1 common
	nop
	btfss	PORTA,4		; is switch closed
	goto	CK_UD		; check for up or down switch
	goto	CK_MD
POL_MD	bsf	INTCON,GIE	; set global interrupt enable for above
	incf	MODE,f		; next mode
	call	DET_DSP		; display detect
	goto	CK_MD

; read data values

CK_UD	bsf	INTCON,GIE	; set global interrupt enable for above
	movlw	EEPROM1
	call 	EEREAD		; increase EEPROM1
	movwf	TEMPW		; store W
	incf	TEMPW,f		; increase
	movlw	EEPROM1
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
	goto	CK_MD

POL_DSP	movlw	B'00001101'	; P for polarity 
	movwf	DISP2		; place in display
	movlw	0xFE
	movwf	DISP1		; blank display for +	
	movlw	EEPROM1
	call 	EEREAD
	movwf	TEST_L
	btfss	TEST_L,0	; if zero then polarity +
	return
	movlw	0x0A		; number for a - after conversion
	call	SVNSEG		; get dash in 7-segment code
	movwf	DISP1		; minus
	return
	
THRSHLD call	DET_DSP		; display threshold (detect)
	call	DELAY
	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bsf	PORTA,3
	nop
	btfss	PORTA,4		; is mode switch pressed
	goto	THRS_MD
	bcf	PORTA,2		; DISP2 common 
	bcf	PORTA,3		; DISP1 common
	nop
	btfss	PORTA,4		; is switch closed
	goto	CKUD_D		; check for up or down switch
	goto	CK_MD
THRS_MD	bsf	INTCON,GIE	; set global interrupt enable for above
	incf	MODE,f		; next mode
	call	LEV_DSP		; level display
	goto	CK_MD

; read data values

CKUD_D	bsf	INTCON,GIE	; set global interrupt enable for above
	movlw	EEPROM2
	call 	EEREAD		; increase EEPROM1
	movwf	TEMPW		; store W
	incf	TEMPW,f		; increase
	movlw	EEPROM2
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
	call	DET_DSP
	goto	CK_MD

DET_DSP	movlw	B'01100001'	; d for Detect bit 0 high for decimal point
	movwf	DISP2		; place in display
	movlw	EEPROM2
	call 	EEREAD
	movwf	TEST_L
	btfsc	TEST_L,0	; if zero then polarity +
	goto 	LOW_D
	movlw	B'00100100'	; High on display 
	movwf	DISP1		; minus
	return

LOW_D	movlw	B'10111000'	; Low on display 
	movwf	DISP1		; minus
	return

LEVEL	call 	LEV_DSP		; display level
	call	DELAY
	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bsf	PORTA,3
	nop
	btfss	PORTA,4		; is mode switch pressed
	goto	C_MDE
	bcf	PORTA,2		; DISP2 common 
	bcf	PORTA,3		; DISP1 common
	nop
	btfss	PORTA,4		; is switch closed
	goto	CKUD_L		; check for up or down switch
	goto	CK_MD
C_MDE	bsf	INTCON,GIE	; set global interrupt enable for above
	incf	MODE,f		; next mode
	call	HYS_DSP		; hysteresis display
	goto	CK_MD

; check if up or down switches
; stop interrupt, turn off display take RA2 and RA3 low to check switch

CKUD_L	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF		; all ones
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bcf	PORTA,3
	nop
	btfss	PORTA,4		; check for up switch
	goto	UP_L		; up in level
	bsf	PORTA,3
	bcf	PORTA,2	
	nop
	btfss	PORTA,4
	goto	DN_L
GBL_CK	bsf	PORTA,2
	bsf	PORTA,3
	bsf	INTCON,GIE	; set global interrupt enable for above
	goto	CK_MD

DN_L	bsf	INTCON,GIE	; set global interrupt enable for above
	movlw	EEPROM3
	call 	EEREAD		; EEPROM3
	movwf	TEMPW		; store W
	btfsc	STATUS,z	; if zero bypass decrement
	goto	GBL_CK
	decf	TEMPW,f		; decrease
	movlw	EEPROM3
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
GBL_DEL	bsf	PORTA,2
	bsf	PORTA,3
	bsf	INTCON,GIE	; set global interrupt enable for above
	goto	CK_MD		; check mode

UP_L	movlw	EEPROM3
	call 	EEREAD		; increase EEPROM3
	movwf	TEMPW		; store W
	sublw	D'98'		
	btfss	STATUS,c	; if negative bypass increment
	goto	GBL_CK
	incf	TEMPW,f		; decrease
	movlw	EEPROM3
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
	goto	GBL_DEL		; delay and check mode


LEV_DSP	movlw	EEPROM3
	call 	EEREAD
	movwf	BIN_0
	call	BCD		; convert to BCD
	swapf	BCD_1,w		; ls digits
	andlw	0x0F		; get ls bits
	btfss	STATUS,z	; check if 0
	goto	NOBLNKL
	movlw	B'11111110'	; display off
	goto	BLANKL
NOBLNKL	call	SVNSEG		; seven segment code
BLANKL	movwf	DISP2		; place in display buffer
	movf	BCD_1,w
	andlw	0x0F
	call	SVNSEG
	movwf	DISP1		; place in display buffer
	bsf	DISP1,0		; (bit 0 high for decimal point)
	return

HYSTER	call	HYS_DSP		; display hysteresis
	call 	DELAY
	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bsf	PORTA,3
	nop
	btfss	PORTA,4		; is mode switch pressed
	goto	RECYC
	bcf	PORTA,2		; DISP2 common 
	bcf	PORTA,3		; DISP1 common
	nop
	btfss	PORTA,4		; is switch closed
	goto	CKUD_H		; check for up or down switch
	goto	CK_MD

RECYC	bsf	INTCON,GIE	; set global interrupt enable for above
	clrf	MODE
	movlw 	B'10000000'	; 0 on display
	movwf	DISP1
	movwf	DISP2
	call	DELAY	
	goto	CK_MD

; check if up or down switches
; stop interrupt, turn off display take RA2 and RA3 low to check switch

CKUD_H	bcf	INTCON,GIE	; clear global interrupt enable for above
	movlw	0xFF		; all ones
	movwf	PORTB		; displays off
	bsf	PORTA,2
	bcf	PORTA,3
	nop
	btfss	PORTA,4		; check for up switch
	goto	UP_H		; up in hysteresis value
	bsf	PORTA,3
	bcf	PORTA,2	
	nop
	btfss	PORTA,4
	goto	DN_H
	goto	GBL_CK	

DN_H	movlw	EEPROM4
	call 	EEREAD		; EEPROM4
	movwf	TEMPW		; store W
	btfsc	STATUS,z	; if zero bypass decrement
	goto	GBL_CK
	decf	TEMPW,f		; decrease
	movlw	EEPROM4
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
	call	HYS_DSP		; display new value
	bsf	PORTA,2
	bsf	PORTA,3
	bsf	INTCON,GIE	; set global interrupt enable for above
	goto	CK_MD		; check mode
	
UP_H	movlw	EEPROM4
	call 	EEREAD		; increase EEPROM4
	movwf	TEMPW		; store W
	sublw	D'98'		
	btfss	STATUS,c	; if negative bypass increment
	goto	GBL_CK
	incf	TEMPW,f		; decrease
	movlw	EEPROM4
	movwf	EEADR		; address to write to 
	
; write new option to EEPROM

	movf	TEMPW,w		; parameter value
	call	EWRITE		; write to EEPROM
	call	HYS_DSP		; display new value
	bsf	PORTA,2
	bsf	PORTA,3
	bsf	INTCON,GIE	; set global interrupt enable for above
	goto	CK_MD		; check mode
	
HYS_DSP movlw	EEPROM4
	call 	EEREAD
	movwf	BIN_0
	call	BCD		; convert to BCD
	swapf	BCD_1,w		; ls digits
	andlw	0x0F		; get ls bits
	btfss	STATUS,z	; check if 0
	goto	NOBLNKH
	movlw	B'11111110'	; display off
	goto	BLANKH
NOBLNKH	call	SVNSEG		; seven segment code
BLANKH	movwf	DISP2		; place in display buffer
	movf	BCD_1,w
	andlw	0x0F
	call	SVNSEG
	movwf	DISP1		; place in display buffer
	bsf	DISP1,0		; (bit 0 high for decimal point)
	return	


; update value
; multiply count7 and count8 (low level counter) by 100
; divide above value by count5 and count6 for percentage
; subtract from 100 if negative edge duty cycle
; convert to BCD     	
	
CALC	movlw	0x02		; number of delay loops
	call	DELAY_1		; add delay for update time
	clrf	AARGB0		; clear
	movlw	D'100'		; %
	movwf	AARGB1

; stop interrupt
	bcf	INTCON,GIE	; clear global interrupt enable for above
	movf	COUNT5,w
	movwf	COUNT_5
	movf	COUNT6,w
	movwf	COUNT_6
	movf	COUNT7,w	; counted value
	movwf	BARGB1
	movf	COUNT8,w
	movwf	BARGB0
	bsf	INTCON,GIE	; set global interrupt enable for above
; allow interrupt

	call	FXM1616U	; multiply by 100

; shift answer
	movf	AARGB2,w	; second ls byte of result
	movwf	STO1
	movf	AARGB3,w
	movwf	AARGB2
	movf	AARGB1,w
	movwf	AARGB0
	movf	STO1,w
	movwf	AARGB1
	
; load denominator

	movf	COUNT_5,w
	movwf	BARGB1
	movf	COUNT_6,w
	movwf	BARGB0
	call	FXD2416U	; divide for %

; increase percentage if over half way at remainder

	movlw	0x7F
	addwf	REMB0,w		; add to remainder
	btfsc	STATUS,c	; increase if carry set
	incf	AARGB2,f

; subtract from 100 if negative edge
 
	movlw	EEPROM1		; check polarity of PWM
	call 	EEREAD
	movwf	TEST_L
	btfsc	TEST_L,0
	goto	CK_OR		; check overrange
	movf	AARGB2,w	; percentage
	movwf	BARGB1		; to subtract from 100
	clrf	BARGB0
	call	D_SUB		; subtract from 100

	btfss	FLAG_0,0	; overrange
	goto	CHK_ONE
	btfss	PORTB,0		; if zero load a 0
	goto	LOAD_O
LOAD_1	movlw	0x0A		; number for a - after conversion
	call	SVNSEG		; get dash in 7-segment code
	movwf	DISP1
	movwf	DISP2
	goto	CK_MD
CHK_ONE	btfss	FLAG_0,1	; low or frequency too low
	goto	CONV		; convert to BCD
	btfsc	PORTB,0		; if high load a -
	goto	LOAD_1
LOAD_O	movlw	0x00		; place 0 in display
	call	SVNSEG
	movwf	DISP1
	movlw	B'11111110'	; display off
	movwf	DISP2
	goto	CK_MD

	
; Check overrange or timeout flags

CK_OR	btfss	FLAG_0,0	; overrange
	goto	CK_ONE
	btfsc	PORTB,0		; if zero load a 0
	goto	LD_ZRO
LD_ONE	movlw	0x0A		; number for a - after conversion
	call	SVNSEG		; get dash in 7-segment code
	movwf	DISP1
	movwf	DISP2
	goto	CK_MD
CK_ONE	btfss	FLAG_0,1	; low or frequency too low
	goto	CONV		; convert to BCD
	btfss	PORTB,0		; if high load a zero
	goto	LD_ONE
LD_ZRO	movlw	0x00		; place 0 in display
	call	SVNSEG
	movwf	DISP1
	movlw	B'11111110'	; display off
	movwf	DISP2
	goto	CK_MD


; convert to BCD	

CONV	movf	AARGB2,w
CONV_1	movwf	BIN_0
	call	BCD		; convert to BCD
	movf	BCD_0,w		; check if 100
	btfss	STATUS,z
	goto	LOAD_1		; dashes on display if 100
	swapf	BCD_1,w		; ls digits
	andlw	0x0F		; get ls bits
	btfss	STATUS,z	; check if 0
	goto	NOBLNK2
	movlw	B'11111110'	; display off
	goto	BLANK2
NOBLNK2	call	SVNSEG		; seven segment code
BLANK2	movwf	DISP2		; place in display buffer
	
	movf	BCD_1,w
	andlw	0x0F
	call	SVNSEG
	movwf	DISP1		; place in display buffer

; compare values
	movlw	EEPROM2		; check threshold detect value
	call 	EEREAD
	movwf	TEST_L
	btfss	TEST_L,0	; if low 
	goto	THDET_L

; for negative detect

	movlw	EEPROM3		; level value
	call 	EEREAD
	movwf	TEST_L
	movf	AARGB2,w
	subwf	TEST_L,w	; compare values
	btfss	STATUS,c
	goto	CLR_B
	bsf	PORTA,1
	goto	CK_MD
CLR_B   movlw	EEPROM3		; level value
	call 	EEREAD
	movwf	TEST_L
	movlw	EEPROM4		; hysteresis value
	call 	EEREAD
	addwf	TEST_L,f
	sublw	D'99'		; check if more than 100%
	btfss	STATUS,c	; check if zero
	goto	CK_MD		; no clear port if more than 100
	movf	TEST_L,w	
	subwf	AARGB2,w	; compare values
	btfsc	STATUS,c	; if negative clear
	bcf	PORTA,1
	btfsc	STATUS,z	; if zero clear
	bcf	PORTA,1
	goto	CK_MD		; return 


THDET_L	movlw	EEPROM3		; level value
	call 	EEREAD
	subwf	AARGB2,w	; compare values
	btfss	STATUS,c
	goto	CLR_A
	bsf	PORTA,1
	goto	CK_MD
CLR_A   movlw	EEPROM3		; level value
	call 	EEREAD
	movwf	TEST_L
	movlw	EEPROM4		; hysteresis value
	call 	EEREAD
	subwf	TEST_L,w
	btfss	STATUS,c	; check if zero
	goto	CK_MD		; no clear port if more than zero
	subwf	AARGB2,w	; compare values
	btfss	STATUS,c	; if negative clear
	bcf	PORTA,1
	btfsc	STATUS,z	; if zero clear
	bcf	PORTA,1
	goto	CK_MD		; return 

; ********************************************************************************
; delay time for switch debounce period

DELAY	movlw	0x5		; number of delay cycles
DELAY_1	movwf	DELCNT

DELAY_M	movlw	D'255'		; set delay period 
	movwf	VALUE_1		; VALUE_1 = w
	movlw	D'255'		; set delay period value 2 
LP_1	movwf	VALUE_2		; VALUE_2 = w
	
LP_2	decfsz	VALUE_2,f	; decrease VALUE_2, skip if zero
	goto 	LP_2
	
	decfsz	VALUE_1,f	; decrease VALUE_1, skip if zero
	goto	LP_1
	decfsz	DELCNT,f
	goto	DELAY_M
	return	
	

; Subroutine to convert from 8-bit binary to 2-digit BCD (packed)
; Binary value is in BIN_0  
; Result in BCD is in BCD_0 & BCD_1.  
; BCD_0 is MSB, BCD_1 is LSB

BCD	bcf	STATUS,c	; clear carry bit
	movlw	D'8'
	movwf	CNT_8		; 8 in count
	clrf	BCD_0
	clrf	BCD_1		; set BCD registers to 0 
LOOPBCD	rlf	BIN_0,f		; shift left binary registers
	rlf	BCD_1,f		; MSB shift left

	rlf	BCD_0,f		; LSB shift left BCD registers
	decfsz	CNT_8,f		; reduce count value return when 0
	goto	DECADJ		; continue decimal adjust
	return			; completed decimal to BCD operation

; subroutine decimal adjust

DECADJ	movlw	BCD_1		; BCD LSB address
	movwf	FSR		; pointer for BCD1
	call	ADJBCD		; subroutine to adjust BCD
	movlw	BCD_0		; BCD MS address
	movwf	FSR		; pointer for BCD0
	call	ADJBCD
	goto	LOOPBCD

; subroutine adjust BCD

ADJBCD	movlw	0x03		; w has 03 
	addwf	INDF,w		; add 03 to BCDx register (x is 0-1)
	movwf	TEMP		; store w
	btfsc	TEMP,3		; test if >7
	movwf	INDF		; save as LS digit
	movlw	0x30		; 3 for MSbyte
	addwf	INDF,w		; add 30 to BCDx register
	movwf	TEMP		; store w in temp
	btfsc	TEMP,7		; test if >7
	movwf	INDF		; save as MS digit
	return			; end subroutine

; subroutine to read EEPROM memory

EEREAD	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 1
	bsf	EECON1,RD	; read EEPROM
RD_RD	nop
	btfsc	EECON1,RD	; skip if RD low (read complete)
	goto 	RD_RD		; wait for low RD (read RD)	
	bcf	STATUS,RP0	; select bank 0
	movf	EEDATA,w	; EEPROM value in w
	return

; subroutine to write to EEPROM

EWRITE	movwf	EEDATA		; data register
	bcf	INTCON,GIE	; disable interrupts
	bsf	STATUS,RP0	; select bank 1
	bsf	EECON1,WREN	; enable write
	movlw	0x55		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	0xAA		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf	EECON1,WR	; set WR bit and begin write sequence
	bsf	INTCON,GIE	; enable interrupts
	bcf	EECON1,WREN	; clear WREN bit
WRITE	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE		; not written yet
	bcf	EECON1,EEIF	; clear write interrupt flag 
	bcf	STATUS,RP0	; select bank 0
	return			; value written 
	
	
; 24/16 Bit Unsigned Fixed Point Divide 

;       Input:  24 bit unsigned fixed point dividend in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point divisor in BARGB0, BARGB1

;       Use:    CALL    FXD2416U

;       Output: 24 bit unsigned fixed point quotient in AARGB0, AARGB1,AARGB2
;               16 bit unsigned fixed point remainder in REMB0, REMB1

;       Result: AARG, REM  <--  AARG / BARG


FXD2416U        CLRF            REMB0
                CLRF            REMB1
                CLRF            TEMPD
                RLF             AARGB0,W
                RLF             REMB1, F
                MOVF            BARGB1,W
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                RLF             AARGB0, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416A      RLF             AARGB0,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46LA
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LA
UADD46LA        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LA 	RLF             AARGB0, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416A
                RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB0,0
                GOTO            UADD46L8
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L8
UADD46L8        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L8         RLF             AARGB1, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416B      RLF             AARGB1,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46LB
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LB
UADD46LB        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LB         RLF             AARGB1, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416B
                RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB1,0
                GOTO            UADD46L16
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46L16
UADD46L16       ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46L16        RLF             AARGB2, F
                MOVLW           7
                MOVWF           LOOPCOUNT
LOOPU2416C      RLF             AARGB2,W
                RLF             REMB1, F
                RLF             REMB0, F
                RLF             TEMPD, F
                MOVF            BARGB1,W
                BTFSS           AARGB2,0
                GOTO            UADD46LC
                SUBWF           REMB1, F
                MOVF            BARGB0,W
                BTFSS           STATUS,C
                INCFSZ          BARGB0,W
                SUBWF           REMB0, F
                CLRW
                BTFSS           STATUS,C
                MOVLW           1
                SUBWF           TEMPD, F
                GOTO            UOK46LC
UADD46LC        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,C
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
                CLRW
                BTFSC           STATUS,C
                MOVLW           1
                ADDWF           TEMPD, F
UOK46LC 	RLF             AARGB2, F
                DECFSZ          LOOPCOUNT, F
                GOTO            LOOPU2416C
                BTFSC           AARGB2,0
                GOTO            UOK46L
                MOVF            BARGB1,W
	        ADDWF           REMB1, F
                MOVF            BARGB0,W
                BTFSC           STATUS,c
                INCFSZ          BARGB0,W
                ADDWF           REMB0, F
UOK46L		RETURN
	
; multiply
;
;       Input:  fixed point arguments in AARG and BARG
;
;       Output: product AARGxBARG in AARG
;

;       16x16 Bit Unsigned Fixed Point Multiply 

;       Input:  16 bit unsigned fixed point multiplicand in AARGB0
;               16 bit unsigned fixed point multiplier in BARGB0

;       Use:    CALL    FXM1616U

;       Output: 32 bit unsigned fixed point product in AARGB0



FXM1616U        CLRF    AARGB2          ; clear partial product
                CLRF    AARGB3
                MOVF    AARGB0,W
                MOVWF   TEMPB0
                MOVF    AARGB1,W
                MOVWF   TEMPB1
                MOVLW   0x08
                MOVWF   LOOPCOUNT
LOOPUM1616A     RRF     BARGB1, F
                BTFSC   STATUS,c
                GOTO    ALUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616A
                MOVWF   LOOPCOUNT
LOOPUM1616B     RRF     BARGB0, F
                BTFSC   STATUS,c
                GOTO    BLUM1616NAP
                DECFSZ  LOOPCOUNT, F
                GOTO    LOOPUM1616B
                CLRF    AARGB0
                CLRF    AARGB1
                RETLW   0x00
BLUM1616NAP     BCF     STATUS,c
                GOTO    BLUM1616NA
ALUM1616NAP     BCF     STATUS,c
                GOTO    ALUM1616NA
ALOOPUM1616     RRF     BARGB1, F
                BTFSS   STATUS,c
                GOTO    ALUM1616NA
                MOVF    TEMPB1,W
                ADDWF   AARGB1, F
                MOVF    TEMPB0,W
                BTFSC   STATUS,c
                INCFSZ  TEMPB0,W
                ADDWF   AARGB0, F
ALUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                DECFSZ LOOPCOUNT, F
                GOTO   ALOOPUM1616
                MOVLW  0x08
                MOVWF  LOOPCOUNT
BLOOPUM1616     RRF    BARGB0, F
                BTFSS  STATUS,c
                GOTO   BLUM1616NA
                MOVF   TEMPB1,W
                ADDWF  AARGB1, F
                MOVF   TEMPB0,W
                BTFSC  STATUS,c
                INCFSZ TEMPB0,W
                ADDWF  AARGB0, F
BLUM1616NA      RRF    AARGB0, F
                RRF    AARGB1, F
                RRF    AARGB2, F
                RRF    AARGB3, F
                DECFSZ LOOPCOUNT, F
                GOTO   BLOOPUM1616
                RETURN

; subroutine subtract AARGB1 AARGB2 - BARGB0 BARGB1 = AARGB
	
D_SUB	movlw	0x00		; 00
	movwf	AARGB1 
	movlw	D'100'		; 100
	movwf	AARGB2 
D_RSUB	call	NEG_A		; complement of A

	movf	BARGB1,w 
	addwf	AARGB2,f 	; add lsb
	btfsc	STATUS,c	; add carry
	incf	AARGB1,f 
	movf	BARGB0,w 
	addwf	AARGB1,f 
	return

NEG_A	comf	BARGB1,f 
	incf	BARGB1,f 
	btfsc	STATUS,z
	decf	BARGB0,f 
	comf	BARGB0,f 
	return	

	
	
	end	
