
;==============================================================================
;	Program: DCRX.asm
;==============================================================================
;	Programmer: Leon Williams
;==============================================================================
;	Date: MAY 2002
;==============================================================================
;	Target Processor: PIC 16F84/04P
;==============================================================================
;	Compiler: MPLAB Version 5.2
;==============================================================================
;	Release version: 1.0
;==============================================================================
;	Revisions: NIL
;==============================================================================
;	Program Description:
;
;	A program to measure the frequency of an oscillator.
;	Frequency is read as a 24 bit word, and converted to packed BCD
;	Sample period is 100mS, so the measured resolution is 10Hz.
;	Frequency is announced in Morse Code.
;
;=========================================================================

	list	p=16f84		; processor type
	radix	hex		; HEX is default language
	errorlevel	-302	; remove warning messages about bank 1/0

;--------------------------------------------------------------------------
;	CONFIGURATION BITS

	__CONFIG 	_CP_OFF & _WDT_OFF & _PWRTE_ON & _XT_OSC

; code protection=off, watch dog timer=off,
; powerup timer =on, oscillator=low speed crystal

;-------------------------------------------------------------------------
;		****Note UPPER CASE*******	

#include 	<p16f84.inc>

;PORT A 

#define	VFOCLOCK	PORTA,3
#define	VFOGATE	PORTA,2
#define	VFOOSC	PORTA,4
#define	MUTE	PORTA,1	; 1= mute audio , 0= norm

;PORT B (RB1, RB2, RB3 D to A outputs)

#define	SWMEM	PORTB,5	; memory switch active low
#define	SWFREQ	PORTB,4	; frequency switch active low

#define	MEM	MYFLAGS,0	; 0 = MEM switch not pressed, 1= MEM switch pressed
#define	FREQ	MYFLAGS,1	; 0 = FREQ switch not pressed, 1= FREQ switch pressed
#define	BOTH	MYFLAGS,2	; 0 = both not pressed, 1= both pressed
#define	BLANKING	MYFLAGS,3	; 0 = blanking not used, 1= blanking used

#define	ZERO	STATUS,Z
#define	CARRY	STATUS,C
#define	OVERFLOW	INTCON,T0IF

#define	SLOW	USER,0	;  bit 1 slow flag=1
#define	MED	USER,1	;  bit 2 medium flag=1
#define	FAST	USER,2	;  bit 3 fast flag=1
#define	LENGTH	USER,4	;  bit 5 length flag, 0= long, 1= short (KHz only)

DOT	equ	0x01
DASH	equ	0x02

; following durations used by tone routine are the value x 1.25mS long
DOTLENS	equ	d'40'	; dot length small 
DOTLENM	equ	d'60'	; dot length medium 
DOTLENL	equ	d'80'	; dot length long
DSHLENS	equ	d'120'	; dash length short
DSHLENM	equ	d'180'	; dash length medium
DSHLENL	equ	d'240'	; dash length long
BIPLEN	equ	d'100'	; bip length
BIPSPLEN	equ	d'100'	; bip space length
BEEPLEN	equ	d'250'	; beep length
BEPSPLEN	equ	d'100'	; beep space length
CLICKLEN	equ	d'15'	; switch click length
MSCHSPS	equ	d'120'	; short morse character space
MSCHSPM	equ	d'180'	; medium morse character space
MSCHSPL	equ	d'240'	; long morse character space

;---------------------------------------------------------------------------------------------
;RAM definitions

W_TEMP	equ	0x0C	; interrupt routine temp register
ST_TEMP	equ	0x0D	; interrupt routine temp register
MYFLAGS	equ	0x0E	; holds various flags
COUNT1	equ	0x0F	; GP counter 1
COUNT2	equ	0x10	; GP counter 2
TEMP1	equ	0x11	; GP temporary register
POINTER	equ	0x12	; GP pointer
SWITCH	equ	0x13	; holds switch flags
USER	equ	0x14	; holds user settings

LOOPS	equ	0x15	; counts 200 loops of 500uS in count routine 
LOOP500	equ	0x16	; counter used in count routine for 500uS delay

DLYCNT1	equ	0x17	; counter1 only used in delay routines
DLYCNT2	equ	0x18	; counter2 only used in delay routines
DELAYN	equ	0x19	; loaded before calling delay10. Total = DELAYN x 10mS
DELAYN1	equ	0x1A	; loaded before calling delay1. Total = DELAYN1 x 1.25mS
BIPS	equ	0x1B	; loaded before calling sendbips,= how many bips sounded

MORSTEMP	equ	0x1C	; temporary register used only in morse send
MORSCNT	equ	0x1D	; counter used in morse send

CYCLES	equ	0x1E	; holds number of 800Hz cycles to send as tone
STEPS	equ	0x1F	; 8 steps per audio cycle used in tone routine only
DLYTONE	equ	0x20	; delay value used in tone routine only
BEEPS	equ	0x21	; loaded before calling sendbeep, = how many beeps sounded
TONEPTR	equ	0x22	; pointer used in table lookups in tone routine only	
	
; each BCD byte is divided into two 4 bit nibbles and contains two BCD digits (packed)	
BCD0	equ	0x30	; lowest BCD byte 
BCD1	equ	0x31	
BCD2	equ	0x32
BCD3	equ	0x33	; highest BCD byte

BIN0	equ	0x34	; low byte of 24 bit binary number to be converted 
BIN1	equ	0x35	; mid byte of 24 bit binary number to be converted 	
BIN2	equ	0x36	; high byte of 24 bit binary number to be converted 	

;EEPROM definitions

EEMEM0	equ	0x00	; memory bcd0 store address in eeprom
EEMEM1	equ	0x01
EEMEM2	equ	0x02
EEMEM3	equ	0x03	; memory bcd3 store address in eeprom

EEFLAGS	equ	0x04	; speed and length flags stored in EEPROM

;***************************MAIN START**********************************************

	org	0x000
	goto	start	; skip over interupt vector and go to start address
	org	0x004	; interupt service
	goto	intserv

start	call	initpic	; initialise PIC

	movlw	EEFLAGS	; get User settings from EEPROM and store in RAM
	movwf	EEADR
	call	eeread	; get byte into EEDATA
	movf	EEDATA,w	; into w
	movwf	USER	; finally into USER register - holds user set options

	movlw	d'25'	; small delay after power up
	movwf	DELAYN
	call	delay10

mainpwr	bsf	MUTE	; turn off radio audio
	call	powertne	; sound power up bips 

mainloop	bcf	MUTE		; enable radio audio again
	movf	PORTB,w		; get a copy of the current port B, 
	nop			; for sleep to detect changed switch inputs.
	bcf	INTCON,RBIF	; clear port b change flag
	bsf	INTCON,RBIE	; set/allow port b change interrupt during sleep
	sleep			; power down and wait for a switch to be pressed
	nop			; NOW AWAKE!!!!!!!!
	bcf	INTCON,RBIF	; clear port b change flag
	bcf	INTCON,RBIE	; clear/disable port b change interrupt option	
	movlw	d'10'		; wait for 100mS before doing anything
	movwf	DELAYN		; lets crystal stabalise a bit more
	call	delay10		; as it just got turned on

	call	switch?	; see which switch is pressed

mainfreq	btfss	FREQ	; is freq switch pressed?
	goto	mainmem	; no, so check if memory pressed
	bcf	FREQ	; yes, reset frequency flag
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	mainloop	; wait again

mainmem	btfss	MEM	; is mem switch pressed?
	goto	mainboth	; no check if both pressed
	bcf	MEM	; yes, reset memory flag
	call	memory	; do memory function
	goto	mainloop	; wait again

mainboth	btfss	BOTH	; are both switches pressed?
	goto	mainloop	; no switches pressed (ERROR) wait again
	bcf	BOTH	; yes, reset both flag
	call	program	; do program
	goto	mainpwr	; go sound bips to show done program mode

;*******************************END MAIN******************************************************


;*******************************START CALLS***************************************************
; The two routines that use table calls are placed in the first 256 bytes

;--------------------------------START SENDMORS---------------------------------------
; This routine sends the digit in BIPS in Morse Code.
; Morse numbers all have 5 dots or dashes. Table position is found by multiplying
; the BIPS value by 5. Multiplying is done by adding 5 to the pointer BIPS times.
; Example: digit to send is 4, BIPS = 0x04. The pointer is reset to zero, and then 
; we add 5 to the pointer 4 times, 5+5+5+5 = 20.
; Position 20 in the table is the first dot of the digit 4. 
; The next 4 reads from the table are accessed by incrementing Position by 1. 
; After 5 reads the routine returns.

sendmors	movlw	0x0f
	andwf	BIPS,f		; clear any bits from top nibble
	clrf	POINTER		; reset table offset pointer
	movlw	d'5'		; counter of data bits sent (all digits have 5)
	movwf 	MORSCNT

	movf	BIPS,f		; test if digit = 0
	btfsc	STATUS,Z		; z set if zero
	goto	sendmlp		; number is zero, so start at top of table

sendmadd	movlw	d'5'		; the digit is not zero, 
	addwf	POINTER,f		; multply BIPS x 5 to move pointer down the table
	decfsz	BIPS,f		; decrement number
	goto	sendmadd		; not done yet, point further down table
	
sendmlp	movf	POINTER,w		; move pointer value into W reg
	call	sendmtbl		; go get the dot or dash from table
	movwf	MORSTEMP		; put w (value from table) into MORSTEMP reg

	btfsc	MORSTEMP,0	; is it a dot?
	call	senddot		; send dot
	btfsc	MORSTEMP,1	; is it a dash?
	call	senddash		; send dash
 	decfsz	MORSCNT,f 	; see if done 5 bits of the number yet
	goto	sendmagn		; not yet
	
	btfsc	FAST		; done all 5 bits, add a space between chars
	goto	sendmspf		; check if short or long space
	btfsc	MED		; is it medium
	goto	sendmspm
	movlw	MSCHSPL		; long (slow morse)
	movwf	DELAYN
	call	delay1
	return			; finished go home

sendmspm	movlw	MSCHSPM		; medium morse
	movwf	DELAYN
	call	delay1
	return			; finished go home
			
sendmspf	movlw	MSCHSPS		; short (fast morse)
	movwf	DELAYN
	call	delay1
	return			; finished go home

sendmagn	incf	POINTER,f		; point to next digit bit
	goto	sendmlp		; around again	

sendmtbl	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	DASH		; 0
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 1
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 2
	retlw	DOT
	retlw	DASH
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 3
	retlw	DOT
	retlw	DOT
	retlw	DASH
	retlw	DASH

	retlw	DOT		; 4
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DASH

	retlw	DOT		; 5
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 6
	retlw	DOT
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 7
	retlw	DASH
	retlw	DOT
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 8
	retlw	DASH
	retlw	DASH
	retlw	DOT
	retlw	DOT

	retlw	DASH		; 9
	retlw	DASH
	retlw	DASH
	retlw	DASH
	retlw	DOT


;---------------------------END SENDMORS------------------------------------------------

;===================================START TONE========================================
; Sends 800Hz (1/800Hz = 1.25mS) tone via 3 bit R2R D to A converter.
; Tone on period = (value in CYCLES + start[Up] and end[Down] cycle) x 1.25mS.  
; There are 8 steps per cycle and each step = 1.25mS/8 = 156uS

; Start with 2 rising amplitude cycles

tone	clrf	TONEPTR		; reset table offset pointer

toneUp	movlw	d'16'		; 16 steps in the first two cycles
	movwf	STEPS

toneUlp	movf	TONEPTR,w		; move pointer value into W reg
	call	toneUtbl		; go get tone value
	movwf	PORTB		; out to port

	movlw	d'47'		; 156uS step delay
	movwf	DLYTONE
toneUdel	decfsz	DLYTONE,f
	goto	toneUdel

	decfsz	STEPS,f		; done a complete cycle?
	goto	toneUagn		; not yet
	goto	toneMid		; done a complete cycle, go do the middle bit

toneUagn	incf	TONEPTR,f		; point to next wave cycle value
	goto	toneUlp		; around again	

; now send the middle section. Number of cycles sent is set in CYCLES

toneMid	clrf	TONEPTR		; reset table offset pointer
	movlw	d'8'		; 8 steps per cycle
	movwf	STEPS

toneMlp	movf	TONEPTR,w		; move pointer value into W reg
	call	toneMtbl		; go get tone value
	movwf	PORTB		; out to port

	movlw	d'47'		; 156uS delay
	movwf	DLYTONE
toneMdel	decfsz	DLYTONE,f
	goto	toneMdel

	decfsz	STEPS,f		; done a complete cycle?
	goto	toneMagn		; not yet
 	decfsz	CYCLES,f 		; done this cycle, see if done all cycles
	goto	toneMid		; not yet, do another cycle
	goto	toneDown		; done all mid cycles, now finish

toneMagn	incf	TONEPTR,f		; point to next wave cycle value
	goto	toneMlp		; around again
	
; End with 2 falling amplitude cycles

toneDown	clrf	TONEPTR		; reset table offset pointer
	movlw	d'16'		; 16 steps in the last 2 cycles
	movwf	STEPS

toneDlp	movf	TONEPTR,w		; move pointer value into W reg
	call	toneDtbl		; go get tone value
	movwf	PORTB		; out to port

	movlw	d'47'		; 156uS step delay
	movwf	DLYTONE
toneDdel	decfsz	DLYTONE,f
	goto	toneDdel

	decfsz	STEPS,f		; done the complete cycle?
	goto	toneDagn		; not yet

	movlw	0x08		; set audio to idle
	movwf	PORTB
	return			; done, go home

toneDagn	incf	TONEPTR,f		; point to next wave cycle value
	goto	toneDlp		; around again	


toneUtbl	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	0x08		;       O
	retlw	0x08		;       0
	retlw	0x0A		;        X
	retlw	0x08		;       O
	retlw	0x08		;       O
	retlw	0x06		;      X
	retlw	0x06		;      X
	retlw	0x08		;       0
	retlw	0x08		;       O
	retlw	0x0A		;        X
	retlw	0x0C		;         X
	retlw	0x0A		;        X
	retlw	0x08		;       O
	retlw	0x06		;      X
	retlw	0x04		;     X
	retlw	0x06		;      X
toneMtbl	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	0x08		;       O
	retlw	0x0C		;         X
	retlw	0x0E		;          X
	retlw	0x0C		;         X
	retlw	0x08		;       O
	retlw	0x04		;     X
	retlw	0x02		;    X
	retlw	0x04		;     X
toneDtbl	addwf	PCL,f		; add offset in w reg to program pointer
	retlw	0x08		;       O
	retlw	0x0A		;        X
	retlw	0x0C		;         X
	retlw	0x0A		;        X
	retlw	0x08		;       O
	retlw	0x06		;      X
	retlw	0x04		;     X
	retlw	0x06		;      X
	retlw	0x08		;       O
	retlw	0x08		;       0
	retlw	0x0A		;        X
	retlw	0x0A		;        X
	retlw	0x08		;       O
	retlw	0x08		;       O
	retlw	0x06		;      X
	retlw	0x08		;       0


;---------------------------------END TONE-------------------------------------------


;-----------------------------START SENDDOT-------------------------------------------
; This routine sends a dot and then a space of 1 dot

senddot	btfsc	FAST		; is it a fast dot?
	goto	senddotf		; is fast
	btfsc	MED		; is it a medium dot?
	goto	senddotm		; is med	
senddots	movlw	DOTLENL		; must be long dot
	movwf	CYCLES
	call	tone

	movlw	DOTLENL		; long space
	movwf	DELAYN1
	call	delay1		; N x 1.25mS delay
	return

senddotm	movlw	DOTLENM		; medium dot
	movwf	CYCLES
	call	tone

	movlw	DOTLENM		; medium space
	movwf	DELAYN1
	call	delay1		
	return

senddotf	movlw	DOTLENS		; short dot
	movwf	CYCLES
	call	tone

	movlw	DOTLENS		; short space
	movwf	DELAYN1
	call	delay1		
	return

;------------------------------END SEND DOT--------------------------------------------

;-----------------------------START SENDDASH-------------------------------------------
; This routine sends a dash and then a space of 1 dot

senddash	btfsc	FAST		; is it a fast dash?
	goto	senddshf		; is fast
	btfsc	MED		; is it a medium dot?
	goto	senddshm		; is med	

senddshs	movlw	DSHLENL		; long dash
	movwf	CYCLES
	call	tone

	movlw	DOTLENL		; long space
	movwf	DELAYN1
	call	delay1	
	return

senddshm	movlw	DSHLENM		; medium dash
	movwf	CYCLES
	call	tone

	movlw	DOTLENM		; medium space
	movwf	DELAYN1
	call	delay1		
	return

senddshf	movlw	DSHLENS		; short dash
	movwf	CYCLES
	call	tone

	movlw	DOTLENS		; short space
	movwf	DELAYN1
	call	delay1		
	return

;------------------------------END SEND DASH--------------------------------------------


;-----------------------------START POWERTNE-------------------------------------------
; This routine sounds a tone sequence each power up, to show all is OK

powertne	movlw	d'3'	; 
	movwf	BIPS
	call	sendbips
	return
;------------------------------END POWERTNE--------------------------------------------

;-----------------------------START PROGTNE1-------------------------------------------
; This routine sounds a tone sequence when program mode entered

progtne1	movlw	d'1'	; 
	movwf	BEEPS
	call	sendbeep
	return
;------------------------------END POWERTNE1--------------------------------------------

;-----------------------------START PROGTNE2-------------------------------------------
; This routine sounds a tone sequence when program setting stage 2 entered

progtne2	movlw	d'2'	; 
	movwf	BEEPS
	call	sendbeep
	return
;------------------------------END POWERTNE2--------------------------------------------

;===============================START SEND BIPS==============================================
; Sounds number of bips set in BIPS. Each bip is BIPLEN x 1.25mS long, and
; BIPSPLEN x 1.25mS apart

sendbips	movlw	0x0f
	andwf	BIPS,f		; clear any bits from top nibble

sndbpslp	movlw	BIPLEN
	movwf	CYCLES
	call	tone
	movlw	BIPSPLEN		
	movwf	DELAYN1
	call	delay1
	decfsz	BIPS,f		; have we done all the bips?
	goto	sndbpslp		; not yet
	return			; yes

;---------------------------------END SEND BIPS----------------------------------------------



;===============================START SEND BEEPS==============================================
; Sounds numer of beeps set in BEEPS. Each beep is BEEPLEN x 1.25mS long, and
; BEPSPLEN x 1.25mS apart

sendbeep	movlw	0x0f
	andwf	BEEPS,f		; clear any bits from top nibble

sndbeplp	movlw	BEEPLEN
	movwf	CYCLES
	call	tone
	movlw	BEPSPLEN		
	movwf	DELAYN1
	call	delay1
	decfsz	BEEPS,f		; have we done all the beeps?
	goto	sndbeplp		; not yet
	return			; yes

;---------------------------------END SEND BEEPS----------------------------------------------


;-----------------------------START CLICK-------------------------------------------
; This routine sounds a short click from audio generator
; to indicate button pressed. Audio tone = CLICKLEN x 1.25mS of 800Hz

click	movlw	CLICKLEN	
	movwf	CYCLES
	call	tone
	return
;------------------------------END CLICK--------------------------------------------


;-----------------------------START PROGRAM-------------------------------------------
; This routine allows user to change settings for morse length and speed of digits

program	call	progtne1	; tone to signal entered program mode

proglen	call	switch?	; wait for a switch press
	btfsc	FREQ	; is it freq?
	goto	proglen1	; yes, so go toggle length of readout
	btfsc	MEM	; is it MEM?
	goto	prognxt	; yes, so go set next program setting
	btfsc	BOTH	; is it both?
	goto	progfin	; both pressed, so store settings and return
	goto	proglen	; no switches pressed, so just wait

proglen1	btfsc	LENGTH	; 0=long 1=short
	goto	proglens	; is currently long, so set short
	bsf	LENGTH	; set flag to short
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	proglen	; wait for next switch press

proglens	bcf	LENGTH	; clear flag to long
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	proglen	; wait for next switch press
;---------------------------------

prognxt	call	progtne2	; indicate in speed mode
	bcf	MEM	; clear flag
progspd	call	switch?	; wait for a switch press
	btfsc	FREQ	; is it freq?
	goto	progspd1	; yes, so go set morse speed
	btfsc	MEM	; is it MEM?
	goto	program	; yes, so go around again
	btfsc	BOTH	; is it both?
	goto	progfin	; both pressed, so store settings and return
	goto	progspd	; no switches pressed, so just wait

progspd1	bcf	FREQ	; clear flag
	btfsc	FAST	; is it currently fast? (1 if set)
	goto	progspdm	; yes, so set to med
	btfsc	MED	; is it currently set to med
	goto	progspds	; yes, so set to slow
	movlw	0xF0	; is slow, clear speed flags and set fast flag
	andwf	USER,f
	bsf	FAST
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	progspd	; wait for next switch press
	
progspdm	movlw	0xF0	; clear speed flags and set med flag
	andwf	USER,f
	bsf	MED
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	progspd	; wait for next switch press

progspds	movlw	0xF0	; clear speed flags and set slow flag
	andwf	USER,f
	bsf	SLOW
	call	measfreq	; measure frequency
	call	sendfreq	; sound frequency
	goto	progspd	; wait for next switch press


progfin	bcf	BOTH	; clear flag
	movf	USER,w	; store user settings in eeprom location eeflags
	movwf	EEDATA
	movlw	EEFLAGS
	movwf	EEADR
	call	eewrite
	return

;------------------------------END PROGRAM--------------------------------------------


;=================================START SENDFREQ====================================
; Sounds the current frequency, and can be done in 2 ways.
; 1. all digits in Morse. 
; 2. only the 3 Khz digits in Morse. 
; The LENGTH  determines if all or only the KHz digits are sounded.
; In most HAM radio situations the MHz will be known, and so there is no need to sound
; it everytime. If we select SHORT and the frequency is 7,130,050Hz, we will only 
; hear digits 1 3 0. 
; Morse is sent according to the setting of the SPEED option in the tone send routines.
; To increase intelligibility and speed up the process, a technique called 
; blank leading zeros is used. This stops the sounding of any Zeros
; ahead of the most significant digits. e.g. A frequency of 250,090 Hz would have a
; BCD register value of 00.25.00.90. We don't need or really want the top most 2 
; zeros to be sounded. The technique starts at the top and if a zero is found we
; don't sound it and move onto the next. If this one is a zero, we do the same and
; so on until a digit other than zero is found. From here any other zeros are sounded.
; Note: we load the indirect register (FSR) with the high BCD byte (BCD3 or 2) at start.
; Because we want to send the nost significant digits first. As a result we move backwards
; because the BCD bytes are in ascending order in RAM.

sendfreq	movlw	d'20'	; 200mS delay before sending
	movwf	DELAYN
	call	delay10

	btfsc	LENGTH	; see if we are going to send all digits or just KHz
	goto	sndflong	; all digits this time

sndfshort	movlw	BCD2	; load indirect register with address of second BCD digits
	movwf	FSR
	movlw	d'3'	; 3 BCD digits to send
	movwf	COUNT1
	bcf	BLANKING	; clear blank leading zeros, send all digits even if 0
	goto	sndfloop

sndflong	movlw	BCD3	; load indirect register with address of first BCD digits
	movwf	FSR
	movlw	d'8'	; 8 BCD digits to send
	movwf	COUNT1
	bsf	BLANKING	; set blank leading zeros to start

sndfloop	movf	INDF,w	; get packed BCD digits
	movwf	TEMP1	
	swapf	TEMP1,w	; get BCD byte, swap the nibbles and place in W
	andlw	0x0F	; clear top bits leaving higher nibble in W
	movwf	BIPS	; put in BIPS for bips routine
	movf	BIPS,f	; test for 0 by moving into its own reg
	btfss	ZERO	
	goto	sndf1ok	; was not 0 can send ok
	btfss	BLANKING	; this digit is 0, see if blanking
	goto	sndf1ok	; no blanking, so do this digit
	goto	sndf2	; this digit is 0 and blanking so go check digit in low nibble

sndf1ok	call	sendmors
	bcf	BLANKING	; sent this digit clear blanking flag

sndf2	decfsz	COUNT1,f	; done all digits?
	goto	sndf2go	; not yet
	goto	sndfdone	; finished all 
	
sndf2go	movf	TEMP1,w	; get BCD byte again
	andlw	0x0F	; clear top bits, leaving lower nibble in W
	movwf	BIPS	; put in BIPS for bips routine
	movf	BIPS,f	; test for 0 by moving into its own reg
	btfss	ZERO	
	goto	sndf2ok	; is not 0 can send ok
	btfss	BLANKING	; this digit is 0, see if blanking
	goto	sndf2ok	; this one is 0 but not blanking so do digit
	goto	sndfnext	; this one = 0 and blanking go check next digit

sndf2ok	call	sendmors
	bcf	BLANKING	; sent this digit so reset flag

sndfnext	decf	FSR,f	; next location in RAM (backwards, started at high end)
	decfsz	COUNT1,f	; done all digits
	goto	sndfloop	; not yet
sndfdone	return		; finished all 

;------------------------------------END SENDFREQ----------------------------------------------


;=================================START SWITCH?===============================================
; this routine polls the switches on PORTB. If a switch is not pressed, simply returns
; with all flags cleared. If a switch is pressed, a debounce period of 100mS is applied
; and the switches are tested again. If a switch is still pressed the switch flag is 
; set, and then waits for the switches to be all un-pressed.
; Switches are active low.

switch?	bcf	FREQ		; clear flags at start - just to be sure
	bcf	MEM
	bcf	BOTH
	movf	PORTB,w		; to check if a switch is pressed firstly,
	andlw	b'00110000'	; get port b and mask off non-switch bits
	sublw	b'00110000'	; check against no switches pressed mask
	btfss	ZERO		; flag =1 if no switch pressed, 0 if one is
	goto	swideb		; comparison bad, flag=0, a switch is pressed
	return			; none pressed, return

swideb	movlw	d'10'	; a switch is pressed, 100mS debounce delay, 
	movwf	DELAYN	; and check again
	call	delay10

	btfss	SWMEM	; MEM still pressed?
	goto	swiboth?	; yes
	btfss	SWFREQ	; MEM not pressed, is FREQ still pressed
	goto	swifreq	; yes
	return		; none still pressed, go home	

swiboth?	btfsc	SWFREQ	; MEM is pressed, see if freq is pressed too
	goto	swimem	; no, so only mem pressed
	bsf 	MUTE	; mute radio audio, leaving only the PIC tone	
	bsf	BOTH	; set both flag
	call	click
	goto	swiwait

swimem	bsf 	MUTE	; mute radio audio, leaving only the PIC tone
	bsf	MEM	; set MEM flag
	call	click
	goto	swiwait

swifreq	bsf 	MUTE	; mute radio audio, leaving only the PIC tone
	bsf	FREQ	; set freq flag
	call	click
	goto	swiwait

swiwait	movf	PORTB,w		; wait for switch to be released, get port b again put in W
	andlw	b'00110000'	; mask off non-switch bits
	sublw	b'00110000'	; check against no switches pressed mask
	btfss	ZERO		; flag =1 if not pressed anymore
	goto	swiwait		; comparison bad, flag=0, so still pressed
	movlw	d'20'		; switch(s) now released, 
	movwf	DELAYN		; 200mS delay and then return
	call	delay10
	return

;---------------------------------END SWITCH?--------------------------------------


;=============================START MEMORY========================================
; Gets here after the MEM switch is pressed and then waits for the next switch press.
; If MEM is next switch pressed, the current frequency is stored in EEPROM.
; If FREQ is the next switch pressed, the frequency stored in EEPROM is sounded.
; Following either of these two actions the routine returns to normal operation.
; 

memory	call	switch?	; check for a switch pressed
	btfsc	FREQ	; was it FREQ
	goto	memfreq	; yes
	btfss	MEM	; not freq was it mem
	goto	memory	; not freq or mem, try again

memmem	call	measfreq	; measure and convert input signal frequency

	bcf	MEM	; clear flag
	movlw	d'4'	; 4 locations
	movwf	COUNT2
	movlw	EEMEM0	; 1st EEPROM location
	movwf	EEADR
	movlw	BCD0	; 1st Ram address
	movwf	FSR	; indirect pointer
memmlp	movf	INDF,w
	movwf	EEDATA
	call	eewrite
	incf	FSR,f
	incf	EEADR,f
	decfsz	COUNT2,f	; finished all 4
	goto	memmlp	; not yet

	movlw	d'50'	; small delay 
	movwf	DELAYN
	call	delay10

	movlw	d'2'	; announce loaded into memory
	movwf	BIPS
	call	sendbips
	return
	
memfreq	bcf	MEM	; clear flag
	movlw	d'4'	; 4 locations
	movwf	COUNT2
	movlw	EEMEM0	; 1st EEPROM location
	movwf	EEADR
	movlw	BCD0	; 1st Ram address
	movwf	FSR	; indirect pointer

memflp	call	eeread	; get byte into EEDATA
	movf	EEDATA,w	;
	movwf	INDF	; into ram

	incf	FSR,f
	incf	EEADR,f
	decfsz	COUNT2,f	; finished all 4
	goto	memflp	; not yet

	call	sendfreq

	return

;-----------------------------END MEMORY-----------------------------------------


;=================================START MEASFREQ======================================
;
measfreq	call	count	; count input VFO for 100mS
	call	B2bcd	; convert the binary values to BCD digits
	return

;---------------------------------END MEASFREQ---------------------------------------



;=================================START COUNT===================================================
; Main counter routine. The accuracy of the measurements relies on this routine
; opening the gate for exactly 100mS. During the 100mS gate period the Timer 0 
; overflow flag is sampled every 500uS.
; Counter is 24 bit, made up of 8 bit pre-scaler(BIN0), 8 bit Timer0(BIN1) and 
; 8 bit software register BIN2. BIN2 is incremented each time T0 overflows
; Maximum 24 bit count is 16,777,215, and because the sample period is 100mS 
; this relates to a VFO frequency of 167MHz, however this is not possible due to 
; hardware limitations. 
; The upper limit according to MICROCHIP data is around 50MHz.
; At this frequency the maximum count will be 5,000,000.
; At 50MHz input, Timer 0 overflows every 1.3mS, so testing each 500uS is adequate.
; Count program loops around 200 x 500uS = 100mS
; Total delay around the loop equals: 6 uS to test T0 overflow + 3x161(normal loop)
; + 2(final loop) + 6(reload+nop) + 3(back to start) = 500uS 
; As a result the cntdelay value used doesn't work out be exactly 500uS but 494uS,
; so LOOP500 value becomes 162. 

count	bcf	VFOGATE		; disable vfo gate
	bcf	VFOCLOCK		; take clock line low
	clrf	TMR0		; clear timer 0 register, also clears pre-scaler
	clrf	BIN2		; clear timer 0 overflow counter
	bcf	OVERFLOW		; clear timer 0 overflow flag

	movlw	d'200'		; Load delay loop counter with 200
	movwf	LOOPS 	  
	movlw	d'162' 		; Load 494uS counter
	movwf	LOOP500 	

	bsf	VFOCLOCK		; take clock high to allow vfo into timer 0 	
	bsf	VFOGATE		; enable gate to allow vfo freq into timer

;now go around a 500uS total loop 200 times

cntloop 	btfss   	OVERFLOW 		; Test for TMR0 overflow
        	goto    	cntNOof  		; no overflow this time
        	incf    	BIN2,f		; TMR0 exceeds FF, inc overflow counter
	bcf	OVERFLOW		; Reset TMR0 interrupt flag
	goto	cntdelay		; now wait 494uS
cntNOof  	nop              		; NOPs to equal up both branches
        	nop		 	; 
	nop			; 	

cntdelay 	decfsz  	LOOP500,f   	; see if delay has ended
        	goto    	cntdelay     	; not yet	
	movlw	d'162' 		; yes, reload 494uS counter
	movwf	LOOP500 
	nop			; nops to make exact delay value 
	nop
	nop
	nop

cntend?	decfsz	LOOPS,f 		; done 200 500uS delays?
	goto	cntloop	 	; not yet

cntfin	bcf	VFOGATE		; finished 100mS delay, close gate
	bcf	VFOCLOCK	

 	btfss   	OVERFLOW 		; Test for final TMR0 overflow
        	goto    	cntfina  		; no overflow at end of 100mS
        	incf    	BIN2,f		; TMR0 exceeds FF, inc overflow counter

cntfina	movf  	TMR0,w		; Move contents of TMR0 into BIN1
	movwf	BIN1 

; This section moves the contents of the prescaler into BIN0. The pre-scaler cannot
; be read directly.
; The value in pre-scaler is found by externally clocking the pre-scaler through
; the VFOCLOCK pin. We firstly load COUNT2 with maximum count of 255. 
; Each time we clock (increment) the pre-scaler we also decrement COUNT2
; until Timer 0 contents change. This indicates that the pre-scaler reached maximum
; count and has clocked timer 0.
; At this point COUNT2 equals the original prescaler value.
; Example: COUNT2 gets loaded with 255, and let's say the pre-scaler has a count 
; of 60 at the closing of the VFOGATE.
; The pre-scaler output will change state after 255-60 clocks of VFOCLOCK = 195.
; COUNT2 is decremented 195 times starting at 255 and so ends up being 60,
; which hey presto is the pre-scaler value. Isn't life wonderful!
; At this point we have a 24 bit value of the frequency to 10Hz resolution.

	movf	TMR0,w		; get timer 0 value, value unimportant
	movwf	TEMP1		; Save Timer 0 value for comparison
	movlw	d'255'
	movwf	COUNT2 		; Load count with 255 (maximum count)

cntpre	decf	COUNT2,f		; Count down
        	bsf	VFOCLOCK  	; Clock Hi
	nop
        	bcf       VFOCLOCK  	; Clock Lo
	movf	TEMP1,w
	subwf	TMR0,w
        	btfsc	ZERO		; are they still the same?
        	goto	cntpre		; Yes (Z=1) check again
	movf	COUNT2,w		; No(Z=0) count = prescaler value
	movwf	BIN0
	
	return
	
;----------------------------------END COUNT----------------------------------------------



;===============================START BINARY TO BCD===========================================
; B2bcd is a program to convert a 24 bit binary number held in BIN0, BIN1 and BIN2 to  
; 8 BCD digits packed into 4 Bytes, using the "Add 3" algorithm, BIN0 is the lower byte.
; The Binary bytes are rotated left, and Bit D7 of each byte goes into the Carry bit. 
; When the next byte is rotated it picks up the Carry bit and puts it into bit D0. 
; In this way we can easily rotate any length binary number held in discrete bytes. 
; However we must rotate the lower byte first, so that the next higher byte gets the 
; carry bit.
; Similiarly the BCD bytes are rotated in the same fashion. Bit D0 of the lowest BCD 
; byte (BCD0) gets bit D7 of the highest Binary byte.
; After we rotate, if any of the 4 bit BCD nibbles are 5 or more, then 3 is added 
; to the value.
; To see if the nibbles are 5 or larger, we firstly add 3 to each and see if D7 for 
; the higher nibble and D3 for the lower nibble is set. If it is, then the original 
; number was larger or equal to 5. If it proves that the nibble was not 5 or larger, 
; the 3 is subtracted to leave the original number.
; The process continues until we have shifted all the bits (24) left out of the 
; binary bytes. The resulting BCD bytes contain the following:
;	_____________________________________________________________
; 	|BYTE  |	HIGH nibble	|	LOW nibble       	|
;	-------------------------------------------------------------
; 	| BCD3 |	10 millions	|	millions		|	
; 	| BCD2 |	100 thousands	|	10 thousands	|
; 	| BCD1 |	thousands		|	hundreds		|
; 	| BCD0 |	tens		|	ones		|
;
; Because the sample period is 100mS the frequency in the BCD regs is 1/10 of the 
; actual frequency.
; Example: 14,318,619Hz input will be stored as 01.43.18.61. To make it look like 
; full frequency the nibbles are shifted one nibble (4 bits) to the left, and the 
; lowest nibble is set to 0, so the stored frequency is now 14.31.86.10
;----------------------------------------------------------------------------------------------
	
B2bcd 	movlw   	d'24' 
        	movwf   	COUNT1 
        	clrf    	BCD3 
	clrf    	BCD2 
        	clrf    	BCD1 
        	clrf	BCD0		
        	
B2bcdagn	rlf	BIN0,f	; rotate all registers one position left <
        	rlf      	BIN1,f 
       	rlf      	BIN2,f 
       	rlf	BCD0,f
       	rlf      	BCD1,f 
        	rlf      	BCD2,f 
        	rlf      	BCD3,f 
        
        	decfsz 	COUNT1,f 	; see if done 24 shifts
         	goto   	B2bcdadj	; not yet, go and adjust last rotation
         	
; done all 24 shifts, BCD digits hold converted number, now move to look like real frequency       	
	movlw   	d'4'	; shift left the digits 4 bits (one nibble)
        	movwf   	COUNT1 
B2bcdfin 	rlf	BCD0,f
       	rlf      	BCD1,f 
        	rlf      	BCD2,f 
        	rlf      	BCD3,f 
        	decfsz	COUNT1,f
        	goto	B2bcdfin	
      	movlw	0xF0
        	andwf	BCD0,f
        	return 		; FINISHED		

B2bcdadj	movlw   	0x33 	; add 3 to each bcd digit (2 bcd digits per byte)
	addwf	BCD3,f	; if we add 3 to each BCD, and it is 5 or larger
        	addwf   	BCD2,f 	; then bit 3 for the lower BCD and bit 7 for the
        	addwf   	BCD1,f 	; higher BCD will be set
        	addwf	BCD0,f

        	movlw   	0x03 	; check if each lower BCD is larger than or equal to 5
        	btfss   	BCD3,3 	; if it was, then leave alone
         	subwf  	BCD3,f 	; if was less than 5, (bit 3 clear) subtract 3 to leave
        	btfss   	BCD2,3 	; the original number
         	subwf  	BCD2,f 
        	btfss   	BCD1,3 
         	subwf  	BCD1,f 
        	btfss   	BCD0,3 
	subwf  	BCD0,f 

        	movlw   	0x30 	; now check if each higher BCD is larger than or equal to 5
        	btfss  	BCD3,7 	; if it was, then leave alone
         	subwf  	BCD3,f 	; if it was less than 5, (bit 7 clear) subtract the 3 to leave
        	btfss   	BCD2,7 	; the original number
         	subwf  	BCD2,f 
        	btfss   	BCD1,7 
         	subwf  	BCD1,f 
          btfss   	BCD0,7 
         	subwf  	BCD0,f 

	goto	B2bcdagn	; do it all again

;-----------------------------END BINARY TO BCD----------------------------------------------



;------------------------------START EEWRITE---------------------------------------
; Writes single byte to EEPROM.
; data must be in EEDATA, address of EEPROM must be in EEADR

eewrite	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; reset all bits to start
	bsf	EECON1,WREN	; enable writes to eeprom
	movlw	0x55		; protection code to prevent false writes
	movwf	EECON2
	movlw	0xAA
	movwf	EECON2
	bsf	EECON1,WR		; write data

eewait	btfss	EECON1,EEIF	; flag = 1 when done
	goto	eewait

	bcf	STATUS,RP0	; back to bank 0
	return

;-------------------------------END EEWRITE-----------------------------------------


;------------------------------START EEREAD---------------------------------------
; Reads a single byte from EEPROM.
; data goes into EEDATA. Address of EEPROM must be in EEADR first

eeread	bsf	STATUS,RP0	; bank 1
	clrf	EECON1		; clear all bits first
	bsf	EECON1,RD		; read from eeprom@eeadr into eedata
	bcf	STATUS,RP0	; back to bank 0
	
	return

;-------------------------------END EEREAD-----------------------------------------


;==========================START DELAY1===============================================
; Delay = 1.25mS x DELAYN1
; This value is equal to 1/800Hz. Used to create a dot space period.
; DELAYN must be loaded before calling this routine
;------------------------------------------------------------------------------------
delay1	movlw	d'5'	; 5 x 250uS = 1.25mS
	movwf	DLYCNT1
delay1A	movlw	d'81'	; 81 x 3 + 6 + nop = 250
	movwf	DLYCNT2
	nop
delay1B	decfsz	DLYCNT2,f	
	goto	delay1B
	decfsz	DLYCNT1,f	; done 5 loops of 250uS?
	goto	delay1A	; not done 5 yet
	decfsz	DELAYN1,f	; have we done all the 1.25mS loops yet
	goto	delay1	; Not yet	
	nop
	nop
	return		; finished

;---------------------------END DELAY1-----------------------------------------------


;==========================START DELAY10===============================================
; Total delay = 10mS x DELAYN
; DELAYN must be loaded before calling this routine
;------------------------------------------------------------------------------------
delay10	movlw	d'40'	; 40 x 250uS = 10mS
	movwf	DLYCNT1
delayl0A	movlw	d'81'	; 81 x 3 + 6 + nop = 250
	movwf	DLYCNT2
	nop
delayl0B	decfsz	DLYCNT2,f	
	goto	delayl0B
	decfsz	DLYCNT1,f	; done 40 loops of 250uS?
	goto	delayl0A	; not done 40 yet
	decfsz	DELAYN,f	; have we done all the 10mS loops yet
	goto	delay10	; Not yet	
	return		; finished

;---------------------------END DELAY10-----------------------------------------------


;--------------------------START INITPIC----------------------------------------------
; initialisation routine, setup defaults for PIC hardware and registers
;

initpic	bcf	INTCON,GIE	; disable global interupts
	bcf	STATUS,RP0	; go to register bank 0
	clrf	STATUS
	movlw	b'00000000'	; disable all interupts
	movwf	INTCON		; 

	bsf	STATUS,RP0	; go to register bank 1
	movlw	b'00100111'	; pullups enabled, ext clock, prescaler = 256
	movwf	OPTION_REG	; 

	movlw	b'00010000'	; 
	movwf	TRISA		; set port A I/O
	movlw	b'00110000'	; 
	movwf	TRISB		; set port B I/O

	clrf	EECON1
	bcf	STATUS,RP0	; go back to register bank 0

	clrf 	PORTB
	clrf	PORTA
	clrf	MYFLAGS
	movlw	0x40
	movwf	PORTB		; set audio A to D to mid point
	bcf	MUTE		; clear mute line

	return			; finished

;-----------------------------END INITPIC------------------------------------------


;--------------------------INTERUPT START-------------------------------------------
; this interrupt routine is not used, but code included just in case something goes 
; wrong

intserv	movwf	W_TEMP		; save w reg
	movf	STATUS,w		; move status reg into w reg
	bcf	STATUS,RP0	; ensure file register set to bank 0
	movwf	ST_TEMP		; save status reg
	bcf	INTCON,GIE	; disable global interupts 

	movf	ST_TEMP,w	; get status
	movwf 	STATUS		; restore status
	swapf	W_TEMP,f	
	swapf	W_TEMP,w		; restore w reg

	retfie			; return to main program

;--------------------------INTERUPT END--------------------------------------------


	end			; no more
