


; IR remote controller
; using IR remote from littlebird electronics

; to test codes see TEST section (do a search for 'TEST') that code is normally remarked out but allows
; address and commands to be placed in EEPROM when unremarked for viewing by reading values using a programmer

; CPU configuration
	ERRORLEVEL -302
	ERRORLEVEL -306

	list P=16F88
	#include p16f88.inc


;Program Configuration Register 1
		__CONFIG    _CONFIG1, _CP_OFF & _CCP1_RB3  & _DEBUG_OFF & _WRT_PROTECT_OFF & _CPD_OFF & _LVP_OFF & _BODEN_OFF & _MCLR_ON & _PWRTE_ON & _WDT_OFF & _INTRC_IO

;Program Configuration Register 2
		__CONFIG    _CONFIG2, _IESO_OFF & _FCMEN_OFF


; Define variables at memory locations

; EEPROM
EEPROM1		equ	H'00'	; stored toggle/momentary 1-7
EEPROM2		equ	H'01'	; stored toggle/momentary 8 & 9
EEPROM3		equ	H'02'	; osctune
EEPROMC		equ	H'03'	; counter
; used for testing of address and command codes and placed in EEPROM
EEPROM4		equ	H'04'	; REMOTE_A, A for address value
EEPROM5		equ	H'05'	; REMOTE_C, C for command value

; RAM
; bank 0 

TOGGLE_MOM1 equ H'20'	; toggle momentary values 1-7
TOGGLE_MOM2 equ H'21'	; toggle momentary values 8 & 9
COUNTER		equ	H'22'	; counter
REMOTE_A	equ	H'23'	; remote control Address byte
REM_A_BAR	equ	H'24'	; complementary value of address
REMOTE_C	equ	H'25'	; remote control Command byte
REM_C_BAR	equ	H'26'	; complementary value of command
BIT_COUNT	equ	H'27'	; 32 bit counter
STORE1		equ	H'28'	; delay store
STORE2		equ	H'29'	; delay store
FREQ_COUNT	equ	H'2A'	; frequency change counter
DIRECTION	equ	H'2B'	; FREQ_COUNT up down direction

; all banks ram
W_TMP		equ	H'70'	; temporary store for w in interrupt
STATUS_TMP	equ	H'71'	; temporary store of status in interrupt 
INT_FLG 	equ H'72'	; interrupt flag
msCOUNT_MS	equ H'73'	; 110ms counter ms byte
msCOUNT_LS	equ H'74'	; 110ms counter	ls byts
LED_FLASH	equ	H'75'	; LED Flasher
DIVIDERM	equ	H'76'	; divide
DIVIDERL	equ	H'77'	; divide
TEMP		equ	H'78'	; temporary

	ORG     2100		; preprogram EEPROM locations
	DE		D'0'		; 1-7 set at 00 (using bits 1-7 and bit zero not used)
	DE		D'0'		; 8 & 9 are bits 0 and 1
	DE		D'0'		; osctune is 0
	DE		D'08'		; initial zero setting
; remaining EEPROM HFF
		
; define reset and interrupt vector start addresses

	org		0			; start at address 0000h
	goto	MAIN		; normal service routines from Reset vector
	org     4			; interrupt vector 0004h, start interrupt routine here
	goto	INTERRUPT

;_____________________________________________________________________________________
; lookup tables
OSC_SET ; oscillator OSCTUNE set tuning value
	addwf	PCL,f		; FREQ_COUNT value added to PCL (FREQ_COUNT = 0 to 16)
	retlw	B'00100000'	; LED1 max slow @ -31
	retlw	B'00100011'	; LEDs1&2
	retlw	B'00100111'	; LED2
	retlw	B'00101101'	; LEDs2&3
	retlw	B'00110000'	; LED3
	retlw	B'00110011'	; LEDs3&4
	retlw	B'00111000'	; LED4
	retlw	B'00111011'	; LEDs4&5
	retlw	B'00000000'	; LED5	zero change
	retlw	B'00000100'	; LEDs5&6
	retlw	B'00000111'	; LED6
	retlw	B'00001100'	; LEDs6&7
	retlw	B'00001111'	; LED7
	retlw	B'00010010'	; LEDs7&8
	retlw	B'00011000'	; LED8
	retlw	B'00011100'	; LEDs8&9
	retlw	B'00011111'	; LED9 max fast & 31

LED_A_SET ; settings for PORTA LEDS
	addwf	PCL,f		; FREQ_COUNT value added to PCL (FREQ_COUNT = 0 to 16)
	retlw	B'00000000'	; LEDs on PORTA off
	retlw	B'00000000'	; LEDs on PORTA off
	retlw	B'00000000'	; LEDs on PORTA off
	retlw	B'00000000'	; LEDs on PORTA off
	retlw	B'00000000'	; LEDs on PORTA off
	retlw	B'00010000'	; LED4 on
	retlw	B'00010000'	; LED4
	retlw	B'00011000'	; LEDs4&5
	retlw	B'00001000'	; LED5	zero change
	retlw	B'00001100'	; LEDs5&6
	retlw	B'00000100'	; LED6
	retlw	B'10000100'	; LEDs6&7
	retlw	B'10000000'	; LED7
	retlw	B'10000001'	; LEDs7&8
	retlw	B'00000001'	; LED8
	retlw	B'00000011'	; LEDs8&9
	retlw	B'00000010'	; LED9 max fast & 31

LED_B_SET ; settings for PORTB LEDs
	addwf	PCL,f		; FREQ_COUNT value added to PCL (FREQ_COUNT = 0 to 16)
	retlw	B'00000010'	; LED1 
	retlw	B'00000110'	; LEDs1&2
	retlw	B'00000100'	; LED2
	retlw	B'00001100'	; LEDs2&3
	retlw	B'00001000'	; LED3
	retlw	B'00001000'	; LED3
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
	retlw	B'00000000'	; LEDs on PORTB off
; ************************************************************************************	
; start interrupt by saving w and status registers before altered by interrupt routine

INTERRUPT
	
	movwf	W_TMP		; w to w_tmp storage
	swapf	STATUS,w	; status to w
	movwf	STATUS_TMP	; status in status_tmp  
	bcf		STATUS,RP0	; select memory bank 0
	bcf		STATUS,RP1

; adjust interrupt rate with counter

	movlw	D'207'		; freq is 8MHz/4/2/255-207. = 50us see lookup table
	addwf	TMR0,f		; add to timer register and takes 2 cycles to start counting
	bcf		INTCON,TMR0IF	; clear TMR0 interrupt flag

	bsf		INT_FLG,0	; set flag to indicate an interrupt

; increase 110ms repeat counter
	incfsz	msCOUNT_LS,f	; increase 110ms counter ls byte
	goto	END_OF_INT
	incf	msCOUNT_MS,f	; increase 110ms counter ms byte

; end of interrupt reclaim w and status 
END_OF_INT
	swapf	STATUS_TMP,w; status temp storage to w
	movwf	STATUS		; w to status register
	swapf	W_TMP,f		; swap upper and lower 4-bits in w_tmp
	swapf   W_TMP,w		; swap bits and into w register
	retfie				; return from interrupt

;********************************************************************************************** 
  
; RESET		
; Set ports A & B

MAIN
	clrf	PORTA
	clrf	PORTB

; set inputs/outputs
	bsf		STATUS,RP0	; select memory bank 1
	movlw	B'00000111'	; comparators off
	movwf	CMCON
	movlw	B'11110001'	; Inputs/outputs
	movwf	TRISB		; port B data direction register
	movlw	B'00000000'	; 
	movwf	OPTION_REG	; TMR0 prescaler is 2, PORTB pullups
	movlw   B'00100000'	; Inputs/ outputs 
	movwf   TRISA		; port A data direction register


; oscillator
	movlw	B'01110000'	; for 8MHz
	movwf	OSCCON		; osc

; analog or IO
	clrf	ANSEL		; all IO
	bcf		STATUS,RP0	; select memory bank 0

; collect momentary/toggle settings and osctune value

	movlw	EEPROM1		; stored toggle/momentary 1-7
	call	EEREAD
	movwf	TOGGLE_MOM1 ; toggle momentary values 1-7
	movlw	EEPROM2		; stored toggle/momentary 1-7
	call	EEREAD
	movwf	TOGGLE_MOM2 ; toggle momentary values 1-7

	movlw	EEPROM3		; stored osctune value
	call	EEREAD
	bsf		STATUS,RP0	; bank 1
	movwf	OSCTUNE
	bcf		STATUS,RP0	; bank 0
	
; presets
	movlw	EEPROMC		; stored counter value
	call	EEREAD		; sets EEADR (address)
	movwf	FREQ_COUNT	; counter for OSCTUNE 
	clrf	DIRECTION	; up down direction for OSCTUNE
	movlw	D'4'
	movwf	DIVIDERL	; ack LED flasher

; interrupt enable 
INTERRUPT_ENABLE
	bcf		INTCON,TMR0IF	; clear TMR0 interrupt flag
	bsf		INTCON,TMR0IE	; set interrupt enable for TMR0 
	bsf		INTCON,GIE		; set global interrupt enable for above

; check if S1 is closed
	btfsc	PORTB,4
	goto	HIGH_WAIT
; ___________________________________________________________________________________
; adjust OSCTUNE 
; display value using LEDs 1-9. LED 5 is the zero change. LED1 is maximum faster, LED9 is maximum slower
; 8 levels for each faster and slower using LEDs at about 1.5% each change (1.5625%) for up to 12.5% faster or slower

ADJ_OSC
	bcf		INTCON,GIE		; clear global interrupt enable 
; get OSCTUNE value
	movf	FREQ_COUNT,w
	call	OSC_SET		; get OSCTUNE value
	bsf		STATUS,RP0	; bank 1
	movwf	OSCTUNE
	bcf		STATUS,RP0	; bank 0
; store in EEPROM
	movlw	EEPROM3		; stored osctune value
	call	EEREAD		; sets EEADR (address)
	bsf		STATUS,RP0	; bank 1
	movf	OSCTUNE,w
	bcf		STATUS,RP0	; bank 0
	call	EEWRITE

	movlw	EEPROMC		; stored counter value
	call	EEREAD		; sets EEADR (address)
	movf	FREQ_COUNT,w
	call	EEWRITE

; get LED settings
	movf	FREQ_COUNT,w
	call	LED_A_SET ; settings for PORTA LEDS
	movwf	PORTA
	movf	FREQ_COUNT,w
	call	LED_B_SET ; settings for PORTB LEDS
	movwf	PORTB
; wait for S1 to open
; check if S1 at RB4 is open
CK_4
	btfss	PORTB,4
	goto	CK_4
; debounce delay
	call	DELAY1
	btfss	PORTB,4
	goto	CK_4
	
; clear LEDs
	clrf	PORTA
	clrf	PORTB

; monitor portb,0 for input
HIGH_WAIT0
	bcf		INTCON,GIE		; clear global interrupt enable 

; wait for S1 closure 
	btfsc	PORTB,4
	goto	WAIT0
; debounce delay
	call	DELAY1
	btfsc	PORTB,4
	goto	WAIT0
; then clear LEDS
	clrf	PORTA
	clrf	PORTB
	goto	DIR

WAIT0
; wait for PORTB,0 to go high
	bsf		PORTA,6 	; ack LED on	
	btfss	PORTB,0
	goto	HIGH_WAIT0

; Wait for a low at PORTB,0
LOW_WAIT0
	bcf		INTCON,GIE		; clear global interrupt enable 
	bsf		PORTA,6 	; ack LED on

	btfss	PORTB,0
	goto	START_ROUTINE0

; wait for S1 closure 
	btfsc	PORTB,4
	goto	LOW_WAIT0
DEBOUNCE
; debounce delay
	clrf	PORTA
	clrf	PORTB
	call	DELAY1
	btfsc	PORTB,4
	goto	LOW_WAIT0

DIR
	btfsc	DIRECTION,0
	goto	REDUCE
; increase	FREQ_COUNT up to 16 (B'00010000') then down to 0
	incf	FREQ_COUNT,w
	andlw	B'00011111'
	movwf	FREQ_COUNT
; check if >16
	btfss	FREQ_COUNT,4
	goto	ADJ_OSC		; change frequency ready for test
; bit 4 is set so other bits need to be 0 otherwise reduce to 15 
	xorlw	D'16'
	btfsc	STATUS,Z
	goto	ADJ_OSC		; change frequency ready for IR test
	incf	DIRECTION,f	; change direction
	decf	FREQ_COUNT,f
REDUCE
	decfsz	FREQ_COUNT,f ; 
	goto	ADJ_OSC		; change frequency ready for IR test
; zero so change direction
	incf	DIRECTION,f	
	goto	ADJ_OSC		; change frequency ready for IR test

START_ROUTINE0
	clrf	PORTA
	clrf	PORTB
; interrupt enable 
	bsf		INTCON,GIE	; set global interrupt enable 
	call	START		; decode IR code
; on return w has 00, FF or A0
	xorlw	H'00'
	btfsc	STATUS,Z	
	goto	LOW_WAIT0	; code not correct
	xorlw	H'FF'
	btfsc	STATUS,Z	
	goto	HIGH_WAIT0	; code not correct
; code is acceptable

; drive LED for each COMMAND
;A
	movlw	H'1F'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED1
;B
	movlw	H'1E'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED2
;C
	movlw	H'1A'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED3
;<
	movlw	H'08'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED4
;O
	movlw	H'04'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED5
;>
	movlw	H'01'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED6
;^
	movlw	H'05'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED7
;V
	movlw	H'00'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED8
; Operate
	movlw	H'1B'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LED9
	goto	HIGH_WAIT	; code not correct

LED1
	bsf		PORTB,1
	call	REPEAT	; check repeat codes
	bcf		PORTB,1	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED2
	bsf		PORTB,2
	call	REPEAT	; check repeat codes
	bcf		PORTB,2	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED3
	bsf		PORTB,3
	call	REPEAT	; check repeat codes
	bcf		PORTB,3	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED4
	bsf		PORTA,4
	call	REPEAT	; check repeat codes
	bcf		PORTA,4	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED5
	bsf		PORTA,3
	call	REPEAT	; check repeat codes
	bcf		PORTA,3	; when repeat codes end clear output
	goto	HIGH_WAIT0
LED6
	bsf		PORTA,2
	call	REPEAT	; check repeat codes
	bcf		PORTA,2	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED7
	bsf		PORTA,7
	call	REPEAT	; check repeat codes
	bcf		PORTA,7	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED8
	bsf		PORTA,0
	call	REPEAT	; check repeat codes
	bcf		PORTA,0	; when repeat codes end clear output 
	goto	HIGH_WAIT0
LED9
	bsf		PORTA,1
	call	REPEAT	; check repeat codes
	bcf		PORTA,1	; when repeat codes end clear output 
	goto	HIGH_WAIT0

;_________________________________________________________________________

; NORMAL START HERE ROUTINE BEGINS

; monitor portb,0 for input
HIGH_WAIT
	decfsz	DIVIDERM,f
	goto	CK_WAIT_H
	decfsz	DIVIDERL,f
	goto	CK_WAIT_H
	movlw	D'4'
	movwf	DIVIDERL
	bcf		PORTA,6 	; ack LED off
	incf	LED_FLASH,f
	movlw	D'1'
	subwf	LED_FLASH,w
	btfss	STATUS,C
	bsf		PORTA,6		; ack LED on

	
; wait for PORTB,0 to go high
CK_WAIT_H		
	btfss	PORTB,0
	goto	HIGH_WAIT

; Wait for a low at PORTB,0
LOW_WAIT
	decfsz	DIVIDERM,f
	goto	CK_WAIT_L
	decfsz	DIVIDERL,f
	goto	CK_WAIT_L
	movlw	D'4'
	movwf	DIVIDERL
	bcf		PORTA,6 	; ack LED off
	incf	LED_FLASH,f
	movlw	D'1'
	subwf	LED_FLASH,w
	btfss	STATUS,C
	bsf		PORTA,6		; ack LED on
CK_WAIT_L	
	btfss	PORTB,0
	goto	START_ROUTINE
	bcf		INT_FLG,0 	; clear flag that indicates an interrupt ocurred
	goto	LOW_WAIT	

START_ROUTINE
	bcf		PORTA,6 	; ack LED off
	call	START		; decode IR code
; on return w has 00, FF or A0
	
	movwf	TEMP
	xorlw	H'A0'		; correct code AOk
	btfsc	STATUS,Z
	goto	DEC0DE
	movf	TEMP,w
	xorlw	H'00'
	btfsc	STATUS,Z	
	goto	LOW_WAIT	; code not correct
	movf	TEMP,w
	xorlw	H'FF'
	btfsc	STATUS,Z	
	goto	HIGH_WAIT	; code not correct

DEC0DE
; check command value
; if values are (incorectly) shifted left (ie ls bits sent left) codes are as follows. Seems to be what is described 
; with Arduino decoding (address H10)  	
; Operate = HD8 ; A = HF8 ; B = H78 ; C	= H58 ; < = H10 ; O = H20 ; >	= H80 ; ^	= HA0 ; V = 00

; If shifted right as it should be (ie ls bit first sent right)codes are (address H08)
; Operate = H1B
; A = H1F
; B = H1E
; C	= H1A
; < = H08
; O = H04
; >	= H01
; ^	= H05
; V = H00

; if no A, B or C links are in then check all commands

	btfss	PORTB,7	; A link
	goto	CK_A
	btfss	PORTB,6	; B link
	goto	CK_B
	btfss	PORTB,5	; C link
	goto	CK_C
; all links are out so goto all commands
	goto	ALL_COMMS

; a link is in 	
; If A, B or C links are in then it is a device mode where the receiver will be an  A, B or C device

;_________________________________________________________________________________
CK_A ; A link is in

; if 'A' LINK is inserted and 'A' LED is on then this is the 'A' device, so check commands except B and C which should be kept off

; is RB1 high (LED1 on)
	btfss	PORTB,1
	goto	NO_A_LED
; if the 'A' LED is on and it is an B or C code switch LEDs off
	movlw	H'1E'		; B code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
	movlw	H'1A'		; C code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
B_C_OFF
	bcf		PORTB,2	; B LED off
	nop
	nop
	bcf		PORTB,3	; C LED off
	goto	DEVICE_COMMS ; use only device commands 	

NO_A_LED
; if the A link is in but the A LED is not on, then check for an A code. If not an 'A' code then exit

	movlw	H'1F'		; A code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	SEL_A
	call	REPEAT	; check repeat codes
	goto	HIGH_WAIT
; if an 'A' code set the A LED on and clear the B and C LEDs 
SEL_A ; select A device
	bsf		PORTB,1	; LED1 on
	goto	B_C_OFF	; B and C LEDs off

;______________________________________________________________________________
CK_B
; if 'B' LINK is inserted and 'B' LED is on then this is the 'B' device, so check commands except A and C which should be kept off

; is RB2 high (LED2 on)
	btfss	PORTB,2
	goto	NO_B_LED
; if the 'B' LED is on and it is an A or C code switch LEDs off
	movlw	H'1F'		; A code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
	movlw	H'1A'		; C code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
A_C_OFF
	bcf		PORTB,1	; A LED off
	nop
	nop
	bcf		PORTB,3	; C LED off
	goto	DEVICE_COMMS ; use only device commands 
NO_B_LED
; if the B link is in but the B LED is not on, then check for a B code. If not a 'B' code then exit

	movlw	H'1E'		; B code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	SEL_B
	call	REPEAT	; check repeat codes
	goto	HIGH_WAIT
; if a 'B' code set the B LED on and clear the A and C LEDs 
SEL_B ; select B device
	bsf		PORTB,2	; LED2 on
	goto	A_C_OFF	; A and C LEDs off
;_________________________________________________________________________________
CK_C ; C link is in

; if 'C' LINK is inserted and 'C' LED is on then this is the 'C' device, so check commands except A and B which should be kept off

; is RB3 high (LED3 on)
	btfss	PORTB,3
	goto	NO_C_LED
; if the 'C' LED is on and it is an A or B code switch LEDs off
	movlw	H'1E'		; B code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
	movlw	H'1F'		; A code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	CLEAR_ABC
A_B_OFF
	bcf		PORTB,1	; A LED off
	nop
	nop
	bcf		PORTB,2	; B LED off
	goto	DEVICE_COMMS ; use only device commands 
NO_C_LED
; if the C link is in but the C LED is not on, then check for a C code. If not a 'C' code then exit

	movlw	H'1A'		; C code
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	SEL_C
	call	REPEAT	; check repeat codes
	goto	HIGH_WAIT
; if an 'C' code sets the C LED on and clears the A and B LEDs 
SEL_C ; select C device
	bsf		PORTB,3	; LED3 on
	goto	A_B_OFF	; A and B LEDs off

;____________________________________________________
CLEAR_ABC
;  clear the on 'A,B,C' LEDs  
	bcf		PORTB,1	; A LED1 off
	nop
	nop
	bcf		PORTB,2	; B LED2 off
	nop
	nop
	bcf		PORTB,3	; C LED2 off
	goto	HIGH_WAIT
;______________________________________________________________________________

ALL_COMMS ; all commands

;A
	movlw	H'1F'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	A_OUT
;B
	movlw	H'1E'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	B_OUT
;C
	movlw	H'1A'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	C_OUT

DEVICE_COMMS ; only device commands

;^
	movlw	H'05'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	UP
;<
	movlw	H'08'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	LEFT
;O
	movlw	H'04'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	ZERO
;>
	movlw	H'01'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	RIGHT
;V
	movlw	H'00'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	DOWN

; Operate
	movlw	H'1B'
	xorwf	REMOTE_C,w
	btfsc	STATUS,Z
	goto	OPERATE

; no valid command	
	call	REPEAT		; when repeat codes end, go to start of checking for codes
	goto	HIGH_WAIT
;_________________________________________________________________________
A_OUT ;(CON1)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_1
; swap mode
	btfss	TOGGLE_MOM1,1	; bit1 has output 1 A mode. When set is toggle
	goto	TOGGLE_1
	bcf		TOGGLE_MOM1,1	; bit1 has output 1 (A) mode. clear is momentary
; store in EEPROM
STO_1
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_1
TOGGLE_1
	bsf		TOGGLE_MOM1,1	; bit 1 has output 1  A mode. set is toggle
	goto	STO_1

SELECT_1
; check if toggle or momentary
	btfss	TOGGLE_MOM1,1	; bit1 has output 1 A mode. When set is toggle
	goto	MOMENTARY_1
; Toggle/swap output state
	btfss	PORTB,1
	goto	SET_B1
	bcf		PORTB,1
	call	REPEAT
	goto	HIGH_WAIT
SET_B1
	bsf		PORTB,1
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_1
; momentary
	bsf		PORTB,1
	call	REPEAT	; check repeat codes
	bcf		PORTB,1	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT
;______________________________________________________________________________
B_OUT ;(CON2)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_2
; swap mode
	btfss	TOGGLE_MOM1,2	; bit2 has output 2 B mode. When set is toggle
	goto	TOGGLE_2
	bcf		TOGGLE_MOM1,2	; bit2 has output 2 (B) mode. clear is momentary
; store in EEPROM
STO_2
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_2
TOGGLE_2
	bsf		TOGGLE_MOM1,2	; bit 2 has output  B mode. set is toggle
	goto	STO_2

SELECT_2
; check if toggle or momentary
	btfss	TOGGLE_MOM1,2	; bit2 has output 2 B mode. When set is toggle
	goto	MOMENTARY_2
; Toggle/swap output state
	btfss	PORTB,2
	goto	SET_B2
	bcf		PORTB,2
	call	REPEAT
	goto	HIGH_WAIT
SET_B2
	bsf		PORTB,2
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_2
; momentary
	bsf		PORTB,2
	call	REPEAT	; check repeat codes
	bcf		PORTB,2	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT
;_________________________________________________________________
C_OUT ;(CON3)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_3
; swap mode
	btfss	TOGGLE_MOM1,3	; bit3 has output 3 C mode. When set is toggle
	goto	TOGGLE_3
	bcf		TOGGLE_MOM1,3	; bit3 has output 3 (C) mode. clear is momentary
; store in EEPROM
STO_3
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_3
TOGGLE_3
	bsf		TOGGLE_MOM1,3	; bit 3 has output 3 C mode. set is toggle
	goto	STO_3

SELECT_3
; check if toggle or momentary
	btfss	TOGGLE_MOM1,3	; bit3 has output 3 C mode. When set is toggle
	goto	MOMENTARY_3
; Toggle/swap output state
	btfss	PORTB,3
	goto	SET_B3
	bcf		PORTB,3
	call	REPEAT
	goto	HIGH_WAIT
SET_B3
	bsf		PORTB,3
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_3
; momentary
	bsf		PORTB,3
	call	REPEAT	; check repeat codes
	bcf		PORTB,3	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT
;_________________________________________________________________________________
UP ;(CON4)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_4
; swap mode
	btfss	TOGGLE_MOM1,4	; bit4 has output 4 up mode. When set is toggle
	goto	TOGGLE_4
	bcf		TOGGLE_MOM1,4	; bit4 has output 4 (up) mode. clear is momentary
; store in EEPROM
STO_4
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_4
TOGGLE_4
	bsf		TOGGLE_MOM1,4	; bit 4 has output 4 up mode. set is toggle
	goto	STO_4

SELECT_4
; check if toggle or momentary
	btfss	TOGGLE_MOM1,4	; bit4 has output 4 up mode. When set is toggle
	goto	MOMENTARY_4
; Toggle/swap output state
	btfss	PORTA,4
	goto	SET_A4
	bcf		PORTA,4
	call	REPEAT
	goto	HIGH_WAIT
SET_A4
	bsf		PORTA,4
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_4
; momentary
	bsf		PORTA,4
	call	REPEAT	; check repeat codes
	bcf		PORTA,4	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT
;_____________________________________________________________________________________

LEFT ;(CON5)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_5
; swap mode
	btfss	TOGGLE_MOM1,5	; bit5 has output 5 left mode. When set is toggle
	goto	TOGGLE_5
	bcf		TOGGLE_MOM1,5	; bit5 has output 5 (left) mode. clear is momentary
; store in EEPROM
STO_5
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_5
TOGGLE_5
	bsf		TOGGLE_MOM1,5	; bit 5 has output 5 left mode. set is toggle
	goto	STO_5

SELECT_5
; check if toggle or momentary
	btfss	TOGGLE_MOM1,5	; bit5 has output 5 left mode. When set is toggle
	goto	MOMENTARY_5
; Toggle/swap output state
	btfss	PORTA,3
	goto	SET_A3
	bcf		PORTA,3
	call	REPEAT
	goto	HIGH_WAIT
SET_A3
	bsf		PORTA,3
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_5
; momentary
	bsf		PORTA,3
	call	REPEAT	; check repeat codes
	bcf		PORTA,3	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT

;_____________________________________________________________________________________
ZERO ;(CON6)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_6
; swap mode
	btfss	TOGGLE_MOM1,6	; bit6 has output 6 zero mode. When set is toggle
	goto	TOGGLE_6
	bcf		TOGGLE_MOM1,6	; bit6 has output 6 (zero) mode. clear is momentary
; store in EEPROM
STO_6
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_6
TOGGLE_6
	bsf		TOGGLE_MOM1,6	; bit 6 has output 6 zero mode. set is toggle
	goto	STO_6

SELECT_6
; check if toggle or momentary
	btfss	TOGGLE_MOM1,6	; bit6 has output 6 zero mode. When set is toggle
	goto	MOMENTARY_6
; Toggle/swap output state
	btfss	PORTA,2
	goto	SET_A2
	bcf		PORTA,2
	call	REPEAT
	goto	HIGH_WAIT
SET_A2
	bsf		PORTA,2
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_6
; momentary
	bsf		PORTA,2
	call	REPEAT	; check repeat codes
	bcf		PORTA,2	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT

;___________________________________________________________________________
RIGHT ;(CON7)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_7
; swap mode
	btfss	TOGGLE_MOM1,7	; bit7 has output 7 (right) mode. When set is toggle
	goto	TOGGLE_7
	bcf		TOGGLE_MOM1,7	; bit7 has output 7 (right) mode. clear is momentary
; store in EEPROM
STO_7
	movlw	D'00'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM1,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_7
TOGGLE_7
	bsf		TOGGLE_MOM1,7	; bit 7 has output 7 right mode. set is toggle
	goto	STO_7

SELECT_7
; check if toggle or momentary
	btfss	TOGGLE_MOM1,7	; bit7 has output 7 right mode. When set is toggle
	goto	MOMENTARY_7
; Toggle/swap output state
	btfss	PORTA,7
	goto	SET_A7
	bcf		PORTA,7
	call	REPEAT
	goto	HIGH_WAIT
SET_A7
	bsf		PORTA,7
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_7
; momentary
	bsf		PORTA,7
	call	REPEAT	; check repeat codes
	bcf		PORTA,7	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT

;__________________________________________________________________________
DOWN ; (CON8)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_8
; swap mode
	btfss	TOGGLE_MOM2,0	; bit0 has output 8 (down) mode. When set is toggle
	goto	TOGGLE_8
	bcf		TOGGLE_MOM2,0	; bit0 has output 8 (down) mode. clear is momentary
; store in EEPROM
STO_8
	movlw	D'01'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM2,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_8
TOGGLE_8
	bsf		TOGGLE_MOM2,0	; bit 0 has output 8 down mode. set is toggle
	goto	STO_8

SELECT_8
; check if toggle or momentary
	btfss	TOGGLE_MOM2,0	; bit0 has output 8 (down) mode. When set is toggle
	goto	MOMENTARY_8
; Toggle/swap output state
	btfss	PORTA,0
	goto	SET_A0
	bcf		PORTA,0
	call	REPEAT
	goto	HIGH_WAIT
SET_A0
	bsf		PORTA,0
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_8
; momentary
	bsf		PORTA,0
	call	REPEAT	; check repeat codes
	bcf		PORTA,0	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT

;_______________________________________________________
OPERATE ; (CON9)
; test switch S1
	btfsc	PORTB,4		; when low switch is pressed
	goto	SELECT_9
; swap mode
	btfss	TOGGLE_MOM2,1	; bit1 has output 9 (operate) mode. When set is toggle
	goto	TOGGLE_9
	bcf		TOGGLE_MOM2,1	; bit1 has output 9 (operate) mode. clear is momentary
; store in EEPROM
STO_9
	movlw	D'01'
	call	EEREAD		; sets EEADR
	movf	TOGGLE_MOM2,w
	bcf		INTCON,GIE		; clear global interrupt enable
	call	EEWRITE
	bsf		INTCON,GIE		; set global interrupt enable
	goto	SELECT_9
TOGGLE_9
	bsf		TOGGLE_MOM2,1	; bit 1 has output 9 operate mode. set is toggle
	goto	STO_9

SELECT_9
; check if toggle or momentary
	btfss	TOGGLE_MOM2,1	; bit1 has output 9 (operate) mode. When set is toggle
	goto	MOMENTARY_9
; Toggle/swap output state
	btfss	PORTA,1
	goto	SET_A1
	bcf		PORTA,1
	call	REPEAT
	goto	HIGH_WAIT
SET_A1
	bsf		PORTA,1
	call	REPEAT			
	goto	HIGH_WAIT

MOMENTARY_9
; momentary
	bsf		PORTA,1
	call	REPEAT	; check repeat codes
	bcf		PORTA,1	; when repeat codes end clear output and go to start of checking for codes
	goto	HIGH_WAIT


; ************************************************
; IR subroutines

; read IR signal that's based on pulse distance protocol
; start train is a 38kHz burst at 9ms and 4.5ms of no burst
; data sent as an 8-bit address (LSB first)then a complementary version of the address
; further data sent as an 8-bit command (LSB first)then a complementary version of the command
; logic 1 is 560us 38kHz burst and 1690us of no burst
; logic 0 is a 560us 38kHz burst and 560us of no burst
; At the end of the 32bit data is a tail pulse comprising 560us of a 38kHz burst
; data is sent once
; a repeat indication (transmit button held on) is 9000us 38kHz burst followed by 2250us no burst and a 560us 38kHz burst
; this is repeated each 110ms
; **********************************


START ; start of Infra Red reception
; start 110ms counter at beginning of code
	clrf	msCOUNT_MS	; 110ms counter
	clrf	msCOUNT_LS


; Check for start train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 4.5ms (no 38kHz signal)
; count 180 x 50us interrupts where PORTB,0 needs to stay low
; 
	movlw	D'160'	; allow an 8ms minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0
	bsf		PORTA,6		; Ack LED on
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0
; PORTB,0 can now go high up to a period of 2ms
; wait for more interrupts
	movlw	D'40'	; 2ms
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_1
	btfsc	PORTB,0
	goto	NINE_ms		; PORTB,0 has gone high within the acceptable ~9ms low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_1
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_1
	retlw 	H'FF' 		; if PORTB,0 stays low  

NINE_ms	; PORTB,0 has stayed low and then gone high within the acceptable ~9ms low period
; Check for 4.5ms high
	bcf		PORTA,6		; Ack LED off
	movlw	D'80'		; allow a 4ms minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_2
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_2 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_2

; PORTB,0 can now go low up to a period of 1ms
; wait for more interrupts
	movlw	D'20'		; 1ms
	movwf	COUNTER
; PORTB,0 can go low in this period
CK_B0_3
	btfss	PORTB,0
	goto	FOUR_POINT_FIVE_ms	; PORTB,0 has gone low within the acceptable ~4.5ms high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_3
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_3
	retlw	H'00'		; PORTB stays high

; end of start train check
FOUR_POINT_FIVE_ms
	
	bsf		PORTA,6		; Ack LED on
	movlw	D'32'		; set up 32bit counter
	movwf	BIT_COUNT

START_DATA ; start of data 
; clear data
	clrf	REMOTE_A	; remote control Address byte
	clrf	REM_A_BAR	; complementary value of address
	clrf	REMOTE_C	; remote control Command byte
	clrf	REM_C_BAR	; complementary value of command

; start reading data
READ_DATA
; a 560us low
	movlw	D'8'		; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_4
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_4 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_4
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'		; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_5
	btfsc	PORTB,0
	goto	ONE_ZERO	; PORTB,0 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_5
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_5
	retlw 	H'FF' 		;  PORTB,0 stayss low  
; test for a zero (high for 560us) or a one (high for 1690us)
ONE_ZERO
	bcf		PORTA,6		; Ack LED off
; Check for 560us high
	movlw	D'8'		; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_6
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_6 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_6

; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'		; 100us
	movwf	COUNTER

; PORTB,0 can go low in this period
CK_B0_7
	btfss	PORTB,0
	goto	LOGIC_0		; PORTB,0 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_7
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_7
	goto	CHECK_FOR_1

LOGIC_0	; logic zero detected
	bsf		PORTA,6		; Ack LED on
	bcf		STATUS,C	; clear carry
; shift detected data into storage
SHIFT_DATA	
	rrf		REM_C_BAR,f	; complementary value of command
	rrf		REMOTE_C,f	; remote control Command byte
	rrf		REM_A_BAR,f	; complementary value of address
	rrf		REMOTE_A,f	; remote control Address byte
	

	decfsz	BIT_COUNT,f	; shift data for 32 bits
	goto	READ_DATA	; contine reading data
	goto	TAIL		; end of 32 bit data, check for the tail (560us of burst)
	
CHECK_FOR_1 ; check for a logic 1

; Check for 1690us high  (already 560us high) so check for 1130us
	movlw	D'15'	; allow a 1100s minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_8
	btfss	PORTB,0
	retlw	H'FF'		; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_8 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_8

; PORTB,0 can now go high up to a period of 50us extra
; wait for more interrupts
	movlw	D'10'	; 50us
	movwf	COUNTER

; PORTB,0 can go low in this period
CK_B0_9
	btfss	PORTB,0
	goto	LOGIC_1		; PORTB,0 has gone low within the acceptable ~560us high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_9
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_9
	retlw	H'00'		; PORTB stays high	

LOGIC_1
	bsf		PORTA,6		; Ack LED on
	bsf		STATUS,C	; data is a 1
	goto	SHIFT_DATA

TAIL ; end of data tail

; a 560us low
	movlw	D'8'	; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_10
	btfsc	PORTB,0
	retlw 	H'00'		; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_10 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_10
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'	; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_11
	btfsc	PORTB,0
	goto	TAIL_DET	; PORTB,0 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_11
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_11
	retlw 	H'FF' 		; if PORTB,0 stays low  

TAIL_DET ; tail is detected
	bcf		PORTA,6		; Ack LED off

; check if Address and Address bar are complementary

	comf	REM_A_BAR,w	; complementary value of address
	xorwf	REMOTE_A,w	; remote control Address byte
	btfss	STATUS,Z
	retlw	H'00'		; PORTB high

; ********************************************

; For the particular remote that uses 08 as address,
; check for a 08 to specify the IR transmitter used or change if different
; or remark out if for use with other IR remotes with different address
	movlw	H'08'
	xorwf	REMOTE_A,w	; value of address
	btfss	STATUS,Z
	goto	HIGH_WAIT
; *******************************************
; check if Command and Command bar are complementary

	comf	REM_C_BAR,w	; complementary value of command
	xorwf	REMOTE_C,w	; remote control Command byte
	btfss	STATUS,Z
	goto	HIGH_WAIT

TEST
; test is normally remarked out
; test for codes. Store in EEPROM so remote address and command (a button code) can be read using PIC programmer (eg PICkit 3)
; to read EEPROM
;	movlw	EEPROM4		; EEPROM address @04
;	call	EEREAD		; sets EEADR
;	movf	REMOTE_A,w	; address value
;	call	EEWRITE
;	movlw	EEPROM5		; EEPROM address @05
;	call	EEREAD
;	movf	REMOTE_C,w	; command value
;	call	EEWRITE
; end test

	retlw	H'A0'		; if data is decoded correctly send an 'AO'k return message

;------------------------------------------------------------------------
REPEAT
; wait for a low after 110ms (see msCOUNT_MS msCOUNT_LS	; 110ms counter)

; REPEAT is a 9ms burst followed by 2250us off and 560us burst repeated 110ms apart
; wait for mscount within a 110ms range

KEEP_CHK_COUNT
	movf	msCOUNT_MS,w
	sublw	H'07'
	btfsc	STATUS,C
	goto	KEEP_CHK_COUNT

; wait for PORTB,0 to go low
KEEP_CHK_LOW
; check if ms count is less than A00
	movf	msCOUNT_MS,w
	sublw	H'0A'
	btfss	STATUS,C
	return
	btfsc	PORTB,0	; when low check if within time frame allowed
	goto	KEEP_CHK_LOW

; start 110ms counter at beginning of repeat frame
	clrf	msCOUNT_MS	; 110ms counter
	clrf	msCOUNT_LS
; Check for repeat train
; comprises a 9ms 38kHz burst, read as a low of 9ms on IR detector, then followed by a high of 2.25ms (no 38kHz signal) then a 560us burst
; count 180 x 50us interrupts where PORTB,0 needs to stay low
; 
	movlw	D'160'	; allow an 8ms minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_12
	bsf		PORTA,6		; Ack LED on
	btfsc	PORTB,0
	return				; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_12 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_12
; PORTB,0 can now go high up to a period of 2ms
; wait for more interrupts
	movlw	D'40'	; 2ms
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_13
	btfsc	PORTB,0
	goto	NINE_ms_1	; PORTB,0 has gone high within the acceptable ~9ms low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_13
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_13
	return

NINE_ms_1

	bcf		PORTA,6		; Ack LED off
; Check for 2.25ms high
	movlw	D'40'	; allow a 2ms minimum 
	movwf	COUNTER
; check PORTB,0

CK_B0_14
	btfss	PORTB,0
	return				; if PORTB,0 goes low then not a continuous high
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_14 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_14

; PORTB,0 can now go low up to a period of 500us
; wait for more interrupts
	movlw	D'10'	; 500us
	movwf	COUNTER
; PORTB,0 can go low in this period
CK_B0_15
	btfss	PORTB,0
	goto	TWOPOINTTWOFIVE_ms	; PORTB,0 has gone low within the acceptable ~2.25ms high period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_15
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_15
	return
TWOPOINTTWOFIVE_ms
; a 560us low
	bsf		PORTA,6		; Ack LED on
	movlw	D'8'	; allow a 500us minimum 
	movwf	COUNTER
; check PORTB,0
CK_B0_16
	btfsc	PORTB,0
	return				; if PORTB,0 goes high then not a continuous low
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_16 
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_16
; PORTB,0 can now go high up to a period of 100us for a 660us maximum
; wait for more interrupts
	movlw	D'6'	; 100us
	movwf	COUNTER
; PORTB,0 can go high in this period
CK_B0_17
	btfsc	PORTB,0
	goto	REP_DET		; PORTB,0 has gone high within the acceptable ~560us low period
	btfss	INT_FLG,0	; check for an interrupt at 50us rate
	goto	CK_B0_17
	bcf		INT_FLG,0
	decfsz	COUNTER,f	; count interrupts
	goto	CK_B0_17
	return

REP_DET

	bcf		PORTA,6		; Ack LED off
	goto	REPEAT		; continue to check repeat pulse train
		  	
; **********************************************************

; general subroutines

DELAY1
	movlw	D'255'	; ~150ms
	movwf	STORE1
LOOP2
	movlw	D'255'
	movwf	STORE2
LOOP1
	nop
	nop
	decfsz	STORE2,f
	goto	LOOP1
	decfsz	STORE1,f
	goto	LOOP2
	return


; subroutine to read EEPROM memory 

EEREAD
	bsf 	STATUS,RP1	; select memory bank 2
	movwf 	EEADR		; indirect special function register
	bsf 	STATUS,RP0	; select memory bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,RD	; read EEPROM
	bcf 	STATUS,RP0	; select memory bank 2
	movf	EEDATA,w	; EEPROM value in w
	bcf		STATUS,RP1	; select bank 0
	return

; subroutine to write to EEPROM

; EEPROM write 

EEWRITE
	bsf		STATUS,RP1	; select bank 2
	movwf	EEDATA		; data register
	bsf		STATUS,RP0	; bank 3
	bcf		EECON1,EEPGD; data memory
	bsf		EECON1,WREN	; enable write
	movlw	H'55'		; place 55H in w for write sequence
	movwf 	EECON2 		; write 55H to EECON2
	movlw 	H'AA'		; AAH to w
	movwf	EECON2		; write AA to EECON2
	bsf		EECON1,WR	; set WR bit and begin write sequence
	bcf		EECON1,WREN	; clear WREN bit
WRITE1	

	btfsc	EECON1,WR	; skip if write complete WR=0 when write complete
	goto 	WRITE1		; not written yet
	bcf		EECON1,EEIF	; clear write interrupt flag
	bcf		STATUS,RP0
	bcf		STATUS,RP1	; bank 0 
	return				; value written
	
	end


	
