#include "io.h"
#include "util.h"

char dt=0;
long displayEnergy=0;
unsigned int adcref1,dacref1;
unsigned int adcref2,dacref2;
unsigned int adcref4,dacref4;

char adccPchans[ADC_RES_COUNT]={VREF    ,VREF    ,AN_J    ,AN_I    ,AN_A    ,ADC_DAC2,AN_C    ,AN_D    ,AN_E    ,AN_F    ,AN_C    ,AN_D    ,VREF    };
char adccNchans[ADC_RES_COUNT]={AN_J    ,AN_I    ,ADC_SE  ,ADC_SE  ,ADC_SE  ,ADC_SE  ,ADC_SE  ,ADC_SE  ,ADC_SE  ,ADC_SE  ,AN_E    ,AN_F    ,ADC_SE  };
char adccRefs  [ADC_RES_COUNT]={FVR_4096,FVR_4096,FVR_4096,FVR_4096,FVR_4096,FVR_1024,FVR_4096,FVR_4096,FVR_4096,FVR_4096,FVR_4096};
//see adccChans_t for enum/ordering
volatile unsigned int adccResInt[ADC_RES_COUNT];
volatile unsigned int adccResLPF[3];

adccChans_t adccPtr=0;
volatile char adccDone=0;    //when a round has been completed
volatile char adccPer=0;    //how many T0 counts to do samples
//from doCalc()
USBstate_t internal;
//loaded by loadADCC()
USBstate_t external;
adccState_t adccState=ADCC_OFF;
char volatile accumE=0;

void ioInit(void){
    //these are in DIA, see pic16f18146.h
    adcref1=readNVM(DIA_FVRA1X);
    dacref1=readNVM(DIA_FVRC1X);
    adcref2=readNVM(DIA_FVRA2X);
    dacref2=readNVM(DIA_FVRC2X);
    adcref4=readNVM(DIA_FVRA4X);
    dacref4=readNVM(DIA_FVRC4X);
    //set everything as analog inputs with pullup to avoid leakage
    ANSELA=0xFF;
    ANSELB=0xFF;
    ANSELC=0xFF;
    WPUA=0xFF;
    WPUB=0xFF;
    WPUC=0xFF;    
    //switches
    anselbits.S1=0; //digital inputs with pullups
    anselbits.S2=0;
    anselbits.S3=0;
    trisbits.S1=1;
    trisbits.S2=1;
    trisbits.S3=1;
    wpubits.S1=1;
    wpubits.S2=1;
    wpubits.S3=1;
    //analogs
    wpubits.AN_A=0;
    wpubits.AN_C=0;
    wpubits.AN_D=0;
    wpubits.AN_E=0;
    wpubits.AN_F=0;
    wpubits.VREF=0;
    wpubits.AN_I=0;
    wpubits.AN_J=0;
    anselbits.AN_A=1;
    anselbits.AN_C=1;
    anselbits.AN_D=1;
    anselbits.AN_E=1;
    anselbits.AN_F=1;
    anselbits.VREF=1;
    anselbits.AN_I=1;
    anselbits.AN_J=1;
    trisbits.AN_A=1;
    trisbits.AN_C=1;
    trisbits.AN_D=1;
    trisbits.AN_E=1;
    trisbits.AN_F=1;
    trisbits.VREF=1;
    trisbits.AN_I=1;
    trisbits.AN_J=1;
    //REG control
    wpubits.REGEN=0;    
    anselbits.REGEN=0;
    latbits.REGEN=0;    //off
    trisbits.REGEN=0;   //enable output
}

void pwmInit(void){
    //PPS
    unsigned char intcon=INTCON;
    GIE=0;
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 0; //clear PPSLOCKED bit
    pps.PWM1S1PIN=0x0B;     //PWM1S1P1_OUT
    PPSLOCK = 0x55; //Required sequence
    PPSLOCK = 0xAA; //Required sequence
    PPSLOCKbits.PPSLOCKED = 1; //set PPSLOCKED bit           
    INTCON=intcon;   
    //pin
    anselbits.PWM1S1PIN=0;
    wpubits.PWM1S1PIN=0;
    trisbits.PWM1S1PIN=0;
    //PWM1S1
    PWM1CON=0;          //reset
    PWM1ERS=0;
    //PWM1CLK=2;          //FOSC
    PWM1CLK=5;          //MFINTOSC 500kHz
    PWM1LDS=0;
    PWM1PR=PWM_PERIOD;
    PWM1CPRE=0;         //no prescale
    PWM1S1CFG=0;        //left aligned, normal polarity
    PWM1S1P1=PWM_PERIOD/2;        //duty
    PWM1GIE=0;          //no ints    
    PWM1CONbits.LD=1;   //load
    PWM1CONbits.EN=1;   //on    
}

void pwmDeInit(void){
    PWM1CONbits.EN=0;   //off    
}

void t0init(void){      //count at 244Hz (ie no interrupts, use raw 16-bit counter value)
    //16bit
    T0CON0=0;               //reset, 8-bit, 1:1 postscale
    T0CON1=0;               //T0CKI, ASYNC off, PS=1:1
    //TMR0L=0;
    //TMR0H=0;                //reset
    T0CON0bits.MD16=1;      //16-bit, period is 65536
    T0CON1bits.CS=5;        //MFINTOSC 500kHz
    //T0CON1bits.CS=4;        //LFINTOSC 31kHz
    //T0CON1bits.CS=2;        //FOSC/4    
    T0CON1bits.ASYNC=1;     //ASYNC
    T0CON1bits.T0CKPS=11;    //1:2^n
    //T0CON0bits.OUTPS=14;    //1:15   1:15*65536 => 4Hz
    T0CON0bits.EN=1;        //on
    //PIR0bits.TMR0IF=0;      //clear
    //PIE0bits.TMR0IE=1;      //enable
/*
    //8bit
    T0CON0=0;               //reset, 8-bit, 1:1    
    T0CON1bits.CS=2;        //FOSC/4=8MHz
    T0CON1bits.ASYNC=1;     //ASYNC
    T0CON1bits.T0CKPS=7;    //1:128
    T0CON0bits.OUTPS=15;    //1:16
    TMR0H=255;              //full period
    T0CON0bits.EN=1;        //on
    PIR0bits.TMR0IF=0;      //clear
    PIE0bits.TMR0IE=1;      //enable
*/
}
unsigned long getT0long(void){  //watch for rollovers and provide extended count
    //should be correct as long as called frequently enough (every 4 min)
    static unsigned int t0Old=0;
    static unsigned long t0hi=0;
    unsigned int t0new=TMR0L; //fill using read sequence, LSB buffers MSB
    t0new=t0new|(((unsigned int)TMR0H) << 8);
    if(t0new<t0Old){//rollover has happened
        t0hi=t0hi+(1UL<<16);
    }
    t0Old=t0new;
    return t0hi|t0new;    
}

void initADCC(void){ //set up with computation
    ADCON0=0;           //reset ADC, single ended, use FOSC
    ADCON1=0;
    ADCON2=0;           //legacy mode
    ADCON3=0;
    ADPRE=0;
    ADACQ=0;            //extra acquisition time, reduce sampling rate
    ADCAP=0;            //extra capacitance
    ADREF=3;            //FVR
    ADCON0bits.CS=1;    //ADCRC    
    ADCON0bits.ADFM0=1; //right justify, two's complement
    ADCON2bits.MD=ADCC_MD_BURST;
    ADCON2bits.ADCRS=2; //right shift
    ADCON3bits.TMD=ADCC_ALWAYS_INT; //set ADTIF on burst complete
    ADCON0bits.ADON=1;  //turn on, ready        
    adccState=ADCC_IDLE;
}

void deInitADCC(void){
    ADCON0=0;
    adccState=ADCC_OFF;
}

unsigned int getADCC(char c){ //computation
    ADCON0bits.IC=0;    //switch back to single ended
    ADPCH=c;
    ADRPT=ADCC_OS;
    ADCNT=0;
    ADACC=0;
    ADCON0bits.GO=1;    //start
    while(ADCON0bits.GO){}     //wait           
    return ADFLTR;    
}

int getADCCdiff(char p, char n){ //differential/computation
    ADCON0bits.IC=1;    //differential
    ADPCH=p;
    ADNCH=n;
    ADRPT=ADCC_OS;
    ADCNT=0;
    ADACC=0;
    ADCON0bits.GO=1;    //start
    while(ADCON0bits.GO){}     //wait           
    return (int)ADFLTR+ADC_DIFF_OFFSET;    
}

int getADCCint(unsigned int u){
    return (int)(u+ADC_DIFF_OFFSET);
}

void isrADCC(void){
    static char t0=0;
    char t1;
    if(PIR6bits.ADIF && PIE6bits.ADIE){
        PIR6bits.ADIF=0;
        if(ADCON0bits.GO==0){   //cleared on completion of sequence
            adccResInt[adccPtr]=ADFLTR;
            adccPtr++;
            if(adccPtr>=ADC_RES_COUNT){
                adccPtr=0;
                adccDone=1;
                t1=TMR0L;
                adccPer=t1-t0;
                t0=t1;
                dt++;
            }
            startADCC();
            if(adccPtr==0){
                //trisbits.S3=0;  //instrumenting
                doCalc();   //housekeeping/derived values etc                        
                //trisbits.S3=1;  //instrumenting        
            }            
        }
    }
}

void stopADCC(void){    //stop, should allow seamless restart
    unsigned char intcon=INTCON;
    GIE=0;
    PIE6bits.ADTIE=0;   //disable
    PIE6bits.ADIE=0;    //disable
    ADCON0bits.GO=0;    //cancel, restart resumes current channel
    adccState=ADCC_IDLE;
    INTCON=intcon;        
}

void startADCC(void){   //enables ISR/ongoing sampling etc    
    initFVR_ADC(adccRefs[adccPtr]);
    ADPCH=adccPchans[adccPtr];
    if(adccNchans[adccPtr]==ADC_SE){
        ADCON0bits.IC=0;    //switch back to single ended        
    }else{
        ADCON0bits.IC=1;    //differential
        ADNCH=adccNchans[adccPtr];        
    }
    ADCON2bits.ACLR=1;  //reset AOV
    ADRPT=ADCC_OS;
    ADCNT=0;
    ADACC=0;
    //PIE6bits.ADTIE=1;
    PIE6bits.ADIE=1;
    PEIE=1;
    adccState=ADCC_RUNNING;
    ADCON0bits.GO=1;    //start
}

void loadADCC(void){
    unsigned char intcon=INTCON;
    GIE=0;
    //load
    external=internal;
    INTCON=intcon;        
}

void initADC(void){
    ADCON0=0;           //reset ADC, single ended, use FOSC
    ADCON1=0;
    ADCON2=0;           //legacy mode
    ADCON3=0;
    ADPRE=0;
    ADACQ=0;
    ADCAP=0;            //extra capacitance
    //ADCLK=63;
    ADCON0bits.CS=1;    //ADCRC    
    ADCON0bits.ADFM0=1; //right justify, two's complement
    //ADREF=0;            //VDD    
    ADREF=3;            //FVR
    ADCON0bits.ADON=1;  //turn on, ready
}

void deInitADC(void){
    ADCON0=0;
}

unsigned int getADC(char c){
    char i;
    unsigned int r=0;
    ADCON0bits.IC=0;    //switch back to single ended
    ADPCH=ADC_VSS;
    ADCON0bits.GO=1;    //dummy VCC
    while(ADCON0bits.GO){}            
    ADPCH=c;
    ADCON0bits.GO=1;    //dummy
    while(ADCON0bits.GO){}            
    for(i=0;i<ADC_OS;i++){
        ADCON0bits.GO=1;
        while(ADCON0bits.GO){}            
        r=r+ADRES;
    }
    return r;
}

int getADCdiff(char p, char n){ //differential
    char i;
    int r=0;
    int a;
    ADCON0bits.IC=1;    //differential
    ADPCH=ADC_VSS;
    ADNCH=ADC_VSS;
    ADCON0bits.GO=1;    //dummy VCC
    while(ADCON0bits.GO){}            
    ADPCH=p;
    ADNCH=n;
    ADCON0bits.GO=1;    //dummy
    while(ADCON0bits.GO){}            
    for(i=0;i<ADC_OS;i++){
        ADCON0bits.GO=1;
        while(ADCON0bits.GO){}            
        a=(int)ADRES;
        r=r+a;
    }
    return r+(2*ADC_OS);    //systematic error?
}

unsigned int getVCC(void){  //in mV
    unsigned long r;
    unsigned char adref=ADREF;
    ADREF=0;            //VDD    
    r=getADC(FVRBUF1);
    ADREF=adref;
    if(r<(adcref4/2)){return 8192;}    //possible overflow/error
    return (unsigned int)(((unsigned long)adcref4)*(4096UL)/r);
}

unsigned int readNVM(unsigned int add){ //maps top bit to NVMREGS
    NVMREGS=0;
    if(add&0x8000){NVMREGS=1;}
    NVMADR=add;
    NVMCON1bits.RD=1;   //start read
    return NVMDAT;    
}

void initFVR_DAC(char n){
    FVRCON=(FVRCON&0xF3)|0x80|((n<<2)&0xC);    
}

void initFVR_ADC(char n){
    FVRCON=(FVRCON&0xFC)|0x80|(n&3);
}

void deInitFVR(void){
    FVRCON=0;
}

void doCalc(void){  //called from ISR!
    //internal: 0)busIlo,1)busIhi,2)busV, 3)vcc, 4)vCC1, 5)vCC2,6)iCC1,7)iCC2,8)vref,9)ANJ,10)ANI
    //see adccChans_t enum
    //internal.busIlo=adjust16bitS((int)adccResInt[CHAN_ILO]+cur.anJoffset,cur.anJfactor);
    //internal.busIhi=adjust16bitS((int)adccResInt[CHAN_IHI]+cur.anIoffset,cur.anIfactor);
    //internal.busV=10UL*adjust16bitU(adccResInt[CHAN_BUSV],cur.anAfactor);   //10mV steps
    internal.busIlo=adjust16bitS((int)getUpdateRes(0,adccResInt[CHAN_ILO])+cur.anJoffset,cur.anJfactor);
    internal.busIhi=adjust16bitS((int)getUpdateRes(1,adccResInt[CHAN_IHI])+cur.anIoffset,cur.anIfactor);
    internal.busV=10UL*adjust16bitU(getUpdateRes(2,adccResInt[CHAN_BUSV]),cur.anAfactor);   //10mV steps    
    if(adccResInt[CHAN_VCC1A]>adccResInt[CHAN_VCC1B]){
        internal.vCC1=adjust16bitU(adccResInt[CHAN_VCC1A],adcref4);
    }else{
        internal.vCC1=adjust16bitU(adccResInt[CHAN_VCC1B],adcref4);
    }
    if(adccResInt[CHAN_VCC2A]>adccResInt[CHAN_VCC2B]){
        internal.vCC2=adjust16bitU(adccResInt[CHAN_VCC2A],adcref4);
    }else{
        internal.vCC2=adjust16bitU(adccResInt[CHAN_VCC2B],adcref4);
    }
    internal.iCC1=(int)adccResInt[CHAN_ICC1];    //no scaling
    internal.iCC2=(int)adccResInt[CHAN_ICC2];    //no scaling
    internal.vcc=adjust16bitU(adccResInt[CHAN_VCC],adcref1*8);    //inbuilt scaling
    internal.vref=adjust16bitU(adccResInt[CHAN_VREF],adcref4);   //inbuilt scaling    
    internal.rawI=adjust16bitU(adccResInt[CHAN_IRAW],adcref4);
    internal.rawJ=adjust16bitU(adccResInt[CHAN_JRAW],adcref4);
    internal.iValid=IVALID_NONE;
    if((internal.rawI<ADC_VALID_UPPER)&&(internal.rawI>ADC_VALID_LOWER)){internal.iValid=IVALID_HI;}
    if((internal.rawJ<ADC_VALID_UPPER)&&(internal.rawJ>ADC_VALID_LOWER)){internal.iValid=IVALID_LO;}
    if(internal.iValid==IVALID_LO){
        internal.busP=(((long)internal.busIlo) * ((long)internal.busV))/100000L;   //10uA x mV => nW => mW    
    }else{  //this likely to be best if neither valid
        internal.busP=(((long)internal.busIhi) * ((long)internal.busV))/1000L;   //mA x mV => uW => mW    
    }    
    if(accumE){internal.busE=internal.busE+internal.busP*adccPer;}
}

void clearAccum(void){  //avoid exposing internal
    internal.busE=0;   
}

unsigned int doOffsetCal(void){ //collect null readings and apply as offset
    int t0,t1;
    long n0=0;
    long n1=0;
    unsigned int c=0;
    while(c<OFFSET_SAMPLE_COUNT){
        if(adccDone){
            adccDone=0;
            t0=(int)adccResInt[CHAN_ILO];      //these are ints stored as uint for consistency with ADRES
            t1=(int)adccResInt[CHAN_IHI];       //save volatiles
            n0=n0+t0;
            n1=n1+t1;
            c=c+1;
        }
    }
    if(c){
        cur.anJoffset=-(int)(n0/c);
        cur.anIoffset=-(int)(n1/c);
    }
    return c;
}

char checkButton(char n){
    if(buttonState(n)){
        while(buttonState(n)){}
        delay(BUTTON_DELAY); //debouncing
        return SHORT_PRESS;
    }
    return 0;    
}

char checkButtonLongShort(char n){  //return long or short
    char t=0;
    char r=SHORT_PRESS;
    if(buttonState(n)){
        while(buttonState(n)){
            t++;
            delay(BUTTON_DELAY);
            if(t>(BUTTON_LONG/BUTTON_DELAY)){r=LONG_PRESS;}
        }
        delay(10); //debouncing
        return r;
    }
    return 0;        
}

char buttonState(char n){
    switch(n){
#ifdef S1
        case 1: return PRESSED(S1); break;
#endif
#ifdef S2
        case 2: return PRESSED(S2); break;
#endif
#ifdef S3
        case 3: return PRESSED(S3); break;
#endif
#ifdef S4
        case 4: return PRESSED(S4); break;
#endif
#ifdef S5
        case 5: return PRESSED(S5); break;
#endif
#ifdef S6
        case 6: return PRESSED(S6); break;
#endif
#ifdef S7
        case 7: return PRESSED(S7); break;
#endif
#ifdef S8
        case 8: return PRESSED(S8); break;
#endif
    }
    return 0;
}

char anyButton(void){
#ifdef S1
    if(PRESSED(S1)){return 1;}
#endif
#ifdef S2
    if(PRESSED(S2)){return 2;}
#endif
#ifdef S3
    if(PRESSED(S3)){return 3;}
#endif
#ifdef S4
    if(PRESSED(S4)){return 4;}
#endif
#ifdef S5
    if(PRESSED(S5)){return 5;}
#endif
#ifdef S6
    if(PRESSED(S6)){return 6;}
#endif
#ifdef S7
    if(PRESSED(S7)){return 7;}
#endif
#ifdef S8
    if(PRESSED(S8)){return 8;}
#endif
    return 0;    
}

char buttonPressAndHold(void){  //return repeats when held
    char t;
    static char last=0;
    char now=anyButton();   //what is pressed?
    if(now==0){  //return immediately unless button pressed
        last=0;
        return 0;
    }else{
        if(last){
            t=HOLD_DELAY/DELAY_INCREMENT;
            while((t--)&&anyButton()){
                delay(DELAY_INCREMENT);
            }
            last=anyButton();
            if(last){
                return now;
            }else{
                return 0;   //stop double-skip on release after hold
            }            
        }else{
            t=PRESS_DELAY/DELAY_INCREMENT;
            while((t--)&&anyButton()){
                delay(DELAY_INCREMENT);
            }
            last=anyButton();
            return now;
        }        
    }    
}

unsigned int getUpdateRes(char i,unsigned int n){   //averaging for display
    unsigned long t;
    t=adccResLPF[i];
    t=t*(DISP_OS-1)+n;
    t=t/DISP_OS;
    adccResLPF[i]=(unsigned int)t;    
    return (unsigned int)t;
}
