#ifndef IO_H
#define	IO_H

#include <xc.h> // include processor files - each processor file is guarded.  
#include "globalPins.h"
#include "portbits.h"

extern char dt;

#define PRESSED(x) (portbits.x==0)
//from DIA
extern long displayEnergy;
extern unsigned int adcref1,dacref1;
extern unsigned int adcref2,dacref2;
extern unsigned int adcref4,dacref4;

//indicate which current value is valid
typedef enum {
    IVALID_NONE=0,
    IVALID_LO,
    IVALID_HI
} iValidity_t;

typedef struct {
    unsigned long busV;
    int busIhi;     //mA
    int busIlo;     //10uA steps
    unsigned int vCC1;    //configuration channel 1
    unsigned int vCC2;    //configuration channel 2
    int iCC1;   //current (mostly interested in direction only)
    int iCC2;   //current (mostly interested in direction only)
    //derived values
    long busP;         //mW
    int64_t busE;       //mJ (ie busPower*t)
    //others:
    unsigned int vcc;
    unsigned int vref;
    unsigned int rawI;
    unsigned int rawJ;
    iValidity_t iValid;
} USBstate_t;

//internal: 0)busIlo,1)busIhi,2)busV, 3)vcc, 4)vCC1, 5)vCC2,6)iCC1,7)iCC2,8)vref
//order of samples in adccResInt etc
typedef enum {
    CHAN_ILO=0,
    CHAN_IHI,
    CHAN_JRAW,
    CHAN_IRAW,
    CHAN_BUSV,
    CHAN_VCC,
    CHAN_VCC1A,
    CHAN_VCC2A,
    CHAN_VCC1B,
    CHAN_VCC2B,
    CHAN_ICC1,
    CHAN_ICC2,
    CHAN_VREF,
    ADC_RES_COUNT   //dummy, not to be used as index
} adccChans_t;

//ADC
//12 bits+ 4 bits=16 bit results
#define ADC_OS (16)
#define ADCC_OS (64)
#define DISP_OS (4)
//systematic offset for differential, calculated at result (ie after oversample/shift)
#define ADC_DIFF_OFFSET (32)
//these in mV
#define ADC_VALID_LOWER (200)
#define ADC_VALID_UPPER (4096-ADC_VALID_LOWER)
#define OFFSET_SAMPLE_COUNT 256

//for FVRCON
#define FVR_OFF (0)
#define FVR_4096 (3)
#define FVR_2048 (2)
#define FVR_1024 (1)
//ADC channels
#define FVRBUF1 0b111110 
#define FVRBUF2 0b111111
#define ADC_VSS 0b111010
#define ADC_DAC1 0b111100
#define ADC_DAC2 0b111101

//for DAC1CON/DAC2CON
//eg DAC1CON=DAC_RA2|DAC_REF_FVR_GND;
#define DAC_RA2 (0b10100000)
#define RAC_RA0 (0b10010000)
#define DAC_OFF (0b00000000)
#define DAC_ON  (0b10000000)
#define DAC_REF_FVR_GND (0b1000)
#define DAC_REF_VDD_GND (0b0000)
#define ADCC_MD_ACCUM (1)
#define ADCC_MD_AVE (2)
#define ADCC_MD_BURST (3)
#define ADCC_MD_LPF (4)
#define ADCC_ALWAYS_INT (7)

//ADC interface
#define ADC_SE (0xFF)
//used to adjust power<>energy calcs
//sample timer runs at 244 counts/s
#define ADC_T0_HZ (244)
//244*3.6 ~=~ 878 (0.05% error)
#define ADC_T0_WATTHOURS (878)

//boost threshold
#define USB_OFF_VOLTAGE 4200

//PWM
#define PWM_PERIOD (50000UL)

//buttons
#define BUTTON_DELAY (10)
#define BUTTON_LONG (500)
#define SHORT_PRESS (1)
#define LONG_PRESS (2)
//buttonPressAndHold timings
#define PRESS_DELAY 200
#define HOLD_DELAY 100
#define DELAY_INCREMENT 10

typedef enum {
    ADCC_OFF=0,
    ADCC_IDLE,
    ADCC_RUNNING,
    ADCC_ERROR
}adccState_t;

extern char adccPchans[ADC_RES_COUNT];
extern char adccNchans[ADC_RES_COUNT];
extern volatile char adccDone;    //when a round has been completed
extern volatile char adccPer;     //how many T0 counts to do samples
//loaded by loadADCC()
extern USBstate_t external;
extern adccState_t adccState;
//flag to enable accumulation
extern volatile char accumE;

void ioInit(void);
void pwmInit(void);
void pwmDeInit(void);
void t0init(void);
unsigned long getT0long(void);  //watch for rollovers and provide extended count
void initADC(void);
void deInitADCC(void);
void deInitADC(void);
unsigned int getADC(char c);
int getADCdiff(char p, char n); //differential
unsigned int getVCC(void);  //in mV
unsigned int readNVM(unsigned int add);
void initFVR_DAC(char n);
void initFVR_ADC(char n);
void deInitFVR(void);

void initADCC(void);   //set up with computation
unsigned int getADCC(char c);  //computation
int getADCCdiff(char p, char n); //differential/computation
void isrADCC(void);
void stopADCC(void);    //stop, should allow seamless restart
void startADCC(void);   //start next sample
void loadADCC(void);    //put into user variables
int getADCCint(unsigned int u);
void doCalc(void);      //ADC results housekeeping, called from ISR!
void clearAccum(void);  //to avoid exposing internal.
unsigned int doOffsetCal(void); //collect null readings and apply as offset
char checkButton(char n);
char checkButtonLongShort(char n);  //return long or short
char buttonState(char n);
char anyButton(void);
char buttonPressAndHold(void);  //return repeats when held
unsigned int getUpdateRes(char i,unsigned int n);

#endif	/* IO_H */

