/*      fix_fft.c - Fixed-point Fast Fourier Transform  */
/*
   fix_fft()       perform FFT or inverse FFT
   window()        applies a Hanning window to the (time) input
   each freq point. Result is an integer array,
   units are dB (values will be negative).
   iscale()        scale an integer value by (numer/denom).
   fix_mpy()       perform fixed-point multiplication.
   Sinewave[8192]  sinewave normalized to 32767 (= 1.0).

   All data are fixed-point short integers, in which
   -32768 to +32768 represent -1.0 to +1.0. Integer arithmetic
   is used for speed, instead of the more natural floating-point.

   For the forward FFT (time -> freq), fixed scaling is
   performed to prevent arithmetic overflow, and to map a 0dB
   sine/cosine wave (i.e. amplitude = 32767) to two -6dB freq
   coefficients. The return value is always 0.

   For the inverse FFT (freq -> time), fixed scaling cannot be
   done, as two 0dB coefficients would sum to a peak amplitude of
   64K, overflowing the 32k range of the fixed-point integers.
   Thus, the fix_fft() routine performs variable scaling, and
   returns a value which is the number of bits LEFT by which
   the output must be shifted to get the actual amplitude
   (i.e. if fix_fft() returns 3, each value of fr[] and fi[]
   must be multiplied by 8 (2**3) for proper scaling.
   Clearly, this cannot be done within the fixed-point short
   integers. In practice, if the result is to be used as a
   filter, the scale_shift can usually be ignored, as the
   result will be approximately correctly normalized as is.

   TURBO C, any memory model; uses inline assembly for speed
   and for carefully-scaled arithmetic.

   Written by:  Tom Roberts  11/8/89
   Made portable:  Malcolm Slaney 12/15/94 malcolm@interval.com

   Timing on a Macintosh PowerBook 180.... (using Symantec C6.0)
   fix_fft (1024 points)             8 ticks
   fft (1024 points - Using SANE)  112 Ticks
   fft (1024 points - Using FPU)    11

 */

/* FIX_MPY() - fixed-point multiplication macro.
   This macro is a statement, not an expression (uses asm).
   BEWARE: make sure _DX is not clobbered by evaluating (A) or DEST.
   args are all of type fixed.
   Scaling ensures that 32767*32767 = 32767. */

#include "fix_fft32.h"

#define FIX_MPY(DEST,A,B)       DEST = ((long long)(A) * (long long)(B))>>31

#define N_WAVE          4096    /* dimension of Sinewave[] */
#define LOG2_N_WAVE     12      /* log2(N_WAVE) */

extern fixed Sinewave[N_WAVE];     /* placed at end of this file for clarity */
extern fixed flattop_window[N_WAVE/2+1];     /* placed at end of this file for clarity */
static fixed fix_mpy (fixed a, fixed b);

/*
   fix_fft() - perform fast Fourier transform.

   if n>0 FFT is done, if n<0 inverse FFT is done
   fr[n],fi[n] are real,imaginary arrays, INPUT AND RESULT.
   size of data = 2**m
   set inverse to 0=dft, 1=idft
 */
int
fix_fft (fixed fr[], fixed fi[], int m, int inverse)
{
  int mr, nn, i, j, l, k, istep, n, scale, shift;
  fixed qr, qi, tr, ti, wr, wi;

  n = 1 << m;

  if (n > N_WAVE)
    return -1;

  mr = 0;
  nn = n - 1;
  scale = 0;

  /* decimation in time - re-order data */
  for (m = 1; m <= nn; ++m) {
    l = n;
    do {
      l >>= 1;
    }
    while (mr + l > nn);
    mr = (mr & (l - 1)) + l;

    if (mr <= m)
      continue;
    tr = fr[m];
    fr[m] = fr[mr];
    fr[mr] = tr;
    ti = fi[m];
    fi[m] = fi[mr];
    fi[mr] = ti;
  }

  l = 1;
  k = LOG2_N_WAVE - 1;
  while (l < n) {
    if (inverse) {
      /* variable scaling, depending upon data */
      shift = 0;
      for (i = 0; i < n; ++i) {
        j = fr[i];
        if (j < 0)
          j = -j;
        m = fi[i];
        if (m < 0)
          m = -m;
        if (j > 1073741823 || m > 1073741823) {
          shift = 1;
          break;
        }
      }
      if (shift)
        ++scale;
    } else {
      /* fixed scaling, for proper normalization -
         there will be log2(n) passes, so this
         results in an overall factor of 1/n,
         distributed to maximize arithmetic accuracy. */
      shift = 1;
    }
    /* it may not be obvious, but the shift will be performed
       on each data point exactly once, during this pass. */
    istep = l << 1;
    for (m = 0; m < l; ++m) {
      j = m << k;
      /* 0 <= j < N_WAVE/2 */
      wr = Sinewave[j + N_WAVE / 4];
      wi = -Sinewave[j];
      if (inverse)
        wi = -wi;
      if (shift) {
        wr >>= 1;
        wi >>= 1;
      }
      for (i = m; i < n; i += istep) {
        j = i + l;
        tr = fix_mpy (wr, fr[j]) -
            fix_mpy (wi, fi[j]);
        ti = fix_mpy (wr, fi[j]) +
            fix_mpy (wi, fr[j]);
        qr = fr[i];
        qi = fi[i];
        if (shift) {
          qr >>= 1;
          qi >>= 1;
        }
        fr[j] = qr - tr;
        fi[j] = qi - ti;
        fr[i] = qr + tr;
        fi[i] = qi + ti;
      }
    }
    --k;
    l = istep;
  }

  return scale;
}

/*      window() - apply a Hanning window       */
void
window_hann (fixed fr[], int n)
{
  int i, j, k;

  j = N_WAVE / n;
  n >>= 1;
  for (i = 0, k = N_WAVE / 4; i < n; ++i, k += j)
    FIX_MPY (fr[i], fr[i], 1073741824 - (Sinewave[k] >> 1));
  n <<= 1;
  for (k -= j; i < n; ++i, k -= j)
    FIX_MPY (fr[i], fr[i], 1073741824 - (Sinewave[k] >> 1));
}

void
window2_hann (fixed fr[], fixed im[], int n)
{
  int i, j, k;

  j = N_WAVE / n;
  for (i = 0, k = N_WAVE / 4; i < n; ++i) {
    if( i&1 ) {
      FIX_MPY (im[i/2], im[i/2], 1073741824 - (Sinewave[k] >> 2) - (Sinewave[k+1] >> 2));
      k += j;
    } else {
      FIX_MPY (fr[i/2], fr[i/2], 1073741824 - (Sinewave[k] >> 1));
    }
  }
  n <<= 1;
  for (k -= j; i < n; ++i) {
    if( i&1 ) {
      FIX_MPY (im[i/2], im[i/2], 1073741824 - (Sinewave[k-1] >> 2) - (Sinewave[k] >> 2));
      k -= j;
    } else {
      FIX_MPY (fr[i/2], fr[i/2], 1073741824 - (Sinewave[k] >> 1));
    }
  }
}

void
window2_flattop (fixed fr[], fixed im[], int n)
{
  int i, j, k;

//  const float a1 = 1.93, a2 = 1.29, a3 = 0.388, a4 = 0.028;

  j = N_WAVE / n;
  for (i = 0, k = 0; i < n; ++i) {
//    float coeff = (1 - a1 * cos(i * 2 * M_PI / (N_WAVE*2)) + a2 * cos(i * 4 * M_PI / (N_WAVE*2)) - a3 * cos(i * 6 * M_PI / (N_WAVE*2)) + a4 * cos(i * 8 * M_PI / (N_WAVE*2))) / 4.6;
    if( i&1 ) {
      FIX_MPY( im[i/2], im[i/2], ((flattop_window[k]>>1) + (flattop_window[k+1]>>1)) );
      k += j;
    } else {
      FIX_MPY( fr[i/2], fr[i/2], flattop_window[k] );
    }
  }
  n <<= 1;
  for (k -= j; i < n; ++i) {
//    float coeff = (1 - a1 * cos(i * 2 * M_PI / (N_WAVE*2)) + a2 * cos(i * 4 * M_PI / (N_WAVE*2)) - a3 * cos(i * 6 * M_PI / (N_WAVE*2)) + a4 * cos(i * 8 * M_PI / (N_WAVE*2))) / 4.6;
    if( i&1 ) {
      FIX_MPY( im[i/2], im[i/2], flattop_window[k] );
      k -= j;
    } else {
      FIX_MPY( fr[i/2], fr[i/2], ((flattop_window[k]>>1) + (flattop_window[k+1]>>1)) );
    }
  }
}

/*
   fix_mpy() - fixed-point multiplication
 */
fixed
fix_mpy (fixed a, fixed b)
{
  FIX_MPY (a, a, b);
  return a;
}

/*
   iscale() - scale an integer value by (numer/denom)
 */
int
iscale (int value, int numer, int denom)
{
  return (long) value *(long) numer / (long) denom;
}

/*
   fix_dot() - dot product of two fixed arrays
 */
fixed
fix_dot (fixed * hpa, fixed * pb, int n)
{
  fixed *pa = hpa;              /* FIXME */
  long sum;
  register fixed a, b;

/*      seg = FP_SEG(hpa);
   off = FP_OFF(hpa);
   seg += off>>4;
   off &= 0x000F;
   pa = MK_FP(seg,off);
 */
  sum = 0L;
  while (n--) {
    a = *pa++;
    b = *pb++;
    FIX_MPY (a, a, b);
    sum += a;
  }

  if (sum > 0x7FFFFFFF)
    sum = 0x7FFFFFFF;
  else if (sum < -0x7FFFFFFF)
    sum = -0x7FFFFFFF;

  return (fixed) sum;

}

void fft_real_result_convolve(fixed* re, fixed* im, int n) {
  int i;
  int scale = N_WAVE / 2 / n;
  for( i = 0; i < n; ++i ) {
    fixed Ar, Ai, Br, Bi;
    if( (i*scale) & 1 ) {
      Ar = 1073741823 - ( Sinewave[           i * scale/2] >> 2) - ( Sinewave[           i * scale/2 + 1] >> 2);
      Ai =              (-Sinewave[N_WAVE/4 + i * scale/2] >> 2) + (-Sinewave[N_WAVE/4 + i * scale/2 + 1] >> 2);
      Br = 1073741823 + ( Sinewave[           i * scale/2] >> 2) + ( Sinewave[           i * scale/2 + 1] >> 2);
      Bi =              ( Sinewave[N_WAVE/4 + i * scale/2] >> 2) + ( Sinewave[N_WAVE/4 + i * scale/2 + 1] >> 2);
    } else {
      Ar = 1073741823 - ( Sinewave[           i * scale/2] >> 1);
      Ai =              (-Sinewave[N_WAVE/4 + i * scale/2] >> 1);
      Br = 1073741823 + ( Sinewave[           i * scale/2] >> 1);
      Bi =              ( Sinewave[N_WAVE/4 + i * scale/2] >> 1);
    }
    fixed re1 = fix_mpy(re[i], Ar) - fix_mpy(im[i], Ai) + fix_mpy(re[n-i], Br) + fix_mpy(im[n-i], Bi);
    fixed im1 = fix_mpy(im[i], Ar) + fix_mpy(re[i], Ai) + fix_mpy(re[n-i], Bi) - fix_mpy(im[n-i], Br);
    re[i] = re1;
    im[i] = im1;
  }
}

#if N_WAVE != 4096
#error N_WAVE != 4096
#endif
    fixed Sinewave[4096] = { 0, 3294197, 6588386, 9882561, 13176711, 16470831, 19764912, 23058947, 26352927, 29646846, 32940694, 36234465, 39528151, 42821744, 46115236, 49408619, 
			52701886, 55995030, 59288041, 62580913, 65873638, 69166207, 72458614, 75750851, 79042909, 82334781, 85626459, 88917936, 92209204, 95500255, 98791081, 102081674, 
			105372028, 108662133, 111951983, 115241569, 118530884, 121819920, 125108670, 128397125, 131685278, 134973121, 138260647, 141547847, 144834714, 148121240, 151407418, 154693239, 
			157978697, 161263783, 164548489, 167832808, 171116732, 174400253, 177683365, 180966058, 184248325, 187530158, 190811551, 194092494, 197372981, 200653003, 203932553, 207211623, 
			210490205, 213768293, 217045877, 220322950, 223599506, 226875534, 230151030, 233425983, 236700387, 239974234, 243247517, 246520227, 249792357, 253063900, 256334846, 259605190, 
			262874923, 266144037, 269412525, 272680379, 275947591, 279214154, 282480060, 285745301, 289009870, 292273759, 295536960, 298799466, 302061268, 305322360, 308582733, 311842381, 
			315101294, 318359466, 321616889, 324873555, 328129456, 331384586, 334638935, 337892497, 341145265, 344397229, 347648383, 350898719, 354148229, 357396906, 360644742, 363891729, 
			367137860, 370383127, 373627522, 376871039, 380113668, 383355403, 386596236, 389836160, 393075166, 396313247, 399550395, 402786603, 406021864, 409256169, 412489511, 415721883, 
			418953276, 422183683, 425413097, 428641510, 431868914, 435095302, 438320666, 441544999, 444768293, 447990540, 451211733, 454431865, 457650927, 460868912, 464085812, 467301621, 
			470516330, 473729932, 476942419, 480153783, 483364018, 486573116, 489781068, 492987869, 496193509, 499397981, 502601278, 505803393, 509004317, 512204044, 515402565, 518599874, 
			521795962, 524990823, 528184448, 531376830, 534567962, 537757836, 540946445, 544133780, 547319836, 550504603, 553688075, 556870244, 560051103, 563230644, 566408859, 569585742, 
			572761285, 575935479, 579108319, 582279796, 585449902, 588618631, 591785975, 594951926, 598116478, 601279622, 604441351, 607601658, 610760535, 613917975, 617073970, 620228513, 
			623381597, 626533214, 629683356, 632832017, 635979189, 639124864, 642269036, 645411696, 648552837, 651692452, 654830534, 657967075, 661102067, 664235504, 667367378, 670497682, 
			673626407, 676753548, 679879096, 683003045, 686125386, 689246113, 692365217, 695482693, 698598532, 701712727, 704825271, 707936157, 711045377, 714152923, 717258789, 720362967, 
			723465451, 726566231, 729665302, 732762657, 735858286, 738952185, 742044344, 745134758, 748223418, 751310317, 754395449, 757478805, 760560379, 763640163, 766718150, 769794333, 
			772868705, 775941258, 779011985, 782080880, 785147933, 788213140, 791276491, 794337981, 797397601, 800455345, 803511206, 806565176, 809617248, 812667415, 815715669, 818762004, 
			821806412, 824848887, 827889421, 830928006, 833964637, 836999305, 840032003, 843062725, 846091463, 849118210, 852142958, 855165702, 858186434, 861205146, 864221831, 867236483, 
			870249094, 873259658, 876268167, 879274613, 882278991, 885281292, 888281511, 891279639, 894275670, 897269597, 900261412, 903251109, 906238680, 909224119, 912207418, 915188571, 
			918167571, 921144410, 924119081, 927091578, 930061893, 933030020, 935995951, 938959680, 941921199, 944880502, 947837581, 950792430, 953745042, 956695410, 959643526, 962589384, 
			965532977, 968474299, 971413341, 974350097, 977284561, 980216725, 983146582, 986074126, 988999350, 991922247, 994842809, 997761030, 1000676904, 1003590423, 1006501580, 1009410369, 
			1012316783, 1015220815, 1018122458, 1021021705, 1023918549, 1026812984, 1029705003, 1032594599, 1035481765, 1038366494, 1041248781, 1044128616, 1047005996, 1049880911, 1052753356, 1055623323, 
			1058490807, 1061355800, 1064218295, 1067078287, 1069935767, 1072790729, 1075643168, 1078493075, 1081340444, 1084185269, 1087027543, 1089867258, 1092704410, 1095538990, 1098370992, 1101200409, 
			1104027236, 1106851464, 1109673088, 1112492100, 1115308495, 1118122266, 1120933405, 1123741907, 1126547764, 1129350971, 1132151520, 1134949405, 1137744620, 1140537157, 1143327010, 1146114173, 
			1148898639, 1151680402, 1154459455, 1157235791, 1160009404, 1162780287, 1165548434, 1168313839, 1171076494, 1173836394, 1176593532, 1179347901, 1182099495, 1184848307, 1187594331, 1190337561, 
			1193077990, 1195815611, 1198550418, 1201282406, 1204011566, 1206737893, 1209461381, 1212182022, 1214899812, 1217614742, 1220326808, 1223036001, 1225742317, 1228445749, 1231146290, 1233843934, 
			1236538674, 1239230505, 1241919420, 1244605413, 1247288476, 1249968605, 1252645793, 1255320033, 1257991319, 1260659645, 1263325004, 1265987391, 1268646799, 1271303221, 1273956652, 1276607085, 
			1279254514, 1281898933, 1284540336, 1287178716, 1289814067, 1292446383, 1295075658, 1297701885, 1300325059, 1302945173, 1305562221, 1308176197, 1310787094, 1313394908, 1315999630, 1318601256, 
			1321199779, 1323795194, 1326387493, 1328976671, 1331562722, 1334145640, 1336725418, 1339302051, 1341875532, 1344445856, 1347013016, 1349577006, 1352137821, 1354695454, 1357249900, 1359801151, 
			1362349203, 1364894049, 1367435684, 1369974100, 1372509293, 1375041257, 1377569984, 1380095471, 1382617709, 1385136695, 1387652420, 1390164881, 1392674071, 1395179983, 1397682612, 1400181953, 
			1402677998, 1405170743, 1407660182, 1410146308, 1412629116, 1415108600, 1417584754, 1420057573, 1422527049, 1424993179, 1427455955, 1429915373, 1432371425, 1434824107, 1437273413, 1439719337, 
			1442161873, 1444601016, 1447036759, 1449469097, 1451898024, 1454323535, 1456745624, 1459164285, 1461579512, 1463991301, 1466399644, 1468804536, 1471205973, 1473603947, 1475998454, 1478389488, 
			1480777043, 1483161114, 1485541694, 1487918779, 1490292363, 1492662440, 1495029005, 1497392051, 1499751575, 1502107569, 1504460028, 1506808947, 1509154321, 1511496144, 1513834409, 1516169113, 
			1518500249, 1520827811, 1523151796, 1525472195, 1527789006, 1530102221, 1532411836, 1534717845, 1537020242, 1539319023, 1541614182, 1543905713, 1546193611, 1548477871, 1550758487, 1553035454, 
			1555308767, 1557578419, 1559844407, 1562106724, 1564365365, 1566620326, 1568871600, 1571119182, 1573363067, 1575603250, 1577839725, 1580072488, 1582301532, 1584526853, 1586748446, 1588966305, 
			1591180424, 1593390800, 1595597426, 1597800298, 1599999410, 1602194757, 1604386334, 1606574135, 1608758156, 1610938392, 1613114836, 1615287485, 1617456334, 1619621376, 1621782607, 1623940021, 
			1626093615, 1628243382, 1630389317, 1632531417, 1634669674, 1636804086, 1638934645, 1641061348, 1643184190, 1645303165, 1647418268, 1649529495, 1651636840, 1653740299, 1655839866, 1657935537, 
			1660027307, 1662115171, 1664199123, 1666279160, 1668355275, 1670427465, 1672495724, 1674560047, 1676620430, 1678676868, 1680729356, 1682777889, 1684822462, 1686863071, 1688899710, 1690932375, 
			1692961061, 1694985763, 1697006477, 1699023198, 1701035921, 1703044641, 1705049354, 1707050054, 1709046738, 1711039400, 1713028036, 1715012641, 1716993210, 1718969739, 1720942223, 1722910658, 
			1724875039, 1726835360, 1728791618, 1730743809, 1732691926, 1734635967, 1736575926, 1738511798, 1740443579, 1742371265, 1744294851, 1746214333, 1748129706, 1750040965, 1751948106, 1753851124, 
			1755750016, 1757644776, 1759535400, 1761421884, 1763304223, 1765182413, 1767056449, 1768926327, 1770792043, 1772653592, 1774510969, 1776364171, 1778213193, 1780058031, 1781898680, 1783735136, 
			1785567395, 1787395452, 1789219303, 1791038944, 1792854371, 1794665579, 1796472564, 1798275321, 1800073847, 1801868137, 1803658188, 1805443994, 1807225552, 1809002857, 1810775905, 1812544693, 
			1814309215, 1816069468, 1817825448, 1819577150, 1821324571, 1823067705, 1824806550, 1826541101, 1828271354, 1829997305, 1831718950, 1833436285, 1835149305, 1836858007, 1838562387, 1840262440, 
			1841958163, 1843649552, 1845336602, 1847019310, 1848697672, 1850371684, 1852041342, 1853706642, 1855367579, 1857024151, 1858676353, 1860324182, 1861967633, 1863606703, 1865241387, 1866871682, 
			1868497584, 1870119090, 1871736195, 1873348896, 1874957188, 1876561068, 1878160533, 1879755578, 1881346200, 1882932395, 1884514160, 1886091490, 1887664381, 1889232831, 1890796835, 1892356390, 
			1893911493, 1895462138, 1897008324, 1898550045, 1900087299, 1901620082, 1903148391, 1904672221, 1906191569, 1907706431, 1909216805, 1910722686, 1912224071, 1913720956, 1915213339, 1916701214, 
			1918184579, 1919663431, 1921137766, 1922607579, 1924072869, 1925533631, 1926989863, 1928441559, 1929888719, 1931331336, 1932769410, 1934202935, 1935631909, 1937056328, 1938476189, 1939891489, 
			1941302224, 1942708390, 1944109986, 1945507007, 1946899449, 1948287311, 1949670588, 1951049277, 1952423376, 1953792880, 1955157786, 1956518092, 1957873794, 1959224889, 1960571374, 1961913245, 
			1963250500, 1964583135, 1965911147, 1967234533, 1968553290, 1969867415, 1971176905, 1972481756, 1973781966, 1975077531, 1976368449, 1977654716, 1978936329, 1980213286, 1981485584, 1982753218, 
			1984016187, 1985274488, 1986528117, 1987777071, 1989021348, 1990260945, 1991495858, 1992726085, 1993951623, 1995172470, 1996388621, 1997600075, 1998806828, 2000008877, 2001206221, 2002398855, 
			2003586778, 2004769986, 2005948476, 2007122247, 2008291294, 2009455616, 2010615209, 2011770071, 2012920199, 2014065591, 2015206243, 2016342154, 2017473319, 2018599738, 2019721406, 2020838322, 
			2021950482, 2023057885, 2024160527, 2025258407, 2026351520, 2027439866, 2028523440, 2029602242, 2030676267, 2031745514, 2032809981, 2033869663, 2034924560, 2035974669, 2037019987, 2038060511, 
			2039096240, 2040127170, 2041153300, 2042174627, 2043191148, 2044202862, 2045209765, 2046211856, 2047209132, 2048201590, 2049189229, 2050172046, 2051150039, 2052123205, 2053091543, 2054055049, 
			2055013722, 2055967559, 2056916558, 2057860717, 2058800034, 2059734506, 2060664132, 2061588909, 2062508834, 2063423906, 2064334123, 2065239482, 2066139982, 2067035619, 2067926393, 2068812301, 
			2069693340, 2070569510, 2071440807, 2072307229, 2073168776, 2074025444, 2074877232, 2075724137, 2076566158, 2077403293, 2078235539, 2079062895, 2079885359, 2080702928, 2081515602, 2082323377, 
			2083126253, 2083924227, 2084717297, 2085505461, 2086288718, 2087067066, 2087840503, 2088609027, 2089372636, 2090131329, 2090885104, 2091633958, 2092377891, 2093116900, 2093850983, 2094580140, 
			2095304368, 2096023666, 2096738031, 2097447462, 2098151959, 2098851517, 2099546137, 2100235817, 2100920555, 2101600349, 2102275197, 2102945099, 2103610052, 2104270056, 2104925108, 2105575206, 
			2106220350, 2106860538, 2107495769, 2108126040, 2108751350, 2109371699, 2109987084, 2110597504, 2111202957, 2111803443, 2112398959, 2112989505, 2113575078, 2114155678, 2114731304, 2115301953, 
			2115867624, 2116428317, 2116984030, 2117534761, 2118080509, 2118621274, 2119157053, 2119687845, 2120213650, 2120734465, 2121250290, 2121761124, 2122266965, 2122767812, 2123263664, 2123754520, 
			2124240379, 2124721239, 2125197099, 2125667958, 2126133816, 2126594670, 2127050521, 2127501366, 2127947205, 2128388037, 2128823860, 2129254674, 2129680478, 2130101271, 2130517051, 2130927817, 
			2131333570, 2131734307, 2132130028, 2132520732, 2132906418, 2133287086, 2133662733, 2134033359, 2134398964, 2134759547, 2135115106, 2135465641, 2135811151, 2136151635, 2136487093, 2136817524, 
			2137142926, 2137463299, 2137778643, 2138088956, 2138394238, 2138694489, 2138989706, 2139279891, 2139565041, 2139845157, 2140120238, 2140390283, 2140655291, 2140915262, 2141170196, 2141420091, 
			2141664947, 2141904763, 2142139539, 2142369275, 2142593969, 2142813622, 2143028233, 2143237800, 2143442325, 2143641805, 2143836242, 2144025634, 2144209981, 2144389282, 2144563537, 2144732746, 
			2144896908, 2145056023, 2145210091, 2145359111, 2145503082, 2145642005, 2145775879, 2145904703, 2146028478, 2146147204, 2146260879, 2146369504, 2146473078, 2146571602, 2146665074, 2146753495, 
			2146836865, 2146915182, 2146988448, 2147056662, 2147119824, 2147177933, 2147230990, 2147278994, 2147321945, 2147359843, 2147392689, 2147420481, 2147443221, 2147460907, 2147473540, 2147481120, 
			2147483647, 2147481120, 2147473540, 2147460907, 2147443221, 2147420481, 2147392689, 2147359843, 2147321945, 2147278994, 2147230990, 2147177933, 2147119824, 2147056662, 2146988448, 2146915182, 
			2146836865, 2146753495, 2146665074, 2146571602, 2146473078, 2146369504, 2146260879, 2146147204, 2146028478, 2145904703, 2145775879, 2145642005, 2145503082, 2145359111, 2145210091, 2145056023, 
			2144896908, 2144732746, 2144563537, 2144389282, 2144209981, 2144025634, 2143836242, 2143641805, 2143442325, 2143237800, 2143028233, 2142813622, 2142593969, 2142369275, 2142139539, 2141904763, 
			2141664947, 2141420091, 2141170196, 2140915262, 2140655291, 2140390283, 2140120238, 2139845157, 2139565041, 2139279891, 2138989706, 2138694489, 2138394238, 2138088956, 2137778643, 2137463299, 
			2137142926, 2136817524, 2136487093, 2136151635, 2135811151, 2135465641, 2135115106, 2134759547, 2134398964, 2134033359, 2133662733, 2133287086, 2132906418, 2132520732, 2132130028, 2131734307, 
			2131333570, 2130927817, 2130517051, 2130101271, 2129680478, 2129254674, 2128823860, 2128388037, 2127947205, 2127501366, 2127050521, 2126594670, 2126133816, 2125667958, 2125197099, 2124721239, 
			2124240379, 2123754520, 2123263664, 2122767812, 2122266965, 2121761124, 2121250290, 2120734465, 2120213650, 2119687845, 2119157053, 2118621274, 2118080509, 2117534761, 2116984030, 2116428317, 
			2115867624, 2115301953, 2114731304, 2114155678, 2113575078, 2112989505, 2112398959, 2111803443, 2111202957, 2110597504, 2109987084, 2109371699, 2108751350, 2108126040, 2107495769, 2106860538, 
			2106220350, 2105575206, 2104925108, 2104270056, 2103610052, 2102945099, 2102275197, 2101600349, 2100920555, 2100235817, 2099546137, 2098851517, 2098151959, 2097447462, 2096738031, 2096023666, 
			2095304368, 2094580140, 2093850983, 2093116900, 2092377891, 2091633958, 2090885104, 2090131329, 2089372636, 2088609027, 2087840503, 2087067066, 2086288718, 2085505461, 2084717297, 2083924227, 
			2083126253, 2082323377, 2081515602, 2080702928, 2079885359, 2079062895, 2078235539, 2077403293, 2076566158, 2075724137, 2074877232, 2074025444, 2073168776, 2072307229, 2071440807, 2070569510, 
			2069693340, 2068812301, 2067926393, 2067035619, 2066139982, 2065239482, 2064334123, 2063423906, 2062508834, 2061588909, 2060664132, 2059734506, 2058800034, 2057860717, 2056916558, 2055967559, 
			2055013722, 2054055049, 2053091543, 2052123205, 2051150039, 2050172046, 2049189229, 2048201590, 2047209132, 2046211856, 2045209765, 2044202862, 2043191148, 2042174627, 2041153300, 2040127170, 
			2039096240, 2038060511, 2037019987, 2035974669, 2034924560, 2033869663, 2032809981, 2031745514, 2030676267, 2029602242, 2028523440, 2027439866, 2026351520, 2025258407, 2024160527, 2023057885, 
			2021950482, 2020838322, 2019721406, 2018599738, 2017473319, 2016342154, 2015206243, 2014065591, 2012920199, 2011770071, 2010615209, 2009455616, 2008291294, 2007122247, 2005948476, 2004769986, 
			2003586778, 2002398855, 2001206221, 2000008877, 1998806828, 1997600075, 1996388621, 1995172470, 1993951623, 1992726085, 1991495858, 1990260945, 1989021348, 1987777071, 1986528117, 1985274488, 
			1984016187, 1982753218, 1981485584, 1980213286, 1978936329, 1977654716, 1976368449, 1975077531, 1973781966, 1972481756, 1971176905, 1969867415, 1968553290, 1967234533, 1965911147, 1964583135, 
			1963250500, 1961913245, 1960571374, 1959224889, 1957873794, 1956518092, 1955157786, 1953792880, 1952423376, 1951049277, 1949670588, 1948287311, 1946899449, 1945507007, 1944109986, 1942708390, 
			1941302224, 1939891489, 1938476189, 1937056328, 1935631909, 1934202935, 1932769410, 1931331336, 1929888719, 1928441559, 1926989863, 1925533631, 1924072869, 1922607579, 1921137766, 1919663431, 
			1918184579, 1916701214, 1915213339, 1913720956, 1912224071, 1910722686, 1909216805, 1907706431, 1906191569, 1904672221, 1903148391, 1901620082, 1900087299, 1898550045, 1897008324, 1895462138, 
			1893911493, 1892356390, 1890796835, 1889232831, 1887664381, 1886091490, 1884514160, 1882932395, 1881346200, 1879755578, 1878160533, 1876561068, 1874957188, 1873348896, 1871736195, 1870119090, 
			1868497584, 1866871682, 1865241387, 1863606703, 1861967633, 1860324182, 1858676353, 1857024151, 1855367579, 1853706642, 1852041342, 1850371684, 1848697672, 1847019310, 1845336602, 1843649552, 
			1841958163, 1840262440, 1838562387, 1836858007, 1835149305, 1833436285, 1831718950, 1829997305, 1828271354, 1826541101, 1824806550, 1823067705, 1821324571, 1819577150, 1817825448, 1816069468, 
			1814309215, 1812544693, 1810775905, 1809002857, 1807225552, 1805443994, 1803658188, 1801868137, 1800073847, 1798275321, 1796472564, 1794665579, 1792854371, 1791038944, 1789219303, 1787395452, 
			1785567395, 1783735136, 1781898680, 1780058031, 1778213193, 1776364171, 1774510969, 1772653592, 1770792043, 1768926327, 1767056449, 1765182413, 1763304223, 1761421884, 1759535400, 1757644776, 
			1755750016, 1753851124, 1751948106, 1750040965, 1748129706, 1746214333, 1744294851, 1742371265, 1740443579, 1738511798, 1736575926, 1734635967, 1732691926, 1730743809, 1728791618, 1726835360, 
			1724875039, 1722910658, 1720942223, 1718969739, 1716993210, 1715012641, 1713028036, 1711039400, 1709046738, 1707050054, 1705049354, 1703044641, 1701035921, 1699023198, 1697006477, 1694985763, 
			1692961061, 1690932375, 1688899710, 1686863071, 1684822462, 1682777889, 1680729356, 1678676868, 1676620430, 1674560047, 1672495724, 1670427465, 1668355275, 1666279160, 1664199123, 1662115171, 
			1660027307, 1657935537, 1655839866, 1653740299, 1651636840, 1649529495, 1647418268, 1645303165, 1643184190, 1641061348, 1638934645, 1636804086, 1634669674, 1632531417, 1630389317, 1628243382, 
			1626093615, 1623940021, 1621782607, 1619621376, 1617456334, 1615287485, 1613114836, 1610938392, 1608758156, 1606574135, 1604386334, 1602194757, 1599999410, 1597800298, 1595597426, 1593390800, 
			1591180424, 1588966305, 1586748446, 1584526853, 1582301532, 1580072488, 1577839725, 1575603250, 1573363067, 1571119182, 1568871600, 1566620326, 1564365365, 1562106724, 1559844407, 1557578419, 
			1555308767, 1553035454, 1550758487, 1548477871, 1546193611, 1543905713, 1541614182, 1539319023, 1537020242, 1534717845, 1532411836, 1530102221, 1527789006, 1525472195, 1523151796, 1520827811, 
			1518500249, 1516169113, 1513834409, 1511496144, 1509154321, 1506808947, 1504460028, 1502107569, 1499751575, 1497392051, 1495029005, 1492662440, 1490292363, 1487918779, 1485541694, 1483161114, 
			1480777043, 1478389488, 1475998454, 1473603947, 1471205973, 1468804536, 1466399644, 1463991301, 1461579512, 1459164285, 1456745624, 1454323535, 1451898024, 1449469097, 1447036759, 1444601016, 
			1442161873, 1439719337, 1437273413, 1434824107, 1432371425, 1429915373, 1427455955, 1424993179, 1422527049, 1420057573, 1417584754, 1415108600, 1412629116, 1410146308, 1407660182, 1405170743, 
			1402677998, 1400181953, 1397682612, 1395179983, 1392674071, 1390164881, 1387652420, 1385136695, 1382617709, 1380095471, 1377569984, 1375041257, 1372509293, 1369974100, 1367435684, 1364894049, 
			1362349203, 1359801151, 1357249900, 1354695454, 1352137821, 1349577006, 1347013016, 1344445856, 1341875532, 1339302051, 1336725418, 1334145640, 1331562722, 1328976671, 1326387493, 1323795194, 
			1321199779, 1318601256, 1315999630, 1313394908, 1310787094, 1308176197, 1305562221, 1302945173, 1300325059, 1297701885, 1295075658, 1292446383, 1289814067, 1287178716, 1284540336, 1281898933, 
			1279254514, 1276607085, 1273956652, 1271303221, 1268646799, 1265987391, 1263325004, 1260659645, 1257991319, 1255320033, 1252645793, 1249968605, 1247288476, 1244605413, 1241919420, 1239230505, 
			1236538674, 1233843934, 1231146290, 1228445749, 1225742317, 1223036001, 1220326808, 1217614742, 1214899812, 1212182022, 1209461381, 1206737893, 1204011566, 1201282406, 1198550418, 1195815611, 
			1193077990, 1190337561, 1187594331, 1184848307, 1182099495, 1179347901, 1176593532, 1173836394, 1171076494, 1168313839, 1165548434, 1162780287, 1160009404, 1157235791, 1154459455, 1151680402, 
			1148898639, 1146114173, 1143327010, 1140537157, 1137744620, 1134949405, 1132151520, 1129350971, 1126547764, 1123741907, 1120933405, 1118122266, 1115308495, 1112492100, 1109673088, 1106851464, 
			1104027236, 1101200409, 1098370992, 1095538990, 1092704410, 1089867258, 1087027543, 1084185269, 1081340444, 1078493075, 1075643168, 1072790729, 1069935767, 1067078287, 1064218295, 1061355800, 
			1058490807, 1055623323, 1052753356, 1049880911, 1047005996, 1044128616, 1041248781, 1038366494, 1035481765, 1032594599, 1029705003, 1026812984, 1023918549, 1021021705, 1018122458, 1015220815, 
			1012316783, 1009410369, 1006501580, 1003590423, 1000676904, 997761030, 994842809, 991922247, 988999350, 986074126, 983146582, 980216725, 977284561, 974350097, 971413341, 968474299, 
			965532977, 962589384, 959643526, 956695410, 953745042, 950792430, 947837581, 944880502, 941921199, 938959680, 935995951, 933030020, 930061893, 927091578, 924119081, 921144410, 
			918167571, 915188571, 912207418, 909224119, 906238680, 903251109, 900261412, 897269597, 894275670, 891279639, 888281511, 885281292, 882278991, 879274613, 876268167, 873259658, 
			870249094, 867236483, 864221831, 861205146, 858186434, 855165702, 852142958, 849118210, 846091463, 843062725, 840032003, 836999305, 833964637, 830928006, 827889421, 824848887, 
			821806412, 818762004, 815715669, 812667415, 809617248, 806565176, 803511206, 800455345, 797397601, 794337981, 791276491, 788213140, 785147933, 782080880, 779011985, 775941258, 
			772868705, 769794333, 766718150, 763640163, 760560379, 757478805, 754395449, 751310317, 748223418, 745134758, 742044344, 738952185, 735858286, 732762657, 729665302, 726566231, 
			723465451, 720362967, 717258789, 714152923, 711045377, 707936157, 704825271, 701712727, 698598532, 695482693, 692365217, 689246113, 686125386, 683003045, 679879096, 676753548, 
			673626407, 670497682, 667367378, 664235504, 661102067, 657967075, 654830534, 651692452, 648552837, 645411696, 642269036, 639124864, 635979189, 632832017, 629683356, 626533214, 
			623381597, 620228513, 617073970, 613917975, 610760535, 607601658, 604441351, 601279622, 598116478, 594951926, 591785975, 588618631, 585449902, 582279796, 579108319, 575935479, 
			572761285, 569585742, 566408859, 563230644, 560051103, 556870244, 553688075, 550504603, 547319836, 544133780, 540946445, 537757836, 534567962, 531376830, 528184448, 524990823, 
			521795962, 518599874, 515402565, 512204044, 509004317, 505803393, 502601278, 499397981, 496193509, 492987869, 489781068, 486573116, 483364018, 480153783, 476942419, 473729932, 
			470516330, 467301621, 464085812, 460868912, 457650927, 454431865, 451211733, 447990540, 444768293, 441544999, 438320666, 435095302, 431868914, 428641510, 425413097, 422183683, 
			418953276, 415721883, 412489511, 409256169, 406021864, 402786603, 399550395, 396313247, 393075166, 389836160, 386596236, 383355403, 380113668, 376871039, 373627522, 370383127, 
			367137860, 363891729, 360644742, 357396906, 354148229, 350898719, 347648383, 344397229, 341145265, 337892497, 334638935, 331384586, 328129456, 324873555, 321616889, 318359466, 
			315101294, 311842381, 308582733, 305322360, 302061268, 298799466, 295536960, 292273759, 289009870, 285745301, 282480060, 279214154, 275947591, 272680379, 269412525, 266144037, 
			262874923, 259605190, 256334846, 253063900, 249792357, 246520227, 243247517, 239974234, 236700387, 233425983, 230151030, 226875534, 223599506, 220322950, 217045877, 213768293, 
			210490205, 207211623, 203932553, 200653003, 197372981, 194092494, 190811551, 187530158, 184248325, 180966058, 177683365, 174400253, 171116732, 167832808, 164548489, 161263783, 
			157978697, 154693239, 151407418, 148121240, 144834714, 141547847, 138260647, 134973121, 131685278, 128397125, 125108670, 121819920, 118530884, 115241569, 111951983, 108662133, 
			105372028, 102081674, 98791081, 95500255, 92209204, 88917936, 85626459, 82334781, 79042909, 75750851, 72458614, 69166207, 65873638, 62580913, 59288041, 55995030, 
			52701886, 49408619, 46115236, 42821744, 39528151, 36234465, 32940694, 29646846, 26352927, 23058947, 19764912, 16470831, 13176711, 9882561, 6588386, 3294197, 
			0, -3294197, -6588386, -9882561, -13176711, -16470831, -19764912, -23058947, -26352927, -29646846, -32940694, -36234465, -39528151, -42821744, -46115236, -49408619, 
			-52701886, -55995030, -59288041, -62580913, -65873638, -69166207, -72458614, -75750851, -79042909, -82334781, -85626459, -88917936, -92209204, -95500255, -98791081, -102081674, 
			-105372028, -108662133, -111951983, -115241569, -118530884, -121819920, -125108670, -128397125, -131685278, -134973121, -138260647, -141547847, -144834714, -148121240, -151407418, -154693239, 
			-157978697, -161263783, -164548489, -167832808, -171116732, -174400253, -177683365, -180966058, -184248325, -187530158, -190811551, -194092494, -197372981, -200653003, -203932553, -207211623, 
			-210490205, -213768293, -217045877, -220322950, -223599506, -226875534, -230151030, -233425983, -236700387, -239974234, -243247517, -246520227, -249792357, -253063900, -256334846, -259605190, 
			-262874923, -266144037, -269412525, -272680379, -275947591, -279214154, -282480060, -285745301, -289009870, -292273759, -295536960, -298799466, -302061268, -305322360, -308582733, -311842381, 
			-315101294, -318359466, -321616889, -324873555, -328129456, -331384586, -334638935, -337892497, -341145265, -344397229, -347648383, -350898719, -354148229, -357396906, -360644742, -363891729, 
			-367137860, -370383127, -373627522, -376871039, -380113668, -383355403, -386596236, -389836160, -393075166, -396313247, -399550395, -402786603, -406021864, -409256169, -412489511, -415721883, 
			-418953276, -422183683, -425413097, -428641510, -431868914, -435095302, -438320666, -441544999, -444768293, -447990540, -451211733, -454431865, -457650927, -460868912, -464085812, -467301621, 
			-470516330, -473729932, -476942419, -480153783, -483364018, -486573116, -489781068, -492987869, -496193509, -499397981, -502601278, -505803393, -509004317, -512204044, -515402565, -518599874, 
			-521795962, -524990823, -528184448, -531376830, -534567962, -537757836, -540946445, -544133780, -547319836, -550504603, -553688075, -556870244, -560051103, -563230644, -566408859, -569585742, 
			-572761285, -575935479, -579108319, -582279796, -585449902, -588618631, -591785975, -594951926, -598116478, -601279622, -604441351, -607601658, -610760535, -613917975, -617073970, -620228513, 
			-623381597, -626533214, -629683356, -632832017, -635979189, -639124864, -642269036, -645411696, -648552837, -651692452, -654830534, -657967075, -661102067, -664235504, -667367378, -670497682, 
			-673626407, -676753548, -679879096, -683003045, -686125386, -689246113, -692365217, -695482693, -698598532, -701712727, -704825271, -707936157, -711045377, -714152923, -717258789, -720362967, 
			-723465451, -726566231, -729665302, -732762657, -735858286, -738952185, -742044344, -745134758, -748223418, -751310317, -754395449, -757478805, -760560379, -763640163, -766718150, -769794333, 
			-772868705, -775941258, -779011985, -782080880, -785147933, -788213140, -791276491, -794337981, -797397601, -800455345, -803511206, -806565176, -809617248, -812667415, -815715669, -818762004, 
			-821806412, -824848887, -827889421, -830928006, -833964637, -836999305, -840032003, -843062725, -846091463, -849118210, -852142958, -855165702, -858186434, -861205146, -864221831, -867236483, 
			-870249094, -873259658, -876268167, -879274613, -882278991, -885281292, -888281511, -891279639, -894275670, -897269597, -900261412, -903251109, -906238680, -909224119, -912207418, -915188571, 
			-918167571, -921144410, -924119081, -927091578, -930061893, -933030020, -935995951, -938959680, -941921199, -944880502, -947837581, -950792430, -953745042, -956695410, -959643526, -962589384, 
			-965532977, -968474299, -971413341, -974350097, -977284561, -980216725, -983146582, -986074126, -988999350, -991922247, -994842809, -997761030, -1000676904, -1003590423, -1006501580, -1009410369, 
			-1012316783, -1015220815, -1018122458, -1021021705, -1023918549, -1026812984, -1029705003, -1032594599, -1035481765, -1038366494, -1041248781, -1044128616, -1047005996, -1049880911, -1052753356, -1055623323, 
			-1058490807, -1061355800, -1064218295, -1067078287, -1069935767, -1072790729, -1075643168, -1078493075, -1081340444, -1084185269, -1087027543, -1089867258, -1092704410, -1095538990, -1098370992, -1101200409, 
			-1104027236, -1106851464, -1109673088, -1112492100, -1115308495, -1118122266, -1120933405, -1123741907, -1126547764, -1129350971, -1132151520, -1134949405, -1137744620, -1140537157, -1143327010, -1146114173, 
			-1148898639, -1151680402, -1154459455, -1157235791, -1160009404, -1162780287, -1165548434, -1168313839, -1171076494, -1173836394, -1176593532, -1179347901, -1182099495, -1184848307, -1187594331, -1190337561, 
			-1193077990, -1195815611, -1198550418, -1201282406, -1204011566, -1206737893, -1209461381, -1212182022, -1214899812, -1217614742, -1220326808, -1223036001, -1225742317, -1228445749, -1231146290, -1233843934, 
			-1236538674, -1239230505, -1241919420, -1244605413, -1247288476, -1249968605, -1252645793, -1255320033, -1257991319, -1260659645, -1263325004, -1265987391, -1268646799, -1271303221, -1273956652, -1276607085, 
			-1279254514, -1281898933, -1284540336, -1287178716, -1289814067, -1292446383, -1295075658, -1297701885, -1300325059, -1302945173, -1305562221, -1308176197, -1310787094, -1313394908, -1315999630, -1318601256, 
			-1321199779, -1323795194, -1326387493, -1328976671, -1331562722, -1334145640, -1336725418, -1339302051, -1341875532, -1344445856, -1347013016, -1349577006, -1352137821, -1354695454, -1357249900, -1359801151, 
			-1362349203, -1364894049, -1367435684, -1369974100, -1372509293, -1375041257, -1377569984, -1380095471, -1382617709, -1385136695, -1387652420, -1390164881, -1392674071, -1395179983, -1397682612, -1400181953, 
			-1402677998, -1405170743, -1407660182, -1410146308, -1412629116, -1415108600, -1417584754, -1420057573, -1422527049, -1424993179, -1427455955, -1429915373, -1432371425, -1434824107, -1437273413, -1439719337, 
			-1442161873, -1444601016, -1447036759, -1449469097, -1451898024, -1454323535, -1456745624, -1459164285, -1461579512, -1463991301, -1466399644, -1468804536, -1471205973, -1473603947, -1475998454, -1478389488, 
			-1480777043, -1483161114, -1485541694, -1487918779, -1490292363, -1492662440, -1495029005, -1497392051, -1499751575, -1502107569, -1504460028, -1506808947, -1509154321, -1511496144, -1513834409, -1516169113, 
			-1518500249, -1520827811, -1523151796, -1525472195, -1527789006, -1530102221, -1532411836, -1534717845, -1537020242, -1539319023, -1541614182, -1543905713, -1546193611, -1548477871, -1550758487, -1553035454, 
			-1555308767, -1557578419, -1559844407, -1562106724, -1564365365, -1566620326, -1568871600, -1571119182, -1573363067, -1575603250, -1577839725, -1580072488, -1582301532, -1584526853, -1586748446, -1588966305, 
			-1591180424, -1593390800, -1595597426, -1597800298, -1599999410, -1602194757, -1604386334, -1606574135, -1608758156, -1610938392, -1613114836, -1615287485, -1617456334, -1619621376, -1621782607, -1623940021, 
			-1626093615, -1628243382, -1630389317, -1632531417, -1634669674, -1636804086, -1638934645, -1641061348, -1643184190, -1645303165, -1647418268, -1649529495, -1651636840, -1653740299, -1655839866, -1657935537, 
			-1660027307, -1662115171, -1664199123, -1666279160, -1668355275, -1670427465, -1672495724, -1674560047, -1676620430, -1678676868, -1680729356, -1682777889, -1684822462, -1686863071, -1688899710, -1690932375, 
			-1692961061, -1694985763, -1697006477, -1699023198, -1701035921, -1703044641, -1705049354, -1707050054, -1709046738, -1711039400, -1713028036, -1715012641, -1716993210, -1718969739, -1720942223, -1722910658, 
			-1724875039, -1726835360, -1728791618, -1730743809, -1732691926, -1734635967, -1736575926, -1738511798, -1740443579, -1742371265, -1744294851, -1746214333, -1748129706, -1750040965, -1751948106, -1753851124, 
			-1755750016, -1757644776, -1759535400, -1761421884, -1763304223, -1765182413, -1767056449, -1768926327, -1770792043, -1772653592, -1774510969, -1776364171, -1778213193, -1780058031, -1781898680, -1783735136, 
			-1785567395, -1787395452, -1789219303, -1791038944, -1792854371, -1794665579, -1796472564, -1798275321, -1800073847, -1801868137, -1803658188, -1805443994, -1807225552, -1809002857, -1810775905, -1812544693, 
			-1814309215, -1816069468, -1817825448, -1819577150, -1821324571, -1823067705, -1824806550, -1826541101, -1828271354, -1829997305, -1831718950, -1833436285, -1835149305, -1836858007, -1838562387, -1840262440, 
			-1841958163, -1843649552, -1845336602, -1847019310, -1848697672, -1850371684, -1852041342, -1853706642, -1855367579, -1857024151, -1858676353, -1860324182, -1861967633, -1863606703, -1865241387, -1866871682, 
			-1868497584, -1870119090, -1871736195, -1873348896, -1874957188, -1876561068, -1878160533, -1879755578, -1881346200, -1882932395, -1884514160, -1886091490, -1887664381, -1889232831, -1890796835, -1892356390, 
			-1893911493, -1895462138, -1897008324, -1898550045, -1900087299, -1901620082, -1903148391, -1904672221, -1906191569, -1907706431, -1909216805, -1910722686, -1912224071, -1913720956, -1915213339, -1916701214, 
			-1918184579, -1919663431, -1921137766, -1922607579, -1924072869, -1925533631, -1926989863, -1928441559, -1929888719, -1931331336, -1932769410, -1934202935, -1935631909, -1937056328, -1938476189, -1939891489, 
			-1941302224, -1942708390, -1944109986, -1945507007, -1946899449, -1948287311, -1949670588, -1951049277, -1952423376, -1953792880, -1955157786, -1956518092, -1957873794, -1959224889, -1960571374, -1961913245, 
			-1963250500, -1964583135, -1965911147, -1967234533, -1968553290, -1969867415, -1971176905, -1972481756, -1973781966, -1975077531, -1976368449, -1977654716, -1978936329, -1980213286, -1981485584, -1982753218, 
			-1984016187, -1985274488, -1986528117, -1987777071, -1989021348, -1990260945, -1991495858, -1992726085, -1993951623, -1995172470, -1996388621, -1997600075, -1998806828, -2000008877, -2001206221, -2002398855, 
			-2003586778, -2004769986, -2005948476, -2007122247, -2008291294, -2009455616, -2010615209, -2011770071, -2012920199, -2014065591, -2015206243, -2016342154, -2017473319, -2018599738, -2019721406, -2020838322, 
			-2021950482, -2023057885, -2024160527, -2025258407, -2026351520, -2027439866, -2028523440, -2029602242, -2030676267, -2031745514, -2032809981, -2033869663, -2034924560, -2035974669, -2037019987, -2038060511, 
			-2039096240, -2040127170, -2041153300, -2042174627, -2043191148, -2044202862, -2045209765, -2046211856, -2047209132, -2048201590, -2049189229, -2050172046, -2051150039, -2052123205, -2053091543, -2054055049, 
			-2055013722, -2055967559, -2056916558, -2057860717, -2058800034, -2059734506, -2060664132, -2061588909, -2062508834, -2063423906, -2064334123, -2065239482, -2066139982, -2067035619, -2067926393, -2068812301, 
			-2069693340, -2070569510, -2071440807, -2072307229, -2073168776, -2074025444, -2074877232, -2075724137, -2076566158, -2077403293, -2078235539, -2079062895, -2079885359, -2080702928, -2081515602, -2082323377, 
			-2083126253, -2083924227, -2084717297, -2085505461, -2086288718, -2087067066, -2087840503, -2088609027, -2089372636, -2090131329, -2090885104, -2091633958, -2092377891, -2093116900, -2093850983, -2094580140, 
			-2095304368, -2096023666, -2096738031, -2097447462, -2098151959, -2098851517, -2099546137, -2100235817, -2100920555, -2101600349, -2102275197, -2102945099, -2103610052, -2104270056, -2104925108, -2105575206, 
			-2106220350, -2106860538, -2107495769, -2108126040, -2108751350, -2109371699, -2109987084, -2110597504, -2111202957, -2111803443, -2112398959, -2112989505, -2113575078, -2114155678, -2114731304, -2115301953, 
			-2115867624, -2116428317, -2116984030, -2117534761, -2118080509, -2118621274, -2119157053, -2119687845, -2120213650, -2120734465, -2121250290, -2121761124, -2122266965, -2122767812, -2123263664, -2123754520, 
			-2124240379, -2124721239, -2125197099, -2125667958, -2126133816, -2126594670, -2127050521, -2127501366, -2127947205, -2128388037, -2128823860, -2129254674, -2129680478, -2130101271, -2130517051, -2130927817, 
			-2131333570, -2131734307, -2132130028, -2132520732, -2132906418, -2133287086, -2133662733, -2134033359, -2134398964, -2134759547, -2135115106, -2135465641, -2135811151, -2136151635, -2136487093, -2136817524, 
			-2137142926, -2137463299, -2137778643, -2138088956, -2138394238, -2138694489, -2138989706, -2139279891, -2139565041, -2139845157, -2140120238, -2140390283, -2140655291, -2140915262, -2141170196, -2141420091, 
			-2141664947, -2141904763, -2142139539, -2142369275, -2142593969, -2142813622, -2143028233, -2143237800, -2143442325, -2143641805, -2143836242, -2144025634, -2144209981, -2144389282, -2144563537, -2144732746, 
			-2144896908, -2145056023, -2145210091, -2145359111, -2145503082, -2145642005, -2145775879, -2145904703, -2146028478, -2146147204, -2146260879, -2146369504, -2146473078, -2146571602, -2146665074, -2146753495, 
			-2146836865, -2146915182, -2146988448, -2147056662, -2147119824, -2147177933, -2147230990, -2147278994, -2147321945, -2147359843, -2147392689, -2147420481, -2147443221, -2147460907, -2147473540, -2147481120, 
			-2147483647, -2147481120, -2147473540, -2147460907, -2147443221, -2147420481, -2147392689, -2147359843, -2147321945, -2147278994, -2147230990, -2147177933, -2147119824, -2147056662, -2146988448, -2146915182, 
			-2146836865, -2146753495, -2146665074, -2146571602, -2146473078, -2146369504, -2146260879, -2146147204, -2146028478, -2145904703, -2145775879, -2145642005, -2145503082, -2145359111, -2145210091, -2145056023, 
			-2144896908, -2144732746, -2144563537, -2144389282, -2144209981, -2144025634, -2143836242, -2143641805, -2143442325, -2143237800, -2143028233, -2142813622, -2142593969, -2142369275, -2142139539, -2141904763, 
			-2141664947, -2141420091, -2141170196, -2140915262, -2140655291, -2140390283, -2140120238, -2139845157, -2139565041, -2139279891, -2138989706, -2138694489, -2138394238, -2138088956, -2137778643, -2137463299, 
			-2137142926, -2136817524, -2136487093, -2136151635, -2135811151, -2135465641, -2135115106, -2134759547, -2134398964, -2134033359, -2133662733, -2133287086, -2132906418, -2132520732, -2132130028, -2131734307, 
			-2131333570, -2130927817, -2130517051, -2130101271, -2129680478, -2129254674, -2128823860, -2128388037, -2127947205, -2127501366, -2127050521, -2126594670, -2126133816, -2125667958, -2125197099, -2124721239, 
			-2124240379, -2123754520, -2123263664, -2122767812, -2122266965, -2121761124, -2121250290, -2120734465, -2120213650, -2119687845, -2119157053, -2118621274, -2118080509, -2117534761, -2116984030, -2116428317, 
			-2115867624, -2115301953, -2114731304, -2114155678, -2113575078, -2112989505, -2112398959, -2111803443, -2111202957, -2110597504, -2109987084, -2109371699, -2108751350, -2108126040, -2107495769, -2106860538, 
			-2106220350, -2105575206, -2104925108, -2104270056, -2103610052, -2102945099, -2102275197, -2101600349, -2100920555, -2100235817, -2099546137, -2098851517, -2098151959, -2097447462, -2096738031, -2096023666, 
			-2095304368, -2094580140, -2093850983, -2093116900, -2092377891, -2091633958, -2090885104, -2090131329, -2089372636, -2088609027, -2087840503, -2087067066, -2086288718, -2085505461, -2084717297, -2083924227, 
			-2083126253, -2082323377, -2081515602, -2080702928, -2079885359, -2079062895, -2078235539, -2077403293, -2076566158, -2075724137, -2074877232, -2074025444, -2073168776, -2072307229, -2071440807, -2070569510, 
			-2069693340, -2068812301, -2067926393, -2067035619, -2066139982, -2065239482, -2064334123, -2063423906, -2062508834, -2061588909, -2060664132, -2059734506, -2058800034, -2057860717, -2056916558, -2055967559, 
			-2055013722, -2054055049, -2053091543, -2052123205, -2051150039, -2050172046, -2049189229, -2048201590, -2047209132, -2046211856, -2045209765, -2044202862, -2043191148, -2042174627, -2041153300, -2040127170, 
			-2039096240, -2038060511, -2037019987, -2035974669, -2034924560, -2033869663, -2032809981, -2031745514, -2030676267, -2029602242, -2028523440, -2027439866, -2026351520, -2025258407, -2024160527, -2023057885, 
			-2021950482, -2020838322, -2019721406, -2018599738, -2017473319, -2016342154, -2015206243, -2014065591, -2012920199, -2011770071, -2010615209, -2009455616, -2008291294, -2007122247, -2005948476, -2004769986, 
			-2003586778, -2002398855, -2001206221, -2000008877, -1998806828, -1997600075, -1996388621, -1995172470, -1993951623, -1992726085, -1991495858, -1990260945, -1989021348, -1987777071, -1986528117, -1985274488, 
			-1984016187, -1982753218, -1981485584, -1980213286, -1978936329, -1977654716, -1976368449, -1975077531, -1973781966, -1972481756, -1971176905, -1969867415, -1968553290, -1967234533, -1965911147, -1964583135, 
			-1963250500, -1961913245, -1960571374, -1959224889, -1957873794, -1956518092, -1955157786, -1953792880, -1952423376, -1951049277, -1949670588, -1948287311, -1946899449, -1945507007, -1944109986, -1942708390, 
			-1941302224, -1939891489, -1938476189, -1937056328, -1935631909, -1934202935, -1932769410, -1931331336, -1929888719, -1928441559, -1926989863, -1925533631, -1924072869, -1922607579, -1921137766, -1919663431, 
			-1918184579, -1916701214, -1915213339, -1913720956, -1912224071, -1910722686, -1909216805, -1907706431, -1906191569, -1904672221, -1903148391, -1901620082, -1900087299, -1898550045, -1897008324, -1895462138, 
			-1893911493, -1892356390, -1890796835, -1889232831, -1887664381, -1886091490, -1884514160, -1882932395, -1881346200, -1879755578, -1878160533, -1876561068, -1874957188, -1873348896, -1871736195, -1870119090, 
			-1868497584, -1866871682, -1865241387, -1863606703, -1861967633, -1860324182, -1858676353, -1857024151, -1855367579, -1853706642, -1852041342, -1850371684, -1848697672, -1847019310, -1845336602, -1843649552, 
			-1841958163, -1840262440, -1838562387, -1836858007, -1835149305, -1833436285, -1831718950, -1829997305, -1828271354, -1826541101, -1824806550, -1823067705, -1821324571, -1819577150, -1817825448, -1816069468, 
			-1814309215, -1812544693, -1810775905, -1809002857, -1807225552, -1805443994, -1803658188, -1801868137, -1800073847, -1798275321, -1796472564, -1794665579, -1792854371, -1791038944, -1789219303, -1787395452, 
			-1785567395, -1783735136, -1781898680, -1780058031, -1778213193, -1776364171, -1774510969, -1772653592, -1770792043, -1768926327, -1767056449, -1765182413, -1763304223, -1761421884, -1759535400, -1757644776, 
			-1755750016, -1753851124, -1751948106, -1750040965, -1748129706, -1746214333, -1744294851, -1742371265, -1740443579, -1738511798, -1736575926, -1734635967, -1732691926, -1730743809, -1728791618, -1726835360, 
			-1724875039, -1722910658, -1720942223, -1718969739, -1716993210, -1715012641, -1713028036, -1711039400, -1709046738, -1707050054, -1705049354, -1703044641, -1701035921, -1699023198, -1697006477, -1694985763, 
			-1692961061, -1690932375, -1688899710, -1686863071, -1684822462, -1682777889, -1680729356, -1678676868, -1676620430, -1674560047, -1672495724, -1670427465, -1668355275, -1666279160, -1664199123, -1662115171, 
			-1660027307, -1657935537, -1655839866, -1653740299, -1651636840, -1649529495, -1647418268, -1645303165, -1643184190, -1641061348, -1638934645, -1636804086, -1634669674, -1632531417, -1630389317, -1628243382, 
			-1626093615, -1623940021, -1621782607, -1619621376, -1617456334, -1615287485, -1613114836, -1610938392, -1608758156, -1606574135, -1604386334, -1602194757, -1599999410, -1597800298, -1595597426, -1593390800, 
			-1591180424, -1588966305, -1586748446, -1584526853, -1582301532, -1580072488, -1577839725, -1575603250, -1573363067, -1571119182, -1568871600, -1566620326, -1564365365, -1562106724, -1559844407, -1557578419, 
			-1555308767, -1553035454, -1550758487, -1548477871, -1546193611, -1543905713, -1541614182, -1539319023, -1537020242, -1534717845, -1532411836, -1530102221, -1527789006, -1525472195, -1523151796, -1520827811, 
			-1518500249, -1516169113, -1513834409, -1511496144, -1509154321, -1506808947, -1504460028, -1502107569, -1499751575, -1497392051, -1495029005, -1492662440, -1490292363, -1487918779, -1485541694, -1483161114, 
			-1480777043, -1478389488, -1475998454, -1473603947, -1471205973, -1468804536, -1466399644, -1463991301, -1461579512, -1459164285, -1456745624, -1454323535, -1451898024, -1449469097, -1447036759, -1444601016, 
			-1442161873, -1439719337, -1437273413, -1434824107, -1432371425, -1429915373, -1427455955, -1424993179, -1422527049, -1420057573, -1417584754, -1415108600, -1412629116, -1410146308, -1407660182, -1405170743, 
			-1402677998, -1400181953, -1397682612, -1395179983, -1392674071, -1390164881, -1387652420, -1385136695, -1382617709, -1380095471, -1377569984, -1375041257, -1372509293, -1369974100, -1367435684, -1364894049, 
			-1362349203, -1359801151, -1357249900, -1354695454, -1352137821, -1349577006, -1347013016, -1344445856, -1341875532, -1339302051, -1336725418, -1334145640, -1331562722, -1328976671, -1326387493, -1323795194, 
			-1321199779, -1318601256, -1315999630, -1313394908, -1310787094, -1308176197, -1305562221, -1302945173, -1300325059, -1297701885, -1295075658, -1292446383, -1289814067, -1287178716, -1284540336, -1281898933, 
			-1279254514, -1276607085, -1273956652, -1271303221, -1268646799, -1265987391, -1263325004, -1260659645, -1257991319, -1255320033, -1252645793, -1249968605, -1247288476, -1244605413, -1241919420, -1239230505, 
			-1236538674, -1233843934, -1231146290, -1228445749, -1225742317, -1223036001, -1220326808, -1217614742, -1214899812, -1212182022, -1209461381, -1206737893, -1204011566, -1201282406, -1198550418, -1195815611, 
			-1193077990, -1190337561, -1187594331, -1184848307, -1182099495, -1179347901, -1176593532, -1173836394, -1171076494, -1168313839, -1165548434, -1162780287, -1160009404, -1157235791, -1154459455, -1151680402, 
			-1148898639, -1146114173, -1143327010, -1140537157, -1137744620, -1134949405, -1132151520, -1129350971, -1126547764, -1123741907, -1120933405, -1118122266, -1115308495, -1112492100, -1109673088, -1106851464, 
			-1104027236, -1101200409, -1098370992, -1095538990, -1092704410, -1089867258, -1087027543, -1084185269, -1081340444, -1078493075, -1075643168, -1072790729, -1069935767, -1067078287, -1064218295, -1061355800, 
			-1058490807, -1055623323, -1052753356, -1049880911, -1047005996, -1044128616, -1041248781, -1038366494, -1035481765, -1032594599, -1029705003, -1026812984, -1023918549, -1021021705, -1018122458, -1015220815, 
			-1012316783, -1009410369, -1006501580, -1003590423, -1000676904, -997761030, -994842809, -991922247, -988999350, -986074126, -983146582, -980216725, -977284561, -974350097, -971413341, -968474299, 
			-965532977, -962589384, -959643526, -956695410, -953745042, -950792430, -947837581, -944880502, -941921199, -938959680, -935995951, -933030020, -930061893, -927091578, -924119081, -921144410, 
			-918167571, -915188571, -912207418, -909224119, -906238680, -903251109, -900261412, -897269597, -894275670, -891279639, -888281511, -885281292, -882278991, -879274613, -876268167, -873259658, 
			-870249094, -867236483, -864221831, -861205146, -858186434, -855165702, -852142958, -849118210, -846091463, -843062725, -840032003, -836999305, -833964637, -830928006, -827889421, -824848887, 
			-821806412, -818762004, -815715669, -812667415, -809617248, -806565176, -803511206, -800455345, -797397601, -794337981, -791276491, -788213140, -785147933, -782080880, -779011985, -775941258, 
			-772868705, -769794333, -766718150, -763640163, -760560379, -757478805, -754395449, -751310317, -748223418, -745134758, -742044344, -738952185, -735858286, -732762657, -729665302, -726566231, 
			-723465451, -720362967, -717258789, -714152923, -711045377, -707936157, -704825271, -701712727, -698598532, -695482693, -692365217, -689246113, -686125386, -683003045, -679879096, -676753548, 
			-673626407, -670497682, -667367378, -664235504, -661102067, -657967075, -654830534, -651692452, -648552837, -645411696, -642269036, -639124864, -635979189, -632832017, -629683356, -626533214, 
			-623381597, -620228513, -617073970, -613917975, -610760535, -607601658, -604441351, -601279622, -598116478, -594951926, -591785975, -588618631, -585449902, -582279796, -579108319, -575935479, 
			-572761285, -569585742, -566408859, -563230644, -560051103, -556870244, -553688075, -550504603, -547319836, -544133780, -540946445, -537757836, -534567962, -531376830, -528184448, -524990823, 
			-521795962, -518599874, -515402565, -512204044, -509004317, -505803393, -502601278, -499397981, -496193509, -492987869, -489781068, -486573116, -483364018, -480153783, -476942419, -473729932, 
			-470516330, -467301621, -464085812, -460868912, -457650927, -454431865, -451211733, -447990540, -444768293, -441544999, -438320666, -435095302, -431868914, -428641510, -425413097, -422183683, 
			-418953276, -415721883, -412489511, -409256169, -406021864, -402786603, -399550395, -396313247, -393075166, -389836160, -386596236, -383355403, -380113668, -376871039, -373627522, -370383127, 
			-367137860, -363891729, -360644742, -357396906, -354148229, -350898719, -347648383, -344397229, -341145265, -337892497, -334638935, -331384586, -328129456, -324873555, -321616889, -318359466, 
			-315101294, -311842381, -308582733, -305322360, -302061268, -298799466, -295536960, -292273759, -289009870, -285745301, -282480060, -279214154, -275947591, -272680379, -269412525, -266144037, 
			-262874923, -259605190, -256334846, -253063900, -249792357, -246520227, -243247517, -239974234, -236700387, -233425983, -230151030, -226875534, -223599506, -220322950, -217045877, -213768293, 
			-210490205, -207211623, -203932553, -200653003, -197372981, -194092494, -190811551, -187530158, -184248325, -180966058, -177683365, -174400253, -171116732, -167832808, -164548489, -161263783, 
			-157978697, -154693239, -151407418, -148121240, -144834714, -141547847, -138260647, -134973121, -131685278, -128397125, -125108670, -121819920, -118530884, -115241569, -111951983, -108662133, 
			-105372028, -102081674, -98791081, -95500255, -92209204, -88917936, -85626459, -82334781, -79042909, -75750851, -72458614, -69166207, -65873638, -62580913, -59288041, -55995030, 
			-52701886, -49408619, -46115236, -42821744, -39528151, -36234465, -32940694, -29646846, -26352927, -23058947, -19764912, -16470831, -13176711, -9882561, -6588386, -3294197 };

fixed flattop_window[2049] = { 0, -99, -403, -910, -1620, -2532, -3648, -4966, -6488, -8213, -10141, -12272, -14607, -17146, -19889, -22836,
			-25987, -29343, -32904, -36670, -40641, -44817, -49200, -53788, -58583, -63585, -68795, -74211, -79836, -85669, -91711, -97961,
			-104422, -111092, -117973, -125065, -132368, -139883, -147610, -155551, -163705, -172073, -180656, -189454, -198468, -207698, -217145, -226810,
			-236694, -246796, -257118, -267660, -278424, -289409, -300617, -312049, -323704, -335584, -347690, -360022, -372582, -385370, -398386, -411632,
			-425109, -438818, -452759, -466933, -481341, -495985, -510865, -525982, -541337, -556931, -572765, -588840, -605157, -621717, -638522, -655571,
			-672867, -690411, -708202, -726244, -744536, -763080, -781877, -800929, -820235, -839798, -859619, -879699, -900039, -920641, -941505, -962633,
			-984026, -1005686, -1027613, -1049810, -1072277, -1095015, -1118027, -1141313, -1164875, -1188714, -1212832, -1237229, -1261908, -1286870, -1312116, -1337647,
			-1363466, -1389573, -1415970, -1442659, -1469641, -1496918, -1524490, -1552360, -1580529, -1608999, -1637771, -1666847, -1696229, -1725917, -1755914, -1786221,
			-1816841, -1847773, -1879021, -1910585, -1942468, -1974671, -2007196, -2040044, -2073218, -2106718, -2140547, -2174706, -2209198, -2244023, -2279184, -2314682,
			-2350520, -2386698, -2423219, -2460085, -2497297, -2534857, -2572767, -2611029, -2649645, -2688616, -2727944, -2767632, -2807681, -2848093, -2888870, -2930014,
			-2971526, -3013409, -3055665, -3098295, -3141302, -3184687, -3228452, -3272600, -3317132, -3362050, -3407357, -3453053, -3499142, -3545626, -3592505, -3639783,
			-3687461, -3735541, -3784026, -3832917, -3882217, -3931927, -3982049, -4032587, -4083541, -4134913, -4186707, -4238924, -4291565, -4344634, -4398132, -4452061,
			-4506424, -4561223, -4616459, -4672135, -4728253, -4784815, -4841823, -4899280, -4957187, -5015547, -5074362, -5133634, -5193366, -5253558, -5314214, -5375336,
			-5436926, -5498986, -5561518, -5624525, -5688008, -5751970, -5816414, -5881340, -5946752, -6012652, -6079042, -6145923, -6213300, -6281172, -6349544, -6418416,
			-6487792, -6557673, -6628062, -6698960, -6770371, -6842296, -6914738, -6987698, -7061180, -7135185, -7209715, -7284773, -7360362, -7436482, -7513137, -7590329,
			-7668059, -7746331, -7825146, -7904507, -7984415, -8064874, -8145886, -8227451, -8309574, -8392256, -8475499, -8559305, -8643678, -8728618, -8814129, -8900212,
			-8986869, -9074104, -9161917, -9250312, -9339291, -9428855, -9519007, -9609749, -9701084, -9793013, -9885539, -9978664, -10072391, -10166720, -10261655, -10357198,
			-10453351, -10550115, -10647494, -10745490, -10844104, -10943338, -11043196, -11143678, -11244788, -11346527, -11448898, -11551902, -11655542, -11759819, -11864737, -11970296,
			-12076500, -12183350, -12290848, -12398997, -12507798, -12617254, -12727367, -12838138, -12949570, -13061664, -13174424, -13287850, -13401945, -13516711, -13632150, -13748264,
			-13865055, -13982525, -14100675, -14219509, -14339027, -14459232, -14580126, -14701710, -14823987, -14946959, -15070626, -15194993, -15320059, -15445827, -15572299, -15699478,
			-15827363, -15955959, -16085265, -16215285, -16346020, -16477471, -16609641, -16742531, -16876144, -17010480, -17145542, -17281332, -17417850, -17555098, -17693080, -17831796,
			-17971248, -18111436, -18252366, -18394034, -18536446, -18679602, -18823502, -18968152, -19113548, -19259696, -19406596, -19554248, -19702656, -19851822, -20001744, -20152426,
			-20303868, -20456074, -20609044, -20762778, -20917278, -21072548, -21228586, -21385396, -21542978, -21701332, -21860462, -22020368, -22181050, -22342512, -22504754, -22667778,
			-22831582, -22996172, -23161546, -23327704, -23494652, -23662386, -23830910, -24000224, -24170330, -24341230, -24512922, -24685408, -24858692, -25032770, -25207648, -25383322,
			-25559798, -25737074, -25915150, -26094030, -26273712, -26454198, -26635490, -26817586, -27000490, -27184200, -27368718, -27554044, -27740180, -27927126, -28114884, -28303452,
			-28492832, -28683024, -28874030, -29065850, -29258484, -29451932, -29646196, -29841276, -30037170, -30233884, -30431412, -30629760, -30828924, -31028906, -31229708, -31431326,
			-31633766, -31837024, -32041100, -32245998, -32451714, -32658252, -32865608, -33073786, -33282784, -33492602, -33703240, -33914700, -34126976, -34340076, -34553996, -34768736,
			-34984296, -35200680, -35417880, -35635896, -35854736, -36074396, -36294872, -36516168, -36738284, -36961216, -37184964, -37409532, -37634912, -37861112, -38088128, -38315960,
			-38544604, -38774064, -39004340, -39235424, -39467324, -39700036, -39933556, -40167888, -40403028, -40638976, -40875736, -41113300, -41351672, -41590848, -41830828, -42071612,
			-42313200, -42555588, -42798776, -43042760, -43287548, -43533128, -43779508, -44026680, -44274648, -44523408, -44772956, -45023296, -45274424, -45526336, -45779036, -46032520,
			-46286784, -46541832, -46797660, -47054264, -47311644, -47569800, -47828728, -48088428, -48348896, -48610136, -48872136, -49134904, -49398436, -49662724, -49927772, -50193580,
			-50460140, -50727452, -50995516, -51264324, -51533884, -51804184, -52075228, -52347012, -52619532, -52892788, -53166780, -53441496, -53716944, -53993120, -54270016, -54547632,
			-54825968, -55105020, -55384780, -55665256, -55946436, -56228324, -56510912, -56794200, -57078188, -57362864, -57648236, -57934292, -58221036, -58508460, -58796564, -59085340,
			-59374792, -59664916, -59955704, -60247156, -60539268, -60832032, -61125456, -61419528, -61714244, -62009604, -62305604, -62602244, -62899512, -63197408, -63495932, -63795080,
			-64094844, -64395220, -64696208, -64997804, -65300000, -65602796, -65906188, -66210168, -66514740, -66819892, -67125624, -67431928, -67738800, -68046248, -68354248, -68662816,
			-68971928, -69281592, -69591808, -69902552, -70213840, -70525656, -70838000, -71150864, -71464248, -71778144, -72092552, -72407456, -72722864, -73038760, -73355152, -73672024,
			-73989376, -74307200, -74625488, -74944248, -75263464, -75583128, -75903248, -76223808, -76544808, -76866240, -77188096, -77510376, -77833072, -78156176, -78479688, -78803592,
			-79127896, -79452592, -79777664, -80103120, -80428936, -80755128, -81081672, -81408576, -81735824, -82063408, -82391336, -82719584, -83048160, -83377056, -83706264, -84035768,
			-84365576, -84695672, -85026064, -85356728, -85687664, -86018864, -86350328, -86682040, -87014000, -87346200, -87678640, -88011296, -88344176, -88677272, -89010568, -89344064,
			-89677752, -90011624, -90345680, -90679896, -91014288, -91348824, -91683520, -92018352, -92353320, -92688408, -93023624, -93358952, -93694384, -94029912, -94365528, -94701224,
			-95037000, -95372840, -95708744, -96044696, -96380688, -96716720, -97052776, -97388856, -97724944, -98061040, -98397128, -98733200, -99069256, -99405280, -99741272, -100077208,
			-100413096, -100748928, -101084680, -101420360, -101755944, -102091440, -102426824, -102762096, -103097248, -103432272, -103767152, -104101880, -104436456, -104770864, -105105096, -105439144,
			-105773000, -106106656, -106440096, -106773312, -107106304, -107439056, -107771560, -108103800, -108435784, -108767480, -109098896, -109430016, -109760832, -110091328, -110421504, -110751344,
			-111080840, -111409984, -111738768, -112067176, -112395200, -112722832, -113050064, -113376880, -113703272, -114029232, -114354744, -114679808, -115004408, -115328536, -115652176, -115975320,
			-116297960, -116620088, -116941688, -117262752, -117583272, -117903232, -118222624, -118541440, -118859664, -119177280, -119494296, -119810688, -120126440, -120441560, -120756016, -121069808,
			-121382928, -121695352, -122007088, -122318104, -122628408, -122937968, -123246792, -123554856, -123862160, -124168680, -124474416, -124779344, -125083464, -125386760, -125689224, -125990840,
			-126291592, -126591472, -126890480, -127188584, -127485784, -127782072, -128077424, -128371832, -128665296, -128957784, -129249304, -129539824, -129829352, -130117856, -130405336, -130691784,
			-130977176, -131261504, -131544760, -131826920, -132107992, -132387936, -132666768, -132944456, -133220992, -133496368, -133770568, -134043576, -134315376, -134585984, -134855344, -135123472,
			-135390352, -135655952, -135920288, -136183328, -136445056, -136705472, -136964560, -137222304, -137478688, -137733712, -137987344, -138239584, -138490400, -138739808, -138987776, -139234288,
			-139479344, -139722928, -139965008, -140205584, -140444656, -140682192, -140918176, -141152608, -141385472, -141616736, -141846416, -142074480, -142300912, -142525712, -142748848, -142970320,
			-143190096, -143408192, -143624576, -143839232, -144052144, -144263312, -144472720, -144680336, -144886160, -145090176, -145292368, -145492720, -145691216, -145887856, -146082608, -146275472,
			-146466416, -146655440, -146842528, -147027664, -147210832, -147392016, -147571216, -147748400, -147923552, -148096656, -148267728, -148436704, -148603616, -148768416, -148931120, -149091680,
			-149250096, -149406368, -149560464, -149712368, -149862064, -150009552, -150154800, -150297792, -150438544, -150577008, -150713168, -150847040, -150978576, -151107776, -151234624, -151359104,
			-151481200, -151600912, -151718192, -151833056, -151945456, -152055424, -152162896, -152267888, -152370368, -152470336, -152567760, -152662640, -152754944, -152844672, -152931808, -153016320,
			-153098208, -153177456, -153254032, -153327936, -153399168, -153467680, -153533472, -153596512, -153656816, -153714352, -153769104, -153821056, -153870192, -153916480, -153959936, -154000528,
			-154038256, -154073072, -154104992, -154133968, -154160016, -154183104, -154203216, -154220352, -154234480, -154245584, -154253648, -154258672, -154260608, -154259472, -154255248, -154247904,
			-154237424, -154223792, -154207008, -154187040, -154163872, -154137504, -154107904, -154075072, -154038976, -153999600, -153956944, -153910976, -153861680, -153809056, -153753072, -153693728,
			-153630976, -153564848, -153495296, -153422304, -153345856, -153265952, -153182576, -153095680, -153005280, -152911360, -152813888, -152712848, -152608240, -152500032, -152388224, -152272784,
			-152153696, -152030960, -151904544, -151774448, -151640640, -151503104, -151361840, -151216816, -151068032, -150915456, -150759088, -150598896, -150434864, -150266992, -150095248, -149919632,
			-149740112, -149556672, -149369312, -149178016, -148982752, -148783504, -148580272, -148373024, -148161760, -147946464, -147727104, -147503664, -147276144, -147044512, -146808768, -146568896,
			-146324864, -146076672, -145824288, -145567712, -145306928, -145041904, -144772656, -144499120, -144221328, -143939232, -143652832, -143362096, -143067040, -142767616, -142463824, -142155648,
			-141843072, -141526080, -141204656, -140878784, -140548448, -140213632, -139874320, -139530496, -139182144, -138829264, -138471824, -138109808, -137743200, -137372000, -136996176, -136615728,
			-136230624, -135840864, -135446416, -135047280, -134643440, -134234864, -133821560, -133403496, -132980672, -132553056, -132120640, -131683416, -131241360, -130794456, -130342696, -129886064,
			-129424544, -128958120, -128486776, -128010504, -127529280, -127043096, -126551928, -126055776, -125554616, -125048440, -124537224, -124020960, -123499632, -122973224, -122441720, -121905112,
			-121363384, -120816512, -120264496, -119707320, -119144960, -118577400, -118004640, -117426664, -116843448, -116254976, -115661248, -115062248, -114457944, -113848344, -113233424, -112613168,
			-111987568, -111356608, -110720280, -110078560, -109431432, -108778896, -108120936, -107457528, -106788664, -106114336, -105434528, -104749224, -104058408, -103362072, -102660208, -101952792,
			-101239816, -100521264, -99797128, -99067384, -98332040, -97591064, -96844448, -96092184, -95334264, -94570656, -93801368, -93026376, -92245664, -91459232, -90667064, -89869136,
			-89065456, -88255992, -87440744, -86619696, -85792832, -84960152, -84121632, -83277264, -82427032, -81570936, -80708952, -79841080, -78967296, -78087592, -77201960, -76310392,
			-75412864, -74509376, -73599912, -72684464, -71763016, -70835560, -69902088, -68962576, -68017032, -67065428, -66107764, -65144024, -64174200, -63198280, -62216256, -61228112,
			-60233844, -59233436, -58226880, -57214168, -56195288, -55170228, -54138980, -53101532, -52057876, -51008004, -49951900, -48889560, -47820976, -46746132, -45665020, -44577632,
			-43483960, -42383992, -41277720, -40165136, -39046228, -37920988, -36789408, -35651480, -34507192, -33356536, -32199506, -31036092, -29866284, -28690076, -27507456, -26318420,
			-25122956, -23921058, -22712718, -21497926, -20276676, -19048960, -17814770, -16574096, -15326933, -14073273, -12813108, -11546431, -10273233, -8993509, -7707251, -6414452,
			-5115104, -3809201, -2496736, -1177703, 147906, 1480097, 2818878, 4164253, 5516231, 6874816, 8240015, 9611835, 10990281, 12375360, 13767076, 15165437,
			16570447, 17982114, 19400440, 20825434, 22257098, 23695440, 25140466, 26592178, 28050582, 29515684, 30987488, 32466000, 33951224, 35443164, 36941824, 38447212,
			39959328, 41478176, 43003764, 44536092, 46075168, 47620996, 49173576, 50732916, 52299012, 53871876, 55451508, 57037912, 58631088, 60231044, 61837784, 63451304,
			65071612, 66698708, 68332600, 69973280, 71620760, 73275048, 74936128, 76604016, 78278712, 79960216, 81648528, 83343656, 85045592, 86754352, 88469920, 90192312,
			91921528, 93657560, 95400416, 97150088, 98906592, 100669920, 102440080, 104217056, 106000864, 107791504, 109588968, 111393256, 113204376, 115022328, 116847104, 118678712,
			120517152, 122362408, 124214504, 126073416, 127939160, 129811728, 131691128, 133577344, 135470384, 137370240, 139276928, 141190432, 143110736, 145037872, 146971824, 148912576,
			150860160, 152814528, 154775712, 156743712, 158718496, 160700096, 162688480, 164683664, 166685632, 168694400, 170709936, 172732272, 174761392, 176797280, 178839936, 180889360,
			182945552, 185008512, 187078240, 189154704, 191237920, 193327888, 195424608, 197528048, 199638240, 201755152, 203878784, 206009136, 208146208, 210289984, 212440464, 214597648,
			216761520, 218932080, 221109328, 223293264, 225483856, 227681120, 229885040, 232095616, 234312832, 236536704, 238767200, 241004336, 243248080, 245498448, 247755408, 250018992,
			252289152, 254565904, 256849232, 259139136, 261435600, 263738640, 266048208, 268364320, 270686976, 273016128, 275351840, 277694016, 280042720, 282397920, 284759584, 287127744,
			289502336, 291883424, 294270944, 296664896, 299065280, 301472064, 303885280, 306304896, 308730880, 311163264, 313602016, 316047104, 318498560, 320956384, 323420512, 325890944,
			328367680, 330850752, 333340064, 335835680, 338337568, 340845664, 343360032, 345880640, 348407456, 350940480, 353479680, 356025056, 358576640, 361134368, 363698208, 366268224,
			368844352, 371426560, 374014880, 376609312, 379209760, 381816288, 384428864, 387047488, 389672096, 392302720, 394939328, 397581920, 400230464, 402884960, 405545376, 408211744,
			410884000, 413562112, 416246144, 418936032, 421631744, 424333280, 427040672, 429753824, 432472800, 435197504, 437928000, 440664224, 443406144, 446153792, 448907136, 451666144,
			454430816, 457201152, 459977088, 462758656, 465545792, 468338528, 471136800, 473940608, 476749984, 479564832, 482385184, 485211008, 488042304, 490879008, 493721152, 496568672,
			499421600, 502279904, 505143552, 508012512, 510886816, 513766368, 516651232, 519541344, 522436672, 525337248, 528243008, 531153952, 534070048, 536991296, 539917632, 542849088,
			545785664, 548727296, 551673920, 554625600, 557582272, 560543936, 563510528, 566482112, 569458560, 572439936, 575426176, 578417344, 581413248, 584414016, 587419584, 590429888,
			593444992, 596464832, 599489344, 602518528, 605552384, 608590912, 611634048, 614681792, 617734080, 620790912, 623852288, 626918208, 629988544, 633063424, 636142656, 639226368,
			642314496, 645406912, 648503744, 651604864, 654710272, 657819968, 660933888, 664052096, 667174464, 670301056, 673431744, 676566592, 679705536, 682848576, 685995648, 689146752,
			692301888, 695460992, 698624064, 701791104, 704961984, 708136768, 711315456, 714497920, 717684224, 720874304, 724068096, 727265664, 730466880, 733671808, 736880384, 740092608,
			743308416, 746527744, 749750656, 752977088, 756207040, 759440384, 762677248, 765917440, 769161088, 772408000, 775658304, 778911936, 782168768, 785428928, 788692224, 791958784,
			795228416, 798501248, 801777216, 805056192, 808338240, 811623296, 814911360, 818202432, 821496384, 824793280, 828092992, 831395648, 834701056, 838009280, 841320256, 844633984,
			847950400, 851269440, 854591168, 857915520, 861242496, 864571968, 867904000, 871238528, 874575488, 877914944, 881256768, 884600960, 887947520, 891296384, 894647552, 898000960,
			901356544, 904714368, 908074368, 911436480, 914800704, 918166976, 921535296, 924905664, 928277952, 931652224, 935028416, 938406464, 941786368, 945168064, 948551616, 951936832,
			955323840, 958712512, 962102912, 965494848, 968888448, 972283584, 975680192, 979078400, 982478016, 985879104, 989281536, 992685376, 996090560, 999497024, 1002904768, 1006313792,
			1009723968, 1013135360, 1016547840, 1019961472, 1023376128, 1026791872, 1030208576, 1033626304, 1037044928, 1040464512, 1043884928, 1047306176, 1050728256, 1054151104, 1057574656, 1060998976,
			1064423872, 1067849472, 1071275712, 1074702464, 1078129792, 1081557504, 1084985728, 1088414464, 1091843584, 1095272960, 1098702848, 1102132864, 1105563136, 1108993792, 1112424448, 1115855360,
			1119286400, 1122717568, 1126148736, 1129579904, 1133010944, 1136442112, 1139873152, 1143303936, 1146734720, 1150165248, 1153595520, 1157025536, 1160455296, 1163884672, 1167313792, 1170742400,
			1174170496, 1177598208, 1181025408, 1184452096, 1187878144, 1191303552, 1194728320, 1198152448, 1201575680, 1204998272, 1208420096, 1211841024, 1215261184, 1218680320, 1222098560, 1225515776,
			1228932096, 1232347264, 1235761280, 1239174272, 1242586112, 1245996672, 1249406080, 1252814208, 1256221056, 1259626496, 1263030528, 1266433280, 1269834496, 1273234176, 1276632448, 1280029056,
			1283424128, 1286817536, 1290209280, 1293599232, 1296987648, 1300374144, 1303758848, 1307141632, 1310522624, 1313901568, 1317278720, 1320653696, 1324026752, 1327397632, 1330766464, 1334133248,
			1337497728, 1340860032, 1344220032, 1347577728, 1350933120, 1354286080, 1357636736, 1360984960, 1364330624, 1367673728, 1371014400, 1374352384, 1377687808, 1381020544, 1384350592, 1387677824,
			1391002368, 1394324096, 1397642880, 1400958848, 1404271872, 1407581952, 1410889088, 1414193152, 1417494016, 1420791936, 1424086656, 1427378176, 1430666496, 1433951488, 1437233280, 1440511616,
			1443786752, 1447058304, 1450326528, 1453591296, 1456852480, 1460110080, 1463364224, 1466614656, 1469861376, 1473104384, 1476343808, 1479579264, 1482811008, 1486038912, 1489262848, 1492482944,
			1495699072, 1498911232, 1502119296, 1505323264, 1508523136, 1511718912, 1514910464, 1518097792, 1521280896, 1524459776, 1527634176, 1530804352, 1533970048, 1537131264, 1540288128, 1543440384,
			1546588160, 1549731200, 1552869760, 1556003584, 1559132800, 1562257280, 1565376896, 1568491776, 1571601792, 1574706944, 1577807232, 1580902528, 1583992832, 1587078144, 1590158336, 1593233536,
			1596303488, 1599368320, 1602427904, 1605482368, 1608531456, 1611575296, 1614613760, 1617646848, 1620674432, 1623696640, 1626713472, 1629724672, 1632730240, 1635730304, 1638724736, 1641713408,
			1644696448, 1647673728, 1650645248, 1653611008, 1656570880, 1659524864, 1662472960, 1665415040, 1668351232, 1671281280, 1674205440, 1677123328, 1680035200, 1682940928, 1685840384, 1688733696,
			1691620608, 1694501376, 1697375744, 1700243712, 1703105280, 1705960448, 1708809216, 1711651328, 1714486912, 1717315968, 1720138496, 1722954240, 1725763456, 1728565888, 1731361536, 1734150400,
			1736932480, 1739707776, 1742476160, 1745237504, 1747992064, 1750739584, 1753480064, 1756213504, 1758939904, 1761659136, 1764371328, 1767076224, 1769773952, 1772464384, 1775147648, 1777823488,
			1780492032, 1783153152, 1785806976, 1788453248, 1791092096, 1793723392, 1796347136, 1798963456, 1801571968, 1804172928, 1806766336, 1809351936, 1811929728, 1814499840, 1817062272, 1819616768,
			1822163328, 1824702080, 1827232896, 1829755776, 1832270720, 1834777600, 1837276416, 1839767168, 1842249728, 1844724352, 1847190656, 1849648768, 1852098688, 1854540288, 1856973696, 1859398784,
			1861815424, 1864223744, 1866623616, 1869015168, 1871398144, 1873772544, 1876138624, 1878496000, 1880844800, 1883185024, 1885516544, 1887839360, 1890153600, 1892459008, 1894755712, 1897043584,
			1899322624, 1901592832, 1903854208, 1906106624, 1908350080, 1910584704, 1912810240, 1915026688, 1917234176, 1919432576, 1921621888, 1923802112, 1925973120, 1928135040, 1930287616, 1932430976,
			1934565120, 1936689920, 1938805504, 1940911616, 1943008512, 1945095936, 1947173888, 1949242368, 1951301504, 1953351040, 1955391104, 1957421568, 1959442432, 1961453696, 1963455360, 1965447424,
			1967429632, 1969402240, 1971365120, 1973318272, 1975261696, 1977195136, 1979118848, 1981032704, 1982936704, 1984830848, 1986715008, 1988589184, 1990453376, 1992307584, 1994151808, 1995985920,
			1997810048, 1999623936, 2001427840, 2003221632, 2005005184, 2006778624, 2008541696, 2010294656, 2012037376, 2013769856, 2015491968, 2017203712, 2018905216, 2020596352, 2022277120, 2023947392,
			2025607296, 2027256704, 2028895744, 2030524288, 2032142208, 2033749632, 2035346560, 2036932864, 2038508672, 2040073728, 2041628288, 2043172096, 2044705280, 2046227712, 2047739392, 2049240448,
			2050730752, 2052210304, 2053678976, 2055136896, 2056583936, 2058020224, 2059445632, 2060860160, 2062263680, 2063656320, 2065038080, 2066408960, 2067768704, 2069117568, 2070455424, 2071782144,
			2073097856, 2074402560, 2075696256, 2076978816, 2078250240, 2079510528, 2080759680, 2081997696, 2083224448, 2084440192, 2085644544, 2086837760, 2088019840, 2089190528, 2090349952, 2091498240,
			2092635136, 2093760640, 2094874880, 2095977856, 2097069440, 2098149632, 2099218432, 2100275840, 2101321856, 2102356480, 2103379584, 2104391296, 2105391616, 2106380288, 2107357696, 2108323456,
			2109277696, 2110220416, 2111151616, 2112071296, 2112979328, 2113875840, 2114760832, 2115634176, 2116495872, 2117346048, 2118184448, 2119011328, 2119826560, 2120630016, 2121421824, 2122202112,
			2122970496, 2123727360, 2124472320, 2125205632, 2125927296, 2126637184, 2127335296, 2128021632, 2128696192, 2129358976, 2130009984, 2130649216, 2131276672, 2131892224, 2132496000, 2133088000,
			2133668096, 2134236416, 2134792832, 2135337344, 2135870080, 2136390912, 2136899968, 2137396992, 2137882112, 2138355456, 2138816896, 2139266304, 2139703936, 2140129536, 2140543232, 2140945024,
			2141334784, 2141712768, 2142078720, 2142432640, 2142774784, 2143104768, 2143422976, 2143729024, 2144023296, 2144305408, 2144575616, 2144833920, 2145080064, 2145314304, 2145536640, 2145746816,
			2145945088, 2146131328, 2146305664, 2146467840, 2146618112, 2146756352, 2146882560, 2146996736, 2147098880, 2147189120, 2147267200, 2147333376, 2147387392, 2147429504, 2147459584, 2147477632, 2147483647 };
