''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This is a demonstration server program running on the Micromite and
' using the ESP8266 module to serve up a WEB page.  Geoff Graham, Nov 2014
'
' This server will run on all versions of the Micromite firmware however
' support for the DHT22 temperature/humidity sensor is only present in
' version 4.6 of MMBasic for the Micromite MkII.
' For details on the Micromite go to:  http://geoffg.net/micromite.html
''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''

' The following four constants should be edited to reflect your requirements
SSID$ = "SSID"            ' Replace SSID with the name of your WiFi network
Passwd$ = "passwd"        ' Replace passwd with the password to your network
Timeout = 20000           ' Error timeout in milliseconds
echo = 0                  ' Set this to non zero to see all communications

' Define an array to record the server activity (ie, statistics)
StatSize = 40
DIM Stats$(StatSize)
shead = 0

' Some housekeeping - autorun on, watchdog on and setup the reset output
OPTION autorun on
WATCHDOG 60 * 1000
PIN(2) = 0
SETPIN 2, dout

PAUSE 1000    ' Let everything settle
VAR RESTORE   ' Restore the connection count (for the statistics)
RTC GETTIME   ' Get the current time

' Setup the automatic timer events
' Timer one will get the time from the RTC every four hours
' Timer two will get the temperature and humidity every two minutes
' Timer three will save the connection count once a day (in case
' of power failure)
SETTICK 4 * 60 * 60 * 1000, GetRtcTime, 1
SETTICK 2 * 60 * 1000, GetDHT22, 2
SETTICK 24 * 60 * 60 * 1000, SaveVars, 3

' Open communications with the ESP8266 module
OPEN "COM1:115200, 1024" AS #1


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This is the restart point.  If the ESP8266 module has crashed the
' timeout will cause us to jump back to here
ReTryWEB:
WATCHDOG 240 * 1000  ' Restart the BASIC program in the event of a crash
error = 0            ' Clear the error flag

' Close the serial port so that it will go to high impedance.  This will
' protect the Rx input to the ESP8266 when it is powered down.
' Then power down the module for one second and re open the serial port
CLOSE #1
PIN(2) = 0 : PAUSE 1000 : PIN(2) = 1 : PAUSE 1000
OPEN "COM1:115200, 1024" AS #1

' Record this event in the log and display it on the console
Stats$(shead) = "Starting the server at " + DATE$ + " " + TIME$
PRINT : PRINT Stats$(shead)
shead = (shead + 1) MOD StatSize

' Check that the ESP8266 module is alive
SendCmd "AT", "OK"
IF error GOTO ReTryWEB

' set the module to client and AP mode and reset it
PRINT #1, "AT+CWMODE=3"
PAUSE 50
SendCmd "AT+RST", "ready"

' Log into the network and setup the server function
PRINT "Found the ESP8266.  Firmware version " + GetVer$()
SendCmd "AT+CWJAP=" + CHR$(34) + SSID$ + CHR$(34) + "," + CHR$(34) + Passwd$ + CHR$(34), "OK"
PRINT "Connected to " + SSID$ + ". IP Address " + GetAddr$()
SendCmd "AT+CIPMUX=1", "OK"
SendCmd "AT+CIPSERVER=1,80", "OK"
IF error GOTO ReTryWEB

PRINT "WEB server started"
s$ = ""
TIMER = 0

' This is the main server loop.
' The program looks for someone connectiong to the server (a line from the
' module starting with +IPD).  It will then get the channel number (in case
' of multiple connections) then send the web page.
'
' When finished it will close the channel and check is there is another
' connection and if there is it will loop to satisify that request.
' If there is no activity for five minutes the program will check that the
' module is alive by sending an AT command.
DO
  ' The watchdog is used to detect if the program has crashed
  WATCHDOG 240 * 1000
  ' Accumulate characters in s$ until a carriage return is received
  ' This includes resetting the timer which is used to detect no activity
  ' and also trimming the lenght of the input if it is too long.
  a$ = INPUT$(1, #1)
  IF a$ <> "" THEN
    TIMER = 0
    IF echo THEN PRINT a$;
  ENDIF
  IF LEN(s$) >= 254 THEN s$ = RIGHT$(s$, 125)
  IF ASC(a$) >= 32 THEN s$ = s$ + a$

  IF ASC(a$) = 13 THEN
    ' We have a carriage return.  Check to see if it is a request for a WEB page
    IF LEFT$(s$, 5) = "+IPD," AND (INSTR(s$, "GET / ") > 1 OR INSTR(s$, "GET /stats ") > 1) THEN
       SendStats = INSTR(s$, "GET /stats ")   ' SendStats is a flag to send the stats only
       channel$ = MID$(s$, 6, 1)              ' Get the channel number for this connection
       PAUSE 10                               ' Let the rest of the data arrive
       DO
         s$ = GetLine$()
         IF error THEN GOTO ReTryWEB
       LOOP UNTIL s$ = "OK"                   ' OK means the end of the messages from the module

       ' We can loop back to here if another connection comes in while we are sending the page
       DO
         ' Record the connection in the statistics and print on the console
         count = count + 1                    ' Count is the number of connections
         Stats$(shead) = DATE$ + " " + TIME$ + " Connection number " + STR$(count) + " from " + GetClientIP$(channel$)
         IF SendStats THEN Stats$(shead) = Stats$(shead) + " for stats"
         PRINT Stats$(shead)
         shead = (shead + 1) MOD StatSize

         ' Send the data requested.
         ' Which can be the statistics (garden.geoffg.net/stats) or the normal page
         IF SendStats THEN
            ServeStats
         ELSE
            ServePage
         ENDIF

         ' close the connection
         SendCmd "AT+CIPCLOSE=" + channel$, "OK"
         IF error THEN GOTO ReTryWEB

         ' Use the "get status" command to see if there is another connection
         ' and retreive its channel number
         channel$ = ""
         ClearBuffer
         PRINT #1, "AT+CIPSTATUS"
         DO
           s$ = GetLine$()
           IF error THEN GOTO ReTryWEB
           IF echo THEN PRINT s$
           IF INSTR(s$, "+CIPSTATUS:") = 1 THEN
            channel$ = MID$(s$, 12, 1)
             ClearBuffer
             s$ = "OK"
           ENDIF
         LOOP UNTIL s$ = "OK"
       ' keep looping while we have a connection to process
       LOOP WHILE channel$ <> ""
    ENDIF

  s$ = ""
  ENDIF

  ' Every five minutes with no activity send an AT command
  ' and check that the module responds with OK
  IF TIMER > 5 * 60 * 1000 THEN
    SendCmd "AT", "OK"
    IF error THEN GOTO ReTryWEB
    ClearBuffer
    TIMER = 0
  ENDIF

LOOP     ' Keep looking for connections to the server
END


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub is called every four hours and is used to correct the Micromite's
' internal clock
SUB GetRtcTime
  RTC GETTIME
END SUB


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub gets the temperature and humidity.
' Support for the DHT22 is new in MMBasic V4.6 for the Micromite MkII
SUB GetDHT22
  DHT22 16, temp, humid
END SUB


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub is called every 24 hours and saves the connection count so that
' we can restore it in case of a power failure
SUB SaveVars
  VAR SAVE count
END SUB



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub sends the main WEB page
SUB ServePage
  ServeLine "<TITLE>Micromite WiFi Server</TITLE>"
  ServeLine "<H3>Welcome to the Micromite Garden Webserver</H3><BR>"
  ServeLine "You are visitor number: " + STR$(count) + "<BR><BR>"
  ServeLine "The Micromite time is " + TIME$
  ServeLine " and the date is " + DATE$ + "<BR><BR>"
  ServeLine "The temperature in the garden is " + STR$(temp) + "&deg;C"
  ServeLine " and the humidity is " + STR$(humid) + "%<BR><BR>"
  ServeLine "The MMBasic program running the server can be downloaded from "
  ServeLine "<a href=" + CHR$(34) + "http://geoffg.net/Downloads/ServerSrc.zip" + CHR$(34) + ">here</a>.<BR>"
END SUB



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub sends the statistics WEB page
SUB ServeStats
  LOCAL i
  ServeLine "<TITLE>WiFi Server Statistics</TITLE>"
  ServeLine "Number of connections: " + STR$(count) + "<BR><BR>"
  i = shead
  DO
    IF Stats$(i) <> "" THEN ServeLine Stats$(i) + "<BR>"
    i = (i + 1) MOD StatSize
  LOOP WHILE i <> shead
END SUB


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This sub sends a single line of the WEB page
' It uses the global channel$ to identify the connection
SUB ServeLine lin$
  IF error THEN EXIT SUB
  PRINT #1, "AT+CIPSEND=" + channel$ + "," + STR$(LEN(lin$)+2)
  PAUSE 10
  SendCmd lin$, "SEND OK"
END SUB



''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This is a general purpose sub which will send a command and check
' for the correct response
SUB SendCmd cmd$, response$
  LOCAL InputStr$
  IF error THEN EXIT SUB
  ClearBuffer
  PRINT #1, cmd$
  DO
    InputStr$ = GetLine$()
  LOOP UNTIL INSTR(InputStr$, response$) > 0 OR error
END SUB


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This will get the version number of the firmware running on the module
FUNCTION GetVer$()
  LOCAL s$, i
  IF error THEN EXIT SUB
  ClearBuffer
  PRINT #1, "AT+GMR"
  FOR i = 1 TO 3
    s$ = GetLine$()
    IF error THEN EXIT FOR
    IF echo THEN PRINT s$
  NEXT i
  GetVer$ = s$
END FUNCTION


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This will get the IP address of a specific connection
' c$ is an ASCII character representing the channel number
FUNCTION GetClientIP$(c$)
  LOCAL s$
  IF error THEN EXIT SUB
  GetClientIP$ = "unknown"
  ClearBuffer
  PRINT #1, "AT+CIPSTATUS"
  DO
    s$ = GetLine$()
    IF error THEN EXIT DO
    IF echo THEN PRINT s$
    IF INSTR(s$, "+CIPSTATUS:" + c$) = 1 THEN
      s$ = MID$(s$, INSTR(s$, ",") + 1)
      s$ = MID$(s$, INSTR(s$, ",") + 2)
      GetClientIP$ = LEFT$(s$, INSTR(s$, ",") - 2)
    ENDIF
  LOOP UNTIL s$ = "OK"
END FUNCTION


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This will get the IP address of the module
' After connecting to the access point it can take several seconds for the
' command to complete.  This sub will keep trying to get the address and will
' only return when it has it (this signifies that the connection is made).
FUNCTION GetAddr$()
  LOCAL s$, i, j
  IF error THEN EXIT SUB
  FOR i = 1 TO 20
    ClearBuffer
    PRINT #1, "AT+CIFSR"
    FOR j = 1 TO 3
      s$ = GetLine$()
      IF error THEN EXIT FOR
      IF echo THEN PRINT s$
    NEXT j
    IF LEN(s$) > 5 OR error THEN EXIT FOR
    PAUSE 1000
  NEXT i
  GetAddr$ = s$
END FUNCTION


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' clear the serial input buffer of any junk
SUB ClearBuffer
  DO : LOOP UNTIL INPUT$(255, #1) = ""
END SUB


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' Get a line of text from the module
' This function can work with lines that are longer than the Micromite's
' maximum string length of 255 characters (unlike LINE INPUT)
FUNCTION GetLine$()
  LOCAL s$, c$
  TIMER = 0
  DO
    c$ = INPUT$(1, #1)
    IF echo THEN PRINT c$;
    IF ASC(c$) = 13 OR LEN(s$) >= 254 THEN
      GetLine$ = s$
    ELSE
      IF ASC(c$) >= 32 THEN s$ = s$ + c$
    ENDIF
    IF TIMER > Timeout THEN error = 1
  LOOP UNTIL ASC(c$) = 13 OR LEN(s$) >= 254 OR error
END FUNCTION


''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''''
' This function is a simple terminal for testing commands on the module.
' It can be run at the MMBasic command prompt just by entering its name.
' For example:
' > Terminal
SUB Terminal
  LOCAL c$
  OPEN "Com1:115200, 1024" AS #1
  DO
    PRINT INPUT$(1, #1);
    PRINT #1, INKEY$;
  LOOP
END SUB