/* ****************************************************************************
 * File:      lcd_X30654.h
 * Date:      13-05-2023
 * Author:    Andrew Levido
 *
 * Copyright 2025 Imbrius Pty Ltd - All Rights Reserved
 *
 *****************************************************************************/
 #ifndef _lcd_X30654_H
 #define _lcd_X30654_H
 
 #ifdef __cplusplus
 extern "C" {
 #endif

/* Description --------------------------------------------------------------*/

/* Includes ----------------------------------------------------------------*/
#include "main.h"

/* Defines -----------------------------------------------------------------*/
#define OLED
#define OLED_I2C_DEVADDR              0x3c

/* Typedefs & Enumerations -------------------------------------------------*/
typedef enum { OLED_OK, OLED_ERR_COMMS } oled_result_t;
typedef enum { OLED_NORMAL, OLED_FLIP } oled_orientation_t;

/* Public Functions Declarations -------------------------------------------*/
/** Initialise LCD Driver
 * 
 * @param oreint  Enumerated orientation specifier
 * @returns       Enumerated result code
*/
oled_result_t oled_init(oled_orientation_t orient);

/** Flip OLED orientation
 * 
 * @param oreint  Enumerated orientation specifier
 * @returns       Enumerated result code
*/
oled_result_t oled_flip(oled_orientation_t orient);


/** Write display buffer to display 
 * 
 * @returns       Enumerated result code
*/
oled_result_t oled_update(void);

/** Write one pixel ti the display buffer. (use  oled_update() to transfer 
 *  buffer to display)
 * 
 * @param x0      x-coord of pixel
 * @param y0      y-coord of pixel
 * @param colour  0 for off, non-zero for on
*/
void oled_put_pixel(uint8_t x0, uint8_t y0, uint8_t value);

/** Write a rectangular block of pixel data to the display buffer (use 
 *  oled_update() to transfer buffer to display)
 * 
 * The block can be any width but must be an integral number of pages long. 
 * The data in the buffer must be a byte array in [width][page]. Cropped outside
 * the display area.
 * 
 * @param x0      x-coord of top left pixel
 * @param page    starting page
 * @param width   x-width in pixels
 * @param height  y-height in pages
 * @param buf     buffer containing the array of pixels
*/
void oled_put_buffer(uint8_t x0, uint8_t page, uint8_t width, uint8_t height, uint8_t * buf);

/** Clear display buffer (use oled_update() to transfer buffer to display) 
*/
void oled_clear_buffer(void);

void oled_off(void);

void oled_on(void); 

/** Clear a rectangular block of pixel data to the display buffer (use 
 *  oled_update() to transfer buffer to display)
 * 
 * The block can be any width but must be an integral number of pages long. 
 * 
 * @param x0      x-coord of top left pixel
 * @param page    starting page
 * @param width   x-width in pixels
 * @param height  y-height in pages
*/
void oled_clear(uint8_t x0, uint8_t page, uint8_t width, uint8_t height);

/* Interrupt Service Routine Declarations ----------------------------------*/

#ifdef __cplusplus
}
#endif

#endif
/* End lcd_X30654.h */