/* ****************************************************************************
 * File:      smartprobe.h
 * Date:      26 Dec 2024
 * Author:    Andrew Levido
 *
 * Copyright 2025 Imbrius Pty Ltd - All Rights Reserved
 *
 *****************************************************************************/
#ifndef _smartprobe_H
#define _smartprobe_H

#ifdef __cplusplus
extern "C" {
#endif

/* Description -----------------------------------------------------------------
 *
 * Header file for the core SmartProbe application. Contains:
 * 
 *  #defines for screen element positions and sizes 
 *  #defines for system constants
 *  function declarations for external functions 
 * 
 *  On-screen elements defined in resources.c as pixel blocks aligned vertically
 *  to OLED pages and horizontally to OLED pixels. 
 */

/* Includes ------------------------------------------------------------------*/
#include "main.h"

/* Defines -------------------------------------------------------------------*/

/* Display 
        0     16    32    48    64    80    96    112    128
        +--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+--+
 page 0 |        [#### Voltage, Continuity ####]        |  Mode Display
      1 |        [#############################]        |
      2 |                                               |
      3 |   [     ]   [     ] [     ] [     ] [     ]   |
      4 |[+][  4  ]   [  3  ] [  2  ] [  1  ] [  V  ]   |  Voltage Display
      5 |   [     ] * [     ] [     ] [     ] [     ]   |
      6 |         [---- Overvoltage Error -----]        |
      7 |[Fatal]  [----------------------------]        |  Error Display
        +-----------------------------------------------+     

      - The voltage display digit 3 and decimal point shift in the 
        low resolution display mode 00.000
      - Error displays are normally off
      - The Voltage display is replaced with "OPEN" in continuity mode if 
        
  */

/* Screen Element Positions & Sizes ------------------------------------------*/
/* Mode Display */
#define SP_MODE_X0              24    // x-position of mode display icon
#define SP_MODE_PAGE            0     // page of mode display icon
#define SP_MODE_WIDTH           80    // width of mode display icon (px)
#define SP_MODE_HEIGHT          2     // height of mode display icon (pages)

/* Voltage Display */
#define SP_VOLTAGE_DISP_WIDTH   128   // width of whole display area
#define SP_DIGIT4_X0            12    // x-position of digit 4 (most sig)
#define SP_DIGIT3_LO_X0         40    // x-position of digit 3 (6.000)
#define SP_DIGIT3_HI_X0         34    // x-position of digit 3 (60.00)
#define SP_DIGIT2_X0            62    // x-position of digit 2 (
#define SP_DIGIT1_X0            84    // x-position of digit 1 (least sig)
#define SP_UNITS_X0             112   // x-position of units
#define SP_DIGIT_PAGE           3     // page of digits
#define SP_DIGIT_WIDTH          16    // width of digits (px)
#define SP_DIGIT_HEIGHT         3     // height of digits (pages)
#define SP_DP_HI_X0             52    // x-position of decimal (60.00)
#define SP_DP_LO_X0             30    // x-position of decimal (6.000)
#define SP_DP_PAGE               5    // page of decimal
#define SP_DP_WIDTH              8    // width of decimal (px)
#define SP_DP_HEIGHT             1    // height of decimal (pages)
#define SP_SIGN_X0               0    // x-position of sign
#define SP_SIGN_PAGE             4    // page of sign
#define SP_SIGN_WIDTH            8    // width of decimal (px)
#define SP_SIGN_HEIGHT           1    // height of decimal (pages)

/* Open Circuit Display */
#define SP_OPEN_X0              24    // x-position of open display icon
#define SP_OPEN_PAGE             3    // page of open display icon
#define SP_OPEN_WIDTH           80    // width of open display icon (px)
#define SP_OPEN_HEIGHT           2    // height of open display icon (pages)

/* Overvoltage Error Display */
#define SP_OVERV_X0             24    // x-position of overvolt icon
#define SP_OVERV_PAGE           6     // page of overvolt icon
#define SP_OVERV_WIDTH          80    // width of overvolt icon (px)
#define SP_OVERV_HEIGHT         2     // height of overvolt icon (pages)

/* Fatal Error Display */
#define SP_ERR_X0_OFFS          0     // x-position of error message
#define SP_ERR_PAGE             7     // page of error message
#define SP_ERR_WIDTH            16    // width of error message (px)
#define SP_ERR_HEIGHT           1     // height of error message (pages)


/* System Constants ----------------------------------------------------------*/
/* Overvoltage Threshold */
#define SP_MAX_VOLTAGE        50000   // ± threshold in millivolts 

/* Coninuity Thresholds */ 
#define SP_OPEN_THRESHOLD     3200    // threshold in millivolts 

/* Display Resolution threshold - the display switches to high range at either
   ±9.999V or this (lower) threshold */
#define SP_HI_V_RANGE_THRESH       6000    // ± threshold in millivolts (<9999)

/* Analog Circuit Gain Factors */
#define SP_HIGH_GAIN         402157   // (1 + 2M/51K) x 10000
#define SP_LOW_GAIN           39412   // (1 + 2M/680K) x 10000

/* Fatal Error Codes */
#define SP_ERR_SYS              0     // System exception or HAL error
#define SP_ERR_OLED             1     // OLED display error
#define SP_ERR_ACC              2     // Accelerometer driver error
#define SP_ERR_SPACEY           3     // Spare code for testing

/* Public Functions Declarations ---------------------------------------------*/

/** SmartProbe application.
 * Main application function. Called once from main() after configuring the 
 * peripherals. Never returns.
 */
void sp_main_loop(void);

/** Display Fatal Error Icon 
 * Displays an error icon on the display and then loops forever
 * 
 * @param code      One of the defined Fatal Error Codes
 */
void sp_fatal_error(uint8_t code);

/** Check continuity comparator. Called every microsecond from the system
 * tick ISR. If the comparator detects a low voltage on the input pin,
 * it beeps and sets the continuity beep flag.
 */
void sp_comp_1ms_tick(void);

#ifdef __cplusplus
}
#endif

#endif

/* End smartprobe */