//all the defines and helpers in util.h
#include "util.h"

void setup1(){
  audioInit(DOUT);
  audioSetRate(OUTSAMPLERATE);
}

void loop1(){ 
  //do nothing except handle interrupts
}

void setup() {
  int i;
  Serial.begin(115200);
  delay(500);
  EEPROM.begin(4096);
  EEPROM.get(0,cur);
  if(dataInvalid(&cur)){
    setDefaults(&cur);
    Serial.println("Defaults Loaded");
  }  
  vCal=cur.vCalLo;
  Wire.setSDA(20);
  Wire.setSCL(21);
  OLED.begin(0x3c, true); // Address 0x3C default
  OLED.setTextSize(1);
  OLED.setTextColor(SH110X_WHITE);
  OLED.setCursor(0, 0);
  OLED.clearDisplay();
  OLED.drawBitmap(0,0,splashBitmap,128,64,SH110X_WHITE);
  OLED.display();
  pinMode(S1,INPUT_PULLUP);
  pinMode(S2,INPUT_PULLUP);
  pinMode(S3,INPUT_PULLUP);
  pinMode(S4,INPUT_PULLUP);  
  buttonSetup();
  wavePtr[0]=Sinewave;
  wavePtr[1]=TriWave;
  wavePtr[2]=SquareWave;
  wavePtr[3]=SawWave;
  wavePtr[4]=WhiteWave;
  waveIndex=0;
  pinMode(23,OUTPUT);
  digitalWrite(23,HIGH);  //force PWM mode for lower ripple  
  Serial.println("Start");
	//audioInit(DOUT);      //this happens on the other core
  //audioSetRate(OUTSAMPLERATE);
  newAmp=(cur.wavCal*pkToRMS[waveMode]*newV)/1000;
  outSampSize=writeWave(newF,0,outSamp[0],newWave); //start with 0 amplitude
  if(audioSpace()){
    audioQueue(outSamp[0],outSampSize);
    audioPlay(AUDIO_LOOP_16);
  }
  dmaSampInit();
  delay(3000);    //allow biases to settle
  while(getButton(S1)){}  //clear any queued button presses
  while(getButton(S2)){}
  while(getButton(S3)){}
  while(getButton(S4)){}
}

void loop() {
  int i;
  unsigned int t;
  if(inMult==10){       //keep this updated
    vCal=cur.vCalHi;
  }else{
    vCal=cur.vCalLo;
  }
  if(getButton(S4)){
    while(getButton(S4)){}
    delay(50);
    mode=(screenmode_t)(mode+1);
    if(mode==DUMMY){mode=(screenmode_t)(0);}
  }
  switch(mode){
    case WAVE_OUT:doWaveScreen();     break;
    case SPECTRUM:doSpecScreen();     break;
    case SCOPE:   doScopeScreen();    break;
    case HARMONIC:doHarmonicScreen(); break;
    case SWEEP:   doSweepScreen();    break;
    case SETTINGS:doSettingsScreen(); break;
  }
  if((refRunning) && (mode!=SETTINGS)){   //reset so it can trigger, turn off reference
    refRunning=0;
    setMode=SET_START;  
    setWave(newF,0);
  }
  if((sweepPhase) && (mode!=SWEEP)){    //stop any active output from a running sweep if mode changes
    sweepPhase=0;
    setWave(newF,0);
  }
  s=scanSerial();
  if(s){
    if(s[0]=='~'){
      rp2040.reboot();
    }else if((s[0]=='w')||(s[0]=='W')){
      if((s[1]>='1')&&(s[1]<='5')){
        waveIndex=s[1]-'1';
        waveMode=(wavemode_t)(waveIndex);
        newWave=wavePtr[waveIndex];
        setWave(newF, newAmp);
        Serial.printf("F=%dHz,Amp=%dmV,%s\r\n",newF,newV,waveModes[waveIndex]);
      }
    }else if((s[0]=='a')||(s[0]=='A')){
      newV=atoi(&s[1]);
      if(newV<0){newV=0;}
      if(newV>2000){newV=2000;}
      newPkPk=2*pkToRMS[waveMode]*newV;
      newAmp=(cur.wavCal*pkToRMS[waveIndex]*newV)/1000;    
      setWave(newF, newAmp);
      Serial.printf("F=%dHz,Amp=%dmV,%s\r\n",newF,newV,waveModes[waveIndex]);
    }else if((s[0]=='p')||(s[0]=='P')){
      newV=atoi(&s[1])/(2*pkToRMS[waveMode]);
      if(newV<0){newV=0;}
      if(newV>2000){newV=2000;}
      newPkPk=2*pkToRMS[waveMode]*newV;
      newAmp=(cur.wavCal*pkToRMS[waveIndex]*newV)/1000;    
      setWave(newF, newAmp);
      Serial.printf("F=%dHz,Amp=%dmV,%s\r\n",newF,newV,waveModes[waveIndex]);
    }else if((s[0]=='f')||(s[0]=='F')){
      newF=atoi(&s[1]);
      if(newF<5){newF=5;}
      if(newF>30000){newF=30000;}
      setWave(newF, newAmp);
      Serial.printf("F=%dHz,Amp=%dmV,%s\r\n",newF,newV,waveModes[waveIndex]);
    }else if((s[0]=='o')||(s[0]=='O')){
      if(s[1]=='1'){
        outputOn=1;
        setWave(newF, newAmp);
        Serial.printf("F=%dHz,Amp=%dmV,%s\r\n",newF,newV,waveModes[waveIndex]);
      }else{
        outputOn=0;
        setWave(5, 0);
        Serial.println("Wave output off.");
      }
    }else if((s[0]=='d')||(s[0]=='D')){
      dumpFlag=1;
    }else if((s[0]=='h')||(s[0]=='H')){
      histFlag=1;
    }else if(s[0]=='?'){
      Serial.println("Commands (followed by Enter):");
      Serial.println("A:  set RMS amplitude in mV (eg a1000)");
      Serial.println("P:  set peak-to-peak amplitude in mV (eg p2800)");
      Serial.println("F:  set frequency in Hz (eg f440)");
      Serial.println("W1: output sinewave");
      Serial.println("W2: output triangle wave");
      Serial.println("W3: output square wave");
      Serial.println("W4: output sawtooth wave");
      Serial.println("W5: output white noise");
      Serial.println("O0: output off");
      Serial.println("O1: output on");
      Serial.println("D:  dump data from Scope, Spectrum, Harmonic or Sweep modes");
      Serial.println(",< and .> emulate DOWN and UP");
      Serial.println(" /  and M emulate OK and MODE");
    }else{

    }
  }
}

void doSettingsScreen(){
  dumpFlag=0;   //nothing to dump in this mode
  if((setMode>SET_START)&&(refRunning==0)){
    refRunning=1;
    newWave=wavePtr[0];   //[0] is sine
    waveIndex=0;
    newAmp=(cur.wavCal*pkToRMS[0]*REF_LEVEL)/1000;    
    newF=REF_FREQ;
    setWave(newF, newAmp);
    delay(50);    //settle
  }
  while(getButton(S3)){   //while to handle repeats
    setMode=(settingmode_t)(setMode+1);
    if(setMode>=SET_DUMMY){setMode=SET_START;}
  }
  OLED.clearDisplay();
  OLED.setCursor(0, 0);
  OLED.println("SETTINGS");
  switch(setMode){
    case SET_START:
      setStart();
      break;
    case SET_INOFFSET:
      setInOffset();
      break;
    case SET_OUTAMP:
      setOutAmp();
      break;
    case SET_INAMPLO:
      setInAmpLo();
      break;
    case SET_INAMPHI:
      setInAmpHi();
      break;
    case SET_SAVE:
      setSave();
      break;
    default:
      setStart();
      break;
  }
  OLED.display();
}

void setStart(){
  OLED.setCursor(70, 0);
  OLED.printf("BAT:%4.1fV",getADC(BATSENSE)*BAT_RATIO);  
  OLED.setCursor(0, 12);
  OLED.print("PRESS OK TO CONTINUE.");
  OLED.setCursor(0, 24);
  OLED.print("Press DOWN for x1");
  OLED.setCursor(0, 36);
  OLED.print("Press UP for x10");
  while(getButton(S1)){
    inMult=1;
    vCal=cur.vCalLo;
  }
  while(getButton(S2)){
    inMult=10;
    vCal=cur.vCalHi;
  }
  OLED.setCursor(0, 48);
  OLED.printf("Set to x%d.",inMult);
}

void setInOffset(){
  int m;
  dmaSampRun(AIN,samp,SAMPSIZE);
  m=getMean();
  OLED.setCursor(0, 12);
  OLED.print("INPUT OFFSET");
  OLED.setCursor(0, 24);
  OLED.print("Leave 'IN' open.");
  OLED.setCursor(0, 36);
  OLED.printf("DOWN to set to %4d.",m);
  OLED.setCursor(0, 48);
  OLED.printf("Current value is %4d",cur.sampOffset);
  while(getButton(S1)){
    cur.sampOffset=m;
  }
  while(getButton(S2)){}
}

void setOutAmp(){
  int f=0;
  OLED.setCursor(0, 12);
  OLED.print("OUTPUT LEVEL");
  OLED.setCursor(0, 24);
  OLED.print("Measure 'OUT' RMS.");
  OLED.setCursor(0, 36);
  OLED.printf("Trim to %4dmV.",REF_LEVEL);
  OLED.setCursor(0, 48);
  OLED.printf("Ratio is %8.2f.",cur.wavCal);
  while(getButton(S1)){
    cur.wavCal=cur.wavCal/1.0001;
    f=1;
  }
  while(getButton(S2)){
    cur.wavCal=cur.wavCal*1.0001;
    f=1;
  }
  if(f){  //update with new
    newWave=wavePtr[0];   //[0] is sine
    waveIndex=0;
    newAmp=(cur.wavCal*pkToRMS[0]*REF_LEVEL)/1000;    
    newF=REF_FREQ;
    setWave(newF, newAmp);
  }
}

void setInAmpLo(){
  float m=0;    
  int i;
  for(i=0;i<4;i++){   //some modest oversampling
    dmaSampRun(AIN,samp,SAMPSIZE);
    m=m+getRMS();
  }
  m=m/4;
  OLED.setCursor(0, 12);
  OLED.print("INPUT LEVEL x1 (SW!)");
  OLED.setCursor(0, 24);
  OLED.print("Connect 'OUT' to 'IN'");  
  OLED.setCursor(0, 36);
  if(m>200){    //validate and also dodge /0, about 560 is expected here
    OLED.printf("DOWN to set %8f.",REF_LEVEL/(m*1000));
  }else{
    OLED.print("No signal error (SW?)");
  }
  OLED.setCursor(0, 48);
  OLED.printf("Current is %8f.",cur.vCalLo);
  while(getButton(S1)){
    if(m>200){
      cur.vCalLo=REF_LEVEL/(m*1000);
    }
  }
  while(getButton(S2)){}
}

void setInAmpHi(){
  float m=0;    
  int i;
  for(i=0;i<4;i++){   //some modest oversampling
    dmaSampRun(AIN,samp,SAMPSIZE);
    m=m+getRMS();
  }
  m=m/4;
  OLED.setCursor(0, 12);
  OLED.print("INPUT LEVEL x10 (SW!)");
  OLED.setCursor(0, 24);
  OLED.print("Connect 'OUT' to 'IN'");  
  OLED.setCursor(0, 36);
  if(m>20){    //validate and also dodge /0 about 56 is expected here
    if(m<100){
      OLED.printf("DOWN to set %8f.",REF_LEVEL/(m*1000));
    }else{
      OLED.print("Check Switch.");  
    }
  }else{
    OLED.print("No signal error.");
  }
  OLED.setCursor(0, 48);
  OLED.printf("Current is %8f.",cur.vCalHi);
  while(getButton(S1)){
    if(m>20){
      cur.vCalHi=REF_LEVEL/(m*1000);
    }
  }
  while(getButton(S2)){}
}

void setSave(){
  static int lastAction=0;        //timer to flash up last action
  static unsigned long lastTime=0;
  if(lastAction){
    if((millis()-lastTime)>2000){
      lastAction=0;
    }
  }  
  OLED.setCursor(0, 12);
  OLED.print("SAVE");
  OLED.setCursor(0, 24);
  OLED.print("UP to load defaults.");  
  OLED.setCursor(0, 36);
  OLED.print("DOWN to save to flash");  
  while(getButton(S1)){   //down
    if(lastAction!=1){
      lastAction=1;
      lastTime=millis();
      EEPROM.put(0,cur);
      if(EEPROM.commit()==false){lastAction=3;}
      Serial.println("Saving");
    }
  }
  while(getButton(S2)){   //up
    if(lastAction!=2){
      lastAction=2;
      lastTime=millis();
      setDefaults(&cur);
    }
  }
  OLED.setCursor(0, 48);
  switch(lastAction){
    case 1:OLED.print("Save Done."); break;
    case 2:OLED.print("Defaults loaded."); break;
    case 3:OLED.print("Save Failed."); break;
    default:break;    //blank
  }
}

void doWaveScreen(){
  static int p=0;
  int flag=0;   //change in output needed
  dumpFlag=0;   //nothing to dump in this mode
  switch(p){
    case 0:   //editing newF
      while(getButton(S1)){   //while to handle repeats
        flag=1;
        if(newF<1001){
          newF=newF-10;
          if(newF<10){newF=10;}
        }else if(newF<2001){
          newF=newF-50;
        }else{
          newF=newF-100;
        }
      }
      while(getButton(S2)){   //while to handle repeats
        flag=1;
        if(newF<999){
          newF=newF+10;
        }else if(newF<1999){
          newF=newF+50;
        }else{
          newF=newF+100;
          if(newF>10000){newF=10000;}
        }
      }
      while(getButton(S3)){   //while to handle repeats
        p=1;
      }
      break;
    case 1:   //editing newV
      while(getButton(S1)){   //while to handle repeats
        flag=1;
        newV=newV-WAVE_AMP_STEPS;
        if(newV<0){newV=0;}
        newPkPk=2*pkToRMS[waveMode]*newV;
      }
      while(getButton(S2)){   //while to handle repeats
        flag=1;
        newV=newV+WAVE_AMP_STEPS;
        if(newV>2000){newV=2000;}
        newPkPk=2*pkToRMS[waveMode]*newV;
      }
      while(getButton(S3)){   //while to handle repeats
        p=2;
      }
      break;
    case 2:   //editing newPkPk
      while(getButton(S1)){   //while to handle repeats
        flag=1;        
        newPkPk=roundf(newPkPk/WAVE_AMP_STEPS)*WAVE_AMP_STEPS-WAVE_AMP_STEPS;
        if(newPkPk<0){newPkPk=0;}
        if(pkToRMS[waveMode]>0.9){
          newV=newPkPk/(2*pkToRMS[waveMode]);
        }
      }
      while(getButton(S2)){   //while to handle repeats
        flag=1;
        newPkPk=roundf(newPkPk/WAVE_AMP_STEPS)*WAVE_AMP_STEPS+WAVE_AMP_STEPS;
        if(newPkPk>5000){newPkPk=5000;}
        if(pkToRMS[waveMode]>0.9){
          newV=newPkPk/(2*pkToRMS[waveMode]);
        }
      }
      while(getButton(S3)){   //while to handle repeats
        p=3;
      }
      break;
    case 3:   //editing waveMode
      if(getButton(S1)){
        waveMode=(wavemode_t)(waveMode-1);
        if(waveMode>=DUMMY_WAVE){waveMode=(wavemode_t)(DUMMY_WAVE-1);}
        newPkPk=2*pkToRMS[waveMode]*newV;
        flag=1;
      }
      if(getButton(S2)){
        waveMode=(wavemode_t)(waveMode+1);
        if(waveMode>=DUMMY_WAVE){waveMode=wavemode_t(0);}
        newPkPk=2*pkToRMS[waveMode]*newV;
        flag=1;
      }
      while(getButton(S3)){   //while to handle repeats
        p=4;
      }
      break;
    case 4:   //toggle on//off      
      if(getButton(S2)){
        outputOn=1;
        flag=1;
        while(getButton(S2)){}
      }
      if(getButton(S1)){    //last to allow off to dominate
        outputOn=0;
        flag=1;
        while(getButton(S1)){}
      }
      while(getButton(S3)){   //while to handle repeats
        p=0;
      }
      break;
    default: p=0;break;       //if we get lost
  }
  if(flag){
    flag=0;
    newWave=wavePtr[waveMode];
    waveIndex=waveMode;
    newAmp=(cur.wavCal*pkToRMS[waveMode]*newV)/1000;    
    while(newAmp>32000){   //if overflow happens
      newV=newV-50;    //retry with fixed value
      newPkPk=2*pkToRMS[waveMode]*newV;
      newAmp=(cur.wavCal*pkToRMS[waveMode]*newV)/1000;    
      flag=1;
    }   
    if(outputOn){
      setWave(newF, newAmp);
    }else{
      setWave(newF,0);
    }
  }
  OLED.clearDisplay();
  OLED.setCursor(0, 0);
  OLED.println("WAVE OUTPUT");
  OLED.setCursor(0, 10);
  if(p==0){
    OLED.printf("FREQ: <%4dHz>",newF);
  }else{
    OLED.printf("FREQ:  %4dHz",newF);
  }
  OLED.setCursor(0, 20);
  if(p==1){
    OLED.printf("AMP: <%4dmV RMS>",newV);
  }else{
    OLED.printf("AMP:  %4dmV RMS",newV);
  }
    OLED.setCursor(0, 30);
  if(p==2){
    OLED.printf("     <%4dmV pk-pk>",int(newPkPk));
  }else{
    OLED.printf("      %4dmV pk-pk",int(newPkPk));
  }
  OLED.setCursor(0, 40);
  if(p==3){
    OLED.printf("WAVE: <%s>",waveModes[waveMode]);
  }else{
    OLED.printf("WAVE:  %s",waveModes[waveMode]);
  }
  OLED.setCursor(0, 50);
  if(p==4){
    OLED.printf("OUTPUT:%s",outputOn?"<ON>  DOWN=OFF":"<OFF> UP=ON");
  }else{
    OLED.printf("OUTPUT: %s ",outputOn?"ON":"OFF");
  }  
  OLED.display();
}

void doScopeScreen(){
  int m,mn,mx,span,y0,y1;  //mean,min,max,span, y0,y1 for drawing lines
  int i=0,j=0;  
  float timeSpan;
  int pixelTime; //samples per pixel on display//=BIN_WIDTH*OVERSAMP;
  static int displaySpan=10000;   //in us
  static int vSpan=4000;         //mV, is auto-fitting, start with 2V pk-pk
  static int drawLines=0;     //slower,  but clearer at higher frequencies
  checkDmaSampRunAsync();
  switch(dmaAsyncState){
    case DMA_ASYNC_IDLE:
      dmaSampRunAsync(AIN,samp,SAMPSIZE);
      break;
    case DMA_ASYNC_RUNNING:
      break;    //do nothing, wait til done
    case DMA_ASYNC_DONE:
      if(getButton(S1)){
        displaySpan=displaySpan/2;
        if(displaySpan<625){displaySpan=625;}
      }
      if(getButton(S2)){
        displaySpan=displaySpan*2;
        if(displaySpan>40000){displaySpan=40000;}
      }
      if(getButton(S3)){
        while(getButton(S3)){}    //avoid repeats
        if(drawLines){
          drawLines=0;
        }else{
          drawLines=1;
        }
      }
      //no subsampling, use smoothed
      pixelTime=(RAWSAMPLERATE*displaySpan)/100000000; //this is in sample counts
      smoothSamp(pixelTime);
      m=getMean();
      mn=getMin();
      mx=getMax();  
      span=mx-mn;
      span=(span*1000)*vCal;  //ie in mV
      if(span<1){span=1;} //avoid divide by zero
      j=0;
      while((j<(SAMPSIZE-pixelTime*102))&&((samp[j]>m)||(samp[j+1]<m))){j++;} //scan until positive going zero cross     
      OLED.clearDisplay();
      OLED.setCursor(0, 0);
      OLED.print("SCOPE");      
      OLED.drawFastHLine(26,63,102,SH110X_WHITE);
      OLED.drawFastHLine(26,13,102,SH110X_WHITE);
      OLED.drawFastVLine(25,13,51,SH110X_WHITE);
      OLED.drawFastVLine(127,13,51,SH110X_WHITE);
      //dashed lines for intermediate
      for(i=0;i<50;i=i+2){
        OLED.drawPixel(51,13+i,SH110X_WHITE);
        OLED.drawPixel(76,13+i,SH110X_WHITE);
        OLED.drawPixel(101,13+i,SH110X_WHITE);
      }
      for(i=0;i<100;i=i+2){
        OLED.drawPixel(27+i,26,SH110X_WHITE);
        OLED.drawPixel(27+i,38,SH110X_WHITE);
        OLED.drawPixel(27+i,51,SH110X_WHITE);
      }
      OLED.setCursor(36,5);      
      OLED.printf("<---%5.2fms--->",displaySpan*0.001);
      //try to fit to nice number of volts
      if(span>vSpan){   //need to increase
        vSpan=vSpan*2;
        if(inMult==10){
          if(vSpan>32000){vSpan=32000;} //32V pk-pk max          
        }else{
          if(vSpan>4000){vSpan=4000;} //4V pk-pk max          
        }        
      }else if(span<(vSpan/2)){ //need to decrease  
        vSpan=vSpan/2;
        if(vSpan<1000){vSpan=1000;} //1V pk-pk min
      }
      if(vSpan>18000){
        OLED.setCursor(0,13);
        OLED.printf("%3.0fV",vSpan/2000.0);      
        OLED.setCursor(0,56);
        OLED.printf("-%3.0f",vSpan/2000.0);
      }else{
        OLED.setCursor(0,13);
        OLED.printf("%3.1fV",vSpan/2000.0);
        OLED.setCursor(0,56);
        OLED.printf("-%3.1f",vSpan/2000.0);
      }
      OLED.setCursor(0,28);
      OLED.print("pkpk");
      OLED.setCursor(0,41);
      if(span>9000){
        OLED.printf("%4.1f",span/1000.0);
      }else{
        OLED.printf("%4.2f",span/1000.0);
      }      
      if(drawLines){
        for(i=0;i<102;i++){
          y1=38+(((m-samp[j])*50000*vCal)/(vSpan));       //=50pixels x 1000mV             
          if(i){OLED.drawLine(i+24,y0,i+25,y1,SH110X_WHITE);}          
          j=j+pixelTime;
          y0=y1;          
        }
      }else{
        for(i=0;i<102;i++){
          y1=38+(((m-samp[j])*50000*vCal)/(vSpan));       //=50pixels x 1000mV             
          OLED.drawPixel(i+25,y1,SH110X_WHITE);
          j=j+pixelTime;
        }
      }
      OLED.display();
      if(dumpFlag){
        dumpFlag=0;
        j=j-pixelTime*102;  //reset
        Serial.println("Oscilloscope Data");
        Serial.println("Time (s), Level (V)");
        for(i=0;i<102;i++){   //do much the same range
          Serial.printf("%8.6f,%7.4f\r\n",float(pixelTime*i)/SAMPRATE,vCal*(samp[j]-cur.sampOffset));        
          j=j+pixelTime;
        }
      }
      dmaAsyncState=DMA_ASYNC_IDLE; //data processed, ready for next
      dmaSampRunAsync(AIN,samp,SAMPSIZE); //start next
      break;    
  }
}

void doSpecScreen(){
  float iPk;
  static int fScale=2000;   //in Hz
  static unsigned int loopt=0;
  static int usePk=0;
  unsigned int tt;
  int i,j,p,ph,iMax;
  long e,t; //total energy
  static int runState=0;  
  while(getButton(S1)){
    fScale=fScale-SPEC_FREQ_STEPS;
    if(fScale<(SPEC_FREQ_STEPS*2)){fScale=SPEC_FREQ_STEPS*2;}
  }
  while(getButton(S2)){
    fScale=fScale+SPEC_FREQ_STEPS;
    if(fScale>10000){fScale=10000;}
  }
  if(getButton(S3)){
    while(getButton(S3)){}    //run down any repeats
    if(usePk){
      usePk=0;
    }else{
      usePk=1;
    }
  }
  checkDmaSampRunAsync();
  switch(dmaAsyncState)  {
    case DMA_ASYNC_IDLE:
      while(nowPtr!=0){}          //synchronise to waveform for phase calcs
      dmaSampRunAsync(AIN,samp,SAMPSIZE);
      break;
    case DMA_ASYNC_RUNNING:
      break;    //do nothing, wait til done
    case DMA_ASYNC_DONE:
      doSubSamp(OVERSAMP);  //resample over lower frequency/higher bit depth
      dmaAsyncState=DMA_ASYNC_IDLE; //data processed, ready for next
      while((nowData[nowPtr]>0)||(nowData[nowPtr+1]<0)){} //find zero crossing
      dmaSampRunAsync(AIN,samp,SAMPSIZE); //start next
      window2_flattop(subSamp, subSampI, SUBSAMPSIZE);
      fix_fft(subSamp, subSampI,SUBSAMPDEPTH,0);
      //convert to real only and get max
      j=0;  //j is max
      e=0;  //e is total energy across spectrum
      for(i=0;i<SUBSAMPSIZE/2;i++){     //avoid aliased samples at high freqs
        //t=(subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]);
        t=((subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]))*FLAT_TOP_SCALING_2;
        subSamp[i]=sqrt(t);
        subSampI[i]=0;
        if(subSamp[i]>j){j=subSamp[i];p=i;}
        e=e+t;
      }
      iPk=getIntPeak(p);
      e=sqrt(e);
      if(e<j){e=j;}   //use most useful value
      if(usePk){e=j;} //scale to peak instead of total
      if(e<1){e=1;}   //validate
      OLED.clearDisplay();
      OLED.setCursor(0, 0);
      OLED.print("SPEC");
      OLED.drawFastHLine(26,13,102,SH110X_WHITE);
      OLED.drawFastVLine(25,13,51,SH110X_WHITE);
      for(i=0;i<40;i=i+5){
        OLED.drawPixel(51,13+i,SH110X_WHITE);
        OLED.drawPixel(76,13+i,SH110X_WHITE);
        OLED.drawPixel(101,13+i,SH110X_WHITE);
      }
      OLED.setCursor(25,5);      
      OLED.printf("0Hz  %5d  %5d",fScale/2,fScale);
      //count through displayable bins
      iMax=fScale/BIN_WIDTH;  //this scales to x=100
      for(i=0;i<=iMax;i++){
        //OLED.drawPixel((i*100)/iMax+26,((subSamp[i]*50)/e)+13,SH110X_WHITE);    //dots only
        OLED.drawFastVLine((i*100)/iMax+26,14,((subSamp[i]*50)/e),SH110X_WHITE);  //lines from axis
      }
      if(dumpFlag){
        dumpFlag=0;
        Serial.println("Spectrum Mode FFT Dump");
        Serial.println("Index,Frequency (Hz),Level (V)");
        for(i=0;i<=iMax;i++){
          Serial.printf("%4d,%9.1f,%8.4f\r\n",i,i*BIN_WIDTH,subSamp[i]*vCal*FFT_SCALING);
        }
      }
      OLED.setCursor(64,55);
      OLED.printf("PK=%5.0fHz",iPk);
      //OLED.printf("PK:%5.0fHz",p*BIN_WIDTH);
      OLED.setCursor(0,25);
      OLED.print(" S3 ");
      OLED.setCursor(0,35);
      OLED.print(usePk?"PEAK":"TOT.");
      OLED.setCursor(0,45);
      e=e*vCal*1000*FFT_SCALING;    //total energy in mV
      if(e<10000){
        OLED.printf("%4d",e);  
      OLED.setCursor(0,55);
      OLED.printf("  mV");
      }else{
        OLED.printf("%4.1f",e/1000.0);  
        OLED.setCursor(0,55);
        OLED.printf("   V");
      }
      OLED.display();
      break;    
  }
}

void doHarmonicScreen(){
  int i,j,p,ph;
  float iPk,ePk,eTot,eHarm,thdn,thd;
  long e,t; //total energy
  checkDmaSampRunAsync();
  switch(dmaAsyncState)  {
    case DMA_ASYNC_IDLE:
      while(nowPtr!=0){}          //synchronise to waveform for phase calcs
      dmaSampRunAsync(AIN,samp,SAMPSIZE);
      break;
    case DMA_ASYNC_RUNNING:
      break;    //do nothing, wait til done
    case DMA_ASYNC_DONE:
      getHist();
      ADCfix();
      doSubSamp(OVERSAMP);  //resample over lower frequency/higher bit depth
      dmaAsyncState=DMA_ASYNC_IDLE; //data processed, ready for next
      //while(nowPtr!=0){}          //synchronise to waveform for phase calcs
      while((nowData[nowPtr]>0)||(nowData[nowPtr+1]<0)){} //find zero crossing
      dmaSampRunAsync(AIN,samp,SAMPSIZE); //start next
      window2_flattop(subSamp, subSampI, SUBSAMPSIZE);
      fix_fft(subSamp, subSampI,SUBSAMPDEPTH,0);
      subSamp[0]=0;
      subSampI[0]=0;
      //convert to real only and get max
      j=0;  //j is max
      e=0;  //e is total energy across spectrum
      for(i=0;i<SUBSAMPSIZE/2;i++){     //avoid aliased samples at high freqs
        //t=(subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]);
        t=((subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]))*FLAT_TOP_SCALING_2;
        subSamp[i]=sqrt(t);
        subSampI[i]=0;
        if(subSamp[i]>j){j=subSamp[i];p=i;}
        e=e+t;
      }
      eTot=sqrt(e);
      e=sqrt(e);
      if(e<j){e=j;}
      if(e<1){e=1;} //validate
      iPk=getIntPeak(p);
      ePk=getEnergy(p,3);
      if(eTot<ePk){eTot=ePk;} //sanity check
      if(ePk<1){ePk=1;}      
      thdn=((eTot-ePk)*100)/ePk;
      if(thdn>99){thdn=99;}     
      eHarm=getEnergies((iPk*2+BIN_WIDTH/2)/BIN_WIDTH,3,20,(iPk+BIN_WIDTH/2)/BIN_WIDTH);
      thd=(eHarm*100)/ePk;
      if(thd>100){thd=100;}     
      OLED.clearDisplay();
      OLED.setCursor(0, 0);
      OLED.print("HARMONIC ANALYSIS");
      OLED.setCursor(0,10);
      OLED.printf("PEAK:%5.0fHz  %5.0fmV",iPk,(ePk*vCal*1000*FFT_SCALING));
      OLED.setCursor(60,20);
      OLED.printf("2nd %5.0fmV",(getEnergy((iPk*2+BIN_WIDTH/2)/BIN_WIDTH,3)*vCal*1000*FFT_SCALING));
      OLED.setCursor(60,30);
      OLED.printf("3rd %5.0fmV",(getEnergy((iPk*3+BIN_WIDTH/2)/BIN_WIDTH,3)*vCal*1000*FFT_SCALING));
      OLED.setCursor(60,40);
      OLED.printf("Even%5.0fmV",(getEnergies((iPk*4+BIN_WIDTH/2)/BIN_WIDTH,3,9,(iPk*2+BIN_WIDTH/2)/BIN_WIDTH)*vCal*1000*FFT_SCALING));  //9 even harmonics from the 4th
      OLED.setCursor(60,50);
      OLED.printf("Odd %5.0fmV",(getEnergies((iPk*5+BIN_WIDTH/2)/BIN_WIDTH,3,9,(iPk*2+BIN_WIDTH/2)/BIN_WIDTH)*vCal*1000*FFT_SCALING));  //9 odd harmonics from the 5th
      OLED.setCursor(0,30);
      OLED.print("THD");
      OLED.setCursor(0,40);
      OLED.printf("%6.2f%%",thd);
      OLED.display();
      if(dumpFlag){
        dumpFlag=0;
        Serial.println("Harmonic Analysis");
        Serial.printf("Peak (Hz),%8.1f\r\n",iPk);
        Serial.printf("Amplitude at peak (V),%6.4f\r\n",(ePk*vCal*FFT_SCALING));
        for(i=2;i<21;i++){
          if(iPk*i<10000){
            Serial.printf("Harmonic %2d (V) at Freq (Hz),%6.4f,%8.1f\r\n",i,(getEnergy((iPk*i+BIN_WIDTH/2)/BIN_WIDTH,3)*vCal*FFT_SCALING),iPk*i);
          }
        }
        Serial.printf("THD (%%),%6.3f\r\n",thd);
      }
      break;
  }
}


void doSweepScreen(){
  static int d0=1;
  static int d1=4;  //start and end decades (log)
  static int dDiv=10;  //steps
  static int setter=0;  //d0,d1,dDiv,amplitude,run
  static int loopMode=0;  //1=single, 2=loop
  static int s=0;
  static int sweepV=500;
  static int vScale=20;   //for display
  float m,m0;
  int i;
  static float e;
  switch(sweepPhase){
    case 0:
      while(getButton(S1)){   //while to handle repeats
        if(setter==0){d0=d0-1;if(d0<1){d0=1;}}
        else if(setter==1){d1=d1-1;if(d1<=d0){d1=d0+1;}}
        else if(setter==2){dDiv=dDiv-1;if(dDiv<2){dDiv=2;}}
        else if(setter==3){sweepV=sweepV-100;if(sweepV<100){sweepV=100;}}
        else if(setter==4){loopMode=1;}
      }
      while(getButton(S2)){   //while to handle repeats
        if(setter==0){d0=d0+1;if(d0>=d1){d0=d1-1;}}
        else if(setter==1){d1=d1+1;if(d1>4){d1=4;}}
        else if(setter==2){dDiv=dDiv+1;if(dDiv>SWEEP_COUNT){dDiv=SWEEP_COUNT;}}
        else if(setter==3){sweepV=sweepV+100;if(sweepV>1000){sweepV=1000;}}
        else if(setter==4){loopMode=2;}
      }
      if(getButton(S3)){
        while(getButton(S1)){}    //run down any repeats
        while(getButton(S2)){}    //run down any repeats
        while(getButton(S3)){}    //run down any repeats
        setter=setter+1;
        if(setter>4){setter=0;}
      }      
      OLED.clearDisplay();
      OLED.setCursor(0, 0);
      OLED.print("SWEEP SETUP");
      OLED.setCursor(0, 10);
      OLED.printf("%c%4dHz%c to %c%5dHz%c",(setter==0)?'<':' ',decades[d0],(setter==0)?'>':' ',(setter==1)?'<':' ',decades[d1],(setter==1)?'>':' ');
      OLED.setCursor(0, 20);
      OLED.printf("Steps:       %c%2d%c",(setter==2)?'<':' ',dDiv,(setter==2)?'>':' ');
      OLED.setCursor(0, 30);
      OLED.printf("Level:   %c%4dmV%c",(setter==3)?'<':' ',sweepV,(setter==3)?'>':' ');
      OLED.setCursor(0, 40);
      OLED.printf("UP=   %cRUN LOOP%c",(setter==4)?'<':' ',(setter==4)?'>':' ');
      OLED.setCursor(0, 50);
      OLED.printf("DOWN= %cRUN SINGLE%c",(setter==4)?'<':' ',(setter==4)?'>':' ');
      OLED.display();
      if(loopMode){
        sweepPhase=1;
        s=0;
        newV=sweepV;
        //e=sweepV*cur.wavCal*vCal/2;     //nominal at set gain
        e=sweepV;     //nominal reference
        sweepFreqs[0]=decades[d0];
        sweepFreqs[dDiv-1]=decades[d1];
        sweepSteps=dDiv;
        if(sweepSteps<2){sweepSteps=2;} //validate
        m0=exp(log(decades[d1]/decades[d0])/(dDiv-1));
        m=m0;
        //Serial.println("Intermediate sweep frequencies:");
        for(i=1;i<dDiv-1;i++){
          sweepFreqs[i]=sweepFreqs[0]*m;
          m=m*m0;
          //Serial.printf("%d\t%d\r\n",i,sweepFreqs[i]);
        }
        OLED.clearDisplay();
        OLED.display(); 
      }
      break;
    case 1:
      runSweep(s);      
      if(e>1){
        sweepDB[s]=8.685889638*log(sweepAmp[s]/e);  //convert from ln to log 10 and multiply by 20
      }else{
        sweepDB[s]=-100;
      }      
      s++;
      if(s>=sweepSteps){   //continue when done
        sweepPhase=2;
        if(loopMode==1){
          loopMode=0;
        }
      }
      while(getButton(S1)){   //while to handle repeats
        vScale=vScale/2;
        if(vScale<5){vScale=5;}
      }
      while(getButton(S2)){   //while to handle repeats
        vScale=vScale*2;
        if(vScale>80){vScale=80;}
      }
      if(getButton(S3)){
        while(getButton(S1)){}    //run down any repeats
        while(getButton(S2)){}    //run down any repeats
        while(getButton(S3)){}    //run down any repeats
        if(loopMode==2){loopMode=1;}  //stop looping
      }
      if((mode!=SWEEP)&&(loopMode==2)){//stop looping if not in sweep mode
        loopMode=1;
      }
      OLED.clearDisplay();
      OLED.setCursor(50, 0);
      OLED.printf("%2d/%2d",s,sweepSteps);
      //graph drawing done in separate section below
      break;
    case 2:
      if(curSweepAmp>0){   //turn off for now
        curSweepAmp=0;
        setWave(sweepF, curSweepAmp);
        outputOn=0;       //flag that wave needs to be turned back on
      }
      OLED.clearDisplay();
      OLED.setCursor(50, 0);
      OLED.print("<OK>");
      //graph drawing done in separate section below
      if(dumpFlag){   //set flag to run a dump only if set at start of Sweep
        dumpFlag=0;
        Serial.println("Sweep results:"); 
        Serial.println("Index,Frequency (Hz),Amplitude (V),Gain (dB)");       
        for(i=0;i<sweepSteps;i++){
          Serial.printf("%3d,%6d,%6.3f,%4d\r\n",i,sweepFreqs[i],sweepAmp[i]*0.001,int(sweepDB[i]));
        }        
      }
      while(getButton(S1)){   //while to handle repeats
        vScale=vScale/2;
        if(vScale<5){vScale=5;}
      }
      while(getButton(S2)){   //while to handle repeats
        vScale=vScale*2;
        if(vScale>80){vScale=80;}
      }
      if(getButton(S3)){
        while(getButton(S1)){}    //run down any repeats
        while(getButton(S2)){}    //run down any repeats
        while(getButton(S3)){}    //run down any repeats
        sweepPhase=0;
      }
      if(loopMode==2){
        sweepPhase=1;
        s=0;
      }
      break;
  }
  //graph drawing
  if((sweepPhase==1)||(sweepPhase==2)){
    OLED.drawFastHLine(26,13,100,SH110X_WHITE);
    OLED.drawFastVLine(25,13,51,SH110X_WHITE);
    OLED.drawFastVLine(125,13,51,SH110X_WHITE);
    if((d1-d0)>0){
      for(i=0;i<50;i=i+3){
        OLED.drawPixel(26+(100/(d1-d0)),13+i,SH110X_WHITE);
        OLED.drawPixel(26+(200/(d1-d0)),13+i,SH110X_WHITE);
        OLED.drawPixel(26+(300/(d1-d0)),13+i,SH110X_WHITE);
      }
      for(i=0;i<50;i=i+5){
        OLED.drawPixel(26+(30/(d1-d0)),13+i,SH110X_WHITE);  //x2
        OLED.drawPixel(26+(70/(d1-d0)),13+i,SH110X_WHITE);  //x5
        OLED.drawPixel(26+(130/(d1-d0)),13+i,SH110X_WHITE);  //x20
        OLED.drawPixel(26+(170/(d1-d0)),13+i,SH110X_WHITE);  //x50
        OLED.drawPixel(26+(230/(d1-d0)),13+i,SH110X_WHITE);  //x200
        OLED.drawPixel(26+(270/(d1-d0)),13+i,SH110X_WHITE);  //x500
      }
    }
    for(i=0;i<100;i=i+3){
      OLED.drawPixel(26+i,23,SH110X_WHITE);
      if(vScale<40){OLED.drawPixel(26+i,23+(60/vScale),SH110X_WHITE);} //-3dB
      OLED.drawPixel(26+i,43,SH110X_WHITE);
    }
    OLED.setCursor(0,5);      
    OLED.printf("%5dHz",decades[d0]);
    OLED.setCursor(85,5);      
    OLED.printf("%5dHz",decades[d1]);
    OLED.setCursor(6,19);      
    OLED.print("0dB");
    OLED.setCursor(0,39);      
    OLED.printf("%4d",-vScale);
    for(i=0;i<sweepSteps-1;i++){
      OLED.drawLine(26+(i*100)/(sweepSteps-1),23-((sweepDB[i]*20)/vScale),26+((i+1)*100)/(sweepSteps-1),23-((sweepDB[i+1]*20)/vScale),SH110X_WHITE);
    }
    OLED.display();
  }
}

void runSweep(int s){
  int i,j,t,p,iPk;
  long e=0;
  t=0;
  checkDmaSampRunAsync();
  while(dmaAsyncState==DMA_ASYNC_RUNNING){checkDmaSampRunAsync();}  //ensure DMA/ADC not busy
  curSweepAmp=(cur.wavCal*pkToRMS[SINE_WAVE]*newV)/1000;    
  sweepF=sweepFreqs[s];
  setWave(sweepF, curSweepAmp);
  delay(100); //settle, at least one cycle at 10Hz
  dmaSampRun(AIN,samp,SAMPSIZE);
  doSubSamp(OVERSAMP);  //resample over lower frequency/higher bit depth
  window2_flattop(subSamp, subSampI, SUBSAMPSIZE);
  fix_fft(subSamp, subSampI,SUBSAMPDEPTH,0);
  j=0;  //j is max
  for(i=0;i<SUBSAMPSIZE/2;i++){     //avoid aliased samples at high freqs
    t=((subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]))*FLAT_TOP_SCALING_2;
    //t=(subSampI[i]*subSampI[i])+(subSamp[i]*subSamp[i]);
    subSamp[i]=sqrt(t);
    subSampI[i]=0;
    if(subSamp[i]>j){j=subSamp[i];p=i;}
    e=e+t;
  }
  e=sqrt(e);
  sweepAmp[s]=getEnergy((sweepFreqs[s]+BIN_WIDTH/2)/BIN_WIDTH,3)*vCal*1000*FFT_SCALING;
  sweepEnergy[s]=e;
}

float getIntPeak(int p){  //find interpolated peak frequency, assuming FFT peak lies in bin p, improved for windowed waveform
  float r;
  float d0,d1; //deviations of lower and upper bins
  if(p==0){
    return ((float)(BIN_WIDTH)*subSamp[1])/(subSamp[0]+subSamp[1]);  //interpolate between 2 lowest    
  }else{
    d0=subSamp[p]-subSamp[p-1];
    d1=subSamp[p]-subSamp[p+1];
    if((d0+d1)>0){
      return ((d0*BIN_WIDTH)/(d0+d1))+((p-0.5)*BIN_WIDTH);
    }else{
      return p*BIN_WIDTH;
    }
  }
}

float getEnergy(int n, int s){    //energy around peak at index n, with spread of s
  float r=0;
  int i;
  if(n>=(SUBSAMPSIZE/2)){return 0;} //too high to measure
  for(i=(n-s);i<=(n+s);i++){
    if(n>=0){
      r=r+subSamp[i]*subSamp[i];
    }
  }
  return sqrt(r);
}

float getEnergies(int n, int s, int c, int j){  //multiple (count c) harmonics starting at n spaced by j
  float r=0;
  int i,t;
  if(j<1){j=1;} //avoid getting stuck
  for(i=n;i<n+c*j;i=i+j){
    t=getEnergy(i,s);
    r=r+t*t;
  }
  return sqrt(r);
}

float getRMS(){
  int i;
  int m=getMean();
  float j;
  j=0;
  for(i=0;i<SAMPSIZE;i++){
    j=j+(samp[i]-m)*(samp[i]-m);
  }
  j=j/SAMPSIZE;
  return sqrt(j);
}

int getMean(){
  int i,j;
  j=0;
  for(i=0;i<SAMPSIZE;i++){
    j=j+samp[i];
  }
  return j/SAMPSIZE;
}

int getMax(){
  int i,j;
  j=0;
  for(i=0;i<SAMPSIZE;i++){
    if(samp[i]>j){j=samp[i];}
  }
  return j;
}

int getMin(){
  int i,j;
  j=65536;
  for(i=0;i<SAMPSIZE;i++){
    if(samp[i]<j){j=samp[i];}
  }
  return j;
}

void getHist(void){
  int i;
  if(histFlag){
    histFlag=0;
    for(i=0;i<ADC_DEPTH;i++){histMap[i]=0;}    
    for(i=0;i<SAMPSIZE;i++){
      histMap[samp[i]]=histMap[samp[i]]+1;
    }
    for(i=0;i<ADC_DEPTH;i++){
      Serial.printf("%d,%d\r\n",i,histMap[i]);
    }
  }  
}

void ADCfix(void){  //adjust for DNL anomalies in RP2040 ADC
  int i,t;
  for(i=0;i<SAMPSIZE;i++){
    t=ADCADJ[samp[i]];
    samp[i]=samp[i]+t;
  }
}

void smoothSamp(int n){    //do smoothing in place (moving average over n) on samples to improve scope display
  int i,j,t;
  for(i=0;i<SAMPSIZE-n;i++){
    t=0;
    for(j=0;j<n;j++){t=t+samp[i+j];}
    samp[i]=t/n;
  }
}

void removeDC(){
  int i,j;
  j=0;
  for(i=0;i<SUBSAMPSIZE;i++){j=j+subSamp[i];}
  j=j/SUBSAMPSIZE;
  for(i=0;i<SUBSAMPSIZE;i++){subSamp[i]=subSamp[i]-j;}
}

void doSubSamp(int s){
  int i,j;
  //resample over lower frequency/higher bit depth
  for(i=0;i<SUBSAMPSIZE;i++){
    subSamp[i]=-s*cur.sampOffset; //place samples in real
    subSampI[i]=0;
    for(j=0;j<s;j++){
      subSamp[i]=subSamp[i]+samp[i*s+j];
    }
    //subSamp[i]=subSamp[i];
  }
}

int writeWave(int f, int a, int16_t* d, const fixed* w){    //write out a wave (w) sample of frequency f, amplitude a, returns number of sample points
  int i,index=0,acc=0,res=0,cycles=0;
  long k;
  for(i=0;i<OUTSAMPLESIZE;i++){       //fill whole array anyway
    //k=w[index];                     //no interpolate between samples
    if(acc>OUTSAMPLERATE/2){          //sample doubling
      k=w[index]+(w[(index+1)%N_WAVE]-w[index])/2;    //average 2 samples and avoid overflow
    }else{
      k=w[index]; 
    }
    FIX_MPY(d[i],a,k);
    acc=acc+(f*N_WAVE);
    while(acc>=OUTSAMPLERATE){
      acc=acc-OUTSAMPLERATE;
      index++;
    }
    if(index>=N_WAVE){ //note complete sample cycles
      index=index-N_WAVE;
      res=i;            //whole number of samples
      cycles++;
    }
    if(cycles>200){    //should be good enough, do one more to round out and bail early
      i++;
      FIX_MPY(d[i],a,w[index]);      
      return res;
    }
  }
  return res;
}

void setWave(int f, int a){
  int i,t;
  for(i=0;i<OUTSAMPLESIZE;i++){outSamp[1][i]=outSamp[0][i];}  //copy to other
  nowData=outSamp[1];     //jump into other, it's the same so this should be seamless
  t=millis();
  outSampSize=writeWave(f,a,outSamp[0],newWave);  //generate new
  if(audioSpace()&&(audioState!=AUDIO_STOPPED)){
    while((nowData[nowPtr]>=32768)&&(nowData[nowPtr+1]<=32768)){} //find zero crossing
  }
  audioStop();
  audioQueue(outSamp[0],outSampSize); //enqueue
  audioPlay(AUDIO_LOOP_16);           //loop at new freq
}