/*      fix_fft.c - Fixed-point Fast Fourier Transform  */
/*
   fix_fft()       perform FFT or inverse FFT
   window()        applies a Hanning window to the (time) input
   each freq point. Result is an integer array,
   units are dB (values will be negative).
   iscale()        scale an integer value by (numer/denom).
   fix_mpy()       perform fixed-point multiplication.
   Sinewave[8192]  sinewave normalized to 32767 (= 1.0).

   All data are fixed-point short integers, in which
   -32768 to +32768 represent -1.0 to +1.0. Integer arithmetic
   is used for speed, instead of the more natural floating-point.

   For the forward FFT (time -> freq), fixed scaling is
   performed to prevent arithmetic overflow, and to map a 0dB
   sine/cosine wave (i.e. amplitude = 32767) to two -6dB freq
   coefficients. The return value is always 0.

   For the inverse FFT (freq -> time), fixed scaling cannot be
   done, as two 0dB coefficients would sum to a peak amplitude of
   64K, overflowing the 32k range of the fixed-point integers.
   Thus, the fix_fft() routine performs variable scaling, and
   returns a value which is the number of bits LEFT by which
   the output must be shifted to get the actual amplitude
   (i.e. if fix_fft() returns 3, each value of fr[] and fi[]
   must be multiplied by 8 (2**3) for proper scaling.
   Clearly, this cannot be done within the fixed-point short
   integers. In practice, if the result is to be used as a
   filter, the scale_shift can usually be ignored, as the
   result will be approximately correctly normalized as is.

   TURBO C, any memory model; uses inline assembly for speed
   and for carefully-scaled arithmetic.

   Written by:  Tom Roberts  11/8/89
   Made portable:  Malcolm Slaney 12/15/94 malcolm@interval.com

   Timing on a Macintosh PowerBook 180.... (using Symantec C6.0)
   fix_fft (1024 points)             8 ticks
   fft (1024 points - Using SANE)  112 Ticks
   fft (1024 points - Using FPU)    11

 */

/* FIX_MPY() - fixed-point multiplication macro.
   This macro is a statement, not an expression (uses asm).
   BEWARE: make sure _DX is not clobbered by evaluating (A) or DEST.
   args are all of type fixed.
   Scaling ensures that 32767*32767 = 32767. */

#include "fix_fft32.h"

static fixed fix_mpy (fixed a, fixed b);

/*
   fix_fft() - perform fast Fourier transform.

   if n>0 FFT is done, if n<0 inverse FFT is done
   fr[n],fi[n] are real,imaginary arrays, INPUT AND RESULT.
   size of data = 2**m
   set inverse to 0=dft, 1=idft
 */
int
fix_fft (fixed fr[], fixed fi[], int m, int inverse)
{
  int mr, nn, i, j, l, k, istep, n, scale, shift;
  fixed qr, qi, tr, ti, wr, wi;

  n = 1 << m;

  if (n > N_WAVE)
    return -1;

  mr = 0;
  nn = n - 1;
  scale = 0;

  /* decimation in time - re-order data */
  for (m = 1; m <= nn; ++m) {
    l = n;
    do {
      l >>= 1;
    }
    while (mr + l > nn);
    mr = (mr & (l - 1)) + l;

    if (mr <= m)
      continue;
    tr = fr[m];
    fr[m] = fr[mr];
    fr[mr] = tr;
    ti = fi[m];
    fi[m] = fi[mr];
    fi[mr] = ti;
  }

  l = 1;
  k = LOG2_N_WAVE - 1;
  while (l < n) {
    if (inverse) {
      /* variable scaling, depending upon data */
      shift = 0;
      for (i = 0; i < n; ++i) {
        j = fr[i];
        if (j < 0)
          j = -j;
        m = fi[i];
        if (m < 0)
          m = -m;
        if (j > 1073741823 || m > 1073741823) {
          shift = 1;
          break;
        }
      }
      if (shift)
        ++scale;
    } else {
      /* fixed scaling, for proper normalization -
         there will be log2(n) passes, so this
         results in an overall factor of 1/n,
         distributed to maximize arithmetic accuracy. */
      shift = 1;
    }
    /* it may not be obvious, but the shift will be performed
       on each data point exactly once, during this pass. */
    istep = l << 1;
    for (m = 0; m < l; ++m) {
      j = m << k;
      /* 0 <= j < N_WAVE/2 */
      wr = Sinewave[j + N_WAVE / 4];
      wi = -Sinewave[j];
      if (inverse)
        wi = -wi;
      if (shift) {
        wr >>= 1;
        wi >>= 1;
      }
      for (i = m; i < n; i += istep) {
        j = i + l;
        tr = fix_mpy (wr, fr[j]) -
            fix_mpy (wi, fi[j]);
        ti = fix_mpy (wr, fi[j]) +
            fix_mpy (wi, fr[j]);
        qr = fr[i];
        qi = fi[i];
        if (shift) {
          qr >>= 1;
          qi >>= 1;
        }
        fr[j] = qr - tr;
        fi[j] = qi - ti;
        fr[i] = qr + tr;
        fi[i] = qi + ti;
      }
    }
    --k;
    l = istep;
  }

  return scale;
}

/*      window() - apply a Hanning window       */
void
window_hann (fixed fr[], int n)
{
  int i, j, k;

  j = N_WAVE / n;
  n >>= 1;
  for (i = 0, k = N_WAVE / 4; i < n; ++i, k += j)
    FIX_MPY (fr[i], fr[i], 1073741824 - (Sinewave[k] >> 1));
  n <<= 1;
  for (k -= j; i < n; ++i, k -= j)
    FIX_MPY (fr[i], fr[i], 1073741824 - (Sinewave[k] >> 1));
}

void
window2_hann (fixed fr[], fixed im[], int n)
{
  int i, j, k;

  j = N_WAVE / n;
  for (i = 0, k = N_WAVE / 4; i < n; ++i) {
    if( i&1 ) {
      FIX_MPY (im[i/2], im[i/2], 1073741824 - (Sinewave[k] >> 2) - (Sinewave[k+1] >> 2));
      k += j;
    } else {
      FIX_MPY (fr[i/2], fr[i/2], 1073741824 - (Sinewave[k] >> 1));
    }
  }
  n <<= 1;
  for (k -= j; i < n; ++i) {
    if( i&1 ) {
      FIX_MPY (im[i/2], im[i/2], 1073741824 - (Sinewave[k-1] >> 2) - (Sinewave[k] >> 2));
      k -= j;
    } else {
      FIX_MPY (fr[i/2], fr[i/2], 1073741824 - (Sinewave[k] >> 1));
    }
  }
}

void
window2_flattop (fixed fr[], fixed im[], int n)
{
  int i, j, k;

//  const float a1 = 1.93, a2 = 1.29, a3 = 0.388, a4 = 0.028;

  j = N_WAVE / n;
  for (i = 0, k = 0; i < n; ++i) {
//    float coeff = (1 - a1 * cos(i * 2 * M_PI / (N_WAVE*2)) + a2 * cos(i * 4 * M_PI / (N_WAVE*2)) - a3 * cos(i * 6 * M_PI / (N_WAVE*2)) + a4 * cos(i * 8 * M_PI / (N_WAVE*2))) / 4.6;
    if( i&1 ) {
      FIX_MPY( im[i/2], im[i/2], ((flattop_window[k]>>1) + (flattop_window[k+1]>>1)) );
      k += j;
    } else {
      FIX_MPY( fr[i/2], fr[i/2], flattop_window[k] );
    }
  }
  n <<= 1;
  for (k -= j; i < n; ++i) {
//    float coeff = (1 - a1 * cos(i * 2 * M_PI / (N_WAVE*2)) + a2 * cos(i * 4 * M_PI / (N_WAVE*2)) - a3 * cos(i * 6 * M_PI / (N_WAVE*2)) + a4 * cos(i * 8 * M_PI / (N_WAVE*2))) / 4.6;
    if( i&1 ) {
      FIX_MPY( im[i/2], im[i/2], flattop_window[k] );
      k -= j;
    } else {
      FIX_MPY( fr[i/2], fr[i/2], ((flattop_window[k]>>1) + (flattop_window[k+1]>>1)) );
    }
  }
}

/*
   fix_mpy() - fixed-point multiplication
 */
fixed
fix_mpy (fixed a, fixed b)
{
  FIX_MPY (a, a, b);
  return a;
}

/*
   iscale() - scale an integer value by (numer/denom)
 */
int
iscale (int value, int numer, int denom)
{
  return (long) value *(long) numer / (long) denom;
}

/*
   fix_dot() - dot product of two fixed arrays
 */
fixed
fix_dot (fixed * hpa, fixed * pb, int n)
{
  fixed *pa = hpa;              /* FIXME */
  long sum;
  register fixed a, b;

/*      seg = FP_SEG(hpa);
   off = FP_OFF(hpa);
   seg += off>>4;
   off &= 0x000F;
   pa = MK_FP(seg,off);
 */
  sum = 0L;
  while (n--) {
    a = *pa++;
    b = *pb++;
    FIX_MPY (a, a, b);
    sum += a;
  }

  if (sum > 0x7FFFFFFF)
    sum = 0x7FFFFFFF;
  else if (sum < -0x7FFFFFFF)
    sum = -0x7FFFFFFF;

  return (fixed) sum;

}

void fft_real_result_convolve(fixed* re, fixed* im, int n) {
  int i;
  int scale = N_WAVE / 2 / n;
  for( i = 0; i < n; ++i ) {
    fixed Ar, Ai, Br, Bi;
    if( (i*scale) & 1 ) {
      Ar = 1073741823 - ( Sinewave[           i * scale/2] >> 2) - ( Sinewave[           i * scale/2 + 1] >> 2);
      Ai =              (-Sinewave[N_WAVE/4 + i * scale/2] >> 2) + (-Sinewave[N_WAVE/4 + i * scale/2 + 1] >> 2);
      Br = 1073741823 + ( Sinewave[           i * scale/2] >> 2) + ( Sinewave[           i * scale/2 + 1] >> 2);
      Bi =              ( Sinewave[N_WAVE/4 + i * scale/2] >> 2) + ( Sinewave[N_WAVE/4 + i * scale/2 + 1] >> 2);
    } else {
      Ar = 1073741823 - ( Sinewave[           i * scale/2] >> 1);
      Ai =              (-Sinewave[N_WAVE/4 + i * scale/2] >> 1);
      Br = 1073741823 + ( Sinewave[           i * scale/2] >> 1);
      Bi =              ( Sinewave[N_WAVE/4 + i * scale/2] >> 1);
    }
    fixed re1 = fix_mpy(re[i], Ar) - fix_mpy(im[i], Ai) + fix_mpy(re[n-i], Br) + fix_mpy(im[n-i], Bi);
    fixed im1 = fix_mpy(im[i], Ar) + fix_mpy(re[i], Ai) + fix_mpy(re[n-i], Bi) - fix_mpy(im[n-i], Br);
    re[i] = re1;
    im[i] = im1;
  }
}

#if N_WAVE != 4096
#error N_WAVE != 4096
#endif
    const fixed Sinewave[4096] = { 0, 3294197, 6588386, 9882561, 13176711, 16470831, 19764912, 23058947, 26352927, 29646846, 32940694, 36234465, 39528151, 42821744, 46115236, 49408619, 
			52701886, 55995030, 59288041, 62580913, 65873638, 69166207, 72458614, 75750851, 79042909, 82334781, 85626459, 88917936, 92209204, 95500255, 98791081, 102081674, 
			105372028, 108662133, 111951983, 115241569, 118530884, 121819920, 125108670, 128397125, 131685278, 134973121, 138260647, 141547847, 144834714, 148121240, 151407418, 154693239, 
			157978697, 161263783, 164548489, 167832808, 171116732, 174400253, 177683365, 180966058, 184248325, 187530158, 190811551, 194092494, 197372981, 200653003, 203932553, 207211623, 
			210490205, 213768293, 217045877, 220322950, 223599506, 226875534, 230151030, 233425983, 236700387, 239974234, 243247517, 246520227, 249792357, 253063900, 256334846, 259605190, 
			262874923, 266144037, 269412525, 272680379, 275947591, 279214154, 282480060, 285745301, 289009870, 292273759, 295536960, 298799466, 302061268, 305322360, 308582733, 311842381, 
			315101294, 318359466, 321616889, 324873555, 328129456, 331384586, 334638935, 337892497, 341145265, 344397229, 347648383, 350898719, 354148229, 357396906, 360644742, 363891729, 
			367137860, 370383127, 373627522, 376871039, 380113668, 383355403, 386596236, 389836160, 393075166, 396313247, 399550395, 402786603, 406021864, 409256169, 412489511, 415721883, 
			418953276, 422183683, 425413097, 428641510, 431868914, 435095302, 438320666, 441544999, 444768293, 447990540, 451211733, 454431865, 457650927, 460868912, 464085812, 467301621, 
			470516330, 473729932, 476942419, 480153783, 483364018, 486573116, 489781068, 492987869, 496193509, 499397981, 502601278, 505803393, 509004317, 512204044, 515402565, 518599874, 
			521795962, 524990823, 528184448, 531376830, 534567962, 537757836, 540946445, 544133780, 547319836, 550504603, 553688075, 556870244, 560051103, 563230644, 566408859, 569585742, 
			572761285, 575935479, 579108319, 582279796, 585449902, 588618631, 591785975, 594951926, 598116478, 601279622, 604441351, 607601658, 610760535, 613917975, 617073970, 620228513, 
			623381597, 626533214, 629683356, 632832017, 635979189, 639124864, 642269036, 645411696, 648552837, 651692452, 654830534, 657967075, 661102067, 664235504, 667367378, 670497682, 
			673626407, 676753548, 679879096, 683003045, 686125386, 689246113, 692365217, 695482693, 698598532, 701712727, 704825271, 707936157, 711045377, 714152923, 717258789, 720362967, 
			723465451, 726566231, 729665302, 732762657, 735858286, 738952185, 742044344, 745134758, 748223418, 751310317, 754395449, 757478805, 760560379, 763640163, 766718150, 769794333, 
			772868705, 775941258, 779011985, 782080880, 785147933, 788213140, 791276491, 794337981, 797397601, 800455345, 803511206, 806565176, 809617248, 812667415, 815715669, 818762004, 
			821806412, 824848887, 827889421, 830928006, 833964637, 836999305, 840032003, 843062725, 846091463, 849118210, 852142958, 855165702, 858186434, 861205146, 864221831, 867236483, 
			870249094, 873259658, 876268167, 879274613, 882278991, 885281292, 888281511, 891279639, 894275670, 897269597, 900261412, 903251109, 906238680, 909224119, 912207418, 915188571, 
			918167571, 921144410, 924119081, 927091578, 930061893, 933030020, 935995951, 938959680, 941921199, 944880502, 947837581, 950792430, 953745042, 956695410, 959643526, 962589384, 
			965532977, 968474299, 971413341, 974350097, 977284561, 980216725, 983146582, 986074126, 988999350, 991922247, 994842809, 997761030, 1000676904, 1003590423, 1006501580, 1009410369, 
			1012316783, 1015220815, 1018122458, 1021021705, 1023918549, 1026812984, 1029705003, 1032594599, 1035481765, 1038366494, 1041248781, 1044128616, 1047005996, 1049880911, 1052753356, 1055623323, 
			1058490807, 1061355800, 1064218295, 1067078287, 1069935767, 1072790729, 1075643168, 1078493075, 1081340444, 1084185269, 1087027543, 1089867258, 1092704410, 1095538990, 1098370992, 1101200409, 
			1104027236, 1106851464, 1109673088, 1112492100, 1115308495, 1118122266, 1120933405, 1123741907, 1126547764, 1129350971, 1132151520, 1134949405, 1137744620, 1140537157, 1143327010, 1146114173, 
			1148898639, 1151680402, 1154459455, 1157235791, 1160009404, 1162780287, 1165548434, 1168313839, 1171076494, 1173836394, 1176593532, 1179347901, 1182099495, 1184848307, 1187594331, 1190337561, 
			1193077990, 1195815611, 1198550418, 1201282406, 1204011566, 1206737893, 1209461381, 1212182022, 1214899812, 1217614742, 1220326808, 1223036001, 1225742317, 1228445749, 1231146290, 1233843934, 
			1236538674, 1239230505, 1241919420, 1244605413, 1247288476, 1249968605, 1252645793, 1255320033, 1257991319, 1260659645, 1263325004, 1265987391, 1268646799, 1271303221, 1273956652, 1276607085, 
			1279254514, 1281898933, 1284540336, 1287178716, 1289814067, 1292446383, 1295075658, 1297701885, 1300325059, 1302945173, 1305562221, 1308176197, 1310787094, 1313394908, 1315999630, 1318601256, 
			1321199779, 1323795194, 1326387493, 1328976671, 1331562722, 1334145640, 1336725418, 1339302051, 1341875532, 1344445856, 1347013016, 1349577006, 1352137821, 1354695454, 1357249900, 1359801151, 
			1362349203, 1364894049, 1367435684, 1369974100, 1372509293, 1375041257, 1377569984, 1380095471, 1382617709, 1385136695, 1387652420, 1390164881, 1392674071, 1395179983, 1397682612, 1400181953, 
			1402677998, 1405170743, 1407660182, 1410146308, 1412629116, 1415108600, 1417584754, 1420057573, 1422527049, 1424993179, 1427455955, 1429915373, 1432371425, 1434824107, 1437273413, 1439719337, 
			1442161873, 1444601016, 1447036759, 1449469097, 1451898024, 1454323535, 1456745624, 1459164285, 1461579512, 1463991301, 1466399644, 1468804536, 1471205973, 1473603947, 1475998454, 1478389488, 
			1480777043, 1483161114, 1485541694, 1487918779, 1490292363, 1492662440, 1495029005, 1497392051, 1499751575, 1502107569, 1504460028, 1506808947, 1509154321, 1511496144, 1513834409, 1516169113, 
			1518500249, 1520827811, 1523151796, 1525472195, 1527789006, 1530102221, 1532411836, 1534717845, 1537020242, 1539319023, 1541614182, 1543905713, 1546193611, 1548477871, 1550758487, 1553035454, 
			1555308767, 1557578419, 1559844407, 1562106724, 1564365365, 1566620326, 1568871600, 1571119182, 1573363067, 1575603250, 1577839725, 1580072488, 1582301532, 1584526853, 1586748446, 1588966305, 
			1591180424, 1593390800, 1595597426, 1597800298, 1599999410, 1602194757, 1604386334, 1606574135, 1608758156, 1610938392, 1613114836, 1615287485, 1617456334, 1619621376, 1621782607, 1623940021, 
			1626093615, 1628243382, 1630389317, 1632531417, 1634669674, 1636804086, 1638934645, 1641061348, 1643184190, 1645303165, 1647418268, 1649529495, 1651636840, 1653740299, 1655839866, 1657935537, 
			1660027307, 1662115171, 1664199123, 1666279160, 1668355275, 1670427465, 1672495724, 1674560047, 1676620430, 1678676868, 1680729356, 1682777889, 1684822462, 1686863071, 1688899710, 1690932375, 
			1692961061, 1694985763, 1697006477, 1699023198, 1701035921, 1703044641, 1705049354, 1707050054, 1709046738, 1711039400, 1713028036, 1715012641, 1716993210, 1718969739, 1720942223, 1722910658, 
			1724875039, 1726835360, 1728791618, 1730743809, 1732691926, 1734635967, 1736575926, 1738511798, 1740443579, 1742371265, 1744294851, 1746214333, 1748129706, 1750040965, 1751948106, 1753851124, 
			1755750016, 1757644776, 1759535400, 1761421884, 1763304223, 1765182413, 1767056449, 1768926327, 1770792043, 1772653592, 1774510969, 1776364171, 1778213193, 1780058031, 1781898680, 1783735136, 
			1785567395, 1787395452, 1789219303, 1791038944, 1792854371, 1794665579, 1796472564, 1798275321, 1800073847, 1801868137, 1803658188, 1805443994, 1807225552, 1809002857, 1810775905, 1812544693, 
			1814309215, 1816069468, 1817825448, 1819577150, 1821324571, 1823067705, 1824806550, 1826541101, 1828271354, 1829997305, 1831718950, 1833436285, 1835149305, 1836858007, 1838562387, 1840262440, 
			1841958163, 1843649552, 1845336602, 1847019310, 1848697672, 1850371684, 1852041342, 1853706642, 1855367579, 1857024151, 1858676353, 1860324182, 1861967633, 1863606703, 1865241387, 1866871682, 
			1868497584, 1870119090, 1871736195, 1873348896, 1874957188, 1876561068, 1878160533, 1879755578, 1881346200, 1882932395, 1884514160, 1886091490, 1887664381, 1889232831, 1890796835, 1892356390, 
			1893911493, 1895462138, 1897008324, 1898550045, 1900087299, 1901620082, 1903148391, 1904672221, 1906191569, 1907706431, 1909216805, 1910722686, 1912224071, 1913720956, 1915213339, 1916701214, 
			1918184579, 1919663431, 1921137766, 1922607579, 1924072869, 1925533631, 1926989863, 1928441559, 1929888719, 1931331336, 1932769410, 1934202935, 1935631909, 1937056328, 1938476189, 1939891489, 
			1941302224, 1942708390, 1944109986, 1945507007, 1946899449, 1948287311, 1949670588, 1951049277, 1952423376, 1953792880, 1955157786, 1956518092, 1957873794, 1959224889, 1960571374, 1961913245, 
			1963250500, 1964583135, 1965911147, 1967234533, 1968553290, 1969867415, 1971176905, 1972481756, 1973781966, 1975077531, 1976368449, 1977654716, 1978936329, 1980213286, 1981485584, 1982753218, 
			1984016187, 1985274488, 1986528117, 1987777071, 1989021348, 1990260945, 1991495858, 1992726085, 1993951623, 1995172470, 1996388621, 1997600075, 1998806828, 2000008877, 2001206221, 2002398855, 
			2003586778, 2004769986, 2005948476, 2007122247, 2008291294, 2009455616, 2010615209, 2011770071, 2012920199, 2014065591, 2015206243, 2016342154, 2017473319, 2018599738, 2019721406, 2020838322, 
			2021950482, 2023057885, 2024160527, 2025258407, 2026351520, 2027439866, 2028523440, 2029602242, 2030676267, 2031745514, 2032809981, 2033869663, 2034924560, 2035974669, 2037019987, 2038060511, 
			2039096240, 2040127170, 2041153300, 2042174627, 2043191148, 2044202862, 2045209765, 2046211856, 2047209132, 2048201590, 2049189229, 2050172046, 2051150039, 2052123205, 2053091543, 2054055049, 
			2055013722, 2055967559, 2056916558, 2057860717, 2058800034, 2059734506, 2060664132, 2061588909, 2062508834, 2063423906, 2064334123, 2065239482, 2066139982, 2067035619, 2067926393, 2068812301, 
			2069693340, 2070569510, 2071440807, 2072307229, 2073168776, 2074025444, 2074877232, 2075724137, 2076566158, 2077403293, 2078235539, 2079062895, 2079885359, 2080702928, 2081515602, 2082323377, 
			2083126253, 2083924227, 2084717297, 2085505461, 2086288718, 2087067066, 2087840503, 2088609027, 2089372636, 2090131329, 2090885104, 2091633958, 2092377891, 2093116900, 2093850983, 2094580140, 
			2095304368, 2096023666, 2096738031, 2097447462, 2098151959, 2098851517, 2099546137, 2100235817, 2100920555, 2101600349, 2102275197, 2102945099, 2103610052, 2104270056, 2104925108, 2105575206, 
			2106220350, 2106860538, 2107495769, 2108126040, 2108751350, 2109371699, 2109987084, 2110597504, 2111202957, 2111803443, 2112398959, 2112989505, 2113575078, 2114155678, 2114731304, 2115301953, 
			2115867624, 2116428317, 2116984030, 2117534761, 2118080509, 2118621274, 2119157053, 2119687845, 2120213650, 2120734465, 2121250290, 2121761124, 2122266965, 2122767812, 2123263664, 2123754520, 
			2124240379, 2124721239, 2125197099, 2125667958, 2126133816, 2126594670, 2127050521, 2127501366, 2127947205, 2128388037, 2128823860, 2129254674, 2129680478, 2130101271, 2130517051, 2130927817, 
			2131333570, 2131734307, 2132130028, 2132520732, 2132906418, 2133287086, 2133662733, 2134033359, 2134398964, 2134759547, 2135115106, 2135465641, 2135811151, 2136151635, 2136487093, 2136817524, 
			2137142926, 2137463299, 2137778643, 2138088956, 2138394238, 2138694489, 2138989706, 2139279891, 2139565041, 2139845157, 2140120238, 2140390283, 2140655291, 2140915262, 2141170196, 2141420091, 
			2141664947, 2141904763, 2142139539, 2142369275, 2142593969, 2142813622, 2143028233, 2143237800, 2143442325, 2143641805, 2143836242, 2144025634, 2144209981, 2144389282, 2144563537, 2144732746, 
			2144896908, 2145056023, 2145210091, 2145359111, 2145503082, 2145642005, 2145775879, 2145904703, 2146028478, 2146147204, 2146260879, 2146369504, 2146473078, 2146571602, 2146665074, 2146753495, 
			2146836865, 2146915182, 2146988448, 2147056662, 2147119824, 2147177933, 2147230990, 2147278994, 2147321945, 2147359843, 2147392689, 2147420481, 2147443221, 2147460907, 2147473540, 2147481120, 
			2147483647, 2147481120, 2147473540, 2147460907, 2147443221, 2147420481, 2147392689, 2147359843, 2147321945, 2147278994, 2147230990, 2147177933, 2147119824, 2147056662, 2146988448, 2146915182, 
			2146836865, 2146753495, 2146665074, 2146571602, 2146473078, 2146369504, 2146260879, 2146147204, 2146028478, 2145904703, 2145775879, 2145642005, 2145503082, 2145359111, 2145210091, 2145056023, 
			2144896908, 2144732746, 2144563537, 2144389282, 2144209981, 2144025634, 2143836242, 2143641805, 2143442325, 2143237800, 2143028233, 2142813622, 2142593969, 2142369275, 2142139539, 2141904763, 
			2141664947, 2141420091, 2141170196, 2140915262, 2140655291, 2140390283, 2140120238, 2139845157, 2139565041, 2139279891, 2138989706, 2138694489, 2138394238, 2138088956, 2137778643, 2137463299, 
			2137142926, 2136817524, 2136487093, 2136151635, 2135811151, 2135465641, 2135115106, 2134759547, 2134398964, 2134033359, 2133662733, 2133287086, 2132906418, 2132520732, 2132130028, 2131734307, 
			2131333570, 2130927817, 2130517051, 2130101271, 2129680478, 2129254674, 2128823860, 2128388037, 2127947205, 2127501366, 2127050521, 2126594670, 2126133816, 2125667958, 2125197099, 2124721239, 
			2124240379, 2123754520, 2123263664, 2122767812, 2122266965, 2121761124, 2121250290, 2120734465, 2120213650, 2119687845, 2119157053, 2118621274, 2118080509, 2117534761, 2116984030, 2116428317, 
			2115867624, 2115301953, 2114731304, 2114155678, 2113575078, 2112989505, 2112398959, 2111803443, 2111202957, 2110597504, 2109987084, 2109371699, 2108751350, 2108126040, 2107495769, 2106860538, 
			2106220350, 2105575206, 2104925108, 2104270056, 2103610052, 2102945099, 2102275197, 2101600349, 2100920555, 2100235817, 2099546137, 2098851517, 2098151959, 2097447462, 2096738031, 2096023666, 
			2095304368, 2094580140, 2093850983, 2093116900, 2092377891, 2091633958, 2090885104, 2090131329, 2089372636, 2088609027, 2087840503, 2087067066, 2086288718, 2085505461, 2084717297, 2083924227, 
			2083126253, 2082323377, 2081515602, 2080702928, 2079885359, 2079062895, 2078235539, 2077403293, 2076566158, 2075724137, 2074877232, 2074025444, 2073168776, 2072307229, 2071440807, 2070569510, 
			2069693340, 2068812301, 2067926393, 2067035619, 2066139982, 2065239482, 2064334123, 2063423906, 2062508834, 2061588909, 2060664132, 2059734506, 2058800034, 2057860717, 2056916558, 2055967559, 
			2055013722, 2054055049, 2053091543, 2052123205, 2051150039, 2050172046, 2049189229, 2048201590, 2047209132, 2046211856, 2045209765, 2044202862, 2043191148, 2042174627, 2041153300, 2040127170, 
			2039096240, 2038060511, 2037019987, 2035974669, 2034924560, 2033869663, 2032809981, 2031745514, 2030676267, 2029602242, 2028523440, 2027439866, 2026351520, 2025258407, 2024160527, 2023057885, 
			2021950482, 2020838322, 2019721406, 2018599738, 2017473319, 2016342154, 2015206243, 2014065591, 2012920199, 2011770071, 2010615209, 2009455616, 2008291294, 2007122247, 2005948476, 2004769986, 
			2003586778, 2002398855, 2001206221, 2000008877, 1998806828, 1997600075, 1996388621, 1995172470, 1993951623, 1992726085, 1991495858, 1990260945, 1989021348, 1987777071, 1986528117, 1985274488, 
			1984016187, 1982753218, 1981485584, 1980213286, 1978936329, 1977654716, 1976368449, 1975077531, 1973781966, 1972481756, 1971176905, 1969867415, 1968553290, 1967234533, 1965911147, 1964583135, 
			1963250500, 1961913245, 1960571374, 1959224889, 1957873794, 1956518092, 1955157786, 1953792880, 1952423376, 1951049277, 1949670588, 1948287311, 1946899449, 1945507007, 1944109986, 1942708390, 
			1941302224, 1939891489, 1938476189, 1937056328, 1935631909, 1934202935, 1932769410, 1931331336, 1929888719, 1928441559, 1926989863, 1925533631, 1924072869, 1922607579, 1921137766, 1919663431, 
			1918184579, 1916701214, 1915213339, 1913720956, 1912224071, 1910722686, 1909216805, 1907706431, 1906191569, 1904672221, 1903148391, 1901620082, 1900087299, 1898550045, 1897008324, 1895462138, 
			1893911493, 1892356390, 1890796835, 1889232831, 1887664381, 1886091490, 1884514160, 1882932395, 1881346200, 1879755578, 1878160533, 1876561068, 1874957188, 1873348896, 1871736195, 1870119090, 
			1868497584, 1866871682, 1865241387, 1863606703, 1861967633, 1860324182, 1858676353, 1857024151, 1855367579, 1853706642, 1852041342, 1850371684, 1848697672, 1847019310, 1845336602, 1843649552, 
			1841958163, 1840262440, 1838562387, 1836858007, 1835149305, 1833436285, 1831718950, 1829997305, 1828271354, 1826541101, 1824806550, 1823067705, 1821324571, 1819577150, 1817825448, 1816069468, 
			1814309215, 1812544693, 1810775905, 1809002857, 1807225552, 1805443994, 1803658188, 1801868137, 1800073847, 1798275321, 1796472564, 1794665579, 1792854371, 1791038944, 1789219303, 1787395452, 
			1785567395, 1783735136, 1781898680, 1780058031, 1778213193, 1776364171, 1774510969, 1772653592, 1770792043, 1768926327, 1767056449, 1765182413, 1763304223, 1761421884, 1759535400, 1757644776, 
			1755750016, 1753851124, 1751948106, 1750040965, 1748129706, 1746214333, 1744294851, 1742371265, 1740443579, 1738511798, 1736575926, 1734635967, 1732691926, 1730743809, 1728791618, 1726835360, 
			1724875039, 1722910658, 1720942223, 1718969739, 1716993210, 1715012641, 1713028036, 1711039400, 1709046738, 1707050054, 1705049354, 1703044641, 1701035921, 1699023198, 1697006477, 1694985763, 
			1692961061, 1690932375, 1688899710, 1686863071, 1684822462, 1682777889, 1680729356, 1678676868, 1676620430, 1674560047, 1672495724, 1670427465, 1668355275, 1666279160, 1664199123, 1662115171, 
			1660027307, 1657935537, 1655839866, 1653740299, 1651636840, 1649529495, 1647418268, 1645303165, 1643184190, 1641061348, 1638934645, 1636804086, 1634669674, 1632531417, 1630389317, 1628243382, 
			1626093615, 1623940021, 1621782607, 1619621376, 1617456334, 1615287485, 1613114836, 1610938392, 1608758156, 1606574135, 1604386334, 1602194757, 1599999410, 1597800298, 1595597426, 1593390800, 
			1591180424, 1588966305, 1586748446, 1584526853, 1582301532, 1580072488, 1577839725, 1575603250, 1573363067, 1571119182, 1568871600, 1566620326, 1564365365, 1562106724, 1559844407, 1557578419, 
			1555308767, 1553035454, 1550758487, 1548477871, 1546193611, 1543905713, 1541614182, 1539319023, 1537020242, 1534717845, 1532411836, 1530102221, 1527789006, 1525472195, 1523151796, 1520827811, 
			1518500249, 1516169113, 1513834409, 1511496144, 1509154321, 1506808947, 1504460028, 1502107569, 1499751575, 1497392051, 1495029005, 1492662440, 1490292363, 1487918779, 1485541694, 1483161114, 
			1480777043, 1478389488, 1475998454, 1473603947, 1471205973, 1468804536, 1466399644, 1463991301, 1461579512, 1459164285, 1456745624, 1454323535, 1451898024, 1449469097, 1447036759, 1444601016, 
			1442161873, 1439719337, 1437273413, 1434824107, 1432371425, 1429915373, 1427455955, 1424993179, 1422527049, 1420057573, 1417584754, 1415108600, 1412629116, 1410146308, 1407660182, 1405170743, 
			1402677998, 1400181953, 1397682612, 1395179983, 1392674071, 1390164881, 1387652420, 1385136695, 1382617709, 1380095471, 1377569984, 1375041257, 1372509293, 1369974100, 1367435684, 1364894049, 
			1362349203, 1359801151, 1357249900, 1354695454, 1352137821, 1349577006, 1347013016, 1344445856, 1341875532, 1339302051, 1336725418, 1334145640, 1331562722, 1328976671, 1326387493, 1323795194, 
			1321199779, 1318601256, 1315999630, 1313394908, 1310787094, 1308176197, 1305562221, 1302945173, 1300325059, 1297701885, 1295075658, 1292446383, 1289814067, 1287178716, 1284540336, 1281898933, 
			1279254514, 1276607085, 1273956652, 1271303221, 1268646799, 1265987391, 1263325004, 1260659645, 1257991319, 1255320033, 1252645793, 1249968605, 1247288476, 1244605413, 1241919420, 1239230505, 
			1236538674, 1233843934, 1231146290, 1228445749, 1225742317, 1223036001, 1220326808, 1217614742, 1214899812, 1212182022, 1209461381, 1206737893, 1204011566, 1201282406, 1198550418, 1195815611, 
			1193077990, 1190337561, 1187594331, 1184848307, 1182099495, 1179347901, 1176593532, 1173836394, 1171076494, 1168313839, 1165548434, 1162780287, 1160009404, 1157235791, 1154459455, 1151680402, 
			1148898639, 1146114173, 1143327010, 1140537157, 1137744620, 1134949405, 1132151520, 1129350971, 1126547764, 1123741907, 1120933405, 1118122266, 1115308495, 1112492100, 1109673088, 1106851464, 
			1104027236, 1101200409, 1098370992, 1095538990, 1092704410, 1089867258, 1087027543, 1084185269, 1081340444, 1078493075, 1075643168, 1072790729, 1069935767, 1067078287, 1064218295, 1061355800, 
			1058490807, 1055623323, 1052753356, 1049880911, 1047005996, 1044128616, 1041248781, 1038366494, 1035481765, 1032594599, 1029705003, 1026812984, 1023918549, 1021021705, 1018122458, 1015220815, 
			1012316783, 1009410369, 1006501580, 1003590423, 1000676904, 997761030, 994842809, 991922247, 988999350, 986074126, 983146582, 980216725, 977284561, 974350097, 971413341, 968474299, 
			965532977, 962589384, 959643526, 956695410, 953745042, 950792430, 947837581, 944880502, 941921199, 938959680, 935995951, 933030020, 930061893, 927091578, 924119081, 921144410, 
			918167571, 915188571, 912207418, 909224119, 906238680, 903251109, 900261412, 897269597, 894275670, 891279639, 888281511, 885281292, 882278991, 879274613, 876268167, 873259658, 
			870249094, 867236483, 864221831, 861205146, 858186434, 855165702, 852142958, 849118210, 846091463, 843062725, 840032003, 836999305, 833964637, 830928006, 827889421, 824848887, 
			821806412, 818762004, 815715669, 812667415, 809617248, 806565176, 803511206, 800455345, 797397601, 794337981, 791276491, 788213140, 785147933, 782080880, 779011985, 775941258, 
			772868705, 769794333, 766718150, 763640163, 760560379, 757478805, 754395449, 751310317, 748223418, 745134758, 742044344, 738952185, 735858286, 732762657, 729665302, 726566231, 
			723465451, 720362967, 717258789, 714152923, 711045377, 707936157, 704825271, 701712727, 698598532, 695482693, 692365217, 689246113, 686125386, 683003045, 679879096, 676753548, 
			673626407, 670497682, 667367378, 664235504, 661102067, 657967075, 654830534, 651692452, 648552837, 645411696, 642269036, 639124864, 635979189, 632832017, 629683356, 626533214, 
			623381597, 620228513, 617073970, 613917975, 610760535, 607601658, 604441351, 601279622, 598116478, 594951926, 591785975, 588618631, 585449902, 582279796, 579108319, 575935479, 
			572761285, 569585742, 566408859, 563230644, 560051103, 556870244, 553688075, 550504603, 547319836, 544133780, 540946445, 537757836, 534567962, 531376830, 528184448, 524990823, 
			521795962, 518599874, 515402565, 512204044, 509004317, 505803393, 502601278, 499397981, 496193509, 492987869, 489781068, 486573116, 483364018, 480153783, 476942419, 473729932, 
			470516330, 467301621, 464085812, 460868912, 457650927, 454431865, 451211733, 447990540, 444768293, 441544999, 438320666, 435095302, 431868914, 428641510, 425413097, 422183683, 
			418953276, 415721883, 412489511, 409256169, 406021864, 402786603, 399550395, 396313247, 393075166, 389836160, 386596236, 383355403, 380113668, 376871039, 373627522, 370383127, 
			367137860, 363891729, 360644742, 357396906, 354148229, 350898719, 347648383, 344397229, 341145265, 337892497, 334638935, 331384586, 328129456, 324873555, 321616889, 318359466, 
			315101294, 311842381, 308582733, 305322360, 302061268, 298799466, 295536960, 292273759, 289009870, 285745301, 282480060, 279214154, 275947591, 272680379, 269412525, 266144037, 
			262874923, 259605190, 256334846, 253063900, 249792357, 246520227, 243247517, 239974234, 236700387, 233425983, 230151030, 226875534, 223599506, 220322950, 217045877, 213768293, 
			210490205, 207211623, 203932553, 200653003, 197372981, 194092494, 190811551, 187530158, 184248325, 180966058, 177683365, 174400253, 171116732, 167832808, 164548489, 161263783, 
			157978697, 154693239, 151407418, 148121240, 144834714, 141547847, 138260647, 134973121, 131685278, 128397125, 125108670, 121819920, 118530884, 115241569, 111951983, 108662133, 
			105372028, 102081674, 98791081, 95500255, 92209204, 88917936, 85626459, 82334781, 79042909, 75750851, 72458614, 69166207, 65873638, 62580913, 59288041, 55995030, 
			52701886, 49408619, 46115236, 42821744, 39528151, 36234465, 32940694, 29646846, 26352927, 23058947, 19764912, 16470831, 13176711, 9882561, 6588386, 3294197, 
			0, -3294197, -6588386, -9882561, -13176711, -16470831, -19764912, -23058947, -26352927, -29646846, -32940694, -36234465, -39528151, -42821744, -46115236, -49408619, 
			-52701886, -55995030, -59288041, -62580913, -65873638, -69166207, -72458614, -75750851, -79042909, -82334781, -85626459, -88917936, -92209204, -95500255, -98791081, -102081674, 
			-105372028, -108662133, -111951983, -115241569, -118530884, -121819920, -125108670, -128397125, -131685278, -134973121, -138260647, -141547847, -144834714, -148121240, -151407418, -154693239, 
			-157978697, -161263783, -164548489, -167832808, -171116732, -174400253, -177683365, -180966058, -184248325, -187530158, -190811551, -194092494, -197372981, -200653003, -203932553, -207211623, 
			-210490205, -213768293, -217045877, -220322950, -223599506, -226875534, -230151030, -233425983, -236700387, -239974234, -243247517, -246520227, -249792357, -253063900, -256334846, -259605190, 
			-262874923, -266144037, -269412525, -272680379, -275947591, -279214154, -282480060, -285745301, -289009870, -292273759, -295536960, -298799466, -302061268, -305322360, -308582733, -311842381, 
			-315101294, -318359466, -321616889, -324873555, -328129456, -331384586, -334638935, -337892497, -341145265, -344397229, -347648383, -350898719, -354148229, -357396906, -360644742, -363891729, 
			-367137860, -370383127, -373627522, -376871039, -380113668, -383355403, -386596236, -389836160, -393075166, -396313247, -399550395, -402786603, -406021864, -409256169, -412489511, -415721883, 
			-418953276, -422183683, -425413097, -428641510, -431868914, -435095302, -438320666, -441544999, -444768293, -447990540, -451211733, -454431865, -457650927, -460868912, -464085812, -467301621, 
			-470516330, -473729932, -476942419, -480153783, -483364018, -486573116, -489781068, -492987869, -496193509, -499397981, -502601278, -505803393, -509004317, -512204044, -515402565, -518599874, 
			-521795962, -524990823, -528184448, -531376830, -534567962, -537757836, -540946445, -544133780, -547319836, -550504603, -553688075, -556870244, -560051103, -563230644, -566408859, -569585742, 
			-572761285, -575935479, -579108319, -582279796, -585449902, -588618631, -591785975, -594951926, -598116478, -601279622, -604441351, -607601658, -610760535, -613917975, -617073970, -620228513, 
			-623381597, -626533214, -629683356, -632832017, -635979189, -639124864, -642269036, -645411696, -648552837, -651692452, -654830534, -657967075, -661102067, -664235504, -667367378, -670497682, 
			-673626407, -676753548, -679879096, -683003045, -686125386, -689246113, -692365217, -695482693, -698598532, -701712727, -704825271, -707936157, -711045377, -714152923, -717258789, -720362967, 
			-723465451, -726566231, -729665302, -732762657, -735858286, -738952185, -742044344, -745134758, -748223418, -751310317, -754395449, -757478805, -760560379, -763640163, -766718150, -769794333, 
			-772868705, -775941258, -779011985, -782080880, -785147933, -788213140, -791276491, -794337981, -797397601, -800455345, -803511206, -806565176, -809617248, -812667415, -815715669, -818762004, 
			-821806412, -824848887, -827889421, -830928006, -833964637, -836999305, -840032003, -843062725, -846091463, -849118210, -852142958, -855165702, -858186434, -861205146, -864221831, -867236483, 
			-870249094, -873259658, -876268167, -879274613, -882278991, -885281292, -888281511, -891279639, -894275670, -897269597, -900261412, -903251109, -906238680, -909224119, -912207418, -915188571, 
			-918167571, -921144410, -924119081, -927091578, -930061893, -933030020, -935995951, -938959680, -941921199, -944880502, -947837581, -950792430, -953745042, -956695410, -959643526, -962589384, 
			-965532977, -968474299, -971413341, -974350097, -977284561, -980216725, -983146582, -986074126, -988999350, -991922247, -994842809, -997761030, -1000676904, -1003590423, -1006501580, -1009410369, 
			-1012316783, -1015220815, -1018122458, -1021021705, -1023918549, -1026812984, -1029705003, -1032594599, -1035481765, -1038366494, -1041248781, -1044128616, -1047005996, -1049880911, -1052753356, -1055623323, 
			-1058490807, -1061355800, -1064218295, -1067078287, -1069935767, -1072790729, -1075643168, -1078493075, -1081340444, -1084185269, -1087027543, -1089867258, -1092704410, -1095538990, -1098370992, -1101200409, 
			-1104027236, -1106851464, -1109673088, -1112492100, -1115308495, -1118122266, -1120933405, -1123741907, -1126547764, -1129350971, -1132151520, -1134949405, -1137744620, -1140537157, -1143327010, -1146114173, 
			-1148898639, -1151680402, -1154459455, -1157235791, -1160009404, -1162780287, -1165548434, -1168313839, -1171076494, -1173836394, -1176593532, -1179347901, -1182099495, -1184848307, -1187594331, -1190337561, 
			-1193077990, -1195815611, -1198550418, -1201282406, -1204011566, -1206737893, -1209461381, -1212182022, -1214899812, -1217614742, -1220326808, -1223036001, -1225742317, -1228445749, -1231146290, -1233843934, 
			-1236538674, -1239230505, -1241919420, -1244605413, -1247288476, -1249968605, -1252645793, -1255320033, -1257991319, -1260659645, -1263325004, -1265987391, -1268646799, -1271303221, -1273956652, -1276607085, 
			-1279254514, -1281898933, -1284540336, -1287178716, -1289814067, -1292446383, -1295075658, -1297701885, -1300325059, -1302945173, -1305562221, -1308176197, -1310787094, -1313394908, -1315999630, -1318601256, 
			-1321199779, -1323795194, -1326387493, -1328976671, -1331562722, -1334145640, -1336725418, -1339302051, -1341875532, -1344445856, -1347013016, -1349577006, -1352137821, -1354695454, -1357249900, -1359801151, 
			-1362349203, -1364894049, -1367435684, -1369974100, -1372509293, -1375041257, -1377569984, -1380095471, -1382617709, -1385136695, -1387652420, -1390164881, -1392674071, -1395179983, -1397682612, -1400181953, 
			-1402677998, -1405170743, -1407660182, -1410146308, -1412629116, -1415108600, -1417584754, -1420057573, -1422527049, -1424993179, -1427455955, -1429915373, -1432371425, -1434824107, -1437273413, -1439719337, 
			-1442161873, -1444601016, -1447036759, -1449469097, -1451898024, -1454323535, -1456745624, -1459164285, -1461579512, -1463991301, -1466399644, -1468804536, -1471205973, -1473603947, -1475998454, -1478389488, 
			-1480777043, -1483161114, -1485541694, -1487918779, -1490292363, -1492662440, -1495029005, -1497392051, -1499751575, -1502107569, -1504460028, -1506808947, -1509154321, -1511496144, -1513834409, -1516169113, 
			-1518500249, -1520827811, -1523151796, -1525472195, -1527789006, -1530102221, -1532411836, -1534717845, -1537020242, -1539319023, -1541614182, -1543905713, -1546193611, -1548477871, -1550758487, -1553035454, 
			-1555308767, -1557578419, -1559844407, -1562106724, -1564365365, -1566620326, -1568871600, -1571119182, -1573363067, -1575603250, -1577839725, -1580072488, -1582301532, -1584526853, -1586748446, -1588966305, 
			-1591180424, -1593390800, -1595597426, -1597800298, -1599999410, -1602194757, -1604386334, -1606574135, -1608758156, -1610938392, -1613114836, -1615287485, -1617456334, -1619621376, -1621782607, -1623940021, 
			-1626093615, -1628243382, -1630389317, -1632531417, -1634669674, -1636804086, -1638934645, -1641061348, -1643184190, -1645303165, -1647418268, -1649529495, -1651636840, -1653740299, -1655839866, -1657935537, 
			-1660027307, -1662115171, -1664199123, -1666279160, -1668355275, -1670427465, -1672495724, -1674560047, -1676620430, -1678676868, -1680729356, -1682777889, -1684822462, -1686863071, -1688899710, -1690932375, 
			-1692961061, -1694985763, -1697006477, -1699023198, -1701035921, -1703044641, -1705049354, -1707050054, -1709046738, -1711039400, -1713028036, -1715012641, -1716993210, -1718969739, -1720942223, -1722910658, 
			-1724875039, -1726835360, -1728791618, -1730743809, -1732691926, -1734635967, -1736575926, -1738511798, -1740443579, -1742371265, -1744294851, -1746214333, -1748129706, -1750040965, -1751948106, -1753851124, 
			-1755750016, -1757644776, -1759535400, -1761421884, -1763304223, -1765182413, -1767056449, -1768926327, -1770792043, -1772653592, -1774510969, -1776364171, -1778213193, -1780058031, -1781898680, -1783735136, 
			-1785567395, -1787395452, -1789219303, -1791038944, -1792854371, -1794665579, -1796472564, -1798275321, -1800073847, -1801868137, -1803658188, -1805443994, -1807225552, -1809002857, -1810775905, -1812544693, 
			-1814309215, -1816069468, -1817825448, -1819577150, -1821324571, -1823067705, -1824806550, -1826541101, -1828271354, -1829997305, -1831718950, -1833436285, -1835149305, -1836858007, -1838562387, -1840262440, 
			-1841958163, -1843649552, -1845336602, -1847019310, -1848697672, -1850371684, -1852041342, -1853706642, -1855367579, -1857024151, -1858676353, -1860324182, -1861967633, -1863606703, -1865241387, -1866871682, 
			-1868497584, -1870119090, -1871736195, -1873348896, -1874957188, -1876561068, -1878160533, -1879755578, -1881346200, -1882932395, -1884514160, -1886091490, -1887664381, -1889232831, -1890796835, -1892356390, 
			-1893911493, -1895462138, -1897008324, -1898550045, -1900087299, -1901620082, -1903148391, -1904672221, -1906191569, -1907706431, -1909216805, -1910722686, -1912224071, -1913720956, -1915213339, -1916701214, 
			-1918184579, -1919663431, -1921137766, -1922607579, -1924072869, -1925533631, -1926989863, -1928441559, -1929888719, -1931331336, -1932769410, -1934202935, -1935631909, -1937056328, -1938476189, -1939891489, 
			-1941302224, -1942708390, -1944109986, -1945507007, -1946899449, -1948287311, -1949670588, -1951049277, -1952423376, -1953792880, -1955157786, -1956518092, -1957873794, -1959224889, -1960571374, -1961913245, 
			-1963250500, -1964583135, -1965911147, -1967234533, -1968553290, -1969867415, -1971176905, -1972481756, -1973781966, -1975077531, -1976368449, -1977654716, -1978936329, -1980213286, -1981485584, -1982753218, 
			-1984016187, -1985274488, -1986528117, -1987777071, -1989021348, -1990260945, -1991495858, -1992726085, -1993951623, -1995172470, -1996388621, -1997600075, -1998806828, -2000008877, -2001206221, -2002398855, 
			-2003586778, -2004769986, -2005948476, -2007122247, -2008291294, -2009455616, -2010615209, -2011770071, -2012920199, -2014065591, -2015206243, -2016342154, -2017473319, -2018599738, -2019721406, -2020838322, 
			-2021950482, -2023057885, -2024160527, -2025258407, -2026351520, -2027439866, -2028523440, -2029602242, -2030676267, -2031745514, -2032809981, -2033869663, -2034924560, -2035974669, -2037019987, -2038060511, 
			-2039096240, -2040127170, -2041153300, -2042174627, -2043191148, -2044202862, -2045209765, -2046211856, -2047209132, -2048201590, -2049189229, -2050172046, -2051150039, -2052123205, -2053091543, -2054055049, 
			-2055013722, -2055967559, -2056916558, -2057860717, -2058800034, -2059734506, -2060664132, -2061588909, -2062508834, -2063423906, -2064334123, -2065239482, -2066139982, -2067035619, -2067926393, -2068812301, 
			-2069693340, -2070569510, -2071440807, -2072307229, -2073168776, -2074025444, -2074877232, -2075724137, -2076566158, -2077403293, -2078235539, -2079062895, -2079885359, -2080702928, -2081515602, -2082323377, 
			-2083126253, -2083924227, -2084717297, -2085505461, -2086288718, -2087067066, -2087840503, -2088609027, -2089372636, -2090131329, -2090885104, -2091633958, -2092377891, -2093116900, -2093850983, -2094580140, 
			-2095304368, -2096023666, -2096738031, -2097447462, -2098151959, -2098851517, -2099546137, -2100235817, -2100920555, -2101600349, -2102275197, -2102945099, -2103610052, -2104270056, -2104925108, -2105575206, 
			-2106220350, -2106860538, -2107495769, -2108126040, -2108751350, -2109371699, -2109987084, -2110597504, -2111202957, -2111803443, -2112398959, -2112989505, -2113575078, -2114155678, -2114731304, -2115301953, 
			-2115867624, -2116428317, -2116984030, -2117534761, -2118080509, -2118621274, -2119157053, -2119687845, -2120213650, -2120734465, -2121250290, -2121761124, -2122266965, -2122767812, -2123263664, -2123754520, 
			-2124240379, -2124721239, -2125197099, -2125667958, -2126133816, -2126594670, -2127050521, -2127501366, -2127947205, -2128388037, -2128823860, -2129254674, -2129680478, -2130101271, -2130517051, -2130927817, 
			-2131333570, -2131734307, -2132130028, -2132520732, -2132906418, -2133287086, -2133662733, -2134033359, -2134398964, -2134759547, -2135115106, -2135465641, -2135811151, -2136151635, -2136487093, -2136817524, 
			-2137142926, -2137463299, -2137778643, -2138088956, -2138394238, -2138694489, -2138989706, -2139279891, -2139565041, -2139845157, -2140120238, -2140390283, -2140655291, -2140915262, -2141170196, -2141420091, 
			-2141664947, -2141904763, -2142139539, -2142369275, -2142593969, -2142813622, -2143028233, -2143237800, -2143442325, -2143641805, -2143836242, -2144025634, -2144209981, -2144389282, -2144563537, -2144732746, 
			-2144896908, -2145056023, -2145210091, -2145359111, -2145503082, -2145642005, -2145775879, -2145904703, -2146028478, -2146147204, -2146260879, -2146369504, -2146473078, -2146571602, -2146665074, -2146753495, 
			-2146836865, -2146915182, -2146988448, -2147056662, -2147119824, -2147177933, -2147230990, -2147278994, -2147321945, -2147359843, -2147392689, -2147420481, -2147443221, -2147460907, -2147473540, -2147481120, 
			-2147483647, -2147481120, -2147473540, -2147460907, -2147443221, -2147420481, -2147392689, -2147359843, -2147321945, -2147278994, -2147230990, -2147177933, -2147119824, -2147056662, -2146988448, -2146915182, 
			-2146836865, -2146753495, -2146665074, -2146571602, -2146473078, -2146369504, -2146260879, -2146147204, -2146028478, -2145904703, -2145775879, -2145642005, -2145503082, -2145359111, -2145210091, -2145056023, 
			-2144896908, -2144732746, -2144563537, -2144389282, -2144209981, -2144025634, -2143836242, -2143641805, -2143442325, -2143237800, -2143028233, -2142813622, -2142593969, -2142369275, -2142139539, -2141904763, 
			-2141664947, -2141420091, -2141170196, -2140915262, -2140655291, -2140390283, -2140120238, -2139845157, -2139565041, -2139279891, -2138989706, -2138694489, -2138394238, -2138088956, -2137778643, -2137463299, 
			-2137142926, -2136817524, -2136487093, -2136151635, -2135811151, -2135465641, -2135115106, -2134759547, -2134398964, -2134033359, -2133662733, -2133287086, -2132906418, -2132520732, -2132130028, -2131734307, 
			-2131333570, -2130927817, -2130517051, -2130101271, -2129680478, -2129254674, -2128823860, -2128388037, -2127947205, -2127501366, -2127050521, -2126594670, -2126133816, -2125667958, -2125197099, -2124721239, 
			-2124240379, -2123754520, -2123263664, -2122767812, -2122266965, -2121761124, -2121250290, -2120734465, -2120213650, -2119687845, -2119157053, -2118621274, -2118080509, -2117534761, -2116984030, -2116428317, 
			-2115867624, -2115301953, -2114731304, -2114155678, -2113575078, -2112989505, -2112398959, -2111803443, -2111202957, -2110597504, -2109987084, -2109371699, -2108751350, -2108126040, -2107495769, -2106860538, 
			-2106220350, -2105575206, -2104925108, -2104270056, -2103610052, -2102945099, -2102275197, -2101600349, -2100920555, -2100235817, -2099546137, -2098851517, -2098151959, -2097447462, -2096738031, -2096023666, 
			-2095304368, -2094580140, -2093850983, -2093116900, -2092377891, -2091633958, -2090885104, -2090131329, -2089372636, -2088609027, -2087840503, -2087067066, -2086288718, -2085505461, -2084717297, -2083924227, 
			-2083126253, -2082323377, -2081515602, -2080702928, -2079885359, -2079062895, -2078235539, -2077403293, -2076566158, -2075724137, -2074877232, -2074025444, -2073168776, -2072307229, -2071440807, -2070569510, 
			-2069693340, -2068812301, -2067926393, -2067035619, -2066139982, -2065239482, -2064334123, -2063423906, -2062508834, -2061588909, -2060664132, -2059734506, -2058800034, -2057860717, -2056916558, -2055967559, 
			-2055013722, -2054055049, -2053091543, -2052123205, -2051150039, -2050172046, -2049189229, -2048201590, -2047209132, -2046211856, -2045209765, -2044202862, -2043191148, -2042174627, -2041153300, -2040127170, 
			-2039096240, -2038060511, -2037019987, -2035974669, -2034924560, -2033869663, -2032809981, -2031745514, -2030676267, -2029602242, -2028523440, -2027439866, -2026351520, -2025258407, -2024160527, -2023057885, 
			-2021950482, -2020838322, -2019721406, -2018599738, -2017473319, -2016342154, -2015206243, -2014065591, -2012920199, -2011770071, -2010615209, -2009455616, -2008291294, -2007122247, -2005948476, -2004769986, 
			-2003586778, -2002398855, -2001206221, -2000008877, -1998806828, -1997600075, -1996388621, -1995172470, -1993951623, -1992726085, -1991495858, -1990260945, -1989021348, -1987777071, -1986528117, -1985274488, 
			-1984016187, -1982753218, -1981485584, -1980213286, -1978936329, -1977654716, -1976368449, -1975077531, -1973781966, -1972481756, -1971176905, -1969867415, -1968553290, -1967234533, -1965911147, -1964583135, 
			-1963250500, -1961913245, -1960571374, -1959224889, -1957873794, -1956518092, -1955157786, -1953792880, -1952423376, -1951049277, -1949670588, -1948287311, -1946899449, -1945507007, -1944109986, -1942708390, 
			-1941302224, -1939891489, -1938476189, -1937056328, -1935631909, -1934202935, -1932769410, -1931331336, -1929888719, -1928441559, -1926989863, -1925533631, -1924072869, -1922607579, -1921137766, -1919663431, 
			-1918184579, -1916701214, -1915213339, -1913720956, -1912224071, -1910722686, -1909216805, -1907706431, -1906191569, -1904672221, -1903148391, -1901620082, -1900087299, -1898550045, -1897008324, -1895462138, 
			-1893911493, -1892356390, -1890796835, -1889232831, -1887664381, -1886091490, -1884514160, -1882932395, -1881346200, -1879755578, -1878160533, -1876561068, -1874957188, -1873348896, -1871736195, -1870119090, 
			-1868497584, -1866871682, -1865241387, -1863606703, -1861967633, -1860324182, -1858676353, -1857024151, -1855367579, -1853706642, -1852041342, -1850371684, -1848697672, -1847019310, -1845336602, -1843649552, 
			-1841958163, -1840262440, -1838562387, -1836858007, -1835149305, -1833436285, -1831718950, -1829997305, -1828271354, -1826541101, -1824806550, -1823067705, -1821324571, -1819577150, -1817825448, -1816069468, 
			-1814309215, -1812544693, -1810775905, -1809002857, -1807225552, -1805443994, -1803658188, -1801868137, -1800073847, -1798275321, -1796472564, -1794665579, -1792854371, -1791038944, -1789219303, -1787395452, 
			-1785567395, -1783735136, -1781898680, -1780058031, -1778213193, -1776364171, -1774510969, -1772653592, -1770792043, -1768926327, -1767056449, -1765182413, -1763304223, -1761421884, -1759535400, -1757644776, 
			-1755750016, -1753851124, -1751948106, -1750040965, -1748129706, -1746214333, -1744294851, -1742371265, -1740443579, -1738511798, -1736575926, -1734635967, -1732691926, -1730743809, -1728791618, -1726835360, 
			-1724875039, -1722910658, -1720942223, -1718969739, -1716993210, -1715012641, -1713028036, -1711039400, -1709046738, -1707050054, -1705049354, -1703044641, -1701035921, -1699023198, -1697006477, -1694985763, 
			-1692961061, -1690932375, -1688899710, -1686863071, -1684822462, -1682777889, -1680729356, -1678676868, -1676620430, -1674560047, -1672495724, -1670427465, -1668355275, -1666279160, -1664199123, -1662115171, 
			-1660027307, -1657935537, -1655839866, -1653740299, -1651636840, -1649529495, -1647418268, -1645303165, -1643184190, -1641061348, -1638934645, -1636804086, -1634669674, -1632531417, -1630389317, -1628243382, 
			-1626093615, -1623940021, -1621782607, -1619621376, -1617456334, -1615287485, -1613114836, -1610938392, -1608758156, -1606574135, -1604386334, -1602194757, -1599999410, -1597800298, -1595597426, -1593390800, 
			-1591180424, -1588966305, -1586748446, -1584526853, -1582301532, -1580072488, -1577839725, -1575603250, -1573363067, -1571119182, -1568871600, -1566620326, -1564365365, -1562106724, -1559844407, -1557578419, 
			-1555308767, -1553035454, -1550758487, -1548477871, -1546193611, -1543905713, -1541614182, -1539319023, -1537020242, -1534717845, -1532411836, -1530102221, -1527789006, -1525472195, -1523151796, -1520827811, 
			-1518500249, -1516169113, -1513834409, -1511496144, -1509154321, -1506808947, -1504460028, -1502107569, -1499751575, -1497392051, -1495029005, -1492662440, -1490292363, -1487918779, -1485541694, -1483161114, 
			-1480777043, -1478389488, -1475998454, -1473603947, -1471205973, -1468804536, -1466399644, -1463991301, -1461579512, -1459164285, -1456745624, -1454323535, -1451898024, -1449469097, -1447036759, -1444601016, 
			-1442161873, -1439719337, -1437273413, -1434824107, -1432371425, -1429915373, -1427455955, -1424993179, -1422527049, -1420057573, -1417584754, -1415108600, -1412629116, -1410146308, -1407660182, -1405170743, 
			-1402677998, -1400181953, -1397682612, -1395179983, -1392674071, -1390164881, -1387652420, -1385136695, -1382617709, -1380095471, -1377569984, -1375041257, -1372509293, -1369974100, -1367435684, -1364894049, 
			-1362349203, -1359801151, -1357249900, -1354695454, -1352137821, -1349577006, -1347013016, -1344445856, -1341875532, -1339302051, -1336725418, -1334145640, -1331562722, -1328976671, -1326387493, -1323795194, 
			-1321199779, -1318601256, -1315999630, -1313394908, -1310787094, -1308176197, -1305562221, -1302945173, -1300325059, -1297701885, -1295075658, -1292446383, -1289814067, -1287178716, -1284540336, -1281898933, 
			-1279254514, -1276607085, -1273956652, -1271303221, -1268646799, -1265987391, -1263325004, -1260659645, -1257991319, -1255320033, -1252645793, -1249968605, -1247288476, -1244605413, -1241919420, -1239230505, 
			-1236538674, -1233843934, -1231146290, -1228445749, -1225742317, -1223036001, -1220326808, -1217614742, -1214899812, -1212182022, -1209461381, -1206737893, -1204011566, -1201282406, -1198550418, -1195815611, 
			-1193077990, -1190337561, -1187594331, -1184848307, -1182099495, -1179347901, -1176593532, -1173836394, -1171076494, -1168313839, -1165548434, -1162780287, -1160009404, -1157235791, -1154459455, -1151680402, 
			-1148898639, -1146114173, -1143327010, -1140537157, -1137744620, -1134949405, -1132151520, -1129350971, -1126547764, -1123741907, -1120933405, -1118122266, -1115308495, -1112492100, -1109673088, -1106851464, 
			-1104027236, -1101200409, -1098370992, -1095538990, -1092704410, -1089867258, -1087027543, -1084185269, -1081340444, -1078493075, -1075643168, -1072790729, -1069935767, -1067078287, -1064218295, -1061355800, 
			-1058490807, -1055623323, -1052753356, -1049880911, -1047005996, -1044128616, -1041248781, -1038366494, -1035481765, -1032594599, -1029705003, -1026812984, -1023918549, -1021021705, -1018122458, -1015220815, 
			-1012316783, -1009410369, -1006501580, -1003590423, -1000676904, -997761030, -994842809, -991922247, -988999350, -986074126, -983146582, -980216725, -977284561, -974350097, -971413341, -968474299, 
			-965532977, -962589384, -959643526, -956695410, -953745042, -950792430, -947837581, -944880502, -941921199, -938959680, -935995951, -933030020, -930061893, -927091578, -924119081, -921144410, 
			-918167571, -915188571, -912207418, -909224119, -906238680, -903251109, -900261412, -897269597, -894275670, -891279639, -888281511, -885281292, -882278991, -879274613, -876268167, -873259658, 
			-870249094, -867236483, -864221831, -861205146, -858186434, -855165702, -852142958, -849118210, -846091463, -843062725, -840032003, -836999305, -833964637, -830928006, -827889421, -824848887, 
			-821806412, -818762004, -815715669, -812667415, -809617248, -806565176, -803511206, -800455345, -797397601, -794337981, -791276491, -788213140, -785147933, -782080880, -779011985, -775941258, 
			-772868705, -769794333, -766718150, -763640163, -760560379, -757478805, -754395449, -751310317, -748223418, -745134758, -742044344, -738952185, -735858286, -732762657, -729665302, -726566231, 
			-723465451, -720362967, -717258789, -714152923, -711045377, -707936157, -704825271, -701712727, -698598532, -695482693, -692365217, -689246113, -686125386, -683003045, -679879096, -676753548, 
			-673626407, -670497682, -667367378, -664235504, -661102067, -657967075, -654830534, -651692452, -648552837, -645411696, -642269036, -639124864, -635979189, -632832017, -629683356, -626533214, 
			-623381597, -620228513, -617073970, -613917975, -610760535, -607601658, -604441351, -601279622, -598116478, -594951926, -591785975, -588618631, -585449902, -582279796, -579108319, -575935479, 
			-572761285, -569585742, -566408859, -563230644, -560051103, -556870244, -553688075, -550504603, -547319836, -544133780, -540946445, -537757836, -534567962, -531376830, -528184448, -524990823, 
			-521795962, -518599874, -515402565, -512204044, -509004317, -505803393, -502601278, -499397981, -496193509, -492987869, -489781068, -486573116, -483364018, -480153783, -476942419, -473729932, 
			-470516330, -467301621, -464085812, -460868912, -457650927, -454431865, -451211733, -447990540, -444768293, -441544999, -438320666, -435095302, -431868914, -428641510, -425413097, -422183683, 
			-418953276, -415721883, -412489511, -409256169, -406021864, -402786603, -399550395, -396313247, -393075166, -389836160, -386596236, -383355403, -380113668, -376871039, -373627522, -370383127, 
			-367137860, -363891729, -360644742, -357396906, -354148229, -350898719, -347648383, -344397229, -341145265, -337892497, -334638935, -331384586, -328129456, -324873555, -321616889, -318359466, 
			-315101294, -311842381, -308582733, -305322360, -302061268, -298799466, -295536960, -292273759, -289009870, -285745301, -282480060, -279214154, -275947591, -272680379, -269412525, -266144037, 
			-262874923, -259605190, -256334846, -253063900, -249792357, -246520227, -243247517, -239974234, -236700387, -233425983, -230151030, -226875534, -223599506, -220322950, -217045877, -213768293, 
			-210490205, -207211623, -203932553, -200653003, -197372981, -194092494, -190811551, -187530158, -184248325, -180966058, -177683365, -174400253, -171116732, -167832808, -164548489, -161263783, 
			-157978697, -154693239, -151407418, -148121240, -144834714, -141547847, -138260647, -134973121, -131685278, -128397125, -125108670, -121819920, -118530884, -115241569, -111951983, -108662133, 
			-105372028, -102081674, -98791081, -95500255, -92209204, -88917936, -85626459, -82334781, -79042909, -75750851, -72458614, -69166207, -65873638, -62580913, -59288041, -55995030, 
			-52701886, -49408619, -46115236, -42821744, -39528151, -36234465, -32940694, -29646846, -26352927, -23058947, -19764912, -16470831, -13176711, -9882561, -6588386, -3294197 };

const fixed flattop_window[2049] = { 0, -99, -403, -910, -1620, -2532, -3648, -4966, -6488, -8213, -10141, -12272, -14607, -17146, -19889, -22836,
			-25987, -29343, -32904, -36670, -40641, -44817, -49200, -53788, -58583, -63585, -68795, -74211, -79836, -85669, -91711, -97961,
			-104422, -111092, -117973, -125065, -132368, -139883, -147610, -155551, -163705, -172073, -180656, -189454, -198468, -207698, -217145, -226810,
			-236694, -246796, -257118, -267660, -278424, -289409, -300617, -312049, -323704, -335584, -347690, -360022, -372582, -385370, -398386, -411632,
			-425109, -438818, -452759, -466933, -481341, -495985, -510865, -525982, -541337, -556931, -572765, -588840, -605157, -621717, -638522, -655571,
			-672867, -690411, -708202, -726244, -744536, -763080, -781877, -800929, -820235, -839798, -859619, -879699, -900039, -920641, -941505, -962633,
			-984026, -1005686, -1027613, -1049810, -1072277, -1095015, -1118027, -1141313, -1164875, -1188714, -1212832, -1237229, -1261908, -1286870, -1312116, -1337647,
			-1363466, -1389573, -1415970, -1442659, -1469641, -1496918, -1524490, -1552360, -1580529, -1608999, -1637771, -1666847, -1696229, -1725917, -1755914, -1786221,
			-1816841, -1847773, -1879021, -1910585, -1942468, -1974671, -2007196, -2040044, -2073218, -2106718, -2140547, -2174706, -2209198, -2244023, -2279184, -2314682,
			-2350520, -2386698, -2423219, -2460085, -2497297, -2534857, -2572767, -2611029, -2649645, -2688616, -2727944, -2767632, -2807681, -2848093, -2888870, -2930014,
			-2971526, -3013409, -3055665, -3098295, -3141302, -3184687, -3228452, -3272600, -3317132, -3362050, -3407357, -3453053, -3499142, -3545626, -3592505, -3639783,
			-3687461, -3735541, -3784026, -3832917, -3882217, -3931927, -3982049, -4032587, -4083541, -4134913, -4186707, -4238924, -4291565, -4344634, -4398132, -4452061,
			-4506424, -4561223, -4616459, -4672135, -4728253, -4784815, -4841823, -4899280, -4957187, -5015547, -5074362, -5133634, -5193366, -5253558, -5314214, -5375336,
			-5436926, -5498986, -5561518, -5624525, -5688008, -5751970, -5816414, -5881340, -5946752, -6012652, -6079042, -6145923, -6213300, -6281172, -6349544, -6418416,
			-6487792, -6557673, -6628062, -6698960, -6770371, -6842296, -6914738, -6987698, -7061180, -7135185, -7209715, -7284773, -7360362, -7436482, -7513137, -7590329,
			-7668059, -7746331, -7825146, -7904507, -7984415, -8064874, -8145886, -8227451, -8309574, -8392256, -8475499, -8559305, -8643678, -8728618, -8814129, -8900212,
			-8986869, -9074104, -9161917, -9250312, -9339291, -9428855, -9519007, -9609749, -9701084, -9793013, -9885539, -9978664, -10072391, -10166720, -10261655, -10357198,
			-10453351, -10550115, -10647494, -10745490, -10844104, -10943338, -11043196, -11143678, -11244788, -11346527, -11448898, -11551902, -11655542, -11759819, -11864737, -11970296,
			-12076500, -12183350, -12290848, -12398997, -12507798, -12617254, -12727367, -12838138, -12949570, -13061664, -13174424, -13287850, -13401945, -13516711, -13632150, -13748264,
			-13865055, -13982525, -14100675, -14219509, -14339027, -14459232, -14580126, -14701710, -14823987, -14946959, -15070626, -15194993, -15320059, -15445827, -15572299, -15699478,
			-15827363, -15955959, -16085265, -16215285, -16346020, -16477471, -16609641, -16742531, -16876144, -17010480, -17145542, -17281332, -17417850, -17555098, -17693080, -17831796,
			-17971248, -18111436, -18252366, -18394034, -18536446, -18679602, -18823502, -18968152, -19113548, -19259696, -19406596, -19554248, -19702656, -19851822, -20001744, -20152426,
			-20303868, -20456074, -20609044, -20762778, -20917278, -21072548, -21228586, -21385396, -21542978, -21701332, -21860462, -22020368, -22181050, -22342512, -22504754, -22667778,
			-22831582, -22996172, -23161546, -23327704, -23494652, -23662386, -23830910, -24000224, -24170330, -24341230, -24512922, -24685408, -24858692, -25032770, -25207648, -25383322,
			-25559798, -25737074, -25915150, -26094030, -26273712, -26454198, -26635490, -26817586, -27000490, -27184200, -27368718, -27554044, -27740180, -27927126, -28114884, -28303452,
			-28492832, -28683024, -28874030, -29065850, -29258484, -29451932, -29646196, -29841276, -30037170, -30233884, -30431412, -30629760, -30828924, -31028906, -31229708, -31431326,
			-31633766, -31837024, -32041100, -32245998, -32451714, -32658252, -32865608, -33073786, -33282784, -33492602, -33703240, -33914700, -34126976, -34340076, -34553996, -34768736,
			-34984296, -35200680, -35417880, -35635896, -35854736, -36074396, -36294872, -36516168, -36738284, -36961216, -37184964, -37409532, -37634912, -37861112, -38088128, -38315960,
			-38544604, -38774064, -39004340, -39235424, -39467324, -39700036, -39933556, -40167888, -40403028, -40638976, -40875736, -41113300, -41351672, -41590848, -41830828, -42071612,
			-42313200, -42555588, -42798776, -43042760, -43287548, -43533128, -43779508, -44026680, -44274648, -44523408, -44772956, -45023296, -45274424, -45526336, -45779036, -46032520,
			-46286784, -46541832, -46797660, -47054264, -47311644, -47569800, -47828728, -48088428, -48348896, -48610136, -48872136, -49134904, -49398436, -49662724, -49927772, -50193580,
			-50460140, -50727452, -50995516, -51264324, -51533884, -51804184, -52075228, -52347012, -52619532, -52892788, -53166780, -53441496, -53716944, -53993120, -54270016, -54547632,
			-54825968, -55105020, -55384780, -55665256, -55946436, -56228324, -56510912, -56794200, -57078188, -57362864, -57648236, -57934292, -58221036, -58508460, -58796564, -59085340,
			-59374792, -59664916, -59955704, -60247156, -60539268, -60832032, -61125456, -61419528, -61714244, -62009604, -62305604, -62602244, -62899512, -63197408, -63495932, -63795080,
			-64094844, -64395220, -64696208, -64997804, -65300000, -65602796, -65906188, -66210168, -66514740, -66819892, -67125624, -67431928, -67738800, -68046248, -68354248, -68662816,
			-68971928, -69281592, -69591808, -69902552, -70213840, -70525656, -70838000, -71150864, -71464248, -71778144, -72092552, -72407456, -72722864, -73038760, -73355152, -73672024,
			-73989376, -74307200, -74625488, -74944248, -75263464, -75583128, -75903248, -76223808, -76544808, -76866240, -77188096, -77510376, -77833072, -78156176, -78479688, -78803592,
			-79127896, -79452592, -79777664, -80103120, -80428936, -80755128, -81081672, -81408576, -81735824, -82063408, -82391336, -82719584, -83048160, -83377056, -83706264, -84035768,
			-84365576, -84695672, -85026064, -85356728, -85687664, -86018864, -86350328, -86682040, -87014000, -87346200, -87678640, -88011296, -88344176, -88677272, -89010568, -89344064,
			-89677752, -90011624, -90345680, -90679896, -91014288, -91348824, -91683520, -92018352, -92353320, -92688408, -93023624, -93358952, -93694384, -94029912, -94365528, -94701224,
			-95037000, -95372840, -95708744, -96044696, -96380688, -96716720, -97052776, -97388856, -97724944, -98061040, -98397128, -98733200, -99069256, -99405280, -99741272, -100077208,
			-100413096, -100748928, -101084680, -101420360, -101755944, -102091440, -102426824, -102762096, -103097248, -103432272, -103767152, -104101880, -104436456, -104770864, -105105096, -105439144,
			-105773000, -106106656, -106440096, -106773312, -107106304, -107439056, -107771560, -108103800, -108435784, -108767480, -109098896, -109430016, -109760832, -110091328, -110421504, -110751344,
			-111080840, -111409984, -111738768, -112067176, -112395200, -112722832, -113050064, -113376880, -113703272, -114029232, -114354744, -114679808, -115004408, -115328536, -115652176, -115975320,
			-116297960, -116620088, -116941688, -117262752, -117583272, -117903232, -118222624, -118541440, -118859664, -119177280, -119494296, -119810688, -120126440, -120441560, -120756016, -121069808,
			-121382928, -121695352, -122007088, -122318104, -122628408, -122937968, -123246792, -123554856, -123862160, -124168680, -124474416, -124779344, -125083464, -125386760, -125689224, -125990840,
			-126291592, -126591472, -126890480, -127188584, -127485784, -127782072, -128077424, -128371832, -128665296, -128957784, -129249304, -129539824, -129829352, -130117856, -130405336, -130691784,
			-130977176, -131261504, -131544760, -131826920, -132107992, -132387936, -132666768, -132944456, -133220992, -133496368, -133770568, -134043576, -134315376, -134585984, -134855344, -135123472,
			-135390352, -135655952, -135920288, -136183328, -136445056, -136705472, -136964560, -137222304, -137478688, -137733712, -137987344, -138239584, -138490400, -138739808, -138987776, -139234288,
			-139479344, -139722928, -139965008, -140205584, -140444656, -140682192, -140918176, -141152608, -141385472, -141616736, -141846416, -142074480, -142300912, -142525712, -142748848, -142970320,
			-143190096, -143408192, -143624576, -143839232, -144052144, -144263312, -144472720, -144680336, -144886160, -145090176, -145292368, -145492720, -145691216, -145887856, -146082608, -146275472,
			-146466416, -146655440, -146842528, -147027664, -147210832, -147392016, -147571216, -147748400, -147923552, -148096656, -148267728, -148436704, -148603616, -148768416, -148931120, -149091680,
			-149250096, -149406368, -149560464, -149712368, -149862064, -150009552, -150154800, -150297792, -150438544, -150577008, -150713168, -150847040, -150978576, -151107776, -151234624, -151359104,
			-151481200, -151600912, -151718192, -151833056, -151945456, -152055424, -152162896, -152267888, -152370368, -152470336, -152567760, -152662640, -152754944, -152844672, -152931808, -153016320,
			-153098208, -153177456, -153254032, -153327936, -153399168, -153467680, -153533472, -153596512, -153656816, -153714352, -153769104, -153821056, -153870192, -153916480, -153959936, -154000528,
			-154038256, -154073072, -154104992, -154133968, -154160016, -154183104, -154203216, -154220352, -154234480, -154245584, -154253648, -154258672, -154260608, -154259472, -154255248, -154247904,
			-154237424, -154223792, -154207008, -154187040, -154163872, -154137504, -154107904, -154075072, -154038976, -153999600, -153956944, -153910976, -153861680, -153809056, -153753072, -153693728,
			-153630976, -153564848, -153495296, -153422304, -153345856, -153265952, -153182576, -153095680, -153005280, -152911360, -152813888, -152712848, -152608240, -152500032, -152388224, -152272784,
			-152153696, -152030960, -151904544, -151774448, -151640640, -151503104, -151361840, -151216816, -151068032, -150915456, -150759088, -150598896, -150434864, -150266992, -150095248, -149919632,
			-149740112, -149556672, -149369312, -149178016, -148982752, -148783504, -148580272, -148373024, -148161760, -147946464, -147727104, -147503664, -147276144, -147044512, -146808768, -146568896,
			-146324864, -146076672, -145824288, -145567712, -145306928, -145041904, -144772656, -144499120, -144221328, -143939232, -143652832, -143362096, -143067040, -142767616, -142463824, -142155648,
			-141843072, -141526080, -141204656, -140878784, -140548448, -140213632, -139874320, -139530496, -139182144, -138829264, -138471824, -138109808, -137743200, -137372000, -136996176, -136615728,
			-136230624, -135840864, -135446416, -135047280, -134643440, -134234864, -133821560, -133403496, -132980672, -132553056, -132120640, -131683416, -131241360, -130794456, -130342696, -129886064,
			-129424544, -128958120, -128486776, -128010504, -127529280, -127043096, -126551928, -126055776, -125554616, -125048440, -124537224, -124020960, -123499632, -122973224, -122441720, -121905112,
			-121363384, -120816512, -120264496, -119707320, -119144960, -118577400, -118004640, -117426664, -116843448, -116254976, -115661248, -115062248, -114457944, -113848344, -113233424, -112613168,
			-111987568, -111356608, -110720280, -110078560, -109431432, -108778896, -108120936, -107457528, -106788664, -106114336, -105434528, -104749224, -104058408, -103362072, -102660208, -101952792,
			-101239816, -100521264, -99797128, -99067384, -98332040, -97591064, -96844448, -96092184, -95334264, -94570656, -93801368, -93026376, -92245664, -91459232, -90667064, -89869136,
			-89065456, -88255992, -87440744, -86619696, -85792832, -84960152, -84121632, -83277264, -82427032, -81570936, -80708952, -79841080, -78967296, -78087592, -77201960, -76310392,
			-75412864, -74509376, -73599912, -72684464, -71763016, -70835560, -69902088, -68962576, -68017032, -67065428, -66107764, -65144024, -64174200, -63198280, -62216256, -61228112,
			-60233844, -59233436, -58226880, -57214168, -56195288, -55170228, -54138980, -53101532, -52057876, -51008004, -49951900, -48889560, -47820976, -46746132, -45665020, -44577632,
			-43483960, -42383992, -41277720, -40165136, -39046228, -37920988, -36789408, -35651480, -34507192, -33356536, -32199506, -31036092, -29866284, -28690076, -27507456, -26318420,
			-25122956, -23921058, -22712718, -21497926, -20276676, -19048960, -17814770, -16574096, -15326933, -14073273, -12813108, -11546431, -10273233, -8993509, -7707251, -6414452,
			-5115104, -3809201, -2496736, -1177703, 147906, 1480097, 2818878, 4164253, 5516231, 6874816, 8240015, 9611835, 10990281, 12375360, 13767076, 15165437,
			16570447, 17982114, 19400440, 20825434, 22257098, 23695440, 25140466, 26592178, 28050582, 29515684, 30987488, 32466000, 33951224, 35443164, 36941824, 38447212,
			39959328, 41478176, 43003764, 44536092, 46075168, 47620996, 49173576, 50732916, 52299012, 53871876, 55451508, 57037912, 58631088, 60231044, 61837784, 63451304,
			65071612, 66698708, 68332600, 69973280, 71620760, 73275048, 74936128, 76604016, 78278712, 79960216, 81648528, 83343656, 85045592, 86754352, 88469920, 90192312,
			91921528, 93657560, 95400416, 97150088, 98906592, 100669920, 102440080, 104217056, 106000864, 107791504, 109588968, 111393256, 113204376, 115022328, 116847104, 118678712,
			120517152, 122362408, 124214504, 126073416, 127939160, 129811728, 131691128, 133577344, 135470384, 137370240, 139276928, 141190432, 143110736, 145037872, 146971824, 148912576,
			150860160, 152814528, 154775712, 156743712, 158718496, 160700096, 162688480, 164683664, 166685632, 168694400, 170709936, 172732272, 174761392, 176797280, 178839936, 180889360,
			182945552, 185008512, 187078240, 189154704, 191237920, 193327888, 195424608, 197528048, 199638240, 201755152, 203878784, 206009136, 208146208, 210289984, 212440464, 214597648,
			216761520, 218932080, 221109328, 223293264, 225483856, 227681120, 229885040, 232095616, 234312832, 236536704, 238767200, 241004336, 243248080, 245498448, 247755408, 250018992,
			252289152, 254565904, 256849232, 259139136, 261435600, 263738640, 266048208, 268364320, 270686976, 273016128, 275351840, 277694016, 280042720, 282397920, 284759584, 287127744,
			289502336, 291883424, 294270944, 296664896, 299065280, 301472064, 303885280, 306304896, 308730880, 311163264, 313602016, 316047104, 318498560, 320956384, 323420512, 325890944,
			328367680, 330850752, 333340064, 335835680, 338337568, 340845664, 343360032, 345880640, 348407456, 350940480, 353479680, 356025056, 358576640, 361134368, 363698208, 366268224,
			368844352, 371426560, 374014880, 376609312, 379209760, 381816288, 384428864, 387047488, 389672096, 392302720, 394939328, 397581920, 400230464, 402884960, 405545376, 408211744,
			410884000, 413562112, 416246144, 418936032, 421631744, 424333280, 427040672, 429753824, 432472800, 435197504, 437928000, 440664224, 443406144, 446153792, 448907136, 451666144,
			454430816, 457201152, 459977088, 462758656, 465545792, 468338528, 471136800, 473940608, 476749984, 479564832, 482385184, 485211008, 488042304, 490879008, 493721152, 496568672,
			499421600, 502279904, 505143552, 508012512, 510886816, 513766368, 516651232, 519541344, 522436672, 525337248, 528243008, 531153952, 534070048, 536991296, 539917632, 542849088,
			545785664, 548727296, 551673920, 554625600, 557582272, 560543936, 563510528, 566482112, 569458560, 572439936, 575426176, 578417344, 581413248, 584414016, 587419584, 590429888,
			593444992, 596464832, 599489344, 602518528, 605552384, 608590912, 611634048, 614681792, 617734080, 620790912, 623852288, 626918208, 629988544, 633063424, 636142656, 639226368,
			642314496, 645406912, 648503744, 651604864, 654710272, 657819968, 660933888, 664052096, 667174464, 670301056, 673431744, 676566592, 679705536, 682848576, 685995648, 689146752,
			692301888, 695460992, 698624064, 701791104, 704961984, 708136768, 711315456, 714497920, 717684224, 720874304, 724068096, 727265664, 730466880, 733671808, 736880384, 740092608,
			743308416, 746527744, 749750656, 752977088, 756207040, 759440384, 762677248, 765917440, 769161088, 772408000, 775658304, 778911936, 782168768, 785428928, 788692224, 791958784,
			795228416, 798501248, 801777216, 805056192, 808338240, 811623296, 814911360, 818202432, 821496384, 824793280, 828092992, 831395648, 834701056, 838009280, 841320256, 844633984,
			847950400, 851269440, 854591168, 857915520, 861242496, 864571968, 867904000, 871238528, 874575488, 877914944, 881256768, 884600960, 887947520, 891296384, 894647552, 898000960,
			901356544, 904714368, 908074368, 911436480, 914800704, 918166976, 921535296, 924905664, 928277952, 931652224, 935028416, 938406464, 941786368, 945168064, 948551616, 951936832,
			955323840, 958712512, 962102912, 965494848, 968888448, 972283584, 975680192, 979078400, 982478016, 985879104, 989281536, 992685376, 996090560, 999497024, 1002904768, 1006313792,
			1009723968, 1013135360, 1016547840, 1019961472, 1023376128, 1026791872, 1030208576, 1033626304, 1037044928, 1040464512, 1043884928, 1047306176, 1050728256, 1054151104, 1057574656, 1060998976,
			1064423872, 1067849472, 1071275712, 1074702464, 1078129792, 1081557504, 1084985728, 1088414464, 1091843584, 1095272960, 1098702848, 1102132864, 1105563136, 1108993792, 1112424448, 1115855360,
			1119286400, 1122717568, 1126148736, 1129579904, 1133010944, 1136442112, 1139873152, 1143303936, 1146734720, 1150165248, 1153595520, 1157025536, 1160455296, 1163884672, 1167313792, 1170742400,
			1174170496, 1177598208, 1181025408, 1184452096, 1187878144, 1191303552, 1194728320, 1198152448, 1201575680, 1204998272, 1208420096, 1211841024, 1215261184, 1218680320, 1222098560, 1225515776,
			1228932096, 1232347264, 1235761280, 1239174272, 1242586112, 1245996672, 1249406080, 1252814208, 1256221056, 1259626496, 1263030528, 1266433280, 1269834496, 1273234176, 1276632448, 1280029056,
			1283424128, 1286817536, 1290209280, 1293599232, 1296987648, 1300374144, 1303758848, 1307141632, 1310522624, 1313901568, 1317278720, 1320653696, 1324026752, 1327397632, 1330766464, 1334133248,
			1337497728, 1340860032, 1344220032, 1347577728, 1350933120, 1354286080, 1357636736, 1360984960, 1364330624, 1367673728, 1371014400, 1374352384, 1377687808, 1381020544, 1384350592, 1387677824,
			1391002368, 1394324096, 1397642880, 1400958848, 1404271872, 1407581952, 1410889088, 1414193152, 1417494016, 1420791936, 1424086656, 1427378176, 1430666496, 1433951488, 1437233280, 1440511616,
			1443786752, 1447058304, 1450326528, 1453591296, 1456852480, 1460110080, 1463364224, 1466614656, 1469861376, 1473104384, 1476343808, 1479579264, 1482811008, 1486038912, 1489262848, 1492482944,
			1495699072, 1498911232, 1502119296, 1505323264, 1508523136, 1511718912, 1514910464, 1518097792, 1521280896, 1524459776, 1527634176, 1530804352, 1533970048, 1537131264, 1540288128, 1543440384,
			1546588160, 1549731200, 1552869760, 1556003584, 1559132800, 1562257280, 1565376896, 1568491776, 1571601792, 1574706944, 1577807232, 1580902528, 1583992832, 1587078144, 1590158336, 1593233536,
			1596303488, 1599368320, 1602427904, 1605482368, 1608531456, 1611575296, 1614613760, 1617646848, 1620674432, 1623696640, 1626713472, 1629724672, 1632730240, 1635730304, 1638724736, 1641713408,
			1644696448, 1647673728, 1650645248, 1653611008, 1656570880, 1659524864, 1662472960, 1665415040, 1668351232, 1671281280, 1674205440, 1677123328, 1680035200, 1682940928, 1685840384, 1688733696,
			1691620608, 1694501376, 1697375744, 1700243712, 1703105280, 1705960448, 1708809216, 1711651328, 1714486912, 1717315968, 1720138496, 1722954240, 1725763456, 1728565888, 1731361536, 1734150400,
			1736932480, 1739707776, 1742476160, 1745237504, 1747992064, 1750739584, 1753480064, 1756213504, 1758939904, 1761659136, 1764371328, 1767076224, 1769773952, 1772464384, 1775147648, 1777823488,
			1780492032, 1783153152, 1785806976, 1788453248, 1791092096, 1793723392, 1796347136, 1798963456, 1801571968, 1804172928, 1806766336, 1809351936, 1811929728, 1814499840, 1817062272, 1819616768,
			1822163328, 1824702080, 1827232896, 1829755776, 1832270720, 1834777600, 1837276416, 1839767168, 1842249728, 1844724352, 1847190656, 1849648768, 1852098688, 1854540288, 1856973696, 1859398784,
			1861815424, 1864223744, 1866623616, 1869015168, 1871398144, 1873772544, 1876138624, 1878496000, 1880844800, 1883185024, 1885516544, 1887839360, 1890153600, 1892459008, 1894755712, 1897043584,
			1899322624, 1901592832, 1903854208, 1906106624, 1908350080, 1910584704, 1912810240, 1915026688, 1917234176, 1919432576, 1921621888, 1923802112, 1925973120, 1928135040, 1930287616, 1932430976,
			1934565120, 1936689920, 1938805504, 1940911616, 1943008512, 1945095936, 1947173888, 1949242368, 1951301504, 1953351040, 1955391104, 1957421568, 1959442432, 1961453696, 1963455360, 1965447424,
			1967429632, 1969402240, 1971365120, 1973318272, 1975261696, 1977195136, 1979118848, 1981032704, 1982936704, 1984830848, 1986715008, 1988589184, 1990453376, 1992307584, 1994151808, 1995985920,
			1997810048, 1999623936, 2001427840, 2003221632, 2005005184, 2006778624, 2008541696, 2010294656, 2012037376, 2013769856, 2015491968, 2017203712, 2018905216, 2020596352, 2022277120, 2023947392,
			2025607296, 2027256704, 2028895744, 2030524288, 2032142208, 2033749632, 2035346560, 2036932864, 2038508672, 2040073728, 2041628288, 2043172096, 2044705280, 2046227712, 2047739392, 2049240448,
			2050730752, 2052210304, 2053678976, 2055136896, 2056583936, 2058020224, 2059445632, 2060860160, 2062263680, 2063656320, 2065038080, 2066408960, 2067768704, 2069117568, 2070455424, 2071782144,
			2073097856, 2074402560, 2075696256, 2076978816, 2078250240, 2079510528, 2080759680, 2081997696, 2083224448, 2084440192, 2085644544, 2086837760, 2088019840, 2089190528, 2090349952, 2091498240,
			2092635136, 2093760640, 2094874880, 2095977856, 2097069440, 2098149632, 2099218432, 2100275840, 2101321856, 2102356480, 2103379584, 2104391296, 2105391616, 2106380288, 2107357696, 2108323456,
			2109277696, 2110220416, 2111151616, 2112071296, 2112979328, 2113875840, 2114760832, 2115634176, 2116495872, 2117346048, 2118184448, 2119011328, 2119826560, 2120630016, 2121421824, 2122202112,
			2122970496, 2123727360, 2124472320, 2125205632, 2125927296, 2126637184, 2127335296, 2128021632, 2128696192, 2129358976, 2130009984, 2130649216, 2131276672, 2131892224, 2132496000, 2133088000,
			2133668096, 2134236416, 2134792832, 2135337344, 2135870080, 2136390912, 2136899968, 2137396992, 2137882112, 2138355456, 2138816896, 2139266304, 2139703936, 2140129536, 2140543232, 2140945024,
			2141334784, 2141712768, 2142078720, 2142432640, 2142774784, 2143104768, 2143422976, 2143729024, 2144023296, 2144305408, 2144575616, 2144833920, 2145080064, 2145314304, 2145536640, 2145746816,
			2145945088, 2146131328, 2146305664, 2146467840, 2146618112, 2146756352, 2146882560, 2146996736, 2147098880, 2147189120, 2147267200, 2147333376, 2147387392, 2147429504, 2147459584, 2147477632, 2147483647 };


    const fixed TriWave[N_WAVE] = {
0,2097151,4194303,6291455,8388607,10485759,12582911,14680063,16777215,18874367,20971519,23068671,25165823,27262975,29360127,31457279,
33554431,35651583,37748735,39845887,41943039,44040191,46137343,48234495,50331647,52428799,54525951,56623103,58720255,60817407,62914559,65011711,
67108863,69206015,71303167,73400319,75497471,77594623,79691775,81788927,83886079,85983231,88080383,90177535,92274687,94371839,96468991,98566143,
100663295,102760447,104857599,106954751,109051903,111149055,113246207,115343359,117440511,119537663,121634815,123731967,125829119,127926271,130023423,132120575,
134217727,136314879,138412031,140509183,142606335,144703487,146800639,148897791,150994943,153092095,155189247,157286399,159383551,161480703,163577855,165675007,
167772159,169869311,171966463,174063615,176160767,178257919,180355071,182452223,184549375,186646527,188743679,190840831,192937983,195035135,197132287,199229439,
201326591,203423743,205520895,207618047,209715199,211812351,213909503,216006655,218103807,220200959,222298111,224395263,226492415,228589567,230686719,232783871,
234881023,236978175,239075327,241172479,243269631,245366783,247463935,249561087,251658239,253755391,255852543,257949695,260046847,262143999,264241151,266338303,
268435455,270532607,272629759,274726911,276824063,278921215,281018367,283115519,285212671,287309823,289406975,291504127,293601279,295698431,297795583,299892735,
301989887,304087039,306184191,308281343,310378495,312475647,314572799,316669951,318767103,320864255,322961407,325058559,327155711,329252863,331350015,333447167,
335544319,337641471,339738623,341835775,343932927,346030079,348127231,350224383,352321535,354418687,356515839,358612991,360710143,362807295,364904447,367001599,
369098751,371195903,373293055,375390207,377487359,379584511,381681663,383778815,385875967,387973119,390070271,392167423,394264575,396361727,398458879,400556031,
402653183,404750335,406847487,408944639,411041791,413138943,415236095,417333247,419430399,421527551,423624703,425721855,427819007,429916159,432013311,434110463,
436207615,438304767,440401919,442499071,444596223,446693375,448790527,450887679,452984831,455081983,457179135,459276287,461373439,463470591,465567743,467664895,
469762047,471859199,473956351,476053503,478150655,480247807,482344959,484442111,486539263,488636415,490733567,492830719,494927871,497025023,499122175,501219327,
503316479,505413631,507510783,509607935,511705087,513802239,515899391,517996543,520093695,522190847,524287999,526385151,528482303,530579455,532676607,534773759,
536870911,538968063,541065215,543162367,545259519,547356671,549453823,551550975,553648127,555745279,557842431,559939583,562036735,564133887,566231039,568328191,
570425343,572522495,574619647,576716799,578813951,580911103,583008255,585105407,587202559,589299711,591396863,593494015,595591167,597688319,599785471,601882623,
603979775,606076927,608174079,610271231,612368383,614465535,616562687,618659839,620756991,622854143,624951295,627048447,629145599,631242751,633339903,635437055,
637534207,639631359,641728511,643825663,645922815,648019967,650117119,652214271,654311423,656408575,658505727,660602879,662700031,664797183,666894335,668991487,
671088639,673185791,675282943,677380095,679477247,681574399,683671551,685768703,687865855,689963007,692060159,694157311,696254463,698351615,700448767,702545919,
704643071,706740223,708837375,710934527,713031679,715128831,717225983,719323135,721420287,723517439,725614591,727711743,729808895,731906047,734003199,736100351,
738197503,740294655,742391807,744488959,746586111,748683263,750780415,752877567,754974719,757071871,759169023,761266175,763363327,765460479,767557631,769654783,
771751935,773849087,775946239,778043391,780140543,782237695,784334847,786431999,788529151,790626303,792723455,794820607,796917759,799014911,801112063,803209215,
805306367,807403519,809500671,811597823,813694975,815792127,817889279,819986431,822083583,824180735,826277887,828375039,830472191,832569343,834666495,836763647,
838860799,840957951,843055103,845152255,847249407,849346559,851443711,853540863,855638015,857735167,859832319,861929471,864026623,866123775,868220927,870318079,
872415231,874512383,876609535,878706687,880803839,882900991,884998143,887095295,889192447,891289599,893386751,895483903,897581055,899678207,901775359,903872511,
905969663,908066815,910163967,912261119,914358271,916455423,918552575,920649727,922746879,924844031,926941183,929038335,931135487,933232639,935329791,937426943,
939524095,941621247,943718399,945815551,947912703,950009855,952107007,954204159,956301311,958398463,960495615,962592767,964689919,966787071,968884223,970981375,
973078527,975175679,977272831,979369983,981467135,983564287,985661439,987758591,989855743,991952895,994050047,996147199,998244351,1000341503,1002438655,1004535807,
1006632959,1008730111,1010827263,1012924415,1015021567,1017118719,1019215871,1021313023,1023410175,1025507327,1027604479,1029701631,1031798783,1033895935,1035993087,1038090239,
1040187391,1042284543,1044381695,1046478847,1048575999,1050673151,1052770303,1054867455,1056964607,1059061759,1061158911,1063256063,1065353215,1067450367,1069547519,1071644671,
1073741823,1075838975,1077936127,1080033279,1082130431,1084227583,1086324735,1088421887,1090519039,1092616191,1094713343,1096810495,1098907647,1101004799,1103101951,1105199103,
1107296255,1109393407,1111490559,1113587711,1115684863,1117782015,1119879167,1121976319,1124073471,1126170623,1128267775,1130364927,1132462079,1134559231,1136656383,1138753535,
1140850687,1142947839,1145044991,1147142143,1149239295,1151336447,1153433599,1155530751,1157627903,1159725055,1161822207,1163919359,1166016511,1168113663,1170210815,1172307967,
1174405119,1176502271,1178599423,1180696575,1182793727,1184890879,1186988031,1189085183,1191182335,1193279487,1195376639,1197473791,1199570943,1201668095,1203765247,1205862399,
1207959551,1210056703,1212153855,1214251007,1216348159,1218445311,1220542463,1222639615,1224736767,1226833919,1228931071,1231028223,1233125375,1235222527,1237319679,1239416831,
1241513983,1243611135,1245708287,1247805439,1249902591,1251999743,1254096895,1256194047,1258291199,1260388351,1262485503,1264582655,1266679807,1268776959,1270874111,1272971263,
1275068415,1277165567,1279262719,1281359871,1283457023,1285554175,1287651327,1289748479,1291845631,1293942783,1296039935,1298137087,1300234239,1302331391,1304428543,1306525695,
1308622847,1310719999,1312817151,1314914303,1317011455,1319108607,1321205759,1323302911,1325400063,1327497215,1329594367,1331691519,1333788671,1335885823,1337982975,1340080127,
1342177279,1344274431,1346371583,1348468735,1350565887,1352663039,1354760191,1356857343,1358954495,1361051647,1363148799,1365245951,1367343103,1369440255,1371537407,1373634559,
1375731711,1377828863,1379926015,1382023167,1384120319,1386217471,1388314623,1390411775,1392508927,1394606079,1396703231,1398800383,1400897535,1402994687,1405091839,1407188991,
1409286143,1411383295,1413480447,1415577599,1417674751,1419771903,1421869055,1423966207,1426063359,1428160511,1430257663,1432354815,1434451967,1436549119,1438646271,1440743423,
1442840575,1444937727,1447034879,1449132031,1451229183,1453326335,1455423487,1457520639,1459617791,1461714943,1463812095,1465909247,1468006399,1470103551,1472200703,1474297855,
1476395007,1478492159,1480589311,1482686463,1484783615,1486880767,1488977919,1491075071,1493172223,1495269375,1497366527,1499463679,1501560831,1503657983,1505755135,1507852287,
1509949439,1512046591,1514143743,1516240895,1518338047,1520435199,1522532351,1524629503,1526726655,1528823807,1530920959,1533018111,1535115263,1537212415,1539309567,1541406719,
1543503871,1545601023,1547698175,1549795327,1551892479,1553989631,1556086783,1558183935,1560281087,1562378239,1564475391,1566572543,1568669695,1570766847,1572863999,1574961151,
1577058303,1579155455,1581252607,1583349759,1585446911,1587544063,1589641215,1591738367,1593835519,1595932671,1598029823,1600126975,1602224127,1604321279,1606418431,1608515583,
1610612735,1612709887,1614807039,1616904191,1619001343,1621098495,1623195647,1625292799,1627389951,1629487103,1631584255,1633681407,1635778559,1637875711,1639972863,1642070015,
1644167167,1646264319,1648361471,1650458623,1652555775,1654652927,1656750079,1658847231,1660944383,1663041535,1665138687,1667235839,1669332991,1671430143,1673527295,1675624447,
1677721599,1679818751,1681915903,1684013055,1686110207,1688207359,1690304511,1692401663,1694498815,1696595967,1698693119,1700790271,1702887423,1704984575,1707081727,1709178879,
1711276031,1713373183,1715470335,1717567487,1719664639,1721761791,1723858943,1725956095,1728053247,1730150399,1732247551,1734344703,1736441855,1738539007,1740636159,1742733311,
1744830463,1746927615,1749024767,1751121919,1753219071,1755316223,1757413375,1759510527,1761607679,1763704831,1765801983,1767899135,1769996287,1772093439,1774190591,1776287743,
1778384895,1780482047,1782579199,1784676351,1786773503,1788870655,1790967807,1793064959,1795162111,1797259263,1799356415,1801453567,1803550719,1805647871,1807745023,1809842175,
1811939327,1814036479,1816133631,1818230783,1820327935,1822425087,1824522239,1826619391,1828716543,1830813695,1832910847,1835007999,1837105151,1839202303,1841299455,1843396607,
1845493759,1847590911,1849688063,1851785215,1853882367,1855979519,1858076671,1860173823,1862270975,1864368127,1866465279,1868562431,1870659583,1872756735,1874853887,1876951039,
1879048191,1881145343,1883242495,1885339647,1887436799,1889533951,1891631103,1893728255,1895825407,1897922559,1900019711,1902116863,1904214015,1906311167,1908408319,1910505471,
1912602623,1914699775,1916796927,1918894079,1920991231,1923088383,1925185535,1927282687,1929379839,1931476991,1933574143,1935671295,1937768447,1939865599,1941962751,1944059903,
1946157055,1948254207,1950351359,1952448511,1954545663,1956642815,1958739967,1960837119,1962934271,1965031423,1967128575,1969225727,1971322879,1973420031,1975517183,1977614335,
1979711487,1981808639,1983905791,1986002943,1988100095,1990197247,1992294399,1994391551,1996488703,1998585855,2000683007,2002780159,2004877311,2006974463,2009071615,2011168767,
2013265919,2015363071,2017460223,2019557375,2021654527,2023751679,2025848831,2027945983,2030043135,2032140287,2034237439,2036334591,2038431743,2040528895,2042626047,2044723199,
2046820351,2048917503,2051014655,2053111807,2055208959,2057306111,2059403263,2061500415,2063597567,2065694719,2067791871,2069889023,2071986175,2074083327,2076180479,2078277631,
2080374783,2082471935,2084569087,2086666239,2088763391,2090860543,2092957695,2095054847,2097151999,2099249151,2101346303,2103443455,2105540607,2107637759,2109734911,2111832063,
2113929215,2116026367,2118123519,2120220671,2122317823,2124414975,2126512127,2128609279,2130706431,2132803583,2134900735,2136997887,2139095039,2141192191,2143289343,2145386495,
2147483647,2145386495,2143289343,2141192191,2139095039,2136997887,2134900735,2132803583,2130706431,2128609279,2126512127,2124414975,2122317823,2120220671,2118123519,2116026367,
2113929215,2111832063,2109734911,2107637759,2105540607,2103443455,2101346303,2099249151,2097151999,2095054847,2092957695,2090860543,2088763391,2086666239,2084569087,2082471935,
2080374783,2078277631,2076180479,2074083327,2071986175,2069889023,2067791871,2065694719,2063597567,2061500415,2059403263,2057306111,2055208959,2053111807,2051014655,2048917503,
2046820351,2044723199,2042626047,2040528895,2038431743,2036334591,2034237439,2032140287,2030043135,2027945983,2025848831,2023751679,2021654527,2019557375,2017460223,2015363071,
2013265919,2011168767,2009071615,2006974463,2004877311,2002780159,2000683007,1998585855,1996488703,1994391551,1992294399,1990197247,1988100095,1986002943,1983905791,1981808639,
1979711487,1977614335,1975517183,1973420031,1971322879,1969225727,1967128575,1965031423,1962934271,1960837119,1958739967,1956642815,1954545663,1952448511,1950351359,1948254207,
1946157055,1944059903,1941962751,1939865599,1937768447,1935671295,1933574143,1931476991,1929379839,1927282687,1925185535,1923088383,1920991231,1918894079,1916796927,1914699775,
1912602623,1910505471,1908408319,1906311167,1904214015,1902116863,1900019711,1897922559,1895825407,1893728255,1891631103,1889533951,1887436799,1885339647,1883242495,1881145343,
1879048191,1876951039,1874853887,1872756735,1870659583,1868562431,1866465279,1864368127,1862270975,1860173823,1858076671,1855979519,1853882367,1851785215,1849688063,1847590911,
1845493759,1843396607,1841299455,1839202303,1837105151,1835007999,1832910847,1830813695,1828716543,1826619391,1824522239,1822425087,1820327935,1818230783,1816133631,1814036479,
1811939327,1809842175,1807745023,1805647871,1803550719,1801453567,1799356415,1797259263,1795162111,1793064959,1790967807,1788870655,1786773503,1784676351,1782579199,1780482047,
1778384895,1776287743,1774190591,1772093439,1769996287,1767899135,1765801983,1763704831,1761607679,1759510527,1757413375,1755316223,1753219071,1751121919,1749024767,1746927615,
1744830463,1742733311,1740636159,1738539007,1736441855,1734344703,1732247551,1730150399,1728053247,1725956095,1723858943,1721761791,1719664639,1717567487,1715470335,1713373183,
1711276031,1709178879,1707081727,1704984575,1702887423,1700790271,1698693119,1696595967,1694498815,1692401663,1690304511,1688207359,1686110207,1684013055,1681915903,1679818751,
1677721599,1675624447,1673527295,1671430143,1669332991,1667235839,1665138687,1663041535,1660944383,1658847231,1656750079,1654652927,1652555775,1650458623,1648361471,1646264319,
1644167167,1642070015,1639972863,1637875711,1635778559,1633681407,1631584255,1629487103,1627389951,1625292799,1623195647,1621098495,1619001343,1616904191,1614807039,1612709887,
1610612735,1608515583,1606418431,1604321279,1602224127,1600126975,1598029823,1595932671,1593835519,1591738367,1589641215,1587544063,1585446911,1583349759,1581252607,1579155455,
1577058303,1574961151,1572863999,1570766847,1568669695,1566572543,1564475391,1562378239,1560281087,1558183935,1556086783,1553989631,1551892479,1549795327,1547698175,1545601023,
1543503871,1541406719,1539309567,1537212415,1535115263,1533018111,1530920959,1528823807,1526726655,1524629503,1522532351,1520435199,1518338047,1516240895,1514143743,1512046591,
1509949439,1507852287,1505755135,1503657983,1501560831,1499463679,1497366527,1495269375,1493172223,1491075071,1488977919,1486880767,1484783615,1482686463,1480589311,1478492159,
1476395007,1474297855,1472200703,1470103551,1468006399,1465909247,1463812095,1461714943,1459617791,1457520639,1455423487,1453326335,1451229183,1449132031,1447034879,1444937727,
1442840575,1440743423,1438646271,1436549119,1434451967,1432354815,1430257663,1428160511,1426063359,1423966207,1421869055,1419771903,1417674751,1415577599,1413480447,1411383295,
1409286143,1407188991,1405091839,1402994687,1400897535,1398800383,1396703231,1394606079,1392508927,1390411775,1388314623,1386217471,1384120319,1382023167,1379926015,1377828863,
1375731711,1373634559,1371537407,1369440255,1367343103,1365245951,1363148799,1361051647,1358954495,1356857343,1354760191,1352663039,1350565887,1348468735,1346371583,1344274431,
1342177279,1340080127,1337982975,1335885823,1333788671,1331691519,1329594367,1327497215,1325400063,1323302911,1321205759,1319108607,1317011455,1314914303,1312817151,1310719999,
1308622847,1306525695,1304428543,1302331391,1300234239,1298137087,1296039935,1293942783,1291845631,1289748479,1287651327,1285554175,1283457023,1281359871,1279262719,1277165567,
1275068415,1272971263,1270874111,1268776959,1266679807,1264582655,1262485503,1260388351,1258291199,1256194047,1254096895,1251999743,1249902591,1247805439,1245708287,1243611135,
1241513983,1239416831,1237319679,1235222527,1233125375,1231028223,1228931071,1226833919,1224736767,1222639615,1220542463,1218445311,1216348159,1214251007,1212153855,1210056703,
1207959551,1205862399,1203765247,1201668095,1199570943,1197473791,1195376639,1193279487,1191182335,1189085183,1186988031,1184890879,1182793727,1180696575,1178599423,1176502271,
1174405119,1172307967,1170210815,1168113663,1166016511,1163919359,1161822207,1159725055,1157627903,1155530751,1153433599,1151336447,1149239295,1147142143,1145044991,1142947839,
1140850687,1138753535,1136656383,1134559231,1132462079,1130364927,1128267775,1126170623,1124073471,1121976319,1119879167,1117782015,1115684863,1113587711,1111490559,1109393407,
1107296255,1105199103,1103101951,1101004799,1098907647,1096810495,1094713343,1092616191,1090519039,1088421887,1086324735,1084227583,1082130431,1080033279,1077936127,1075838975,
1073741823,1071644671,1069547519,1067450367,1065353215,1063256063,1061158911,1059061759,1056964607,1054867455,1052770303,1050673151,1048575999,1046478847,1044381695,1042284543,
1040187391,1038090239,1035993087,1033895935,1031798783,1029701631,1027604479,1025507327,1023410175,1021313023,1019215871,1017118719,1015021567,1012924415,1010827263,1008730111,
1006632959,1004535807,1002438655,1000341503,998244351,996147199,994050047,991952895,989855743,987758591,985661439,983564287,981467135,979369983,977272831,975175679,
973078527,970981375,968884223,966787071,964689919,962592767,960495615,958398463,956301311,954204159,952107007,950009855,947912703,945815551,943718399,941621247,
939524095,937426943,935329791,933232639,931135487,929038335,926941183,924844031,922746879,920649727,918552575,916455423,914358271,912261119,910163967,908066815,
905969663,903872511,901775359,899678207,897581055,895483903,893386751,891289599,889192447,887095295,884998143,882900991,880803839,878706687,876609535,874512383,
872415231,870318079,868220927,866123775,864026623,861929471,859832319,857735167,855638015,853540863,851443711,849346559,847249407,845152255,843055103,840957951,
838860799,836763647,834666495,832569343,830472191,828375039,826277887,824180735,822083583,819986431,817889279,815792127,813694975,811597823,809500671,807403519,
805306367,803209215,801112063,799014911,796917759,794820607,792723455,790626303,788529151,786431999,784334847,782237695,780140543,778043391,775946239,773849087,
771751935,769654783,767557631,765460479,763363327,761266175,759169023,757071871,754974719,752877567,750780415,748683263,746586111,744488959,742391807,740294655,
738197503,736100351,734003199,731906047,729808895,727711743,725614591,723517439,721420287,719323135,717225983,715128831,713031679,710934527,708837375,706740223,
704643071,702545919,700448767,698351615,696254463,694157311,692060159,689963007,687865855,685768703,683671551,681574399,679477247,677380095,675282943,673185791,
671088639,668991487,666894335,664797183,662700031,660602879,658505727,656408575,654311423,652214271,650117119,648019967,645922815,643825663,641728511,639631359,
637534207,635437055,633339903,631242751,629145599,627048447,624951295,622854143,620756991,618659839,616562687,614465535,612368383,610271231,608174079,606076927,
603979775,601882623,599785471,597688319,595591167,593494015,591396863,589299711,587202559,585105407,583008255,580911103,578813951,576716799,574619647,572522495,
570425343,568328191,566231039,564133887,562036735,559939583,557842431,555745279,553648127,551550975,549453823,547356671,545259519,543162367,541065215,538968063,
536870911,534773759,532676607,530579455,528482303,526385151,524287999,522190847,520093695,517996543,515899391,513802239,511705087,509607935,507510783,505413631,
503316479,501219327,499122175,497025023,494927871,492830719,490733567,488636415,486539263,484442111,482344959,480247807,478150655,476053503,473956351,471859199,
469762047,467664895,465567743,463470591,461373439,459276287,457179135,455081983,452984831,450887679,448790527,446693375,444596223,442499071,440401919,438304767,
436207615,434110463,432013311,429916159,427819007,425721855,423624703,421527551,419430399,417333247,415236095,413138943,411041791,408944639,406847487,404750335,
402653183,400556031,398458879,396361727,394264575,392167423,390070271,387973119,385875967,383778815,381681663,379584511,377487359,375390207,373293055,371195903,
369098751,367001599,364904447,362807295,360710143,358612991,356515839,354418687,352321535,350224383,348127231,346030079,343932927,341835775,339738623,337641471,
335544319,333447167,331350015,329252863,327155711,325058559,322961407,320864255,318767103,316669951,314572799,312475647,310378495,308281343,306184191,304087039,
301989887,299892735,297795583,295698431,293601279,291504127,289406975,287309823,285212671,283115519,281018367,278921215,276824063,274726911,272629759,270532607,
268435455,266338303,264241151,262143999,260046847,257949695,255852543,253755391,251658239,249561087,247463935,245366783,243269631,241172479,239075327,236978175,
234881023,232783871,230686719,228589567,226492415,224395263,222298111,220200959,218103807,216006655,213909503,211812351,209715199,207618047,205520895,203423743,
201326591,199229439,197132287,195035135,192937983,190840831,188743679,186646527,184549375,182452223,180355071,178257919,176160767,174063615,171966463,169869311,
167772159,165675007,163577855,161480703,159383551,157286399,155189247,153092095,150994943,148897791,146800639,144703487,142606335,140509183,138412031,136314879,
134217727,132120575,130023423,127926271,125829119,123731967,121634815,119537663,117440511,115343359,113246207,111149055,109051903,106954751,104857599,102760447,
100663295,98566143,96468991,94371839,92274687,90177535,88080383,85983231,83886079,81788927,79691775,77594623,75497471,73400319,71303167,69206015,
67108863,65011711,62914559,60817407,58720255,56623103,54525951,52428799,50331647,48234495,46137343,44040191,41943039,39845887,37748735,35651583,
33554431,31457279,29360127,27262975,25165823,23068671,20971519,18874367,16777215,14680063,12582911,10485759,8388607,6291455,4194303,2097151,
0,-2097151,-4194303,-6291455,-8388607,-10485759,-12582911,-14680063,-16777215,-18874367,-20971519,-23068671,-25165823,-27262975,-29360127,-31457279,
-33554431,-35651583,-37748735,-39845887,-41943039,-44040191,-46137343,-48234495,-50331647,-52428799,-54525951,-56623103,-58720255,-60817407,-62914559,-65011711,
-67108863,-69206015,-71303167,-73400319,-75497471,-77594623,-79691775,-81788927,-83886079,-85983231,-88080383,-90177535,-92274687,-94371839,-96468991,-98566143,
-100663295,-102760447,-104857599,-106954751,-109051903,-111149055,-113246207,-115343359,-117440511,-119537663,-121634815,-123731967,-125829119,-127926271,-130023423,-132120575,
-134217727,-136314879,-138412031,-140509183,-142606335,-144703487,-146800639,-148897791,-150994943,-153092095,-155189247,-157286399,-159383551,-161480703,-163577855,-165675007,
-167772159,-169869311,-171966463,-174063615,-176160767,-178257919,-180355071,-182452223,-184549375,-186646527,-188743679,-190840831,-192937983,-195035135,-197132287,-199229439,
-201326591,-203423743,-205520895,-207618047,-209715199,-211812351,-213909503,-216006655,-218103807,-220200959,-222298111,-224395263,-226492415,-228589567,-230686719,-232783871,
-234881023,-236978175,-239075327,-241172479,-243269631,-245366783,-247463935,-249561087,-251658239,-253755391,-255852543,-257949695,-260046847,-262143999,-264241151,-266338303,
-268435455,-270532607,-272629759,-274726911,-276824063,-278921215,-281018367,-283115519,-285212671,-287309823,-289406975,-291504127,-293601279,-295698431,-297795583,-299892735,
-301989887,-304087039,-306184191,-308281343,-310378495,-312475647,-314572799,-316669951,-318767103,-320864255,-322961407,-325058559,-327155711,-329252863,-331350015,-333447167,
-335544319,-337641471,-339738623,-341835775,-343932927,-346030079,-348127231,-350224383,-352321535,-354418687,-356515839,-358612991,-360710143,-362807295,-364904447,-367001599,
-369098751,-371195903,-373293055,-375390207,-377487359,-379584511,-381681663,-383778815,-385875967,-387973119,-390070271,-392167423,-394264575,-396361727,-398458879,-400556031,
-402653183,-404750335,-406847487,-408944639,-411041791,-413138943,-415236095,-417333247,-419430399,-421527551,-423624703,-425721855,-427819007,-429916159,-432013311,-434110463,
-436207615,-438304767,-440401919,-442499071,-444596223,-446693375,-448790527,-450887679,-452984831,-455081983,-457179135,-459276287,-461373439,-463470591,-465567743,-467664895,
-469762047,-471859199,-473956351,-476053503,-478150655,-480247807,-482344959,-484442111,-486539263,-488636415,-490733567,-492830719,-494927871,-497025023,-499122175,-501219327,
-503316479,-505413631,-507510783,-509607935,-511705087,-513802239,-515899391,-517996543,-520093695,-522190847,-524287999,-526385151,-528482303,-530579455,-532676607,-534773759,
-536870911,-538968063,-541065215,-543162367,-545259519,-547356671,-549453823,-551550975,-553648127,-555745279,-557842431,-559939583,-562036735,-564133887,-566231039,-568328191,
-570425343,-572522495,-574619647,-576716799,-578813951,-580911103,-583008255,-585105407,-587202559,-589299711,-591396863,-593494015,-595591167,-597688319,-599785471,-601882623,
-603979775,-606076927,-608174079,-610271231,-612368383,-614465535,-616562687,-618659839,-620756991,-622854143,-624951295,-627048447,-629145599,-631242751,-633339903,-635437055,
-637534207,-639631359,-641728511,-643825663,-645922815,-648019967,-650117119,-652214271,-654311423,-656408575,-658505727,-660602879,-662700031,-664797183,-666894335,-668991487,
-671088639,-673185791,-675282943,-677380095,-679477247,-681574399,-683671551,-685768703,-687865855,-689963007,-692060159,-694157311,-696254463,-698351615,-700448767,-702545919,
-704643071,-706740223,-708837375,-710934527,-713031679,-715128831,-717225983,-719323135,-721420287,-723517439,-725614591,-727711743,-729808895,-731906047,-734003199,-736100351,
-738197503,-740294655,-742391807,-744488959,-746586111,-748683263,-750780415,-752877567,-754974719,-757071871,-759169023,-761266175,-763363327,-765460479,-767557631,-769654783,
-771751935,-773849087,-775946239,-778043391,-780140543,-782237695,-784334847,-786431999,-788529151,-790626303,-792723455,-794820607,-796917759,-799014911,-801112063,-803209215,
-805306367,-807403519,-809500671,-811597823,-813694975,-815792127,-817889279,-819986431,-822083583,-824180735,-826277887,-828375039,-830472191,-832569343,-834666495,-836763647,
-838860799,-840957951,-843055103,-845152255,-847249407,-849346559,-851443711,-853540863,-855638015,-857735167,-859832319,-861929471,-864026623,-866123775,-868220927,-870318079,
-872415231,-874512383,-876609535,-878706687,-880803839,-882900991,-884998143,-887095295,-889192447,-891289599,-893386751,-895483903,-897581055,-899678207,-901775359,-903872511,
-905969663,-908066815,-910163967,-912261119,-914358271,-916455423,-918552575,-920649727,-922746879,-924844031,-926941183,-929038335,-931135487,-933232639,-935329791,-937426943,
-939524095,-941621247,-943718399,-945815551,-947912703,-950009855,-952107007,-954204159,-956301311,-958398463,-960495615,-962592767,-964689919,-966787071,-968884223,-970981375,
-973078527,-975175679,-977272831,-979369983,-981467135,-983564287,-985661439,-987758591,-989855743,-991952895,-994050047,-996147199,-998244351,-1000341503,-1002438655,-1004535807,
-1006632959,-1008730111,-1010827263,-1012924415,-1015021567,-1017118719,-1019215871,-1021313023,-1023410175,-1025507327,-1027604479,-1029701631,-1031798783,-1033895935,-1035993087,-1038090239,
-1040187391,-1042284543,-1044381695,-1046478847,-1048575999,-1050673151,-1052770303,-1054867455,-1056964607,-1059061759,-1061158911,-1063256063,-1065353215,-1067450367,-1069547519,-1071644671,
-1073741823,-1075838975,-1077936127,-1080033279,-1082130431,-1084227583,-1086324735,-1088421887,-1090519039,-1092616191,-1094713343,-1096810495,-1098907647,-1101004799,-1103101951,-1105199103,
-1107296255,-1109393407,-1111490559,-1113587711,-1115684863,-1117782015,-1119879167,-1121976319,-1124073471,-1126170623,-1128267775,-1130364927,-1132462079,-1134559231,-1136656383,-1138753535,
-1140850687,-1142947839,-1145044991,-1147142143,-1149239295,-1151336447,-1153433599,-1155530751,-1157627903,-1159725055,-1161822207,-1163919359,-1166016511,-1168113663,-1170210815,-1172307967,
-1174405119,-1176502271,-1178599423,-1180696575,-1182793727,-1184890879,-1186988031,-1189085183,-1191182335,-1193279487,-1195376639,-1197473791,-1199570943,-1201668095,-1203765247,-1205862399,
-1207959551,-1210056703,-1212153855,-1214251007,-1216348159,-1218445311,-1220542463,-1222639615,-1224736767,-1226833919,-1228931071,-1231028223,-1233125375,-1235222527,-1237319679,-1239416831,
-1241513983,-1243611135,-1245708287,-1247805439,-1249902591,-1251999743,-1254096895,-1256194047,-1258291199,-1260388351,-1262485503,-1264582655,-1266679807,-1268776959,-1270874111,-1272971263,
-1275068415,-1277165567,-1279262719,-1281359871,-1283457023,-1285554175,-1287651327,-1289748479,-1291845631,-1293942783,-1296039935,-1298137087,-1300234239,-1302331391,-1304428543,-1306525695,
-1308622847,-1310719999,-1312817151,-1314914303,-1317011455,-1319108607,-1321205759,-1323302911,-1325400063,-1327497215,-1329594367,-1331691519,-1333788671,-1335885823,-1337982975,-1340080127,
-1342177279,-1344274431,-1346371583,-1348468735,-1350565887,-1352663039,-1354760191,-1356857343,-1358954495,-1361051647,-1363148799,-1365245951,-1367343103,-1369440255,-1371537407,-1373634559,
-1375731711,-1377828863,-1379926015,-1382023167,-1384120319,-1386217471,-1388314623,-1390411775,-1392508927,-1394606079,-1396703231,-1398800383,-1400897535,-1402994687,-1405091839,-1407188991,
-1409286143,-1411383295,-1413480447,-1415577599,-1417674751,-1419771903,-1421869055,-1423966207,-1426063359,-1428160511,-1430257663,-1432354815,-1434451967,-1436549119,-1438646271,-1440743423,
-1442840575,-1444937727,-1447034879,-1449132031,-1451229183,-1453326335,-1455423487,-1457520639,-1459617791,-1461714943,-1463812095,-1465909247,-1468006399,-1470103551,-1472200703,-1474297855,
-1476395007,-1478492159,-1480589311,-1482686463,-1484783615,-1486880767,-1488977919,-1491075071,-1493172223,-1495269375,-1497366527,-1499463679,-1501560831,-1503657983,-1505755135,-1507852287,
-1509949439,-1512046591,-1514143743,-1516240895,-1518338047,-1520435199,-1522532351,-1524629503,-1526726655,-1528823807,-1530920959,-1533018111,-1535115263,-1537212415,-1539309567,-1541406719,
-1543503871,-1545601023,-1547698175,-1549795327,-1551892479,-1553989631,-1556086783,-1558183935,-1560281087,-1562378239,-1564475391,-1566572543,-1568669695,-1570766847,-1572863999,-1574961151,
-1577058303,-1579155455,-1581252607,-1583349759,-1585446911,-1587544063,-1589641215,-1591738367,-1593835519,-1595932671,-1598029823,-1600126975,-1602224127,-1604321279,-1606418431,-1608515583,
-1610612735,-1612709887,-1614807039,-1616904191,-1619001343,-1621098495,-1623195647,-1625292799,-1627389951,-1629487103,-1631584255,-1633681407,-1635778559,-1637875711,-1639972863,-1642070015,
-1644167167,-1646264319,-1648361471,-1650458623,-1652555775,-1654652927,-1656750079,-1658847231,-1660944383,-1663041535,-1665138687,-1667235839,-1669332991,-1671430143,-1673527295,-1675624447,
-1677721599,-1679818751,-1681915903,-1684013055,-1686110207,-1688207359,-1690304511,-1692401663,-1694498815,-1696595967,-1698693119,-1700790271,-1702887423,-1704984575,-1707081727,-1709178879,
-1711276031,-1713373183,-1715470335,-1717567487,-1719664639,-1721761791,-1723858943,-1725956095,-1728053247,-1730150399,-1732247551,-1734344703,-1736441855,-1738539007,-1740636159,-1742733311,
-1744830463,-1746927615,-1749024767,-1751121919,-1753219071,-1755316223,-1757413375,-1759510527,-1761607679,-1763704831,-1765801983,-1767899135,-1769996287,-1772093439,-1774190591,-1776287743,
-1778384895,-1780482047,-1782579199,-1784676351,-1786773503,-1788870655,-1790967807,-1793064959,-1795162111,-1797259263,-1799356415,-1801453567,-1803550719,-1805647871,-1807745023,-1809842175,
-1811939327,-1814036479,-1816133631,-1818230783,-1820327935,-1822425087,-1824522239,-1826619391,-1828716543,-1830813695,-1832910847,-1835007999,-1837105151,-1839202303,-1841299455,-1843396607,
-1845493759,-1847590911,-1849688063,-1851785215,-1853882367,-1855979519,-1858076671,-1860173823,-1862270975,-1864368127,-1866465279,-1868562431,-1870659583,-1872756735,-1874853887,-1876951039,
-1879048191,-1881145343,-1883242495,-1885339647,-1887436799,-1889533951,-1891631103,-1893728255,-1895825407,-1897922559,-1900019711,-1902116863,-1904214015,-1906311167,-1908408319,-1910505471,
-1912602623,-1914699775,-1916796927,-1918894079,-1920991231,-1923088383,-1925185535,-1927282687,-1929379839,-1931476991,-1933574143,-1935671295,-1937768447,-1939865599,-1941962751,-1944059903,
-1946157055,-1948254207,-1950351359,-1952448511,-1954545663,-1956642815,-1958739967,-1960837119,-1962934271,-1965031423,-1967128575,-1969225727,-1971322879,-1973420031,-1975517183,-1977614335,
-1979711487,-1981808639,-1983905791,-1986002943,-1988100095,-1990197247,-1992294399,-1994391551,-1996488703,-1998585855,-2000683007,-2002780159,-2004877311,-2006974463,-2009071615,-2011168767,
-2013265919,-2015363071,-2017460223,-2019557375,-2021654527,-2023751679,-2025848831,-2027945983,-2030043135,-2032140287,-2034237439,-2036334591,-2038431743,-2040528895,-2042626047,-2044723199,
-2046820351,-2048917503,-2051014655,-2053111807,-2055208959,-2057306111,-2059403263,-2061500415,-2063597567,-2065694719,-2067791871,-2069889023,-2071986175,-2074083327,-2076180479,-2078277631,
-2080374783,-2082471935,-2084569087,-2086666239,-2088763391,-2090860543,-2092957695,-2095054847,-2097151999,-2099249151,-2101346303,-2103443455,-2105540607,-2107637759,-2109734911,-2111832063,
-2113929215,-2116026367,-2118123519,-2120220671,-2122317823,-2124414975,-2126512127,-2128609279,-2130706431,-2132803583,-2134900735,-2136997887,-2139095039,-2141192191,-2143289343,-2145386495,
-2147483647,-2145386495,-2143289343,-2141192191,-2139095039,-2136997887,-2134900735,-2132803583,-2130706431,-2128609279,-2126512127,-2124414975,-2122317823,-2120220671,-2118123519,-2116026367,
-2113929215,-2111832063,-2109734911,-2107637759,-2105540607,-2103443455,-2101346303,-2099249151,-2097151999,-2095054847,-2092957695,-2090860543,-2088763391,-2086666239,-2084569087,-2082471935,
-2080374783,-2078277631,-2076180479,-2074083327,-2071986175,-2069889023,-2067791871,-2065694719,-2063597567,-2061500415,-2059403263,-2057306111,-2055208959,-2053111807,-2051014655,-2048917503,
-2046820351,-2044723199,-2042626047,-2040528895,-2038431743,-2036334591,-2034237439,-2032140287,-2030043135,-2027945983,-2025848831,-2023751679,-2021654527,-2019557375,-2017460223,-2015363071,
-2013265919,-2011168767,-2009071615,-2006974463,-2004877311,-2002780159,-2000683007,-1998585855,-1996488703,-1994391551,-1992294399,-1990197247,-1988100095,-1986002943,-1983905791,-1981808639,
-1979711487,-1977614335,-1975517183,-1973420031,-1971322879,-1969225727,-1967128575,-1965031423,-1962934271,-1960837119,-1958739967,-1956642815,-1954545663,-1952448511,-1950351359,-1948254207,
-1946157055,-1944059903,-1941962751,-1939865599,-1937768447,-1935671295,-1933574143,-1931476991,-1929379839,-1927282687,-1925185535,-1923088383,-1920991231,-1918894079,-1916796927,-1914699775,
-1912602623,-1910505471,-1908408319,-1906311167,-1904214015,-1902116863,-1900019711,-1897922559,-1895825407,-1893728255,-1891631103,-1889533951,-1887436799,-1885339647,-1883242495,-1881145343,
-1879048191,-1876951039,-1874853887,-1872756735,-1870659583,-1868562431,-1866465279,-1864368127,-1862270975,-1860173823,-1858076671,-1855979519,-1853882367,-1851785215,-1849688063,-1847590911,
-1845493759,-1843396607,-1841299455,-1839202303,-1837105151,-1835007999,-1832910847,-1830813695,-1828716543,-1826619391,-1824522239,-1822425087,-1820327935,-1818230783,-1816133631,-1814036479,
-1811939327,-1809842175,-1807745023,-1805647871,-1803550719,-1801453567,-1799356415,-1797259263,-1795162111,-1793064959,-1790967807,-1788870655,-1786773503,-1784676351,-1782579199,-1780482047,
-1778384895,-1776287743,-1774190591,-1772093439,-1769996287,-1767899135,-1765801983,-1763704831,-1761607679,-1759510527,-1757413375,-1755316223,-1753219071,-1751121919,-1749024767,-1746927615,
-1744830463,-1742733311,-1740636159,-1738539007,-1736441855,-1734344703,-1732247551,-1730150399,-1728053247,-1725956095,-1723858943,-1721761791,-1719664639,-1717567487,-1715470335,-1713373183,
-1711276031,-1709178879,-1707081727,-1704984575,-1702887423,-1700790271,-1698693119,-1696595967,-1694498815,-1692401663,-1690304511,-1688207359,-1686110207,-1684013055,-1681915903,-1679818751,
-1677721599,-1675624447,-1673527295,-1671430143,-1669332991,-1667235839,-1665138687,-1663041535,-1660944383,-1658847231,-1656750079,-1654652927,-1652555775,-1650458623,-1648361471,-1646264319,
-1644167167,-1642070015,-1639972863,-1637875711,-1635778559,-1633681407,-1631584255,-1629487103,-1627389951,-1625292799,-1623195647,-1621098495,-1619001343,-1616904191,-1614807039,-1612709887,
-1610612735,-1608515583,-1606418431,-1604321279,-1602224127,-1600126975,-1598029823,-1595932671,-1593835519,-1591738367,-1589641215,-1587544063,-1585446911,-1583349759,-1581252607,-1579155455,
-1577058303,-1574961151,-1572863999,-1570766847,-1568669695,-1566572543,-1564475391,-1562378239,-1560281087,-1558183935,-1556086783,-1553989631,-1551892479,-1549795327,-1547698175,-1545601023,
-1543503871,-1541406719,-1539309567,-1537212415,-1535115263,-1533018111,-1530920959,-1528823807,-1526726655,-1524629503,-1522532351,-1520435199,-1518338047,-1516240895,-1514143743,-1512046591,
-1509949439,-1507852287,-1505755135,-1503657983,-1501560831,-1499463679,-1497366527,-1495269375,-1493172223,-1491075071,-1488977919,-1486880767,-1484783615,-1482686463,-1480589311,-1478492159,
-1476395007,-1474297855,-1472200703,-1470103551,-1468006399,-1465909247,-1463812095,-1461714943,-1459617791,-1457520639,-1455423487,-1453326335,-1451229183,-1449132031,-1447034879,-1444937727,
-1442840575,-1440743423,-1438646271,-1436549119,-1434451967,-1432354815,-1430257663,-1428160511,-1426063359,-1423966207,-1421869055,-1419771903,-1417674751,-1415577599,-1413480447,-1411383295,
-1409286143,-1407188991,-1405091839,-1402994687,-1400897535,-1398800383,-1396703231,-1394606079,-1392508927,-1390411775,-1388314623,-1386217471,-1384120319,-1382023167,-1379926015,-1377828863,
-1375731711,-1373634559,-1371537407,-1369440255,-1367343103,-1365245951,-1363148799,-1361051647,-1358954495,-1356857343,-1354760191,-1352663039,-1350565887,-1348468735,-1346371583,-1344274431,
-1342177279,-1340080127,-1337982975,-1335885823,-1333788671,-1331691519,-1329594367,-1327497215,-1325400063,-1323302911,-1321205759,-1319108607,-1317011455,-1314914303,-1312817151,-1310719999,
-1308622847,-1306525695,-1304428543,-1302331391,-1300234239,-1298137087,-1296039935,-1293942783,-1291845631,-1289748479,-1287651327,-1285554175,-1283457023,-1281359871,-1279262719,-1277165567,
-1275068415,-1272971263,-1270874111,-1268776959,-1266679807,-1264582655,-1262485503,-1260388351,-1258291199,-1256194047,-1254096895,-1251999743,-1249902591,-1247805439,-1245708287,-1243611135,
-1241513983,-1239416831,-1237319679,-1235222527,-1233125375,-1231028223,-1228931071,-1226833919,-1224736767,-1222639615,-1220542463,-1218445311,-1216348159,-1214251007,-1212153855,-1210056703,
-1207959551,-1205862399,-1203765247,-1201668095,-1199570943,-1197473791,-1195376639,-1193279487,-1191182335,-1189085183,-1186988031,-1184890879,-1182793727,-1180696575,-1178599423,-1176502271,
-1174405119,-1172307967,-1170210815,-1168113663,-1166016511,-1163919359,-1161822207,-1159725055,-1157627903,-1155530751,-1153433599,-1151336447,-1149239295,-1147142143,-1145044991,-1142947839,
-1140850687,-1138753535,-1136656383,-1134559231,-1132462079,-1130364927,-1128267775,-1126170623,-1124073471,-1121976319,-1119879167,-1117782015,-1115684863,-1113587711,-1111490559,-1109393407,
-1107296255,-1105199103,-1103101951,-1101004799,-1098907647,-1096810495,-1094713343,-1092616191,-1090519039,-1088421887,-1086324735,-1084227583,-1082130431,-1080033279,-1077936127,-1075838975,
-1073741823,-1071644671,-1069547519,-1067450367,-1065353215,-1063256063,-1061158911,-1059061759,-1056964607,-1054867455,-1052770303,-1050673151,-1048575999,-1046478847,-1044381695,-1042284543,
-1040187391,-1038090239,-1035993087,-1033895935,-1031798783,-1029701631,-1027604479,-1025507327,-1023410175,-1021313023,-1019215871,-1017118719,-1015021567,-1012924415,-1010827263,-1008730111,
-1006632959,-1004535807,-1002438655,-1000341503,-998244351,-996147199,-994050047,-991952895,-989855743,-987758591,-985661439,-983564287,-981467135,-979369983,-977272831,-975175679,
-973078527,-970981375,-968884223,-966787071,-964689919,-962592767,-960495615,-958398463,-956301311,-954204159,-952107007,-950009855,-947912703,-945815551,-943718399,-941621247,
-939524095,-937426943,-935329791,-933232639,-931135487,-929038335,-926941183,-924844031,-922746879,-920649727,-918552575,-916455423,-914358271,-912261119,-910163967,-908066815,
-905969663,-903872511,-901775359,-899678207,-897581055,-895483903,-893386751,-891289599,-889192447,-887095295,-884998143,-882900991,-880803839,-878706687,-876609535,-874512383,
-872415231,-870318079,-868220927,-866123775,-864026623,-861929471,-859832319,-857735167,-855638015,-853540863,-851443711,-849346559,-847249407,-845152255,-843055103,-840957951,
-838860799,-836763647,-834666495,-832569343,-830472191,-828375039,-826277887,-824180735,-822083583,-819986431,-817889279,-815792127,-813694975,-811597823,-809500671,-807403519,
-805306367,-803209215,-801112063,-799014911,-796917759,-794820607,-792723455,-790626303,-788529151,-786431999,-784334847,-782237695,-780140543,-778043391,-775946239,-773849087,
-771751935,-769654783,-767557631,-765460479,-763363327,-761266175,-759169023,-757071871,-754974719,-752877567,-750780415,-748683263,-746586111,-744488959,-742391807,-740294655,
-738197503,-736100351,-734003199,-731906047,-729808895,-727711743,-725614591,-723517439,-721420287,-719323135,-717225983,-715128831,-713031679,-710934527,-708837375,-706740223,
-704643071,-702545919,-700448767,-698351615,-696254463,-694157311,-692060159,-689963007,-687865855,-685768703,-683671551,-681574399,-679477247,-677380095,-675282943,-673185791,
-671088639,-668991487,-666894335,-664797183,-662700031,-660602879,-658505727,-656408575,-654311423,-652214271,-650117119,-648019967,-645922815,-643825663,-641728511,-639631359,
-637534207,-635437055,-633339903,-631242751,-629145599,-627048447,-624951295,-622854143,-620756991,-618659839,-616562687,-614465535,-612368383,-610271231,-608174079,-606076927,
-603979775,-601882623,-599785471,-597688319,-595591167,-593494015,-591396863,-589299711,-587202559,-585105407,-583008255,-580911103,-578813951,-576716799,-574619647,-572522495,
-570425343,-568328191,-566231039,-564133887,-562036735,-559939583,-557842431,-555745279,-553648127,-551550975,-549453823,-547356671,-545259519,-543162367,-541065215,-538968063,
-536870911,-534773759,-532676607,-530579455,-528482303,-526385151,-524287999,-522190847,-520093695,-517996543,-515899391,-513802239,-511705087,-509607935,-507510783,-505413631,
-503316479,-501219327,-499122175,-497025023,-494927871,-492830719,-490733567,-488636415,-486539263,-484442111,-482344959,-480247807,-478150655,-476053503,-473956351,-471859199,
-469762047,-467664895,-465567743,-463470591,-461373439,-459276287,-457179135,-455081983,-452984831,-450887679,-448790527,-446693375,-444596223,-442499071,-440401919,-438304767,
-436207615,-434110463,-432013311,-429916159,-427819007,-425721855,-423624703,-421527551,-419430399,-417333247,-415236095,-413138943,-411041791,-408944639,-406847487,-404750335,
-402653183,-400556031,-398458879,-396361727,-394264575,-392167423,-390070271,-387973119,-385875967,-383778815,-381681663,-379584511,-377487359,-375390207,-373293055,-371195903,
-369098751,-367001599,-364904447,-362807295,-360710143,-358612991,-356515839,-354418687,-352321535,-350224383,-348127231,-346030079,-343932927,-341835775,-339738623,-337641471,
-335544319,-333447167,-331350015,-329252863,-327155711,-325058559,-322961407,-320864255,-318767103,-316669951,-314572799,-312475647,-310378495,-308281343,-306184191,-304087039,
-301989887,-299892735,-297795583,-295698431,-293601279,-291504127,-289406975,-287309823,-285212671,-283115519,-281018367,-278921215,-276824063,-274726911,-272629759,-270532607,
-268435455,-266338303,-264241151,-262143999,-260046847,-257949695,-255852543,-253755391,-251658239,-249561087,-247463935,-245366783,-243269631,-241172479,-239075327,-236978175,
-234881023,-232783871,-230686719,-228589567,-226492415,-224395263,-222298111,-220200959,-218103807,-216006655,-213909503,-211812351,-209715199,-207618047,-205520895,-203423743,
-201326591,-199229439,-197132287,-195035135,-192937983,-190840831,-188743679,-186646527,-184549375,-182452223,-180355071,-178257919,-176160767,-174063615,-171966463,-169869311,
-167772159,-165675007,-163577855,-161480703,-159383551,-157286399,-155189247,-153092095,-150994943,-148897791,-146800639,-144703487,-142606335,-140509183,-138412031,-136314879,
-134217727,-132120575,-130023423,-127926271,-125829119,-123731967,-121634815,-119537663,-117440511,-115343359,-113246207,-111149055,-109051903,-106954751,-104857599,-102760447,
-100663295,-98566143,-96468991,-94371839,-92274687,-90177535,-88080383,-85983231,-83886079,-81788927,-79691775,-77594623,-75497471,-73400319,-71303167,-69206015,
-67108863,-65011711,-62914559,-60817407,-58720255,-56623103,-54525951,-52428799,-50331647,-48234495,-46137343,-44040191,-41943039,-39845887,-37748735,-35651583,
-33554431,-31457279,-29360127,-27262975,-25165823,-23068671,-20971519,-18874367,-16777215,-14680063,-12582911,-10485759,-8388607,-6291455,-4194303,-2097151

    };

    const fixed SquareWave[N_WAVE] = {
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,
-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647,-2147483647
    };

    const fixed SawWave[N_WAVE] = {
2146959359,2145910783,2144862207,2143813631,2142765055,2141716479,2140667903,2139619327,2138570751,2137522175,2136473599,2135425023,2134376447,2133327871,2132279295,2131230719,
2130182143,2129133567,2128084991,2127036415,2125987839,2124939263,2123890687,2122842111,2121793535,2120744959,2119696383,2118647807,2117599231,2116550655,2115502079,2114453503,
2113404927,2112356351,2111307775,2110259199,2109210623,2108162047,2107113471,2106064895,2105016319,2103967743,2102919167,2101870591,2100822015,2099773439,2098724863,2097676287,
2096627711,2095579135,2094530559,2093481983,2092433407,2091384831,2090336255,2089287679,2088239103,2087190527,2086141951,2085093375,2084044799,2082996223,2081947647,2080899071,
2079850495,2078801919,2077753343,2076704767,2075656191,2074607615,2073559039,2072510463,2071461887,2070413311,2069364735,2068316159,2067267583,2066219007,2065170431,2064121855,
2063073279,2062024703,2060976127,2059927551,2058878975,2057830399,2056781823,2055733247,2054684671,2053636095,2052587519,2051538943,2050490367,2049441791,2048393215,2047344639,
2046296063,2045247487,2044198911,2043150335,2042101759,2041053183,2040004607,2038956031,2037907455,2036858879,2035810303,2034761727,2033713151,2032664575,2031615999,2030567423,
2029518847,2028470271,2027421695,2026373119,2025324543,2024275967,2023227391,2022178815,2021130239,2020081663,2019033087,2017984511,2016935935,2015887359,2014838783,2013790207,
2012741631,2011693055,2010644479,2009595903,2008547327,2007498751,2006450175,2005401599,2004353023,2003304447,2002255871,2001207295,2000158719,1999110143,1998061567,1997012991,
1995964415,1994915839,1993867263,1992818687,1991770111,1990721535,1989672959,1988624383,1987575807,1986527231,1985478655,1984430079,1983381503,1982332927,1981284351,1980235775,
1979187199,1978138623,1977090047,1976041471,1974992895,1973944319,1972895743,1971847167,1970798591,1969750015,1968701439,1967652863,1966604287,1965555711,1964507135,1963458559,
1962409983,1961361407,1960312831,1959264255,1958215679,1957167103,1956118527,1955069951,1954021375,1952972799,1951924223,1950875647,1949827071,1948778495,1947729919,1946681343,
1945632767,1944584191,1943535615,1942487039,1941438463,1940389887,1939341311,1938292735,1937244159,1936195583,1935147007,1934098431,1933049855,1932001279,1930952703,1929904127,
1928855551,1927806975,1926758399,1925709823,1924661247,1923612671,1922564095,1921515519,1920466943,1919418367,1918369791,1917321215,1916272639,1915224063,1914175487,1913126911,
1912078335,1911029759,1909981183,1908932607,1907884031,1906835455,1905786879,1904738303,1903689727,1902641151,1901592575,1900543999,1899495423,1898446847,1897398271,1896349695,
1895301119,1894252543,1893203967,1892155391,1891106815,1890058239,1889009663,1887961087,1886912511,1885863935,1884815359,1883766783,1882718207,1881669631,1880621055,1879572479,
1878523903,1877475327,1876426751,1875378175,1874329599,1873281023,1872232447,1871183871,1870135295,1869086719,1868038143,1866989567,1865940991,1864892415,1863843839,1862795263,
1861746687,1860698111,1859649535,1858600959,1857552383,1856503807,1855455231,1854406655,1853358079,1852309503,1851260927,1850212351,1849163775,1848115199,1847066623,1846018047,
1844969471,1843920895,1842872319,1841823743,1840775167,1839726591,1838678015,1837629439,1836580863,1835532287,1834483711,1833435135,1832386559,1831337983,1830289407,1829240831,
1828192255,1827143679,1826095103,1825046527,1823997951,1822949375,1821900799,1820852223,1819803647,1818755071,1817706495,1816657919,1815609343,1814560767,1813512191,1812463615,
1811415039,1810366463,1809317887,1808269311,1807220735,1806172159,1805123583,1804075007,1803026431,1801977855,1800929279,1799880703,1798832127,1797783551,1796734975,1795686399,
1794637823,1793589247,1792540671,1791492095,1790443519,1789394943,1788346367,1787297791,1786249215,1785200639,1784152063,1783103487,1782054911,1781006335,1779957759,1778909183,
1777860607,1776812031,1775763455,1774714879,1773666303,1772617727,1771569151,1770520575,1769471999,1768423423,1767374847,1766326271,1765277695,1764229119,1763180543,1762131967,
1761083391,1760034815,1758986239,1757937663,1756889087,1755840511,1754791935,1753743359,1752694783,1751646207,1750597631,1749549055,1748500479,1747451903,1746403327,1745354751,
1744306175,1743257599,1742209023,1741160447,1740111871,1739063295,1738014719,1736966143,1735917567,1734868991,1733820415,1732771839,1731723263,1730674687,1729626111,1728577535,
1727528959,1726480383,1725431807,1724383231,1723334655,1722286079,1721237503,1720188927,1719140351,1718091775,1717043199,1715994623,1714946047,1713897471,1712848895,1711800319,
1710751743,1709703167,1708654591,1707606015,1706557439,1705508863,1704460287,1703411711,1702363135,1701314559,1700265983,1699217407,1698168831,1697120255,1696071679,1695023103,
1693974527,1692925951,1691877375,1690828799,1689780223,1688731647,1687683071,1686634495,1685585919,1684537343,1683488767,1682440191,1681391615,1680343039,1679294463,1678245887,
1677197311,1676148735,1675100159,1674051583,1673003007,1671954431,1670905855,1669857279,1668808703,1667760127,1666711551,1665662975,1664614399,1663565823,1662517247,1661468671,
1660420095,1659371519,1658322943,1657274367,1656225791,1655177215,1654128639,1653080063,1652031487,1650982911,1649934335,1648885759,1647837183,1646788607,1645740031,1644691455,
1643642879,1642594303,1641545727,1640497151,1639448575,1638399999,1637351423,1636302847,1635254271,1634205695,1633157119,1632108543,1631059967,1630011391,1628962815,1627914239,
1626865663,1625817087,1624768511,1623719935,1622671359,1621622783,1620574207,1619525631,1618477055,1617428479,1616379903,1615331327,1614282751,1613234175,1612185599,1611137023,
1610088447,1609039871,1607991295,1606942719,1605894143,1604845567,1603796991,1602748415,1601699839,1600651263,1599602687,1598554111,1597505535,1596456959,1595408383,1594359807,
1593311231,1592262655,1591214079,1590165503,1589116927,1588068351,1587019775,1585971199,1584922623,1583874047,1582825471,1581776895,1580728319,1579679743,1578631167,1577582591,
1576534015,1575485439,1574436863,1573388287,1572339711,1571291135,1570242559,1569193983,1568145407,1567096831,1566048255,1564999679,1563951103,1562902527,1561853951,1560805375,
1559756799,1558708223,1557659647,1556611071,1555562495,1554513919,1553465343,1552416767,1551368191,1550319615,1549271039,1548222463,1547173887,1546125311,1545076735,1544028159,
1542979583,1541931007,1540882431,1539833855,1538785279,1537736703,1536688127,1535639551,1534590975,1533542399,1532493823,1531445247,1530396671,1529348095,1528299519,1527250943,
1526202367,1525153791,1524105215,1523056639,1522008063,1520959487,1519910911,1518862335,1517813759,1516765183,1515716607,1514668031,1513619455,1512570879,1511522303,1510473727,
1509425151,1508376575,1507327999,1506279423,1505230847,1504182271,1503133695,1502085119,1501036543,1499987967,1498939391,1497890815,1496842239,1495793663,1494745087,1493696511,
1492647935,1491599359,1490550783,1489502207,1488453631,1487405055,1486356479,1485307903,1484259327,1483210751,1482162175,1481113599,1480065023,1479016447,1477967871,1476919295,
1475870719,1474822143,1473773567,1472724991,1471676415,1470627839,1469579263,1468530687,1467482111,1466433535,1465384959,1464336383,1463287807,1462239231,1461190655,1460142079,
1459093503,1458044927,1456996351,1455947775,1454899199,1453850623,1452802047,1451753471,1450704895,1449656319,1448607743,1447559167,1446510591,1445462015,1444413439,1443364863,
1442316287,1441267711,1440219135,1439170559,1438121983,1437073407,1436024831,1434976255,1433927679,1432879103,1431830527,1430781951,1429733375,1428684799,1427636223,1426587647,
1425539071,1424490495,1423441919,1422393343,1421344767,1420296191,1419247615,1418199039,1417150463,1416101887,1415053311,1414004735,1412956159,1411907583,1410859007,1409810431,
1408761855,1407713279,1406664703,1405616127,1404567551,1403518975,1402470399,1401421823,1400373247,1399324671,1398276095,1397227519,1396178943,1395130367,1394081791,1393033215,
1391984639,1390936063,1389887487,1388838911,1387790335,1386741759,1385693183,1384644607,1383596031,1382547455,1381498879,1380450303,1379401727,1378353151,1377304575,1376255999,
1375207423,1374158847,1373110271,1372061695,1371013119,1369964543,1368915967,1367867391,1366818815,1365770239,1364721663,1363673087,1362624511,1361575935,1360527359,1359478783,
1358430207,1357381631,1356333055,1355284479,1354235903,1353187327,1352138751,1351090175,1350041599,1348993023,1347944447,1346895871,1345847295,1344798719,1343750143,1342701567,
1341652991,1340604415,1339555839,1338507263,1337458687,1336410111,1335361535,1334312959,1333264383,1332215807,1331167231,1330118655,1329070079,1328021503,1326972927,1325924351,
1324875775,1323827199,1322778623,1321730047,1320681471,1319632895,1318584319,1317535743,1316487167,1315438591,1314390015,1313341439,1312292863,1311244287,1310195711,1309147135,
1308098559,1307049983,1306001407,1304952831,1303904255,1302855679,1301807103,1300758527,1299709951,1298661375,1297612799,1296564223,1295515647,1294467071,1293418495,1292369919,
1291321343,1290272767,1289224191,1288175615,1287127039,1286078463,1285029887,1283981311,1282932735,1281884159,1280835583,1279787007,1278738431,1277689855,1276641279,1275592703,
1274544127,1273495551,1272446975,1271398399,1270349823,1269301247,1268252671,1267204095,1266155519,1265106943,1264058367,1263009791,1261961215,1260912639,1259864063,1258815487,
1257766911,1256718335,1255669759,1254621183,1253572607,1252524031,1251475455,1250426879,1249378303,1248329727,1247281151,1246232575,1245183999,1244135423,1243086847,1242038271,
1240989695,1239941119,1238892543,1237843967,1236795391,1235746815,1234698239,1233649663,1232601087,1231552511,1230503935,1229455359,1228406783,1227358207,1226309631,1225261055,
1224212479,1223163903,1222115327,1221066751,1220018175,1218969599,1217921023,1216872447,1215823871,1214775295,1213726719,1212678143,1211629567,1210580991,1209532415,1208483839,
1207435263,1206386687,1205338111,1204289535,1203240959,1202192383,1201143807,1200095231,1199046655,1197998079,1196949503,1195900927,1194852351,1193803775,1192755199,1191706623,
1190658047,1189609471,1188560895,1187512319,1186463743,1185415167,1184366591,1183318015,1182269439,1181220863,1180172287,1179123711,1178075135,1177026559,1175977983,1174929407,
1173880831,1172832255,1171783679,1170735103,1169686527,1168637951,1167589375,1166540799,1165492223,1164443647,1163395071,1162346495,1161297919,1160249343,1159200767,1158152191,
1157103615,1156055039,1155006463,1153957887,1152909311,1151860735,1150812159,1149763583,1148715007,1147666431,1146617855,1145569279,1144520703,1143472127,1142423551,1141374975,
1140326399,1139277823,1138229247,1137180671,1136132095,1135083519,1134034943,1132986367,1131937791,1130889215,1129840639,1128792063,1127743487,1126694911,1125646335,1124597759,
1123549183,1122500607,1121452031,1120403455,1119354879,1118306303,1117257727,1116209151,1115160575,1114111999,1113063423,1112014847,1110966271,1109917695,1108869119,1107820543,
1106771967,1105723391,1104674815,1103626239,1102577663,1101529087,1100480511,1099431935,1098383359,1097334783,1096286207,1095237631,1094189055,1093140479,1092091903,1091043327,
1089994751,1088946175,1087897599,1086849023,1085800447,1084751871,1083703295,1082654719,1081606143,1080557567,1079508991,1078460415,1077411839,1076363263,1075314687,1074266111,
1073217535,1072168959,1071120383,1070071807,1069023231,1067974655,1066926079,1065877503,1064828927,1063780351,1062731775,1061683199,1060634623,1059586047,1058537471,1057488895,
1056440319,1055391743,1054343167,1053294591,1052246015,1051197439,1050148863,1049100287,1048051711,1047003135,1045954559,1044905983,1043857407,1042808831,1041760255,1040711679,
1039663103,1038614527,1037565951,1036517375,1035468799,1034420223,1033371647,1032323071,1031274495,1030225919,1029177343,1028128767,1027080191,1026031615,1024983039,1023934463,
1022885887,1021837311,1020788735,1019740159,1018691583,1017643007,1016594431,1015545855,1014497279,1013448703,1012400127,1011351551,1010302975,1009254399,1008205823,1007157247,
1006108671,1005060095,1004011519,1002962943,1001914367,1000865791,999817215,998768639,997720063,996671487,995622911,994574335,993525759,992477183,991428607,990380031,
989331455,988282879,987234303,986185727,985137151,984088575,983039999,981991423,980942847,979894271,978845695,977797119,976748543,975699967,974651391,973602815,
972554239,971505663,970457087,969408511,968359935,967311359,966262783,965214207,964165631,963117055,962068479,961019903,959971327,958922751,957874175,956825599,
955777023,954728447,953679871,952631295,951582719,950534143,949485567,948436991,947388415,946339839,945291263,944242687,943194111,942145535,941096959,940048383,
938999807,937951231,936902655,935854079,934805503,933756927,932708351,931659775,930611199,929562623,928514047,927465471,926416895,925368319,924319743,923271167,
922222591,921174015,920125439,919076863,918028287,916979711,915931135,914882559,913833983,912785407,911736831,910688255,909639679,908591103,907542527,906493951,
905445375,904396799,903348223,902299647,901251071,900202495,899153919,898105343,897056767,896008191,894959615,893911039,892862463,891813887,890765311,889716735,
888668159,887619583,886571007,885522431,884473855,883425279,882376703,881328127,880279551,879230975,878182399,877133823,876085247,875036671,873988095,872939519,
871890943,870842367,869793791,868745215,867696639,866648063,865599487,864550911,863502335,862453759,861405183,860356607,859308031,858259455,857210879,856162303,
855113727,854065151,853016575,851967999,850919423,849870847,848822271,847773695,846725119,845676543,844627967,843579391,842530815,841482239,840433663,839385087,
838336511,837287935,836239359,835190783,834142207,833093631,832045055,830996479,829947903,828899327,827850751,826802175,825753599,824705023,823656447,822607871,
821559295,820510719,819462143,818413567,817364991,816316415,815267839,814219263,813170687,812122111,811073535,810024959,808976383,807927807,806879231,805830655,
804782079,803733503,802684927,801636351,800587775,799539199,798490623,797442047,796393471,795344895,794296319,793247743,792199167,791150591,790102015,789053439,
788004863,786956287,785907711,784859135,783810559,782761983,781713407,780664831,779616255,778567679,777519103,776470527,775421951,774373375,773324799,772276223,
771227647,770179071,769130495,768081919,767033343,765984767,764936191,763887615,762839039,761790463,760741887,759693311,758644735,757596159,756547583,755499007,
754450431,753401855,752353279,751304703,750256127,749207551,748158975,747110399,746061823,745013247,743964671,742916095,741867519,740818943,739770367,738721791,
737673215,736624639,735576063,734527487,733478911,732430335,731381759,730333183,729284607,728236031,727187455,726138879,725090303,724041727,722993151,721944575,
720895999,719847423,718798847,717750271,716701695,715653119,714604543,713555967,712507391,711458815,710410239,709361663,708313087,707264511,706215935,705167359,
704118783,703070207,702021631,700973055,699924479,698875903,697827327,696778751,695730175,694681599,693633023,692584447,691535871,690487295,689438719,688390143,
687341567,686292991,685244415,684195839,683147263,682098687,681050111,680001535,678952959,677904383,676855807,675807231,674758655,673710079,672661503,671612927,
670564351,669515775,668467199,667418623,666370047,665321471,664272895,663224319,662175743,661127167,660078591,659030015,657981439,656932863,655884287,654835711,
653787135,652738559,651689983,650641407,649592831,648544255,647495679,646447103,645398527,644349951,643301375,642252799,641204223,640155647,639107071,638058495,
637009919,635961343,634912767,633864191,632815615,631767039,630718463,629669887,628621311,627572735,626524159,625475583,624427007,623378431,622329855,621281279,
620232703,619184127,618135551,617086975,616038399,614989823,613941247,612892671,611844095,610795519,609746943,608698367,607649791,606601215,605552639,604504063,
603455487,602406911,601358335,600309759,599261183,598212607,597164031,596115455,595066879,594018303,592969727,591921151,590872575,589823999,588775423,587726847,
586678271,585629695,584581119,583532543,582483967,581435391,580386815,579338239,578289663,577241087,576192511,575143935,574095359,573046783,571998207,570949631,
569901055,568852479,567803903,566755327,565706751,564658175,563609599,562561023,561512447,560463871,559415295,558366719,557318143,556269567,555220991,554172415,
553123839,552075263,551026687,549978111,548929535,547880959,546832383,545783807,544735231,543686655,542638079,541589503,540540927,539492351,538443775,537395199,
536346623,535298047,534249471,533200895,532152319,531103743,530055167,529006591,527958015,526909439,525860863,524812287,523763711,522715135,521666559,520617983,
519569407,518520831,517472255,516423679,515375103,514326527,513277951,512229375,511180799,510132223,509083647,508035071,506986495,505937919,504889343,503840767,
502792191,501743615,500695039,499646463,498597887,497549311,496500735,495452159,494403583,493355007,492306431,491257855,490209279,489160703,488112127,487063551,
486014975,484966399,483917823,482869247,481820671,480772095,479723519,478674943,477626367,476577791,475529215,474480639,473432063,472383487,471334911,470286335,
469237759,468189183,467140607,466092031,465043455,463994879,462946303,461897727,460849151,459800575,458751999,457703423,456654847,455606271,454557695,453509119,
452460543,451411967,450363391,449314815,448266239,447217663,446169087,445120511,444071935,443023359,441974783,440926207,439877631,438829055,437780479,436731903,
435683327,434634751,433586175,432537599,431489023,430440447,429391871,428343295,427294719,426246143,425197567,424148991,423100415,422051839,421003263,419954687,
418906111,417857535,416808959,415760383,414711807,413663231,412614655,411566079,410517503,409468927,408420351,407371775,406323199,405274623,404226047,403177471,
402128895,401080319,400031743,398983167,397934591,396886015,395837439,394788863,393740287,392691711,391643135,390594559,389545983,388497407,387448831,386400255,
385351679,384303103,383254527,382205951,381157375,380108799,379060223,378011647,376963071,375914495,374865919,373817343,372768767,371720191,370671615,369623039,
368574463,367525887,366477311,365428735,364380159,363331583,362283007,361234431,360185855,359137279,358088703,357040127,355991551,354942975,353894399,352845823,
351797247,350748671,349700095,348651519,347602943,346554367,345505791,344457215,343408639,342360063,341311487,340262911,339214335,338165759,337117183,336068607,
335020031,333971455,332922879,331874303,330825727,329777151,328728575,327679999,326631423,325582847,324534271,323485695,322437119,321388543,320339967,319291391,
318242815,317194239,316145663,315097087,314048511,312999935,311951359,310902783,309854207,308805631,307757055,306708479,305659903,304611327,303562751,302514175,
301465599,300417023,299368447,298319871,297271295,296222719,295174143,294125567,293076991,292028415,290979839,289931263,288882687,287834111,286785535,285736959,
284688383,283639807,282591231,281542655,280494079,279445503,278396927,277348351,276299775,275251199,274202623,273154047,272105471,271056895,270008319,268959743,
267911167,266862591,265814015,264765439,263716863,262668287,261619711,260571135,259522559,258473983,257425407,256376831,255328255,254279679,253231103,252182527,
251133951,250085375,249036799,247988223,246939647,245891071,244842495,243793919,242745343,241696767,240648191,239599615,238551039,237502463,236453887,235405311,
234356735,233308159,232259583,231211007,230162431,229113855,228065279,227016703,225968127,224919551,223870975,222822399,221773823,220725247,219676671,218628095,
217579519,216530943,215482367,214433791,213385215,212336639,211288063,210239487,209190911,208142335,207093759,206045183,204996607,203948031,202899455,201850879,
200802303,199753727,198705151,197656575,196607999,195559423,194510847,193462271,192413695,191365119,190316543,189267967,188219391,187170815,186122239,185073663,
184025087,182976511,181927935,180879359,179830783,178782207,177733631,176685055,175636479,174587903,173539327,172490751,171442175,170393599,169345023,168296447,
167247871,166199295,165150719,164102143,163053567,162004991,160956415,159907839,158859263,157810687,156762111,155713535,154664959,153616383,152567807,151519231,
150470655,149422079,148373503,147324927,146276351,145227775,144179199,143130623,142082047,141033471,139984895,138936319,137887743,136839167,135790591,134742015,
133693439,132644863,131596287,130547711,129499135,128450559,127401983,126353407,125304831,124256255,123207679,122159103,121110527,120061951,119013375,117964799,
116916223,115867647,114819071,113770495,112721919,111673343,110624767,109576191,108527615,107479039,106430463,105381887,104333311,103284735,102236159,101187583,
100139007,99090431,98041855,96993279,95944703,94896127,93847551,92798975,91750399,90701823,89653247,88604671,87556095,86507519,85458943,84410367,
83361791,82313215,81264639,80216063,79167487,78118911,77070335,76021759,74973183,73924607,72876031,71827455,70778879,69730303,68681727,67633151,
66584575,65535999,64487423,63438847,62390271,61341695,60293119,59244543,58195967,57147391,56098815,55050239,54001663,52953087,51904511,50855935,
49807359,48758783,47710207,46661631,45613055,44564479,43515903,42467327,41418751,40370175,39321599,38273023,37224447,36175871,35127295,34078719,
33030143,31981567,30932991,29884415,28835839,27787263,26738687,25690111,24641535,23592959,22544383,21495807,20447231,19398655,18350079,17301503,
16252927,15204351,14155775,13107199,12058623,11010047,9961471,8912895,7864319,6815743,5767167,4718591,3670015,2621439,1572863,524287,
-524288,-1572864,-2621440,-3670016,-4718592,-5767168,-6815744,-7864320,-8912896,-9961472,-11010048,-12058624,-13107200,-14155776,-15204352,-16252928,
-17301504,-18350080,-19398656,-20447232,-21495808,-22544384,-23592960,-24641536,-25690112,-26738688,-27787264,-28835840,-29884416,-30932992,-31981568,-33030144,
-34078720,-35127296,-36175872,-37224448,-38273024,-39321600,-40370176,-41418752,-42467328,-43515904,-44564480,-45613056,-46661632,-47710208,-48758784,-49807360,
-50855936,-51904512,-52953088,-54001664,-55050240,-56098816,-57147392,-58195968,-59244544,-60293120,-61341696,-62390272,-63438848,-64487424,-65536000,-66584576,
-67633152,-68681728,-69730304,-70778880,-71827456,-72876032,-73924608,-74973184,-76021760,-77070336,-78118912,-79167488,-80216064,-81264640,-82313216,-83361792,
-84410368,-85458944,-86507520,-87556096,-88604672,-89653248,-90701824,-91750400,-92798976,-93847552,-94896128,-95944704,-96993280,-98041856,-99090432,-100139008,
-101187584,-102236160,-103284736,-104333312,-105381888,-106430464,-107479040,-108527616,-109576192,-110624768,-111673344,-112721920,-113770496,-114819072,-115867648,-116916224,
-117964800,-119013376,-120061952,-121110528,-122159104,-123207680,-124256256,-125304832,-126353408,-127401984,-128450560,-129499136,-130547712,-131596288,-132644864,-133693440,
-134742016,-135790592,-136839168,-137887744,-138936320,-139984896,-141033472,-142082048,-143130624,-144179200,-145227776,-146276352,-147324928,-148373504,-149422080,-150470656,
-151519232,-152567808,-153616384,-154664960,-155713536,-156762112,-157810688,-158859264,-159907840,-160956416,-162004992,-163053568,-164102144,-165150720,-166199296,-167247872,
-168296448,-169345024,-170393600,-171442176,-172490752,-173539328,-174587904,-175636480,-176685056,-177733632,-178782208,-179830784,-180879360,-181927936,-182976512,-184025088,
-185073664,-186122240,-187170816,-188219392,-189267968,-190316544,-191365120,-192413696,-193462272,-194510848,-195559424,-196608000,-197656576,-198705152,-199753728,-200802304,
-201850880,-202899456,-203948032,-204996608,-206045184,-207093760,-208142336,-209190912,-210239488,-211288064,-212336640,-213385216,-214433792,-215482368,-216530944,-217579520,
-218628096,-219676672,-220725248,-221773824,-222822400,-223870976,-224919552,-225968128,-227016704,-228065280,-229113856,-230162432,-231211008,-232259584,-233308160,-234356736,
-235405312,-236453888,-237502464,-238551040,-239599616,-240648192,-241696768,-242745344,-243793920,-244842496,-245891072,-246939648,-247988224,-249036800,-250085376,-251133952,
-252182528,-253231104,-254279680,-255328256,-256376832,-257425408,-258473984,-259522560,-260571136,-261619712,-262668288,-263716864,-264765440,-265814016,-266862592,-267911168,
-268959744,-270008320,-271056896,-272105472,-273154048,-274202624,-275251200,-276299776,-277348352,-278396928,-279445504,-280494080,-281542656,-282591232,-283639808,-284688384,
-285736960,-286785536,-287834112,-288882688,-289931264,-290979840,-292028416,-293076992,-294125568,-295174144,-296222720,-297271296,-298319872,-299368448,-300417024,-301465600,
-302514176,-303562752,-304611328,-305659904,-306708480,-307757056,-308805632,-309854208,-310902784,-311951360,-312999936,-314048512,-315097088,-316145664,-317194240,-318242816,
-319291392,-320339968,-321388544,-322437120,-323485696,-324534272,-325582848,-326631424,-327680000,-328728576,-329777152,-330825728,-331874304,-332922880,-333971456,-335020032,
-336068608,-337117184,-338165760,-339214336,-340262912,-341311488,-342360064,-343408640,-344457216,-345505792,-346554368,-347602944,-348651520,-349700096,-350748672,-351797248,
-352845824,-353894400,-354942976,-355991552,-357040128,-358088704,-359137280,-360185856,-361234432,-362283008,-363331584,-364380160,-365428736,-366477312,-367525888,-368574464,
-369623040,-370671616,-371720192,-372768768,-373817344,-374865920,-375914496,-376963072,-378011648,-379060224,-380108800,-381157376,-382205952,-383254528,-384303104,-385351680,
-386400256,-387448832,-388497408,-389545984,-390594560,-391643136,-392691712,-393740288,-394788864,-395837440,-396886016,-397934592,-398983168,-400031744,-401080320,-402128896,
-403177472,-404226048,-405274624,-406323200,-407371776,-408420352,-409468928,-410517504,-411566080,-412614656,-413663232,-414711808,-415760384,-416808960,-417857536,-418906112,
-419954688,-421003264,-422051840,-423100416,-424148992,-425197568,-426246144,-427294720,-428343296,-429391872,-430440448,-431489024,-432537600,-433586176,-434634752,-435683328,
-436731904,-437780480,-438829056,-439877632,-440926208,-441974784,-443023360,-444071936,-445120512,-446169088,-447217664,-448266240,-449314816,-450363392,-451411968,-452460544,
-453509120,-454557696,-455606272,-456654848,-457703424,-458752000,-459800576,-460849152,-461897728,-462946304,-463994880,-465043456,-466092032,-467140608,-468189184,-469237760,
-470286336,-471334912,-472383488,-473432064,-474480640,-475529216,-476577792,-477626368,-478674944,-479723520,-480772096,-481820672,-482869248,-483917824,-484966400,-486014976,
-487063552,-488112128,-489160704,-490209280,-491257856,-492306432,-493355008,-494403584,-495452160,-496500736,-497549312,-498597888,-499646464,-500695040,-501743616,-502792192,
-503840768,-504889344,-505937920,-506986496,-508035072,-509083648,-510132224,-511180800,-512229376,-513277952,-514326528,-515375104,-516423680,-517472256,-518520832,-519569408,
-520617984,-521666560,-522715136,-523763712,-524812288,-525860864,-526909440,-527958016,-529006592,-530055168,-531103744,-532152320,-533200896,-534249472,-535298048,-536346624,
-537395200,-538443776,-539492352,-540540928,-541589504,-542638080,-543686656,-544735232,-545783808,-546832384,-547880960,-548929536,-549978112,-551026688,-552075264,-553123840,
-554172416,-555220992,-556269568,-557318144,-558366720,-559415296,-560463872,-561512448,-562561024,-563609600,-564658176,-565706752,-566755328,-567803904,-568852480,-569901056,
-570949632,-571998208,-573046784,-574095360,-575143936,-576192512,-577241088,-578289664,-579338240,-580386816,-581435392,-582483968,-583532544,-584581120,-585629696,-586678272,
-587726848,-588775424,-589824000,-590872576,-591921152,-592969728,-594018304,-595066880,-596115456,-597164032,-598212608,-599261184,-600309760,-601358336,-602406912,-603455488,
-604504064,-605552640,-606601216,-607649792,-608698368,-609746944,-610795520,-611844096,-612892672,-613941248,-614989824,-616038400,-617086976,-618135552,-619184128,-620232704,
-621281280,-622329856,-623378432,-624427008,-625475584,-626524160,-627572736,-628621312,-629669888,-630718464,-631767040,-632815616,-633864192,-634912768,-635961344,-637009920,
-638058496,-639107072,-640155648,-641204224,-642252800,-643301376,-644349952,-645398528,-646447104,-647495680,-648544256,-649592832,-650641408,-651689984,-652738560,-653787136,
-654835712,-655884288,-656932864,-657981440,-659030016,-660078592,-661127168,-662175744,-663224320,-664272896,-665321472,-666370048,-667418624,-668467200,-669515776,-670564352,
-671612928,-672661504,-673710080,-674758656,-675807232,-676855808,-677904384,-678952960,-680001536,-681050112,-682098688,-683147264,-684195840,-685244416,-686292992,-687341568,
-688390144,-689438720,-690487296,-691535872,-692584448,-693633024,-694681600,-695730176,-696778752,-697827328,-698875904,-699924480,-700973056,-702021632,-703070208,-704118784,
-705167360,-706215936,-707264512,-708313088,-709361664,-710410240,-711458816,-712507392,-713555968,-714604544,-715653120,-716701696,-717750272,-718798848,-719847424,-720896000,
-721944576,-722993152,-724041728,-725090304,-726138880,-727187456,-728236032,-729284608,-730333184,-731381760,-732430336,-733478912,-734527488,-735576064,-736624640,-737673216,
-738721792,-739770368,-740818944,-741867520,-742916096,-743964672,-745013248,-746061824,-747110400,-748158976,-749207552,-750256128,-751304704,-752353280,-753401856,-754450432,
-755499008,-756547584,-757596160,-758644736,-759693312,-760741888,-761790464,-762839040,-763887616,-764936192,-765984768,-767033344,-768081920,-769130496,-770179072,-771227648,
-772276224,-773324800,-774373376,-775421952,-776470528,-777519104,-778567680,-779616256,-780664832,-781713408,-782761984,-783810560,-784859136,-785907712,-786956288,-788004864,
-789053440,-790102016,-791150592,-792199168,-793247744,-794296320,-795344896,-796393472,-797442048,-798490624,-799539200,-800587776,-801636352,-802684928,-803733504,-804782080,
-805830656,-806879232,-807927808,-808976384,-810024960,-811073536,-812122112,-813170688,-814219264,-815267840,-816316416,-817364992,-818413568,-819462144,-820510720,-821559296,
-822607872,-823656448,-824705024,-825753600,-826802176,-827850752,-828899328,-829947904,-830996480,-832045056,-833093632,-834142208,-835190784,-836239360,-837287936,-838336512,
-839385088,-840433664,-841482240,-842530816,-843579392,-844627968,-845676544,-846725120,-847773696,-848822272,-849870848,-850919424,-851968000,-853016576,-854065152,-855113728,
-856162304,-857210880,-858259456,-859308032,-860356608,-861405184,-862453760,-863502336,-864550912,-865599488,-866648064,-867696640,-868745216,-869793792,-870842368,-871890944,
-872939520,-873988096,-875036672,-876085248,-877133824,-878182400,-879230976,-880279552,-881328128,-882376704,-883425280,-884473856,-885522432,-886571008,-887619584,-888668160,
-889716736,-890765312,-891813888,-892862464,-893911040,-894959616,-896008192,-897056768,-898105344,-899153920,-900202496,-901251072,-902299648,-903348224,-904396800,-905445376,
-906493952,-907542528,-908591104,-909639680,-910688256,-911736832,-912785408,-913833984,-914882560,-915931136,-916979712,-918028288,-919076864,-920125440,-921174016,-922222592,
-923271168,-924319744,-925368320,-926416896,-927465472,-928514048,-929562624,-930611200,-931659776,-932708352,-933756928,-934805504,-935854080,-936902656,-937951232,-938999808,
-940048384,-941096960,-942145536,-943194112,-944242688,-945291264,-946339840,-947388416,-948436992,-949485568,-950534144,-951582720,-952631296,-953679872,-954728448,-955777024,
-956825600,-957874176,-958922752,-959971328,-961019904,-962068480,-963117056,-964165632,-965214208,-966262784,-967311360,-968359936,-969408512,-970457088,-971505664,-972554240,
-973602816,-974651392,-975699968,-976748544,-977797120,-978845696,-979894272,-980942848,-981991424,-983040000,-984088576,-985137152,-986185728,-987234304,-988282880,-989331456,
-990380032,-991428608,-992477184,-993525760,-994574336,-995622912,-996671488,-997720064,-998768640,-999817216,-1000865792,-1001914368,-1002962944,-1004011520,-1005060096,-1006108672,
-1007157248,-1008205824,-1009254400,-1010302976,-1011351552,-1012400128,-1013448704,-1014497280,-1015545856,-1016594432,-1017643008,-1018691584,-1019740160,-1020788736,-1021837312,-1022885888,
-1023934464,-1024983040,-1026031616,-1027080192,-1028128768,-1029177344,-1030225920,-1031274496,-1032323072,-1033371648,-1034420224,-1035468800,-1036517376,-1037565952,-1038614528,-1039663104,
-1040711680,-1041760256,-1042808832,-1043857408,-1044905984,-1045954560,-1047003136,-1048051712,-1049100288,-1050148864,-1051197440,-1052246016,-1053294592,-1054343168,-1055391744,-1056440320,
-1057488896,-1058537472,-1059586048,-1060634624,-1061683200,-1062731776,-1063780352,-1064828928,-1065877504,-1066926080,-1067974656,-1069023232,-1070071808,-1071120384,-1072168960,-1073217536,
-1074266112,-1075314688,-1076363264,-1077411840,-1078460416,-1079508992,-1080557568,-1081606144,-1082654720,-1083703296,-1084751872,-1085800448,-1086849024,-1087897600,-1088946176,-1089994752,
-1091043328,-1092091904,-1093140480,-1094189056,-1095237632,-1096286208,-1097334784,-1098383360,-1099431936,-1100480512,-1101529088,-1102577664,-1103626240,-1104674816,-1105723392,-1106771968,
-1107820544,-1108869120,-1109917696,-1110966272,-1112014848,-1113063424,-1114112000,-1115160576,-1116209152,-1117257728,-1118306304,-1119354880,-1120403456,-1121452032,-1122500608,-1123549184,
-1124597760,-1125646336,-1126694912,-1127743488,-1128792064,-1129840640,-1130889216,-1131937792,-1132986368,-1134034944,-1135083520,-1136132096,-1137180672,-1138229248,-1139277824,-1140326400,
-1141374976,-1142423552,-1143472128,-1144520704,-1145569280,-1146617856,-1147666432,-1148715008,-1149763584,-1150812160,-1151860736,-1152909312,-1153957888,-1155006464,-1156055040,-1157103616,
-1158152192,-1159200768,-1160249344,-1161297920,-1162346496,-1163395072,-1164443648,-1165492224,-1166540800,-1167589376,-1168637952,-1169686528,-1170735104,-1171783680,-1172832256,-1173880832,
-1174929408,-1175977984,-1177026560,-1178075136,-1179123712,-1180172288,-1181220864,-1182269440,-1183318016,-1184366592,-1185415168,-1186463744,-1187512320,-1188560896,-1189609472,-1190658048,
-1191706624,-1192755200,-1193803776,-1194852352,-1195900928,-1196949504,-1197998080,-1199046656,-1200095232,-1201143808,-1202192384,-1203240960,-1204289536,-1205338112,-1206386688,-1207435264,
-1208483840,-1209532416,-1210580992,-1211629568,-1212678144,-1213726720,-1214775296,-1215823872,-1216872448,-1217921024,-1218969600,-1220018176,-1221066752,-1222115328,-1223163904,-1224212480,
-1225261056,-1226309632,-1227358208,-1228406784,-1229455360,-1230503936,-1231552512,-1232601088,-1233649664,-1234698240,-1235746816,-1236795392,-1237843968,-1238892544,-1239941120,-1240989696,
-1242038272,-1243086848,-1244135424,-1245184000,-1246232576,-1247281152,-1248329728,-1249378304,-1250426880,-1251475456,-1252524032,-1253572608,-1254621184,-1255669760,-1256718336,-1257766912,
-1258815488,-1259864064,-1260912640,-1261961216,-1263009792,-1264058368,-1265106944,-1266155520,-1267204096,-1268252672,-1269301248,-1270349824,-1271398400,-1272446976,-1273495552,-1274544128,
-1275592704,-1276641280,-1277689856,-1278738432,-1279787008,-1280835584,-1281884160,-1282932736,-1283981312,-1285029888,-1286078464,-1287127040,-1288175616,-1289224192,-1290272768,-1291321344,
-1292369920,-1293418496,-1294467072,-1295515648,-1296564224,-1297612800,-1298661376,-1299709952,-1300758528,-1301807104,-1302855680,-1303904256,-1304952832,-1306001408,-1307049984,-1308098560,
-1309147136,-1310195712,-1311244288,-1312292864,-1313341440,-1314390016,-1315438592,-1316487168,-1317535744,-1318584320,-1319632896,-1320681472,-1321730048,-1322778624,-1323827200,-1324875776,
-1325924352,-1326972928,-1328021504,-1329070080,-1330118656,-1331167232,-1332215808,-1333264384,-1334312960,-1335361536,-1336410112,-1337458688,-1338507264,-1339555840,-1340604416,-1341652992,
-1342701568,-1343750144,-1344798720,-1345847296,-1346895872,-1347944448,-1348993024,-1350041600,-1351090176,-1352138752,-1353187328,-1354235904,-1355284480,-1356333056,-1357381632,-1358430208,
-1359478784,-1360527360,-1361575936,-1362624512,-1363673088,-1364721664,-1365770240,-1366818816,-1367867392,-1368915968,-1369964544,-1371013120,-1372061696,-1373110272,-1374158848,-1375207424,
-1376256000,-1377304576,-1378353152,-1379401728,-1380450304,-1381498880,-1382547456,-1383596032,-1384644608,-1385693184,-1386741760,-1387790336,-1388838912,-1389887488,-1390936064,-1391984640,
-1393033216,-1394081792,-1395130368,-1396178944,-1397227520,-1398276096,-1399324672,-1400373248,-1401421824,-1402470400,-1403518976,-1404567552,-1405616128,-1406664704,-1407713280,-1408761856,
-1409810432,-1410859008,-1411907584,-1412956160,-1414004736,-1415053312,-1416101888,-1417150464,-1418199040,-1419247616,-1420296192,-1421344768,-1422393344,-1423441920,-1424490496,-1425539072,
-1426587648,-1427636224,-1428684800,-1429733376,-1430781952,-1431830528,-1432879104,-1433927680,-1434976256,-1436024832,-1437073408,-1438121984,-1439170560,-1440219136,-1441267712,-1442316288,
-1443364864,-1444413440,-1445462016,-1446510592,-1447559168,-1448607744,-1449656320,-1450704896,-1451753472,-1452802048,-1453850624,-1454899200,-1455947776,-1456996352,-1458044928,-1459093504,
-1460142080,-1461190656,-1462239232,-1463287808,-1464336384,-1465384960,-1466433536,-1467482112,-1468530688,-1469579264,-1470627840,-1471676416,-1472724992,-1473773568,-1474822144,-1475870720,
-1476919296,-1477967872,-1479016448,-1480065024,-1481113600,-1482162176,-1483210752,-1484259328,-1485307904,-1486356480,-1487405056,-1488453632,-1489502208,-1490550784,-1491599360,-1492647936,
-1493696512,-1494745088,-1495793664,-1496842240,-1497890816,-1498939392,-1499987968,-1501036544,-1502085120,-1503133696,-1504182272,-1505230848,-1506279424,-1507328000,-1508376576,-1509425152,
-1510473728,-1511522304,-1512570880,-1513619456,-1514668032,-1515716608,-1516765184,-1517813760,-1518862336,-1519910912,-1520959488,-1522008064,-1523056640,-1524105216,-1525153792,-1526202368,
-1527250944,-1528299520,-1529348096,-1530396672,-1531445248,-1532493824,-1533542400,-1534590976,-1535639552,-1536688128,-1537736704,-1538785280,-1539833856,-1540882432,-1541931008,-1542979584,
-1544028160,-1545076736,-1546125312,-1547173888,-1548222464,-1549271040,-1550319616,-1551368192,-1552416768,-1553465344,-1554513920,-1555562496,-1556611072,-1557659648,-1558708224,-1559756800,
-1560805376,-1561853952,-1562902528,-1563951104,-1564999680,-1566048256,-1567096832,-1568145408,-1569193984,-1570242560,-1571291136,-1572339712,-1573388288,-1574436864,-1575485440,-1576534016,
-1577582592,-1578631168,-1579679744,-1580728320,-1581776896,-1582825472,-1583874048,-1584922624,-1585971200,-1587019776,-1588068352,-1589116928,-1590165504,-1591214080,-1592262656,-1593311232,
-1594359808,-1595408384,-1596456960,-1597505536,-1598554112,-1599602688,-1600651264,-1601699840,-1602748416,-1603796992,-1604845568,-1605894144,-1606942720,-1607991296,-1609039872,-1610088448,
-1611137024,-1612185600,-1613234176,-1614282752,-1615331328,-1616379904,-1617428480,-1618477056,-1619525632,-1620574208,-1621622784,-1622671360,-1623719936,-1624768512,-1625817088,-1626865664,
-1627914240,-1628962816,-1630011392,-1631059968,-1632108544,-1633157120,-1634205696,-1635254272,-1636302848,-1637351424,-1638400000,-1639448576,-1640497152,-1641545728,-1642594304,-1643642880,
-1644691456,-1645740032,-1646788608,-1647837184,-1648885760,-1649934336,-1650982912,-1652031488,-1653080064,-1654128640,-1655177216,-1656225792,-1657274368,-1658322944,-1659371520,-1660420096,
-1661468672,-1662517248,-1663565824,-1664614400,-1665662976,-1666711552,-1667760128,-1668808704,-1669857280,-1670905856,-1671954432,-1673003008,-1674051584,-1675100160,-1676148736,-1677197312,
-1678245888,-1679294464,-1680343040,-1681391616,-1682440192,-1683488768,-1684537344,-1685585920,-1686634496,-1687683072,-1688731648,-1689780224,-1690828800,-1691877376,-1692925952,-1693974528,
-1695023104,-1696071680,-1697120256,-1698168832,-1699217408,-1700265984,-1701314560,-1702363136,-1703411712,-1704460288,-1705508864,-1706557440,-1707606016,-1708654592,-1709703168,-1710751744,
-1711800320,-1712848896,-1713897472,-1714946048,-1715994624,-1717043200,-1718091776,-1719140352,-1720188928,-1721237504,-1722286080,-1723334656,-1724383232,-1725431808,-1726480384,-1727528960,
-1728577536,-1729626112,-1730674688,-1731723264,-1732771840,-1733820416,-1734868992,-1735917568,-1736966144,-1738014720,-1739063296,-1740111872,-1741160448,-1742209024,-1743257600,-1744306176,
-1745354752,-1746403328,-1747451904,-1748500480,-1749549056,-1750597632,-1751646208,-1752694784,-1753743360,-1754791936,-1755840512,-1756889088,-1757937664,-1758986240,-1760034816,-1761083392,
-1762131968,-1763180544,-1764229120,-1765277696,-1766326272,-1767374848,-1768423424,-1769472000,-1770520576,-1771569152,-1772617728,-1773666304,-1774714880,-1775763456,-1776812032,-1777860608,
-1778909184,-1779957760,-1781006336,-1782054912,-1783103488,-1784152064,-1785200640,-1786249216,-1787297792,-1788346368,-1789394944,-1790443520,-1791492096,-1792540672,-1793589248,-1794637824,
-1795686400,-1796734976,-1797783552,-1798832128,-1799880704,-1800929280,-1801977856,-1803026432,-1804075008,-1805123584,-1806172160,-1807220736,-1808269312,-1809317888,-1810366464,-1811415040,
-1812463616,-1813512192,-1814560768,-1815609344,-1816657920,-1817706496,-1818755072,-1819803648,-1820852224,-1821900800,-1822949376,-1823997952,-1825046528,-1826095104,-1827143680,-1828192256,
-1829240832,-1830289408,-1831337984,-1832386560,-1833435136,-1834483712,-1835532288,-1836580864,-1837629440,-1838678016,-1839726592,-1840775168,-1841823744,-1842872320,-1843920896,-1844969472,
-1846018048,-1847066624,-1848115200,-1849163776,-1850212352,-1851260928,-1852309504,-1853358080,-1854406656,-1855455232,-1856503808,-1857552384,-1858600960,-1859649536,-1860698112,-1861746688,
-1862795264,-1863843840,-1864892416,-1865940992,-1866989568,-1868038144,-1869086720,-1870135296,-1871183872,-1872232448,-1873281024,-1874329600,-1875378176,-1876426752,-1877475328,-1878523904,
-1879572480,-1880621056,-1881669632,-1882718208,-1883766784,-1884815360,-1885863936,-1886912512,-1887961088,-1889009664,-1890058240,-1891106816,-1892155392,-1893203968,-1894252544,-1895301120,
-1896349696,-1897398272,-1898446848,-1899495424,-1900544000,-1901592576,-1902641152,-1903689728,-1904738304,-1905786880,-1906835456,-1907884032,-1908932608,-1909981184,-1911029760,-1912078336,
-1913126912,-1914175488,-1915224064,-1916272640,-1917321216,-1918369792,-1919418368,-1920466944,-1921515520,-1922564096,-1923612672,-1924661248,-1925709824,-1926758400,-1927806976,-1928855552,
-1929904128,-1930952704,-1932001280,-1933049856,-1934098432,-1935147008,-1936195584,-1937244160,-1938292736,-1939341312,-1940389888,-1941438464,-1942487040,-1943535616,-1944584192,-1945632768,
-1946681344,-1947729920,-1948778496,-1949827072,-1950875648,-1951924224,-1952972800,-1954021376,-1955069952,-1956118528,-1957167104,-1958215680,-1959264256,-1960312832,-1961361408,-1962409984,
-1963458560,-1964507136,-1965555712,-1966604288,-1967652864,-1968701440,-1969750016,-1970798592,-1971847168,-1972895744,-1973944320,-1974992896,-1976041472,-1977090048,-1978138624,-1979187200,
-1980235776,-1981284352,-1982332928,-1983381504,-1984430080,-1985478656,-1986527232,-1987575808,-1988624384,-1989672960,-1990721536,-1991770112,-1992818688,-1993867264,-1994915840,-1995964416,
-1997012992,-1998061568,-1999110144,-2000158720,-2001207296,-2002255872,-2003304448,-2004353024,-2005401600,-2006450176,-2007498752,-2008547328,-2009595904,-2010644480,-2011693056,-2012741632,
-2013790208,-2014838784,-2015887360,-2016935936,-2017984512,-2019033088,-2020081664,-2021130240,-2022178816,-2023227392,-2024275968,-2025324544,-2026373120,-2027421696,-2028470272,-2029518848,
-2030567424,-2031616000,-2032664576,-2033713152,-2034761728,-2035810304,-2036858880,-2037907456,-2038956032,-2040004608,-2041053184,-2042101760,-2043150336,-2044198912,-2045247488,-2046296064,
-2047344640,-2048393216,-2049441792,-2050490368,-2051538944,-2052587520,-2053636096,-2054684672,-2055733248,-2056781824,-2057830400,-2058878976,-2059927552,-2060976128,-2062024704,-2063073280,
-2064121856,-2065170432,-2066219008,-2067267584,-2068316160,-2069364736,-2070413312,-2071461888,-2072510464,-2073559040,-2074607616,-2075656192,-2076704768,-2077753344,-2078801920,-2079850496,
-2080899072,-2081947648,-2082996224,-2084044800,-2085093376,-2086141952,-2087190528,-2088239104,-2089287680,-2090336256,-2091384832,-2092433408,-2093481984,-2094530560,-2095579136,-2096627712,
-2097676288,-2098724864,-2099773440,-2100822016,-2101870592,-2102919168,-2103967744,-2105016320,-2106064896,-2107113472,-2108162048,-2109210624,-2110259200,-2111307776,-2112356352,-2113404928,
-2114453504,-2115502080,-2116550656,-2117599232,-2118647808,-2119696384,-2120744960,-2121793536,-2122842112,-2123890688,-2124939264,-2125987840,-2127036416,-2128084992,-2129133568,-2130182144,
-2131230720,-2132279296,-2133327872,-2134376448,-2135425024,-2136473600,-2137522176,-2138570752,-2139619328,-2140667904,-2141716480,-2142765056,-2143813632,-2144862208,-2145910784,-2146959360
    };

    const fixed WhiteWave[N_WAVE] = {
977138961,298164032,-99546916,1956562030,844234724,1014768408,-1486338227,726975776,1775670258,1185817714,1980327372,-84283377,7597864,909681492,-513345240,-633791163,
656715234,699850289,-181605720,-520972399,-593746947,1160869567,1833242123,235956454,-796226680,-690543418,308990192,-796783559,1077147862,-394251613,-341804240,1551127904,
168783077,-2027538989,-433387027,-212032531,743023750,316265813,68583127,1780774153,-1547949090,-1621055118,-960942606,1152527641,1959042671,-1264081317,1131378755,-850310450,
1392912621,719977593,165717357,1461526666,-183206074,-1204244787,-859460728,915338578,433869943,652167762,1498042377,-222477022,-937675029,1526013910,1537016253,2136096022,
-1402802110,2114909632,-1816770355,1756538270,-155659095,-948616762,-585513282,1659125952,-178489143,1407084372,-822014279,-1645427055,-1640808780,-1820968349,-1633810364,1730745619,
-819449803,-471550611,-320417383,2093478945,-1300746417,-360087951,-777688638,-1782465589,1260315050,-2109805990,-1662274667,-157273660,-2015341310,-630548704,-668147746,-1314496453,
1114949561,-603187902,-1453611535,172240758,-172046699,1580872556,-139930186,-2055187108,-1466027278,388210205,1488234642,-228852807,-1240934457,-2043324737,-1500861984,-849571448,
1922232651,-1219725085,1436519814,-2057453940,-1878480031,1599037889,-1221005726,-85757449,-628235918,10117630,442998684,-1552721603,-1126824998,-1602118718,-1859287990,-1939013720,
1564087402,1815918806,587302138,-1065147421,-218538726,-1108539483,-1299198105,498618975,1309270371,-1549233564,1001411163,1518562358,1130636331,-1722316773,-1686072230,1078480855,
2067557236,-1282011855,1894196841,1680034930,-478563388,875500793,423200070,1801995061,-1937575394,1252147420,1660317400,1526909594,-48729664,-602615899,531331472,1070735484,
466940672,-1806000358,1118330349,1649462178,13660379,-362000067,100722948,-1673911258,543091972,-2114884316,-1170074406,338198014,-2028235483,-1316017760,-1705266616,-908857410,
-2082548737,1981087199,1380637778,524416186,-597310334,682993695,-226631264,-1709772976,-785597130,1739151713,221541101,1505828258,452992606,-1271467853,1300013557,-1140190069,
472324526,-655952212,-1849007292,425659380,-1674142961,1465683205,1000157556,-1064229758,-1122912419,1766150163,-652447691,202955037,-1320979110,-228395374,-431648916,2129650427,
354671958,1871056676,-1256260616,654556515,-1981380734,-878450003,-1928205616,130437215,1110482582,-1895182454,-1037525887,1929099951,-1407906717,2115484758,1402737951,-1988570853,
-1919916361,-190369965,-1111037892,717611983,-1113873980,-299828486,1391372078,342988329,1342592986,1949816222,456637155,649700157,561073221,545146035,-253219693,1417493205,
-2008030348,-204577007,-1880530868,-548720553,586032111,418452715,-655337834,2143909975,770096863,-2140734334,-1427629715,-1280161062,-346141268,1445715997,2034558408,1982466200,
-97470301,632085800,392916596,-1741140801,-1847513769,-1403960053,-1570015859,-1516913359,1916527083,1326883200,540559458,914826159,-417929357,452408895,-1384937476,42657391,
-894065311,2114187816,1857907919,2009044231,1092254580,-2119101686,-1139361940,-715479270,145606104,1969134043,1433563379,-1435235089,-1843442934,201400764,1930229125,-273287113,
-439163746,735242587,-1534770481,-1131940666,317378625,-1288640071,1916998566,336533785,1442833116,-1986476725,297850648,1120840512,-1714786023,706021729,1349242307,518914939,
314621929,914443021,349735444,-1598997061,-1289138768,599070069,-914208197,-700532825,150362102,-1185561186,-332242593,1221570647,485699853,-867396586,701609143,-752723525,
721614964,1087971247,1137747221,1043420231,1456731917,-527577869,-98675308,1650350683,1242538840,-1119910003,290552670,1442605436,147542625,-2063395526,1536098990,-1437231306,
-1101522357,-1733495441,1385565383,-5147107,817508059,1986558622,-1013330664,2036627258,-442458562,-1327524792,767722379,-1944734488,-1326190917,1190752503,-1207094485,1420077331,
1888896702,1098533845,926544299,1365847353,-1753289751,-1988961213,-1762183210,-1858513586,-1346091829,1119100560,480484414,-269293351,1910413822,-1067727467,-2071992907,608534939,
-450268094,1644366520,14559908,1963244528,-2100551673,449170804,-2025410487,-1970602593,607182191,-317552478,1968151620,-390852273,-646876620,-1796560429,1401250986,392515064,
165536723,-33855639,80061426,1021098878,-1127145289,1018858724,-846330830,279974890,60570093,-690740587,-1666431384,2145382725,735867934,1869894256,-471959240,-1233031644,
980717327,-372046733,1925559979,244267907,-2012891586,1313097364,571876512,1693485001,989463511,1530987822,-1990166761,559943082,-1399904492,226237490,-1953329047,1407774539,
1759923255,920135621,-786428749,-403693291,1461726059,-877788584,481714940,706452731,1601165709,1184045854,1293398779,487852867,-1306195154,-438863469,386536150,1999083598,
-1088305409,984728960,-1849759460,-683208617,1028485830,1807190528,-1474183762,2125346889,-343312360,2092065975,1793233737,1226081487,507900923,2106067916,-1520157138,1224942160,
-673117001,170998159,-1999354878,-1486657309,1940189429,-52806349,1669290231,918049751,12658561,-2035182259,-196420616,1335086995,-1744610228,816813133,-1803261449,1597517394,
245568873,-390692967,-512112354,1918656118,-1639698021,1935735584,-83770803,511280217,-1962995499,-1407556334,1477091984,1301492240,647666978,-703594328,-1261892237,2115146915,
1541490367,355107065,-1034589717,1904913031,-410588413,148109268,2123627089,1820992573,667921653,1360329782,1573131799,-1321812016,576888641,1369182866,-2126265454,547376816,
1748547754,828064618,824849517,921888636,-1606625045,286508539,647387613,769021575,-1788318104,-856051999,-1446955952,-1746620206,2130049823,458030546,-940934349,-2054470491,
-1860935884,-418636302,-1413364591,-411670941,-1127447773,785473220,-1557710070,-1192161823,869545776,-1005328583,721852690,-1778879696,926344664,-1153022733,1616530574,653352498,
-1100500471,1121919644,2022574494,1840441165,-2047326379,52574200,-1216485256,-1254445837,-1625809401,343925765,-1265326374,-1486407938,2093324978,531744576,1785544078,-206933244,
1165039836,-570795460,-2139159895,177664008,79828087,-465272942,-216803012,-1268490213,201239569,-2050375376,-733013734,-615685120,-481516706,-2115725630,702259152,495197302,
814085505,2056881923,-2144929518,-1971484960,-1932873745,-837879643,830537200,-2038199242,-1614532409,-1544728511,1067409985,1851610383,38856337,-611946940,1588378140,1173786375,
-2130764740,-1800056789,-179783441,808697068,325124454,-483602607,-362177268,-52847803,-1088178217,-598883074,-1136760297,-848350762,-737090968,-1628870098,707703045,-184637456,
-1045933190,112414609,2141941869,2011954263,2138725958,236183687,-1193732551,1330501073,897537137,1380672672,-1027981551,-1088196864,-413488367,1532505584,99743709,-1736949835,
-1277728224,-1277908427,636893315,-343824614,-153829205,-136452745,245703374,-1231111254,-1232734965,1284679509,409583421,1459816049,1036896313,-1325466123,360849876,-498459579,
754193476,-1397333756,-963320789,1451497567,-998477653,-1982868086,-1985957222,-1195843830,-1764447279,1465452943,-1639899069,-1861135493,-1689465596,171296630,1524681192,161549115,
313143611,-80918720,-2006514954,-957762602,1089885210,-1953156211,-1800751313,-978935917,-1203370927,-912165378,-1308726805,-1798558152,-803029278,748583133,-1440613260,-1435370390,
-1240753804,-231939410,-180034585,1885586363,-1687910263,-1162904276,76522257,52643249,-1750072127,-640309226,-446364151,-646121088,-528099090,1025306867,1327601259,1150247344,
324908023,-2056092877,-1425079121,1551838359,1840860315,-400397943,18431987,-365209511,1847477530,1118337384,1065697041,528221112,-1071592295,503805916,631166186,1245376119,
-1243926500,-150889343,-131526239,-481146328,-369551177,213409340,1297134827,1990955603,2107654601,488665682,1990287185,-653287431,561456170,1853502479,-81347631,106942921,
-1257040247,-622328750,-1470739174,1635030994,4630450,-1272807777,-1182421330,-1378388761,1659913582,1153530733,189454481,-1043010920,-59667123,-1381752720,142978434,-145076538,
1665852074,-1380932923,554196829,308557030,368543209,-1629671925,-1204048836,-1037057961,862062650,472927782,292004784,-608186288,1018575210,-1768672134,-1089429728,22716510,
1763814019,-2089241355,345512414,1648603277,-1895104327,200260934,-1100965099,-2008968746,-847548685,724148994,497953651,1771451234,-171821953,-1220831440,-1453758242,-1529016758,
2080109904,754506460,476981433,1466805448,1453032554,-1864503448,-139283977,1278648184,-1002876131,302089008,705810375,-1550667444,1554888220,72994324,-1350079363,1855304594,
-15458926,1525474691,-1215794243,1895179733,1454834181,-1323870944,-1505353771,1738717921,2111685099,1812813454,-1812225926,189982291,953360650,-103148989,-402392518,371519288,
-635850444,-419733836,2073410921,-1587985729,1037017722,-1600557498,546861168,-409182961,-1455693150,237150839,1478532087,-1666936692,-601053437,-1149403982,-132331292,-705626796,
-442800482,-301925508,-1019570741,-1979765092,270614276,-1038747844,-1153778335,1194222654,1852237995,261968884,-2051770812,-711238077,-187935570,980817527,-2036407597,-493746735,
1688409046,310545032,980907512,-69536487,-363633823,862747944,90792165,-1415225309,-1844581026,-1696602542,1012000623,-619697244,-1997745685,1519760521,-788158003,-1081007459,
-1824162020,80566760,-582041612,1962634495,1022490475,861868349,-780213475,199386683,1180482510,1589958666,1101345009,-1042980462,2046531824,1942774251,885855877,-112382478,
-407934377,1338809669,-1150902126,-1549622778,1428656192,269385257,-174920635,-1767220024,44891262,691674545,608743767,916304926,-966749045,1062133368,2045224352,849344584,
1616831491,-354807813,361705627,1081213423,1349408513,1971634492,1832328745,632633807,782805067,1610794016,515395493,-680998054,326328566,480954320,-2135399761,-1804085766,
1733190101,485590481,-1666088226,1960235830,-1554821871,-262441323,1741068079,-1833509250,2071747991,-1184452353,-1740041241,1386303306,954922428,1383293421,-901294744,963614519,
456616995,1931385384,1011798581,-1022773348,-1971816305,-266084892,-245982248,-1076663631,111206222,-1116237976,930691642,-2137681317,1026332229,565496280,-1920272159,1707193750,
-1980411713,2143156011,22201461,1324333966,2084908301,1336989183,1406842168,-765752602,-432613903,-580849136,2127342325,-466208005,1125381941,1586226630,1399578623,-1273672946,
-522734117,-1285969336,-8106750,-668214970,-1985906829,1425869748,451916948,204988451,-184750260,-680836234,47227256,-1946389550,225433782,1104914951,-1033403144,861264300,
-2078567245,-503428301,233690312,-2144765340,-422743948,1811993343,835189588,-921972212,415247911,-716426461,-146079713,-1513048224,-439323124,-676842898,-1158868948,-1241988526,
1347421813,637685678,-536981876,-1832100957,1079358660,553764155,-933231860,1313214486,1674194944,-2082914561,1482205083,604214389,1653232769,-1618082626,-544677381,1065086717,
1541023084,1129574923,2057999355,-1697491937,1330956955,54113745,-986380501,443515549,1831432480,-1346729891,-1241611956,1203294682,678066451,1321761356,76563418,-46926527,
1934771364,-2035728884,1748590209,367790347,492038239,-409105910,-1685236234,1629673178,-2131148089,-707680918,-759759875,1819677519,770817448,1465233854,2082809254,-2055524141,
-911028376,-1410550353,1476373349,-1200973226,494996853,-1918858783,-862634277,-490631685,1445203619,-934973065,-392753477,-120376894,2031377088,112040013,1976759543,647474930,
1058932913,-872725193,-1804350817,209749399,403041975,986147016,390455190,-1132472193,-1883387251,1349687198,716292021,-1492620320,1074522371,-1600618142,-1721546939,-1869286142,
-1486686550,-1809490204,-1436206885,-1934074417,245249378,86155469,2004724561,1592683407,1420332462,-413184254,-33380010,-76456587,-1863960400,1999278272,178458651,611564885,
-1102048844,-1583674770,-1296256054,1094257189,-906517987,-1537360049,-1169789116,-366576774,-2094852574,1420097272,-1285445708,-793039215,-1378769777,696366298,-1925082022,733607942,
-2089271805,1804649045,-831355608,-1754035736,-1547213972,763083991,136533602,759231967,-1014463092,914211848,651446462,142827411,-1894683554,1965123044,453180535,1759556535,
1528948574,-1042115833,273160536,-1033079238,50230350,939535159,364348740,369727676,829965812,-83848698,-312317748,-646354022,-954419741,-1538471554,-1042019634,1227737561,
-516745026,-302747153,321244616,-1098572176,421904330,671966854,331875517,-747255154,1532192060,1515413620,503768203,118810576,-938091119,-1432262800,926529326,-1316816562,
-1551477366,-396908009,-149672134,1751949224,-1652074154,1943071556,1253624560,447286820,719349537,1042760238,802763917,1946788418,-1024884238,-2021555971,-1517037606,-7765518,
2021842120,-438836911,-2008820577,-640605155,1356946703,1202842982,618425796,1089707039,-2144561794,-2135825389,1642464466,-1683687958,-507972112,-1741745034,1972295960,2020938681,
-1706303001,1845783038,1438656578,1139410814,1734679936,428147080,477019000,-1172301496,-1539711447,245158312,-1410785611,1228323983,2000670064,-621457806,278819013,19135050,
-2070780962,-792030698,1110939345,74607632,1345343350,1087270907,1171900972,1012411540,719590678,1253041691,-724742802,669805805,1011304911,-372480602,-1235906561,1329491715,
-158377808,-1227413530,-1211123085,-536174341,-215241472,-415696199,365978777,1824344490,-1693781982,519939177,1308467425,299686541,1532478703,596786249,-413313192,-1048405581,
1765633768,1749341138,-29313552,-2052267598,-1741009387,-665561131,1889092421,1796994283,405468257,1611327602,-1142289867,862506423,1234374099,-1384564797,-328903130,1040477307,
1988533778,1206007891,1972910316,1798754954,-371326130,1917237441,1517269141,-1639476274,-369851154,-1992204543,402404179,-1492862872,-13742018,-1922136924,-1308900909,2009159294,
505871543,1250740973,1692679384,194638484,-1639162919,2121240640,724567050,-864844682,-423400233,215044653,-1507031318,-743593206,-1444342875,1315794001,1815344946,-39863616,
-656368453,1248723875,-979868907,467470139,452989327,-813169878,1786190106,1533072781,773245346,-141813133,-1837945621,1236111068,-331836598,-267581241,-2063366140,-981920739,
1628488673,-491795262,548861182,-443187606,551919719,-1619201927,1518585650,718199338,99654880,-501926494,-1295399855,635149043,339011897,-1542366888,1008475175,984350189,
1734049696,-9335138,-1332109756,575875192,-1342462336,-1113161178,20911161,1466196564,-1330681519,-769994634,1716490305,968480546,-1325001044,-733367270,-2126048646,-730569355,
-2134667304,-284273819,599050768,-1614498035,-602191696,-1395986736,843533503,1738374522,-1893965472,1316182365,-697503088,1750645289,1019188945,1153282225,1778185737,819839606,
793976061,1295718416,805657497,-1194339644,1954014323,-194020005,-1620983729,-2066729778,1291001673,-1590457346,494456315,1207994114,-1415698998,495709140,-1449566368,815438745,
1138549104,-334912907,-1562096217,-523235277,162272581,-1508490144,-2068562460,966967590,-1041355854,-502727729,-203519734,305753160,1105973795,1126530437,803717974,-1223255552,
-1140417221,-2146820386,2099234588,1532293201,-566782148,1742604069,1307261861,1724651377,1352090008,797851044,-902144233,1924825883,526436748,-995849635,2031417560,919860032,
1563421911,-779656535,1484986180,-1013850851,27163871,1932395349,682720058,-1834985191,1449722280,-1034522745,79332596,-1343690119,-997604724,-2136040430,682774710,753711080,
-544380370,867203021,-636981646,-1691729,-766763246,-148671606,-245970846,254689664,309811246,189246264,-717043026,-1741593759,2069729525,1955207022,-946358611,-1793764840,
575669803,1227526177,310627833,1627415905,1131086288,1038386012,-1534010744,1422761311,88900544,-1039045214,-498585471,2011736288,781907941,880304425,1704117762,-1676877185,
-675896668,-1332914127,-2122224016,-90070989,1975048154,1516371267,682737462,1292310042,2025972101,-1580577196,-1387286878,-1333465169,-1944035306,992439701,1184218911,1698296099,
387505085,1870629132,820864105,1052970787,2038468154,-1781799781,1016080005,-1967794900,715466149,-1455716754,-1008864187,-1009751366,1477406135,1586754433,1062039390,1014343962,
-1366070782,-1215676747,1890289099,-1807784716,25908690,1230113927,-1948401810,1843020086,-1348203960,-1732656724,-1381939075,-501241242,-662657303,1096804509,1345709875,1125874745,
1432981310,-654223336,-1805785893,-860793429,-1653067771,-1287495885,1584743837,-1417211425,1218260365,1953287654,525532437,-499452313,-1158698958,-846826768,551117953,2010748784,
-300734057,190846150,761780512,-1902705270,-403302650,1186191581,-213727914,-2104858210,920087578,-1608705272,2021097136,-1715077624,1058231548,-2073123669,1849132002,114769399,
-776403829,495960702,-1546858191,958996523,1681202320,-579906037,-2045026564,-320329247,1534665807,1312636856,231444941,1491077592,791511058,1938058146,-863056808,33530618,
538964905,-68372162,-628543668,-103554802,-1125696270,-600392385,-1675259266,1387987187,578572669,-1588580090,1761785135,1763840974,-1859692482,2006694219,-379286489,500313809,
1945645249,1965749051,-2063257191,985870522,1333452434,717369832,-1238624783,-55888518,-1840971051,898430926,61396411,-1806028401,-483225032,1485464231,534018617,-1848029409,
-1449833371,-300946314,-1006747715,1377908776,1244800531,-815320853,-1667676406,-611610082,1811153878,635913208,1129169150,1219309599,1691706017,-1850866400,1176592753,1337007325,
1012650923,26489357,-1969283177,-1174512667,-876930985,344298459,727131250,187291695,-1226776180,-1513732692,-105062557,691044326,-1341801365,-551264651,386299040,1809770402,
-1425873547,-1843608362,-648200837,223118616,685593518,1248340024,1876220411,290329611,1547263543,1446664717,1608848966,-1897965578,-81628128,1935864823,708029071,-1214215992,
883029760,1390019210,-2135129956,1403479312,884828740,1749846299,1481768056,-568590760,243094479,809041136,-1418763281,1128740894,341619927,1422192644,1566782588,-439504431,
930146840,-2032569892,-674443648,385544351,-885691344,-1348556573,-650067949,-702855223,-491864071,1822010965,413341604,408575,1987647723,-559413809,265878042,-413542172,
-96230348,-1433419732,1756020185,17741588,-889987373,-948378245,273650340,1048197771,-2119675836,-1772398444,1654908222,-1472368777,698215264,-1957422451,-1308140151,-2134969798,
325640588,414034082,-1407556646,-779785802,-96343726,-800246513,1544044518,1280863277,-311852185,-301286561,-320646778,653575583,-2081595478,231416845,-270583514,-1561646533,
-616707311,-255821290,99511409,830566516,-951714864,-1349571947,946972171,819837491,-384402478,-1246153781,1137076201,1691531749,1662031500,31626985,-11858735,1970728422,
1151942864,148538009,-1222551503,-2017240122,765566989,1543860341,1532329061,1718895915,-481125196,-1730135126,-673443298,1089408398,668784452,-812941750,-677419721,2090140182,
1710105700,1645979770,1579199790,-636499215,1241108731,-1100127095,297706183,680935475,1006318506,-339516480,1100362952,1399754838,-1106951008,163933571,-267704150,-1308501932,
-255106705,-1814074468,1404355761,-704292438,-746501756,2105714442,-290947648,1075322065,1841327099,532864943,1401726512,1697916636,-328590019,1091528328,-1889758586,1639475757,
-615000865,1756503286,952321231,-791085772,306764941,1963740016,-1954293072,-1716648771,259666648,1230102568,2041703396,-597889198,1481029497,-1218440742,-1316875608,-1254015231,
-1913186357,1936177651,-389241311,-2065940191,639637786,-2089646872,375233262,-318472288,-1189994220,936604156,493746604,-1668813775,655881849,602023400,1295832273,-907235724,
-214195480,1257180819,67990154,-2018721864,-1684240529,1526098832,-171468728,391647150,-2084854500,2040748779,1694756555,-1668701123,-699208559,1448329306,974972079,-252337851,
-1529014461,-1671998307,1140238941,1435385726,758908389,1959460744,-255712907,1172427551,-421773885,1453820517,1992816032,2089660327,-853795053,1502396327,1311465115,1788377533,
-1929889649,-324074450,97495282,-1966447793,739109992,-1161970585,1968312617,1664238374,426167809,-1113489775,-1502265458,-1117194619,-98270575,973049060,257690903,-2099389519,
-1002841776,-866852704,-334457175,1858535113,-819565839,-344182485,-2068830348,423672535,-1834829773,-127067893,-1205838963,1235132771,1323433654,1820963618,-1660956143,1312309100,
524768069,610573530,898924149,1802572953,-1789737890,1663028272,-831191594,-1999647089,-305756247,-1398665977,-1293945072,-1726059639,184186271,-618079888,-591816421,202912325,
2139591782,-1282442858,-1034408129,-487263477,1114660538,-1771129853,-1608993634,-1289479396,-1773784620,1707483869,-1261939502,382071140,1805005466,1153784847,-825512390,452677508,
-1327672681,-1479405154,1015905027,-453320535,-366918287,-1341877290,-802905191,112338622,1419709627,-1920459799,1882595284,-1707237341,-152660194,-1589274803,678961843,14201118,
606999359,-1607907840,1855084458,619220279,1674826772,-2076038766,-1678724923,-1304021899,-2069985748,481020853,-1131764808,1544030514,-1616143078,-280176583,-332514851,1421127158,
-1901866901,962072200,1848791593,-1538727723,-1799219581,412785836,-364478465,-1476317188,-442770645,-1701784240,1901538267,418392335,-1242716214,479343296,-958493975,1113672746,
217741508,1896833881,2118740946,-1019547643,-914076959,433920478,388430547,-845249483,678550110,1526946460,1800177046,-1859990616,-776335882,1165807504,313071035,-1828164093,
96038579,1595239959,-312460728,-864375781,144897686,-660965059,1106780267,1936681978,-1210967004,-1553673858,1096642699,-1358714218,310449858,-478460711,-1365026508,325666680,
-46896441,124058966,1795053847,-1116715686,1321012677,1999895547,-2046531219,-921507552,-1395118399,-796436206,-256945260,-151277028,-117549605,-1047163004,785958723,-1898482357,
-124040922,-1610570524,1802978251,-1059983356,216274349,660886176,720663429,-1000764928,-1004401359,1231193025,371138786,-1902541061,2080064443,-1163167199,-1571008016,1931986573,
355685523,-2045473160,-1552952993,1653790692,341921797,2118541822,-1008322062,1089316308,-1918158563,297361437,-1660118721,-654421304,1186499047,298902632,-1024836478,-22158934,
521090371,-1443037488,227304750,1860928535,-2022676415,-1985883269,-1881777936,1114881041,439012527,1154883669,2081440730,323430437,1873731234,-109542775,1776447027,-1176173483,
-503549505,176094023,1724818714,-293250348,1695619165,767922136,-46011826,-86329166,1392410765,-758225879,1335500305,1189622085,587651402,1341251121,-1951000334,-1470398447,
650022411,-1261632972,606872890,-1719173289,642155308,-1693640140,-988513273,-1858075813,-758068265,-783231587,1062178013,-191305862,28713541,683499893,-1540856010,-593040820,
1674298710,-2001988136,-599951848,-153014089,928064027,-1392145111,-95934053,1466201065,-1635715086,1428875744,11832661,6215028,-903294757,1027680413,-1636992726,906805430,
2056615082,-21531261,-203809714,-1448862166,-691278156,1631042818,2131235885,170835545,-2114775002,-1594914662,1759691592,302623763,-1764684633,1178714880,-604645443,437789709,
-1830022233,1659427547,-1707287794,1412158108,-1021879554,-1875288330,523180256,615580483,-1780309382,-1214611814,-416461544,1380973480,-1085698223,821725196,-1791649875,-145417043,
2042626922,2099632893,1577362376,1761931130,-461763754,1983468397,-1278523663,1219432109,632263803,56091344,-886263452,-1227656662,-1524173399,-1727087930,1028017404,-1009619047,
1280518851,33412478,882328912,-2080306736,-653167410,-1197048870,557553335,-455549740,454565837,1741196416,157041569,1200758680,-1833091374,312644196,1715295601,262146804,
550031506,-1712859136,-1259020999,623411967,-1398293842,-1696694856,1078094947,1955416988,1447878044,-694423092,311545683,-762424493,-427329593,448205962,-1958223126,1228289630,
-2125449174,-1969745694,-857017779,-1425403634,1727430449,-1857952047,-1963196170,-1631462775,886789923,-1153782453,1012536796,-1512235641,-170547589,55582955,705232316,-341724824,
-763420165,-2115520640,-1060797013,-1636234109,801553977,2021325166,299423544,1018396459,-678752839,1922621326,-1720622932,668417445,-793370369,1299802598,1511129504,327163854,
2108296812,1283682992,1803574206,-1021962170,-1009723578,779061372,-1435748998,-362426744,1849944840,955052604,1056357364,-1459150536,306139524,-1196392685,-1601296198,-438225598,
614993005,1740764058,1507055388,-1100588845,-1921714141,1980405157,1375763086,-1602763317,195158884,-123988399,1725912215,99884383,-1174170006,1455992790,-127911086,627495884,
1000675409,-1087817281,1327281594,2031896053,-1710540994,-1174426962,513464806,-462087699,-1922772771,-2087609507,-5901532,278246784,-1989514532,138270135,-286420384,-487738643,
-583840655,-437633467,-1586096100,-1500999572,-1825090552,2058504391,-122626842,-1058739557,1687800145,716920234,-510377066,-194666043,327792727,1727371751,-1155925406,-1132104798,
-1962614308,-132322594,-354073979,-605883686,2095544523,-105154206,-46673232,-253056447,-1780264389,68046376,1343607815,744028562,-431910514,1572960848,2000610674,1213566741,
-1921466207,-668769044,-839822826,419380867,-406414329,-422963466,841450601,-806122244,1111454571,2010559346,700108278,-711919338,987726386,-1955059687,-312294478,48523214,
1259732511,285369618,-2007795363,-1441812823,-1343237367,-147874418,1200559053,-738134502,-1274165976,-727278543,-1147034583,-1204794710,7650062,-513074031,-363694996,485296282,
958898212,753959363,1614553703,968255183,538033150,-1768294233,-846509711,1972458828,673834279,1852399037,1209344240,-1284129041,1914011182,-1449723082,738274754,-1661833718,
-1205525227,-1297269028,1331617404,-835841271,-820329357,1171479614,-1618596383,-1926389969,-1803450968,1024761393,1361607810,-701817900,-2020581829,-41115547,-730551133,-273350690,
-929971016,-1916149405,1458801273,-317466089,-1342313602,1052135794,-2033081165,-545996768,618208356,1472560960,-1131472436,-1242806921,-1756142259,1284281337,-1112251775,-162509262,
-1787523555,1030907025,-1089870090,-1619230752,1808045440,1635559654,-1188707143,-1941811757,224181256,98998091,223988694,356741454,1844828823,2088385245,-1727246674,326264711,
4182271,-281235751,1863852965,-718971038,-499875639,-557105608,178654933,-664023172,588883803,-1459156533,-1660431218,1008769489,-1729029467,497191236,423938221,1265630733,
-1863133727,-534079123,957647550,-44656321,1426588444,1832661274,1789514180,-128665685,-250636757,1593751886,1800738463,-681159606,1343261762,-620977772,194119192,-694003317,
267520026,-1638868311,122973513,-1008041376,-1070953222,-1293409635,-1403173778,808462856,1403049775,1516528449,1080867307,-1604816830,-1050153048,1953628571,-1556592938,1657158086,
1560627504,689872266,-1089726569,-1568314748,1603312547,6352873,233239887,-669435762,-2094217692,1793806078,806331770,943886035,-160317320,-503342958,-293896092,1471716633,
1647861909,403470921,991448327,-915771802,1006122647,-1856486019,1647722535,-680569246,-626686781,-1655408599,1271813762,-2066707934,940234907,-447431864,-1547483096,-26155739,
-1437030322,-1388296831,2022578496,201620890,737059042,-2089958571,-279803087,-1135559735,-647433010,-187829769,-1597992960,-1203360253,387260342,1543573924,-1288280557,-1961778302,
1034900740,-434776465,-1423512071,1066201901,175048323,1350938635,-209748238,1530622968,1036046448,222350998,1570843282,-1104791220,-481277224,1746485726,124447801,647203623,
2054538391,1708378480,-1773221520,-172190701,1117365616,2056831644,466209254,17816268,905379482,622305914,2074410437,-1159352211,-762660256,637857029,-1021168593,54986762,
-432198768,-453595102,741094372,-2031194701,1503325452,-1806530754,929954404,72524233,-765353043,1836836043,723094852,-1494794027,582136717,-1827256891,-1199298829,278785166,
-1028227685,-197728854,-1257344742,-235374889,1460459489,1529970603,-191598593,-1229164819,-2121139703,1455246711,1996596607,-734483511,357564711,1742409200,1756645386,646377559,
-1414685377,1137224201,1880439670,-1372569504,878196695,-70244055,481524448,-2057588831,794934368,-1642703535,879820477,-2115877667,-1109830468,-1695091659,806167354,-1406625124,
-273161719,-23335708,-709138764,2137733320,-692906746,397575496,-1269853169,788003817,362169695,1827807356,908401485,-2037709597,-616519907,1740378186,-1355786333,-850582028,
-2143569861,46857918,1472540242,1884128321,657233929,-639409284,501319235,-1987555171,1571615587,490476423,495026296,-650027907,1433391829,1865087944,-62437597,-1992496479,
1739210792,1653101507,-1516414249,-1488624128,-521760024,-304963386,-1560031460,1521030974,-2017980466,-840916996,-1732174800,42200631,1649784198,-230803173,-1468665307,-252176452,
-18841230,1736606128,942224577,138844179,614898706,1120358812,-1657382008,-1506840321,560721511,-756635983,-826764091,-780092120,-1748526070,553391653,-1884568421,-431894637,
-13898956,-869950599,-1710171208,-2072512557,-903615297,-310396145,-1584002609,-1173656403,210886394,-1219177410,-1658095677,113675357,-1339771926,1854853350,921018426,1153938759,
1460463153,-143802268,-942695149,290735308,2133807230,-104194395,1591009704,-445382018,413863220,-349909924,471728804,-438116831,288704038,537167769,199661449,1431933110,
-1170207056,-1670689948,1119687168,939111100,-1889940108,1350480222,1634163929,-852485015,-1523374879,-1537970014,-1531537965,-34666303,-928680839,-1344218316,-77658672,189982691,
535275245,1323065686,876362543,1052206995,1957242888,620204927,948240623,353572152,278694722,68227730,-1561537965,-1570088971,-2135018518,-1136805970,-1686515855,-1523890857,
-1202245242,-1885307377,-1662127598,-742644003,-1213038087,1590356152,-733762607,1034823988,230091078,1542329140,2053984081,-588315321,-1348355387,2138243970,1213010891,1086328509,
-835582849,-1262789383,-1416431133,-1078729961,-1800490343,225473300,1334505074,1500869560,-669926682,-1709770030,1987705343,1916670649,1271274364,-1881207666,448633064,1194616269,
-35936333,-2049368359,1697192421,-1000105077,-1271271748,-1065218169,-1957069190,684083475,686715950,-1778650542,625038255,-356892782,-434706373,290571380,-232660235,-1737119329,
-1431233075,1418263466,-696602848,-390856721,-1493032924,1121120984,975265953,1763988875,145474389,931600255,-2080880622,-149165875,-1972671306,166270009,-794940766,938739135,
-1104764597,-1580361827,141799578,-1445045206,-751543775,-1897152070,1797928873,-1310192528,202992635,838933949,-605878955,305144693,1538466071,-276015941,719995893,644005450,
1478642855,-707445553,2038074502,1684369390,-935243734,680871081,-1386112320,869845045,-782207567,670500487,1252178658,1347831692,798816314,379480532,-2076351383,1388438634,
1557352217,767240885,-189589976,1841600771,-58977584,-63522425,1107143983,-234211909,1424971771,2081265235,-1271558397,-226567744,1850129605,437289277,1472583513,1972898211,
-1535268848,1334905131,1644236081,952410442,265627372,-1982473106,1013144955,-260516145,-184701996,384219955,-1917718807,727363530,1068036874,-1754785787,2117395184,1894672461,
790559312,-1058385698,-1290349376,-1220612542,-724207835,-519698808,-988443469,1865862597,-31025487,211707480,-1823788464,1786601883,-1829045295,-411696106,-339815344,-1548568557,
-1360900421,834189610,-1658807804,-1583996225,47108715,1664825424,1198747035,-1489849038,2087038440,-727647271,2073794974,-508328655,-750941299,1406046137,-451302245,1199339897,
393190893,-1282697649,-2074859934,-1071672161,693024979,1556617650,612971297,1803028624,-1488232810,670371300,170062382,2140746805,1381997589,1164275814,-266706121,-872742267,
1589593122,1231209051,2045320941,-1433412326,788205398,770323143,1243560726,-1514393720,-1120538793,-11441862,170341912,1441046102,-106037685,1311959379,-1649307741,11624747,
-420270352,502969050,2133317078,2084141361,-1075279286,1741629871,-1062602195,-2006449606,1268620999,-982451706,-140603316,1013239909,-1123045585,1264602773,-21474602,666292487,
-1156943829,1472529687,-1946577810,-1383006323,-319914138,1599123405,433961020,-1452302598,83621407,64009926,518801171,1444917006,449849043,1590677365,-553436673,-669973343,
677241962,-1980288078,-1938703447,1358618404,-927077836,1081064694,-2002567278,104758320,688141875,-1314694978,-1789807718,-1839377336,-333337314,1856411333,-1681387850,1381129696,
-502093290,-203210448,-206854348,-1817571427,-1624241091,-767487596,-1855728845,-1385527203,-1385395949,1413633739,131847991,-2053951260,-812719223,289252481,1567728586,-1944857879,
-1708647148,-926311183,-2032240217,-1285527051,-1756906230,601081529,1076590571,-119484998,-1955669269,-1535010360,-518195896,-1822080450,-985407414,733482168,-1730894302,670638575,
-2009410236,-2125959628,855126562,1851724265,584149830,-546318634,571593737,-1423790358,337015911,640745123,-1204925936,699059918,-676501063,1913992116,110372436,433348931,
805311214,1454287108,1698852588,-46814702,1324317150,1129658544,-1722804795,-1398807749,1025521804,1592596061,-2065199250,-1853209702,-1098414183,-1962751361,-681684818,-1857742936,
-936035062,695599394,-1568532210,1884777262,-2057829302,583831187,-932415115,-1865169292,-148875477,362772661,-1136233767,-1975684368,-1113112357,280238235,-1862481595,-2061216688,
-157872776,1575631706,-588993091,-1998026743,-512654014,-1120720467,1274042246,-937511336,373222046,-1265791908,684926404,-1488933746,2011889254,-1901133968,46637315,-1190625271,
-575950942,2054409232,1217998689,-101261067,-2072267827,211764567,1946923345,-1454138840,-11016500,-1163677355,388053986,12222101,-528640388,-201531451,49693582,1180458457,
-2029347714,1387457005,-2113184494,-1130625012,61422436,-713170188,-1089559753,947444605,-2079940938,497288446,-2140969269,-1506937624,704216404,855698169,-1040772696,669422198,
-1887732161,878280062,-1445531695,1198740019,1961064457,769219386,93403173,-1025312463,-1772261720,1954209996,-797983027,363699877,-2054303399,-244446814,-1679608133,-167130355,
-565728846,281877668,1128946161,-1261535124,595231012,1497016771,-1739771691,442708060,-2070209183,1833373489,2008323085,712627659,-827618854,804533894,-1739661679,1724529667,
-1218106393,-39517588,-303175425,-121640385,-1502076782,-380520912,568641803,-1775112351,726707371,1387212949,1293444914,171962324,-730070697,267802167,-2063771404,-641213454,
-1156552843,100878116,-1388258852,-1120511001,922642940,-792653690,-1077185317,1135376820,-649968938,1905631392,818032552,1814171873,1451381269,-1765475518,-477894828,-2040017346,
-399064662,704260554,1028714760,-1270389101,-1674397289,-1837913847,-829481294,878643472,-1307742586,166660083,-403315912,1544626251,-1957880478,-1765556981,-1067156694,1524161139,
-331646688,1863841516,467100694,1067182428,2121321322,-1657075887,-1805077313,875735761,1238228827,-1215432430,1099468039,210749736,773066589,-1548572190,1324334760,-577734137,
-1179543272,-1665391265,-872023415,1694205539,1170246221,-2145334193,1676819527,-104454241,-1695779824,-923805367,-1301651530,-618931393,1446126385,-1416935469,345434447,-293816784,
-411319844,-1036886981,1164175534,-1641799423,828554727,-1191032900,567579048,827754990,-1774194081,1074663038,1157885908,-1629473512,-336201532,1811290898,-869505326,270110515,
595493220,-1313091330,-633145174,1250326307,-393301623,184982103,859983799,-806278415,-100197093,1691910399,-1369320345,-809127814,915971316,-1219458330,1380243114,133735923,
-1279869651,531684889,-1771077583,-514189009,-1771014162,-919743056,-2023732726,-1300100033,-1996263901,705289684,-656453728,192044100,290699781,-1092860777,-1099298600,79111058,
-448468257,1014953782,193669228,-1674498406,-1425955004,-1741597170,2121927581,-724964875,1008148833,1409666135,-938807664,588420547,-1222206202,-301035875,-814537251,1773479891,
2009848340,-172609971,205486104,1825643812,1041970168,-293547007,1321340088,997337759,-1550398318,-124000048,-448971138,-572971556,996353283,-124892397,1748227215,1235265192,
1043958653,-499447946,-1898689865,808706407,-661082345,-1020553144,-988193852,-2070538739,1708535482,-1874982519,884635268,1661385158,1184413464,-1105118605,-761955095,-1478992647,
-213028286,1725758068,-1226389813,-421919708,-2119429772,1168061817,475991934,-2125306474,1245440872,1106888482,-1794470176,-1338934781,-91441979,-868057075,-332031514,-847515735,
1865142873,641011174,1519535702,-1236171016,1310701414,1186301505,380018864,1402776846,-927814859,-1747488959,-509612510,-1266379965,299680691,2102338577,-1433918321,382248688,
-539439111,-1777749178,289363503,1946783679,4023809,1324080447,-2146936539,-540126265,1865218518,-401559870,-87706098,1425757065,-1659903660,-1405680988,1745490591,-1575855435,
-529810188,-1600832801,1350666289,1530631007,-1369882330,1813331179,-635727324,-2134732666,414329294,-1266950688,-1272888090,841488587,-2026092895,1677958961,-672779970,1378720810,
-1319174118,1291532817,-778526784,2067600401,-1123194821,1365741030,1942672227,536835834,-605466840,2143770903,663251502,23280891,-649389944,-1443072814,1080903205,-1307207272,
1598926535,2036408192,1293058253,1719512022,-462451786,419386220,-1057494372,-1061594962,-246188214,1794070839,-949738540,-1594898353,1730892909,1249880101,-2048910210,-704495017,
-320069919,-307563824,196136459,-317607658,-1470560398,995592687,-514951409,-14921068,-1698766968,-1831214971,503893983,-613570653,1968408637,1553169993,533664249,-350374539,
140210096,1981006026,-133774461,1017999839,-986762715,1081020064,1891863264,1970677885,1431672973,1875754028,479842227,273269992,-549277030,1182645242,1675806375,-1290746064,
-1245672541,-908511870,1432206480,-1226234442,46169117,-2038283785,-1813127563,-1247925847,181794889,953702777,1503474072,609066329,-340799427,-619449953,1655920630,1755210368,
-761383602,455396980,371476672,-14286954,1073304239,-1349078879,1678093544,-869708410,836918338,-74796206,-2087857216,1714370896,38009537,-462564818,-1597548458,-389696103,
136481916,959110594,-359927239,581863289,-2119995186,-1313063029,-1058225161,10163895,2087346967,1999137160,-566433726,-768149820,-798723242,1330563988,771739407,1912224201,
1678627080,-1330245785,589888551,-1739003714,1897655504,2136603526,-1923200990,-958361724,1632987263,-443649941,1403802584,-768732486,1197101282,175706890,-931228864,-1601337153,
264165525,12403556,-1869041095,-1627844613,-1398835792,941494110,-1598154756,-375558823,-109374953,-1291549502,-1348729080,-1369875757,641409984,138370290,2116006431,812581202,
1053575428,840269097,-174601527,1742360584,1478006764,-685088060,2033767944,-1184127400,-1985231885,1038541256,494596043,-532610794,-1331628351,227173026,-1734491918,2077910765,
-2043034585,-456067074,1242383031,1815270403,759685151,-1030226569,-1989214787,-650926866,1511116605,1462240050,1371017956,1692094642,1344113571,-299147820,1908838128,1877260089,
-1346999939,1446540002,78066238,-1114820934,-420373876,992315870,2109620409,-525185750,1800912544,1397177693,727802281,1827300270,-2008714528,2037115049,-827867419,1479927225,
644211576,-475098680,1771179202,-505291261,1602414804,714564578,2085435938,-1651730891,-1951833140,-1300526015,-890176473,1748548567,-811096619,1133429781,346231129,-3080453,
204308513,1771687260,1657579281,-789260240,2036174083,673727826,69639963,512845082,-905095343,1016286964,52632126,652233464,1551256083,1284514173,1076410798,-59916802,
-70951568,-1432445896,-1676516735,-320133216,1836785225,-1387220035,1138166338,-1196847302,-142182812,-806408055,-1091665628,466434728,1335796876,-1765084733,-1758619628,1034951910,
24638411,719346238,-1759893734,-1967607048,1265495726,-1709646662,-1653097370,1774903101,-530518349,1024775900,1503189355,-1535853,-397995321,-198407855,-146813965,-877492957,
-1007405102,968972738,1132550694,-840906349,1187203158,-684919910,-252832317,1274251950,-1239766618,-714638784,-232775137,1224313429,-1886871824,2053523291,-1018194570,-1226183006,
-333222465,-877348977,-160695435,383306981,-621386920,-1738687586,-1350683268,-1836597456,722114942,1923651013,-2111946849,386801722,1113682618,-71490488,310603841,-1349892207,
844652461,-1274960781,-1264476604,-27722988,1787795773,-1335672839,1823857395,946777931,-1794076730,-989133570,1544247351,47453766,-1019304458,128166117,-1094957468,-608850680,
1576762598,1674945937,-627888056,187637513,-613180121,2110119713,-2027342843,928235903,2133434752,-1961507122,1007710513,-1709025408,1066813810,1864507629,-1846302721,-1555270863,
2031886306,1255622129,-1225725257,1584191243,-1953821739,-998039608,616056690,1019424189,-703682472,-1781283643,-798461438,-2122845151,1874190012,1836333622,1203114268,-654634886,
386466087,620565038,-1789285404,630884361,62524848,-431353052,157743977,-444302973,127058910,562340897,-1520357176,-339284232,1626722744,1942294141,-1641615721,-1060230236,
33586771,-1494801252,2054983032,-558587061,-1088189639,1168902181,761285679,-1070061877,-550408493,-1317275497,-1539634027,962289420,-624092168,730543979,1174191542,595826183,
463774515,596739707,-725019929,-1544637805,410135452,1662781867,43506343,-300581263,21659072,-1440329398,25647340,342487948,-475716556,2135274282,-729342111,-994905639,
1387786741,1467339877,1891295445,-16505666,245650124,925032423,1469059867,410212245,-2010181484,534088083,2000317793,-1845445620,812021098,26298667,34206482,-1977849297,
1286926200,-781062220,-911023798,-764156739,-2144763459,-651574145,1248361320,-1362408712,108424333,-1541730314,1308048543,-1397997403,-116892352,498476065,-947285103,719113662,
-650472493,1836088639,636462167,-958817406,-480510999,83996228,-1175644789,-1498836102,921606558,-1663884175,1224629661,-1610992897,465080628,-197714381,1259931909,1927243513,
875346043,-53356076,-819426943,-324599266,-474273814,1494530897,-949845739,930113957,1038355225,-655528664,55047892,1835249947,-252429728,-431449682,-603857036,931541896,
1284649699,414139059,173935189,-1170438889,421486883,-1891497881,415013808,-1537904459,830318066,563652010,-2010566005,-1635923168,-887551765,1309085804,1295679246,-1575214574,
-197451151,818419761,645019555,1366998057,-2120650967,260805501,1946756276,-1216686661,-192696652,-1215442355,897691887,2139774616,-2071625765,-1009003013,-1314487514,-485690384,
405735387,-1595478628,-1210584237,343657227,1137847344,-262069054,1157206290,226703979,-1835522382,-423986569,-385285334,-2090951997,1275519963,536764490,-2057341302,-32290399,
1539654923,-339154327,-1206943446,-556974558,-140580585,1665163859,651654271,1747202905,427020319,-755417773,2134746073,-1643489033,-1060699596,1671667922,-2128479493,1634532813,
-1922650364,1158407410,-1639288912,-383413325,1716228424,231920547,1510486923,1151077877,-269815368,-1926267311,-1292568502,1467493820,-1523433051,1013555012,-764033823,328507487,
-2009210731,239679378,-709315886,-1781823871,-730035037,520060063,2030056464,574075384,-1658863299,-1731747321,-1013286090,656740957,-781110779,-1678467462,111388769,1471255057,
398263962,2027218820,1183516121,1392522249,1878091397,-1758128085,-1100491106,-1615953409,1409256639,-418671631,768929871,1327181173,945269460,921960362,1029449969,-1520963801,
-853193407,-1429403632,67503704,-1969849168,-1467027815,339257748,156972713,-1960762415,-300012171,1209550519,1141423760,598890477,-441174160,719252151,1128010588,-1199906122,
-1181883599,-848324279,1665810576,1893704915,8868122,-1958902967,2066188340,-821610917,-223403982,-1148577599,1005442386,-938548045,750407884,178530548,972095281,321689774,
670017900,1854097445,184888512,-1336208266,479640429,37738505,-367030128,1840366157,1556527065,446211465,1789932324,-1014091831,180439454,-1183963707,1020632494,658392348,
326022788,1891218413,-112101285,929051733,1351158551,-590705399,-1372783192,226651980,-299184520,-524928453,39604871,-1077723564,1809308409,291764059,1314744062,1547213789,
1823268199,-272984608,1013455413,1759441429,1693060673,-1144332836,-674190072,-235040630,-1317150970,1898240578,-380409551,216191167,-710251580,89779246,-239849340,1458274261,
-410217963,-2065701515,1557395926,-1548257226,639189208,-411951560,1660665446,-614983444,-865182369,-1447878618,-258537864,-555565752,676065709,-2048357504,-1650533390,1613595091,
-219879814,709289760,-909788361,130782805,-1276877364,766266431,1416164101,-1636972098,1948379902,295181025,1631512904,559799906,-592133002,-1315133826,-1479756115,162663595,
-1043507244,-1076819224,1032965536,739790085,-1192973936,-1437059402,978158746,465039085,1499024378,758095452,695971522,523677918,-778302377,-1220770366,-1059772862,340401072,
730149031,1631334284,-376767858,-1208963599,1484208953,-1510899390,-519349223,190032684,-1801575230,-957267577,-318062812,-1971160242,-6417017,1454065285,-645351376,-1574555356,
943581513,-2054948454,1281977671,446464124,610345218,-493624810,1457484502,-1904354836,711300052,83085252,1827824215,89609107,-2041959123,-805870489,-1425324144,-1715290382,
-463635609,-1010408270,-1812155340,1443583783,1216296824,935611079,748063409,205107210,-2075564007,1463940435,1323913937,-1601637211,1444180680,-1551582068,446608962,-1582749982,
-410022079,90937169,37750458,-1874547812,640106733,1856720804,284408150,519455125,-1973565050,-610685616,-2069184662,-682607306,-1818438204,-1031568011,-1517246616,302603510,
1697084028,-450494835,-1779356092,-1355793669,-369645652,1121614710,1124782988,-970765869,1195113355,1134094021,1680298151,-1066651581,-1930760795,-1805979072,-1930820033,1421165525,
-916774186,673463353,-2029827407,1861457007,-1166358923,-73571569,-84313399,1924818942,1650611228,-1946212954,1672265550,-1835992832,1979242451,69002639,2041101677,-1804421711,
436326986,1499878185,-1254709414,1655446765,-1338062471,-778953196,-2119309132,-1422085206,-1773079508,-1245871889,-2142821969,-1415307780,-352907735,-200235120,-48317272,1009198447,
-1203720574,-410433366,27744092,-725028232,-218177207,-1280225706,1642526716,-1888452018,-711048137,1942144978,1913114444,-144099505,-2005612502,-1378060770,-130564653,1757892282,
-670829329,1512915117,539281264,1241295697,-878487580,1872614058,-515601567,-1577050532,-382797810,-1542204239,-63128513,598227761,-165907902,-1557468921,-1721101320,220656680,
    };
