#include "dmasamp.h"
#include "hardware/adc.h"
#include "hardware/dma.h"

uint dma_chan;
dma_channel_config cfg;
int dmaAsyncState=0;

void dmaSampInit(){
  //C SDK setup to use DMA
  adc_init();
  adc_fifo_setup(
    true,    // Write each completed conversion to the sample FIFO
    true,    // Enable DMA data request (DREQ)
    1,       // DREQ (and IRQ) asserted when at least 1 sample present
    false,   // We won't see the ERR bit because of 8 bit reads; disable.
    false    // retain 16 bits
  );
  adc_set_clkdiv(97);   //48Mhz/98=490kHz sample rate with 12x oversample gives ~40960Hz subsamples or ~10Hz FFT resolution with 4096 bins
  // Set up the DMA to start transferring data as soon as it appears in FIFO
  dma_chan = dma_claim_unused_channel(true);
  cfg = dma_channel_get_default_config(dma_chan);
  // Reading from constant address, writing to incrementing byte addresses
  channel_config_set_transfer_data_size(&cfg, DMA_SIZE_16);
  channel_config_set_read_increment(&cfg, false);
  channel_config_set_write_increment(&cfg, true);
  // Pace transfers based on availability of ADC samples
  channel_config_set_dreq(&cfg, DREQ_ADC);
}

void dmaSampRun(int anchan, uint16_t* capture_buf, unsigned int capture_depth){
  if((anchan<26)||(anchan>29)){return;} //fail, invalid channel
  adc_gpio_init(anchan);
  adc_select_input(anchan-26); //map to ADC channel
  dma_channel_configure(dma_chan, &cfg,
    capture_buf,    // dst
    &adc_hw->fifo,  // src
    capture_depth,  // transfer count
    true            // start immediately
  );
  //start
  adc_run(true);
  dma_channel_wait_for_finish_blocking(dma_chan);
  //done
  adc_run(false);
  adc_fifo_drain();
}

int dmaSampRunAsync(int anchan, uint16_t* capture_buf, unsigned int capture_depth){
  if((anchan<26)||(anchan>29)){ //fail, invalid channel
    dmaAsyncState=DMA_ASYNC_IDLE;
    return dmaAsyncState;
  }
  adc_gpio_init(anchan);
  adc_select_input(anchan-26); //map to ADC channel
  dma_channel_configure(dma_chan, &cfg,
    capture_buf,    // dst
    &adc_hw->fifo,  // src
    capture_depth,  // transfer count
    true            // start immediately
  );
  //start
  adc_run(true);
  dmaAsyncState=DMA_ASYNC_RUNNING;
  return dmaAsyncState;   //running
}

int checkDmaSampRunAsync(){
  if(dmaAsyncState==DMA_ASYNC_RUNNING){
    if(dma_channel_is_busy(dma_chan)){
      return dmaAsyncState; //still running
    }else{
      dma_channel_wait_for_finish_blocking(dma_chan);
      adc_run(false);
      adc_fifo_drain();
      dmaAsyncState=DMA_ASYNC_DONE;
      return dmaAsyncState;
    }
  }else{
    return dmaAsyncState;
  }
}

int resetDmaSampRunAsync(){
  if(dmaAsyncState==DMA_ASYNC_DONE){
    dmaAsyncState=DMA_ASYNC_IDLE;
  }
  return dmaAsyncState;
}

int getADC(int anchan){
  int i,j;
  uint16_t s[GETADCOS];  
  dmaSampRun(anchan,s,GETADCOS);    //settle on new channel
  dmaSampRun(anchan,s,GETADCOS);    //modest number of samples
  j=0;
  for(i=0;i<GETADCOS;i++){
    j=j+s[i];
  }
  return j/GETADCOS;
}
