////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates very basic launching of linux threads
// and shows how you can have more threads running than actual CPU cores
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
#include <stdio.h>
#include <stdlib.h>
#include <pthread.h>
#include <unistd.h>
#include <inttypes.h>
///////////////////////////////////////////////////////////////////////////////////////////
struct _launch_struct
{
  uint8_t	id ;
  void		*(*func)(void *) ;
} ;

typedef struct _launch_struct LS ;
///////////////////////////////////////////////////////////////////////////////////////////
// The child_function is a 'reentrant' subroutine. You can have as many copies of the same
// subroutine running at exactly the same time, on different CPU cores. The term
// reentrant means that you can enter the same subroutine many times, and different
// threads can be executing different parts of the same subroutine at once.
//
// In order for each of the copies of the same subroutine to keep their independence,
// a special argument is passed at run-time and the subroutine can use this argument
// to work out which copy of the subroutine it is. This is not always important, and
// it will depend on what you are using threads for as to whether this matters.

void *
child_function (void *arg)
{
  int		*i, len ;

  // read the parameters that were passed - we will use it to number our children
  i = (int *)arg ;

  len = (rand() % 15) + 1 ; // Pick a random number between 1 and 15 seconds
  fprintf (stderr, "CHILD  %d: sleep=%d seconds\n", *i, len) ;
  sleep (len) ;

  fprintf (stderr, "CHILD  %d: sleep ended\n", *i) ;
  free (arg) ; // return the arg memory to the OS
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
// Our launch function is just a demo. Its not necessary to launch a second level of thread
// but I just wanted to show it can be done. Its also an example of a reentrant function.
//
// The way posix threads work, you need to malloc some memory for the parameter you are going
// to pass to the child as its argument. If you have nothing to pass, that's OK too. Just
// pass a null pointer.

void *
launch_function (void *arg)
{
  LS		*p ;
  int		*i, ret ;
  pthread_t	child_thread_id ;

  // read the parameters that were passed
  p = (LS *)arg ;

  // Create a new child using the paramter
  i = malloc (sizeof(int)) ; // We need new memory to pass to the child
  *i = p->id ;

  // Launch the child
  ret = pthread_create (&child_thread_id, NULL, p->func, (void *)i) ;
  fprintf (stderr, "LAUNCH %d: Created child. ret=%08x, thread=%lx\n", p->id, ret, child_thread_id) ;

  // Wait for the child thread to exit
  ret = pthread_join (child_thread_id, NULL) ;
  fprintf (stderr, "LAUNCH %d: Child returned. ret=%08x, thread=%lxx\n", p->id, ret, child_thread_id) ;

  free (arg) ; // return the arg memory to the OS
  pthread_exit (NULL) ;
}
///////////////////////////////////////////////////////////////////////////////////////////
#define N 8

int
main()
{
  pthread_t	thread [N];
  int		i, ret ;
  LS		*p ;
  time_t	t ;

  // Seed the random number generator using system time
  srand ((unsigned) time(&t)) ;

  printf ("This demo programme will create %d different threads called LAUNCH, and each of those\n", N) ;
  printf ("will in turn create one CHILD thread each. The children will sleep for a random amount\n") ;
  printf ("of time and then return. You will see logs when the LAUNCH and CHILD threads start and\n") ;
  printf ("end.\n\n") ;

  // Launch N threads, mallocing some memory so we can pass a parameter that gives each
  // child a unique number
  for (i=0 ; i < N ; i++)
  {
    p = malloc (sizeof(LS)) ; // allocate memory for the parameter and initialise it
    p->id = i ;
    p->func = child_function ;

    // This is the Linux call to start a thread
    ret = pthread_create (&thread[i], NULL, launch_function, (void *)p) ;
    fprintf (stderr, "MAIN   %d: ret=%08x, thread=%lx\n", i, ret, thread[i]) ;
  }

  // Note: all the memory we allocated for the parameters is no longer ours! It belongs to the
  // threads, so we should not access it any more from here.

  // After the thread(s) have launched, we need to wait for them to finish.
  // The joining need not actually happen in the same order that the threads finish.
  // As implemented here, we wait for them to finish in a particular sequence, but the
  // actual ordering of the pthread_exit calls in real time might be different.
  for (i=0 ; i < N ; i++)
  {
    ret = pthread_join (thread[i], NULL) ;
    fprintf (stderr, "MAIN   %d thread returned: ret=%08x\n", i, ret) ;
  }

  fprintf (stderr, "Finishing up\n") ;
}
