////////////////////////////////////////////////////////////////////////////////////////
// This is a programme that demonstrates the use of pipes to manage the bluetoothctl
// interface
///////////////////////////////////////////////////////////////////////////////////////
//                           Digital Alarm Clock
//                Copyright (C) 2023 - Stefan Keller-Tuberg
//                       skt@keller-tuberg.homeip.net
//
// This file is part of the Digital Alarm Clock project.
//
// The Digital Alarm Clock project is free software: you can redistribute it and/or
// modify it under the terms of the GNU General Public License as published by the
// Free Software Foundation, either version 3 of the License, or (at your option)
// any later version.
//
// The Digital Alarm Clock project is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
// or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
// more details.
//
// You should have received a copy of the GNU General Public License along with
// the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
///////////////////////////////////////////////////////////////////////////////////////////
#define _GNU_SOURCE // otherwise strcasestr won't be loaded

#include <stddef.h>
#include <stdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <unistd.h>
#include <pthread.h>
#include <errno.h>
#include <inttypes.h>
#include <fcntl.h>
#include <pthread.h>
#include <sys/time.h>
#include <unistd.h>
#include <ctype.h>
///////////////////////////////////////////////////////////////////////////////////////////
// The definitions below look like a mess. The mess is caused by the 'pipe' system call
// and the way that the parent and child processes use DIFFERENT identifiers for the same pipe

#define PARENT_READ_PIPE	(0)	// The parent will read from this pipe (and child will write)
#define PARENT_WRITE_PIPE	(1)	// The parent will write into this pipe (and child will read)
#define NUM_PIPES		(2)

#define READ_FD			(0)	// read file desriptor offset used by pipe call
#define WRITE_FD		(1)	// write file descriptor offset used by pipe call
#define NUM_FDS			(2)

int	pipes [NUM_PIPES][NUM_FDS] ;	// Will hold pipe identifiers for stdin and out comms to forked task

#define PARENT_READ_FD		(pipes[PARENT_READ_PIPE][READ_FD])
#define PARENT_WRITE_FD		(pipes[PARENT_WRITE_PIPE][WRITE_FD])
#define CHILD_READ_FD		(pipes[PARENT_WRITE_PIPE][READ_FD])
#define CHILD_WRITE_FD		(pipes[PARENT_READ_PIPE][WRITE_FD])
///////////////////////////////////////////////////////////////////////////////////////////
#define MAC_ADDRESS_LEN		(18) // This is a fixed length format - colon separated, null terminated
#define BLUETOOTH_DEV_RECS	(20)

struct _bluetooth_device_record
{
  char		mac_address [MAC_ADDRESS_LEN] ;
  char		device_name [MAC_ADDRESS_LEN] ; // use the same length as the mac address, for convenience
} ;

typedef struct _bluetooth_device_record BLUETOOTH_DEV_REC ;

BLUETOOTH_DEV_REC	Bluetooth_Devices [BLUETOOTH_DEV_RECS] ;
///////////////////////////////////////////////////////////////////////////////////////////
// Structure for specifying expected strings and responses
struct _expected_strings
{
  char	const 	*target_string ;
  uint8_t const	find_prompt ;
  char	const	*command ;
} ;

typedef struct _expected_strings EXPECT ;

EXPECT Bluetooth_Initialisation[] =
{
  {"Pairable: yes",		1,		"power off"},
  {"Powered: no",		1,		"agent off"},
  {"Pairable: no",		1,		"power on"},
  {"Powered: yes",		1,		"discoverable on"},
  {"succeeded",			1,		"pairable on"},
  {"Pairable: yes",		1,		"agent NoInputNoOutput"},
  {"Agent registered",		1,		"default-agent"},
  {NULL,			0,		NULL}
} ;
///////////////////////////////////////////////////////////////////////////////////////////
#define UNUSED(x)	(void)(x)	// This macro silences unused parameter warnings
///////////////////////////////////////////////////////////////////////////////////////////
#define MAX_LINE	(1000)
#define MAX_LINES	(300)

char	*Read_Lines [MAX_LINES] ;
char	This_Question [MAX_LINE] ;
char	This_Line [MAX_LINE] ;
int	RB_insert, RB_extract, Running_State, Check_Bluetooth_Devices, Disconnect_Bluetooth_Devices, Bluetooth_Exit_Timer ;

char	*Bluetoothctl_Devices_Cmd ;
char	*Bluetoothctl_Devices_String ;
///////////////////////////////////////////////////////////////////////////////////////////
typedef struct timespec         TS ;
#define NS_PER_SECOND		(1000000000)
#define THREAD_START_DELAY	(100000)		// THREAD_START_DELAY is expressed in nanoseconds (i.e. 100us)
#define HUNDRED_MS		(100000000)		// THREAD_START_DELAY is expressed in nanoseconds (i.e. 100ms)
#define QUARTER_SECOND		(NS_PER_SECOND / 4)	// THREAD_START_DELAY is expressed in nanoseconds (i.e. 250ms)
///////////////////////////////////////////////////////////////////////////////////////////
// The state enum is used to record initialisation status, so the threads know where they stand
enum _state
{
  _INITIALISING=0, // define the message queues
  _RUNNING,
  _SHUTTING_DOWN,
} ;
///////////////////////////////////////////////////////////////////////////////////////////
// Sleep_ns is a wrapper to make calls to nanosleep easier
// It is set up as a delay call with ns granularity
//
// The maximum sleep time is ~4 seconds, as the number of nanoseconds is a 32 bit unsigned int

void
Sleep_ns (const uint32_t ns)
{
  TS    ts ;

  if (ns == 0)
    return ;

  ts.tv_sec = ns / NS_PER_SECOND ;
  ts.tv_nsec = ((int)ns - (ts.tv_sec * NS_PER_SECOND)) ;        // nanoseconds (fractional seconds)

  // nanosleep should return 0 if it slept correctly for the full duration
  // However it can be interrupted by a signal, and if so, we need to process
  // the signal and reinitiate the remainder of the delay.
  for (; nanosleep(&ts, &ts) != 0 ;)
  {
    // if interrupted by a signal, errno will be EINTR, and we just try again
    if (errno != EINTR)
      fprintf (stderr, "Problem with Sleep_ns (0x%08x ==> %d.%d): %s\n", ns, (int)ts.tv_sec, (int)ts.tv_nsec, strerror(errno)) ;
  }
} // Sleep_ns
///////////////////////////////////////////////////////////////////////////////////////////
// skip_to_string and read_response are two routines that parse the incoming character
// stream.
//
// skip_to_string discards characters until either the prompt (# ) or one of the target
// strings are reached. The target strings are optional.
//   Returns 0 if reason for return is a prompt,
//   Returns 1 if reason for return is string1
//   Returns 2 if reason for return is string2
// The complete line of text up until the characters that triggered the return will be
// stored in This_Question and the full line in This_Line.
//
// read_response reads WHOLE LINES until the target string is reached, discarding the
// beginning and ending partial lines.

static char	*Last_Point = NULL ;

int
skip_to_string (const char *string1, const char *string2)
{
  char		*p ;
  const char	*ret_string ;
  int		next_index, len, ret_val ;

  // Loop waiting for text
  for (;;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Last_Point points to any residual characters past last searched string
    if (Last_Point == NULL)
    {
      if (Read_Lines[RB_extract] == NULL)
      {
	// There are no characters to search - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // Read_Lines[RB_extract] is not NULL at this point
      Last_Point = Read_Lines[RB_extract] ;
    }

    // Is it the end of the line?
    if (*Last_Point == '\0')
    {
      if (Read_Lines[next_index] == NULL)
      {
	// There are no characters to search - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // The next index is not NULL at this point - we must be at the end of the line
      goto no_match_do_free ;
    }

    // We have a line of text pointed to by Last_Point.
    // Does it contain the skip string?
    if ((p = strcasestr (Last_Point, "# ")) != NULL)
    {
      // We have found a prompt. update Last_Point to point immediately after the prompt
      p += 2 ;
      Last_Point = p ;
      return 0 ;
    }

    if ((string1 != NULL) && ((p = strcasestr (Last_Point, string1)) != NULL))
    {
      ret_string = string1 ;      // We have found string1
      ret_val = 1 ;
      goto clean_exit ;
    }

    if ((string2 != NULL) && ((p = strcasestr (Last_Point, string2)) != NULL))
    {
      ret_string = string2 ;      // We have found string2
      ret_val = 2 ;

    clean_exit:
      // Update Last_Point to point immediately after the string
      len = strlen (ret_string) ;
      p += len ;
      Last_Point = p ;
      strcpy (This_Line, Read_Lines[RB_extract]) ;
      strcpy (This_Question, Read_Lines[RB_extract]) ;

      // We are going to return with the This_Question string containing the full line of text up until the string
      // because it will be useful to determine if we have seen this before or not
      if ((p = strcasestr (This_Question, ret_string)) != NULL)
      {
	p += len ;
	*p = '\0' ;
      }

      return ret_val ;
    }

    // At this point, there is no match - need to wait for more characters
    if (Read_Lines[next_index] != NULL)
    {
    no_match_do_free:
      free (Read_Lines[RB_extract]) ;
      Read_Lines[RB_extract] = NULL ;
      RB_extract = next_index ;
      Last_Point = NULL ;
      continue ;
    }

    Sleep_ns (QUARTER_SECOND) ;
  }
} // skip_to_string
///////////////////////////////////////////////////////////////////////////////////////////
// read_response will read characters up until the end of the line, or prompt '# '
int
read_response (char *full_line, const int len)
{
  int		next_index, n ;
  char		*p ;

  // Does Last_Point point part way into a line? If so, discard the remaining
  // characters and wait for the next whole line.
  for (; Last_Point != NULL ;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Does the prompt string exist part way along this line?
    if ((p = strcasestr (Last_Point, "# ")) != NULL)
    {
      p += 2 ; // the length of the prompt search string
      Last_Point = p ;
      return 0 ;
    }

    // We don't have a prompt string, and we still have a partial line. Wait until the
    // next line commences so we can safely discard the partial line
    if (Read_Lines[next_index] != NULL)
    {
      // discard the remaining characters in this line. Point to the next line
      free (Read_Lines[RB_extract]) ;
      Read_Lines[RB_extract] = NULL ;
      RB_extract = next_index ;
      Last_Point = NULL ;
      break ;
    }

    // Hang around for a bit waiting
    Sleep_ns (QUARTER_SECOND) ;
  }

  // At this point, Last_Point is NULL and we are scanning from the beginning of a line
  // Loop waiting for whole lines of text
  for (;;)
  {
    next_index = RB_extract + 1 ;

    if (next_index >= MAX_LINES)
      next_index = 0 ;

    // Last_Point points to any residual characters past last searched string
    if (Last_Point == NULL)
    {
      if (Read_Lines[RB_extract] == NULL)
      {
	// There are no characters to read - delay and try again soon
	Sleep_ns (QUARTER_SECOND) ;
	continue ;
      }

      // Read_Lines[RB_extract] is definitely not NULL at this point
      Last_Point = Read_Lines[RB_extract] ;
    }

    // At this point, Last_Point points to the beginning of the line we are scanning
    // Does the prompt string exist part way along this line?
    if ((p = strcasestr (Last_Point, "# ")) != NULL)
    {
      p += 2 ; // the length of the prompt search string
      Last_Point = p ;
      return 0 ;
    }

    // No prompt. Do we have a complete line? We only know when there's a new line present
    if (Read_Lines[next_index] != NULL)
      break ;

    // There are no characters to search - delay and try again soon
    Sleep_ns (QUARTER_SECOND) ;
  }

  // We have a complete line of text pointed to by RB_extract at this point.
  // We are sure the prompt is not in this complete line
  if ((n = strlen (Last_Point)) >= len)
  {
    fprintf (stderr, "Truncating response. %d chars available and string is %d chars\n", len, n) ;
    n = len - 1 ;
    Last_Point[n] = '\0' ;
  }

  memmove (full_line, Last_Point, n+1) ;
  free (Read_Lines[RB_extract]) ;
  Read_Lines[RB_extract] = NULL ;
  RB_extract = next_index ;
  Last_Point = NULL ;
  return n ; // number of characters
} // read_response
///////////////////////////////////////////////////////////////////////////////////////////
// send_command writes a command to the pipe and hits 'return'

void
send_command (const char *cmd)
{
  int		n, c ;

  // Send a command and wait for the prompt
  if ((n = strlen (cmd)) > 0)
  {
    write (PARENT_WRITE_FD, cmd, n) ;

    // Send the newline character
    c = '\n' ;
    write (PARENT_WRITE_FD, &c, 1) ;

    // now wait for the prompt
    skip_to_string (NULL, NULL) ;
  }
} // send_command
///////////////////////////////////////////////////////////////////////////////////////////
void
find_prompt_and_issue_command (const EXPECT *e)
{
  int		n ;
  char		c ;

  // Wait until receiving the two strings
  if (e->target_string != NULL)
    for (; skip_to_string (e->target_string, NULL) == 0 ;) ; // 0 means prompt was found

  if (e->find_prompt)
    skip_to_string (NULL, NULL) ;

  // The strings have been received. Send the command
  if ((e->command != NULL) && ((n = strlen (e->command)) > 0))
  {
    write (PARENT_WRITE_FD, e->command, n) ;
    c = '\n' ;
    write (PARENT_WRITE_FD, &c, 1) ;
  }

  // wait for the command to be echoed back at us
  for (; skip_to_string (e->command, NULL) == 0 ;) ; // 0 means prompt was found and we should keep looking
} // find_prompt_and_issue_command
///////////////////////////////////////////////////////////////////////////////////////////
#define MAX_FIELD		(100)
#define NUM_MAC_ADDRESSES	(100)

void
build_connected_list (void)
{
  int		n, i, j, c, ret_reason, connected ;
  char		format[MAX_FIELD], field1[MAX_FIELD] ;
  char		buffer [MAX_LINE] ;
  char		*mac_addresses [NUM_MAC_ADDRESSES] ;

  fprintf (stderr, "Rebuilding connected devices list\n") ;

  // The 'paired-devices' command will cause a list of the previously paired devices to be displayed
  // Pick out the MAC addresses and check to see if they are currently connected
  // If no devices have been paired, the output will be blank and it will return to a prompt
  // MAC address is the second field
  sprintf (format, "%%*[^ ] %%%d[^ ]", MAX_FIELD) ; // prepare this string used for sscanf

  // Create a list of MAC addresses
  for (n=0 ; (Running_State < _SHUTTING_DOWN) && (n < NUM_MAC_ADDRESSES) && (skip_to_string ("Device", NULL) == 1) ;)
  {
    if (sscanf (This_Line, format, field1) == 1) // read MAC address from the device line
    {
      mac_addresses[n] = malloc (MAC_ADDRESS_LEN) ;
      memmove (mac_addresses[n], field1, MAC_ADDRESS_LEN) ;
      n++ ;
    }
  }

  // we have a list of previously known mac addresses - now need to check if any of them are currently connected
  for (i=j=0 ; (Running_State < _SHUTTING_DOWN) && (i < n) && (j < BLUETOOTH_DEV_RECS) ; i++)
  {
    c = snprintf (buffer, sizeof(buffer), "info %s\n", mac_addresses[i]) + 1 ;
    write (PARENT_WRITE_FD, buffer, c) ;
    connected = 0 ;

    while ((ret_reason = skip_to_string ("Name", "Connected")) != 0)
    {
      // The connection status is the second field once more
      if (sscanf (This_Line, format, field1) == 1) // read connected status from the connected line
      {
        // Create a connection record for this device
        memmove (Bluetooth_Devices[j].mac_address, mac_addresses[i], MAC_ADDRESS_LEN) ;

        if (ret_reason == 1)
        {
          memmove (Bluetooth_Devices[j].device_name, field1, MAC_ADDRESS_LEN) ;
          Bluetooth_Devices[j].device_name[MAC_ADDRESS_LEN-1] = '\0' ; // Ensure the string is null terminated (if its a long name)
        }
        else
          connected = (*field1 == 'y') || (*field1 == 'Y') ;
      }
    }

    // If this device is connected, then keep the record
    if (connected)
      j++ ;

    free (mac_addresses[i]) ;
  }

  // Terminate the list of connected devices
  if (j < BLUETOOTH_DEV_RECS)
    Bluetooth_Devices[j].mac_address[0] = '\0' ;

  fprintf (stderr, "\nConnected devices:\n") ;

  for (n=0 ; (n < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
    fprintf (stderr, " %s (%s) connected\n", Bluetooth_Devices[n].mac_address, Bluetooth_Devices[n].device_name) ;
} // build_connected_list
///////////////////////////////////////////////////////////////////////////////////////////
// The job of the reader thread is to read characters into a buffer, malloc some memory
// and then store the line that has just been read.
//
// We have to do this because a pipe doesn't contain a lot of buffer/storage, and will not
// block the sending process if it outputs more characters than are able to be processed.
// So we need a thread to consume and buffer the characters as they arrive from the pipe.

void *
reader_thread (void *arg)
{
  int                   c, n ;
  char			*buffer ;

  UNUSED(arg) ; // silence the compiler warning

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  for (n=0 ; n < MAX_LINES ; n++)
    Read_Lines[n] = NULL ;

  RB_insert = RB_extract = 0 ;

  // Loop reading characters as lines of text, then storing the lines in
  // a circular buffer
  for (; Running_State < _SHUTTING_DOWN ; )
  {
    // Fetch a fresh memory buffer and erase it
    buffer = malloc (MAX_LINE+1) ;
    Read_Lines [RB_insert] = buffer ;
    memset (buffer, 0x00, MAX_LINE+1) ;

    for (n=0 ; n < MAX_LINE ; n++)
    {
      // Read one line of characters into buffer
      if ((c = read (PARENT_READ_FD, &buffer[n], 1)) != 1)
      {
	fprintf (stderr, "reader thread: %s\n", strerror(errno)) ;
	goto exit_reader ;
      }

      // Did we just read a newline?
      if ((buffer[n] == '\n') || (buffer[n] == '\r'))
	break ;
    }

    // Remove the trailing newline
    buffer[n] = '\0' ;

    // Is the line entirely blank?
    for (c=0 ; c < n ; c++)
      if (! isspace(buffer[c]))
	break ;

    // If the line is blank, then ignore
    if ((n <= 1) || (c == n))
    {
      free (Read_Lines [RB_insert]) ; // we are going to loop and check for shutting down, so free buffer and get it again
      continue ;
    }

    // Is the line is too long
    if (n >= MAX_LINE)
    {
      fprintf (stderr, "Reading a line that is too long for allocated buffer!\n") ;
      break ;
    }

    // The line is not blank and not too long
fprintf (stderr, "READER: %s\n", buffer) ;

    if (++RB_insert >= MAX_LINES)
      RB_insert = 0 ;

    if (Read_Lines [RB_insert] != NULL)
    {
      fprintf (stderr, "Read buffer overrun!\n") ;
      break ;
    }
  }

 exit_reader:
  fprintf (stderr, "Dropped out bottom of reader thread: Running state=%d\n", Running_State) ;
  Running_State = _SHUTTING_DOWN ; // aborting
  pthread_exit (NULL) ;
} // reader_thread
///////////////////////////////////////////////////////////////////////////////////////////
// The job of the consumer thread is to parse the incoming stream in an orderly manner
// and to issue commands in an orderly manner. The thread also initialises the bluetoothctl task
void *
consumer_thread (void *arg)
{
  int		n, c, ret_reason ;
  char		command[100] ;
  char		buffer [MAX_LINE] ;
  char		prev_question [MAX_LINE] ;

  UNUSED(arg) ; // silence the compiler warning

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately

  memset (Bluetooth_Devices, 0x00, sizeof(Bluetooth_Devices)) ;
  memset (prev_question, 0x00, sizeof(prev_question)) ;
  Check_Bluetooth_Devices = 0 ;
  Disconnect_Bluetooth_Devices = 0 ;
  memset (Bluetooth_Devices, 0x00, sizeof(Bluetooth_Devices)) ;

  // Wait a short time for all of the threads to have been launched (and all message queues to have been defined)
  for (; Running_State == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  // Power cycle the bluetooth interface and then issue devices command
  for (n=0 ; (Running_State < _SHUTTING_DOWN) && (Bluetooth_Initialisation[n].command != NULL) ; n++)
  {
    // Wait until prompt (which is a '#' character)
    find_prompt_and_issue_command (&Bluetooth_Initialisation[n]) ;
  }

  skip_to_string ("successful", NULL) ;
  write (PARENT_WRITE_FD, Bluetoothctl_Devices_Cmd, strlen(Bluetoothctl_Devices_Cmd)) ;

  // Process the paired-devices command (which was the last command issued during initialisation)
  build_connected_list() ;
  fprintf (stderr, "\nBluetooth has been initialised and is running normally\n") ;

  // Normal running happens below
  for (; Running_State < _SHUTTING_DOWN ;)
  {
    if ((ret_reason = skip_to_string ("(yes/no): ", "Connected: ")) == 1)
    {
      // non-zero return code from 'skip_to_string' means yes/no question was found
      // Unfortunately, the same question can be repeated multiple times. We only want to respond ONCE.
      // Luckily, the questions are numbered but it means we need to keep a copy of the numbered question
      if (strcmp (This_Question, prev_question) == 0)
	continue ; // ignore if the same

      strncpy (prev_question, This_Question, sizeof(prev_question)) ;
      Check_Bluetooth_Devices = 20 ; // Check for new devices 5 seconds after sending last yes/no
      write (PARENT_WRITE_FD, "yes\n", 4) ;
    }

    else if (ret_reason == 2)
    {
      // There has been a change in device connection status
      goto rebuild_connection_list ;
    }

    else if (Check_Bluetooth_Devices == 1)
    {
      // issue a devices command and keep track of the paired devices
    rebuild_connection_list:
      Check_Bluetooth_Devices = 0 ;
      write (PARENT_WRITE_FD, Bluetoothctl_Devices_Cmd, strlen(Bluetoothctl_Devices_Cmd)) ;

      // wait for the command to be echoed back at us
      for (; skip_to_string (Bluetoothctl_Devices_String, NULL) == 0 ;) ; // 0 means prompt was found, and we should keep looking
      memset (Bluetooth_Devices, 0x00, sizeof(Bluetooth_Devices)) ;
      build_connected_list() ;

      // Now trust any devices that we have paired with - this will make it easier for them to reconnect later
      for (n=0 ; (n < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
      {
        c = snprintf (buffer, sizeof(buffer), "trust %s\n", Bluetooth_Devices[n].mac_address) + 1 ;
        write (PARENT_WRITE_FD, buffer, c) ;
        skip_to_string (NULL, NULL) ; // wait for the prompt
      }
    }

    else if (Disconnect_Bluetooth_Devices)
    {
      // Disconnect all current bluetooth devices
      for (n=0 ; (n < BLUETOOTH_DEV_RECS) && (Bluetooth_Devices[n].mac_address[0] != '\0') ; n++)
      {
	// Create a command using this device's mac address
	c = snprintf (command, sizeof(command), "disconnect %s\n", Bluetooth_Devices[n].mac_address) + 1 ;
	write (PARENT_WRITE_FD, command, c) ;

fprintf (stderr, "*** Issued disconnect %s\n", Bluetooth_Devices[n].mac_address) ;

	// now wait for the prompt before issuing the next command
	skip_to_string (NULL, NULL) ;
      }

      Disconnect_Bluetooth_Devices = 0 ;
    }

  }

  Running_State = _SHUTTING_DOWN ; // aborting
  pthread_exit (NULL) ;
} // consumer_thread
///////////////////////////////////////////////////////////////////////////////////////////
// The job of the timeout thread is to handle some downcounters. This thread is a bit of a kludge
// The exit timer is just for demo purposes so that this programme quits cleanly after 75 seconds.

void *
bluetooth_timeout_thread (void *arg)
{
#if 0
  int			n, c ;
#endif

  UNUSED(arg) ; // silence the compiler warning

  pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL) ; // This should be set by default anyway
  pthread_setcanceltype (PTHREAD_CANCEL_ASYNCHRONOUS, NULL) ; // This sets thread to be killed immediately
  Check_Bluetooth_Devices = 0 ;
  Bluetooth_Exit_Timer = 300 ; // 75 seconds - just an arbitrary delay

  // Wait a short time for all of the threads to have been launched (and all message queues to have been defined)
  for (; Running_State == _INITIALISING ;)
    Sleep_ns (THREAD_START_DELAY) ;

  for (; Running_State < _SHUTTING_DOWN ;)
  {
#if 0
    // Count the number of entries in the queues
    for (n=c=0 ; n < MAX_LINES ; n++)
      if (Read_Lines[n] != NULL)
	c++ ;

    fprintf (stderr, "RB_insert=%d, RB_extract=%d, c=%d\n", RB_insert, RB_extract, c) ;
#endif
    if ((Check_Bluetooth_Devices > 1) && (--Check_Bluetooth_Devices == 1))
    {
      // Issue a dummy return press, so that the consumer thread gets tripped
      write (PARENT_WRITE_FD, "\n", 1) ;
    }

    if (Bluetooth_Exit_Timer == 30) // just a random demo to verify the disconnect function works OK
    {
      Disconnect_Bluetooth_Devices = 1 ;

      // Issue a dummy return press, so that the consumer thread gets tripped
      write (PARENT_WRITE_FD, "\n", 1) ;
    }

    if (--Bluetooth_Exit_Timer <= 0)
    {
      write (PARENT_WRITE_FD, "exit\n", 5) ;
      Sleep_ns (NS_PER_SECOND) ; // sleeps one second
      break ;
    }

    Sleep_ns (QUARTER_SECOND) ;
  }

  Running_State = _SHUTTING_DOWN ; // aborting
  pthread_exit (NULL) ;
} // bluetooth_timeout_thread
///////////////////////////////////////////////////////////////////////////////////////////
int
main (int argc, char **argv)
{
  int		c, n ;
  FILE		*g ;
  char		*s, temp_str[1000] ;
  pthread_t	r_id, w_id, t_id ;

  if (argc != 1)
  {
    fprintf (stderr, "Usage: %s\n", argv[0]) ;
    fprintf (stderr, "\n") ;
    fprintf (stderr, "Creates a bluetoothctl child and readies it to pair with a device.\n") ;
    fprintf (stderr, "Using a device such as a phone or computer, look for the raspberry pi,\n") ;
    fprintf (stderr, "and pair with it. Once paired, you should be able to stream audio to\n") ;
    fprintf (stderr, "the Pi and listen to it out of the Pi's headphone jack.\n") ;
    return EXIT_FAILURE ;
  }


  // Determine the version of bluetoothctl that is installed, and display
  // The Bookworm and Bullseye versions of the OS have different bluetoothctl syntax for the paired devices command
  Bluetoothctl_Devices_Cmd = "paired-devices\n" ;
  Bluetoothctl_Devices_String = "paired-devices" ;

  if ((g = popen ("bluetoothctl --version", "r")) == NULL)
  {
    fprintf (stderr, "Cannot popen to determine bluetoothctl version: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  else if (fgets (temp_str, sizeof(temp_str), g) == NULL)
  {
    pclose (g) ;
    fprintf (stderr, "Cannot read bluetoothctl version: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }

  // find the character after the space and read the number... this is the version number
  for (s=temp_str ; (*s != '\0') && (!isspace(*s)) ; s++) ;
  for (; (*s != '\0') && (isspace (*s)) ; s++) ;

  // Versions greater than 5.65 have slightly different syntax
  if ((*s != '\0') && (atof(s) >= 5.65))
  {
    Bluetoothctl_Devices_Cmd = "devices Paired\n" ; // Note the capital P in Paired! It is case sensitive!!
    Bluetoothctl_Devices_String = "devices Paired" ;
  }

  pclose (g) ;

  // The pipe function creates a pipe and puts the file descriptors for the reading and writing ends of the
  // pipe (respectively) into file_descriptor[0] and file_descriptor[1].
  // create two pipes
  if ((pipe(pipes[PARENT_READ_PIPE]) < 0) || (pipe(pipes[PARENT_WRITE_PIPE]) < 0))
  {
    printf ("Cannot create pipes: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------------
  // fork to create a child process
  if ((c = fork()) == 0)
  {
    // I am the child
    // duplicate the pipes created by the parent and use these for stdin and out
    dup2 (CHILD_READ_FD, STDIN_FILENO) ;
    dup2 (CHILD_WRITE_FD, STDOUT_FILENO) ;
    dup2 (CHILD_WRITE_FD, STDERR_FILENO) ;

    // close the file descriptors that are not required by the child
    close (CHILD_READ_FD) ;
    close (CHILD_WRITE_FD) ;
    close (PARENT_READ_FD) ;
    close (PARENT_WRITE_FD) ;

    // We are now ready to switch and become the bluetoothctl process
    char	*e_argv[] = {"bluetoothctl", NULL} ;

    if ((n = execvp(e_argv[0], e_argv)) < 0)
    {
      fprintf (stderr, "Cannot execvp: %s\n", strerror(errno)) ;
      return EXIT_FAILURE ;
    }

    // Not Reached
  }
  else if (c < 0)
  {
    printf ("Cannot fork: %s\n", strerror(errno)) ;
    return EXIT_FAILURE ;
  }
  //------------------------------------------------------------
  // I am the parent - close the file descriptors not required by the parent (ie the child ends of the pipes)
  close (CHILD_READ_FD) ;
  close (CHILD_WRITE_FD) ;
  sleep (0.5) ;
  Running_State = _INITIALISING ; // true means running

  // Create two threads to manage reading and writing to the pipes. Dummy arguments are used
  pthread_create (&r_id, NULL, reader_thread, (void *)0) ;
  pthread_create (&w_id, NULL, consumer_thread, (void *)0) ;
  pthread_create (&t_id, NULL, bluetooth_timeout_thread, (void *)0) ;

  Sleep_ns (HUNDRED_MS) ;
  Running_State = _RUNNING ; // true means running

  // Hang around and wait for the threads to complete
  pthread_join (r_id, NULL) ;

  pthread_cancel (r_id) ;
  pthread_cancel (w_id) ;
  pthread_cancel (t_id) ;

  pthread_join (t_id, NULL) ;
fprintf (stderr, "joined 3\n") ;
  pthread_join (w_id, NULL) ;
fprintf (stderr, "joined 2\n") ;

  // close (PARENT_READ_FD) ; Don't need to close the file descriptor if we fclose the handle
  return EXIT_SUCCESS ;
}
