#!/bin/bash
#                           Digital Alarm Clock
#                Copyright (C) 2023,2024 - Stefan Keller-Tuberg
#                       skt@keller-tuberg.homeip.net
#
# This file is part of the Digital Alarm Clock project.
#
# The Digital Alarm Clock project is free software: you can redistribute it and/or
# modify it under the terms of the GNU General Public License as published by the
# Free Software Foundation, either version 3 of the License, or (at your option)
# any later version.
#
# The Digital Alarm Clock project is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
# or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for
# more details.
#
# You should have received a copy of the GNU General Public License along with
# the Digital Alarm Clock project.  If not, see <http://www.gnu.org/licenses/>.
####################################################################################################################
THIS_SCRIPT="/cgi-bin/setup.cgi"

# Load common functions and data
INCLUDE_COMMON="common.sh"

if test -f "${INCLUDE_COMMON}" ; then
  . ${INCLUDE_COMMON}
fi

Make_Temp "$0"
Read_HTML_Parameters
Process_Inputs
####################################################################################################################
Emit_HTML_Headers icons,noborder
echo ' <body>'
echo "  <div class='background'>"

PREV_HOSTNAME=$(hostname)
[[ -n "${STREAM}" ]] && DEFAULT_STREAM_OR_FILE="${STREAM}"
####################################################################################################################
# Process and commands and directives

if [[ "${command}" == "commence_default_stream" ]] ; then
  browse='B'
  mode='Streaming'
  command=''

  [[ -f "${DEFAULT_STREAM_OR_FILE}" ]] && BROWSE_DIR="$(dirname "${DEFAULT_STREAM_OR_FILE}")"
  [[ -d "${DEFAULT_STREAM_OR_FILE}" ]] && BROWSE_DIR="${DEFAULT_STREAM_OR_FILE}"
fi

if [[ "${command}" == "choose_default_streaming_folder" ]] ; then
  browse=''
  command=''

  if [[ -d "${BROWSE_DIR}" ]] ; then
    DEFAULT_STREAM_OR_FILE="${BROWSE_DIR}"
  else
    BROWSE_ERROR="${BROWSE_DIR}"
  fi
  mode=''
fi

if [[ "${command}" == "commence_fallback" ]] ; then
  browse='B'
  mode='Fallback'
  command=''

  [[ -f "${FALLBACK_ALARM_FILE}" ]] && BROWSE_DIR="$(dirname "${FALLBACK_ALARM_FILE}")"
  [[ -d "${FALLBACK_ALARM_FILE}" ]] && BROWSE_DIR="${FALLBACK_ALARM_FILE}"
fi

if [[ "${command}" == "commence_default_folder" ]] ; then
  browse='D'
  mode='DefaultDir'
  command=''
  [[ -f "${BROWSE_DIR}" ]] && BROWSE_DIR="$(dirname "${BROWSE_DIR}")"
fi

if [[ "${command}" == "choose_default_folder" ]] ; then
  browse=''
  command=''

  if [[ -d "${BROWSE_DIR}" ]] ; then
    DEFAULT_BROWSE_DIR="${BROWSE_DIR}"
  else
    BROWSE_ERROR="${BROWSE_DIR}"
  fi
  mode=''
fi

if [[ "${command}" == "cease_fallback" ]] ; then
  browse=''
  command=''
  mode=''
fi

if [[ -n "${stop_browsing}" ]] ; then
  # We can arrive here either when browsing for fallback file, or browsing for default folder
  browse=''
  NEW_VALUE="$(Percent_Decode "${stop_browsing}")"

  if [[ "${mode}" == 'Streaming' ]] ; then
    if [[ -f "${NEW_VALUE}" ]] ; then
      Check_Valid_Stream "${NEW_VALUE}"

      if [[ -z "${MERROR}" ]] ; then
        DEFAULT_STREAM_OR_FILE="${NEW_VALUE}"
      else
	FALLBACK_ERROR="${MERROR}"
      fi

    elif [[ -d "${NEW_VALUE}" ]] ; then
      # Its a folder - get a list of all music and create a playlist
      Create_Playlist "${NEW_VALUE}" "${TMP}"

      if [[ -n "${MEDIA_CMD}" ]] ; then
	# We have at least one file in the temporary file
	DEFAULT_STREAM_OR_FILE="${MEDIA_CMD}"
      else
	FALLBACK_ERROR="No audio files found under ${NEW_VALUE}"
      fi
    else
	FALLBACK_ERROR="${NEW_VALUE}"
    fi

  elif [[ "${mode}" == 'Fallback' ]] ; then
    if [[ -f "${NEW_VALUE}" ]] ; then
      Check_Valid_Stream "${NEW_VALUE}"

      if [[ -z "${MERROR}" ]] ; then
        FALLBACK_ALARM_FILE="${NEW_VALUE}"
      else
	FALLBACK_ERROR="${MERROR}"
      fi

    elif [[ -d "${NEW_VALUE}" ]] ; then
      Create_Playlist "${NEW_VALUE}" "${TMP}" "${FALLBACK_PLAYLIST}"

      if [[ -n "${MEDIA_CMD}" ]] ; then
	# We have at least one file in the temporary file
	FALLBACK_ALARM_FILE="${MEDIA_CMD}"
      else
	FALLBACK_ERROR="No audio files found under ${NEW_VALUE}"
      fi
    else
	FALLBACK_ERROR="${NEW_VALUE}"
    fi

  elif [[ "${mode}" == 'DefaultDir' ]] ; then
    if [[ -d "${NEW_VALUE}" ]] ; then
      DEFAULT_BROWSE_DIR="${NEW_VALUE}"
    else
      BROWSE_ERROR="${NEW_VALUE}"
    fi
  fi

  mode=''
fi

[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${DEFAULT_BROWSE_DIR}"
[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${MEDIA_DIR}"
[[ ! -d "${FALLBACK_ALARM_FILE}" ]] && [[ ! -f "${FALLBACK_ALARM_FILE}" ]] && [[ "${FALLBACK_ALARM_FILE}" != "radio" ]] && FALLBACK_ALARM_FILE="${MEDIA_DIR}"
[[ "${browse}" == 'F' ]] && [[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="$(dirname "${FALLBACK_ALARM_FILE}")"
[[ "${BROWSE_DIR}" == "." ]] && BROWSE_DIR="${MEDIA_DIR}"
[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${DEFAULT_BROWSE_DIR}"
[[ ! -d "${BROWSE_DIR}" ]] && BROWSE_DIR="${MEDIA_DIR}"

if [[ "${command}" == "measure_ambient_min" ]] ; then
  AMBIENT_MEASUREMENT=$($EXE -a | cut -d, -f1)
  [[ -n "${AMBIENT_MEASUREMENT}" ]] && [[ $AMBIENT_MEASUREMENT -ge 0 ]] && [[ AMBIENT_MEASUREMENT -le 4095 ]] && DEFAULT_MIN_AMBIENT="${AMBIENT_MEASUREMENT}"
fi

if [[ "${command}" == "measure_ambient_max" ]] ; then
  AMBIENT_MEASUREMENT=$($EXE -a | cut -d, -f1)
  [[ -n "${AMBIENT_MEASUREMENT}" ]] && [[ $AMBIENT_MEASUREMENT -ge 0 ]] && [[ AMBIENT_MEASUREMENT -le 4095 ]] && DEFAULT_MAX_AMBIENT="${AMBIENT_MEASUREMENT}"
fi

if [[ "${command}" == "display_test" ]] ; then
  RET=$($EXE -z)
fi

if [[ "${command}" == "brightness_test" ]] ; then
  RET=$($EXE -y)
fi

if [[ "${command}" == "new_hostname" ]] ; then
  if [[ -z "${new_hostname}" ]] ; then
    echo '   <fieldset>'
    echo '    <legend><b>Error!</b></legend>'
    echo '    <p style="color:red;font-size:25px;">No new clock name was specified!</p>'
    echo '    <p>Fill in the new clock name text box.</p>'
    echo '   </fieldset>'
  else
    NEW_HOSTNAME="$(Percent_Decode "${new_hostname}")"
    NEW_HOSTNAME_VALUE="${NEW_HOSTNAME}"
  fi
fi

if [[ "${command}" == "save" ]] ; then
  echo "# Raspberry Pi Alarm Clock setup file" > "${TMP}"
  echo "#" >> "${TMP}"
  echo "# The LOG_FILE setting must be a path to a writable filename (eg use /var/log/alarm-clock.log)" >> "${TMP}"
  echo "# The VERBOSE setting is expressed in '0x----' hex as a bitmask (eg. 0x200) - see alarm-clock.c" >> "${TMP}"
  echo "#" >> "${TMP}"
  echo "# The following provide a guide for determining the VERBOSE bitmask:" >> "${TMP}"
  echo "#  DISPLAY   = 0x0001" >> "${TMP}"
  echo "#  CLOCK     = 0x0002" >> "${TMP}"
  echo "#  DIMMER    = 0x0004" >> "${TMP}"
  echo "#  TIMER     = 0x0008" >> "${TMP}"
  echo "#  SETUP     = 0x0010" >> "${TMP}"
  echo "#  MEDIA     = 0x0020" >> "${TMP}"
  echo "#  AMP	     = 0x0040" >> "${TMP}"
  echo "#  BUTTON    = 0x0080" >> "${TMP}"
  echo "#  AMP_MEDIA = 0x0100" >> "${TMP}"
  echo "#  ALARM     = 0x0200" >> "${TMP}"
  echo "#  MULTICAST = 0x0400" >> "${TMP}"
  echo "#  BLUETOOTH = 0x0800" >> "${TMP}"
  echo "#" >> "${TMP}"
  echo "# If used, it is recommended that VERBOSE and LOG_FILE are defined before any other variables." >> "${TMP}"

  if [[ -n "${VERBOSE}" ]] && [[ -n "${LOG_FILE}" ]] ; then
    echo "VERBOSE=${VERBOSE}" >> "${TMP}"
    echo "LOG_FILE=\"${LOG_FILE}\"" >> "${TMP}"

  else
    echo "# Uncomment the following two lines. Remember to set the VERBOSE preferences to initiate logging." >> "${TMP}"

    if [[ -n "${VERBOSE}" ]] ; then
      echo "VERBOSE=${VERBOSE}" >> "${TMP}"
    else
      echo "#VERBOSE=0x090   # 0x090 sets the BUTTON and SETUP verbose bits" >> "${TMP}"
    fi

    if [[ -n "${LOG_FILE}" ]] ; then
      echo "LOG_FILE=\"${LOG_FILE}\"" >> "${TMP}"
    else
      echo "#LOG_FILE=/var/log/alarm-clock.log" >> "${TMP}"
    fi
  fi

  echo >> "${TMP}"
  echo "DEFAULT_1224=\"${DEFAULT_1224}\"" >> "${TMP}"
  echo "DEFAULT_HOUR_ZERO=\"${DEFAULT_HOUR_ZERO}\"" >> "${TMP}"
  echo "DEFAULT_SNOOZE_PRESS=\"${DEFAULT_SNOOZE_PRESS}\"" >> "${TMP}"
  echo "VOLUME=\"${VOLUME}\"" >> "${TMP}"
  echo "MIN_VOLUME=\"${MIN_VOLUME}\"" >> "${TMP}"
  echo "DEFAULT_TIME=\"${DEFAULT_TIME}\"" >> "${TMP}"
  echo "SHUFFLE=\"${SHUFFLE}\"" >> "${TMP}"
  echo "DEFAULT_ALARM_DURATION=${DEFAULT_ALARM_DURATION}" >> "${TMP}"
  echo "DEFAULT_MEDIA_DURATION=${DEFAULT_MEDIA_DURATION}" >> "${TMP}"
  echo "DEFAULT_SNOOZE_DURATION=${DEFAULT_SNOOZE_DURATION}" >> "${TMP}"
  echo "DEFAULT_BROWSE_DIR=\"${DEFAULT_BROWSE_DIR}\"" >> "${TMP}"
  echo "FALLBACK_ALARM_FILE=\"${FALLBACK_ALARM_FILE}\"" >> "${TMP}"
  echo "DEFAULT_STREAM_OR_FILE=\"${DEFAULT_STREAM_OR_FILE}\"" >> "${TMP}"
  echo "DEFAULT_MIN_LED=${DEFAULT_MIN_LED}" >> "${TMP}"
  echo "DEFAULT_MAX_LED=${DEFAULT_MAX_LED}" >> "${TMP}"
  echo "DEFAULT_MIN_AMBIENT=${DEFAULT_MIN_AMBIENT}" >> "${TMP}"
  echo "DEFAULT_MAX_AMBIENT=${DEFAULT_MAX_AMBIENT}" >> "${TMP}"
  echo "CLOCK_TYPE=\"${CLOCK_TYPE}\"" >> "${TMP}"

  # Does the temporary file exist and is it non-empty size??
  if [[ -s "${TMP}" ]] ; then
    for ((i=9 ; i > 0 ; i--)) ; do
      j=$((i-1))
      [[ -f "${SETUP_FILE}.${j}" ]] && mv "${SETUP_FILE}.${j}" "${SETUP_FILE}.${i}"
    done

    mv "${SETUP_FILE}" "${SETUP_FILE}.${j}"
    mv "${TMP}" "${SETUP_FILE}"
    chmod 666 "${SETUP_FILE}"
  fi

  command=''

  # Issue the command to reload the new settings
  $EXE -r > /dev/null 2>&1
fi

if [[ "${command}" == "pair" ]] ; then
  command=''
  $($EXE -b pair)
  PAIRING_INITIATED='Y'
fi

if [[ "${command}" == "forget" ]] ; then
  command=''
  $($EXE -b unpair)
  UNPAIRING_INITIATED='Y'
fi

# Has a new hostname been specified?
if [[ "${NEW_HOSTNAME}" == "${PREV_HOSTNAME}" ]] ; then
  HOSTNAME_ERROR="${NEW_HOSTNAME}"

elif [[ -n "${NEW_HOSTNAME}" ]] ; then
  # Are there any other clocks that have been discovered?
  readarray -t ClockList < <( $EXE -c clocks )

  for ((i=0 ; i < ${#ClockList[@]} ; i++)) ; do
    # There are four columns delimited by commas
    IFS=',' read -ra Array <<< "${ClockList[$i]}"

    # If the first character in the first column is an asterisk, this is an error message
    [[ "${Array[0]}" == ERROR* ]] && continue

    # Check to see if the new clock name matches an existing clock
    if [[ "${NEW_HOSTNAME}" == "${Array[3]}" ]] ; then
      HOSTNAME_DUPLICATE="${NEW_HOSTNAME}"
      break
    fi
  done

  # Check for special characters and flag an error if present
  CHAR_SET="A-Za-z0-9.\-"
  if [[ "${NEW_HOSTNAME}" =~ [^${CHAR_SET}] ]] ; then
    HOSTNAME_ERROR="${NEW_HOSTNAME}"

  elif [[ -z "${HOSTNAME_DUPLICATE}" ]] ; then
    sudo bash -c "hostnamectl set-hostname ${NEW_HOSTNAME} 2>&1 > /dev/null && sed -i s/${PREV_HOSTNAME}/${NEW_HOSTNAME}/g /etc/hosts 2>&1 > /dev/null" && PREV_HOSTNAME="${NEW_HOSTNAME}"

    # Issue the command to reload the new settings
    $EXE -r > /dev/null 2>&1
  fi
fi
####################################################################################################################
if [[ -n "${FALLBACK_ERROR}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>${FALLBACK_ERROR}!</p>"
  echo '    <p>Not a valid audio file/filename for the alarm clock. Try a different filename or renaming.</p>'
  echo '    <p>If the directory or file names contain spaces or special characters, the files will not be recognised. You can run the fix-media-tree script from an ssh session to remove those characters.</p>'
  echo '   </fieldset>'
fi

if [[ -n "${STREAM_ERROR}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>${STREAM_ERROR} doesn't contain valid audio!</p>"
  echo '    <p>Not a valid playlist stream. Try a different URL/file.</p>'
  echo '   </fieldset>'
fi

if [[ -n "${BROWSE_ERROR}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>${BROWSE_ERROR} isn't a valid folder!</p>"
  echo '    <p>Try selecting a different folder.</p>'
  echo '   </fieldset>'
fi

if [[ -n "${HOSTNAME_DUPLICATE}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>The name '${HOSTNAME_DUPLICATE}' already exists!!</p>"
  echo "    <p>Specify a clock name that doesn't already exist on your network.</p>"
  echo '   </fieldset>'
fi

if [[ -n "${HOSTNAME_ERROR}" ]] ; then
  echo '   <fieldset>'
  echo '    <b>Error!</b>'
  echo "    <p style='color:red;font-size:25px;'>${HOSTNAME_ERROR} can't be used!</p>"
  echo "    <p>Specify a clock name that is different than the current name, and doesn't contain special characters or spaces.</p>"
  echo '   </fieldset>'
fi
#--------------------------------------------------------
echo "   <form action='${THIS_SCRIPT}' method='get'>"
Include_Defaults

if [[ "${mode}" == 'Streaming' ]] ; then
  Include_FileSystem_Browser
  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Browsing for default folder</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='choose_default_streaming_folder'>Choose ${BROWSE_DIR} as the default streaming folder</button><br>"
  echo '    </fieldset>'
#--------------------------------------------------------
elif [[ "${mode}" == 'Fallback' ]] ; then
  Include_FileSystem_Browser
  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Brosing for fallback media</b></legend>'
  echo "     <button class='blue' type='submit' name='command' value='cease_fallback'>Cease browsing for fallback file or folder (making no selection)</button><br>"
  echo '    </fieldset>'
#--------------------------------------------------------
elif [[ "${mode}" == 'DefaultDir' ]] ; then
  Include_FileSystem_Browser
  echo '    <br>'
  echo '    <fieldset>'
  echo "     <legend><b>Browsing ${BROWSE_DIR} for default folder</b></legend>"
  echo "     <button class='blue' type='submit' name='command' value='choose_default_folder'>Choose ${BROWSE_DIR} as the default browsing folder</button><br>"
  echo '    </fieldset>'
#--------------------------------------------------------
else
  echo '    <fieldset>'
  echo '     <legend><b>Bluetooth Actions</b></legend>'

  echo "     <table class='noborder'>"
  echo '      <tr>'
  echo '       <td>'
  echo "        <div class='tip'>"
  echo "         <button class='green' name='command' value='pair' type='submit'>Pair Bluetooth Device<br><i class='fa fa-bluetooth fa-2x' style='padding-top: 7px'></i></button>"
  echo "         <span class='text'>To pair a new bluetooth device, click this button and this clock will temporarily become a discoverable bluetooth device.<br><br>On your phone or tablet, enter your bluetooth settings mode and look for the clock. Then pair with the clock as you would pair with a car audio system or another bluetooth speaker.<br><br>Once paired your device should be able to reconnect to the clock when you need without having to initiate the pairing process again.<br><br>If you can't reconnect your device to the clock, try turning off bluetooth on your device, then turning it back on again, then try to connect again. If that doesn't work, use the 'Reset Pairing' function.<br><br>If you unpair a device, you may be prevented from re-pairing it if it is not first un-paired from the other device too. If that happens, use the clock's 'Reset Pairing' function to clear out all records of previously paired devices.<br><br>Pairing may also be initiated by pressing two of the clock's alarm switches at the same time.</span>"
  echo '        </div>'
  echo '       </td>'

  echo '       <td>'
  echo "        <div class='tip'>"
  echo "         <button class='red' name='command' value='forget' type='submit'>Reset Pairing<br><i class='fa fa-bluetooth fa-2x' style='padding-top: 7px'></i></button>"
  echo "         <span class='text'>If you unpair a device, you may be prevented from re-pairing it if it is not first un-paired from the other device too. If that happens, use the clock's 'Reset Pairing' function to clear out all records of previously paired devices. Then re-pair your device(s) again.</span>"
  echo '        </div>'
  echo '       </td>'
  echo '      </tr>'
  echo '     </table>'

  if [[ -n "${PAIRING_INITIATED}" ]] ; then
    echo "    <p style='color:green;font-size:22px;'>Pairing initiated!</p>"
  fi

  if [[ -n "${UNPAIRING_INITIATED}" ]] ; then
    echo "    <p style='color:red;font-size:22px;'>Unpaired all devices!</p>"
  fi

  echo '    </fieldset>'
  echo '    <br>'

  if [[ -n "${DEFAULT_1224}" ]] ; then
    if [[ ${DEFAULT_1224} -eq 12 ]] ; then
      CHECK12="checked='true'"
    else
      CHECK24="checked='true'"
      DEFAULT_1224=24
    fi
  else
    CHECK24="checked='true'"
    DEFAULT_1224=24
  fi

  # Note: double comma syntax below converts string to lower case
  if [[ "${DEFAULT_HOUR_ZERO,,}" == "nonblank" ]] ; then
    CHECK_WITH="checked='true'"

  elif [[ "${DEFAULT_HOUR_ZERO,,}" == "blank" ]] ; then
    CHECK_WITHOUT="checked='true'"

  else
    DEFAULT_HOUR_ZERO="blank"
    CHECK_WITHOUT="checked='true'"
  fi

  # Note: double comma syntax below converts string to lower case
  if [[ "${DEFAULT_SNOOZE_PRESS,,}" == "ignore" ]] ; then
    CHECK_IGNORE="checked='true'"

  elif [[ "${DEFAULT_SNOOZE_PRESS,,}" == "restart" ]] ; then
    CHECK_RESTART="checked='true'"

  else
    DEFAULT_SNOOZE_PRESS="restart"
    CHECK_RESTART="checked='true'"
  fi

  if [[ -n "${CLOCK_TYPE}" ]] ; then
    if [[ ${CLOCK_TYPE:0:1} == "A" ]] ; then
      ACLOCK_CLUSTERED="checked='true'"
      CLOCK_TYPE="AClustered"

    elif [[ ${CLOCK_TYPE:0:1} == "C" ]] ; then
      CLOCK_CLUSTERED="checked='true'"
      CLOCK_TYPE="Clustered"

    elif [[ ${CLOCK_TYPE:0:1} == "Z" ]] ; then
      ZCLOCK_CLUSTERED="checked='true'"
      CLOCK_TYPE="ZClustered"

    else
      CLOCK_STANDALONE="checked='true'"
      CLOCK_TYPE="Standalone"
    fi
  else
    CLOCK_STANDALONE="checked='true'"
    CLOCK_TYPE="Standalone"
  fi

  # input form that lists existing alarms, and offers the option to delete them.
  echo '    <fieldset>'
  echo '     <legend><b>Clock Settings</b></legend>'
  echo "     <div class='tip'>"
  echo '      Clock displays 12 or 24 hour time:<br>'
  echo "      <input type='radio' name='default_1224' value='12' ${CHECK12}>12 hour time format<br>"
  echo "      <input type='radio' name='default_1224' value='24' ${CHECK24}>24 hour time format"
  echo "      <span class='text'>This selection sets the time preference for the clock display, but not the browser display. The way times and dates appear in your browser is determined by the computer's locale setting.</span>"
  echo '     </div>'

  echo '     <br><br>Clock displays hours leading zero:<br>'
  echo "     <input type='radio' name='default_hour_zero' value='nonblank' ${CHECK_WITH}>Display hours leading zero<br>"
  echo "     <input type='radio' name='default_hour_zero' value='blank' ${CHECK_WITHOUT}>Blank hours leading zero<br>"

  echo "     <div class='tip'>"
  echo '      <br>Second press of snooze:<br>'
  echo "      <input type='radio' name='default_snooze_press' value='ignore' ${CHECK_IGNORE}>Ignore<br>"
  echo "      <input type='radio' name='default_snooze_press' value='restart' ${CHECK_RESTART}>Restart paused media"
  echo "      <span class='text'>When set to 'Ignore' and media is playing either because of an alarm or because it has been manually initiated, short-pressing the snooze button causes the media to pause and the snooze timer to commence. Additional short-presses of the snooze button are ignored until the snooze timer completes.<br><br>When set to 'Restart paused media' and media is playing, each press of the snooze button toggles the media between paused and playing.<br><br>Long-pressing the snooze button when media is playing always stops and cancels the media stream or alarm.</span>"
  echo '     </div>'

  echo "    <br><br>"
  Do_Slider default_snooze_duration "Default snooze duration" 1 60 ${DEFAULT_SNOOZE_DURATION} 'N' "minutes"
  Do_Slider default_media_duration "Default media playing duration" 1 180 ${DEFAULT_MEDIA_DURATION} 'N' "minutes"

  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo '     <legend><b>Display brightness</b></legend>'
  echo "     <div class='tip'>"
  Do_Slider default_min_led "Minimum" 1 255 ${DEFAULT_MIN_LED} 'B'
  Do_Slider default_max_led "Maximum" 1 255 ${DEFAULT_MAX_LED} 'B'
  echo "      <span class='text'>These settings establish the dimmest and brightest LED levels that will be used by the clock.<br><br>The clock will adjust the instantaneous brightness according to how bright the room is. As the room darkens, the display will become dimmer until the minimum dimness is reached. As the room brightens, the display will become brighter until the maximum brightness is reached.<br><br>Tip: (1) Adjust the minimum brightness slider in a darkened room so that you can still read the display. Its value will probably be in single digits if the clock is going to be used in a darkened bedroom. Then (2) adjust the maximum brightness slider in a bright room so that you can easily read the display. Finally (3) adjust the the ambient light thresholds settings below.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo '     <legend><b>Ambient light thresholds</b></legend>'
  AMBIENT_MEASUREMENT=$($EXE -a | cut -d, -f1)
  echo "     Current light level: ${AMBIENT_MEASUREMENT}<br><br>"

  echo "     <div class='tip'>"
  Do_Slider default_min_ambient "Applies minimum LED brightness at and below light level" 0 4095 ${DEFAULT_MIN_AMBIENT}
  echo "      <button type='submit' name='command' value='measure_ambient_min'>Set minimum using current ambient light level</button><br><br>"
  Do_Slider default_max_ambient "Applies maximum LED brightness at and above light level" 0 4095 ${DEFAULT_MAX_AMBIENT}
  echo "      <button type='submit' name='command' value='measure_ambient_max'>Set maximum using current ambient light level</button><br>"
  echo "      <span class='text'>When the ambient light level is low, the clock display will dim. When the ambient light level is bright, the clock display will brighten.<br><br>Tip: (1) Adjust the minimum and maximum brightness sliders above first. Then (2) darken the room to its night-time light level and look at the light level measurement. The minimum brightness slider should be at or a little greater than the measured level. (3) Brighten the room to its day-time light level, read the ambient level, and set the maximum brightness slider at or a little lower than the measured level.</span>"

  echo '     </div>'
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Default browsing folder</b></legend>'
  echo "     <div class='tip'>"
  Do_Text_Input 'default_browse_dir' "${DEFAULT_BROWSE_DIR}" 'Enter full path to the default browsing folder' "100" "" 'This folder will appear as the starting location when initiating the file browser'
  echo '     </div>'
  echo '     <br>'

  echo "     <div class='tip'>"
  echo "      <button class='blue' type='submit' name='command' value='commence_default_folder'>Browse filesystem to define the default browsing folder</button>"
  echo "      <span class='text'>When choosing a default browsing folder: USB devices get auto-mounted under the folder called '/media'.<br><br>If copying media into the Raspbery Pi's SDMMC filesystem, you could also create a folder tree under '/media'.<br><br>Another possibility is to manually edit the Pi's configuration outside of this setup page to automount network drives from your home network - this is generally done by modifying '/etc/fstab' which you can google - and you can automount those network drives under '/media' also.</span>"
  echo '     </div>'
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>Playback volume</b></legend>'
  echo "     <div class='tip'>"
  Do_Slider volume "Playback Volume" 0 31 ${VOLUME}
  Do_Slider min_volume "Minimum Alarm Volume" 0 31 ${MIN_VOLUME}
  echo "      <span class='text'>The playback volume is the system-wide volume level and can be adjusted using either this web slider, or by using the clock's <b>+</b> and <b>-</b> buttons.<br><br>(Note: The <b>+</b> and <b>-</b> buttons are not active if no media is playing)<br><br>Adjustments made with this slider will immediately have effect on playing media volume, but the volume change will not be saved / remembered unless you click the 'Save These Defaults' button below.<br><br>The minimum alarm volume defines the lowest system-wide volume that will apply to an alarm. Alarm offsets, if specified with any alarm, are still added to the minimum alarm volume. If you've set the system-wide volume setting lower than the minimum alarm volume (for example using the volume down button last time you used the media player), the minimum alarm volume setting will ensure you will still be woken reliably.<br><br>Note: Playback volume can be adjusted in real-time. Minimum alarm volume is only changed when the changes have been saved using the 'Save These Defaults' button.</span>"
  echo '     </div>'
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo '     <legend><b>New alarm defaults</b></legend>'
  # Note: the format for time displayed by the browser is not flexible!
  # In general, the browser uses the LOCALE to determine format, and you can't change it!
  echo '     Default new alarm time:'
  echo "     <input type='time' name='default_time' value='${DEFAULT_TIME}' required>"

  echo '     <hr>'
  Do_Slider default_alarm_duration "Default new alarm duration" 1 180 ${DEFAULT_ALARM_DURATION} 'N' "minutes"

  echo '     <hr>'
  echo "     <div class='tip'>"
  echo '      Default new alarm source: either a streaming URL, <i>OR</i> a file, folder or playlist <i>OR</i> specify radio:<br>'

  EXTRA=""
  [[ -f "${DEFAULT_STREAM_OR_FILE}" ]] && [[ $(file "${DEFAULT_STREAM_OR_FILE}" | sed 's/^.*:[[:space:]]\+//g;s/[[:space:]].*//g') == "ASCII" ]] && EXTRA="($(wc -l < ${DEFAULT_STREAM_OR_FILE}) items)"
  Do_Text_Input 'stream' "${DEFAULT_STREAM_OR_FILE}" 'Enter a streaming URL or a path to a file, folder or playlist, or the word radio' "100" "${EXTRA}"

  echo "      <button class='blue' type='submit' name='command' value='commence_default_stream'>Browse filesystem to find default streaming file, folder or playlist</button><br>"
  echo "      <span class='text'>This streaming URL, file, folder or playlist will appear as the default when creating new alarms.<br><br>Set this field to 'radio' (without the quotes) to default to turning on the radio.<br><br>If the specified path is a directory, a playlist file will be automatically generated and saved into the ${PLAYLIST_DIR} folder. Note that if the characters in the directory or file names contain spaces or special characters, the media files may not be recognised. You can run the fix-media-tree script from an ssh session to remove those characters.</span>"
  echo '     </div>'

  echo '<br>'
  Insert_Examples

  echo '     <hr>'
  Do_Choice N SHUFFLE "Shuffle when more than one media file in a folder or playlist" "Shuffle" "Normal"

  echo '     <hr>'
  echo "     <div class='tip'>"
  echo '      Fallback alarm file, folder or playlist:<br>'

  [[ -n "${FALLBACK_ALARM_FILE}" ]] && FALLBACK_ALARM_FILE_VALUE="value='${FALLBACK_ALARM_FILE}'"
  echo -n "      <input type='text' name='fallback_alarm_file' ${FALLBACK_ALARM_FILE_VALUE} placeholder='Enter full path to an audio file or folder that will be played if the alarm stream fails' size='100'>"
  [[ -f "${FALLBACK_ALARM_FILE}" ]] && [[ $(file "${FALLBACK_ALARM_FILE}" | sed 's/^.*:[[:space:]]\+//g;s/[[:space:]].*//g') == "ASCII" ]] && echo -n " ($(wc -l < ${FALLBACK_ALARM_FILE}) items)"
  echo '      <br>'

  echo "      <button class='blue' type='submit' name='command' value='commence_fallback'>Browse filesystem to find a fallback alarm file, folder or playlist</button>"
  echo "      <span class='text'>A local file or folder (i.e. reliably accessible) which is played if the alarm stream fails or ends.<br><br>Set this field to 'radio' (without the quotes) to fall back to the the analogue radio.<br><br>If the specified path is a directory, a playlist file will be automatically generated and saved into the ${PLAYLIST_DIR} folder. Note that if the characters in the directory or file names contain spaces or special characters, the media files may not be recognised. You can run the fix-media-tree script from an ssh session to remove those characters.</span>"
  echo '     </div>'
  echo '    </fieldset>'

  echo '    <br>'

  echo '    <fieldset>'
  echo '     <legend><b>Stand alone / Clustered</b></legend>'
  echo "     <div class='tip'>"
  echo '      Is this clock a stand alone or clustered clock:<br>'
  echo "      <span class='text'>Clock types:<br><br>Stand Alone: Stand alone clocks operate independently of any other clocks. They do not synchronise their alarms with other clocks, are not controlled by other clocks, and do not control other clocks. The clock does not share alarm settings or button activity with other clocks. Stand alone clocks are audible.<br><br>If you have more than one clock, you can optionally group two or more clocks into a cluster. Each clustered clock will sync its alarms with the others so that alarm changes on any clock in the cluster propagate shortly after to the other clocks in the cluster.<br><br>Clustered clocks may either be 'silent' or 'audible'. Silent clocks will not normally sound any alarms even if alarms are defined within the cluster. The exception to this is if and only if a silent clustered clock cannot see any audible clustered clocks on the network. In that case, the silent clock will sound its alarms.<br><br>You would want to declare a clock to be silent if two or more clustered clocks are within hearing distance of each other (for example on either side of a bed) because digital media isn't perfectly synchronised across devices and audio time lags don't sound good.<br><br>Autonomous: The clock shares alarm settings with other clocks. It can control other clocks but cannot be controlled by other clocks. Autonomous clocks are audible.<br><br>Clustered: The clock shares alarm settings with other clocks. It can both control and be controlled by other clocks. Clustered clocks are audible.<br><br>Silent clustered: The clock shares alarm settings with other clocks. It can control other clocks. Silent clustered clocks will not sound alarms UNLESS there are no other audible clustered clocks visible on the network!</span>"
  echo '     </div>'
  echo '     <br>'
  echo "     <input type='radio' name='clock_type' value='Standalone' ${CLOCK_STANDALONE}>Stand alone clock<br>"
  echo "     <input type='radio' name='clock_type' value='ZClustered' ${ZCLOCK_CLUSTERED}>Clustered clock with silent alarms<br>"
  echo "     <input type='radio' name='clock_type' value='Clustered' ${CLOCK_CLUSTERED}>Clustered clock with audible alarms. Can both control and be controlled by other clustered clocks<br>"
  echo "     <input type='radio' name='clock_type' value='AClustered' ${ACLOCK_CLUSTERED}>Clustered clock with audible alarms. Autonomous. Can control but cannot be controled by other clustered clocks"
  echo '    </fieldset>'
  echo '    <br>'

  echo '    <fieldset>'
  echo "     <div class='tip'>"
  echo "      <button class='green' type='submit' name='command' value='save'>Save These Defaults<br><i class='fa fa-save fa-2x' style='padding-top: 7px'></i></button>"
  echo "      <span class='text'>Clicking here will cause changes in this form to be saved as the clock's new defaults</span>"
  echo '     </div>'
  echo '    </fieldset>'

  echo '    <br>'
  echo '    <fieldset>'
  echo "     <div class='tip'>"
  echo "      <button class='blue' type='submit' name='command' value='display_test'>Start the display test<br><i class='fa fa-lightbulb-o fa-2x' style='padding-top: 7px'></i></button>"
  echo "      <span class='text'>Clicking here will initiate the display test for verifying that all segments and dots are illuminating.<br><br>The clock display will revert to the time of day after the test has completed.</span>"
  echo '     </div>'

  echo "     <div class='tip'>"
  echo "      <button class='blue' type='submit' name='command' value='brightness_test'>Start the LED brightness test<br><i class='fa fa-sun-o fa-2x' style='padding-top: 7px'></i></button>"
  echo "      <span class='text'>Clicking here will initiate the LED brightness test for verifying brightness uniformity of the LED segments and dots. The displayed minimum brightness will be the currently saved minimum display brightness setting. The displayed maximum brightness will be the currently saved maximum display brightness setting.<br><br>The clock display will revert to the time of day after the test has completed.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '   </form>'

  # Change the hostname
  echo '   <form>'
  echo '    <fieldset>'
  echo '     <legend><b>Change Clock Name</b></legend>'
  echo "     <div class='tip'>"
  echo "      This clock's current name is <b>${PREV_HOSTNAME}</b><br>"
  Do_Text_Input 'new_hostname' "${NEW_HOSTNAME_VALUE}" 'Enter a new clock name, or leave blank to keep the current name' "63"
  echo "      <button type='submit' class='green' name='command' value='new_hostname'>Change the clock's name</button>"
  echo "      <span class='text'>Clicking here will change the clock's name on the local network, including its web name, ssh hostname and samba fileserver name.<br><br>After changing this clock's name, other clocks should recognise the name change quickly, however other devices on your network may take some time.</span>"
  echo '     </div>'
  echo '    </fieldset>'
  echo '   </form>'

  echo "   <form action='/index.html'>"
  echo '    <fieldset>'
  echo "     <div class='tip'>"
  echo "      <button class='red' type='submit'>Return to main page</button>"
  echo "      <span class='text'>Clicking here will cause any unsaved changes in this form to be discarded and lost</span>"
  echo '     </div>'
  echo '    </fieldset>'
fi

echo '   </form>'
echo '  </div>'
Insert_Debug_Section
####################################################################################################################
echo ' </body>'
echo '</html>'

[[ -f "${TMP}" ]] && rm -f "${TMP}"
exit 0
